/* $Id: fors_dark-test.c,v 1.6 2013-09-11 13:47:53 cgarcia Exp $
 *
 * This file is part of the FORS Library
 * Copyright (C) 2002-2006 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: cgarcia $
 * $Date: 2013-09-11 13:47:53 $
 * $Revision: 1.6 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <fors_dark_impl.h>
#include <fors_dfs.h>
#include <fors_utils.h>

#include <test_simulate.h>
#include <test.h>

#include <cpl.h>

/**
 * @defgroup fors_dark_test  test of master dark recipe
 */

/**@{*/

#undef cleanup
#define cleanup \
do { \
    cpl_frameset_delete(frames); \
    cpl_parameterlist_delete(parameters); \
    fors_image_delete(&raw_dark); \
    fors_image_delete(&master_dark); \
    fors_image_delete(&master_bias); \
    fors_setting_delete(&setting); \
} while(0)

/**
 * @brief   Test dark recipe
 */
static void
test_dark(void)
{
    /* Input */
    cpl_frameset *frames          = cpl_frameset_new();
    cpl_parameterlist *parameters = cpl_parameterlist_new();

    /* Output */
    fors_image *master_dark  = NULL;
    fors_image *master_bias  = NULL;
    fors_image *raw_dark  = NULL;

    fors_setting *setting = NULL;

    /* Simulate data */
    const char *dark_filename[] = {"dark_1.fits",
                                   "dark_2.fits",
                                   "dark_3.fits"};

    {
        unsigned i;
        
        for (i = 0; i < sizeof(dark_filename)/sizeof(char *); i++) {
            cpl_frameset_insert(frames, 
                                create_dark(dark_filename[i],
                                            DARK, CPL_FRAME_GROUP_RAW));
        }
    }

    setting = fors_setting_new(cpl_frameset_get_position(frames, 0));

    cpl_frameset_insert(frames, 
                        create_master_bias("dark_master_bias.fits", 
                                    MASTER_BIAS, CPL_FRAME_GROUP_CALIB));
    
    fors_dark_define_parameters(parameters);
    assure( !cpl_error_get_code(), return, 
            "Create parameters failed");
    
    fors_parameterlist_set_defaults(parameters);

    /* Call recipe */
    fors_dark(frames, parameters);
    assure( !cpl_error_get_code(), return, 
            "Execution error");

    /* Test results */
    {
        /* New and previous frames */
        test( cpl_frameset_find(frames, MASTER_DARK) != NULL );
        test( cpl_frameset_find(frames, MASTER_BIAS) != NULL );
        test( cpl_frameset_find(frames, DARK) != NULL );
        
        master_dark = fors_image_load(
            cpl_frameset_find(frames, MASTER_DARK));
        
        master_bias = fors_image_load(
            cpl_frameset_find(frames, MASTER_BIAS));
        
        raw_dark    = fors_image_load(
            cpl_frameset_find(frames, DARK));
        
        /* Verify relation
           master_dark = raw_dark - master_bias */
        test_rel( fors_image_get_mean(master_dark, NULL),
                  fors_image_get_mean(raw_dark, NULL) -
                  fors_image_get_mean(master_bias, NULL) - 200, //Hardcoded overscan level
                  0.03);

        /* Verify that error decreased  */
        {
            test( fors_image_get_error_mean(master_dark, NULL) /
                  (fors_image_get_mean(master_dark, NULL) +
                   fors_image_get_mean(master_bias, NULL) + 200) //Hardcoded overscan level
                  <
                  fors_image_get_error_mean(raw_dark, NULL) /
                  fors_image_get_mean(raw_dark, NULL));
        }
    }

    cleanup;
    return;
}

/**
 * @brief   Test of image module
 */
int main(void)
{
    TEST_INIT;
    
    test_dark();

    TEST_END;
}

/**@}*/
