/* $Id: levelloader.c,v 1.22 2003/06/12 13:18:54 eneggers Exp $ */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>

#include <libxml/xmlmemory.h>
#include <libxml/parser.h>

#include "levelloader.h"


/*  structure for saving xml stuff in memory */
typedef struct {
  char *str;
  int len;
} level_string; 

/* Level_setting */
static Level_setting *new_level_setting(void) {
	Level_setting *s;

	s = malloc(sizeof(Level_setting));
	s->name = NULL;
	s->type = -1;

	return s;
};

static void parse_level_setting_vector(Level *l, xmlNodePtr cur) {
	Level_setting *s;
	char *x, *y;

	s = new_level_setting();
	
	s->name = (char *)xmlGetProp(cur, (const xmlChar *)"name");
	s->type = LEVEL_VECTOR;

	x = (char *)xmlGetProp(cur, (const xmlChar *)"value");
	y = strchr(x, ',');
	*y = '\0'; y++;

	(s->data).vval = new_vector(
		(int32_t)atoi(x),
		(int32_t)atoi(y)
	);
	
	list_append(l->settings, s);
}

static void parse_level_setting_int(Level *l, xmlNodePtr cur) {
	Level_setting *s;

	s = new_level_setting();
	
	s->name = (char *)xmlGetProp(cur, (const xmlChar *)"name");
	s->type = LEVEL_INT;
	(s->data).ival = (int)atoi((char *)xmlGetProp(cur, (const xmlChar *)"value")),
	
	list_append(l->settings, s);
}

static void parse_level_setting_int32_t(Level *l, xmlNodePtr cur) {
	Level_setting *s;

	s = new_level_setting();
	
	s->name = (char *)xmlGetProp(cur, (const xmlChar *)"name");
	s->type = LEVEL_INT32_T;
	(s->data).i32val = (int32_t)atoi((char *)xmlGetProp(cur, (const xmlChar *)"value")),
	
	list_append(l->settings, s);
}

static void parse_level_setting(Level *l, xmlNodePtr cur) {
	char *type;
	type = (char*)xmlGetProp(cur, (const xmlChar *)"type");
	if(!strcmp(type, "vector")) parse_level_setting_vector(l, cur);
	else if(!strcmp(type, "int")) parse_level_setting_int(l, cur);
	else if(!strcmp(type, "int32_t")) parse_level_setting_int32_t(l, cur);
}

Level_setting *level_next_setting(Level *l) {
	if(l->_next_setting == NULL) {
		l->_next_setting = new_list_ptr(l->settings);
		if(!list_ptr_first(l->_next_setting)) return NULL;
	} else {
		if(!list_ptr_next(l->_next_setting)) return NULL;
	}
	
	return (Level_setting *)list_ptr_get_data(l->_next_setting);
}

void level_reset_setting(Level *l) {
	if(l != NULL) {
		l->_next_setting = NULL;
	}
}

/* Level_graphic */
static _Level_graphic *new_graphic(void) {
	_Level_graphic *g;

	g = malloc(sizeof(_Level_graphic));
	g->type = -1;
	g->gfxid = -1;
	g->svgseries = NULL;

	return g;
}

static int cmp_gfxid(void *graphic, void *gfxid) {
	if(((_Level_graphic*)graphic)->gfxid == *((int*)gfxid)) {
		return TRUE;
	}
	return FALSE;
}

static char *level_get_svgseries_gfxid(Level *l, int gfxid) {
	_Level_graphic *g =
		(_Level_graphic *) list_search(l->graphics, cmp_gfxid, &gfxid);

	if(g != NULL) {
		return g->svgseries;
	}

	return NULL;
}

static int cmp_type(void *graphic, void *type) {
	_Level_graphic *g = (_Level_graphic*)graphic;
	if(g->gfxid == -1 && g->type == *((int*)type)) {
		return TRUE;
	}
	return FALSE;
}

char *level_get_svgseries(Level *l, int type) {
	_Level_graphic *g =
		(_Level_graphic *) list_search(l->graphics, cmp_type, &type);

	if(g != NULL) {
		return g->svgseries;
	}

	return NULL;
}

static int
level_stringio_write(void *context, const char *string, int len) {
  level_string *outstr = (level_string *) context;

  outstr->str = realloc(outstr->str,outstr->len + len + 1);
  memcpy(outstr->str + outstr->len, string,len);
  outstr->len += len;
  outstr->str[outstr->len] = '\0';
  return len;
};

static 
void parse_level_graphic(Level *l, xmlNodePtr cur) {
	_Level_graphic *g;
	xmlOutputBufferPtr buf;
	xmlChar *x;
  xmlDocPtr doc;
  level_string outputstring;

	g = new_graphic();

	x = xmlGetProp(cur, (const xmlChar *)"gfxid");
	if(x != NULL)
		g->gfxid = (int32_t)atoi((char *)x);

	x = xmlGetProp(cur, (const xmlChar *)"type");
	if(x != NULL)
		g->type = (int32_t)atoi((char *)x);

	cur = cur->xmlChildrenNode;
  do {
    if (cur->type == XML_ELEMENT_NODE) {
      outputstring.len = 0;
      outputstring.str = NULL;
      buf = xmlOutputBufferCreateIO(level_stringio_write,NULL,
                                            &outputstring,NULL);
      doc = xmlNewDoc((const xmlChar *)"1.0");
      xmlDocSetRootElement(doc,xmlDocCopyNode(cur,doc,1));

      xmlSaveFileTo(buf,doc,"utf-8");

     	g->svgseries = outputstring.str;

      xmlFreeDoc(doc);
      break;
    }
  } while ((cur = cur->next) != NULL);
	list_append(l->graphics, g);
}

/* Level_player */
static int cmp_id(void *player, void *id) {
	if(((Level_player*)player)->id == *((int*)id)) {
		return TRUE;
	}
	return FALSE;
}

Level_player *level_get_player(Level *l, int id) {
	return (Level_player *)list_search(l->players,cmp_id,&id);
}

static Level_player *new_player(void) {
	Level_player *p;

	p = malloc(sizeof(Level_player));
	p->pos = new_vector(0,0);
	p->size = new_vector(0,0);
	p->home_pos = new_vector(0,0);
	p->home_size = new_vector(0,0);

	p->id = -1;
	p->svgseries = NULL;
	p->home_svgseries = NULL;

	return p;
}

static void parse_level_background(Level *l, xmlNodePtr cur) {
	xmlChar *x;

	x = xmlGetProp(cur, (const xmlChar *)"gfxid");
	if(x != NULL)
		l->background = level_get_svgseries_gfxid(l, atoi((char *)x));
}

static void parse_level_player(Level *l, xmlNodePtr cur) {
	Level_player *p;
	xmlChar *x;

	p = new_player();

	p->id = (int32_t)atoi((char *)xmlGetProp(cur, (const xmlChar *)"id"));
	p->pos->x = (int32_t)atoi((char *)xmlGetProp(cur, (const xmlChar *)"x"));
	p->pos->y = (int32_t)atoi((char *)xmlGetProp(cur, (const xmlChar *)"y"));
	p->size->x = (int32_t)atoi((char *)xmlGetProp(cur, (const xmlChar *)"width"));
	p->size->y = (int32_t)atoi((char *)xmlGetProp(cur, (const xmlChar *)"height"));
	p->home_pos->x = (int32_t)atoi((char *)xmlGetProp(cur, (const xmlChar *)"hx"));
	p->home_pos->y = (int32_t)atoi((char *)xmlGetProp(cur, (const xmlChar *)"hy"));
	p->home_size->x = (int32_t)atoi((char *)xmlGetProp(cur, (const xmlChar *)"hwidth"));
	p->home_size->y = (int32_t)atoi((char *)xmlGetProp(cur, (const xmlChar *)"hheight"));

	x = xmlGetProp(cur, (const xmlChar *)"gfxid");
	if(x != NULL)
		p->svgseries = level_get_svgseries_gfxid(l, atoi((char *)x));
	else
		p->svgseries = NULL;

	x = xmlGetProp(cur, (const xmlChar *)"hgfxid");
	if(x != NULL)
		p->home_svgseries = level_get_svgseries_gfxid(l, atoi((char *)x));
	else
		p->home_svgseries = NULL;

	list_append(l->players, p);
	l->numplayers++;
}

/* Level_type */
static int cmp_otype(void *lt, void *id) {
	if(((Level_type*)lt)->id == *((int*)id)) {
		return TRUE;
	}
	return FALSE;
}

Level_type *level_get_type(Level *l, int o_type) {
	return (Level_type *)list_search(l->types,cmp_otype,&o_type);
}

Vector *level_get_type_size(Level *l, int o_type) {
	Level_type *t = level_get_type(l, o_type);
	if(t != NULL) return t->size;
	return NULL;
}

static Level_type *new_leveltype(void) {
	Level_type *t;

	t = malloc(sizeof(Level_type));
	t->size = new_vector(0,0);
	t->id = -1;
	t->svgseries = NULL;

	return t;
}

static void parse_level_type(Level *l, xmlNodePtr cur) {
	Level_type *t = new_leveltype();

	t->id = atoi((char *)xmlGetProp(cur, (const xmlChar *)"id"));
	t->size->x = (int32_t)atoi((char *)xmlGetProp(cur, (const xmlChar *)"width"));
	t->size->y = (int32_t)atoi((char *)xmlGetProp(cur, (const xmlChar *)"height"));

	t->svgseries = level_get_svgseries(l, t->id);

	list_append(l->types, t);
}

/* Level_object */
Level_object *level_next_object(Level *l) {
	if(l->_next_object == NULL) {
		l->_next_object = new_list_ptr(l->objects);
		if(!list_ptr_first(l->_next_object)) return NULL;
	} else {
		if(!list_ptr_next(l->_next_object)) return NULL;
	}
	
	return (Level_object *)list_ptr_get_data(l->_next_object);
}

void level_reset_object(Level *l) {
	if(l != NULL) {
		l->_next_object = NULL;
	}
}

static Level_object *new_levelobject(void) {
	Level_object *lo;

	lo = malloc(sizeof(Level_object));
	lo->pos = new_vector(0,0);
	lo->size = new_vector(0,0);
	
	lo->type = -1;
	lo->svgseries = NULL;
	
	return lo;
}

static void parse_level_object(Level *l, xmlNodePtr cur) {
	Level_object *lo = new_levelobject();
	xmlChar *x;

	lo->type = atoi((char *)xmlGetProp(cur, (const xmlChar *)"type"));
	lo->pos->x = (int32_t)atoi((char *)xmlGetProp(cur, (const xmlChar *)"x"));
	lo->pos->y = (int32_t)atoi((char *)xmlGetProp(cur, (const xmlChar *)"y"));
	lo->size->x = (int32_t)atoi((char *)xmlGetProp(cur, (const xmlChar *)"width"));
	lo->size->y = (int32_t)atoi((char *)xmlGetProp(cur, (const xmlChar *)"height"));

	x = xmlGetProp(cur, (const xmlChar *)"gfxid");
	if(x != NULL)
		lo->svgseries = level_get_svgseries_gfxid(l, atoi((char *)x));
	else
		lo->svgseries = NULL;

	list_append(l->objects, lo);
}

/* constructor */
static Level *_new_level(xmlDocPtr doc) {
	Level *l;
	
	xmlNodePtr cur;

	l = malloc(sizeof(Level));
	l->name = NULL;
	l->background = NULL;
	l->size = new_vector(0,0);
	l->objects = new_list();
	l->settings = new_list();
	l->players = new_list();
	l->graphics = new_list();
	l->types = new_list();
	l->numplayers = 0;
	l->_next_object = NULL;
	l->_next_setting = NULL;

	if (doc == NULL) {
		fprintf(stderr,"Document not parsed successfully. \n");
		del_level(l);
		return NULL;
	}

	cur = xmlDocGetRootElement(doc);
	if (cur == NULL) {
		fprintf(stderr,"empty document\n");
		xmlFreeDoc(doc);
		del_level(l);
		return NULL;
	}

	if (xmlStrcmp(cur->name, (const xmlChar *) "level")) {
		fprintf(stderr,"document of the wrong type, root node != level");
		xmlFreeDoc(doc);
		del_level(l);
		return NULL;
	}

	l->name = (char *)xmlGetProp(cur, (const xmlChar *)"name");
	l->size->x = (int32_t)atoi((char *)xmlGetProp(cur, (const xmlChar *)"width"));
	l->size->y = (int32_t)atoi((char *)xmlGetProp(cur, (const xmlChar *)"height"));

	cur = cur->xmlChildrenNode;
	while (cur != NULL) {
		if ((!xmlStrcmp(cur->name, (const xmlChar *)"graphic"))) {
			parse_level_graphic(l, cur);
		}
		cur = cur->next;
	}
	cur = xmlDocGetRootElement(doc);
	cur = cur->xmlChildrenNode;
	while (cur != NULL) {
		if ((!xmlStrcmp(cur->name, (const xmlChar *)"type"))) {
			parse_level_type(l, cur);
		} else if ((!xmlStrcmp(cur->name, (const xmlChar *)"player"))) {
			parse_level_player(l, cur);
		} else if ((!xmlStrcmp(cur->name, (const xmlChar *)"object"))) {
			parse_level_object(l, cur);
		} else if ((!xmlStrcmp(cur->name, (const xmlChar *)"setting"))) {
			parse_level_setting(l, cur);
		} else if ((!xmlStrcmp(cur->name, (const xmlChar *)"background"))) {
			parse_level_background(l, cur);
		}
		cur = cur->next;
	}

	return l;
}

Level *new_levelfile(char *filename) {
	xmlDocPtr doc = xmlParseFile(filename);
	return _new_level(doc);
}

Level *new_level(char *data) {
	xmlDocPtr doc = xmlParseDoc((xmlChar *)data);
	return _new_level(doc);
}

/* deconstructors */

static void del_type(Level_type *t) {
	del_vector(t->size);
	free(t);
}

static void del_player(Level_player *p) {
	del_vector(p->pos);
	del_vector(p->size);
	del_vector(p->home_pos);
	del_vector(p->home_size);
	free(p);
}

static void del_object(Level_object *o) {
	del_vector(o->pos);
	del_vector(o->size);
	free(o);
}

static void del_setting(Level_setting *s) {
	free(s->name);
	if(s->type == LEVEL_VECTOR)
		del_vector((s->data).vval);
	free(s);
}

static void del_graphic(_Level_graphic *g) {
	if(g->svgseries != NULL) free(g->svgseries);
	free(g);
}

void del_level(Level *l) {
	Level_player *p;
	Level_object *o;
	Level_setting *s;
	Level_type *t;
	_Level_graphic *g;

	if(l==NULL) return;

	if(l->size != NULL) del_vector(l->size);

	if(l->_next_object != NULL) del_list_ptr(l->_next_object);
	if(l->_next_setting != NULL) del_list_ptr(l->_next_setting);

	while((p = (Level_player *)list_pop(l->players)) != NULL)
		del_player(p);

	while((o = (Level_object *)list_pop(l->objects)) != NULL)
		del_object(o);
	
	while((s = (Level_setting *)list_pop(l->settings)) != NULL)
		del_setting(s);
	
	while((t = (Level_type *)list_pop(l->types)) != NULL)
		del_type(t);
	
	while((g = (_Level_graphic *)list_pop(l->graphics)) != NULL)
		del_graphic(g);

	del_list(l->graphics);
	del_list(l->objects);
	del_list(l->settings);
	del_list(l->players);
	del_list(l->types);

	if(l->name != NULL) free(l->name);

	free(l);
}
