<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet
    version="1.0"
    xmlns="http://www.w3.org/1999/xhtml"
    xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:str="http://exslt.org/strings"
    xmlns:func="http://exslt.org/functions"
    xmlns:gsa="http://openvas.org"
    xmlns:gsa-i18n="http://openvas.org/i18n"
    xmlns:vuln="http://scap.nist.gov/schema/vulnerability/0.4"
    xmlns:cpe-lang="http://cpe.mitre.org/language/2.0"
    xmlns:scap-core="http://scap.nist.gov/schema/scap-core/0.1"
    xmlns:cve="http://scap.nist.gov/schema/feed/vulnerability/2.0"
    xmlns:cvss="http://scap.nist.gov/schema/cvss-v2/0.2"
    xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
    xmlns:patch="http://scap.nist.gov/schema/patch/0.1"
    xmlns:meta="http://scap.nist.gov/schema/cpe-dictionary-metadata/0.2"
    xmlns:ns6="http://scap.nist.gov/schema/scap-core/0.1"
    xmlns:config="http://scap.nist.gov/schema/configuration/0.1"
    xmlns:cpe="http://cpe.mitre.org/dictionary/2.0"
    xmlns:oval="http://oval.mitre.org/XMLSchema/oval-common-5"
    xmlns:oval_definitions="http://oval.mitre.org/XMLSchema/oval-definitions-5"
    xmlns:dfncert="http://www.dfn-cert.de/dfncert.dtd"
    xmlns:atom="http://www.w3.org/2005/Atom"
    xsi:schemaLocation="http://scap.nist.gov/schema/configuration/0.1 http://nvd.nist.gov/schema/configuration_0.1.xsd http://scap.nist.gov/schema/scap-core/0.3 http://nvd.nist.gov/schema/scap-core_0.3.xsd http://cpe.mitre.org/dictionary/2.0 http://cpe.mitre.org/files/cpe-dictionary_2.2.xsd http://scap.nist.gov/schema/scap-core/0.1 http://nvd.nist.gov/schema/scap-core_0.1.xsd http://scap.nist.gov/schema/cpe-dictionary-metadata/0.2 http://nvd.nist.gov/schema/cpe-dictionary-metadata_0.2.xsd"
    xmlns:date="http://exslt.org/dates-and-times"
    xmlns:exslt="http://exslt.org/common"
    exclude-result-prefixes="vuln cpe-lang scap-core cve cvss xsi patch meta ns6 config cpe oval oval_definitions dfncert atom"
    extension-element-prefixes="str func date exslt gsa gsa-i18n">
  <xsl:output
      method="html"
      doctype-system="http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd"
      doctype-public="-//W3C//DTD XHTML 1.0 Transitional//EN"
      encoding="UTF-8"/>

<!--
Greenbone Security Assistant
$Id$
Description: OpenVAS Manager Protocol (OMP) stylesheet

Authors:
Matthew Mundell <matthew.mundell@greenbone.net>
Jan-Oliver Wagner <jan-oliver.wagner@greenbone.net>
Michael Wiegand <michael.wiegand@greenbone.net>
Timo Pollmeier <timo.pollmeier@greenbone.net>

Copyright:
Copyright (C) 2009-2015 Greenbone Networks GmbH

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
-->


<!-- BEGIN GLOBAL VARIABLES -->

<xsl:variable name="icon-width" select="19"/>
<xsl:variable name="trash-actions-width" select="3 + (2 * $icon-width)"/>

<!-- BEGIN XPATH FUNCTIONS -->

<func:function name="gsa:envelope-filter">
  <xsl:choose>
    <xsl:when test="string-length (/envelope/params/filter) &gt; 0 and string-length (/envelope/params/filter_extra) &gt; 0">
      <func:result select="concat (/envelope/params/filter, ' ', /envelope/params/filter_extra)"/>
    </xsl:when>
    <xsl:otherwise>
      <func:result select="concat (/envelope/params/filter, /envelope/params/filter_extra)"/>
    </xsl:otherwise>
  </xsl:choose>
</func:function>

<func:function name="gsa:may">
  <xsl:param name="name"/>
  <xsl:param name="permissions" select="permissions"/>
  <func:result select="gsa:may-op ($name) and (boolean ($permissions/permission[name='Everything']) or boolean ($permissions/permission[name=$name]))"/>
</func:function>

<xsl:variable name="capabilities" select="/envelope/capabilities/help_response/schema"/>

<func:function name="gsa:may-op">
  <xsl:param name="name"/>
  <func:result select="boolean ($capabilities/command[gsa:lower-case (name) = gsa:lower-case ($name)])"/>
</func:function>

<func:function name="gsa:may-clone">
  <xsl:param name="type"/>
  <xsl:param name="owner" select="owner"/>
  <func:result select="gsa:may-op (concat ('create_', $type))"/>
</func:function>

<func:function name="gsa:may-get-trash">
  <func:result select="boolean ($capabilities/command[substring (gsa:lower-case (name), 1, 4) = 'get_' and gsa:lower-case (name) != 'get_version' and gsa:lower-case (name) != 'get_info' and gsa:lower-case (name) != 'get_nvts' and gsa:lower-case (name) != 'get_system_reports'  and gsa:lower-case (name) != 'get_settings'])"/>
</func:function>

<func:function name="gsa:build-levels">
  <xsl:param name="filters"></xsl:param>
  <func:result>
    <xsl:for-each select="$filters/filter">
      <xsl:choose>
        <xsl:when test="text()='High'">h</xsl:when>
        <xsl:when test="text()='Medium'">m</xsl:when>
        <xsl:when test="text()='Low'">l</xsl:when>
        <xsl:when test="text()='Log'">g</xsl:when>
        <xsl:when test="text()='False Positive'">f</xsl:when>
      </xsl:choose>
    </xsl:for-each>
  </func:result>
</func:function>

<func:function name="gsa:build-filter">
  <xsl:param name="filters"></xsl:param>
  <xsl:param name="replace"></xsl:param>
  <xsl:param name="with"></xsl:param>

  <func:result>
    <xsl:for-each select="$filters/keywords/keyword">
      <xsl:choose>
        <xsl:when test="column = $replace">
          <xsl:value-of select="$with"/>
          <xsl:text> </xsl:text>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="column"/>
          <xsl:value-of select="relation"/>
          <xsl:value-of select="value"/>
          <xsl:text> </xsl:text>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:for-each>
  </func:result>
</func:function>

<func:function name="gsa:join">
  <xsl:param name="nodes"/>
  <func:result>
    <xsl:for-each select="$nodes">
      <xsl:value-of select="name"/>
      <xsl:text> </xsl:text>
    </xsl:for-each>
  </func:result>
</func:function>

<func:function name="gsa:build-css-classes">
  <xsl:param name="prefix"/>
  <xsl:param name="nodes"/>

  <xsl:variable name="classes" select="exslt:node-set($nodes)/classes"/>

  <func:result>
    <xsl:for-each select="$classes/class">
      <xsl:value-of select="$prefix"/><xsl:value-of select="."/><xsl:text> </xsl:text>
    </xsl:for-each>
  </func:result>
</func:function>

<func:function name="gsa:actions-width">
  <xsl:param name="icon-count"/>
  <func:result select="15 + ($icon-count * $icon-width)"/>
</func:function>

<func:function name="gsa:token">
  <xsl:choose>
    <xsl:when test="string-length (/envelope/params/debug) = 0">
      <func:result select="concat ('&amp;token=', /envelope/token)"/>
    </xsl:when>
    <xsl:otherwise>
      <func:result select="concat ('&amp;token=', /envelope/token, '&amp;debug=', /envelope/params/debug)"/>
    </xsl:otherwise>
  </xsl:choose>
</func:function>

<func:function name="gsa:capitalise">
  <xsl:param name="string"/>
  <func:result select="concat (gsa:upper-case (substring ($string, 1, 1)), substring ($string, 2))"/>
</func:function>

<func:function name="gsa:lower-case">
  <xsl:param name="string"/>
  <func:result select="translate($string, 'ABCDEFGHIJKLMNOPQRSTUVWXYZ', 'abcdefghijklmnopqrstuvwxyz')"/>
</func:function>

<func:function name="gsa:upper-case">
  <xsl:param name="string"/>
  <func:result select="translate($string, 'abcdefghijklmnopqrstuvwxyz', 'ABCDEFGHIJKLMNOPQRSTUVWXYZ')"/>
</func:function>

<func:function name="gsa:escape-js">
  <xsl:param name="string"/>
  <xsl:variable name='apos'>'</xsl:variable>
  <!-- Escape as XML entities where applicable -->
  <func:result select="str:replace (str:replace (str:replace (str:replace (str:replace (str:replace (
                       $string, '&amp;', '&amp;amp;'), '\', '\x2F'), '&quot;', '&amp;quot;'), $apos, '&amp;apos;'), '&lt;', '&amp;lt;'), '&gt;', '&amp;gt;')"/>
</func:function>

<func:function name="gsa:date-tz">
  <xsl:param name="time"></xsl:param>
  <func:result>
    <!-- 2013-03-26T13:15:00-04:00 -->
    <!-- 2013-03-26T13:15:00Z -->
    <!-- 2013-03-26T13:15:00+04:00 -->
    <xsl:variable name="length" select="string-length ($time)"/>
    <xsl:if test="$length &gt; 0">
      <xsl:choose>
        <xsl:when test="substring ($time, $length) = 'Z'">
          <xsl:value-of select="'UTC'"/>
        </xsl:when>
        <xsl:when test="contains ('+-', substring ($time, $length - 5, 1)) and (substring ($time, $length - 2, 1) = ':')">
          <xsl:value-of select="substring ($time, $length - 5)"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="'ERROR'"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:if>
  </func:result>
</func:function>

<func:function name="gsa:long-time-tz">
  <xsl:param name="time"></xsl:param>
  <func:result>
    <xsl:if test="string-length ($time) &gt; 0">
      <xsl:value-of select="concat (date:day-abbreviation ($time), ' ', date:month-abbreviation ($time), ' ', date:day-in-month ($time), ' ', format-number(date:hour-in-day($time), '00'), ':', format-number(date:minute-in-hour($time), '00'), ':', format-number(date:second-in-minute($time), '00'), ' ', date:year($time), ' ', gsa:date-tz($time))"/>
    </xsl:if>
  </func:result>
</func:function>

<func:function name="gsa:long-time">
  <xsl:param name="time"></xsl:param>
  <func:result>
    <xsl:if test="string-length ($time) &gt; 0">
      <xsl:value-of select="concat (date:day-abbreviation ($time), ' ', date:month-abbreviation ($time), ' ', date:day-in-month ($time), ' ', format-number(date:hour-in-day($time), '00'), ':', format-number(date:minute-in-hour($time), '00'), ':', format-number(date:second-in-minute($time), '00'), ' ', date:year($time))"/>
    </xsl:if>
  </func:result>
</func:function>

<func:function name="gsa:date">
  <xsl:param name="datetime"></xsl:param>
  <func:result>
    <xsl:if test="string-length ($datetime) &gt; 0">
      <xsl:value-of select="concat (date:day-abbreviation ($datetime), ' ', date:month-abbreviation ($datetime), ' ', date:day-in-month ($datetime), ' ', date:year($datetime))"/>
    </xsl:if>
  </func:result>
</func:function>

<func:function name="gsa:type-many">
  <xsl:param name="type"></xsl:param>
  <func:result>
    <xsl:choose>
      <xsl:when test="$type = 'info' or $type = 'allinfo'">
        <xsl:value-of select="$type"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$type"/><xsl:text>s</xsl:text>
      </xsl:otherwise>
    </xsl:choose>
  </func:result>
</func:function>

<func:function name="gsa:html-attribute-quote">
  <xsl:param name="text"/>
  <func:result>
    <xsl:value-of select="translate ($text, '&quot;', '&amp;quot;')"/>
  </func:result>
</func:function>

<!-- This is only safe for HTML attributes. -->
<func:function name="gsa:param-or">
  <xsl:param name="name"/>
  <xsl:param name="alternative"/>
  <xsl:choose>
    <xsl:when test="/envelope/params/node()[name()=$name]">
      <func:result>
        <xsl:value-of select="gsa:html-attribute-quote (/envelope/params/node()[name()=$name])"/>
      </func:result>
    </xsl:when>
    <xsl:when test="/envelope/params/_param[name=$name]">
      <func:result>
        <xsl:value-of select="gsa:html-attribute-quote (/envelope/params/_param[name=$name]/value)"/>
      </func:result>
    </xsl:when>
    <xsl:otherwise>
      <func:result>
        <xsl:value-of select="$alternative"/>
      </func:result>
    </xsl:otherwise>
  </xsl:choose>
</func:function>

<func:function name="gsa:get-nvt-tag">
    <xsl:param name="tags"/>
    <xsl:param name="name"/>
  <xsl:variable name="after">
    <xsl:value-of select="substring-after (nvt/tags, concat ($name, '='))"/>
  </xsl:variable>
  <xsl:choose>
      <xsl:when test="contains ($after, '|')">
        <func:result select="substring-before ($after, '|')"/>
      </xsl:when>
      <xsl:otherwise>
        <func:result select="$after"/>
      </xsl:otherwise>
  </xsl:choose>
</func:function>

<func:function name="gsa:cvss-risk-factor">
  <xsl:param name="cvss_score"/>
  <xsl:variable name="type" select="/envelope/severity"/>
  <xsl:variable name="threat">
    <xsl:choose>
      <xsl:when test="$type = 'classic'">
        <xsl:choose>
          <xsl:when test="$cvss_score = 0.0">Log</xsl:when>
          <xsl:when test="$cvss_score &gt;= 0.1 and $cvss_score &lt;= 2.0">Low</xsl:when>
          <xsl:when test="$cvss_score &gt;= 2.1 and $cvss_score &lt;= 5.0">Medium</xsl:when>
          <xsl:when test="$cvss_score &gt;= 5.1 and $cvss_score &lt;= 8.0">High</xsl:when>
          <xsl:when test="$cvss_score &gt;= 8.1 and $cvss_score &lt;= 10.0">High</xsl:when>
          <xsl:otherwise>None</xsl:otherwise>
        </xsl:choose>
      </xsl:when>
      <xsl:when test="$type = 'pci-dss'">
        <xsl:choose>
          <xsl:when test="$cvss_score &gt;= 0.0 and $cvss_score &lt; 4.0">Log</xsl:when>
          <xsl:when test="$cvss_score &gt;= 4.0">High</xsl:when>
          <xsl:otherwise>None</xsl:otherwise>
        </xsl:choose>
      </xsl:when>
      <!-- Default to nist/bsi -->
      <xsl:otherwise>
        <xsl:choose>
          <xsl:when test="$cvss_score = 0.0">Log</xsl:when>
          <xsl:when test="$cvss_score &gt;= 0.1 and $cvss_score &lt;= 3.9">Low</xsl:when>
          <xsl:when test="$cvss_score &gt;= 4.0 and $cvss_score &lt;= 6.9">Medium</xsl:when>
          <xsl:when test="$cvss_score &gt;= 7.0 and $cvss_score &lt;= 10.0">High</xsl:when>
          <xsl:otherwise>None</xsl:otherwise>
        </xsl:choose>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <func:result select="$threat"/>
</func:function>

<func:function name="gsa:result-cvss-risk-factor">
  <xsl:param name="cvss_score"/>
  <xsl:variable name="threat">
    <xsl:choose>
      <xsl:when test="$cvss_score &gt; 0.0">
        <xsl:value-of select="gsa:cvss-risk-factor($cvss_score)"/>
      </xsl:when>
      <xsl:when test="$cvss_score = 0.0">Log</xsl:when>
      <xsl:when test="$cvss_score = -1.0">False Positive</xsl:when>
      <xsl:when test="$cvss_score = -2.0">Debug</xsl:when>
      <xsl:when test="$cvss_score = -3.0">Error</xsl:when>
      <xsl:otherwise>N/A</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <func:result select="$threat"/>
</func:function>

<func:function name="gsa:risk-factor-max-cvss">
  <xsl:param name="threat"/>
  <xsl:param name="type"><xsl:value-of select="/envelope/severity"/></xsl:param>
  <xsl:variable name="cvss">
    <xsl:choose>
      <xsl:when test="$type = 'classic'">
        <xsl:choose>
          <xsl:when test="gsa:lower-case($threat) = 'none' or gsa:lower-case($threat) = 'log'">0.0</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'low'">2.0</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'medium'">5.0</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'high'">10.0</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'critical'">10.0</xsl:when>
          <xsl:otherwise>0.0</xsl:otherwise>
        </xsl:choose>
      </xsl:when>

      <xsl:when test="$type = 'pci-dss'">
        <xsl:choose>
          <xsl:when test="gsa:lower-case($threat) = 'none' or gsa:lower-case($threat) = 'log'">3.9</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'low'">3.9</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'medium'">3.9</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'high'">10.0</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'critical'">10.0</xsl:when>
          <xsl:otherwise>0.0</xsl:otherwise>
        </xsl:choose>
      </xsl:when>
      <!-- Default to nist/bsi -->
      <xsl:otherwise>
        <xsl:choose>
          <xsl:when test="gsa:lower-case($threat) = 'none' or gsa:lower-case($threat) = 'log'">0.0</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'low'">3.9</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'medium'">6.9</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'high'">10.0</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'critical'">10.0</xsl:when>
          <xsl:otherwise>0.0</xsl:otherwise>
        </xsl:choose>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <func:result select="$cvss"/>
</func:function>

<func:function name="gsa:risk-factor-min-cvss">
  <xsl:param name="threat"/>
  <xsl:param name="type"><xsl:value-of select="/envelope/severity"/></xsl:param>
  <xsl:variable name="cvss">
    <xsl:choose>
      <xsl:when test="$type = 'classic'">
        <xsl:choose>
          <xsl:when test="gsa:lower-case($threat) = 'none' or gsa:lower-case($threat) = 'log'">0.0</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'low'">0.1</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'medium'">2.1</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'high'">5.1</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'critical'">10.0</xsl:when>
          <xsl:otherwise>0.0</xsl:otherwise>
        </xsl:choose>
      </xsl:when>

      <xsl:when test="$type = 'pci-dss'">
        <xsl:choose>
          <xsl:when test="gsa:lower-case($threat) = 'none' or gsa:lower-case($threat) = 'none'">0.0</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'low'">3.9</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'medium'">3.9</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'high'">4.0</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'critical'">10.0</xsl:when>
          <xsl:otherwise>0.0</xsl:otherwise>
        </xsl:choose>
      </xsl:when>
      <!-- Default to nist/bsi -->
      <xsl:otherwise>
        <xsl:choose>
          <xsl:when test="gsa:lower-case($threat) = 'none' or gsa:lower-case($threat) = 'log'">0.0</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'low'">0.1</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'medium'">4.0</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'high'">7.0</xsl:when>
          <xsl:when test="gsa:lower-case($threat) = 'critical'">10.0</xsl:when>
          <xsl:otherwise>0.0</xsl:otherwise>
        </xsl:choose>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <func:result select="$cvss"/>
</func:function>

<func:function name="gsa:threat-color">
  <xsl:param name="threat"/>
  <xsl:variable name="color">
    <xsl:choose>
      <xsl:when test="gsa:lower-case($threat) = 'high'">red</xsl:when>
      <xsl:when test="gsa:lower-case($threat) = 'medium'">orange</xsl:when>
      <xsl:when test="gsa:lower-case($threat) = 'low'">lightskyblue</xsl:when>
      <xsl:when test="gsa:lower-case($threat) = 'none' or gsa:lower-case($threat) = 'log'">silver</xsl:when>
      <xsl:otherwise></xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <func:result select="$color"/>
</func:function>

<func:function name="gsa:column-filter-name">
  <xsl:param name="type"/>
  <func:result select="str:replace (str:replace (gsa:lower-case ($type), '&#xa0;', '_'), ' ', '_')"/>
</func:function>

<func:function name="gsa:type-string">
  <xsl:param name="type"/>
  <func:result select="str:replace (gsa:lower-case ($type), ' ', '_')"/>
</func:function>

<func:function name="gsa:command-type-plural">
  <xsl:param name="command"/>
  <xsl:variable name="type"
                select="gsa:command-type ($command)"/>
  <xsl:choose>
    <xsl:when test="$type = 'NVT family'">
      <func:result select="'NVT families'"/>
    </xsl:when>
    <xsl:otherwise>
      <func:result select="concat ($type, 's')"/>
    </xsl:otherwise>
  </xsl:choose>
</func:function>

<func:function name="gsa:command-type">
  <xsl:param name="command"/>
  <xsl:variable name="after"
                select="substring-after (str:replace (gsa:lower-case ($command), '_', ' '), ' ')"/>
  <xsl:variable name="type">
    <xsl:choose>
      <xsl:when test="substring ($after, string-length ($after)) = 's'">
        <xsl:value-of select="substring ($after, 1, string-length ($after) - 1)"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$after"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:choose>
    <xsl:when test="$type = 'lsc credential'">
      <func:result select="'credential'"/>
    </xsl:when>
    <xsl:when test="$type = 'config'">
      <func:result select="'scan config'"/>
    </xsl:when>
    <xsl:when test="$type = 'nvt'">
      <func:result select="'NVT'"/>
    </xsl:when>
    <xsl:when test="$type = 'nvt familie'">
      <func:result select="'NVT family'"/>
    </xsl:when>
    <xsl:otherwise>
      <func:result select="$type"/>
    </xsl:otherwise>
  </xsl:choose>
</func:function>

<func:function name="gsa:join-capital">
  <xsl:param name="nodes"/>
  <func:result>
    <xsl:for-each select="$nodes">
      <xsl:value-of select="gsa:capitalise (text ())"/>
      <xsl:if test="position() != last()">
        <xsl:text> </xsl:text>
      </xsl:if>
    </xsl:for-each>
  </func:result>
</func:function>

<func:function name="gsa:command-type-label">
  <xsl:param name="command"/>
  <func:result select="gsa:capitalise (gsa:command-type ($command))"/>
</func:function>

<func:function name="gsa:type-name">
  <xsl:param name="type"/>
  <xsl:choose>
    <xsl:when test="$type = 'nvt' or $type = 'cve' or $type = 'cpe'">
      <func:result select="gsa:upper-case ($type)"/>
    </xsl:when>
    <xsl:when test="$type = 'os'">
      <func:result select="'Operating System'"/>
    </xsl:when>
    <xsl:when test="$type = 'ovaldef'">
      <func:result select="'OVAL Definition'"/>
    </xsl:when>
    <xsl:when test="$type = 'cert_bund_adv'">
      <func:result select="'CERT-Bund Advisory'"/>
    </xsl:when>
    <xsl:when test="$type = 'dfn_cert_adv'">
      <func:result select="'DFN-CERT Advisory'"/>
    </xsl:when>
    <xsl:when test="$type = 'allinfo'">
      <func:result select="'All SecInfo'"/>
    </xsl:when>
    <xsl:otherwise>
      <func:result select="gsa:join-capital (str:split ($type, '_'))"/>
    </xsl:otherwise>
  </xsl:choose>
</func:function>

<func:function name="gsa:type-name-plural">
  <xsl:param name="type"/>
  <xsl:choose>
    <xsl:when test="$type = 'cert_bund_adv'">
      <func:result select="'CERT-Bund Advisories'"/>
    </xsl:when>
    <xsl:when test="$type = 'dfn_cert_adv'">
      <func:result select="'DFN-CERT Advisories'"/>
    </xsl:when>
    <xsl:when test="$type = 'allinfo'">
      <func:result select="'All SecInfo'"/>
    </xsl:when>
    <xsl:otherwise>
      <func:result select="concat(gsa:type-name ($type), 's')"/>
    </xsl:otherwise>
  </xsl:choose>
</func:function>

<func:function name="gsa:field-name">
  <xsl:param name="field"/>
  <xsl:choose>
    <xsl:when test="$field = 'created'">
      <func:result select="'creation time'"/>
    </xsl:when>
    <xsl:when test="$field = 'modified'">
      <func:result select="'modification time'"/>
    </xsl:when>
    <xsl:when test="$field = 'qod'">
      <func:result select="'QoD'"/>
    </xsl:when>
    <xsl:when test="$field = 'qod_type'">
      <func:result select="'QoD type'"/>
    </xsl:when>
    <xsl:otherwise>
      <func:result select="translate ($field, '_', ' ')"/>
    </xsl:otherwise>
  </xsl:choose>
</func:function>

<func:function name="gsa:alert-in-trash">
  <xsl:for-each select="alert">
    <xsl:if test="trash/text() != '0'">
      <func:result>1</func:result>
    </xsl:if>
  </xsl:for-each>
  <func:result>0</func:result>
</func:function>

<func:function name="gsa:table-row-class">
  <xsl:param name="position"/>
  <func:result>
    <xsl:choose>
      <xsl:when test="$position &lt; 0"></xsl:when>
      <xsl:when test="$position mod 2 = 0">even</xsl:when>
      <xsl:otherwise>odd</xsl:otherwise>
    </xsl:choose>
  </func:result>
</func:function>

<func:function name="gsa:date-diff-text">
  <xsl:param name="difference"/>

  <xsl:variable name="fromepoch"
                select="date:add ('1970-01-01T00:00:00Z', $difference)"/>
  <xsl:variable name="seconds"
                select="date:second-in-minute($fromepoch)"/>
  <xsl:variable name="minutes"
                select="date:minute-in-hour($fromepoch)"/>
  <xsl:variable name="hours"
                select="date:hour-in-day($fromepoch)"/>
  <xsl:variable name="days"
                select="date:day-in-year($fromepoch) - 1"/>

  <func:result>
      <xsl:if test="$days">
          <xsl:value-of select="concat (gsa-i18n:strformat (gsa:n-i18n ('%1 day', '%1 days', $days, ''), $days), ' ')"/>
      </xsl:if>
      <xsl:if test="$hours">
          <xsl:value-of select="concat (gsa-i18n:strformat (gsa:n-i18n ('%1 hour', '%1 hours', $hours, ''), $hours), ' ')"/>
      </xsl:if>
      <xsl:if test="$minutes">
          <xsl:value-of select="concat (gsa-i18n:strformat (gsa:n-i18n ('%1 minute', '%1 minutes', $minutes, ''), $minutes), ' ')"/>
      </xsl:if>
      <xsl:if test="$seconds">
          <xsl:value-of select="concat (gsa-i18n:strformat (gsa:n-i18n ('%1 second', '%1 seconds', $seconds, ''), $seconds), ' ')"/>
      </xsl:if>
  </func:result>
</func:function>

<func:function name="gsa:date-diff">
  <xsl:param name="start"/>
  <xsl:param name="end"/>

  <xsl:variable name="difference" select="date:difference ($start, $end)"/>
  <func:result>
    <xsl:value-of select="gsa:date-diff-text ($difference)"/>
  </func:result>
</func:function>

<func:function name="gsa:report-host-has-os">
  <xsl:param name="report"/>
  <xsl:param name="ip"/>
  <xsl:param name="os"/>
  <func:result>
    <xsl:choose>
      <xsl:when test="$report/host[ip = $ip and detail/name = 'best_os_cpe' and detail/value = $os]">1</xsl:when>
      <xsl:otherwise>0</xsl:otherwise>
    </xsl:choose>
  </func:result>
</func:function>

<func:function name="gsa:host-has-unknown-os">
  <xsl:param name="report"/>
  <xsl:param name="ip"/>
  <func:result>
    <xsl:choose>
      <xsl:when test="$report/host[ip = $ip and ((detail/name = 'best_os_cpe') = 0)]">1</xsl:when>
      <xsl:otherwise>0</xsl:otherwise>
    </xsl:choose>
  </func:result>
</func:function>

<func:function name="gsa:report-section-title">
  <xsl:param name="section"/>
  <xsl:param name="type"/>
  <func:result>
    <xsl:choose>
      <xsl:when test="$section = 'results' and $type = 'prognostic'"><xsl:value-of select="gsa:i18n ('Report: Prognostic Results')"/></xsl:when>
      <xsl:when test="$section = 'results' and $type = 'delta'"><xsl:value-of select="gsa:i18n ('Report: Delta Results')"/></xsl:when>
      <xsl:when test="$section = 'results'"><xsl:value-of select="gsa:i18n ('Report: Results')"/></xsl:when>
      <xsl:when test="$section = 'summary' and $type = 'prognostic'"><xsl:value-of select="gsa:i18n ('Report: Prognostic Summary and Download')"/></xsl:when>
      <xsl:when test="$section = 'summary' and $type = 'delta'"><xsl:value-of select="gsa:i18n ('Report: Delta Summary and Download')"/></xsl:when>
      <xsl:when test="$section = 'summary'"><xsl:value-of select="gsa:i18n ('Report: Summary and Download')"/></xsl:when>
      <xsl:when test="$section = 'hosts' and $type = 'prognostic'"><xsl:value-of select="gsa:i18n ('Report: Prognostic Hosts')"/></xsl:when>
      <xsl:when test="$section = 'hosts'"><xsl:value-of select="gsa:i18n ('Report: Hosts')"/></xsl:when>
      <xsl:when test="$section = 'ports'"><xsl:value-of select="gsa:i18n ('Report: Ports')"/></xsl:when>
      <xsl:when test="$section = 'os'"><xsl:value-of select="gsa:i18n ('Report: Operating Systems')"/></xsl:when>
      <xsl:when test="$section = 'apps' and $type = 'prognostic'"><xsl:value-of select="gsa:i18n ('Report: Prognostic Applications')"/></xsl:when>
      <xsl:when test="$section = 'apps'"><xsl:value-of select="gsa:i18n ('Report: Applications')"/></xsl:when>
      <xsl:when test="$section = 'vulns'"><xsl:value-of select="gsa:i18n ('Report: Vulnerabilities')"/></xsl:when>
      <xsl:when test="$section = 'cves'"><xsl:value-of select="gsa:i18n ('Report: CVEs')"/></xsl:when>
      <xsl:when test="$section = 'closed_cves'"><xsl:value-of select="gsa:i18n ('Report: Closed CVEs')"/></xsl:when>
      <xsl:when test="$section = 'topology'"><xsl:value-of select="gsa:i18n ('Report: Topology')"/></xsl:when>
      <xsl:when test="$section = 'ssl_certs'"><xsl:value-of select="gsa:i18n ('Report: SSL Certificates')"/></xsl:when>
      <xsl:when test="$section = 'errors'"><xsl:value-of select="gsa:i18n ('Report: Error Messages')"/></xsl:when>
    </xsl:choose>
  </func:result>
</func:function>

<func:function name="gsa:has-long-word">
  <xsl:param name="string"/>
  <xsl:param name="max" select="44"/>
  <func:result select="count (str:split ($string, ' ')[string-length (.) &gt; $max]) &gt; 0"/>
</func:function>

<func:function name="gsa:permission-description">
  <xsl:param name="name"/>
  <xsl:param name="resource"/>
  <xsl:variable name="lower" select="gsa:lower-case ($name)"/>
  <xsl:variable name="has-resource" select="boolean ($resource) and string-length ($resource/type) &gt; 0"/>
  <func:result>
    <xsl:choose>
      <xsl:when test="$has-resource and $lower = 'super'">
        <xsl:value-of select="gsa:i18n ('has super access to ')"/>
        <xsl:value-of select="gsa:i18n (gsa:command-type ($lower), 'Type Lower')"/>
        <xsl:text> </xsl:text>
        <xsl:value-of select="$resource/type"/>
        <xsl:text> </xsl:text>
        <xsl:value-of select="$resource/name"/>
      </xsl:when>
      <xsl:when test="$lower = 'super'">
        <xsl:value-of select="gsa:i18n ('has super access to all users')"/>
      </xsl:when>
      <xsl:when test="$lower = 'authenticate'">
        <xsl:value-of select="gsa:i18n ('may login')"/>
      </xsl:when>
      <xsl:when test="$lower = 'commands'">
        <xsl:value-of select="gsa:i18n ('may run multiple OMP commands in one')"/>
      </xsl:when>
      <xsl:when test="$lower = 'everything'">
        <xsl:value-of select="gsa:i18n ('has all permissions')"/>
      </xsl:when>
      <xsl:when test="$lower = 'empty_trashcan'">
        <xsl:value-of select="gsa:i18n ('may empty the trashcan')"/>
      </xsl:when>
      <xsl:when test="$lower = 'get_dependencies'">
        <xsl:value-of select="gsa:i18n ('may get the dependencies of NVTs')"/>
      </xsl:when>
      <xsl:when test="$lower = 'get_version'">
        <xsl:value-of select="gsa:i18n ('may get version information')"/>
      </xsl:when>
      <xsl:when test="$lower = 'help'">
        <xsl:value-of select="gsa:i18n ('may get the help text')"/>
      </xsl:when>
      <xsl:when test="$lower = 'modify_auth'">
        <xsl:value-of select="gsa:i18n ('has write access to the authentication configuration')"/>
      </xsl:when>
      <xsl:when test="$lower = 'restore'">
        <xsl:value-of select="gsa:i18n ('may restore items from the trashcan')"/>
      </xsl:when>
      <!-- i18n with concat : see dynamic_strings.xsl - permission-descriptions -->
      <xsl:when test="substring-before ($lower, '_') = 'create'">
        <xsl:value-of select="gsa:i18n (concat ('may create a new ', gsa:command-type ($lower)))"/>
      </xsl:when>
      <xsl:when test="$lower = 'get_info'">
        <xsl:value-of select="gsa:i18n ('has read access to SecInfo')"/>
      </xsl:when>
      <xsl:when test="$has-resource and substring-before ($lower, '_') = 'delete'">
        <xsl:value-of select="gsa-i18n:strformat (gsa:i18n (concat ('may delete ', gsa:command-type ($lower), ' %1')), $resource/name)"/>
      </xsl:when>
      <xsl:when test="substring-before ($lower, '_') = 'delete'">
        <xsl:value-of select="gsa:i18n (concat ('may delete an existing ', gsa:command-type ($lower)))"/>
      </xsl:when>
      <xsl:when test="$has-resource and substring-before ($lower, '_') = 'get'">
        <xsl:value-of select="gsa-i18n:strformat (gsa:i18n (concat ('has read access to ', gsa:command-type ($lower), ' %1')), $resource/name)"/>
      </xsl:when>
      <xsl:when test="substring-before ($lower, '_') = 'get'">
        <xsl:value-of select="gsa:i18n (concat ('has read access to ', gsa:command-type-plural ($lower)))"/>
      </xsl:when>
      <xsl:when test="$has-resource and substring-before ($lower, '_') = 'modify'">
        <xsl:value-of select="gsa-i18n:strformat (gsa:i18n (concat ('has write access to ', gsa:command-type ($lower), ' %1')), $resource/name)"/>
      </xsl:when>
      <xsl:when test="substring-before ($lower, '_') = 'modify'">
        <xsl:value-of select="gsa:i18n (concat ('has write access to ', gsa:command-type-plural ($lower)))"/>
      </xsl:when>

      <xsl:when test="substring-before ($lower, '_') = 'describe'">
        <xsl:variable name="described" select="substring-after ($lower, '_')"/>
        <xsl:variable name="text">
          <xsl:choose>
            <xsl:when test="$described = 'auth'">
              <xsl:value-of select="gsa:i18n ('may get details about the authentication configuration')"/>
            </xsl:when>
            <xsl:otherwise>
              <!-- This should only be a fallback for unexpected output -->
              <xsl:value-of select="gsa-i18n:strformat (gsa:i18n ('may get details about %1') , $described)"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:variable>
        <xsl:value-of select="$text"/>
      </xsl:when>

      <xsl:when test="substring-before ($lower, '_') = 'sync'">
        <xsl:variable name="to_sync" select="substring-after ($lower, '_')"/>
        <xsl:variable name="text">
          <xsl:choose>
            <xsl:when test="$to_sync = 'cert'">
              <xsl:value-of select="gsa:i18n ('may sync the CERT feed')"/>
            </xsl:when>
            <xsl:when test="$to_sync = 'feed'">
              <xsl:value-of select="gsa:i18n ('may sync the NVT feed')"/>
            </xsl:when>
            <xsl:when test="$to_sync = 'scap'">
              <xsl:value-of select="gsa:i18n ('may sync the SCAP feed')"/>
            </xsl:when>
            <xsl:otherwise>
              <!-- This should only be a fallback for unexpected output -->
              <xsl:value-of select="gsa-i18n:strformat (gsa:i18n ('may sync %1'), $to_sync)"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:variable>
        <xsl:value-of select="$text"/>
      </xsl:when>

      <xsl:when test="contains ($lower, '_')">
        <!-- see dynamic_strings.xsl - permission-descriptions (verify_...) -->
        <xsl:value-of select="gsa:i18n (concat ('may ', substring-before ($lower, '_'), ' ', gsa:command-type-plural ($lower)))"/>
      </xsl:when>
      <xsl:otherwise><xsl:value-of select="$lower"/></xsl:otherwise>
    </xsl:choose>
  </func:result>
</func:function>

<func:function name="gsa:view_details_title">
  <xsl:param name="type"/>
  <xsl:param name="name"/>
  <xsl:variable name="cap_type" select="gsa:type-name($type)"/>
  <func:result>
    <!-- i18n with concat : see dynamic_strings.xsl - type-details-long -->
    <xsl:value-of select="gsa-i18n:strformat (gsa:i18n (concat ('View details of ', $cap_type, ' %1')), $name)"/>
  </func:result>
</func:function>

<func:function name="gsa:is_absolute_path">
  <xsl:param name="path"/>

  <xsl:variable name="first"
    select="substring ($path, 1, 1)"/>

  <xsl:choose>
    <xsl:when test="$first = '/'">
      <func:result select="true()"/>
    </xsl:when>
    <xsl:otherwise>
      <func:result select="false()"/>
    </xsl:otherwise>
  </xsl:choose>
</func:function>

<func:function name="gsa:column_is_extra">
  <xsl:param name="column"/>
  <xsl:choose>
    <xsl:when test="$column = 'apply_overrides' or $column = 'autofp' or $column = 'rows' or $column = 'first' or $column = 'sort' or $column = 'sort-reverse' or $column = 'notes' or $column = 'overrides' or $column = 'timezone' or $column = 'result_hosts_only' or $column = 'levels' or $column = 'min_qod' or $column = 'delta_states'">
      <func:result select="true()"/>
    </xsl:when>
    <xsl:otherwise>
      <func:result select="false()"/>
    </xsl:otherwise>
  </xsl:choose>
</func:function>

<!-- BEGIN NAMED TEMPLATES -->

<xsl:template name="shy-long-rest">
  <xsl:param name="string"/>
  <xsl:param name="max" select="44"/>
  <xsl:param name="chunk" select="5"/>
  <xsl:text disable-output-escaping="yes">&amp;shy;</xsl:text>
  <xsl:value-of select="substring ($string, 1, $chunk)"/>
  <xsl:if test="string-length ($string) &gt; $chunk">
    <xsl:call-template name="shy-long-rest">
      <xsl:with-param name="string"
                      select="substring ($string, $chunk + 1)"/>
    </xsl:call-template>
  </xsl:if>
</xsl:template>

<xsl:template name="shy-long-words">
  <xsl:param name="string"/>
  <xsl:param name="max" select="44"/>
  <xsl:param name="chunk" select="5"/>
  <xsl:for-each select="str:split ($string, ' ')">
    <xsl:choose>
      <xsl:when test="string-length (.) &gt; $max">
        <xsl:value-of select="substring (., 1, $chunk)"/>
        <xsl:call-template name="shy-long-rest">
          <xsl:with-param name="string"
                          select="substring (., $chunk + 1)"/>
          <xsl:with-param name="chunk" select="$chunk"/>
        </xsl:call-template>
        <xsl:text> </xsl:text>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="."/>
        <xsl:text> </xsl:text>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:for-each>
</xsl:template>

<!-- Currently only a very simple formatting method to produce
     nice HTML from a structured text:
     - create paragraphs for each text block separated with a empty line
-->
<xsl:template name="structured-text">
  <xsl:param name="string"/>

  <xsl:for-each select="str:split($string, '&#10;&#10;')">
    <p>
      <xsl:value-of select="."/>
    </p>
  </xsl:for-each>
</xsl:template>

<xsl:template name="prognostic-description">
  <xsl:param name="string"/>

  <xsl:for-each select="str:split($string, '&#10;&#10;')">
    <p>
      <xsl:for-each select="str:split(., '&#10;')">
        <xsl:value-of select="."/>
        <br/>
      </xsl:for-each>
    </p>
  </xsl:for-each>
</xsl:template>

<xsl:template name="feedback-icon">
<!-- You may fill in here to_name and to_adress and un-comment the block
     to enable a feedback button for support or similar purposes. -->
<!--
  <xsl:param name="to_name" select="'FILL IN NAME'"/>
  <xsl:param name="to_address" select="'FILL IN EMAIL ADDRESS'"/>
  <xsl:param name="subject" select="'Feedback'"/>
  <xsl:param name="body" select="'Dear%20{str:encode-uri ($to_name, true ())},&#xA;&#xA;'"/>
  <a class="icon icon-sm" href="mailto:{str:encode-uri ($to_name, true ())}%20%3C{str:encode-uri ($to_address, true ())}%3E?subject={str:encode-uri ($subject, true ())}&amp;body=Dear%20{str:encode-uri ($to_name, true ())},&#xA;&#xA;{str:encode-uri ($body, true ())}">
    <img src="img/feedback.svg" title="{gsa:i18n ('Send feedback to')} {$to_name}" alt="{gsa:i18n('Feedback')}"/>
  </a>
-->
</xsl:template>

<xsl:template name="filter-window-pager">
  <xsl:param name="type"/>
  <xsl:param name="list"/>
  <xsl:param name="count"/>
  <xsl:param name="filtered_count"/>
  <xsl:param name="full_count"/>
  <xsl:param name="extra_params"/>

  <xsl:variable name="get_cmd">
    <xsl:choose>
      <xsl:when test="$type='report_result'">
        <xsl:value-of select="'get_report_section'"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="concat ('get_', gsa:type-many($type))"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:choose>
    <xsl:when test="$count &gt; 0">
      <xsl:variable name="last" select="$list/@start + $count - 1"/>

      <!-- Table has rows. -->

      <div class="pager">

        <!-- Left icons. -->
        <div class="pagination pagination-left">
          <xsl:choose>
            <xsl:when test = "$list/@start &gt; 1">
              <a href="?cmd={$get_cmd}{$extra_params}&amp;filter=first=1 rows={$list/@max} {filters/term}&amp;token={/envelope/token}"
                class="icon icon-sm">
                <img src="/img/first.svg" title="{gsa:i18n ('First', 'Pagination')}"/></a>
            </xsl:when>
            <xsl:otherwise>
              <img class="icon icon-sm" src="/img/first_inactive.svg" title="{gsa:i18n ('Already on first page', 'Pagination')}"/>
            </xsl:otherwise>
          </xsl:choose>
          <xsl:choose>
            <xsl:when test="$list/@start > $list/@max and $list/@max &gt; 0">
              <a href="?cmd={$get_cmd}{$extra_params}&amp;filter=first={$list/@start - $list/@max} rows={$list/@max} {filters/term}&amp;token={/envelope/token}"
                class="icon icon-sm">
                <img src="/img/previous.svg" title="{gsa:i18n ('Previous', 'Pagination')}"/></a>
            </xsl:when>
            <xsl:when test="$list/@start &gt; 1 and $list/@max &gt; 0">
              <a href="?cmd={$get_cmd}{$extra_params}&amp;filter=first=1 rows={$list/@max} {filters/term}&amp;token={/envelope/token}"
                class="icon icon-sm">
                <img src="/img/previous.svg" title="{gsa:i18n ('Previous', 'Pagination')}"/></a>
            </xsl:when>
            <xsl:otherwise>
              <img class="icon icon-sm" src="/img/previous_inactive.svg" title="{gsa:i18n ('Already on first page', 'Pagination')}"/>
            </xsl:otherwise>
          </xsl:choose>
        </div>

        <!-- Text. -->
        <div class="pagination pagination-text">
          <xsl:value-of select="gsa-i18n:strformat (gsa:i18n ('%1 - %2 of %3'), $list/@start, $last, $filtered_count)"/>
        </div>

        <!-- Right icons. -->
        <div class="pagination pagination-right">
          <xsl:choose>
            <xsl:when test = "$last &lt; $filtered_count">
              <a href="?cmd={$get_cmd}{$extra_params}&amp;filter=first={$list/@start + $list/@max} rows={$list/@max} {filters/term}&amp;token={/envelope/token}"
                class="icon icon-sm">
                <img src="/img/next.svg" title="{gsa:i18n ('Next', 'Pagination')}"/></a>
            </xsl:when>
            <xsl:otherwise>
              <img class="icon icon-sm" src="/img/next_inactive.svg" title="{gsa:i18n ('Already on last page', 'Pagination')}"/>
            </xsl:otherwise>
          </xsl:choose>
          <xsl:choose>
            <xsl:when test = "$last &lt; $filtered_count">
              <a href="?cmd={$get_cmd}{$extra_params}&amp;filter=first={floor(($filtered_count - 1) div $list/@max) * $list/@max + 1} rows={$list/@max} {filters/term}&amp;token={/envelope/token}"
                class="icon icon-sm">
                <img src="/img/last.svg" title="{gsa:i18n ('Last', 'Pagination')}"/></a>
            </xsl:when>
            <xsl:otherwise>
              <img class="icon icon-sm" src="/img/last_inactive.svg" title="{gsa:i18n ('Already on last page', 'Pagination')}"/>
            </xsl:otherwise>
          </xsl:choose>
        </div>
      </div>
    </xsl:when>
  </xsl:choose>
</xsl:template>

<xsl:template name="filter-criteria">
  <xsl:variable name="operator_count" select="count (filters/keywords/keyword[column='' and (value='and' or value='not' or value='or')])"/>
  <xsl:for-each select="filters/keywords/keyword[not (gsa:column_is_extra (column) or (column = 'task_id' and $operator_count = 0))]">
    <xsl:value-of select="column"/>
    <xsl:choose>
      <xsl:when test="column = '' and relation != '='">
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="relation"/>
      </xsl:otherwise>
    </xsl:choose>
    <xsl:if test="boolean (quoted)">"</xsl:if>
    <xsl:value-of select="value"/>
    <xsl:if test="boolean (quoted)">"</xsl:if>
    <xsl:text> </xsl:text>
  </xsl:for-each>
</xsl:template>

<xsl:template name="filter-extra">
  <xsl:variable name="operator_count" select="count (filters/keywords/keyword[column='' and (value='and' or value='not' or value='or')])"/>
  <xsl:for-each select="filters/keywords/keyword[gsa:column_is_extra (column) or (column = 'task_id' and $operator_count = 0)]">
    <xsl:value-of select="column"/>
    <xsl:choose>
      <xsl:when test="column = '' and relation != '='">
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="relation"/>
      </xsl:otherwise>
    </xsl:choose>
    <xsl:if test="boolean (quoted)">"</xsl:if>
    <xsl:value-of select="value"/>
    <xsl:if test="boolean (quoted)">"</xsl:if>
    <xsl:text> </xsl:text>
  </xsl:for-each>
</xsl:template>

<xsl:template name="filter-window-part">
  <xsl:param name="type"/>
  <xsl:param name="list"/>
  <xsl:param name="extra_params"/>
  <xsl:param name="columns"/>
  <xsl:param name="filter_options" select="''"/>
  <xsl:param name="filters" select="../filters"/>
  <xsl:param name="full-count" select="1"/>

  <xsl:variable name="filter_options_nodes" select="exslt:node-set($filter_options)"/>

  <xsl:variable name="criteria">
    <xsl:call-template name="filter-criteria"/>
  </xsl:variable>
  <xsl:variable name="extra">
    <xsl:call-template name="filter-extra"/>
  </xsl:variable>
  <xsl:variable name="get_cmd">
    <xsl:choose>
      <xsl:when test="$type='report_result'">
        <xsl:value-of select="'get_report_section'"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="concat ('get_', gsa:type-many($type))"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>

  <xsl:variable name="extra_params_string">
    <xsl:for-each select="exslt:node-set($extra_params)/param">
      <xsl:text>&amp;</xsl:text>
      <xsl:value-of select="name"/>
      <xsl:text>=</xsl:text>
      <xsl:value-of select="value"/>
    </xsl:for-each>
  </xsl:variable>

  <xsl:variable name="max">
    <xsl:choose>
      <xsl:when test="$full-count&lt;1">
        <xsl:value-of select="1"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$full-count"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>

  <xsl:variable name="min_qod_value">
    <xsl:choose>
      <xsl:when test="not (filters/keywords/keyword[column = 'min_qod']/value != '')">
        <xsl:value-of select="70"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="filters/keywords/keyword[column = 'min_qod']/value"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>

  <div>
    <form class="form-inline" action="" method="get" enctype="multipart/form-data" name="filterform">
      <input type="hidden" name="token" value="{/envelope/token}"/>
      <input type="hidden" name="cmd" value="{$get_cmd}"/>
      <xsl:for-each select="exslt:node-set($extra_params)/param">
        <input type="hidden" name="{name}" value="{value}"/>
      </xsl:for-each>
      <div class="form-group">
        <label for="filtername" class="control-label">
          <b><xsl:value-of select="gsa:i18n ('Filter')"/></b>:
        </label>
        <input type="text" name="filter" size="53"
          id="filtername"
          class="form-control"
          value="{$criteria}"
          maxlength="1000"/>
        <input type="image"
          name="Update Filter"
          class="icon icon-sm"
          title="{gsa:i18n ('Update Filter')}"
          src="/img/refresh.svg"
          alt="{gsa:i18n ('Update', 'Action Verb')}"/>
        <a href="?token={/envelope/token}&amp;cmd={$get_cmd}&amp;filt_id=--{$extra_params_string}"
          class="icon icon-sm"
          title="{gsa:i18n ('Reset Filter')}">
          <img src="/img/delete.svg" />
        </a>
        <a href="/help/powerfilter.html?token={/envelope/token}"
          class="icon icon-sm"
          title="{gsa:i18n ('Help')}: {gsa:i18n ('Powerfilter')}">
          <img src="/img/help.svg" />
        </a>
        <a href="#" class="icon icon-sm edit-filter-action-icon" data-id="filterbox">
          <img src="/img/edit.svg"/>
        </a>
        <xsl:variable name="extras">
          <xsl:for-each select="exslt:node-set($extra_params)/param">
            <xsl:value-of select="concat ('&amp;', name, '=', value)"/>
          </xsl:for-each>
        </xsl:variable>
        <input type="hidden" name="filter_extra" value="{$extra}" />
      </div>
      <div class="footnote">
        <xsl:value-of select="$extra"/>
      </div>
    </form>
  </div>
  <xsl:if test="gsa:may-op ('create_filter')">
    <div>
      <form class="form-inline" action="" method="post" enctype="multipart/form-data">
        <div class="form-group">
          <input type="hidden" name="token" value="{/envelope/token}"/>
          <input type="hidden" name="cmd" value="create_filter"/>
          <input type="hidden" name="caller" value="{/envelope/current_page}"/>
          <input type="hidden" name="comment" value=""/>
          <input type="hidden" name="term" value="{filters/term}"/>
          <xsl:choose>
            <xsl:when test="$type = 'report_result'">
              <input type="hidden" name="optional_resource_type" value="result"/>
              <input type="hidden" name="next" value="get_report_section"/>
            </xsl:when>
            <xsl:otherwise>
              <input type="hidden" name="optional_resource_type" value="{$type}"/>
              <input type="hidden" name="next" value="get_{gsa:type-many($type)}"/>
            </xsl:otherwise>
          </xsl:choose>
          <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
          <xsl:for-each select="exslt:node-set($extra_params)/param">
            <input type="hidden" name="{name}" value="{value}"/>
          </xsl:for-each>
          <input type="text" name="name" value="" size="10"
            class="form-control"
            maxlength="80"/>

          <xsl:variable name="type-name">
            <xsl:choose>
              <xsl:when test="$type = 'report_result'">
                <xsl:value-of select="Result"/>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="gsa:type-name ($type)"/>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:variable>
          <!-- i18n with concat : see dynamic_strings.xsl - type-new-filter -->
          <input type="image"
            name="New Filter"
            class="icon icon-sm"
            src="/img/new.svg"
            alt="{gsa:i18n ('New Filter')}"
            title="{gsa:i18n (concat ('New ', $type-name, ' Filter from current term'))}" />
        </div>
      </form>
    </div>
  </xsl:if>
  <xsl:if test="gsa:may-op ('get_filters')">
    <div>
      <form class="form-inline" action="" method="get" name="switch_filter" enctype="multipart/form-data">
        <div class="form-group">
          <input type="hidden" name="token" value="{/envelope/token}"/>
          <xsl:choose>
            <xsl:when test="$type = 'report_result'">
              <input type="hidden" name="cmd" value="get_report_section"/>
            </xsl:when>
            <xsl:otherwise>
              <input type="hidden" name="cmd" value="get_{gsa:type-many($type)}"/>
            </xsl:otherwise>
          </xsl:choose>
          <xsl:for-each select="exslt:node-set($extra_params)/param">
            <input type="hidden" name="{name}" value="{value}"/>
          </xsl:for-each>
          <select style="margin-bottom: 0px; max-width: 100px;" name="filt_id" onchange="switch_filter.submit()">
            <option value="--">--</option>
            <xsl:variable name="id" select="filters/@id"/>
            <xsl:for-each select="$filters/get_filters_response/filter">
              <xsl:choose>
                <xsl:when test="@id = $id">
                  <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
                </xsl:when>
                <xsl:otherwise>
                  <option value="{@id}"><xsl:value-of select="name"/></option>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:for-each>
          </select>
        </div>
      </form>
    </div>
  </xsl:if>

  <div id="filterbox" style="display: none">
    <form class="form-horizontal" action="" method="get" name="filterform">
      <xsl:choose>
        <xsl:when test="$type = 'report_result'">
          <input type="hidden" name="cmd" value="get_report_section"/>
        </xsl:when>
        <xsl:otherwise>
          <input type="hidden" name="cmd" value="get_{gsa:type-many($type)}"/>
        </xsl:otherwise>
      </xsl:choose>
      <xsl:for-each select="exslt:node-set($extra_params)/param">
        <input type="hidden" name="{name}" value="{value}"/>
      </xsl:for-each>
      <input type="hidden" name="token" value="{/envelope/token}"/>
      <input type="hidden" name="build_filter" value="0"/>
      <div class="form-group">
        <label for="dfilter" class="col-2 control-label">
          <xsl:value-of select="gsa:i18n ('Filter')"/>:
        </label>
        <div class="col-10">
          <input type="text" name="filter" size="53"
            id="dfilter"
            class="form-control"
            value="{$criteria}"
            maxlength="1000"/>
        </div>
      </div>
      <xsl:if test="filters/keywords/keyword[column='task_id'] and ../get_tasks_response/task">
        <div class="form-group">
          <xsl:variable name="task_id"
            select="filters/keywords/keyword[column='task_id']/value"/>
          <label for="task_id" class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Task')"/>:</label>
          <div class="col-10">
            <select class="col-10 form-control" id="task_id" name="task_id" size="1">
              <xsl:for-each select="../get_tasks_response/task">
                <xsl:call-template name="opt">
                  <xsl:with-param name="value" select="@id"/>
                  <xsl:with-param name="content" select="name/text()"/>
                  <xsl:with-param name="select-value" select="$task_id"/>
                </xsl:call-template>
              </xsl:for-each>
            </select>
          </div>
        </div>
      </xsl:if>
      <xsl:if test="delta or $filter_options_nodes/option[text()='delta_states']">
        <div class="form-group">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Show delta results')"/>:
          </label>
          <span class="col-10">
            <span class="checkbox">
              <label>
                <xsl:choose>
                  <xsl:when test="filters/delta/same = 0">
                    <input type="checkbox" name="delta_state_same" value="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="checkbox" name="delta_state_same"
                      value="1" checked="1"/>
                  </xsl:otherwise>
                </xsl:choose>
                = <xsl:value-of select="gsa:i18n ('same', 'Delta Result')"/>
              </label>
            </span>
            <span class="checkbox">
              <label>
                <xsl:choose>
                  <xsl:when test="filters/delta/new = 0">
                    <input type="checkbox" name="delta_state_new" value="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="checkbox" name="delta_state_new"
                      value="1" checked="1"/>
                  </xsl:otherwise>
                </xsl:choose>
                + <xsl:value-of select="gsa:i18n ('new', 'Delta Result')"/>
              </label>
            </span>
            <span class="checkbox">
              <label>
                <xsl:choose>
                  <xsl:when test="filters/delta/gone = 0">
                    <input type="checkbox" name="delta_state_gone" value="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="checkbox" name="delta_state_gone"
                      value="1" checked="1"/>
                  </xsl:otherwise>
                </xsl:choose>
                &#8722; <xsl:value-of select="gsa:i18n ('gone', 'Delta Result')"/>
              </label>
            </span>
            <span class="checkbox">
              <label>
                <xsl:choose>
                  <xsl:when test="filters/delta/changed = 0">
                    <input type="checkbox" name="delta_state_changed" value="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="checkbox" name="delta_state_changed"
                      value="1" checked="1"/>
                  </xsl:otherwise>
                </xsl:choose>
                ~ <xsl:value-of select="gsa:i18n ('changed', 'Delta Result')"/>
              </label>
            </span>
          </span>
        </div>
      </xsl:if>
      <xsl:if test="filters/keywords/keyword[column='apply_overrides'] or $filter_options_nodes/option[text()='apply_overrides']">
        <div class="form-group">
          <xsl:variable name="apply_overrides"
            select="filters/keywords/keyword[column='apply_overrides']/value"/>
          <!-- TODO: Rename "overrides" to "apply_overrides" where it
                      controls whether overrides are applied -->
          <xsl:variable name="apply_overrides_param_name">
            <xsl:choose>
              <xsl:when test="$type = 'report_result'">apply_overrides</xsl:when>
              <xsl:otherwise>overrides</xsl:otherwise>
            </xsl:choose>
          </xsl:variable>
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Apply overrides')"/>:
          </label>
          <div class="col-10 checkbox">
            <label>
              <xsl:choose>
                <xsl:when test="$apply_overrides = 0">
                  <input type="checkbox" name="{$apply_overrides_param_name}"
                    value="1"/>
                </xsl:when>
                <xsl:otherwise>
                  <input type="checkbox" name="{$apply_overrides_param_name}"
                    value="1" checked="1"/>
                </xsl:otherwise>
              </xsl:choose>
            </label>
          </div>
        </div>
      </xsl:if>
      <xsl:if test="filters/keywords/keyword[column='autofp'] or $filter_options_nodes/option[text()='autofp']">
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Auto-FP')"/>:</label>
          <div class="col-10 checkbox">
            <label>
              <xsl:choose>
                <xsl:when test="filters/keywords/keyword[column='autofp']/value = 0">
                  <input class="form-enable-control" id="autofp" type="checkbox"
                    name="autofp" value="1" disable-on="not(:checked)"/>
                </xsl:when>
                <xsl:otherwise>
                  <input class="form-enable-control" id="autofp" type="checkbox"
                    name="autofp" value="1" checked="1" disable-on="not(:checked)"/>
                </xsl:otherwise>
              </xsl:choose>
              <xsl:value-of select="gsa:i18n ('Trust vendor security updates')"/>
            </label>
            <div>
              <label class="radio-inline">
                <xsl:choose>
                  <xsl:when test="filters/keywords/keyword[column='autofp']/value = 2">
                    <input type="radio" name="autofp_value" value="1"
                      class="form-enable-item--autofp"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="radio" name="autofp_value" value="1" checked="1"
                      class="form-enable-item--autofp"/>
                  </xsl:otherwise>
                </xsl:choose>
                <xsl:value-of select="gsa:i18n ('Full CVE match')"/>
              </label>
              <label class="radio-inline">
                <xsl:choose>
                  <xsl:when test="filters/keywords/keyword[column='autofp']/value = 2">
                    <input type="radio" name="autofp_value" value="2" checked="1"
                      class="form-enable-item--autofp"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="radio" name="autofp_value" value="2"
                      class="form-enable-item--autofp"/>
                  </xsl:otherwise>
                </xsl:choose>
                <xsl:value-of select="gsa:i18n ('Partial CVE match')"/>
              </label>
            </div>
          </div>
        </div>
      </xsl:if>
      <xsl:if test="filters/keywords/keyword[column='notes'] or $filter_options_nodes/option[text()='notes']">
        <div class="form-group">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Show Notes')"/>:
          </label>
          <div class="col-10 checkbox">
            <label>
              <xsl:choose>
                <xsl:when test="filters/keywords/keyword[column='notes']/value = '0'">
                  <input type="checkbox" name="notes" value="1"/>
                </xsl:when>
                <xsl:otherwise>
                  <input type="checkbox" name="notes" value="1"
                    checked="1"/>
                </xsl:otherwise>
              </xsl:choose>
            </label>
          </div>
        </div>
      </xsl:if>
      <xsl:if test="filters/keywords/keyword[column='overrides'] or $filter_options_nodes/option[text()='overrides']">
        <div class="form-group">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Show Overrides')"/>:
          </label>
          <div class="col-10 checkbox">
            <label>
              <xsl:choose>
                <xsl:when test="filters/keywords/keyword[column='overrides']/value = '0'">
                  <input type="checkbox" name="overrides" value="1"/>
                </xsl:when>
                <xsl:otherwise>
                  <input type="checkbox" name="overrides" value="1"
                    checked="1"/>
                </xsl:otherwise>
              </xsl:choose>
            </label>
          </div>
        </div>
      </xsl:if>
      <xsl:if test="filters/keywords/keyword[column='result_hosts_only'] or $filter_options_nodes/option[text()='result_hosts_only']">
        <div class="form-group">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Only show hosts that have results')"/>:
          </label>
          <div class="col-10 checkbox">
            <label>
              <xsl:choose>
                <xsl:when test="filters/keywords/keyword[column='result_hosts_only']/value = '0'">
                  <input type="checkbox" name="result_hosts_only" value="1"/>
                </xsl:when>
                <xsl:otherwise>
                  <input type="checkbox" name="result_hosts_only" value="1"
                    checked="1"/>
                </xsl:otherwise>
              </xsl:choose>
            </label>
          </div>
        </div>
      </xsl:if>
      <xsl:if test="filters/keywords/keyword[column='min_qod'] or $filter_options_nodes/option[text()='min_qod']">
        <div class="form-group">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('QoD')"/>:
          </label>
          <span class="col-10">
            <div class="form-item">
              <label>
                <xsl:value-of select="gsa:i18n ('must be at least', 'QoD')"/>
              </label>
              <xsl:text> </xsl:text>
              <div min="0" max="100" step="1" class="slider" name="min_qod" type="int" value="{$min_qod_value}"></div>
            </div>
          </span>
        </div>
      </xsl:if>
      <xsl:if test="filters/keywords/keyword[column='timezone'] or $filter_options_nodes/option[text()='timezone']">
        <div class="form-group">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Timezone')"/>:
          </label>
          <span class="col-10">
            <xsl:call-template name="timezone-select">
              <xsl:with-param name="timezone" select="timezone"/>
              <xsl:with-param name="input-name" select="'timezone'"/>
            </xsl:call-template>
          </span>
        </div>
      </xsl:if>

      <xsl:if test="filters/keywords/keyword[column='levels'] or $filter_options_nodes/option[text()='levels']">
        <div class="form-group">
          <xsl:variable name="high_filter"
                        select="filters/filter[text()='High'] or contains (filters/keywords/keyword[column='levels']/value, 'h')"/>
          <xsl:variable name="medium_filter"
                        select="filters/filter[text()='Medium'] or contains (filters/keywords/keyword[column='levels']/value, 'm')"/>
          <xsl:variable name="low_filter"
                        select="filters/filter[text()='Low'] or contains (filters/keywords/keyword[column='levels']/value, 'l')"/>
          <xsl:variable name="log_filter"
                        select="filters/filter[text()='Log'] or contains (filters/keywords/keyword[column='levels']/value, 'g')"/>
          <xsl:variable name="false_positive_filter"
                        select="filters/filter[text()='False Postive'] or contains (filters/keywords/keyword[column='levels']/value, 'f')"/>
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Severity (Class)')"/>:
          </label>
          <div class="col-10">
            <label class="checkbox-inline">
              <xsl:choose>
                <xsl:when test="$high_filter">
                  <input type="checkbox" name="level_high" value="1"
                    checked="1"/>
                </xsl:when>
                <xsl:otherwise>
                  <input type="checkbox" name="level_high" value="1"/>
                </xsl:otherwise>
              </xsl:choose>
              <xsl:call-template name="severity-label">
                <xsl:with-param name="level" select="'High'"/>
              </xsl:call-template>
            </label>
            <label class="checkbox-inline">
              <xsl:choose>
                <xsl:when test="$medium_filter">
                  <input type="checkbox" name="level_medium" value="1"
                    checked="1"/>
                </xsl:when>
                <xsl:otherwise>
                  <input type="checkbox" name="level_medium" value="1"/>
                </xsl:otherwise>
              </xsl:choose>
              <xsl:call-template name="severity-label">
                <xsl:with-param name="level" select="'Medium'"/>
              </xsl:call-template>
            </label>
            <label class="checkbox-inline">
              <xsl:choose>
                <xsl:when test="$low_filter">
                  <input type="checkbox" name="level_low" value="1"
                    checked="1"/>
                </xsl:when>
                <xsl:otherwise>
                  <input type="checkbox" name="level_low" value="1"/>
                </xsl:otherwise>
              </xsl:choose>
              <xsl:call-template name="severity-label">
                <xsl:with-param name="level" select="'Low'"/>
              </xsl:call-template>
            </label>
            <label class="checkbox-inline">
              <xsl:choose>
                <xsl:when test="$log_filter">
                  <input type="checkbox" name="level_log" value="1"
                    checked="1"/>
                </xsl:when>
                <xsl:otherwise>
                  <input type="checkbox" name="level_log" value="1"/>
                </xsl:otherwise>
              </xsl:choose>
              <xsl:call-template name="severity-label">
                <xsl:with-param name="level" select="'Log'"/>
              </xsl:call-template>
            </label>
            <label class="checkbox-inline">
              <xsl:choose>
                <xsl:when test="$false_positive_filter">
                  <input type="checkbox"
                    name="level_false_positive"
                    value="1"
                    checked="1"/>
                </xsl:when>
                <xsl:otherwise>
                  <input type="checkbox"
                    name="level_false_positive"
                    value="1"/>
                </xsl:otherwise>
              </xsl:choose>
              <xsl:call-template name="severity-label">
                <xsl:with-param name="level" select="'False Positive'"/>
              </xsl:call-template>
            </label>
          </div>
        </div>
      </xsl:if>
      <xsl:if test="filters/keywords/keyword[column='first'] or $filter_options_nodes/option[text()='first']">
        <xsl:variable name="first_param_name">
          <xsl:choose>
            <xsl:when test="$type = 'report_result'">first_result</xsl:when>
            <xsl:otherwise>first</xsl:otherwise>
          </xsl:choose>
        </xsl:variable>
        <div class="form-group">
          <label for="{$first_param_name}" class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('First result')"/>:
          </label>
          <div class="col-4">
            <input type="number" name="{$first_param_name}" size="5"
              class="form-control spinner"
              min="1"
              max="{$max}"
              data-type="int"
              value="{filters/keywords/keyword[column='first']/value}"
              maxlength="400"/>
          </div>
        </div>
      </xsl:if>
      <xsl:if test="filters/keywords/keyword[column='rows'] or $filter_options_nodes/option[text()='rows']">
        <xsl:variable name="max_param_name">
          <xsl:choose>
            <xsl:when test="$type = 'report_result'">max_results</xsl:when>
            <xsl:otherwise>max</xsl:otherwise>
          </xsl:choose>
        </xsl:variable>
        <div class="form-group">
          <label for="{$max_param_name}" class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Results per page')"/>:
          </label>
          <div class="col-4">
            <input name="{$max_param_name}" size="5"
              class="form-control spinner"
              min="1"
              type="number"
              data-type="int"
              value="{filters/keywords/keyword[column='rows']/value}"
              maxlength="400"/>
          </div>
        </div>
      </xsl:if>
      <xsl:if test="exslt:node-set ($columns)">
        <div class="form-group">
          <label for="sort_field" class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Sort by')"/>:</label>
          <div class="col-10">
            <xsl:variable name="sort" select="sort/field/text ()"/>
            <div class="form-item">
              <select name="sort_field" size="1">
                <xsl:for-each select="exslt:node-set ($columns)/column">
                  <xsl:variable name="single" select="count (column) = 0"/>
                  <xsl:choose>
                    <xsl:when test="boolean (hide_in_filter)"/>
                    <xsl:when test="($single) and ((boolean (field) and field = $sort) or (gsa:column-filter-name (name) = $sort))">
                      <option value="{$sort}" selected="1">
                        <xsl:value-of select="name"/>
                      </option>
                    </xsl:when>
                    <xsl:when test="$single and boolean (field)">
                      <option value="{field}">
                        <xsl:value-of select="name"/>
                      </option>
                    </xsl:when>
                    <xsl:when test="$single">
                      <option value="{gsa:column-filter-name (name)}">
                        <xsl:value-of select="name"/>
                      </option>
                    </xsl:when>
                    <xsl:otherwise>
                      <xsl:for-each select="column">
                        <xsl:choose>
                          <xsl:when test="(boolean (field) and (field = $sort)) or (gsa:column-filter-name (name) = $sort)">
                            <option value="{$sort}" selected="1">
                              <xsl:value-of select="concat(../name, ': ', name)"/>
                            </option>
                          </xsl:when>
                          <xsl:when test="boolean (field)">
                            <option value="{field}">
                              <xsl:value-of select="concat(../name, ': ', name)"/>
                            </option>
                          </xsl:when>
                          <xsl:otherwise>
                            <option value="{gsa:column-filter-name (name)}">
                              <xsl:value-of select="concat(../name, ': ', name)"/>
                            </option>
                          </xsl:otherwise>
                        </xsl:choose>
                      </xsl:for-each>
                    </xsl:otherwise>
                  </xsl:choose>
                </xsl:for-each>
              </select>
            </div>
            <xsl:variable name="order" select="sort/field/order"/>
            <div class="form-item">
              <label class="radio-inline">
                <xsl:choose>
                  <xsl:when test="$order = 'ascending'">
                    <input type="radio" name="sort_order" value="ascending" checked="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="radio" name="sort_order" value="ascending"/>
                  </xsl:otherwise>
                </xsl:choose>
                <xsl:value-of select="gsa:i18n ('Ascending')"/>
              </label>
              <label class="radio-inline">
                <xsl:choose>
                  <xsl:when test="$order = 'descending'">
                    <input type="radio" name="sort_order" value="descending" checked="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="radio" name="sort_order" value="descending"/>
                  </xsl:otherwise>
                </xsl:choose>
                <xsl:value-of select="gsa:i18n ('Descending')"/>
              </label>
            </div>
          </div>
        </div>
      </xsl:if>
    </form>
  </div>
</xsl:template>

<xsl:template name="edit-header-icons">
  <xsl:param name="type"/>
  <xsl:param name="cap-type"/>
  <xsl:param name="cap-type-plural" select="concat ($cap-type, 's')"/>
  <xsl:param name="id"/>
  <!-- i18n with concat : see dynamic_strings.xsl - type-edit -->
  <xsl:variable name="help_url">
    <xsl:choose>
      <xsl:when test="$type = 'config'">
        <xsl:value-of select="concat ('/help/config_editor.html?token=', /envelope/token)"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="concat ('/help/', $type, 's.html?token=', /envelope/token, '#edit_', $type)"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <a href="{$help_url}" title="{gsa:i18n ('Help')}: {gsa:i18n (concat ('Edit ', $cap-type))}"
    class="icon icon-sm">
    <img src="/img/help.svg"/>
  </a>
  <!-- dynamic i18n : see dynamic_strings.xsl - type-name-plural -->
  <a href="/omp?cmd=get_{$type}s&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
     title="{gsa:i18n ($cap-type-plural)}" class="icon icon-sm">
    <img src="/img/list.svg" alt="{gsa:i18n ($cap-type-plural)}"/>
  </a>
  <!-- i18n with concat : see dynamic_strings.xsl - type-name-details -->
  <div class="small_inline_form" style="display: inline; margin-left: 15px; font-weight: normal;">
      <a href="/omp?cmd=get_{$type}&amp;{$type}_id={$id}&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
        class="icon icon-sm"
        title="{gsa:i18n (concat ($cap-type, ' Details'))}">
      <img src="/img/details.svg" alt="{gsa:i18n ('Details')}"/>
    </a>
  </div>
</xsl:template>

<xsl:template name="get-settings-resource">
  <xsl:param name="id"/>
  <xsl:param name="type"/>
  <xsl:param name="cap_type" select="gsa:capitalise ($type)"/>
  <xsl:param name="resources"/>

  <xsl:choose>
    <xsl:when test="$id">
      <!-- i18n with concat : see dynamic_strings.xsl - type-name-details -->
      <a href="/omp?cmd=get_{$type}&amp;{$type}_id={$id}&amp;token={/envelope/token}"
         title="{gsa:i18n (concat ($cap_type, ' Details'), $type)}">
        <xsl:value-of select="$resources[@id=$id]/name"/>
      </a>
    </xsl:when>
    <xsl:otherwise>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="get-settings-filter">
  <xsl:param name="filter"/>

  <xsl:choose>
    <xsl:when test="$filter">
      <a href="/omp?cmd=get_filter&amp;filter_id={$filter}&amp;token={/envelope/token}"
         title="{gsa:i18n ('Filter Details')}">
        <xsl:value-of select="commands_response/get_filters_response/filter[@id=$filter]/name"/>
      </a>
    </xsl:when>
    <xsl:otherwise>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="edit-settings-resource">
  <xsl:param name="setting"/>
  <xsl:param name="param_name" select="concat('settings_default:', $setting)"/>
  <xsl:param name="resources"/>
  <xsl:param name="selected_id" select="@id"/>

  <select style="margin-bottom: 0px;" name="{$param_name}" class="setting-control" data-setting="{$setting}">
    <option value="">--</option>
    <xsl:for-each select="$resources">
      <xsl:choose>
        <xsl:when test="@id = $selected_id">
          <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
        </xsl:when>
        <xsl:otherwise>
          <option value="{@id}"><xsl:value-of select="name"/></option>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:for-each>
  </select>
</xsl:template>

<xsl:template name="edit-settings-filters">
  <xsl:param name="uuid"/>
  <xsl:param name="filter-type"/>
  <xsl:param name="filter"/>
  <select style="margin-bottom: 0px;" name="settings_filter:{$uuid}" class="setting-control" data-setting="{$uuid}">
    <option value="">--</option>
    <xsl:variable name="id" select="filters/@id"/>
    <xsl:for-each select="commands_response/get_filters_response/filter[type=$filter-type or type='']">
      <xsl:choose>
        <xsl:when test="@id = $filter">
          <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
        </xsl:when>
        <xsl:otherwise>
          <option value="{@id}"><xsl:value-of select="name"/></option>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:for-each>
  </select>
</xsl:template>

<xsl:template name="severity-settings-list">
  <xsl:param name="default"/>
  <select style="margin-bottom: 0px;" name="severity_class" class="setting-control" data-setting="severity_class">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'nist'"/>
      <xsl:with-param name="content" select="'NVD Vulnerability Severity Ratings'"/>
      <xsl:with-param name="select-value" select="$default"/>
    </xsl:call-template>
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'bsi'"/>
      <xsl:with-param name="content" select="'BSI Schwachstellenampel (Germany)'"/>
      <xsl:with-param name="select-value" select="$default"/>
    </xsl:call-template>
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'classic'"/>
      <xsl:with-param name="content" select="'OpenVAS Classic'"/>
      <xsl:with-param name="select-value" select="$default"/>
    </xsl:call-template>
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'pci-dss'"/>
      <xsl:with-param name="content" select="'PCI-DSS'"/>
      <xsl:with-param name="select-value" select="$default"/>
    </xsl:call-template>
  </select>
</xsl:template>

<xsl:template name="severity-settings-name">
  <xsl:param name="type"/>
  <xsl:choose>
    <xsl:when test="$type = 'nist'">NVD Vulnerability Severity Ratings</xsl:when>
    <xsl:when test="$type = 'bsi'">BSI Schwachstellenampel (Germany)</xsl:when>
    <xsl:when test="$type = 'classic'">OpenVAS Classic</xsl:when>
    <xsl:when test="$type = 'pci-dss'">PCI-DSS</xsl:when>
  </xsl:choose>
</xsl:template>

<xsl:template name="list-window-line-icons">
  <xsl:param name="resource" select="."/>
  <xsl:param name="type"/>
  <xsl:param name="cap-type"/>

  <xsl:param name="id"/>
  <xsl:param name="noedit"/>
  <xsl:param name="noclone"/>
  <xsl:param name="grey-clone"/>
  <xsl:param name="noexport"/>
  <xsl:param name="notrash"/>
  <xsl:param name="params" select="''"/>
  <xsl:param name="next" select="concat ('get_', $type, 's')"/>
  <xsl:param name="next_type" select="''"/>
  <xsl:param name="next_id" select="''"/>

  <xsl:variable name="next_params_string">
    <xsl:choose>
      <xsl:when test="$next_type != '' and $next_id != ''">
        <xsl:value-of select="concat ('&amp;next_type=', $next_type, '&amp;next_id=', $next_id)"/>
      </xsl:when>
      <xsl:otherwise/>
    </xsl:choose>
  </xsl:variable>

  <xsl:choose>
    <xsl:when test="$notrash">
    </xsl:when>
    <xsl:when test="gsa:may (concat ('delete_', $type)) and $resource/writable!='0' and $resource/in_use='0'">
      <xsl:call-template name="trashcan-icon">
        <xsl:with-param name="type" select="$type"/>
        <xsl:with-param name="id" select="$resource/@id"/>
        <xsl:with-param name="params">
          <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
          <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
          <xsl:if test="$next != ''">
            <input type="hidden" name="next" value="{$next}"/>
          </xsl:if>
          <xsl:if test="$next_id != '' and $next_type != ''">
            <input type="hidden" name="{$next_type}_id" value="{$next_id}"/>
          </xsl:if>
        </xsl:with-param>
      </xsl:call-template>
    </xsl:when>
    <xsl:otherwise>
      <xsl:variable name="inactive_text">
        <!-- i18n with concat : see dynamic_strings.xsl - type-action-denied -->
        <xsl:choose>
          <xsl:when test="$resource/in_use != '0'">
            <xsl:value-of select="gsa:i18n (concat ($cap-type, ' is still in use'))"/>
          </xsl:when>
          <xsl:when test="$resource/writable = '0'">
            <xsl:value-of select="gsa:i18n (concat ($cap-type, ' is not writable'))"/>
          </xsl:when>
          <xsl:when test="not(gsa:may (concat ('delete_', $type)))">
            <xsl:value-of select="gsa:i18n (concat ('Permission to move ', $cap-type, ' to trashcan denied'))"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="gsa:i18n ('Cannot move to trashcan.')"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:variable>
      <img src="/img/trashcan_inactive.svg" class="icon icon-sm"
           alt="{gsa:i18n ('To Trashcan', 'Action Verb')}"
           title="{$inactive_text}"/>
    </xsl:otherwise>
  </xsl:choose>
  <xsl:choose>
    <xsl:when test="$noedit">
    </xsl:when>
    <xsl:otherwise>
      <xsl:choose>
        <xsl:when test="gsa:may (concat ('modify_', $type)) and $resource/writable!='0'">
          <!-- i18n with concat : see dynamic_strings.xsl - type-edit -->
          <a href="/omp?cmd=edit_{$type}&amp;{$type}_id={$resource/@id}&amp;next={$next}{$next_params_string}{$params}&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
             title="{gsa:i18n (concat ('Edit ', $cap-type))}" data-reload="window"
             class="edit-action-icon icon icon-sm" data-type="{$type}" data-id="{$resource/@id}">
            <img src="/img/edit.svg" alt="{gsa:i18n ('Edit', 'Action Verb')}"/>
          </a>
        </xsl:when>
        <xsl:otherwise>
          <xsl:variable name="inactive_text">
            <!-- i18n with concat : see dynamic_strings.xsl - type-action-denied -->
            <xsl:choose>
              <xsl:when test="$resource/writable = '0'">
                <xsl:value-of select="gsa:i18n (concat ($cap-type, ' is not writable'))"/>
              </xsl:when>
              <xsl:when test="not(gsa:may (concat ('delete_', $type)))">
                <xsl:value-of select="gsa:i18n (concat ('Permission to edit ', $cap-type, ' denied'))"/>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="gsa:i18n (concat ('Cannot modify ', $cap-type))"/>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:variable>
          <img src="/img/edit_inactive.svg" alt="{gsa:i18n ('Edit', 'Action Verb')}"
            title="{$inactive_text}" class="icon icon-sm"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:otherwise>
  </xsl:choose>
  <xsl:choose>
    <xsl:when test="$noclone">
    </xsl:when>
    <xsl:when test="$grey-clone">
      <!-- i18n with concat : see dynamic_strings.xsl - type-action-denied -->
      <img src="/img/clone_inactive.svg"
           alt="{gsa:i18n ('Clone', 'Action Verb')}"
           value="Clone" class="icon icon-sm"
           title="{gsa:i18n (concat ($cap-type, ' may not be cloned'))}"/>
    </xsl:when>
    <xsl:when test="gsa:may-clone ($type)">
      <div class="icon icon-sm">
        <form action="/omp" method="post" enctype="multipart/form-data">
          <input type="hidden" name="token" value="{/envelope/token}"/>
          <input type="hidden" name="caller" value="{/envelope/current_page}"/>
          <input type="hidden" name="cmd" value="clone"/>
          <input type="hidden" name="resource_type" value="{$type}"/>
          <input type="hidden" name="next" value="get_{$type}"/>
          <input type="hidden" name="id" value="{$resource/@id}"/>
          <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
          <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
          <input type="image" src="/img/clone.svg" alt="{gsa:i18n ('Clone', 'Action Verb')}"
                 name="Clone" value="Clone" title="{gsa:i18n ('Clone', 'Action Verb')}"/>
        </form>
      </div>
    </xsl:when>
    <xsl:when test="$resource/owner/name = /envelope/login/text() or string-length ($resource/owner/name) = 0">
      <!-- i18n with concat : see dynamic_strings.xsl - type-action-denied -->
      <img src="/img/clone_inactive.svg"
           alt="{gsa:i18n ('Clone', 'Action Verb')}"
           value="Clone" class="icon icon-sm"
           title="{gsa:i18n (concat ($cap-type, ' must be owned or global'))}"/>
    </xsl:when>
    <xsl:otherwise>
      <img src="/img/clone_inactive.svg"
           alt="{gsa:i18n ('Clone', 'Action Verb')}"
           value="Clone" class="icon icon-sm"
           title="{gsa:i18n ('Permission to clone denied')}"/>
    </xsl:otherwise>
  </xsl:choose>
  <xsl:choose>
    <xsl:when test="$noexport">
    </xsl:when>
    <xsl:otherwise>
      <!-- i18n with concat : see dynamic_strings.xsl - type-export -->
      <a href="/omp?cmd=export_{$type}&amp;{$type}_id={$resource/@id}&amp;next={$next}{$params}&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
        class="icon icon-sm"
        title="{gsa:i18n (concat ('Export ', $cap-type))}">
        <img src="/img/download.svg" alt="{gsa:i18n ('Export', 'Action Verb')}"/>
      </a>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="trash-delete-icon">
  <xsl:param name="type"></xsl:param>
  <xsl:param name="id"></xsl:param>
  <xsl:param name="params"></xsl:param>

  <div class="icon icon-sm">
    <form action="/omp" method="post" enctype="multipart/form-data">
      <input type="hidden" name="token" value="{/envelope/token}"/>
      <input type="hidden" name="caller" value="{/envelope/current_page}"/>
      <input type="hidden" name="cmd" value="delete_trash_{$type}"/>
      <input type="hidden" name="next" value="get_trash"/>
      <input type="hidden" name="{$type}_id" value="{$id}"/>
      <input type="image" src="/img/delete.svg" alt="{gsa:i18n ('Delete')}"
             name="Delete" value="Delete" title="{gsa:i18n ('Delete')}"/>
      <xsl:copy-of select="$params"/>
    </form>
  </div>
</xsl:template>

<xsl:template name="delete-icon">
  <xsl:param name="type"></xsl:param>
  <xsl:param name="id"></xsl:param>
  <xsl:param name="params"></xsl:param>

  <div class="icon icon-sm">
    <xsl:choose>
      <xsl:when test="$type = 'user'">
        <form action="/omp" method="get" enctype="multipart/form-data">
          <input type="hidden" name="token" value="{/envelope/token}"/>
          <input type="hidden" name="caller" value="{/envelope/current_page}"/>
          <input type="hidden" name="cmd" value="delete_{$type}_confirm"/>
          <input type="hidden" name="{$type}_id" value="{$id}"/>
          <input type="image" src="/img/delete.svg" alt="{gsa:i18n ('Delete')}"
            class="delete-action-icon" data-reload="next" data-type="{$type}" data-id="{$id}"
            name="Delete" value="Delete" title="{gsa:i18n ('Delete')}"/>
          <xsl:copy-of select="$params"/>
        </form>
      </xsl:when>
      <xsl:otherwise>
        <form style="display: inline; font-size: 0px;" action="/omp" method="post" enctype="multipart/form-data">
          <input type="hidden" name="token" value="{/envelope/token}"/>
          <input type="hidden" name="caller" value="{/envelope/current_page}"/>
          <input type="hidden" name="cmd" value="delete_{$type}"/>
          <input type="hidden" name="{$type}_id" value="{$id}"/>
          <input type="image" src="/img/delete.svg" alt="{gsa:i18n ('Delete')}"
                name="Delete" value="Delete" title="{gsa:i18n ('Delete')}"/>
          <xsl:copy-of select="$params"/>
        </form>
      </xsl:otherwise>
    </xsl:choose>
  </div>
</xsl:template>

<xsl:template name="restore-icon">
  <xsl:param name="id"></xsl:param>

  <xsl:if test="gsa:may-op ('restore')">
    <div class="icon icon-sm">
      <form action="/omp"
            method="post" enctype="multipart/form-data">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden" name="cmd" value="restore"/>
        <input type="hidden" name="target_id" value="{$id}"/>
        <input type="image" src="/img/restore.svg" alt="{gsa:i18n ('Restore')}"
               name="Restore" value="Restore" title="{gsa:i18n ('Restore')}"/>
      </form>
    </div>
  </xsl:if>
</xsl:template>

<xsl:template name="resume-icon">
  <xsl:param name="type"></xsl:param>
  <xsl:param name="id"></xsl:param>
  <xsl:param name="params"></xsl:param>
  <xsl:param name="cmd">resume_<xsl:value-of select="type"/></xsl:param>

  <div class="icon icon-sm">
    <form action="/omp" method="post" enctype="multipart/form-data">
      <input type="hidden" name="token" value="{/envelope/token}"/>
      <input type="hidden" name="caller" value="{/envelope/current_page}"/>
      <input type="hidden" name="cmd" value="{$cmd}"/>
      <input type="hidden" name="{$type}_id" value="{$id}"/>
      <input type="image" src="/img/resume.svg" alt="{gsa:i18n ('Resume')}"
             name="Resume" value="Resume" title="{gsa:i18n ('Resume')}"/>
      <xsl:copy-of select="$params"/>
    </form>
  </div>
</xsl:template>

<xsl:template name="start-icon">
  <xsl:param name="type"></xsl:param>
  <xsl:param name="id"></xsl:param>
  <xsl:param name="params"></xsl:param>
  <xsl:param name="cmd">start_<xsl:value-of select="$type"/></xsl:param>
  <xsl:param name="alt"><xsl:value-of select="gsa:i18n('Start', 'Action Verb')"/></xsl:param>
  <xsl:param name="name">Start</xsl:param>

  <div class="icon icon-sm">
    <form action="/omp" method="post" enctype="multipart/form-data">
      <input type="hidden" name="token" value="{/envelope/token}"/>
      <input type="hidden" name="caller" value="{/envelope/current_page}"/>
      <input type="hidden" name="cmd" value="{$cmd}"/>
      <input type="hidden" name="{$type}_id" value="{$id}"/>
      <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
      <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
      <input type="image" src="/img/start.svg" alt="{$alt}"
             name="{$name}" value="{$name}" title="{$alt}"/>
      <xsl:copy-of select="$params"/>
    </form>
  </div>
</xsl:template>

<xsl:template name="stop-icon">
  <xsl:param name="type"></xsl:param>
  <xsl:param name="id"></xsl:param>
  <xsl:param name="params"></xsl:param>

  <div class="icon icon-sm">
    <form action="/omp" method="post" enctype="multipart/form-data">
      <input type="hidden" name="token" value="{/envelope/token}"/>
      <input type="hidden" name="caller" value="{/envelope/current_page}"/>
      <input type="hidden" name="cmd" value="stop_{$type}"/>
      <input type="hidden" name="{$type}_id" value="{$id}"/>
      <input type="image" src="/img/stop.svg" alt="{gsa:i18n('Stop', 'Action Verb')}"
             name="Stop" value="Stop" title="{gsa:i18n('Stop', 'Action Verb')}"/>
      <xsl:copy-of select="$params"/>
    </form>
  </div>
</xsl:template>

<xsl:template name="trashcan-icon">
  <xsl:param name="type"></xsl:param>
  <xsl:param name="id"></xsl:param>
  <xsl:param name="fragment"></xsl:param>
  <xsl:param name="params"></xsl:param>

  <div class="icon icon-sm ajax-post" data-reload="next" data-busy-text="{gsa:i18n ('Moving to trashcan...')}">
    <img src="/img/trashcan.svg" alt="{gsa:i18n ('To Trashcan', 'Action Verb')}"
      name="To Trashcan" title="{gsa:i18n ('Move To Trashcan', 'Action Verb')}"/>
    <form action="/omp{$fragment}" method="post" enctype="multipart/form-data">
      <input type="hidden" name="token" value="{/envelope/token}"/>
      <input type="hidden" name="caller" value="{/envelope/current_page}"/>
      <input type="hidden" name="cmd" value="delete_{$type}"/>
      <input type="hidden" name="{$type}_id" value="{$id}"/>
      <xsl:copy-of select="$params"/>
    </form>
  </div>
</xsl:template>

<xsl:template name="highlight-diff">
  <xsl:param name="string"></xsl:param>

  <xsl:for-each select="str:tokenize($string, '&#10;')">
    <xsl:call-template name="highlight-diff-line">
      <xsl:with-param name="string"><xsl:value-of select="."/></xsl:with-param>
      <xsl:with-param name="class-string">
        <xsl:choose>
          <xsl:when test="(substring (., 1, 1) = '\') and preceding-sibling::*">
            <!-- Use class from previous line for one like
                 "\ No newline at end of file" -->
            <xsl:value-of select="preceding-sibling::*[1]"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="."/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:with-param>
    </xsl:call-template>
  </xsl:for-each>
</xsl:template>

<!-- This is called within a PRE. -->
<xsl:template name="break-diff-line">
  <xsl:param name="string"></xsl:param>
  <xsl:param name="break-length" select="90"/>
  <xsl:choose>
    <xsl:when test="string-length ($string) &gt; $break-length">
      <xsl:value-of select="substring ($string, 1, $break-length)"/>
      <xsl:text>&#8629;&#10;</xsl:text>
      <xsl:call-template name="break-diff-line">
        <xsl:with-param name="string" select="substring ($string, $break-length+1, string-length ($string))"/>
      </xsl:call-template>
    </xsl:when>
    <xsl:otherwise>
      <xsl:value-of select="$string"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- This is called within a PRE. -->
<xsl:template name="highlight-diff-line">
  <xsl:param name="string"></xsl:param>
  <!-- class-string : String to base class on (e.g. for \ ... lines) -->
  <xsl:param name="class-string" select="$string"/>
  <xsl:choose>
    <xsl:when test="string-length($string) = 0">
      <!-- The string is empty. -->
    </xsl:when>
    <xsl:when test="(substring($class-string, 1, 1) = '@')">
      <div class="diff-line-hunk">
        <xsl:call-template name="break-diff-line">
          <xsl:with-param name="string" select="$string"/>
        </xsl:call-template>
      </div>
    </xsl:when>
    <xsl:when test="(substring($class-string, 1, 1) = '+')">
      <div class="diff-line-plus">
        <xsl:call-template name="break-diff-line">
          <xsl:with-param name="string" select="$string"/>
        </xsl:call-template>
      </div>
    </xsl:when>
    <xsl:when test="(substring($class-string, 1, 1) = '-')">
      <div class="diff-line-minus">
        <xsl:call-template name="break-diff-line">
          <xsl:with-param name="string" select="$string"/>
        </xsl:call-template>
      </div>
    </xsl:when>
    <xsl:otherwise>
      <div class="diff-line">
        <xsl:call-template name="break-diff-line">
          <xsl:with-param name="string" select="$string"/>
        </xsl:call-template>
      </div>
    </xsl:otherwise>
  </xsl:choose>

</xsl:template>

<xsl:template name="severity-bar">
  <xsl:param name="extra_text"></xsl:param>
  <xsl:param name="notext"></xsl:param>
  <xsl:param name="cvss"></xsl:param>
  <xsl:param name="threat"><xsl:value-of select="gsa:cvss-risk-factor($cvss)"/></xsl:param>
  <xsl:param name="title"><xsl:value-of select="gsa:i18n($threat, 'Severity')"/></xsl:param>
  <xsl:param name="scale">10</xsl:param>

  <xsl:variable name="fill">
    <xsl:value-of select="number($cvss) * $scale"/>
  </xsl:variable>
  <xsl:variable name="width"><xsl:value-of select="10 * $scale"/></xsl:variable>
  <div class="progressbar_box" title="{$title}" style="width:{$width}px;">
    <xsl:choose>
      <xsl:when test="$threat = 'None'">
        <div class="progressbar_bar_done" style="width:0px;"></div>
      </xsl:when>
      <xsl:when test="$threat = 'Log'">
        <div class="progressbar_bar_gray" style="width:{$fill}px;"></div>
      </xsl:when>
      <xsl:when test="$threat = 'Low'">
        <div class="progressbar_bar_done" style="width:{$fill}px;"></div>
      </xsl:when>
      <xsl:when test="$threat = 'Medium'">
        <div class="progressbar_bar_request" style="width:{$fill}px;"></div>
      </xsl:when>
      <xsl:when test="$threat = 'High'">
        <div class="progressbar_bar_error" style="width:{$fill}px;"></div>
      </xsl:when>
    </xsl:choose>
      <div class="progressbar_text">
        <xsl:if test="not($notext)">
          <xsl:value-of select="$cvss"/>
        </xsl:if>
        <xsl:if test="$extra_text">
          <xsl:value-of select="$extra_text"/>
        </xsl:if>
      </div>
  </div>
</xsl:template>

<xsl:template name="severity-label">
  <xsl:param name="level"/>
  <xsl:param name="font-size" select="'9'"/>
  <xsl:param name="width" select="floor($font-size * 6.0)"/>
  <xsl:choose>
    <xsl:when test="$level = 'High'">
      <div class="label_high" style="font-size:{$font-size}px; min-width:{$width}px"><xsl:value-of select="gsa:i18n ('High', 'Severity')"/></div>
    </xsl:when>
    <xsl:when test="$level = 'Medium'">
      <div class="label_medium" style="font-size:{$font-size}px; min-width:{$width}px"><xsl:value-of select="gsa:i18n ('Medium', 'Severity')"/></div>
    </xsl:when>
    <xsl:when test="$level = 'Low'">
      <div class="label_low" style="font-size:{$font-size}px; min-width:{$width}px"><xsl:value-of select="gsa:i18n ('Low', 'Severity')"/></div>
    </xsl:when>
    <xsl:when test="$level = 'Log'">
      <div class="label_log" style="font-size:{$font-size}px; min-width:{$width}px"><xsl:value-of select="gsa:i18n ('Log', 'Severity')"/></div>
    </xsl:when>
    <xsl:when test="$level = 'False Positive' or $level = 'False&#xa0;Positive'">
      <div class="label_none" style="font-size:{$font-size}px; min-width:{$width}px"><xsl:value-of select="gsa:i18n ('False Pos.', 'Severity')"/></div>
    </xsl:when>
    <xsl:otherwise>
      <div class="label_none" style="font-size:{$font-size}px; min-width:{$width}px"><xsl:value-of select="$level"/></div>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="build-levels">
  <xsl:param name="filters"></xsl:param>
  <xsl:for-each select="$filters">
    <xsl:choose>
      <xsl:when test="text()='High'">h</xsl:when>
      <xsl:when test="text()='Medium'">m</xsl:when>
      <xsl:when test="text()='Low'">l</xsl:when>
      <xsl:when test="text()='Log'">g</xsl:when>
      <xsl:when test="text()='False Positive'">f</xsl:when>
    </xsl:choose>
  </xsl:for-each>
</xsl:template>

<xsl:template name="scanner-type-name">
  <xsl:param name="type"/>
  <xsl:choose>
    <xsl:when test="$type = '1'">OSP Scanner</xsl:when>
    <xsl:when test="$type = '2'">OpenVAS Scanner</xsl:when>
    <xsl:when test="$type = '3'">CVE Scanner</xsl:when>
    <xsl:when test="$type = '4'">OMP Slave</xsl:when>
    <xsl:otherwise>Unknown type (<xsl:value-of select="type"/>)</xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="scanner-type-list">
  <xsl:param name="default"/>
  <xsl:call-template name="opt">
    <xsl:with-param name="value" select="4"/>
    <xsl:with-param name="content" select="'OMP Slave'"/>
    <xsl:with-param name="select-value" select="$default"/>
  </xsl:call-template>
  <xsl:call-template name="opt">
    <xsl:with-param name="value" select="2"/>
    <xsl:with-param name="content" select="'OpenVAS Scanner'"/>
    <xsl:with-param name="select-value" select="$default"/>
  </xsl:call-template>
  <xsl:call-template name="opt">
    <xsl:with-param name="value" select="1"/>
    <xsl:with-param name="content" select="'OSP Scanner'"/>
    <xsl:with-param name="select-value" select="$default"/>
  </xsl:call-template>
</xsl:template>

<xsl:template name="solution-icon">
  <xsl:param name="solution_type" select="''"/>
  <xsl:choose>
    <xsl:when test="$solution_type = ''">
    </xsl:when>
    <xsl:when test="$solution_type = 'Workaround'">
      <img class="icon icon-sm" src="/img/st_workaround.svg" title="{$solution_type}" alt="{$solution_type}"/>
    </xsl:when>
    <xsl:when test="$solution_type = 'Mitigation'">
      <img class="icon icon-sm" src="/img/st_mitigate.svg" title="{$solution_type}" alt="{$solution_type}"/>
    </xsl:when>
    <xsl:when test="$solution_type = 'VendorFix'">
      <img class="icon icon-sm" src="/img/st_vendorfix.svg" title="{$solution_type}" alt="{$solution_type}"/>
    </xsl:when>
    <xsl:when test="$solution_type = 'NoneAvailable'">
      <img class="icon icon-sm" src="/img/st_nonavailable.svg" title="{$solution_type}" alt="{$solution_type}"/>
    </xsl:when>
    <xsl:when test="$solution_type = 'WillNotFix'">
      <img class="icon icon-sm" src="/img/st_willnotfix.svg" title="{$solution_type}" alt="{$solution_type}"/>
    </xsl:when>
    <xsl:otherwise>
      <img class="icon icon-sm" src="/img/os_unknown.svg" title="{$solution_type}" alt="{$solution_type}"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- GENERAL ERROR MESSAGES -->

<xsl:template match="action_status">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation" select="../prev_action"/>
    <xsl:with-param name="status" select="text()"/>
    <xsl:with-param name="msg" select="../action_message"/>
  </xsl:call-template>
</xsl:template>

<!-- BEGIN GENERAL TAGS VIEWS -->

<xsl:template name="user-tags-window-checked">
  <xsl:param name="resource_type"/>
  <xsl:param name="resource_subtype"/>
  <xsl:param name="resource_id"/>
  <xsl:param name="next"/>
  <xsl:param name="report_section"/>
  <xsl:param name="user_tags"/>
  <xsl:param name="tag_names"/>

  <div class="section-header">
    <a href="#" class="icon icon-sm icon-action toggle-action-icon"
      data-target="#usertags-box" data-name="User Tags" data-variable="usertags-box--collapsed">
      <img src="/img/fold.svg"/>
    </a>
    <a href="/help/user-tags.html?token={/envelope/token}"
       class="icon icon-sm icon-action"
       title="{gsa:i18n ('Help')}: {gsa:i18n ('User Tags list')}">
      <img src="/img/help.svg"/>
    </a>
      <xsl:choose>
        <xsl:when test="not (gsa:may-op ('create_tag'))"/>
        <xsl:when test="$report_section != ''">
          <a href="/omp?cmd=new_tag&amp;resource_id={$resource_id}&amp;resource_type={$resource_type}&amp;next={$next}&amp;next_type={$resource_type}&amp;next_id={$resource_id}&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;report_section={$report_section}&amp;token={/envelope/token}"
             title="{gsa:i18n ('New tag')}"
             data-reload="window"
             class="new-action-icon icon icon-sm icon-action" data-type="tag" data-extra="resource_id={$resource_id}&amp;resource_type={$resource_type}">
            <img src="/img/new.svg" alt="{gsa:i18n ('Add tag')}"/>
          </a>
        </xsl:when>
        <xsl:when test="$resource_subtype != ''">
          <a href="/omp?cmd=new_tag&amp;resource_id={$resource_id}&amp;resource_type={$resource_subtype}&amp;next={$next}&amp;next_type={$resource_type}&amp;next_subtype={$resource_subtype}&amp;next_id={$resource_id}&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
             title="{gsa:i18n ('New Tag')}"
             data-reload="window"
             class="new-action-icon icon icon-sm icon-action" data-type="tag" data-extra="resource_id={$resource_id}&amp;resource_type={$resource_subtype}">
            <img src="/img/new.svg" alt="{gsa:i18n ('Add tag')}"/>
          </a>
        </xsl:when>
        <xsl:otherwise>
          <a href="/omp?cmd=new_tag&amp;resource_id={$resource_id}&amp;resource_type={$resource_type}&amp;next={$next}&amp;next_type={$resource_type}&amp;next_id={$resource_id}&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
             title="{gsa:i18n ('New Tag')}"
             data-reload="window"
             class="new-action-icon icon icon-sm icon-action" data-type="tag" data-extra="resource_id={$resource_id}&amp;resource_type={$resource_type}">
            <img src="/img/new.svg" alt="{gsa:i18n ('Add tag')}"/>
          </a>
        </xsl:otherwise>
      </xsl:choose>
    <h2>
      <a href="/omp?cmd=get_tags&amp;filter=resource_uuid={$resource_id}&amp;token={/envelope/token}"
         title="{gsa:i18n ('Tags')}">
        <img class="icon icon-sm" src="/img/tag.svg" alt="Tags"/>
      </a>
      <xsl:value-of select="gsa:i18n ('User Tags')"/>
      <xsl:choose>
        <xsl:when test="$user_tags/count != 0">
          (<xsl:value-of select="$user_tags/count"/>)
        </xsl:when>
        <xsl:otherwise>
          (<xsl:value-of select="gsa:i18n ('none', 'Tags')"/>)
        </xsl:otherwise>
      </xsl:choose>
    </h2>
  </div>

  <div class="section-box" id="usertags-box">
    <xsl:if test="count(//delete_tag_response[@status!=200]|//modify_tag_response[@status!=200]|//create_tag_response[@status!=201]) = 0">
      <a name="user_tags"/>
    </xsl:if>
    <xsl:choose>
      <xsl:when test="count($tag_names/tag) > 0">
        <div class="ajax-post" data-reload="next" data-button="input.icon" data-busy-text="{gsa:i18n ('Adding Tag...')}">
          <form class="form-inline" action="/omp#user_tags" method="post" enctype="multipart/form-data">
            <input type="hidden" name="comment"/>
            <input type="hidden" name="active" value="1"/>
            <input type="hidden" name="caller" value="{/envelope/current_page}"/>
            <input type="hidden" name="token" value="{/envelope/token}"/>
            <input type="hidden" name="cmd" value="create_tag"/>
            <input type="hidden" name="resource_id" value="{$resource_id}"/>

            <div class="form-group">
              <label class="control-label">
                <b><xsl:value-of select="gsa:i18n ('Add Tag')"/>:</b>
              </label>
              <select style="margin-bottom: 0px;" name="tag_name" size="1">
                <xsl:for-each select="$tag_names/tag">
                  <xsl:call-template name="opt">
                    <xsl:with-param name="value" select="name/text()"/>
                  </xsl:call-template>
                </xsl:for-each>
              </select>
            </div>

            <div class="form-group">
              <label class="control-label">
                <xsl:value-of select="gsa:i18n ('with Value', 'Tag')"/>:
              </label>
              <input type="text" class="form-control" name="tag_value"/>
            </div>
            <div class="form-group">
              <input type="image" src="/img/tag.svg" alt="{gsa:i18n ('Add Tag')}"
                name="Add Tag" value="Add Tag" title="{gsa:i18n ('Add Tag')}"
                class="icon icon-sm"/>
            </div>
            <xsl:choose>
              <xsl:when test="$resource_subtype!=''">
                <input type="hidden" name="resource_type" value="{$resource_subtype}"/>
              </xsl:when>
              <xsl:otherwise>
                <input type="hidden" name="resource_type" value="{$resource_type}"/>
              </xsl:otherwise>
            </xsl:choose>
            <input type="hidden" name="resource_id" value="{$resource_id}"/>
            <input type="hidden" name="next" value="{$next}"/>
            <xsl:choose>
              <xsl:when test="$resource_type='nvt'">
                <input type="hidden"
                        name="oid"
                        value="{$resource_id}"/>
              </xsl:when>
              <xsl:otherwise>
                <input type="hidden"
                        name="{$resource_type}_id"
                        value="{$resource_id}"/>
              </xsl:otherwise>
            </xsl:choose>
            <xsl:if test="$resource_type='info'">
              <input type="hidden"
                    name="details"
                    value="1"/>
            </xsl:if>
            <xsl:if test="$resource_subtype != ''">
              <input type="hidden"
                      name="{$resource_type}_type"
                      value="{$resource_subtype}"/>
            </xsl:if>
            <xsl:if test="$report_section != ''">
              <input type="hidden"
                      name="report_section"
                      value="{$report_section}"/>
            </xsl:if>
          </form>
        </div>
      </xsl:when>
      <xsl:otherwise/>
    </xsl:choose>
    <xsl:choose>
      <xsl:when test="$user_tags/count != 0">
        <table class="gbntable">
          <tr class="gbntablehead2">
            <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
            <td><xsl:value-of select="gsa:i18n ('Value')"/></td>
            <td><xsl:value-of select="gsa:i18n ('Comment')"/></td>
            <td style="width: {gsa:actions-width (3)}px"><xsl:value-of select="gsa:i18n ('Actions')"/></td>
          </tr>
          <xsl:apply-templates select="$user_tags/tag" mode="for_resource">
            <xsl:with-param name="resource_type" select="$resource_type"/>
            <xsl:with-param name="resource_subtype" select="$resource_subtype"/>
            <xsl:with-param name="resource_id"   select="$resource_id"/>
            <xsl:with-param name="next" select="$next"/>
            <xsl:with-param name="report_section" select="$report_section"/>
          </xsl:apply-templates>
        </table>
      </xsl:when>
      <xsl:otherwise/>
    </xsl:choose>
  </div>
</xsl:template>

<xsl:template name="user-tags-window">
  <xsl:param name="resource_type"/>
  <xsl:param name="resource_subtype"/>
  <xsl:param name="resource_id" select="@id"/>
  <xsl:param name="next" select="concat('get_',$resource_type)"/>
  <xsl:param name="report_section" select="''"/>
  <xsl:param name="user_tags" select="user_tags" />
  <xsl:param name="tag_names" select="../../get_tags_response"/>
  <xsl:if test="gsa:may-op ('get_tags')">
    <xsl:call-template name="user-tags-window-checked">
      <xsl:with-param name="resource_type" select="$resource_type"/>
      <xsl:with-param name="resource_subtype" select="$resource_subtype"/>
      <xsl:with-param name="resource_id" select="$resource_id"/>
      <xsl:with-param name="next" select="$next"/>
      <xsl:with-param name="report_section" select="$report_section"/>
      <xsl:with-param name="user_tags" select="$user_tags"/>
      <xsl:with-param name="tag_names" select="$tag_names"/>
    </xsl:call-template>
  </xsl:if>
</xsl:template>

<xsl:template match="tag" mode="for_resource">
  <xsl:param name="resource_type"/>
  <xsl:param name="resource_subtype"/>
  <xsl:param name="resource_id"/>
  <xsl:param name="next"/>
  <xsl:param name="report_section" select="''"/>

  <tr class="{gsa:table-row-class(position())}">
    <td>
      <a href="/omp?cmd=get_tag&amp;tag_id={@id}&amp;token={/envelope/token}"
          title="{gsa:i18n ('Tag Details')}">
        <xsl:value-of select="name"/>
      </a>
    </td>
    <td><xsl:value-of select="value"/></td>
    <td><xsl:value-of select="comment"/></td>
    <td class="table-actions">

      <xsl:call-template name="toggle-tag-icon">
        <xsl:with-param name="id" select="@id"/>
        <xsl:with-param name="enable" select="0"/>
        <xsl:with-param name="params">
          <input type="hidden" name="next" value="{$next}"/>
          <xsl:choose>
            <xsl:when test="$resource_type='nvt'">
              <input type="hidden" name="oid" value="{$resource_id}"/>
            </xsl:when>
            <xsl:otherwise>
              <input type="hidden" name="{concat($resource_type,'_id')}" value="{$resource_id}"/>
            </xsl:otherwise>
          </xsl:choose>
          <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
          <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
          <xsl:if test="$resource_subtype != ''">
            <input type="hidden"
                   name="{$resource_type}_type"
                   value="{$resource_subtype}"/>
          </xsl:if>
          <xsl:if test="$resource_type = 'info'">
            <input type="hidden"
                   name="details"
                   value="1"/>
          </xsl:if>
          <xsl:if test="$report_section != ''">
            <input type="hidden"
                    name="report_section"
                    value="{$report_section}"/>
          </xsl:if>
        </xsl:with-param>
        <xsl:with-param name="fragment" select="'#user_tags'"/>
      </xsl:call-template>

      <xsl:call-template name="trashcan-icon">
        <xsl:with-param name="type" select="'tag'"/>
        <xsl:with-param name="id" select="@id"/>
        <xsl:with-param name="params">
          <input type="hidden" name="next" value="{$next}"/>
          <xsl:choose>
            <xsl:when test="$resource_type='nvt'">
              <input type="hidden" name="oid" value="{$resource_id}"/>
            </xsl:when>
            <xsl:otherwise>
              <input type="hidden" name="{concat($resource_type,'_id')}" value="{$resource_id}"/>
            </xsl:otherwise>
          </xsl:choose>
          <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
          <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
          <xsl:if test="$resource_subtype != ''">
            <input type="hidden"
                   name="{$resource_type}_type"
                   value="{$resource_subtype}"/>
          </xsl:if>
          <xsl:if test="$resource_type = 'info'">
            <input type="hidden"
                   name="details"
                   value="1"/>
          </xsl:if>
          <xsl:if test="$report_section != ''">
            <input type="hidden"
                    name="report_section"
                    value="{$report_section}"/>
          </xsl:if>
        </xsl:with-param>
        <xsl:with-param name="fragment" select="'#user_tags'"/>
      </xsl:call-template>

      <xsl:choose>
        <xsl:when test="$report_section != ''">
          <a href="/omp?cmd=edit_tag&amp;tag_id={@id}&amp;next={$next}&amp;next_type={$resource_type}&amp;next_subtype={$resource_subtype}&amp;next_id={$resource_id}&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;report_section={$report_section}&amp;token={/envelope/token}"
             class="edit-action-icon icon icon-sm" data-type="tag" data-id="{@id}"
             title="{gsa:i18n ('Edit Tag')}">
            <img src="/img/edit.svg"/>
          </a>
        </xsl:when>
        <xsl:when test="$resource_subtype!=''">
          <a href="/omp?cmd=edit_tag&amp;tag_id={@id}&amp;next={$next}&amp;next_type={$resource_type}&amp;next_subtype={$resource_subtype}&amp;next_id={$resource_id}&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
             class="edit-action-icon icon icon-sm" data-type="tag" data-id="{@id}"
             title="{gsa:i18n ('Edit Tag')}">
            <img src="/img/edit.svg"/>
          </a>
        </xsl:when>
        <xsl:otherwise>
          <a href="/omp?cmd=edit_tag&amp;tag_id={@id}&amp;next={$next}&amp;next_type={$resource_type}&amp;next_subtype={$resource_subtype}&amp;next_id={$resource_id}&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
             class="edit-action-icon icon icon-sm" data-type="tag" data-id="{@id}"
             title="{gsa:i18n ('Edit Tag')}">
            <img src="/img/edit.svg"/>
          </a>
        </xsl:otherwise>
      </xsl:choose>
    </td>
  </tr>
</xsl:template>

<xsl:template name="toggle-tag-icon">
  <xsl:param name="id"></xsl:param>
  <xsl:param name="enable"></xsl:param>
  <xsl:param name="fragment"></xsl:param>
  <xsl:param name="params"></xsl:param>

  <xsl:if test="gsa:may-op ('modify_tag')">
    <div class="icon icon-sm ajax-post" data-reload="next" data-busy-text="{gsa:i18n ('Toggling Tag...')}">
      <xsl:choose>
        <xsl:when test="$enable">
          <img src="/img/enable.svg" alt="{gsa:i18n ('Enable Tag')}"
            name="Enable Tag" title="{gsa:i18n ('Enable Tag')}"/>
        </xsl:when>
        <xsl:otherwise>
          <img src="/img/disable.svg" alt="{gsa:i18n ('Disable Tag')}"
            name="Disable Tag" title="{gsa:i18n ('Disable Tag')}"/>
        </xsl:otherwise>
      </xsl:choose>
      <form action="/omp{$fragment}" method="post" enctype="multipart/form-data">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden" name="cmd" value="toggle_tag"/>
        <input type="hidden" name="enable" value="{$enable}"/>
        <input type="hidden" name="tag_id" value="{$id}"/>
        <xsl:copy-of select="$params"/>
      </form>
    </div>
  </xsl:if>
</xsl:template>

<xsl:template name="user_tag_list">
  <xsl:param name="user_tags" select="user_tags" />
  <xsl:for-each select="user_tags/tag">
    <a href="/omp?cmd=get_tag&amp;tag_id={@id}&amp;token={/envelope/token}">
      <xsl:value-of select="name"/>
      <xsl:if test="value != ''">=<xsl:value-of select="value"/></xsl:if>
    </a>
    <xsl:if test="position()!=last()"><xsl:text>, </xsl:text></xsl:if>
  </xsl:for-each>
</xsl:template>

<!-- Resource Permissions -->
<xsl:template name="resource-permissions-window">
  <xsl:param name="resource_type"/>
  <xsl:param name="resource_id" select="@id"/>
  <xsl:param name="next" select="concat('get_',$resource_type)"/>
  <xsl:param name="report_section" select="''"/>
  <!-- i18n with concat : see dynamic_strings.xsl - type-permissions -->
  <xsl:param name="permissions" select="../../permissions/get_permissions_response"/>
  <xsl:param name="related" select="''"/>
  <xsl:variable name="token" select="/envelope/token"/>
  <xsl:if test="gsa:may-op ('get_permissions')">
    <xsl:variable name="related_params">
      <xsl:for-each select="exslt:node-set ($related)/*">
        <xsl:text>related:</xsl:text>
        <xsl:value-of select="@id"/>
        <xsl:text>=</xsl:text>
        <xsl:value-of select="name(.)"/>
        <xsl:if test="position() != last()">
          <xsl:text>&amp;</xsl:text>
        </xsl:if>
      </xsl:for-each>
    </xsl:variable>

    <div class="section-header">
      <a href="#" class="toggle-action-icon icon icon-sm icon-action"
        data-target="#permission-box" data-name="Permissions"
        data-variable="permission-box--collapsed">
          <img src="/img/fold.svg"/>
      </a>
      <a href="/help/resource_permissions.html?token={/envelope/token}"
         class="icon icon-sm icon-action"
         title="Help: Resource Permissions">
        <img src="/img/help.svg"/>
      </a>
      <xsl:choose>
        <xsl:when test="gsa:may-op ('create_permission')">
          <a href="/omp?cmd=new_permissions&amp;next={$next}&amp;next_id={$resource_id}&amp;next_type={$resource_type}&amp;resource_id={$resource_id}&amp;restrict_type={$resource_type}&amp;{$related_params}token={/envelope/token}"
             class="new-action-icon icon icon-sm icon-action"
             data-reload="window"
             data-type="permissions"
             data-extra="resource_id={$resource_id}&amp;restrict_type={$resource_type}&amp;{$related_params}"
             title="{gsa:i18n ('Create Multiple Permissions')}">
            <img src="/img/new.svg"/>
          </a>
        </xsl:when>
        <xsl:otherwise/>
      </xsl:choose>
      <h2>
        <a href="/omp?cmd=get_permissions&amp;filter=name:^.*({$resource_type})s?$ and resource_uuid={$resource_id}&amp;token={/envelope/token}"
           title="{gsa:i18n ('Permissions')}">
          <img class="icon icon-sm" src="/img/permission.svg" alt="Permissions"/>
        </a>
        <xsl:value-of select="gsa:i18n ('Permissions')"/>
        <xsl:choose>
          <xsl:when test="$permissions/permission_count/filtered != 0">
            (<xsl:value-of select="$permissions/permission_count/filtered"/>)
          </xsl:when>
          <xsl:otherwise>
            (<xsl:value-of select="gsa:i18n ('none', 'Permissions')"/>)
          </xsl:otherwise>
        </xsl:choose>
      </h2>
    </div>

    <div class="section-box" id="permission-box">
      <table class="gbntable">
        <tr class="gbntablehead2">
          <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
          <td><xsl:value-of select="gsa:i18n ('Description')"/></td>
          <td><xsl:value-of select="gsa:i18n ('Resource Type')"/></td>
          <td><xsl:value-of select="gsa:i18n ('Resource')"/></td>
          <td><xsl:value-of select="gsa:i18n ('Subject Type', 'Permission')"/></td>
          <td><xsl:value-of select="gsa:i18n ('Subject', 'Permission')"/></td>
          <td style="width: {gsa:actions-width (4)}px"><xsl:value-of select="gsa:i18n ('Actions')"/></td>
        </tr>
        <xsl:apply-templates select="$permissions/permission">
          <xsl:with-param name="next" select="$next"/>
          <xsl:with-param name="next_type" select="$resource_type"/>
          <xsl:with-param name="next_id" select="$resource_id"/>
        </xsl:apply-templates>
      </table>
    </div>
  </xsl:if>
</xsl:template>

<!-- BEGIN REPORTS MANAGEMENT -->

<xsl:template match="sort">
</xsl:template>

<xsl:template match="apply_overrides">
</xsl:template>

<xsl:template match="all">
</xsl:template>

<xsl:template name="html-import-report-form">
  <div class="edit-dialog">
    <div class="title">
      <xsl:value-of select="gsa:i18n ('Import Report')"/>
    </div>
    <div class="content">
      <form action="/omp" method="post" enctype="multipart/form-data">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="import_report"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden" name="next" value="get_report"/>
        <xsl:if test="string-length (/envelope/params/filt_id) = 0">
          <input type="hidden" name="overrides" value="{/envelope/params/overrides}"/>
        </xsl:if>
        <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
        <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
        <table class="table-form">
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Report')"/></td>
            <td><input type="file" name="xml_file" size="30"/></td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Container Task')"/></td>
            <td>
              <xsl:variable name="task_id" select="/envelope/params/task_id"/>
              <select name="task_id">
                <xsl:for-each select="get_tasks_response/task[target/@id='']">
                  <xsl:choose>
                    <xsl:when test="@id = $task_id">
                      <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
                    </xsl:when>
                    <xsl:otherwise>
                      <option value="{@id}"><xsl:value-of select="name"/></option>
                    </xsl:otherwise>
                  </xsl:choose>
                </xsl:for-each>
              </select>
              <a href="#" title="{ gsa:i18n('Create a new container task') }"
                 class="icon icon-sm new-action-icon" data-type="container_task" data-done="select[name=task_id]">
                <img src="/img/new.svg" class="valign-middle"/>
              </a>
            </td>
          </tr>
          <xsl:if test="gsa:may-op ('create_asset')">
            <tr>
              <td><xsl:value-of select="gsa:i18n ('Add to Assets')"/></td>
              <td>
                <div>
                  <xsl:value-of select="gsa-i18n:strformat (gsa:i18n ('Add to Assets with QoD>=%1%% and Overrides enabled'), 70)"/>
                </div>
                <label>
                  <input type="radio" name="in_assets" value="1" checked="1"/>
                  <xsl:value-of select="gsa:i18n ('yes')"/>
                </label>
                <label>
                  <input type="radio" name="in_assets" value="0"/>
                  <xsl:value-of select="gsa:i18n ('no')"/>
                </label>
              </td>
            </tr>
          </xsl:if>
        </table>
      </form>
    </div>
  </div>
</xsl:template>

<xsl:template match="upload_report">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="create_report_response" mode="upload"/>
  <xsl:apply-templates select="commands_response/delete_report_response"/>
  <xsl:call-template name="html-import-report-form"/>
</xsl:template>

<xsl:template match="get_reports_response" mode="alert">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Run Alert</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template name="assets">
  <xsl:variable name="levels"
                select="report/filters/keywords/keyword[column='levels']/value"/>
  <xsl:variable name="apply-overrides"
                select="report/filters/keywords/keyword[column='apply_overrides']/value"/>
  <xsl:variable name="phrase"
                select="translate (report/filters/keywords/keyword[column = '']/value, '&quot;', '')"/>
  <xsl:if test="report/@scap_loaded = 0">
    <div>
      <xsl:call-template name="error_window">
        <xsl:with-param name="heading">Warning: SCAP Database Missing</xsl:with-param>
        <xsl:with-param name="message">
          SCAP database missing on OMP server.  Prognostic reporting disabled.
          <a href="/help/hosts_classic.html?token={/envelope/token}#scap_missing"
            title="Help: SCAP database missing"
            class="icon icon-sm">
            <img style="margin-left:5px" src="/img/help.svg"/>
          </a>
        </xsl:with-param>
      </xsl:call-template>
    </div>
  </xsl:if>
  <div class="gb_window">
    <div class="gb_window_part_left"></div>
    <div class="gb_window_part_right"></div>
    <div class="gb_window_part_center">
      <xsl:value-of select="gsa:i18n ('Host Filtering')"/>
      <a href="/help/hosts_classic.html?token={/envelope/token}" title="{concat(gsa:i18n('Help'),': ',gsa:i18n('Hosts'))}"
        class="icon icon-sm">
        <img src="/img/help.svg"/>
      </a>
      <div class="small_inline_form" style="display: inline; margin-left: 40px; font-weight: normal;">
        <form action="" method="get" name="switch_overrides" enctype="multipart/form-data">
          <input type="hidden" name="token" value="{/envelope/token}"/>
          <input type="hidden" name="cmd" value="get_report"/>
          <input type="hidden" name="type" value="assets"/>
          <input type="hidden" name="levels" value="{$levels}"/>
          <input type="hidden" name="search_phrase" value="{$phrase}"/>
          <!-- Switch back to the first page if the override state changes, because
               this could lead to changes in the number of hosts in the table. -->
          <input type="hidden" name="first_result" value="1"/>
          <input type="hidden" name="max_results" value="{report/hosts/@max}"/>
          <select style="margin-bottom: 0px;" name="apply_overrides" size="1" onchange="switch_overrides.submit ()">
            <xsl:choose>
              <xsl:when test="$apply-overrides = 0">
                <option value="0" selected="1">&#8730;<xsl:value-of select="gsa:i18n ('No Overrides')"/></option>
                <option value="1" ><xsl:value-of select="gsa:i18n ('Apply overrides')"/></option>
              </xsl:when>
              <xsl:otherwise>
                <option value="0"><xsl:value-of select="gsa:i18n ('No Overrides')"/></option>
                <option value="1" selected="1">&#8730;<xsl:value-of select="gsa:i18n ('Apply overrides')"/></option>
              </xsl:otherwise>
            </xsl:choose>
          </select>
          <input type="image"
                 name="{gsa:i18n ('Update', 'Action Verb')}"
                 src="/img/refresh.svg"
                 class="icon icon-sm"
                 title="{gsa:i18n ('Update', 'Action Verb')}"
                 alt="{gsa:i18n ('Update', 'Action Verb')}" style="margin-left:3px;margin-right:3px;"/>
        </form>
      </div>
    </div>
    <div class="gb_window_part_content">
      <div style="background-color: #EEEEEE;">
        <xsl:variable name="sort_field">
          <xsl:value-of select="report/sort/field/text()"/>
        </xsl:variable>
        <xsl:variable name="sort_order">
          <xsl:value-of select="report/sort/field/order"/>
        </xsl:variable>
        <form action="" method="get" enctype="multipart/form-data">
          <input type="hidden" name="token" value="{/envelope/token}"/>
          <input type="hidden" name="cmd" value="get_report"/>
          <input type="hidden" name="type" value="assets"/>
          <input type="hidden" name="sort_field" value="{$sort_field}"/>
          <input type="hidden" name="sort_order" value="{$sort_order}"/>
          <input type="hidden"
                 name="apply_overrides"
                 value="{$apply-overrides}"/>
          <div class="form-group">
            <xsl:value-of select="gsa:i18n ('Results per page')"/>:
            <input type="text" name="max_results" size="5"
                   value="{report/hosts/@max}"
                   maxlength="400"/>
          </div>
          <div class="form-group">
            <xsl:value-of select="gsa:i18n ('Text phrase')"/>:
            <input type="text" name="search_phrase" size="50"
                   value="{$phrase}"
                   maxlength="400"/>
          </div>
          <div class="pull-right">
            <input type="submit" value="{gsa:i18n ('Apply')}" title="{gsa:i18n ('Apply')}"/>
          </div>
          <div class="form-group">
            <xsl:value-of select="gsa:i18n ('Severity')"/>:
            <table style="display: inline">
              <tr>
                <td class="threat_info_table_h">
                  <label>
                    <xsl:choose>
                      <xsl:when test="report/filters/filter[text()='High']">
                        <input type="checkbox" name="level_high" value="1"
                               checked="1"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <input type="checkbox" name="level_high" value="1"/>
                      </xsl:otherwise>
                    </xsl:choose>
                    <xsl:call-template name="severity-label">
                      <xsl:with-param name="level" select="'High'"/>
                    </xsl:call-template>
                  </label>
                </td>
                <td class="threat_info_table_h">
                  <label>
                    <xsl:choose>
                      <xsl:when test="report/filters/filter[text()='Medium']">
                        <input type="checkbox" name="level_medium" value="1"
                               checked="1"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <input type="checkbox" name="level_medium" value="1"/>
                      </xsl:otherwise>
                    </xsl:choose>
                    <xsl:call-template name="severity-label">
                      <xsl:with-param name="level" select="'Medium'"/>
                    </xsl:call-template>
                  </label>
                </td>
                <td class="threat_info_table_h">
                  <label>
                    <xsl:choose>
                      <xsl:when test="report/filters/filter[text()='Low']">
                        <input type="checkbox" name="level_low" value="1"
                               checked="1"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <input type="checkbox" name="level_low" value="1"/>
                      </xsl:otherwise>
                    </xsl:choose>
                    <xsl:call-template name="severity-label">
                      <xsl:with-param name="level" select="'Low'"/>
                    </xsl:call-template>
                  </label>
                </td>
                <td class="threat_info_table_h">
                  <label>
                    <xsl:choose>
                      <xsl:when test="report/filters/filter[text()='Log']">
                        <input type="checkbox" name="level_log" value="1"
                               checked="1"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <input type="checkbox" name="level_log" value="1"/>
                      </xsl:otherwise>
                    </xsl:choose>
                    <xsl:call-template name="severity-label">
                      <xsl:with-param name="level" select="'Log'"/>
                    </xsl:call-template>
                  </label>
                </td>
              </tr>
            </table>
          </div>
        </form>
      </div>
    </div>
  </div>
  <div class="gb_window">
    <div class="gb_window_part_left"></div>
    <div class="gb_window_part_right"></div>
    <div class="gb_window_part_center">
      <xsl:value-of select="gsa:i18n ('Filtered Hosts')"/>
      <xsl:choose>
        <xsl:when test="count(report/host) &gt; 0">
          <xsl:variable name="last" select="report/hosts/@start + count(report/host) - 1"/>
          <xsl:choose>
            <xsl:when test = "report/hosts/@start &gt; 1">
              <a href="?cmd=get_report&amp;type=assets&amp;first_result=1&amp;max_results={report/hosts/@max}&amp;sort_field={report/sort/field/text()}&amp;sort_order={report/sort/field/order}&amp;overrides={report/filters/apply_overrides}&amp;search_phrase={report/filters/phrase}&amp;levels={$levels}&amp;search_phrase={report/filters/phrase}&amp;token={/envelope/token}"
                class="icon icon-sm">
                <img style="margin-left:10px;margin-right:3px;" src="/img/first.svg" title="{gsa:i18n ('First', 'Pagination')}"/>
              </a>
              <a href="?cmd=get_report&amp;type=assets&amp;first_result={report/hosts/@start - report/hosts/@max}&amp;max_results={report/hosts/@max}&amp;sort_field={report/sort/field/text()}&amp;sort_order={report/sort/field/order}&amp;overrides={report/filters/apply_overrides}&amp;search_phrase={report/filters/phrase}&amp;levels={$levels}&amp;search_phrase={report/filters/phrase}&amp;token={/envelope/token}"
                class="icon icon-sm">
                <img style="margin-right:3px;" src="/img/previous.svg" title="{gsa:i18n ('Previous', 'Pagination')}"/>
              </a>
            </xsl:when>
            <xsl:otherwise>
              <img class="icon icon-sm" style="margin-left:10px;margin-right:3px;" src="/img/first_inactive.svg" title="{gsa:i18n ('Already on first page', 'Pagination')}"/>
              <img class="icon icon-sm" style="margin-right:3px;" src="/img/previous_inactive.svg" title="{gsa:i18n ('Already on first page', 'Pagination')}"/>
            </xsl:otherwise>
          </xsl:choose>
          <xsl:value-of select="gsa-i18n:strformat (gsa:i18n ('%1 - %2 of %3'), report/hosts/@start, $last, report/host_count/filtered)"/>
          <xsl:choose>
            <xsl:when test = "$last &lt; report/host_count/filtered">
              <a href="?cmd=get_report&amp;type=assets&amp;first_result={report/hosts/@start + report/hosts/@max}&amp;max_results={report/hosts/@max}&amp;overrides={report/filters/apply_overrides}&amp;search_phrase={report/filters/phrase}&amp;levels={$levels}&amp;search_phrase={report/filters/phrase}&amp;token={/envelope/token}"
                class="icon icon-sm">
                <img style="margin-left:3px;" src="/img/next.svg" title="{gsa:i18n ('Next', 'Pagination')}"/>
              </a>
              <a href="?cmd=get_report&amp;type=assets&amp;first_result={floor((report/host_count/filtered - 1) div report/hosts/@max) * report/hosts/@max + 1}&amp;max_results={report/hosts/@max}&amp;overrides={report/filters/apply_overrides}&amp;search_phrase={report/filters/phrase}&amp;levels={$levels}&amp;search_phrase={report/filters/phrase}&amp;token={/envelope/token}"
                class="icon icon-sm">
                <img style="margin-left:3px;margin-right:10px;" src="/img/last.svg" title="{gsa:i18n ('Last', 'Pagination')}"/>
              </a>
            </xsl:when>
            <xsl:otherwise>
              <img class="icon icon-sm" style="margin-left:3px;" src="/img/next_inactive.svg" title="{gsa:i18n ('Already on last page', 'Pagination')}"/>
              <img class="icon icon-sm" style="margin-left:3px;margin-right:10px;" src="/img/last_inactive.svg" title="{gsa:i18n ('Already on last page', 'Pagination')}"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:when>
        <xsl:otherwise>
        </xsl:otherwise>
      </xsl:choose>
      <a href="/help/hosts_classic.html?token={/envelope/token}" title="{concat(gsa:i18n('Help'),': ',gsa:i18n('Hosts'))}"
        class="icon icon-sm">
        <img src="/img/help.svg"/>
      </a>
      <xsl:choose>
        <xsl:when test="count (report/host) = 0">
          <xsl:variable name="desc"
                        select="gsa:i18n ('No hosts available for Prognostic Report')"/>
          <img src="/img/prognosis_inactive.svg"
            class="icon icon-sm"
            title="{$desc}"
            alt="{$desc}"/>
        </xsl:when>
        <xsl:when test="report/@scap_loaded = 0">
          <xsl:variable name="desc"
                        select="gsa:i18n ('No SCAP data available for Prognostic Report')"/>
          <img src="/img/prognosis_inactive.svg"
            class="icon icon-sm"
            title="{$desc}"
            alt="{$desc}"/>
        </xsl:when>
        <xsl:otherwise>
          <a href="/omp?cmd=get_report&amp;type=prognostic&amp;pos=1&amp;host_search_phrase={report/filters/phrase}&amp;host_levels={gsa:build-levels(report/filters)}&amp;host_first_result={report/hosts/@start}&amp;host_max_results={report/hosts/@max}&amp;result_hosts_only=1&amp;overrides={$apply-overrides}&amp;token={/envelope/token}"
            class="icon icon-sm"
            title="{gsa:i18n ('Prognostic Report')}">
            <img src="/img/prognosis.svg" alt="{gsa:i18n ('Prognostic Report')}"/>
          </a>
        </xsl:otherwise>
      </xsl:choose>
    </div>
    <xsl:choose>
      <xsl:when test="count (report/host)=0">
        <div class="gb_window_part_content">
          <xsl:value-of select="gsa:i18n ('0 hosts')"/>
        </div>
      </xsl:when>
      <xsl:otherwise>
        <div class="gb_window_part_content">
          <xsl:apply-templates select="report" mode="assets"/>
        </div>
      </xsl:otherwise>
    </xsl:choose>
  </div>
</xsl:template>

<xsl:template match="report" mode="sorting-link">
  <xsl:param name="field"/>
  <xsl:param name="order"/>
  <xsl:param name="levels"/>
  <xsl:param name="name"><xsl:value-of select="$field"/></xsl:param>

  <xsl:choose>
    <xsl:when test="sort/field/text() = $field and sort/field/order = $order">
      <xsl:value-of select="concat($name, ' ', $order)"/>
    </xsl:when>
    <xsl:otherwise>
        <a href="/omp?cmd=get_report&amp;report_id={@id}&amp;delta_report_id={delta/report/@id}&amp;delta_states={filters/delta/text()}&amp;sort_field={$field}&amp;sort_order={$order}&amp;max_results={results/@max}&amp;levels={$levels}&amp;notes={filters/notes}&amp;details={/envelope/params/details}&amp;overrides={filters/overrides}&amp;result_hosts_only={filters/result_hosts_only}&amp;autofp={filters/autofp}&amp;token={/envelope/token}">
        <xsl:value-of select="concat($name, ' ', $order)"/>
      </a>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="result-details-icon-img">
  <xsl:param name="details"/>
  <xsl:choose>
    <xsl:when test="$details = 1">
      <img src="/img/fold.svg"
        class="icon icon-sm"
        alt="{gsa:i18n ('Collapse details of all vulnerabilities')}"
        title="{gsa:i18n ('Collapse details of all vulnerabilities')}"/>
    </xsl:when>
    <xsl:otherwise>
      <img src="/img/unfold.svg"
        class="icon icon-sm"
        alt="{gsa:i18n ('Expand to full details of all vulnerabilities')}"
        title="{gsa:i18n ('Expand to full details of all vulnerabilities')}"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="report" mode="result-details-icon">
  <xsl:variable name="details">
    <xsl:choose>
      <xsl:when test="/envelope/params/details &gt; 0">1</xsl:when>
      <xsl:otherwise>0</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:variable name="filter_term">
    <xsl:choose>
      <xsl:when test="/envelope/params/cmd='get_report_section' and /envelope/params/report_section != 'results'">
        <xsl:value-of select="/envelope/params/filter"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="filters/term"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>

  <xsl:variable name="host" select="/envelope/params/host"/>
  <xsl:variable name="pos" select="/envelope/params/pos"/>
  <xsl:variable name="delta" select="delta/report/@id"/>

  <xsl:variable name="expand" select="($details - 1)*($details - 1)"/>
  <xsl:variable name="apply_filter" select="/envelope/params/apply_filter"/>
  <xsl:variable name="link">
    <xsl:choose>
      <xsl:when test="@type='prognostic'">
        <xsl:value-of select="concat('/omp?cmd=get_report&amp;type=prognostic&amp;host=', $host, '&amp;pos=',$pos ,'&amp;details=', $expand, '&amp;apply_filter=', $apply_filter, '&amp;filter=', $filter_term, '&amp;filt_id=', /envelope/params/filt_id, '&amp;token=', /envelope/token)"/>
      </xsl:when>
      <xsl:when test="@type='delta'">
        <xsl:value-of select="concat('/omp?cmd=get_report&amp;report_id=', @id, '&amp;delta_report_id=', $delta, '&amp;details=', $expand, '&amp;apply_filter=', $apply_filter, '&amp;filter=', $filter_term, '&amp;filt_id=', /envelope/params/filt_id, '&amp;token=', /envelope/token)"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="concat('/omp?cmd=get_report&amp;report_id=', @id, '&amp;details=', $expand, '&amp;apply_filter=', $apply_filter, '&amp;filter=', $filter_term, '&amp;filt_id=', /envelope/params/filt_id, '&amp;token=', /envelope/token)"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:variable name="title">
    <xsl:choose>
      <xsl:when test="$expand=1">
        <xsl:value-of select="'Expand to full details of all vulnerabilities'"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="'Collapse details of all vulnerabilities'"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <a href="{$link}" title="{$title}">
     <xsl:call-template name="result-details-icon-img">
       <xsl:with-param name="details" select="$details"/>
     </xsl:call-template>
  </a>
</xsl:template>

<xsl:template name="result-overrides-icon-img">
  <xsl:param name="overrides"/>
  <xsl:choose>
    <xsl:when test="$overrides = 1">
      <img src="/img/overrides_enabled.svg"
        alt="{gsa:i18n('Overrides are Applied')}"
        value="Overrides are Applied"
        title="{gsa:i18n('Overrides are Applied')}"
        class="icon icon-sm"/>
    </xsl:when>
    <xsl:otherwise>
      <img src="/img/overrides_disabled.svg"
        alt="{gsa:i18n('No Overrides')}"
        value="No Overrides"
        title="{gsa:i18n('No Overrides')}"
        class="icon icon-sm"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="report" mode="result-overrides-icon">
  <xsl:variable name="details">
    <xsl:choose>
      <xsl:when test="/envelope/params/details &gt; 0">1</xsl:when>
      <xsl:otherwise>0</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>

  <xsl:variable name="overrides">
    <xsl:choose>
      <xsl:when test="filters/keywords/keyword[column = 'apply_overrides']/value = 0">0</xsl:when>
      <xsl:otherwise>1</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:variable name="link-overrides">
    <xsl:choose>
      <xsl:when test="$overrides = 0">1</xsl:when>
      <xsl:otherwise>0</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:variable name="host" select="/envelope/params/host"/>
  <xsl:variable name="pos" select="/envelope/params/pos"/>
  <xsl:variable name="delta" select="delta/report/@id"/>
  <xsl:variable name="apply_filter" select="/envelope/params/apply_filter"/>

  <xsl:variable name="link">
    <xsl:choose>
      <xsl:when test="@type='prognostic'">
        <xsl:value-of select="concat('/omp?cmd=get_report&amp;type=prognostic&amp;host=', $host, '&amp;pos=',$pos ,'&amp;details=', $details, '&amp;apply_filter=', $apply_filter, '&amp;filter=apply_overrides=', $link-overrides, ' ', filters/term, '&amp;filt_id=', /envelope/params/filt_id, '&amp;token=', /envelope/token)"/>
      </xsl:when>
      <xsl:when test="@type='delta'">
        <xsl:value-of select="concat('/omp?cmd=get_report&amp;report_id=', @id, '&amp;delta_report_id=', $delta, '&amp;details=', $details, '&amp;apply_filter=', $apply_filter, '&amp;filter=apply_overrides=', $link-overrides, ' ', filters/term, '&amp;token=', /envelope/token)"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="concat('/omp?cmd=get_report&amp;report_id=', @id, '&amp;details=', $details, '&amp;apply_filter=', $apply_filter, '&amp;filter=apply_overrides=', $link-overrides, ' ', filters/term, '&amp;filt_id=&amp;token=', /envelope/token)"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:variable name="title">
    <xsl:choose>
      <xsl:when test="$overrides='1'">
        <xsl:value-of select="'Overrides are Applied'"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="'No Overrides'"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <a href="{$link}" title="{$title}">
    <xsl:call-template name="result-overrides-icon-img">
      <xsl:with-param name="overrides" select="$overrides"/>
    </xsl:call-template>
  </a>
</xsl:template>

<xsl:template match="report" mode="filterbox">
  <input type="hidden" name="build_filter" value="0"/>
  <div id="filterbox" style="display: none;">
    <div style="background-color: #EEEEEE;">
      <xsl:choose>
        <xsl:when test="/envelope/params/report_section != ''">
          <input type="hidden" name="report_section" value="{/envelope/params/report_section}"/>
          <input type="hidden" name="cmd" value="get_report_section"/>
        </xsl:when>
        <xsl:otherwise>
          <input type="hidden" name="cmd" value="get_report"/>
        </xsl:otherwise>
      </xsl:choose>
      <input type="hidden" name="report_id" value="{report/@id}"/>
      <xsl:if test="@type='prognostic'">
        <input type="hidden" name="type" value="prognostic"/>
        <input type="hidden" name="host" value="{report/filters/host}"/>
        <input type="hidden" name="host_search_phrase" value="{../../host_search_phrase}"/>
        <input type="hidden" name="host_levels" value="{../../host_levels}"/>
        <input type="hidden" name="host_first_result" value="{../../results/@start}"/>
        <input type="hidden" name="host_max_results" value="{../../results/@max}"/>
        <input type="hidden" name="pos" value="{/envelope/params/pos}"/>
      </xsl:if>
      <input type="hidden" name="details" value="{/envelope/params/details}"/>
      <input type="hidden" name="token" value="{/envelope/token}"/>
      <xsl:if test="../../delta">
        <input type="hidden" name="delta_report_id" value="{report/delta/report/@id}"/>
        <div class="pull-right">
          <div class="form-group"><xsl:value-of select="gsa:i18n ('Show delta results')"/>:</div>
          <div style="margin-left: 8px;">
            <label>
              <xsl:choose>
                <xsl:when test="report/filters/delta/same = 0">
                  <input type="checkbox" name="delta_state_same" value="1"/>
                </xsl:when>
                <xsl:otherwise>
                  <input type="checkbox" name="delta_state_same"
                          value="1" checked="1"/>
                </xsl:otherwise>
              </xsl:choose>
              = <xsl:value-of select="gsa:i18n ('same', 'Delta Result')"/>
            </label>
          </div>
          <div style="margin-left: 8px;">
            <label>
              <xsl:choose>
                <xsl:when test="report/filters/delta/new = 0">
                  <input type="checkbox" name="delta_state_new" value="1"/>
                </xsl:when>
                <xsl:otherwise>
                  <input type="checkbox" name="delta_state_new"
                          value="1" checked="1"/>
                </xsl:otherwise>
              </xsl:choose>
              + <xsl:value-of select="gsa:i18n ('new', 'Delta Result')"/>
            </label>
          </div>
          <div style="margin-left: 8px;">
            <label>
              <xsl:choose>
                <xsl:when test="report/filters/delta/gone = 0">
                  <input type="checkbox" name="delta_state_gone" value="1"/>
                </xsl:when>
                <xsl:otherwise>
                  <input type="checkbox" name="delta_state_gone"
                          value="1" checked="1"/>
                </xsl:otherwise>
              </xsl:choose>
              &#8722; <xsl:value-of select="gsa:i18n ('gone', 'Delta Result')"/>
            </label>
          </div>
          <div style="margin-left: 8px;">
            <label>
              <xsl:choose>
                <xsl:when test="report/filters/delta/changed = 0">
                  <input type="checkbox" name="delta_state_changed" value="1"/>
                </xsl:when>
                <xsl:otherwise>
                  <input type="checkbox" name="delta_state_changed"
                          value="1" checked="1"/>
                </xsl:otherwise>
              </xsl:choose>
              ~ <xsl:value-of select="gsa:i18n ('changed', 'Delta Result')"/>
            </label>
          </div>
        </div>
      </xsl:if>

      <xsl:if test="not (/envelope/params/report_section) or /envelope/params/report_section = 'results'">
        <div class="form-group">
          <xsl:value-of select="gsa:i18n ('Results per page')"/>:
          <input type="text" name="max_results" size="5"
                value="{report/results/@max}"
                maxlength="400"/>
        </div>
      </xsl:if>

      <div class="form-group">
        <label>
          <xsl:choose>
            <xsl:when test="report/filters/keywords/keyword[column = 'apply_overrides']/value = 0">
              <input type="checkbox" name="apply_overrides" value="1"/>
            </xsl:when>
            <xsl:otherwise>
              <input type="checkbox" name="apply_overrides" value="1" checked="1"/>
            </xsl:otherwise>
          </xsl:choose>
          <xsl:value-of select="gsa:i18n ('Apply Overrides')"/>
        </label>
      </div>

      <xsl:choose>
        <xsl:when test="@type='prognostic'">
        </xsl:when>
        <xsl:otherwise>
          <div class="form-group">
            <xsl:value-of select="gsa:i18n ('Auto-FP')"/>:
            <div style="margin-left: 30px">
              <label>
                <xsl:choose>
                  <xsl:when test="report/filters/keywords/keyword[column = 'autofp']/value = 0">
                    <input type="checkbox" name="autofp" value="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="checkbox" name="autofp" value="1" checked="1"/>
                  </xsl:otherwise>
                </xsl:choose>
                <xsl:value-of select="gsa:i18n ('Trust vendor security updates')"/>
              </label>
              <div style="margin-left: 30px">
                <label>
                  <xsl:choose>
                    <xsl:when test="report/filters/keywords/keyword[column = 'autofp']/value = 2">
                      <input type="radio" name="autofp_value" value="1"/>
                    </xsl:when>
                    <xsl:otherwise>
                      <input type="radio" name="autofp_value" value="1" checked="1"/>
                    </xsl:otherwise>
                  </xsl:choose>
                  <xsl:value-of select="gsa:i18n ('Full CVE match')"/>
                </label>
                <label>
                  <xsl:choose>
                    <xsl:when test="report/filters/keywords/keyword[column = 'autofp']/value = 2">
                      <input type="radio" name="autofp_value" value="2" checked="1"/>
                    </xsl:when>
                    <xsl:otherwise>
                      <input type="radio" name="autofp_value" value="2"/>
                    </xsl:otherwise>
                  </xsl:choose>
                  <xsl:value-of select="gsa:i18n ('Partial CVE match')"/>
                </label>
              </div>
            </div>
          </div>
        </xsl:otherwise>
      </xsl:choose>

      <xsl:choose>
        <xsl:when test="@type='prognostic'">
        </xsl:when>
        <xsl:otherwise>
          <div class="form-group">
            <label>
              <xsl:choose>
                <xsl:when test="report/filters/keywords/keyword[column = 'notes']/value = 0">
                  <input type="checkbox" name="notes" value="1"/>
                </xsl:when>
                <xsl:otherwise>
                  <input type="checkbox" name="notes" value="1" checked="1"/>
                </xsl:otherwise>
              </xsl:choose>
              <xsl:value-of select="gsa:i18n ('Show Notes')"/>
            </label>
          </div>
        </xsl:otherwise>
      </xsl:choose>

      <div class="form-group">
        <label>
          <xsl:choose>
            <xsl:when test="report/filters/keywords/keyword[column = 'overrides']/value = 0">
              <input type="checkbox" name="overrides" value="1"/>
            </xsl:when>
            <xsl:otherwise>
              <input type="checkbox" name="overrides" value="1" checked="1"/>
            </xsl:otherwise>
          </xsl:choose>
          <xsl:value-of select="gsa:i18n ('Show Overrides')"/>
        </label>
      </div>

      <div class="form-group">
        <xsl:choose>
          <xsl:when test="report/filters/keywords/keyword[column = 'result_hosts_only']/value = 0">
            <label>
              <input type="checkbox" name="result_hosts_only" value="1"/>
              <xsl:value-of select="gsa:i18n ('Only show hosts that have results')"/>
            </label>
          </xsl:when>
          <xsl:otherwise>
            <label>
              <input type="checkbox" name="result_hosts_only" value="1" checked="1"/>
              <xsl:value-of select="gsa:i18n ('Only show hosts that have results')"/>
            </label>
          </xsl:otherwise>
        </xsl:choose>
      </div>
      <div class="form-group">
        <label>
          QoD &gt;=
        </label>
        <select name="min_qod">
          <xsl:call-template name="opt">
            <xsl:with-param name="value" select="'100'"/>
            <xsl:with-param name="select-value" select="report/filters/keywords/keyword[column = 'min_qod']/value"/>
          </xsl:call-template>
          <xsl:call-template name="opt">
            <xsl:with-param name="value" select="'90'"/>
            <xsl:with-param name="select-value" select="report/filters/keywords/keyword[column = 'min_qod']/value"/>
          </xsl:call-template>
          <xsl:call-template name="opt">
            <xsl:with-param name="value" select="'80'"/>
            <xsl:with-param name="select-value" select="report/filters/keywords/keyword[column = 'min_qod']/value"/>
          </xsl:call-template>
          <xsl:choose>
            <xsl:when test="not (report/filters/keywords/keyword[column = 'min_qod']/value != '')">
              <xsl:call-template name="opt">
                <xsl:with-param name="value" select="'70'"/>
                <xsl:with-param name="select-value" select="'70'"/>
              </xsl:call-template>
            </xsl:when>
            <xsl:otherwise>
              <xsl:call-template name="opt">
                <xsl:with-param name="value" select="'70'"/>
                <xsl:with-param name="select-value" select="report/filters/keywords/keyword[column = 'min_qod']/value"/>
              </xsl:call-template>
            </xsl:otherwise>
          </xsl:choose>
          <xsl:call-template name="opt">
            <xsl:with-param name="value" select="'60'"/>
            <xsl:with-param name="select-value" select="report/filters/keywords/keyword[column = 'min_qod']/value"/>
          </xsl:call-template>
          <xsl:call-template name="opt">
            <xsl:with-param name="value" select="'50'"/>
            <xsl:with-param name="select-value" select="report/filters/keywords/keyword[column = 'min_qod']/value"/>
          </xsl:call-template>
          <xsl:call-template name="opt">
            <xsl:with-param name="value" select="'40'"/>
            <xsl:with-param name="select-value" select="report/filters/keywords/keyword[column = 'min_qod']/value"/>
          </xsl:call-template>
          <xsl:call-template name="opt">
            <xsl:with-param name="value" select="'30'"/>
            <xsl:with-param name="select-value" select="report/filters/keywords/keyword[column = 'min_qod']/value"/>
          </xsl:call-template>
          <xsl:call-template name="opt">
            <xsl:with-param name="value" select="'20'"/>
            <xsl:with-param name="select-value" select="report/filters/keywords/keyword[column = 'min_qod']/value"/>
          </xsl:call-template>
          <xsl:call-template name="opt">
            <xsl:with-param name="value" select="'10'"/>
            <xsl:with-param name="select-value" select="report/filters/keywords/keyword[column = 'min_qod']/value"/>
          </xsl:call-template>
          <xsl:call-template name="opt">
            <xsl:with-param name="value" select="'0'"/>
            <xsl:with-param name="select-value" select="report/filters/keywords/keyword[column = 'min_qod']/value"/>
          </xsl:call-template>
        </select>
        %
      </div>
      <div class="form-group">
        <xsl:value-of select="gsa:i18n ('Timezone')"/>:
        <xsl:call-template name="timezone-select">
          <xsl:with-param name="timezone" select="report/timezone"/>
          <xsl:with-param name="input-name" select="'timezone'"/>
        </xsl:call-template>
      </div>
      <div class="pull-right">
        <input type="submit" value="{gsa:i18n ('Apply')}" title="{gsa:i18n ('Apply')}"/>
      </div>
      <div class="form-group">
        <xsl:value-of select="gsa:i18n ('Severity')"/>:
        <table style="display: inline">
          <tr>
            <td class="threat_info_table_h">
              <label>
                <xsl:choose>
                  <xsl:when test="report/filters/filter[text()='High']">
                    <input type="checkbox" name="level_high" value="1"
                            checked="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="checkbox" name="level_high" value="1"/>
                  </xsl:otherwise>
                </xsl:choose>
                <xsl:call-template name="severity-label">
                  <xsl:with-param name="level" select="'High'"/>
                </xsl:call-template>
              </label>
            </td>
            <td class="threat_info_table_h">
              <label>
                <xsl:choose>
                  <xsl:when test="report/filters/filter[text()='Medium']">
                    <input type="checkbox" name="level_medium" value="1"
                            checked="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="checkbox" name="level_medium" value="1"/>
                  </xsl:otherwise>
                </xsl:choose>
                <xsl:call-template name="severity-label">
                  <xsl:with-param name="level" select="'Medium'"/>
                </xsl:call-template>
              </label>
            </td>
            <td class="threat_info_table_h">
              <label>
                <xsl:choose>
                  <xsl:when test="report/filters/filter[text()='Low']">
                    <input type="checkbox" name="level_low" value="1"
                            checked="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="checkbox" name="level_low" value="1"/>
                  </xsl:otherwise>
                </xsl:choose>
                <xsl:call-template name="severity-label">
                  <xsl:with-param name="level" select="'Low'"/>
                </xsl:call-template>
              </label>
            </td>
            <td class="threat_info_table_h">
              <label>
                <xsl:choose>
                  <xsl:when test="report/filters/filter[text()='Log']">
                    <input type="checkbox" name="level_log" value="1"
                            checked="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="checkbox" name="level_log" value="1"/>
                  </xsl:otherwise>
                </xsl:choose>
                <xsl:call-template name="severity-label">
                  <xsl:with-param name="level" select="'Log'"/>
                </xsl:call-template>
              </label>
            </td>
            <td class="threat_info_table_h">
              <label>
                <xsl:choose>
                  <xsl:when test="report/filters/filter[text()='False Positive']">
                    <input type="checkbox"
                            name="level_false_positive"
                            value="1"
                            checked="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="checkbox"
                            name="level_false_positive"
                            value="1"/>
                  </xsl:otherwise>
                </xsl:choose>
                <xsl:call-template name="severity-label">
                  <xsl:with-param name="level" select="'False Positive'"/>
                </xsl:call-template>
              </label>
            </td>
          </tr>
        </table>
      </div>
    </div>
  </div>
</xsl:template>

<xsl:template match="report" mode="section-filter-restricted">
  <xsl:param name="report_section" select="'results'"/>
  <xsl:param name="extra_params" select="''"/>

  <xsl:variable name="filter_term" select="/envelope/params/filter"/>
  <xsl:variable name="apply-overrides"
                select="filters/apply_overrides"/>

  <div id="list-window-filter" class="col-8">
    <form name="filterform" method="get" action="" enctype="multipart/form-data" class="pull-right">
      <input type="hidden" name="token" value="{/envelope/token}"/>
      <input type="hidden" name="cmd" value="get_report_section"/>
      <input type="hidden" name="report_id" value="{@id}"/>
      <input type="hidden" name="report_section" value="{$report_section}"/>
      <input type="hidden" name="overrides" value="{$apply-overrides}"/>
      <input type="hidden" name="details" value="{/envelope/params/details}"/>
      <xsl:if test="@type='prognostic'">
        <input type="hidden" name="type" value="prognostic"/>
        <input type="hidden" name="host" value="{filters/host}"/>
        <input type="hidden" name="host_search_phrase" value="{/envelope/params/host_search_phrase}"/>
        <input type="hidden" name="host_levels" value="{/envelope/params/host_levels}"/>
        <input type="hidden" name="host_first_result" value="{/envelope/params/host_first_result}"/>
        <input type="hidden" name="host_max_results" value="{/envelope/params/host_max_results}"/>
        <input type="hidden" name="pos" value="{/envelope/params/pos}"/>
      </xsl:if>
      <xsl:if test="@type='delta'">
        <input type="hidden" name="delta_report_id" value="{delta/report/@id}"/>
      </xsl:if>
      <select name="apply_filter" style="min-width:250px">
        <xsl:choose>
          <xsl:when test="/envelope/params/apply_filter = 'no_pagination' or not(/envelope/params/apply_filter != '')">
            <option value="no_pagination" selected="1">&#8730;<xsl:value-of select="gsa:i18n ('Use filtered results (all pages)')"/></option>
          </xsl:when>
          <xsl:otherwise>
            <option value="no_pagination"><xsl:value-of select="gsa:i18n ('Use filtered results (all pages)')"/></option>
          </xsl:otherwise>
        </xsl:choose>
        <xsl:choose>
          <xsl:when test="/envelope/params/apply_filter = 'no'">
            <option value="no" selected="1">&#8730;<xsl:value-of select="gsa:i18n ('Use all unfiltered results')"/></option>
          </xsl:when>
          <xsl:otherwise>
            <option value="no"><xsl:value-of select="gsa:i18n ('Use all unfiltered results')"/></option>
          </xsl:otherwise>
        </xsl:choose>
        <xsl:choose>
          <xsl:when test="/envelope/params/apply_filter = 'full' or /envelope/params/apply_filter = ''">
            <option value="full" selected="1">&#8730;<xsl:value-of select="gsa:i18n ('Use filtered results (current page)')"/></option>
          </xsl:when>
          <xsl:otherwise>
            <option value="full"><xsl:value-of select="gsa:i18n ('Use filtered results (current page)')"/></option>
          </xsl:otherwise>
        </xsl:choose>
      </select>
      <xsl:text> </xsl:text>
      <xsl:choose>
        <xsl:when test="/envelope/params/apply_filter = 'no'">
          <input type="text" name="filter" size="53"
                  value="{$filter_term}" style="color:silver"
                  maxlength="1000"/>
        </xsl:when>
        <xsl:otherwise>
          <input type="text" name="filter" size="53"
                  value="{$filter_term}"
                  maxlength="1000"/>
        </xsl:otherwise>
      </xsl:choose>
      <input type="image"
        name="Update Filter"
        title="{gsa:i18n ('Update Filter')}"
        src="/img/refresh.svg"
        class="icon icon-sm"
        alt="{gsa:i18n ('Update', 'Action Verb')}" style="vertical-align:middle;margin-left:3px;margin-right:3px;"/>
      <a href="/help/powerfilter.html?token={/envelope/token}" title="{gsa:i18n ('Help')}: {gsa:i18n ('Powerfilter')}"
        class="icon icon-sm">
        <img src="/img/help.svg"/>
      </a>
    </form>
  </div>
</xsl:template>

<xsl:template match="report" mode="section-filter-full">
  <xsl:param name="report_section" select="'results'"/>
  <xsl:param name="extra_params" select="''"/>

  <div id="list-window-filter" class="col-8">
    <xsl:call-template name="filter-window-part">
      <xsl:with-param name="type" select="'report_result'"/>
      <xsl:with-param name="subtype" select="''"/>
      <xsl:with-param name="list" select="report/results"/>
      <xsl:with-param name="full-count" select="result_count/full/text ()"/>
      <xsl:with-param name="columns" xmlns="">
        <column>
          <name><xsl:value-of select="gsa:i18n('Vulnerability')"/></name>
          <field>vulnerability</field>
        </column>
        <column>
          <name><xsl:value-of select="gsa:i18n('Solution type')"/></name>
          <field>solution_type</field>
        </column>
        <column>
          <name><xsl:value-of select="gsa:i18n('Severity')"/></name>
          <field>severity</field>
        </column>
        <column>
          <name><xsl:value-of select="gsa:i18n('QoD')"/></name>
          <field>qod</field>
        </column>
        <column>
          <name><xsl:value-of select="gsa:i18n('Host')"/></name>
          <field>host</field>
        </column>
        <column>
          <name><xsl:value-of select="gsa:i18n('Location', 'Host')"/></name>
          <field>location</field>
        </column>
      </xsl:with-param>
      <xsl:with-param name="filter_options" xmlns="">
        <xsl:if test="delta">
          <option>delta_states</option>
        </xsl:if>
        <option>apply_overrides</option>
        <option>autofp</option>
        <option>notes</option>
        <option>overrides</option>
        <option>result_hosts_only</option>
        <option>min_qod</option>
        <option>timezone</option>
        <option>levels</option>
        <option>first</option>
        <option>rows</option>
      </xsl:with-param>
      <xsl:with-param name="extra_params" xmlns="">
        <xsl:copy-of select="$extra_params"/>
        <param>
          <name>report_id</name>
          <value><xsl:value-of select="@id"/></value>
        </param>
        <param>
          <name>report_section</name>
          <value><xsl:value-of select="$report_section"/></value>
        </param>
        <xsl:if test="../@type != '' and ../@type != 'scan'">
          <param>
            <name>type</name>
            <value><xsl:value-of select="../@type"/></value>
          </param>
        </xsl:if>
        <xsl:if test="../@type='prognostic'">
          <param>
            <name>host</name>
            <value><xsl:value-of select="filters/host"/></value>
          </param>
          <param>
            <name>host_search_phrase</name>
            <value><xsl:value-of select="/envelope/params/host_search_phrase"/></value>
          </param>
          <param>
            <name>host_levels</name>
            <value><xsl:value-of select="/envelope/params/host_levels"/></value>
          </param>
          <param>
            <name>host_first_result</name>
            <value><xsl:value-of select="/envelope/params/host_first_result"/></value>
          </param>
          <param>
            <name>host_max_results</name>
            <value><xsl:value-of select="/envelope/params/host_max_results"/></value>
          </param>
          <param>
            <name>pos</name>
            <value><xsl:value-of select="/envelope/params/pos"/></value>
          </param>
        </xsl:if>
        <xsl:if test="delta/report/@id">
          <param>
            <name>delta_report_id</name>
            <value><xsl:value-of select="delta/report/@id"/></value>
          </param>
        </xsl:if>
      </xsl:with-param>
      <xsl:with-param name="filters" select="../../../filters"/>
      <xsl:with-param name="report_section" select="$report_section"/>
    </xsl:call-template>
  </div>
</xsl:template>

<xsl:template match="report" mode="section-pager">
  <xsl:param name="section"/>
  <xsl:param name="count"/>
  <xsl:param name="filtered-count"/>
  <xsl:param name="full-count"/>

  <xsl:call-template name="filter-window-pager">
    <xsl:with-param name="type" select="'report_result'"/>
    <xsl:with-param name="list" select="results"/>
    <xsl:with-param name="count" select="$count"/>
    <xsl:with-param name="filtered_count" select="$filtered-count"/>
    <xsl:with-param name="full_count" select="$full-count"/>
    <xsl:with-param name="extra_params">
      <xsl:text>&amp;report_id=</xsl:text><xsl:value-of select="@id"/>
      <xsl:text>&amp;report_section=</xsl:text><xsl:value-of select="$section"/>
      <xsl:text>&amp;apply_overrides=</xsl:text><xsl:value-of select="/envelope/params/apply_overrides"/>
      <xsl:text>&amp;details=</xsl:text><xsl:value-of select="/envelope/params/details"/>
      <xsl:if test="@type='prognostic'">
        <xsl:text>&amp;type=</xsl:text><xsl:value-of select="'prognostic'"/>
        <xsl:text>&amp;host=</xsl:text><xsl:value-of select="filters/host"/>
        <xsl:text>&amp;host_search_phrase=</xsl:text><xsl:value-of select="/envelope/params/host_search_phrase"/>
        <xsl:text>&amp;host_levels=</xsl:text><xsl:value-of select="/envelope/params/host_levels"/>
        <xsl:text>&amp;host_first_result=</xsl:text><xsl:value-of select="/envelope/params/host_first_result"/>
        <xsl:text>&amp;host_max_results=</xsl:text><xsl:value-of select="/envelope/params/host_max_results"/>
        <xsl:text>&amp;pos=</xsl:text><xsl:value-of select="/envelope/params/pos"/>
      </xsl:if>
      <xsl:if test="@type='delta'">
        <xsl:text>&amp;delta_report_id=</xsl:text><xsl:value-of select="delta/report/@id"/>
      </xsl:if>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="report" mode="report-section-toolbar">
  <xsl:param name="section" select="'results'"/>
  <xsl:param name="extra_params" select="''"/>

  <div class="toolbar">

    <div class="col-4">
      <xsl:call-template name="report-icons">
        <xsl:with-param name="section" select="$section"/>
      </xsl:call-template>

      <span class="divider"/>

      <!-- Status bar -->
      <xsl:choose>
        <xsl:when test="not(/envelope/params/delta_report_id != '')">
          <div class="report-section-severity">
            <xsl:apply-templates select="report" mode="section-link">
              <xsl:with-param name="section" select="'summary'"/>
              <xsl:with-param name="type">
                <xsl:choose>
                  <xsl:when test="@type"><xsl:value-of select="@type"/></xsl:when>
                  <xsl:otherwise>normal</xsl:otherwise>
                </xsl:choose>
              </xsl:with-param>
              <xsl:with-param name="count" select="-1"/>
              <xsl:with-param name="link_style" select="'element'"/>

              <xsl:with-param name="element">
                <xsl:call-template name="status_bar">
                  <xsl:with-param name="status">
                    <xsl:choose>
                      <xsl:when test="report/task/target/@id='' and report/scan_run_status='Running'">
                        <xsl:text>Uploading</xsl:text>
                      </xsl:when>
                      <xsl:when test="report/task/target/@id=''">
                        <xsl:text>Container</xsl:text>
                      </xsl:when>
                      <xsl:otherwise>
                        <xsl:value-of select="report/scan_run_status"/>
                      </xsl:otherwise>
                    </xsl:choose>
                  </xsl:with-param>
                  <xsl:with-param name="progress">
                    <xsl:value-of select="../../get_tasks_response/task/progress/text()"/>
                  </xsl:with-param>
                </xsl:call-template>
              </xsl:with-param>
            </xsl:apply-templates>
          </div>
        </xsl:when>
      </xsl:choose>
    </div>

    <xsl:choose>
      <xsl:when test="($section='results' or $section='summary')">
        <xsl:apply-templates select="report" mode="section-filter-full">
          <xsl:with-param name="extra_params" select="$extra_params"/>
          <xsl:with-param name="report_section" select="$section"/>
        </xsl:apply-templates>
      </xsl:when>
      <xsl:otherwise>
        <xsl:apply-templates select="report" mode="section-filter-restricted">
          <xsl:with-param name="extra_params" select="$extra_params"/>
          <xsl:with-param name="report_section" select="$section"/>
        </xsl:apply-templates>
      </xsl:otherwise>
    </xsl:choose>
  </div>
</xsl:template>

<xsl:template name="report-section-header">
  <xsl:param name="section" select="'results'"/>
  <xsl:param name="filtered-count" select="''"/>
  <xsl:param name="full-count" select="''"/>

  <div id="list-window-header">
    <div id="list-window-title">
      <div class="section-header">
        <div class="section-header-info">
          <table>
            <tr>
              <td><xsl:value-of select="gsa:i18n ('ID')"/>:</td>
              <td>
                <xsl:value-of select="@id"/>
              </td>
            </tr>
            <tr>
              <td><xsl:value-of select="gsa:i18n ('Modified', 'Date')"/>:</td>
              <td><xsl:value-of select="gsa:long-time (modification_time)"/></td>
            </tr>
            <tr>
              <td><xsl:value-of select="gsa:i18n ('Created', 'Date')"/>:</td>
              <td><xsl:value-of select="gsa:long-time (creation_time)"/></td>
            </tr>
            <tr>
              <td><xsl:value-of select="gsa:i18n ('Owner')"/>:</td>
              <td><xsl:value-of select="owner/name"/></td>
            </tr>
          </table>
        </div>

        <xsl:choose>
          <xsl:when test="0">
          </xsl:when>
          <xsl:otherwise>
            <img class="icon icon-lg" src="/img/vul_report.svg"/>
          </xsl:otherwise>
        </xsl:choose>

        <h1>
          <xsl:apply-templates select="report" mode="section-list">
            <xsl:with-param name="current" select="$section"/>
          </xsl:apply-templates>
          <xsl:if test="$filtered-count != ''">
            <xsl:text> (</xsl:text>
            <xsl:choose>
              <xsl:when test="$full-count != ''">
                <xsl:value-of select="gsa-i18n:strformat (gsa:i18n ('%1 of %2'), $filtered-count, $full-count)"/>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="$filtered-count"/>
              </xsl:otherwise>
            </xsl:choose>
            <xsl:text>)</xsl:text>
          </xsl:if>
        </h1>
      </div>
    </div>
  </div>
</xsl:template>

<xsl:template match="report" mode="results">
  <xsl:variable name="levels"
                select="report/filters/keywords/keyword[column = 'levels']/value"/>
  <xsl:variable name="apply-overrides"
                select="report/filters/keywords/keyword[column = 'apply_overrides']/value"/>
  <xsl:variable name="type">
    <xsl:choose>
      <xsl:when test="@type"><xsl:value-of select="@type"/></xsl:when>
      <xsl:otherwise>normal</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:apply-templates select="." mode="report-section-toolbar">
    <xsl:with-param name="section" select="'results'"/>
  </xsl:apply-templates>
  <xsl:call-template name="report-section-header">
    <xsl:with-param name="section" select="'results'"/>
    <xsl:with-param name="filtered-count" select="report/result_count/filtered"/>
    <xsl:with-param name="full-count" select="report/result_count/full"/>
  </xsl:call-template>

  <div id="table-box" class="section-box">
      <xsl:choose>
        <xsl:when test="count(report/results/result) &gt; 0">
          <div id="reports">
            <div class="footnote" style="text-align:right;">
              <xsl:apply-templates select="report" mode="section-pager">
                <xsl:with-param name="report_section" select="'results'"/>
                <xsl:with-param name="count" select="count (report/results/result)"/>
                <xsl:with-param name="filtered-count" select="report/result_count/filtered"/>
                <xsl:with-param name="full-count" select="report/result_count/full"/>
              </xsl:apply-templates>
            </div>
            <table class="gbntable">
              <xsl:apply-templates select="report" mode="details"/>
              <tr>
                <td class="footnote" colspan="1000">
                  <xsl:variable name="prognostic">
                    <xsl:if test="report/@type='prognostic'">
                      <xsl:value-of select="concat ('&amp;type=prognostic&amp;host=', report/filters/host, '&amp;host_search_phrase=', host_search_phrase, '&amp;host_levels=', host_levels, '&amp;host_first_result=', results/@start, '&amp;host_max_results=', results/@max, '&amp;pos=', /envelope/params/pos)"/>
                    </xsl:if>
                  </xsl:variable>
                  <xsl:variable name="delta">
                    <xsl:if test="report/@type='delta'">
                      <xsl:value-of select="concat ('&amp;delta_report_id=', report/delta/report/@id)"/>
                    </xsl:if>
                  </xsl:variable>
                  <div class="pull-right">
                    <xsl:apply-templates select="report" mode="section-pager">
                      <xsl:with-param name="report_section" select="'results'"/>
                      <xsl:with-param name="count" select="count (report/results/result)"/>
                      <xsl:with-param name="filtered-count" select="report/result_count/filtered"/>
                      <xsl:with-param name="full-count" select="report/result_count/full"/>
                    </xsl:apply-templates>
                  </div>
                  (<xsl:value-of select="gsa:i18n('Applied filter:')"/>
                  <a class="footnote"
                     href="/omp?cmd=get_report_section&amp;report_id={report/@id}&amp;report_section=results&amp;overrides={$apply-overrides}&amp;details={/envelope/params/details}&amp;filter={report/filters/term}{$prognostic}{$delta}&amp;token={/envelope/token}">
                    <xsl:value-of select="report/filters/term"/>
                  </a>)
                </td>
              </tr>
            </table>
          </div>
        </xsl:when>
        <xsl:otherwise>
          <xsl:variable name="report_url" select="concat ('/omp?token=', /envelope/token, '&amp;cmd=get_report_section&amp;report_id=', report/@id, '&amp;report_section=results')"/>
          <div class="result-info">
            <xsl:choose>
              <xsl:when test="report/result_count/full = 0">
                <p class="alert alert-info"><xsl:value-of select="gsa:i18n ('The report is empty. This can happen for the following reasons:')"/></p>
                <ul>
                  <xsl:choose>
                    <xsl:when test="report/task/progress = 1">
                      <li class="panel panel-info">
                        <div class="panel-heading">
                          <xsl:value-of select="gsa:i18n ('The scan just started and no results have arrived yet.')"/>
                        </div>
                        <p class="panel-body">
                          <a href="{/envelope/current_page}&amp;token={/envelope/token}">
                            <img src="/img/refresh.svg" class="icon icon-lg valign-middle"/>
                            <span>
                              <xsl:value-of select="gsa:i18n ('Click here to reload this page and update the status.')"/>
                            </span>
                          </a>
                        </p>
                      </li>
                    </xsl:when>
                    <xsl:when test="report/task/progress &gt; 1">
                      <li class="panel panel-info">
                        <div class="panel-heading">
                          <xsl:value-of select="gsa:i18n ('The scan is still running and no results have arrived yet.')"/>
                        </div>
                        <p class="panel-body">
                          <a href="{/envelope/current_page}&amp;token={/envelope/token}">
                            <img src="/img/refresh.svg" class="icon icon-lg valign-middle"/>
                            <span>
                              <xsl:value-of select="gsa:i18n ('Click here to reload this page and update the status.')"/>
                            </span>
                          </a>
                        </p>
                      </li>
                    </xsl:when>
                    <xsl:otherwise>
                      <li class="panel panel-info">
                        <div class="panel-heading">
                          <xsl:value-of select="gsa:i18n ('The target hosts could be regarded dead.')"/>
                        </div>
                        <div class="panel-body">
                          <xsl:choose>
                            <xsl:when test="gsa:may-op ('modify_target')">
                              <!-- i18n with concat : see dynamic_strings.xsl - type-edit -->
                              <a href="/omp?cmd=edit_target&amp;target_id={report/task/target/@id}&amp;next=get_report&amp;filter={str:encode-uri (/envelope/params/filter, true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}" data-reload="window"
                                class="edit-action-icon" data-type="target" data-id="{report/task/target/@id}"
                                title="{gsa:i18n ('Edit Target')}">
                                <img src="/img/target.svg" class="icon icon-lg"/>
                                <span>
                                  <xsl:value-of select="gsa:i18n ('You could change the Alive Test method of the target. However, if the targets are indeed dead, the scan duration might increase significantly.')"/>
                                  <xsl:text> (</xsl:text>
                                  <xsl:value-of select="gsa:i18n ('Click here to edit the target')"/>
                                  <xsl:text>)</xsl:text>
                                </span>
                              </a>
                            </xsl:when>
                            <xsl:otherwise>
                              <img src="/img/target.svg" class="icon icon-lg"/>
                              <span>
                                <xsl:value-of select="gsa:i18n ('You could change the Alive Test method of the target. However, if the targets are indeed dead, the scan duration might increase significantly.')"/>
                              </span>
                            </xsl:otherwise>
                          </xsl:choose>
                        </div>
                      </li>
                    </xsl:otherwise>
                  </xsl:choose>
                </ul>
              </xsl:when>
              <xsl:when test="report/result_count/full &gt; 0">
                <p class="alert alert-info">
                  <xsl:value-of select="gsa:i18n ('The report is empty.')"/>
                  <xsl:value-of select="' '"/>
                  <b><xsl:value-of select="gsa-i18n:strformat (gsa:i18n ('The filter does not match any of %1 results.'), report/result_count/full)"/></b>
                </p>
                <ul>
                  <xsl:if test="not (contains ($levels, 'g'))">
                    <xsl:variable name="filter" select="gsa:build-filter (report/filters, 'levels', 'levels=hmlg')" />
                    <li class="panel panel-info">
                      <div class="panel-heading">
                        <xsl:choose>
                          <xsl:when test="number(report/severity/full) = 0">
                            <xsl:value-of select="gsa:i18n ('The report only contains log messages, which are currently excluded.')"/>
                          </xsl:when>
                          <xsl:otherwise>
                            <xsl:value-of select="gsa:i18n ('Log messages are currently excluded.')"/>
                          </xsl:otherwise>
                        </xsl:choose>
                      </div>
                      <p class="panel-body">
                        <a href="{$report_url}&amp;filter={$filter}"
                          title="{gsa:i18n ('Add log messages to the filter')}">
                          <img src="/img/filter.svg" class="valign-middle icon icon-lg"/>
                          <span>
                            <xsl:value-of select="gsa:i18n ('Include log messages in your filter setting.')"/>
                          </span>
                        </a>
                      </p>
                    </li>
                  </xsl:if>
                  <xsl:if test="contains (report/filters/term, 'severity>')">
                    <xsl:variable name="filter" select="gsa:build-filter (report/filters, 'severity', '')" />
                    <li class="panel panel-info">
                      <div class="panel-heading">
                        <xsl:value-of select="gsa:i18n ('You are using keywords setting a minimum limit on severity.')"/>
                      </div>
                      <p class="panel-body">
                        <a href="{$report_url}&amp;filter={$filter}"
                          title="{gsa:i18n ('Remove severity limit')}">
                          <img src="/img/filter.svg" class="valign-middle icon icon-lg"/>
                          <span>
                            <xsl:value-of select="gsa:i18n ('Remove the severity limit from your filter settings.')"/>
                          </span>
                        </a>
                      </p>
                    </li>
                  </xsl:if>
                  <xsl:if test="report/filters/keywords/keyword[column='min_qod']/value > 30">
                    <xsl:variable name="filter" select="gsa:build-filter (report/filters, 'min_qod', 'min_qod=30')" />
                    <li class="panel panel-info">
                      <div class="panel-heading">
                        <xsl:value-of select="gsa:i18n ('There may be results below the current minimum Quality of Detection level.')"/>
                      </div>
                      <p class="panel-body">
                        <a href="{$report_url}&amp;filter={$filter}"
                          title="{gsa:i18n ('Descrease minimum QoD')}">
                          <img src="/img/filter.svg" class="valign-middle icon icon-lg"/>
                          <span>
                            <xsl:value-of select="gsa:i18n ('Decrease the minimum QoD in the Filter to 30 percent to see those results.')"/>
                          </span>
                        </a>
                      </p>
                    </li>
                  </xsl:if>
                  <xsl:if test="report/filters/keywords/keyword[column='qod' and not (relation='&lt;')]">
                    <xsl:variable name="filter" select="gsa:build-filter (report/filters, 'qod', '')" />
                    <li class="panel panel-info">
                      <div class="panel-heading">
                        <xsl:value-of select="gsa:i18n ('You are using keywords setting a lower limit on QoD.')"/>
                      </div>
                      <p class="panel-body">
                        <a href="{$report_url}&amp;filter={$filter}"
                          title="{gsa:i18n ('Remove QoD limit')}">
                          <img src="/img/filter.svg" class="valign-middle icon icon-lg"/>
                          <span>
                            <xsl:value-of select="gsa:i18n ('Remove Quality of Detection limit.')"/>
                          </span>
                        </a>
                      </p>
                    </li>
                  </xsl:if>
                  <li class="panel panel-info">
                    <div class="panel-heading">
                      <xsl:value-of select="gsa:i18n ('Your filter settings may be too refined.')"/>
                    </div>
                    <p class="panel-body">
                      <a href="#" class="edit-filter-action-icon"
                        data-id="filterbox" title="{gsa:i18n ('Edit filter')}">
                        <img src="/img/edit.svg" class="valign-middle icon icon-lg"/>
                        <span>
                          <xsl:value-of select="gsa:i18n ('Adjust and update your filter settings.')"/>
                        </span>
                      </a>
                    </p>
                  </li>
                  <li class="panel panel-info">
                    <div class="panel-heading">
                      <xsl:value-of select="gsa:i18n ('Your last filter change may be too restrictive.')"/>
                    </div>
                    <p class="panel-body">
                      <a href="/omp?token={/envelope/token}&amp;cmd=get_report_section&amp;report_id={report/@id}&amp;report_section=results&amp;filt_id=--"
                        title="{gsa:i18n ('Reset filter')}">
                        <img src="/img/delete.svg" class="valign-middle icon icon-lg"/>
                        <span>
                          <xsl:value-of select="gsa:i18n ('Reset the filter settings to the defaults.')"/>
                        </span>
                      </a>
                    </p>
                  </li>
                </ul>
              </xsl:when>
            </xsl:choose>
          </div>
        </xsl:otherwise>
      </xsl:choose>
  </div>
</xsl:template>


<!-- BEGIN TASKS MANAGEMENT -->

<xsl:template name="task-icons">
  <xsl:param name="next" select="'get_tasks'"/>
  <xsl:param name="show-start-when-scheduled" select="false ()"/>
  <xsl:param name="show-stop-when-scheduled" select="false ()"/>
  <xsl:choose>
    <xsl:when test="target/@id = ''">
      <a href="/omp?cmd=upload_report&amp;next=get_report&amp;task_id={@id}&amp;filter={str:encode-uri (filters/term, true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
         class="upload-action-icon icon icon-sm" data-type="report"
         data-task_id="{@id}" data-reload="window"
         title="{gsa:i18n ('Import Report')}">
        <img src="/img/upload.svg"/>
      </a>
    </xsl:when>
    <xsl:when test="gsa:may ('start_task') = 0">
      <img class="icon icon-sm" src="/img/start_inactive.svg"
        alt="{gsa:i18n ('Start', 'Action Verb')}" title="{gsa:i18n ('Permission to start task denied')}"/>
    </xsl:when>
    <xsl:when test="string-length(schedule/@id) &gt; 0">
      <xsl:choose>
        <xsl:when test="boolean (schedule/permissions) and count (schedule/permissions/permission) = 0">
          <img class="icon icon-sm" src="/img/scheduled_inactive.svg"
               alt="{gsa:i18n ('Schedule Unavailable')}"
               title="{gsa:i18n ('Schedule Unavailable')} ({gsa:i18n ('Name')}: {schedule/name}, ID: {schedule/@id})"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:variable name="next_due_string">
            <xsl:choose>
              <xsl:when test="schedule/next_time = 'over'">
    <xsl:text>
    (</xsl:text>
                <xsl:value-of select="gsa:i18n ('Next due: over', 'Task|Schedule')"/>
                <xsl:text>)</xsl:text>
              </xsl:when>
              <xsl:otherwise>
    <xsl:text>
    (</xsl:text>
                <xsl:value-of select="gsa:i18n ('Next due', 'Task|Schedule')"/>: <xsl:value-of select="gsa:long-time (schedule/next_time)"/>
                <xsl:choose>
                  <xsl:when test="schedule_periods = 1">
                    <xsl:value-of select="concat (', ', gsa:i18n ('Once'))"/>
                  </xsl:when>
                  <xsl:when test="schedule_periods &gt; 1">
                    <xsl:value-of select="concat (', ', schedule_periods, ' ', gsa:i18n ('more times'))"/>
                  </xsl:when>
                  <xsl:otherwise>
                  </xsl:otherwise>
                </xsl:choose>
                <xsl:text>)</xsl:text>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:variable>
          <a href="/omp?cmd=get_schedule&amp;schedule_id={schedule/@id}&amp;token={/envelope/token}"
             class="icon icon-sm"
             title="{concat (gsa:view_details_title ('schedule', schedule/name), $next_due_string)}">
            <img src="/img/scheduled.svg" alt="{gsa:i18n ('Schedule Details')}"/>
          </a>
        </xsl:otherwise>
      </xsl:choose>
      <xsl:if test="boolean ($show-start-when-scheduled)">
        <xsl:choose>
          <xsl:when test="status!='Running' and status!='Stop Requested' and status!='Delete Requested' and status!='Ultimate Delete Requested' and status!='Resume Requested' and status!='Requested'">
            <xsl:call-template name="start-icon">
              <xsl:with-param name="type">task</xsl:with-param>
              <xsl:with-param name="id" select="@id"/>
              <xsl:with-param name="params">
                <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
                <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
                <input type="hidden" name="next" value="{$next}"/>
              </xsl:with-param>
            </xsl:call-template>
          </xsl:when>
          <xsl:when test="status='Running'">
          </xsl:when>
          <xsl:otherwise>
            <img class="icon icon-sm" src="/img/start_inactive.svg" alt="{gsa:i18n ('Start', 'Action Verb')}" title="{gsa:i18n ('Task is already active')}"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:if>
    </xsl:when>
    <xsl:when test="status='Running'">
      <xsl:call-template name="stop-icon">
        <xsl:with-param name="type">task</xsl:with-param>
        <xsl:with-param name="id" select="@id"/>
        <xsl:with-param name="params">
          <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
          <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
          <input type="hidden" name="next" value="{$next}"/>
        </xsl:with-param>
      </xsl:call-template>
    </xsl:when>
    <xsl:when test="status='Stop Requested' or status='Delete Requested' or status='Ultimate Delete Requested' or status='Resume Requested' or status='Requested'">
      <img class="icon icon-sm" src="/img/start_inactive.svg" alt="{gsa:i18n ('Start', 'Action Verb')}" title="{gsa:i18n ('Task is already active')}"/>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="start-icon">
        <xsl:with-param name="type">task</xsl:with-param>
        <xsl:with-param name="id" select="@id"/>
        <xsl:with-param name="params">
          <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
          <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
          <input type="hidden" name="next" value="{$next}"/>
        </xsl:with-param>
      </xsl:call-template>
    </xsl:otherwise>
  </xsl:choose>
  <xsl:choose>
    <xsl:when test="(string-length (/envelope/params/enable_stop) &gt; 0 and /envelope/params/enable_stop = 1) or (boolean ($show-stop-when-scheduled) and status='Running' and string-length(schedule/@id) &gt; 0)">
      <xsl:call-template name="stop-icon">
        <xsl:with-param name="type">task</xsl:with-param>
        <xsl:with-param name="id" select="@id"/>
        <xsl:with-param name="params">
          <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
          <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
          <input type="hidden" name="next" value="{$next}"/>
        </xsl:with-param>
      </xsl:call-template>
    </xsl:when>
  </xsl:choose>
  <xsl:choose>
    <xsl:when test="target/@id = ''">
      <img src="/img/resume_inactive.svg" alt="{gsa:i18n ('Resume')}" title="{gsa:i18n ('Task is a container')}"
         class="icon icon-sm"/>
    </xsl:when>
    <xsl:when test="string-length(schedule/@id) &gt; 0">
      <img src="/img/resume_inactive.svg" alt="{gsa:i18n ('Resume')}" title="{gsa:i18n ('Task is scheduled')}"
           class="icon icon-sm"/>
    </xsl:when>
    <xsl:when test="status='Stopped'">
      <xsl:choose>
        <xsl:when test="gsa:may ('resume_task') = 0">
          <img src="/img/resume_inactive.svg" alt="{gsa:i18n ('Resume')}" title="{gsa:i18n ('Permission to resume task denied')}"
             class="icon icon-sm"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="resume-icon">
            <xsl:with-param name="type">task</xsl:with-param>
            <xsl:with-param name="cmd">resume_task</xsl:with-param>
            <xsl:with-param name="id" select="@id"/>
            <xsl:with-param name="params">
              <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
              <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
              <input type="hidden" name="next" value="{$next}"/>
            </xsl:with-param>
          </xsl:call-template>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>
    <xsl:otherwise>
      <img src="/img/resume_inactive.svg" alt="{gsa:i18n ('Resume')}" title="{gsa:i18n ('Task is not stopped')}"
           class="icon icon-sm"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="move_task_icon">
  <xsl:param name="task" select="."/>
  <xsl:param name="slaves" select="../../../get_scanners_response/scanner[type=4]"/>
  <xsl:param name="next" select="'get_task'"/>
  <xsl:variable name="current_slave_id" select="$task/scanner/@id"/>
  <xsl:choose>
    <xsl:when test="gsa:may-op ('get_scanners') and gsa:may-op ('modify_task') and count ($slaves)">
      <span class="icon-menu">
        <xsl:variable name="slave_count" select="count ($slaves [@id != $current_slave_id])"/>
        <img src="/img/wizard.svg" class="icon icon-sm"/>
        <ul>
          <xsl:if test="$current_slave_id != '08b69003-5fc2-4037-a479-93b440211c73'">
            <xsl:variable name="class">
              <xsl:text>first</xsl:text>
              <xsl:if test="$slave_count = 0"> last</xsl:if>
            </xsl:variable>
            <li class="{$class}">
              <a href="#" class="{$class}" onclick="move_task_form.submit();">
                <xsl:value-of select="gsa:i18n ('Move to Master', 'Task')"/>
              </a>
            </li>
          </xsl:if>

          <xsl:for-each select="$slaves [@id != $current_slave_id]">
            <xsl:variable name="class">
              <xsl:choose>
                <xsl:when test="$slave_count = 1 and $current_slave_id = ''">first last</xsl:when>
                <xsl:when test="position () = 1 and $current_slave_id = ''">first</xsl:when>
                <xsl:when test="position () = last ()">last</xsl:when>
              </xsl:choose>
            </xsl:variable>
            <li class="{$class}">
              <a href="#" class="{$class}" onclick="move_task_form.slave_id.value = '{@id}'; move_task_form.submit();">
                <xsl:value-of select="gsa-i18n:strformat (gsa:i18n ('Move to Slave &quot;%1&quot;', 'Task'), name)"/>
              </a>
            </li>
          </xsl:for-each>
        </ul>
      </span>
      <form style="display:none" method="post" name="move_task_form" action="">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="move_task"/>
        <input type="hidden" name="task_id" value="{$task/@id}"/>
        <input type="hidden" name="slave_id" value=""/>
        <input type="hidden" name="next" value="{$next}"/>
        <input type="hidden" name="filter" value="{/envelope/params/filter}"/>
        <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
      </form>
    </xsl:when>
  </xsl:choose>
</xsl:template>

<xsl:template match="task" mode="details">
  <xsl:variable name="apply-overrides" select="../apply_overrides"/>
  <xsl:variable name="min-qod" select="/envelope/params/min_qod"/>

  <div class="toolbar">
    <xsl:call-template name="details-header-icons">
      <xsl:with-param name="cap-type" select="'Task'"/>
      <xsl:with-param name="type" select="'task'"/>
    </xsl:call-template>
    <xsl:choose>
      <xsl:when test="alterable = 0">
      </xsl:when>
      <xsl:otherwise>
        <img src="/img/alterable.svg" class="icon icon-sm"
             alt="{gsa:i18n ('This is an Alterable Task. Reports may not relate to current Scan Config or Target!')}"
             title="{gsa:i18n ('This is an Alterable Task. Reports may not relate to current Scan Config or Target!')}"/>
      </xsl:otherwise>
    </xsl:choose>
    <span class="divider"/>
    <xsl:call-template name="task-icons">
      <xsl:with-param name="next" select="'get_task'"/>
      <xsl:with-param name="show-start-when-scheduled" select="1"/>
      <xsl:with-param name="show-stop-when-scheduled" select="1"/>
    </xsl:call-template>
    <xsl:call-template name="move_task_icon"/>
  </div>

  <div class="section-header">
    <xsl:call-template name="minor-details"/>
    <h1>
      <a href="/omp?cmd=get_tasks&amp;token={/envelope/token}"
         title="{gsa:i18n ('Tasks')}">
        <img class="icon icon-lg" src="/img/task.svg" alt="Tasks"/>
      </a>
      <xsl:value-of select="gsa:i18n ('Task')"/>:
      <xsl:value-of select="name"/>
      <xsl:text> </xsl:text>
    </h1>
  </div>

  <div class="section-box">
    <table>
      <tr>
        <td><b><xsl:value-of select="gsa:i18n ('Name')"/>:</b></td>
        <td><b><xsl:value-of select="name"/></b></td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Comment')"/>:</td>
        <td><xsl:value-of select="comment"/></td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Target')"/>:</td>
        <td>
          <xsl:choose>
            <xsl:when test="boolean (target/permissions) and count (target/permissions/permission) = 0">
              <xsl:value-of select="gsa:i18n('Unavailable')"/>
              <xsl:text> (</xsl:text>
              <xsl:value-of select="gsa:i18n ('Name')"/>
              <xsl:text>: </xsl:text>
              <xsl:value-of select="target/name"/>
              <xsl:text>, </xsl:text>
              <xsl:value-of select="gsa:i18n ('ID')"/>: <xsl:value-of select="target/@id"/>
              <xsl:text>)</xsl:text>
            </xsl:when>
            <xsl:otherwise>
              <a href="/omp?cmd=get_target&amp;target_id={target/@id}&amp;token={/envelope/token}">
                <xsl:value-of select="target/name"/>
              </a>
            </xsl:otherwise>
          </xsl:choose>
        </td>
      </tr>
      <xsl:if test="gsa:may-op ('get_alerts') or count (alert) &gt; 0">
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Alerts')"/>:</td>
          <td>
            <xsl:for-each select="alert">
              <xsl:choose>
                <xsl:when test="boolean (permissions) and count (permissions/permission) = 0">
                  <xsl:value-of select="gsa:i18n('Unavailable')"/>
                  <xsl:text> (</xsl:text>
                  <xsl:value-of select="gsa:i18n ('Name')"/>
                  <xsl:text>: </xsl:text>
                  <xsl:value-of select="name"/>
                  <xsl:text>, </xsl:text>
                  <xsl:value-of select="gsa:i18n ('ID')"/>: <xsl:value-of select="@id"/>
                  <xsl:text>)</xsl:text>
                </xsl:when>
                <xsl:when test="gsa:may-op ('get_alerts')">
                  <a href="/omp?cmd=get_alert&amp;alert_id={@id}&amp;token={/envelope/token}">
                    <xsl:value-of select="name"/>
                  </a>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:value-of select="name"/>
                </xsl:otherwise>
              </xsl:choose>
              <xsl:if test="position() != last()">, </xsl:if>
            </xsl:for-each>
          </td>
        </tr>
      </xsl:if>
      <xsl:if test="gsa:may-op ('get_schedules') or boolean (schedule)">
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Schedule')"/>:</td>
          <td>
            <xsl:if test="schedule">
              <xsl:choose>
                <xsl:when test="gsa:may-op ('get_schedules')">
                  <xsl:choose>
                    <xsl:when test="boolean (schedule/permissions) and count (schedule/permissions/permission) = 0">
                      <xsl:value-of select="gsa:i18n('Unavailable')"/>
                      <xsl:text> (</xsl:text>
                      <xsl:value-of select="gsa:i18n ('Name')"/>
                      <xsl:text>: </xsl:text>
                      <xsl:value-of select="schedule/name"/>
                      <xsl:text>, </xsl:text>
                      <xsl:value-of select="gsa:i18n ('ID')"/>: <xsl:value-of select="schedule/@id"/>
                      <xsl:text>)</xsl:text>
                    </xsl:when>
                    <xsl:otherwise>
                      <a href="/omp?cmd=get_schedule&amp;schedule_id={schedule/@id}&amp;token={/envelope/token}">
                        <xsl:value-of select="schedule/name"/>
                      </a>
                    </xsl:otherwise>
                  </xsl:choose>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:value-of select="schedule/name"/>
                </xsl:otherwise>
              </xsl:choose>
              <xsl:choose>
                <xsl:when test="schedule/next_time = 'over'">
                  (<xsl:value-of select="gsa:i18n ('Next due: over', 'Task|Schedule')"/>)
                </xsl:when>
                <xsl:otherwise>
                  <xsl:text> (</xsl:text>
                  <xsl:value-of select="gsa:i18n ('Next due', 'Task|Schedule')"/>: <xsl:value-of select="gsa:long-time (schedule/next_time)"/>
                  <xsl:choose>
                    <xsl:when test="schedule_periods = 1">
                      <xsl:value-of select="concat (', ', gsa:i18n ('Once'))"/>
                    </xsl:when>
                    <xsl:when test="schedule_periods &gt; 1">
                      <xsl:value-of select="concat (', ', schedule_periods, ' ', gsa:i18n ('more times'))"/>
                    </xsl:when>
                    <xsl:otherwise>
                    </xsl:otherwise>
                  </xsl:choose>
                  <xsl:text>)</xsl:text>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:if>
          </td>
        </tr>
      </xsl:if>
      <xsl:variable name="in_assets" select="preferences/preference[scanner_name='in_assets']"/>
      <xsl:if test="target/@id != ''">
        <tr>
          <td>
            <xsl:value-of select="gsa:i18n ('Add to Assets')"/>:
          </td>
          <td>
            <xsl:value-of select="gsa:i18n (normalize-space($in_assets/value), 'Task')"/>
          </td>
        </tr>
        <xsl:if test="normalize-space($in_assets/value) = 'yes'">
          <tr>
            <td></td>
            <td>
              <xsl:value-of select="gsa:i18n ('Apply Overrides')"/>:
              <xsl:value-of select="preferences/preference[scanner_name='assets_apply_overrides']/value"/>
            </td>
          </tr>
          <tr>
            <td></td>
            <td>
              <xsl:value-of select="gsa:i18n ('Min QoD')"/>:
              <xsl:value-of select="preferences/preference[scanner_name='assets_min_qod']/value"/>
              <xsl:text>%</xsl:text>
            </td>
          </tr>
        </xsl:if>
      </xsl:if>
      <tr>
        <td>
          <xsl:value-of select="gsa:i18n ('Alterable Task')"/>:
        </td>
        <td>
          <xsl:variable name="yes" select="alterable"/>
          <xsl:choose>
            <xsl:when test="string-length ($yes) = 0 or $yes = 0">
              <xsl:value-of select="gsa:i18n ('no')"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="gsa:i18n ('yes')"/>
            </xsl:otherwise>
          </xsl:choose>
        </td>
      </tr>
      <tr>
        <xsl:variable name="auto_delete" select="preferences/preference[scanner_name='auto_delete']/value"/>
        <xsl:variable name="auto_delete_data" select="preferences/preference[scanner_name='auto_delete_data']/value"/>
        <td>
          <xsl:value-of select="gsa:i18n ('Auto Delete Reports')"/>:
        </td>
        <td>
          <xsl:choose>
            <xsl:when test="$auto_delete = 'keep'">
              <xsl:value-of select="gsa:i18n ('Automatically delete oldest reports but always keep newest ', 'Task|Auto Delete Reports')"/>
              <xsl:value-of select="$auto_delete_data"/>
              <xsl:value-of select="gsa:i18n (' reports', 'Task|Auto Delete Reports')"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="gsa:i18n ('Do not automatically delete reports')"/>
            </xsl:otherwise>
          </xsl:choose>
        </td>
      </tr>
      <xsl:if test="target/@id != ''">
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Scanner')"/>:</td>
          <td>
            <xsl:choose>
              <xsl:when test="boolean (scanner/permissions) and count (scanner/permissions/permission) = 0">
                <xsl:value-of select="gsa:i18n('Unavailable')"/>
                <xsl:text> (</xsl:text>
                <xsl:value-of select="gsa:i18n ('Name')"/>
                <xsl:text>: </xsl:text>
                <xsl:value-of select="scanner/name"/>
                <xsl:text>, </xsl:text>
                <xsl:value-of select="gsa:i18n ('ID')"/>: <xsl:value-of select="scanner/@id"/>
                <xsl:text>)</xsl:text>
              </xsl:when>
              <xsl:otherwise>
                <xsl:choose>
                  <xsl:when test="gsa:may-op ('get_scanners')">
                    <a href="/omp?cmd=get_scanner&amp;scanner_id={scanner/@id}&amp;token={/envelope/token}">
                      <xsl:value-of select="scanner/name"/>
                    </a>
                    (<xsl:value-of select="gsa:i18n ('Type')"/>:
                    <xsl:call-template name="scanner-type-name">
                      <xsl:with-param name="type" select="scanner/type"/>
                    </xsl:call-template>)
                  </xsl:when>
                  <xsl:otherwise>
                    <xsl:value-of select="scanner/name"/>
                  </xsl:otherwise>
                </xsl:choose>
              </xsl:otherwise>
            </xsl:choose>
          </td>
        </tr>
        <xsl:if test="string-length (config/@id) &gt; 0">
          <tr>
            <td></td>
            <td>
              <xsl:value-of select="gsa:i18n ('Scan Config')"/>:
              <xsl:choose>
                <xsl:when test="boolean (config/permissions) and count (config/permissions/permission) = 0">
                  <xsl:value-of select="gsa:i18n('Unavailable')"/>
                  <xsl:text> (</xsl:text>
                  <xsl:value-of select="gsa:i18n ('Name')"/>
                  <xsl:text>: </xsl:text>
                  <xsl:value-of select="config/name"/>
                  <xsl:text>, </xsl:text>
                  <xsl:value-of select="gsa:i18n ('ID')"/>: <xsl:value-of select="config/@id"/>
                  <xsl:text>)</xsl:text>
                </xsl:when>
                <xsl:otherwise>
                  <a href="/omp?cmd=get_config&amp;config_id={config/@id}&amp;token={/envelope/token}">
                    <xsl:value-of select="config/name"/>
                  </a>
                </xsl:otherwise>
              </xsl:choose>
            </td>
          </tr>
        </xsl:if>
        <xsl:if test="config/type = 0">
          <tr>
            <td></td>
            <td>
              <xsl:value-of select="gsa:i18n ('Order for target hosts')"/>:
              <xsl:choose>
                <xsl:when test="hosts_ordering = 'sequential'"><xsl:value-of select="gsa:i18n ('Sequential', 'Task|Hosts Ordering')"/></xsl:when>
                <xsl:when test="hosts_ordering = 'random'"><xsl:value-of select="gsa:i18n ('Random', 'Task|Hosts Ordering')"/></xsl:when>
                <xsl:when test="hosts_ordering = 'reverse'"><xsl:value-of select="gsa:i18n ('Reverse', 'Task|Hosts Ordering')"/></xsl:when>
                <xsl:otherwise><xsl:value-of select="gsa:i18n ('N/A')"/></xsl:otherwise>
              </xsl:choose>
            </td>
          </tr>
          <tr>
            <td></td>
            <td>
              <xsl:value-of select="gsa:i18n ('Network Source Interface')"/>:
              <xsl:value-of select="preferences/preference[scanner_name='source_iface']/value"/>
            </td>
          </tr>
          <tr>
            <td></td>
            <td>
              <xsl:value-of select="gsa:i18n (normalize-space (preferences/preference[scanner_name='max_checks']/name), 'Task')"/>:
              <xsl:value-of select="preferences/preference[scanner_name='max_checks']/value"/>
            </td>
          </tr>
          <tr>
            <td></td>
            <td>
              <xsl:value-of select="gsa:i18n (normalize-space (preferences/preference[scanner_name='max_hosts']/name), 'Task')"/>:
              <xsl:value-of select="preferences/preference[scanner_name='max_hosts']/value"/>
            </td>
          </tr>
        </xsl:if>
      </xsl:if>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Status')"/>:</td>
        <td>
          <xsl:call-template name="status_bar">
            <xsl:with-param name="status">
              <xsl:choose>
                <xsl:when test="target/@id='' and status='Running'">
                  <xsl:value-of select="'Uploading'"/>
                </xsl:when>
                <xsl:when test="target/@id=''">
                  <xsl:value-of select="'Container'"/>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:value-of select="status"/>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:with-param>
            <xsl:with-param name="progress">
              <xsl:value-of select="progress/text()"/>
            </xsl:with-param>
          </xsl:call-template>
        </td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Duration of last scan')"/>:</td>
        <td>
          <xsl:choose>
            <xsl:when test="last_report/report/scan_end">
              <xsl:value-of select="gsa:date-diff (last_report/report/scan_start, last_report/report/scan_end)"/>
            </xsl:when>
          </xsl:choose>
        </td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Average scan duration')"/>:</td>
        <td>
          <xsl:value-of select="gsa:date-diff-text (date:duration (average_duration))"/>
        </td>
      </tr>
      <tr>
        <td>
          <xsl:value-of select="gsa:i18n ('Reports')"/>:
        </td>
        <td>
          <a href="/omp?cmd=get_reports&amp;replace_task_id=1&amp;filt_id=-2&amp;filter=task_id={@id} apply_overrides={$apply-overrides} min_qod={$min-qod} sort-reverse=date&amp;task_filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
             title="{gsa-i18n:strformat (gsa:i18n ('Reports on Task %1'), name)}">
            <xsl:value-of select="report_count/text ()"/>
          </a>
          <xsl:if test="current_report/report/timestamp">
            <xsl:value-of select="concat(', ', gsa:i18n ('Current', 'Task|Report'), ': ')"/>
            <a href="/omp?cmd=get_report&amp;report_id={current_report/report/@id}&amp;overrides={$apply-overrides}&amp;;min_qod={$min-qod}&amp;token={/envelope/token}">
              <xsl:call-template name="short_timestamp_current"/>
            </a>
          </xsl:if>
           <xsl:value-of select="concat(' (', gsa:i18n ('Finished', 'Task|Reports'), ': ')"/>
           <a href="/omp?cmd=get_reports&amp;replace_task_id=1&amp;filt_id=-2&amp;filter=task_id={@id} and status=Done apply_overrides={$apply-overrides} min_qod={$min-qod} sort-reverse=date&amp;task_filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
             title="{gsa-i18n:strformat (gsa:i18n ('Reports on Task %1'), name)}">
            <xsl:value-of select="report_count/finished"/>
           </a>
           <xsl:if test="last_report/report/timestamp">
             <xsl:value-of select="concat(', ', gsa:i18n ('Last', 'Task|Report'), ': ')"/>
             <a href="/omp?cmd=get_report&amp;report_id={last_report/report/@id}&amp;overrides={$apply-overrides}&amp;min_qod={$min-qod}&amp;token={/envelope/token}">
               <xsl:call-template name="short_timestamp_last"/>
             </a>
           </xsl:if>)
        </td>
      </tr>
      <tr>
        <td>
          <xsl:value-of select="gsa:i18n ('Results')"/>:
        </td>
        <td>
          <a href="/omp?cmd=get_results&amp;filter=severity&gt;Error and task_id={@id} sort=nvt&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
             title="{gsa-i18n:strformat (gsa:i18n ('Results on Task %1'), name)}">
            <xsl:value-of select="result_count/text ()"/>
          </a>
        </td>
      </tr>
      <tr>
        <td>
          <xsl:value-of select="gsa:i18n ('Notes')"/>:
        </td>
        <td>
          <a href="/omp?cmd=get_notes&amp;filter=task_id={@id} sort=nvt&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
             title="{gsa-i18n:strformat (gsa:i18n ('Notes on Task %1'), name)}">
            <xsl:value-of select="count (../../get_notes_response/note)"/>
          </a>
        </td>
      </tr>
      <tr>
        <td>
          <xsl:value-of select="gsa:i18n ('Overrides')"/>:
        </td>
        <td>
          <a href="/omp?cmd=get_overrides&amp;filter=task_id={@id} sort=nvt&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
             title="{gsa-i18n:strformat (gsa:i18n ('Overrides on Task %1'), name)}">
            <xsl:value-of select="count (../../get_overrides_response/override)"/>
          </a>
        </td>
      </tr>
    </table>
  </div>

  <xsl:call-template name="user-tags-window">
    <xsl:with-param name="resource_type" select="'task'"/>
    <xsl:with-param name="tag_names" select="../../../get_tags_response"/>
  </xsl:call-template>

  <xsl:call-template name="resource-permissions-window">
    <xsl:with-param name="resource_type" select="'task'"/>
    <xsl:with-param name="permissions" select="../../../permissions/get_permissions_response"/>
    <xsl:with-param name="related">
      <xsl:variable name="detailed_target" select="../../../get_targets_response/target"/>
      <xsl:variable name="detailed_alerts" select="../../../get_alerts_response/alert"/>
      <xsl:if test="target/@id != ''">
        <target id="{target/@id}"/>
        <xsl:if test="$detailed_target/ssh_credential/@id != ''">
          <credential id="{$detailed_target/ssh_credential/@id}"/>
        </xsl:if>
        <xsl:if test="$detailed_target/smb_credential/@id != '' and $detailed_target/smb_credential/@id != $detailed_target/ssh_credential/@id">
          <credential id="{$detailed_target/smb_credential/@id}"/>
        </xsl:if>
        <xsl:if test="$detailed_target/esxi_credential/@id != '' and $detailed_target/esxi_credential/@id != $detailed_target/ssh_credential/@id and $detailed_target/esxi_credential/@id != $detailed_target/smb_credential/@id">
          <credential id="{$detailed_target/esxi_credential/@id}"/>
        </xsl:if>
        <xsl:if test="$detailed_target/snmp_credential/@id != '' and $detailed_target/snmp_credential/@id != $detailed_target/ssh_credential/@id and $detailed_target/snmp_credential/@id != $detailed_target/smb_credential/@id and $detailed_target/snmp_credential/@id != $detailed_target/esxi_credential/@id">
          <credential id="{$detailed_target/snmp_credential/@id}"/>
        </xsl:if>
        <xsl:if test="$detailed_target/port_list/@id != ''">
          <port_list id="{$detailed_target/port_list/@id}"/>
        </xsl:if>
      </xsl:if>
      <xsl:for-each select="alert">
        <xsl:if test="@id != ''">
          <xsl:variable name="alert_id" select="@id"/>
          <alert id="{$alert_id}"/>
          <xsl:if test="$detailed_alerts[@id=$alert_id]/filter/@id != ''">
            <filter id="{$detailed_alerts[@id=$alert_id]/filter/@id}"/>
          </xsl:if>
        </xsl:if>
      </xsl:for-each>
      <xsl:if test="config/@id != ''">
        <config id="{config/@id}"/>
      </xsl:if>
      <xsl:if test="scanner/@id != ''">
        <scanner id="{scanner/@id}"/>
      </xsl:if>
      <xsl:if test="schedule/@id != ''">
        <schedule id="{schedule/@id}"/>
      </xsl:if>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template name="short_timestamp_first">
  <xsl:if test="first_report/report/timestamp">
    <xsl:value-of select="concat (date:month-abbreviation(first_report/report/timestamp), ' ', date:day-in-month(first_report/report/timestamp), ' ', date:year(first_report/report/timestamp))"/>
  </xsl:if>
</xsl:template>

<xsl:template name="short_timestamp_last">
  <xsl:if test="last_report/report/timestamp">
    <xsl:value-of select="concat (date:month-abbreviation(last_report/report/timestamp), ' ', date:day-in-month(last_report/report/timestamp), ' ', date:year(last_report/report/timestamp))"/>
  </xsl:if>
</xsl:template>

<xsl:template name="short_timestamp_second_last">
  <xsl:if test="first_report/report/timestamp">
    <xsl:value-of select="concat (date:month-abbreviation(second_last_report/report/timestamp), ' ', date:day-in-month(second_last_report/report/timestamp), ' ', date:year(second_last_report/report/timestamp))"/>
  </xsl:if>
</xsl:template>

<xsl:template name="short_timestamp_current">
  <xsl:if test="current_report/report/timestamp">
    <xsl:value-of select="concat (date:month-abbreviation(current_report/report/timestamp), ' ', date:day-in-month(current_report/report/timestamp), ' ', date:year(current_report/report/timestamp))"/>
  </xsl:if>
</xsl:template>

<!-- TREND METER -->
<xsl:template name="trend_meter">
  <xsl:choose>
    <xsl:when test="trend = 'up'">
      <img src="/img/trend_up.svg" alt="{gsa:i18n ('Severity increased')}"
        class="icon icon-sm"
        title="{gsa:i18n ('Severity increased')}"/>
    </xsl:when>
    <xsl:when test="trend = 'down'">
      <img src="/img/trend_down.svg" alt="{gsa:i18n ('Severity decreased')}"
        class="icon icon-sm"
        title="{gsa:i18n ('Severity decreased')}"/>
    </xsl:when>
    <xsl:when test="trend = 'more'">
      <img src="/img/trend_more.svg" alt="{gsa:i18n ('Vulnerability count increased')}"
        class="icon icon-sm"
        title="{gsa:i18n ('Vulnerability count increased')}"/>
    </xsl:when>
    <xsl:when test="trend = 'less'">
      <img src="/img/trend_less.svg" alt="{gsa:i18n ('Vulnerability count decreased')}"
        class="icon icon-sm"
        title="{gsa:i18n ('Vulnerability count decreased')}"/>
    </xsl:when>
    <xsl:when test="trend = 'same'">
      <img src="/img/trend_nochange.svg" alt="{gsa:i18n ('Vulnerabilities did not change')}"
        class="icon icon-sm"
        title="{gsa:i18n ('Vulnerabilities did not change')}"/>
    </xsl:when>
    <xsl:otherwise>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="target" mode="newtask">
  <option value="{@id}"><xsl:value-of select="name"/></option>
</xsl:template>

<xsl:template match="config" mode="newtask">
  <option value="{@id}"><xsl:value-of select="name"/></option>
</xsl:template>

<xsl:template match="alert" mode="newtask">
  <xsl:param name="select_id" select="''"/>
  <xsl:choose>
    <xsl:when test="@id = $select_id">
      <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
    </xsl:when>
    <xsl:otherwise>
      <option value="{@id}"><xsl:value-of select="name"/></option>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="group" mode="newtask">
  <option value="{@id}"><xsl:value-of select="name"/></option>
</xsl:template>

<xsl:template match="schedule" mode="newtask">
  <option value="{@id}"><xsl:value-of select="name"/></option>
</xsl:template>

<xsl:template name="status_bar">
  <xsl:param name="status">(Unknown)</xsl:param>
  <xsl:param name="progress">(Unknown)</xsl:param>
  <xsl:param name="title_suffix"></xsl:param>
  <xsl:choose>
    <xsl:when test="$status='Running'">
      <div class="progressbar_box" title="{gsa:i18n ($status, 'Status')}">
        <div class="progressbar_bar" style="width:{$progress}px;"></div>
        <div class="progressbar_text">
          <xsl:value-of select="$progress"/> %
        </div>
      </div>
    </xsl:when>
    <xsl:when test="$status='New'">
      <div class="progressbar_box" title="{gsa:i18n ($status, 'Status')}{$title_suffix}">
        <div class="progressbar_bar_new" style="width:100px;"></div>
        <div class="progressbar_text">
          <i><b><xsl:value-of select="gsa:i18n ($status, 'Status')"/></b></i>
        </div>
      </div>
    </xsl:when>
    <xsl:when test="$status='Requested'">
      <div class="progressbar_box" title="{gsa:i18n ($status, 'Status')}{$title_suffix}">
        <div class="progressbar_bar_request" style="width:100px;"></div>
        <div class="progressbar_text"><xsl:value-of select="gsa:i18n ($status, 'Status')"/></div>
      </div>
    </xsl:when>
    <xsl:when test="$status='Delete Requested'">
      <div class="progressbar_box" title="{gsa:i18n ($status, 'Status')}{$title_suffix}">
        <div class="progressbar_bar_request" style="width:100px;"></div>
        <div class="progressbar_text"><xsl:value-of select="gsa:i18n ($status, 'Status')"/></div>
      </div>
    </xsl:when>
    <xsl:when test="$status='Ultimate Delete Requested'">
      <div class="progressbar_box" title="{gsa:i18n ('Delete Requested', 'Status')}">
        <div class="progressbar_bar_request" style="width:100px;"></div>
        <div class="progressbar_text"><xsl:value-of select="gsa:i18n ('Delete Requested', 'Status')"/></div>
      </div>
    </xsl:when>
    <xsl:when test="$status='Resume Requested'">
      <div class="progressbar_box" title="{gsa:i18n ($status, 'Status')}{$title_suffix}">
        <div class="progressbar_bar_request" style="width:100px;"></div>
        <div class="progressbar_text"><xsl:value-of select="gsa:i18n ($status, 'Status')"/></div>
      </div>
    </xsl:when>
    <xsl:when test="$status='Stop Requested'">
      <div class="progressbar_box" title="{gsa:i18n ($status, 'Status')}{$title_suffix}">
        <div class="progressbar_bar_request" style="width:100px;"></div>
        <div class="progressbar_text"><xsl:value-of select="gsa:i18n ($status, 'Status')"/></div>
      </div>
    </xsl:when>
    <xsl:when test="$status='Stopped'">
      <div class="progressbar_box" title="{gsa:i18n ($status, 'Status')}{$title_suffix}">
        <div class="progressbar_bar_request" style="width:{$progress}px;"></div>
        <div class="progressbar_text">
          <xsl:value-of select="gsa:i18n ($status, 'Status')"/>
          <xsl:if test="$progress &gt;= 0">
            <xsl:value-of select="gsa:i18n (' at ', 'Status')"/> <xsl:value-of select="$progress"/> %
          </xsl:if>
        </div>
      </div>
    </xsl:when>
    <xsl:when test="$status='Internal Error'">
      <div class="progressbar_box" title="{gsa:i18n ($status, 'Status')}{$title_suffix}">
        <div class="progressbar_bar_error" style="width:100px;"></div>
        <div class="progressbar_text"><xsl:value-of select="gsa:i18n ($status, 'Status')"/></div>
      </div>
    </xsl:when>
    <xsl:when test="$status='Done'">
      <div class="progressbar_box" title="{gsa:i18n ($status, 'Status')}{$title_suffix}">
        <div class="progressbar_bar_done" style="width:100px;"></div>
        <div class="progressbar_text"><xsl:value-of select="gsa:i18n ($status, 'Status')"/></div>
      </div>
    </xsl:when>
    <xsl:when test="$status='Uploading'">
      <div class="progressbar_box" title="{gsa:i18n ($status, 'Status')}{$title_suffix}">
        <div class="progressbar_bar_done" style="width:{$progress}px;"></div>
        <div class="progressbar_text">
          <xsl:value-of select="gsa:i18n ($status, 'Status')"/>
          <xsl:if test="$progress &gt;= 0">
            <xsl:text>: </xsl:text>
            <xsl:value-of select="$progress"/> %
          </xsl:if>
        </div>
      </div>
    </xsl:when>
    <xsl:when test="$status='Container'">
      <div class="progressbar_box" title="{gsa:i18n ($status, 'Status')}{$title_suffix}">
        <div class="progressbar_bar_done" style="width:100px;"></div>
        <div class="progressbar_text"><xsl:value-of select="gsa:i18n ($status, 'Status')"/></div>
      </div>
    </xsl:when>
    <xsl:otherwise>
      <xsl:value-of select="gsa:i18n ($status, 'Status')"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>


<!-- BEGIN GENERIC MANAGEMENT -->

<xsl:template name="list-window">
  <xsl:param name="type"/>
  <xsl:param name="subtype"/>
  <xsl:param name="cap-type"/>
  <xsl:param name="cap-type-plural" select="concat ($cap-type, 's')"/>
  <xsl:param name="resources-summary"/>
  <xsl:param name="resources"/>
  <xsl:param name="count"/>
  <xsl:param name="filtered-count"/>
  <xsl:param name="full-count"/>
  <xsl:param name="columns"/>
  <xsl:param name="icon-count" select="8"/>
  <xsl:param name="new-icon" select="gsa:may-op (concat ('create_', $type))"/>
  <xsl:param name="upload-icon" select="false ()"/>
  <xsl:param name="default-filter"/>
  <xsl:param name="extra_params"/>
  <xsl:param name="extra_params_string">
    <xsl:for-each select="exslt:node-set($extra_params)/param">
      <xsl:text>&amp;</xsl:text>
      <xsl:value-of select="name"/>
      <xsl:text>=</xsl:text>
      <xsl:value-of select="value"/>
    </xsl:for-each>
  </xsl:param>
  <xsl:param name="no_bulk" select="0"/>
  <xsl:param name="top-visualization" select="''"/>

  <xsl:variable name="apply-overrides"
                select="filters/keywords/keyword[column='apply_overrides']/value"/>
  <xsl:variable name="subtype_param">
    <xsl:if test="$subtype != ''">
      <xsl:value-of select="concat ('&amp;', $type, '_type=', $subtype)"/>
    </xsl:if>
  </xsl:variable>

  <div class="toolbar row">
    <div class="col-4">
    <xsl:choose>
      <xsl:when test="$subtype != ''">
        <a href="/help/{gsa:type-many($subtype)}.html?token={/envelope/token}"
           class="icon icon-sm"
           title="{gsa:i18n ('Help')}: {gsa:i18n ($cap-type-plural)}">
          <img src="/img/help.svg"/>
        </a>
      </xsl:when>
      <xsl:otherwise>
        <a href="/help/{gsa:type-many($type)}.html?token={/envelope/token}"
           class="icon icon-sm"
           title="{gsa:i18n ('Help')}: {gsa:i18n ($cap-type-plural)}">
          <img src="/img/help.svg"/>
        </a>
      </xsl:otherwise>
    </xsl:choose>
    <xsl:if test="gsa:may-op ('run_wizard')">
      <xsl:call-template name="wizard-icon"/>
    </xsl:if>
    <xsl:choose>
      <xsl:when test="$type = 'report'"/>
      <xsl:when test="$type = 'info'"/>
      <xsl:when test="$new-icon and $subtype != ''">
        <!-- i18n with concat : see dynamic_strings.xsl - type-new -->
        <a href="/omp?cmd=new_{$subtype}{$extra_params_string}&amp;next=get_{$type}&amp;filter={str:encode-uri (filters/term, true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
           class="new-action-icon icon icon-sm" data-type="{$subtype}" data-reload="window"
           title="{gsa:i18n (concat ('New ', $cap-type))}">
          <img src="/img/new.svg"/>
        </a>
      </xsl:when>
      <xsl:when test="$new-icon and $type = 'task'">
        <span class="icon-menu">
          <a href="/omp?cmd=new_task{$extra_params_string}&amp;next=get_task&amp;filter={str:encode-uri (filters/term, true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
             class="new-action-icon icon icon-sm" data-type="task" data-reload="window"
             data-dialog-id="create_new_task"
             title="{gsa:i18n ('New Task')}">
            <img src="/img/new.svg"/>
          </a>
          <ul>
            <li>
              <a href="/omp?cmd=new_task{$extra_params_string}&amp;next=get_task&amp;filter={str:encode-uri (filters/term, true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
                 class="new-action-icon" data-type="task" data-reload="window"
                 data-dialog-id="create_new_task"
                 title="{gsa:i18n ('New Task')}">
                <xsl:value-of select="gsa:i18n ('New Task')"/>
              </a>
            </li>
            <li>
              <a href="/omp?cmd=new_container_task{$extra_params_string}&amp;next=get_task&amp;filter={str:encode-uri (filters/term, true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
                 class="last new-action-icon" data-type="container_task" data-reload="window"
                 data-dialog-id="create_new_container_task"
                 title="{gsa:i18n ('New Container Task')}">
                <xsl:value-of select="gsa:i18n ('New Container Task')"/>
              </a>
            </li>
          </ul>
        </span>
      </xsl:when>
      <xsl:when test="$new-icon and $type = 'config'">
        <a href="/omp?cmd=new_{$type}{$extra_params_string}&amp;next=get_{$type}&amp;filter={str:encode-uri (filters/term, true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
           class="new-action-icon icon icon-sm" data-type="{$type}" data-reload="dialog"
           data-dialog-id="create_new_{$type}"
           title="{gsa:i18n (concat ('New ', $cap-type))}">
           <span class="success-dialog" data-type="{$type}" data-cmd="edit_{$type}"
             data-reload="window" data-close-reload="window"/>
          <img src="/img/new.svg"/>
        </a>
      </xsl:when>
      <xsl:when test="$new-icon">
        <!-- i18n with concat : see dynamic_strings.xsl - type-new -->
        <a href="/omp?cmd=new_{$type}{$extra_params_string}&amp;next=get_{$type}&amp;filter={str:encode-uri (filters/term, true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
           data-dialog-id="create_new_{$type}"
           class="new-action-icon icon icon-sm" data-type="{$type}" data-reload="window"
           title="{gsa:i18n (concat ('New ', $cap-type))}">
          <img src="/img/new.svg"/>
        </a>
      </xsl:when>
    </xsl:choose>
    <xsl:choose>
      <xsl:when test="$upload-icon">
        <!-- i18n with concat : see dynamic_strings.xsl - type-upload -->
        <a href="/omp?cmd=upload_{$type}{$extra_params_string}&amp;next=get_{$type}&amp;filter={str:encode-uri (filters/term, true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
           class="upload-action-icon icon icon-sm" data-type="{$type}"
           data-dialog-id="upload_{$type}"
           title="{gsa:i18n (concat ('Import ', $cap-type))}">
          <img src="/img/upload.svg"/>
        </a>
      </xsl:when>
    </xsl:choose>
    </div>

    <div id="list-window-filter" class="col-8">
      <xsl:call-template name="filter-window-part">
        <xsl:with-param name="type" select="$type"/>
        <xsl:with-param name="subtype" select="$subtype"/>
        <xsl:with-param name="list" select="$resources-summary"/>
        <xsl:with-param name="full-count" select="$full-count"/>
        <xsl:with-param name="columns" select="$columns" xmlns=""/>
        <xsl:with-param name="filter_options" xmlns="">
          <xsl:if test="$type='result' or $type='report' or $type='task'">
            <option>apply_overrides</option>
            <option>min_qod</option>
          </xsl:if>
          <xsl:if test="$type='result'">
            <option>autofp</option>
            <option>levels</option>
          </xsl:if>
          <option>first</option>
          <option>rows</option>
        </xsl:with-param>
        <xsl:with-param name="extra_params" xmlns="">
          <xsl:copy-of select="$extra_params"/>
          <xsl:if test="$subtype != ''">
            <param>
              <name><xsl:value-of select="$type"/>_type</name>
              <value><xsl:value-of select="$subtype"/></value>
            </param>
          </xsl:if>
        </xsl:with-param>
      </xsl:call-template>
    </div>
  </div>

  <div id="list-window-header">
    <div class="section-header">
      <h1>
        <xsl:choose>
          <xsl:when test="$subtype != ''">
            <img class="icon icon-lg" src="/img/{$subtype}.svg"/>
          </xsl:when>
          <xsl:otherwise>
            <img class="icon icon-lg" src="/img/{$type}.svg"/>
          </xsl:otherwise>
        </xsl:choose>

        <xsl:value-of select="gsa:i18n ($cap-type-plural)"/>
        (<xsl:value-of select="gsa-i18n:strformat (gsa:i18n ('%1 of %2'), $filtered-count, $full-count)"/>)
      </h1>

      <xsl:if test="$top-visualization != ''">
        <div class="dashboard-controls" id="top-dashboard-controls"/>
      </xsl:if>
    </div>
  </div>

  <xsl:if test="$top-visualization != ''">
    <div id="top-dashboard-section" class="section-box">
      <xsl:copy-of select="$top-visualization"/>
    </div>
  </xsl:if>

  <div class="section-box resources" id="table-box">
    <div class="header">
      <xsl:call-template name="filter-window-pager">
        <xsl:with-param name="type" select="$type"/>
        <xsl:with-param name="list" select="$resources-summary"/>
        <xsl:with-param name="count" select="$count"/>
        <xsl:with-param name="filtered_count" select="$filtered-count"/>
        <xsl:with-param name="full_count" select="$full-count"/>
        <xsl:with-param name="extra_params" select="concat($subtype_param, $extra_params_string)"/>
      </xsl:call-template>
    </div>

      <!-- The entire table of resources, in a variable. -->
      <xsl:variable name="table">
        <table class="gbntable">

          <!-- Column headings, top row. -->
          <tr class="gbntablehead2">
            <xsl:variable name="current" select="."/>
            <xsl:variable name="token" select="/envelope/token"/>
            <!-- Generate given column headings. -->
            <xsl:for-each select="exslt:node-set ($columns)/column">
              <xsl:choose>
                <xsl:when test="boolean (hide_in_table)"/>
                <xsl:when test="count (column) = 0 and field != ''">
                  <!-- Single column. -->
                  <td rowspan="2">
                    <xsl:copy-of select="html/before/*"/>
                    <xsl:call-template name="column-name">
                      <xsl:with-param name="head" select="name"/>
                      <xsl:with-param name="image" select="image"/>
                      <xsl:with-param name="name" select="field"/>
                      <xsl:with-param name="type" select="$type"/>
                      <xsl:with-param name="current" select="$current"/>
                      <xsl:with-param name="token" select="$token"/>
                      <xsl:with-param name="extra_params" select="concat($subtype_param, $extra_params_string)"/>
                      <xsl:with-param name="sort-reverse" select="boolean (sort-reverse)"/>
                      <xsl:with-param name="i18n-context" select="$cap-type"/>
                    </xsl:call-template>
                    <xsl:copy-of select="html/after/*"/>
                  </td>
                </xsl:when>
                <xsl:when test="count (column) = 0">
                  <!-- Single column without a sort field. -->
                  <td rowspan="2">
                    <xsl:copy-of select="html/before/*"/>
                    <!-- FIXME : Test if translated name is given everywhere -->
                    <xsl:value-of select="name"/>
                    <xsl:copy-of select="html/after/*"/>
                  </td>
                </xsl:when>
                <xsl:otherwise>
                  <!-- Column with subcolumns. -->
                  <td colspan="{count (column)}">
                    <xsl:copy-of select="html/before/*"/>
                    <!-- FIXME : Test if translated name is given everywhere -->
                    <xsl:value-of select="name"/>
                    <xsl:copy-of select="html/after/*"/>
                  </td>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:for-each>
            <!-- Action column. -->
            <xsl:if test="$icon-count &gt; 0">
              <td style="width: {gsa:actions-width ($icon-count)}px" rowspan="2"><xsl:value-of select="gsa:i18n ('Actions')"/></td>
            </xsl:if>
          </tr>

          <!-- Column headings, second row. -->
          <tr class="gbntablehead2">
            <xsl:variable name="current" select="."/>
            <xsl:variable name="token" select="/envelope/token"/>
            <xsl:for-each select="exslt:node-set ($columns)/column">
              <xsl:choose>
                <xsl:when test="count (column) = 0">
                  <!-- Single column.  Done in top row. -->
                </xsl:when>
                <xsl:otherwise>
                  <!-- Column with subcolumns.  Output the subcolumns. -->
                  <xsl:for-each select="column">
                    <td style="font-size:10px;">
                      <xsl:copy-of select="html/before/*"/>
                      <xsl:call-template name="column-name">
                        <xsl:with-param name="head" select="name"/>
                        <xsl:with-param name="image" select="image"/>
                        <xsl:with-param name="name" select="field"/>
                        <xsl:with-param name="type" select="$type"/>
                        <xsl:with-param name="current" select="$current"/>
                        <xsl:with-param name="token" select="$token"/>
                        <xsl:with-param name="extra_params" select="concat($subtype_param, $extra_params_string)"/>
                        <xsl:with-param name="sort-reverse" select="boolean (sort-reverse)"/>
                        <xsl:with-param name="i18n-context" select="$cap-type"/>
                      </xsl:call-template>
                      <xsl:copy-of select="html/after/*"/>
                    </td>
                  </xsl:for-each>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:for-each>
          </tr>

          <!-- A nested variable: Form inputs for the bulk icons. -->
          <xsl:variable name="bulk-elements">
            <xsl:variable name="selection_type">
              <xsl:choose>
                <xsl:when test="/envelope/params/bulk_select = 1">selection</xsl:when>
                <xsl:when test="/envelope/params/bulk_select = 2">all filtered</xsl:when>
                <xsl:otherwise>page contents</xsl:otherwise>
              </xsl:choose>
            </xsl:variable>
            <input type="hidden" name="token" value="{/envelope/token}"/>
            <xsl:text> </xsl:text>
            <input type="hidden" name="cmd" value="process_bulk"/>
            <input type="hidden" name="next" value="get_{$type}s"/>
            <input type="hidden" name="filter" value="{filters/term}"/>
            <input type="hidden" name="filt_id" value="{filters/@id}"/>
            <input type="hidden" name="bulk_select" value="{/envelope/params/bulk_select}"/>
            <xsl:if test="$subtype">
              <input type="hidden" name="{$type}_type" value="{$subtype}"/>
            </xsl:if>

            <xsl:for-each select="exslt:node-set($extra_params)/param">
              <input type="hidden" name="{name}" value="{value}"/>
            </xsl:for-each>

            <input type="hidden" name="resource_type" value="{$type}"/>

            <!-- i18n with concat : see dynamic_strings.xsl - bulk-actions -->
            <xsl:if test="gsa:may-op (concat ('delete_', $type)) and ($type != 'info' and $type != 'user' and $type != 'report' and $type != 'asset')">
              <input type="image" class="icon icon-sm bulk-dialog-icon" data-type="{$type}" name="bulk_trash" title="{gsa:i18n (concat ('Move ', $selection_type, ' to trashcan'))}" src="/img/trashcan.svg"/>
            </xsl:if>
            <xsl:if test="gsa:may-op (concat ('delete_', $type)) and ($type = 'user' or $type = 'report' or $type = 'asset')">
              <input type="image" class="icon icon-sm bulk-dialog-icon" data-type="{$type}" name="bulk_delete" title="{gsa:i18n (concat ('Delete ', $selection_type))}" src="/img/delete.svg"/>
            </xsl:if>
            <xsl:if test="$type = 'asset' and $subtype = 'host' and gsa:may-op ('create_target')">
              <input type="image" class="icon icon-sm bulk-dialog-icon" data-type="{$type}" name="bulk_create" title="{gsa:i18n (concat ('Create Target from ', $selection_type))}" src="/img/new.svg"/>
            </xsl:if>
            <xsl:if test="$type != 'report'">
              <input class="icon icon-sm" type="image" name="bulk_export" title="{gsa:i18n (concat ('Export ', $selection_type))}" src="/img/download.svg"/>
            </xsl:if>
          </xsl:variable>

          <!-- Resource rows, with extra row if bulk is enabled. -->
          <tbody>
            <xsl:apply-templates select="$resources"/>
          </tbody>
          <xsl:choose>
            <xsl:when test="$no_bulk">
            </xsl:when>
            <xsl:when test="not (/envelope/params/bulk_select = 1)">
              <!-- Bulk "Apply to page contents" or "Apply to all filtered". -->
              <tfoot>
                <tr>
                  <td colspan="{count (exslt:node-set ($columns)/column/column) + count (exslt:node-set ($columns)/column[count (column) = 0]) + ($icon-count &gt; 0)}"  style="text-align:right;" class="small_inline_form">
                    <form name="bulk-actions" method="post" action="/omp" enctype="multipart/form-data" class="small_inline_form">
                      <xsl:choose>
                        <xsl:when test="$type = 'asset' and ($subtype = 'host' or $subtype = 'os')">
                          <xsl:choose>
                            <xsl:when test="/envelope/params/bulk_select = 2">
                              <input type="hidden" name="{$subtype}_count" value="{$filtered-count}"/>
                            </xsl:when>
                            <xsl:otherwise>
                              <input type="hidden" name="{$subtype}_count" value="{$count}"/>
                            </xsl:otherwise>
                          </xsl:choose>
                          <xsl:for-each select="$resources">
                            <input type="hidden" name="bulk_selected:{../@id}" value="1"/>
                          </xsl:for-each>
                        </xsl:when>
                        <xsl:when test="$type = 'info'">
                          <xsl:for-each select="$resources">
                            <input type="hidden" name="bulk_selected:{../@id}" value="1"/>
                          </xsl:for-each>
                        </xsl:when>
                        <xsl:otherwise>
                          <xsl:for-each select="$resources">
                            <input type="hidden" name="bulk_selected:{@id}" value="1"/>
                          </xsl:for-each>
                        </xsl:otherwise>
                      </xsl:choose>
                      <xsl:copy-of select="$bulk-elements"/>
                    </form>
                  </td>
                </tr>
              </tfoot>
            </xsl:when>
            <xsl:otherwise>
              <!-- Bulk "Apply to selection" (the page with checkboxes). -->
              <tfoot>
                <tr>
                  <td colspan="{count (exslt:node-set ($columns)/column/column) + count (exslt:node-set ($columns)/column[count (column) = 0]) + ($icon-count &gt; 0)}"  style="text-align:right;" class="small_inline_form">
                    <xsl:choose>
                      <xsl:when test="$type = 'asset' and ($subtype = 'host' and $subtype = 'os')">
                        <input type="hidden" name="{$subtype}_count" value="0"/>
                      </xsl:when>
                    </xsl:choose>
                    <xsl:copy-of select="$bulk-elements"/>
                  </td>
                </tr>
              </tfoot>
            </xsl:otherwise>
          </xsl:choose>
        </table>
      </xsl:variable>

      <!-- Output the table from the variable. -->
      <xsl:choose>
        <xsl:when test="/envelope/params/bulk_select = 1">
          <!-- Bulk "Apply to selection" (the page with checkboxes). -->
          <form name="bulk-actions" method="post" action="/omp" enctype="multipart/form-data">
            <xsl:copy-of select="$table"/>
          </form>
        </xsl:when>
        <xsl:otherwise>
          <xsl:copy-of select="$table"/>
        </xsl:otherwise>
      </xsl:choose>

      <!-- The bulk dropdown and refresh icon, during bulk selection. -->
      <xsl:if test="not ($no_bulk)">
        <form name="bulk_select_type_form" class="small_inline_form bulk-select-type">
          <input type="hidden" name="token" value="{/envelope/token}"/>
          <input type="hidden" name="cmd" value="get_{gsa:type-many($type)}"/>
          <xsl:if test="$subtype">
            <input type="hidden" name="{$type}_type" value="{$subtype}"/>
          </xsl:if>
          <xsl:for-each select="exslt:node-set($extra_params)/param">
            <input type="hidden" name="{name}" value="{value}"/>
          </xsl:for-each>
          <input type="hidden" name="filter" value="{filters/term}"/>
          <input type="hidden" name="filt_id" value="{filters/@id}"/>
          <select name="bulk_select" onchange="bulk_select_type_form.submit()">
            <!-- TODO selection by current parameter value + check marks -->
            <xsl:choose>
              <xsl:when test="not (/envelope/params/bulk_select != 0)">
                <option value="0" selected="1">&#8730;<xsl:value-of select="gsa:i18n('Apply to page contents')"/></option>
              </xsl:when>
              <xsl:otherwise>
                <option value="0"><xsl:value-of select="gsa:i18n('Apply to page contents')"/></option>
              </xsl:otherwise>
            </xsl:choose>
            <xsl:choose>
              <xsl:when test="/envelope/params/bulk_select = '1'">
                <option value="1" selected="1">&#8730;<xsl:value-of select="gsa:i18n('Apply to selection')"/></option>
              </xsl:when>
              <xsl:otherwise>
                <option value="1"><xsl:value-of select="gsa:i18n('Apply to selection')"/></option>
              </xsl:otherwise>
            </xsl:choose>
            <xsl:choose>
              <xsl:when test="/envelope/params/bulk_select = '2'">
                <option value="2" selected="1">&#8730;<xsl:value-of select="gsa:i18n('Apply to all filtered')"/></option>
              </xsl:when>
              <xsl:otherwise>
                <option value="2"><xsl:value-of select="gsa:i18n('Apply to all filtered')"/></option>
              </xsl:otherwise>
            </xsl:choose>
          </select>
        </form>
      </xsl:if>

      <!-- Bottom line with applied filter and pager. -->
      <xsl:if test="string-length (filters/term) &gt; 0">
        <div class="footer">
          <div class="applied-filter">
            (<xsl:value-of select="gsa:i18n('Applied filter')"/>:
            <a href="/omp?cmd=get_{gsa:type-many($type)}{$extra_params_string}&amp;filter={str:encode-uri (filters/term, true ())}&amp;token={/envelope/token}">
              <xsl:value-of select="filters/term"/>
            </a>)
          </div>
          <xsl:call-template name="filter-window-pager">
            <xsl:with-param name="type" select="$type"/>
            <xsl:with-param name="list" select="$resources-summary"/>
            <xsl:with-param name="count" select="$count"/>
            <xsl:with-param name="filtered_count" select="$filtered-count"/>
            <xsl:with-param name="full_count" select="$full-count"/>
            <xsl:with-param name="extra_params" select="concat($subtype_param, $extra_params_string)"/>
          </xsl:call-template>
        </div>
      </xsl:if>

  </div> <!-- /table-box -->

  <!-- Wizard. -->
  <xsl:call-template name="wizard"/>
</xsl:template>

<xsl:template name="minor-details">
  <div class="section-header-info">
    <table>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('ID')"/>:</td>
        <td><xsl:value-of select="@id"/></td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Created', 'Date')"/>:</td>
        <td><xsl:value-of select="gsa:long-time (creation_time)"/></td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Modified', 'Date')"/>:</td>
        <td><xsl:value-of select="gsa:long-time (modification_time)"/></td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Owner')"/>:</td>
        <td><xsl:value-of select="owner/name"/></td>
      </tr>
    </table>
  </div>
</xsl:template>

<xsl:template name="details-header-icons">
  <xsl:param name="cap-type"/>
  <xsl:param name="cap-type-plural" select="concat ($cap-type, 's')"/>
  <xsl:param name="type"/>
  <xsl:param name="noedit"/>
  <xsl:param name="nonew"/>
  <xsl:param name="noupload" select="true ()"/>
  <xsl:param name="noclone" select="$nonew"/>
  <xsl:param name="grey-clone" select="0"/>
  <xsl:param name="noexport"/>
  <xsl:param name="filter" select="/envelope/params/filter"/>
  <xsl:param name="filt_id" select="/envelope/params/filt_id"/>

  <!-- i18n with concat : see dynamic_strings.xsl - type-details -->
  <a class="icon icon-sm" href="/help/{$type}_details.html?token={/envelope/token}"
    title="{gsa:i18n ('Help')}: {gsa:i18n(concat($cap-type, ' Details'))}">
    <img src="/img/help.svg"/>
  </a>
  <xsl:choose>
    <xsl:when test="$nonew"/>
    <xsl:when test="gsa:may-op (concat ('create_', $type)) and $type = 'task'">
      <span class="icon-menu">
        <a href="/omp?cmd=new_task&amp;next=get_task&amp;filter={str:encode-uri (filters/term, true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
           class="new-action-icon icon icon-sm" data-type="task"
           title="{gsa:i18n ('New Task')}">
          <img src="/img/new.svg"/>
        </a>
        <ul>
          <li>
            <a href="/omp?cmd=new_task&amp;next=get_task&amp;filter={str:encode-uri (filters/term, true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
               class="new-action-icon" data-type="task"
               title="{gsa:i18n ('New Task')}">
              <xsl:value-of select="gsa:i18n ('New Task')"/>
            </a>
          </li>
          <li>
            <a href="/omp?cmd=new_container_task&amp;next=get_task&amp;filter={str:encode-uri (filters/term, true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
               class="new-action-icon" data-type="container_task"
               title="{gsa:i18n ('New Container Task')}">
              <xsl:value-of select="gsa:i18n ('New Container Task')"/>
            </a>
          </li>
        </ul>
      </span>
    </xsl:when>
    <xsl:when test="gsa:may-op (concat ('create_', $type))">
      <!-- i18n with concat : see dynamic_strings.xsl - type-new -->
      <a href="/omp?cmd=new_{$type}&amp;next=get_{$type}&amp;filter={str:encode-uri ($filter, true ())}&amp;filt_id={$filt_id}&amp;{$type}_id={@id}&amp;token={/envelope/token}"
         class="new-action-icon icon icon-sm" data-type="{$type}" data-reload="window"
         title="{gsa:i18n (concat ('New ', $cap-type))}">
        <img src="/img/new.svg"/>
      </a>
    </xsl:when>
  </xsl:choose>
  <xsl:choose>
    <xsl:when test="$noupload"/>
    <xsl:when test="gsa:may-op (concat ('create_', $type))">
      <a href="/omp?cmd=upload_{$type}&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
         class="upload-action-icon icon icon-sm" data-type="port_list" data-reload="window"
         title="{gsa:i18n ('Import Port List')}">
        <img src="/img/upload.svg"/>
      </a>
    </xsl:when>
  </xsl:choose>
  <xsl:choose>
    <xsl:when test="$noclone"/>
    <xsl:when test="$grey-clone">
      <!-- i18n with concat : see dynamic_strings.xsl - type-action-denied -->
      <img src="/img/clone_inactive.svg"
           alt="{gsa:i18n ('Clone', 'Action Verb')}"
           value="Clone"
           title="{gsa:i18n (concat ($cap-type, ' may not be cloned'))}"
           class="icon icon-sm"/>
    </xsl:when>
    <xsl:when test="gsa:may-clone ($type, owner)">
      <xsl:choose>
        <xsl:when test="writable='0' and $type='permission'">
          <!-- i18n with concat : see dynamic_strings.xsl - type-action-denied -->
          <img src="/img/clone_inactive.svg"
               alt="{gsa:i18n ('Clone', 'Action Verb')}"
               value="Clone"
               title="{gsa:i18n (concat ($cap-type, ' must be owned or global'))}"
               class="icon icon-sm"/>
        </xsl:when>
        <xsl:otherwise>
          <div class="icon icon-sm ajax-post" data-reload="next" data-busy-text="{gsa:i18n ('Cloning...')}">
            <img src="/img/clone.svg"
              alt="{gsa:i18n ('Clone', 'Action Verb')}"
              title="{gsa:i18n ('Clone', 'Action Verb')}"/>
            <form action="/omp" method="post" enctype="multipart/form-data">
              <input type="hidden" name="token" value="{/envelope/token}"/>
              <input type="hidden" name="caller" value="{/envelope/current_page}"/>
              <input type="hidden" name="cmd" value="clone"/>
              <input type="hidden" name="resource_type" value="{$type}"/>
              <input type="hidden" name="next" value="get_{$type}"/>
              <input type="hidden" name="id" value="{@id}"/>
              <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
              <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
            </form>
          </div>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>
  </xsl:choose>
  <a href="/omp?cmd=get_{$type}s&amp;filter={str:encode-uri ($filter, true ())}&amp;filt_id={$filt_id}&amp;token={/envelope/token}"
     title="{gsa:i18n ($cap-type-plural)}" class="icon icon-sm">
    <img src="/img/list.svg" alt="{gsa:i18n ($cap-type-plural)}"/>
  </a>
  <span class="divider"/>
  <xsl:choose>
    <xsl:when test="$type = 'user'">
      <xsl:choose>
        <xsl:when test="name=/envelope/login/text()">
          <img src="/img/delete_inactive.svg" alt="{gsa:i18n ('Delete')}"
                title="{gsa:i18n ('Currently logged in as this user')}"
                class="icon icon-sm"/>
        </xsl:when>
        <xsl:when test="gsa:may (concat ('delete_', $type)) and writable!='0' and in_use='0'">
          <xsl:call-template name="delete-icon">
            <xsl:with-param name="type" select="$type"/>
            <xsl:with-param name="id" select="@id"/>
            <xsl:with-param name="params">
              <input type="hidden" name="filter" value="{$filter}"/>
              <input type="hidden" name="filt_id" value="{$filt_id}"/>
            </xsl:with-param>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <xsl:variable name="inactive_text">
            <xsl:choose>
              <!-- i18n with concat : see dynamic_strings.xsl - type-action-denied -->
              <xsl:when test="in_use != '0'">
                <xsl:value-of select="gsa:i18n (concat ($cap-type, ' is still in use'))"/>
              </xsl:when>
              <xsl:when test="writable = '0'">
                <xsl:value-of select="gsa:i18n (concat ($cap-type, ' is not writable'))"/>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="gsa:i18n (concat ($cap-type, ' cannot be deleted'))"/>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:variable>
          <img src="/img/delete_inactive.svg" alt="{gsa:i18n ('Delete')}"
                title="{$inactive_text}"
                class="icon icon-sm"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>
    <xsl:otherwise>
      <xsl:choose>
        <xsl:when test="gsa:may (concat ('delete_', $type)) and writable!='0' and in_use='0'">
          <xsl:call-template name="trashcan-icon">
            <xsl:with-param name="type" select="$type"/>
            <xsl:with-param name="id" select="@id"/>
            <xsl:with-param name="params">
              <input type="hidden" name="filter" value="{$filter}"/>
              <input type="hidden" name="filt_id" value="{$filt_id}"/>
            </xsl:with-param>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <xsl:variable name="inactive_text">
            <xsl:choose>
              <!-- i18n with concat : see dynamic_strings.xsl - type-action-denied -->
              <xsl:when test="in_use != '0'">
                <xsl:value-of select="gsa:i18n (concat ($cap-type, ' is still in use'))"/>
              </xsl:when>
              <xsl:when test="writable = '0'">
                <xsl:value-of select="gsa:i18n (concat ($cap-type, ' is not writable'))"/>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="gsa:i18n (concat ($cap-type, ' cannot be moved to the trashcan'))"/>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:variable>
          <img src="/img/trashcan_inactive.svg" alt="{gsa:i18n ('To Trashcan', 'Action Verb')}"
                title="{$inactive_text}"
                class="icon icon-sm"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:otherwise>
  </xsl:choose>
  <xsl:choose>
    <xsl:when test="$noedit">
    </xsl:when>
    <xsl:otherwise>
      <xsl:choose>
        <xsl:when test="gsa:may (concat ('modify_', $type)) and writable!='0'">
          <!-- i18n with concat : see dynamic_strings.xsl - type-edit -->
          <a href="/omp?cmd=edit_{$type}&amp;{$type}_id={@id}&amp;next=get_{$type}&amp;filter={str:encode-uri ($filter, true ())}&amp;filt_id={$filt_id}&amp;token={/envelope/token}" data-reload="window"
              class="edit-action-icon icon icon-sm" data-type="{$type}" data-id="{@id}"
              title="{gsa:i18n (concat ('Edit ', $cap-type))}">
            <img src="/img/edit.svg"/>
          </a>
        </xsl:when>
        <xsl:otherwise>
          <!-- i18n with concat : see dynamic_strings.xsl - type-action-denied -->
          <img src="/img/edit_inactive.svg" alt="{gsa:i18n ('Edit', 'Action Verb')}"
                title="{gsa:i18n (concat ($cap-type, ' is not writable'))}"
                class="icon icon-sm"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:otherwise>
  </xsl:choose>
  <xsl:choose>
    <xsl:when test="$noexport">
    </xsl:when>
    <xsl:otherwise>
      <!-- i18n with concat : see dynamic_strings.xsl - type-export-xml -->
      <a href="/omp?cmd=export_{$type}&amp;{$type}_id={@id}&amp;filter={str:encode-uri ($filter, true ())}&amp;filt_id={$filt_id}&amp;token={/envelope/token}"
          title="{gsa:i18n (concat ('Export ', $cap-type, ' as XML'))}"
          class="icon icon-sm">
        <img src="/img/download.svg" alt="{gsa:i18n ('Export XML', 'Action Verb')}"/>
      </a>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="gsad_msg">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">
      <xsl:value-of select="@operation"/>
    </xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
    <xsl:with-param name="details">
      <xsl:value-of select="text()"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template name="certificate-info-table">
  <xsl:param name="certificate_info"/>
  <table>
    <tr>
      <td><xsl:value-of select="gsa:i18n ('Activation', 'Certificate')"/>:</td>
      <td>
        <xsl:value-of select="$certificate_info/activation_time"/>
        <xsl:if test="$certificate_info/time_status = 'inactive'">
          <xsl:text> </xsl:text>
          <b>(<xsl:value-of select="gsa:i18n ('not active yet', 'Certificate')"/>)</b>
        </xsl:if>
      </td>
    </tr>
    <tr>
      <td><xsl:value-of select="gsa:i18n ('Expiration', 'Certificate')"/>:</td>
      <td>
        <xsl:value-of select="$certificate_info/expiration_time"/>
        <xsl:if test="$certificate_info/time_status = 'expired'">
          <xsl:text> </xsl:text>
          <b>(<xsl:value-of select="gsa:i18n ('expired', 'Certificate')"/>)</b>
        </xsl:if>
      </td>
    </tr>
    <tr>
      <td><xsl:value-of select="gsa:i18n ('MD5 Fingerprint')"/>:</td>
      <td><xsl:value-of select="$certificate_info/md5_fingerprint"/></td>
    </tr>
    <tr>
      <td><xsl:value-of select="gsa:i18n ('Issued by', 'Certificate')"/>:</td>
      <td><xsl:value-of select="$certificate_info/issuer"/></td>
    </tr>
  </table>
</xsl:template>

<xsl:template name="certificate-status">
  <xsl:param name="certificate_info"/>

  <xsl:choose>
    <xsl:when test="$certificate_info/time_status = 'expired'">
      <xsl:value-of select="gsa-i18n:strformat (gsa:i18n ('Certificate currently in use expired %1'), $certificate_info/expiration_time)"/>
    </xsl:when>
    <xsl:when test="$certificate_info/time_status = 'inactive'">
      <xsl:value-of select="gsa-i18n:strformat (gsa:i18n ('Certificate currently in use is not valid until %1'), $certificate_info/activation_time)"/>
    </xsl:when>
    <xsl:otherwise>
      <xsl:value-of select="gsa-i18n:strformat (gsa:i18n ('Certificate currently in use will expire %1'), $certificate_info/expiration_time)"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="commands_response">
  <xsl:apply-templates/>
</xsl:template>


<!-- BEGIN TASKS MANAGEMENT -->

<xsl:template match="message">
  <div class="message">
    <xsl:apply-templates/>
  </div>
</xsl:template>

<xsl:template match="error">
  <div class="error">
    <xsl:apply-templates/>
  </div>
</xsl:template>

<xsl:template match="status">
</xsl:template>

<xsl:template match="hole">
  H=<xsl:apply-templates/>
</xsl:template>

<xsl:template match="warning">
  W=<xsl:apply-templates/>
</xsl:template>

<xsl:template match="info">
  I=<xsl:apply-templates/>
</xsl:template>

<xsl:template match="debug">
  D=<xsl:apply-templates/>
</xsl:template>

<xsl:template match="log">
  L=<xsl:apply-templates/>
</xsl:template>

<xsl:template match="false_positive">
  F=<xsl:apply-templates/>
</xsl:template>

<xsl:template match="result_count">
  <div>
    <xsl:apply-templates/>
  </div>
</xsl:template>

<xsl:template match="create_report_response" mode="upload">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Import Report</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="create_report_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Create Container Task</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="create_task_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Create Task</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="delete_task_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Delete Task</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="delete_report_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Delete Report</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="run_wizard_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Run Wizard</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="start_task_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Start Task</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="stop_task_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Stop Task</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="resume_task_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Resume Task</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="move_task_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Move Task</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!-- NEW_TASK -->

<xsl:template match="task_count">
</xsl:template>

<xsl:template match="new_container_task">
  <div class="edit-dialog">
    <div class="title">
      <xsl:value-of select="gsa:i18n ('New Container Task')"/>
   </div>
   <div class="content">
    <form action="/omp" method="post" enctype="multipart/form-data" class="form-horizontal">
      <input type="hidden" name="token" value="{/envelope/token}"/>
      <input type="hidden" name="cmd" value="create_container_task"/>
      <input type="hidden" name="caller" value="{/envelope/current_page}"/>
      <input type="hidden" name="next" value="get_task"/>
      <xsl:if test="string-length (/envelope/params/filt_id) = 0">
        <input type="hidden" name="overrides" value="{/envelope/params/overrides}"/>
      </xsl:if>
      <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
      <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
      <div class="form-group">
        <label class="col-2 control-label">
          <xsl:value-of select="gsa:i18n ('Name')"/>
        </label>
        <div class="col-10">
          <input type="text" name="name" value="unnamed" size="30"
            maxlength="80" class="form-control"/>
        </div>
      </div>
      <div class="form-group">
        <label class="col-2 control-label">
          <xsl:value-of select="gsa:i18n ('Comment')"/>
        </label>
        <div class="col-10">
          <input type="text" name="comment" size="30" maxlength="400"
            class="form-control"/>
        </div>
      </div>
    </form>
   </div>
  </div>
</xsl:template>

<xsl:template name="new-task-scanner-fields">
  <xsl:param name="scanner-type" select="2"/>

  <div class="form-group offset-container offset-2 form-selection-item-scanner form-selection-item-scanner--{$scanner-type}">
    <input type="hidden" name="scanner_type" value="{$scanner-type}" class="form-selection-input-scanner form-selection-input-scanner--{$scanner-type}"/>
    <div class="form-group">
      <label class="col-4 control-label">
        <xsl:value-of select="gsa:i18n ('Scan Config')"/>
      </label>
      <div class="col-8">
        <xsl:variable name="config_id" select="config_id"/>
        <select name="config_id" class="form-selection-input-scanner form-selection-input-scanner--{$scanner-type}">
          <!-- Skip the "empty" config. -->
          <xsl:for-each select="get_configs_response/config[@id!='085569ce-73ed-11df-83c3-002264764cea' and type = 0]">
            <xsl:choose>
              <xsl:when test="@id = $config_id">
                <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
              </xsl:when>
              <xsl:otherwise>
                <option value="{@id}"><xsl:value-of select="name"/></option>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:for-each>
        </select>
      </div>
    </div>
    <div class="form-group">
      <label class="col-4 control-label">
        <xsl:value-of select="gsa:i18n ('Network Source Interface')"/>
      </label>
      <div class="col-8">
        <div class="form-item">
          <input type="text" name="source_iface"
            class="form-control"
            value="{/envelope/params/source_iface}"/>
        </div>
      </div>
    </div>
    <div class="form-group">
      <label class="col-4 control-label">
        <xsl:value-of select="gsa:i18n ('Order for target hosts')"/>
      </label>
      <div class="col-8">
        <select name="hosts_ordering">
          <option value="sequential" selected="1"><xsl:value-of select="gsa:i18n ('Sequential', 'Task|Hosts Ordering')"/></option>
          <option value="random"><xsl:value-of select="gsa:i18n ('Random', 'Task|Hosts Ordering')"/></option>
          <option value="reverse"><xsl:value-of select="gsa:i18n ('Reverse', 'Task|Hosts Ordering')"/></option>
        </select>
      </div>
    </div>
    <div class="form-group">
      <label class="col-4 control-label">
        <xsl:value-of select="gsa:i18n ('Maximum concurrently executed NVTs per host')"/>
      </label>
      <div class="col-8">
        <input type="text" name="max_checks" value="{gsa:param-or ('max_checks', '4')}"
          data-type="int" min="0" class="spinner"
          size="10" maxlength="10"/>
      </div>
    </div>
    <div class="form-group">
      <label class="col-4 control-label">
        <xsl:value-of select="gsa:i18n ('Maximum concurrently scanned hosts')"/>
      </label>
      <div class="col-8">
        <input type="text" name="max_hosts" value="{gsa:param-or ('max_hosts', '20')}"
          data-type="int" class="spinner" min="0"
          size="10" maxlength="10"/>
      </div>
    </div>
  </div>
</xsl:template>

<xsl:template match="new_task">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="create_task_response"/>
  <xsl:apply-templates select="create_report_response"/>

  <div class="edit-dialog">
   <div class="title"><xsl:value-of select="gsa:i18n ('New Task')"/>
   </div>
   <div class="content">
    <form action="/omp" method="post" enctype="multipart/form-data" class="form-horizontal">
      <input type="hidden" name="token" value="{/envelope/token}"/>
      <input type="hidden" name="cmd" value="create_task"/>
      <input type="hidden" name="caller" value="{/envelope/current_page}"/>
      <input type="hidden" name="next" value="get_task"/>
      <xsl:if test="string-length (/envelope/params/filt_id) = 0">
        <input type="hidden" name="overrides" value="{/envelope/params/overrides}"/>
      </xsl:if>
      <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
      <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
      <xsl:if test="not (gsa:may-op ('get_schedules'))">
        <input type="hidden" name="schedule_id" value="0"/>
      </xsl:if>
      <div class="form-group">
        <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Name')"/></label>
        <div class="col-10">
          <input type="text" name="name" value="{gsa:param-or ('name', 'unnamed')}" size="30"
            class="form-control"
            maxlength="80"/>
        </div>
      </div>
      <div class="form-group">
        <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Comment')"/></label>
        <div class="col-10" colspan="2">
          <input type="text" name="comment" value="{gsa:param-or ('comment', '')}" size="30" maxlength="400"
            class="form-control" />
        </div>
      </div>
      <div class="form-group">
        <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Scan Targets')"/></label>
        <div class="col-10">
          <div class="form-item">
            <select name="target_id">
              <xsl:variable name="target_id">
                <xsl:value-of select="target_id"/>
              </xsl:variable>
              <xsl:for-each select="get_targets_response/target">
                <xsl:choose>
                  <xsl:when test="@id = $target_id">
                    <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
                  </xsl:when>
                  <xsl:otherwise>
                    <option value="{@id}"><xsl:value-of select="name"/></option>
                  </xsl:otherwise>
                </xsl:choose>
              </xsl:for-each>
            </select>
          </div>
          <div class="form-item">
            <a href="#" title="{ gsa:i18n('Create a new target') }"
              class="new-action-icon icon icon-sm" data-type="target" data-done="select[name=target_id]">
              <img src="/img/new.svg"/>
            </a>
          </div>
        </div>
      </div>
      <xsl:if test="gsa:may-op ('get_alerts')">
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Alerts')"/></label>
          <div class="col-10">
            <xsl:variable name="alerts"
              select="get_alerts_response/alert"/>
            <div class="form-item">
              <select name="alert_ids:" multiple="multiple" class="form-control" id="alert_ids">
                <xsl:for-each select="$alerts">
                  <option value="{@id}"><xsl:value-of select="name"/></option>
                </xsl:for-each>
              </select>
            </div>
            <div class="form-item">
              <a href="#" title="{ gsa:i18n('Create a new alert') }"
                  class="new-action-icon icon icon-sm" data-type="alert" data-done="#alert_ids">
                  <img src="/img/new.svg"/>
              </a>
            </div>
          </div>
        </div>
      </xsl:if>
      <xsl:if test="gsa:may-op ('get_schedules')">
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Schedule')"/></label>
          <div class="col-10">
            <div class="form-item">
              <select name="schedule_id">
                <xsl:variable name="schedule_id"
                  select="schedule_id"/>
                <xsl:choose>
                  <xsl:when test="string-length ($schedule_id) &gt; 0">
                    <option value="0">--</option>
                  </xsl:when>
                  <xsl:otherwise>
                    <option value="0" selected="1">--</option>
                  </xsl:otherwise>
                </xsl:choose>
                <xsl:for-each select="get_schedules_response/schedule">
                  <xsl:choose>
                    <xsl:when test="@id = $schedule_id">
                      <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
                    </xsl:when>
                    <xsl:otherwise>
                      <option value="{@id}"><xsl:value-of select="name"/></option>
                    </xsl:otherwise>
                  </xsl:choose>
                </xsl:for-each>
              </select>
            </div>
            <div class="form-item">
              <div class="checkbox">
                <label>
                  <input name="schedule_periods" type="checkbox" value="1"
                    title="{gsa:i18n ('Once')}"/>
                  <xsl:value-of select="gsa:i18n ('Once')"/>
                </label>
              </div>
            </div>
            <div class="form-item">
              <a href="#" title="{ gsa:i18n('Create a new schedule') }"
                class="new-action-icon icon icon-sm" data-type="schedule"
                data-done="select[name=schedule_id]">
                <img src="/img/new.svg"/>
              </a>
            </div>
          </div>
        </div>
      </xsl:if>
      <div class="form-group">
        <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Add results to Assets')"/></label>
        <div class="col-10">
          <xsl:variable name="yes" select="/envelope/params/in_assets"/>
          <div class="form-item">
            <div class="radio">
              <label>
                <xsl:choose>
                  <xsl:when test="string-length ($yes) = 0 or $yes = 1">
                    <input type="radio" name="in_assets" value="1" checked="1"
                      disable-on="0"
                      class="form-enable-control" id="in-assets"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="radio" name="in_assets" value="1"
                      disable-on="0"
                      class="form-enable-control" id="in-assets"/>
                  </xsl:otherwise>
                </xsl:choose>
                <xsl:value-of select="gsa:i18n ('yes')"/>
              </label>
            </div>
          </div>
          <div class="form-item">
            <div class="radio">
              <label>
                <xsl:choose>
                  <xsl:when test="string-length ($yes) = 0 or $yes = 1">
                    <input type="radio" name="in_assets" value="0"
                      disable-on="0"
                      class="form-enable-control" id="in-assets"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="radio" name="in_assets" value="0" checked="1"
                      disable-on="0"
                      class="form-enable-control" id="in-assets"/>
                  </xsl:otherwise>
                </xsl:choose>
                <xsl:value-of select="gsa:i18n ('no')"/>
              </label>
            </div>
          </div>
        </div>
      </div>
      <div class="offset-container offset-2 col-10 form-enable-item--in-assets">
        <div class="form-group">
          <label class="col-3 control-label"><xsl:value-of select="gsa:i18n ('Apply Overrides')"/></label>
          <div class="col-9">
            <div class="form-item">
              <div class="radio">
                <label>
                  <input type="radio" name="apply_overrides" value="1" checked="1"
                    class="form-enable-item--in-assets" />
                  <xsl:value-of select="gsa:i18n ('yes')"/>
                </label>
              </div>
            </div>
            <div class="form-item">
              <div class="radio">
                <label>
                  <input type="radio" name="apply_overrides" value="0"
                    class="form-enable-item--in-assets" />
                  <xsl:value-of select="gsa:i18n ('no')"/>
                </label>
              </div>
            </div>
          </div>
        </div>
        <div class="form-group">
          <label class="col-3 control-label"><xsl:value-of select="gsa:i18n ('Min QoD')"/></label>
          <div class="col-9">
            <div class="form-item">
              <input type="text" name="min_qod" value="70" size="4"
                class="spinner form-enable-item--in-assets"
                data-type="int" min="0" max="100"/>
            </div>
            <div class="form-item">
              <xsl:text>%</xsl:text>
            </div>
          </div>
        </div>
      </div>
      <div class="form-group">
        <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Alterable Task')"/></label>
        <div class="col-10">
          <xsl:variable name="yes" select="/envelope/params/alterable"/>
          <div class="form-item">
            <div class="radio">
              <label>
                <xsl:choose>
                  <xsl:when test="string-length ($yes) = 0 or $yes = 0">
                    <input type="radio" name="alterable" value="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="radio" name="alterable" value="1" checked="1"/>
                  </xsl:otherwise>
                </xsl:choose>
                <xsl:value-of select="gsa:i18n ('yes')"/>
              </label>
            </div>
          </div>
          <div class="form-item">
            <div class="radio">
              <label>
                <xsl:choose>
                  <xsl:when test="string-length ($yes) = 0 or $yes = 0">
                    <input type="radio" name="alterable" value="0" checked="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="radio" name="alterable" value="0"/>
                  </xsl:otherwise>
                </xsl:choose>
                <xsl:value-of select="gsa:i18n ('no')"/>
              </label>
            </div>
          </div>
        </div>
      </div>
      <div class="form-group">
        <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Auto Delete Reports')"/></label>
        <div class="col-10">
          <xsl:variable name="auto_delete" select="/envelope/params/auto_delete"/>
          <xsl:variable name="auto_delete_data" select="/envelope/params/auto_delete_data"/>
          <div class="radio">
            <label>
              <xsl:choose>
                <xsl:when test="$auto_delete = 'keep'">
                  <input type="radio" name="auto_delete" value="no"/>
                </xsl:when>
                <xsl:otherwise>
                  <input type="radio" name="auto_delete" value="no" checked="1"/>
                </xsl:otherwise>
              </xsl:choose>
              <xsl:value-of select="gsa:i18n ('Do not automatically delete reports')"/>
            </label>
          </div>
          <div class="radio">
            <label>
              <xsl:choose>
                <xsl:when test="$auto_delete = 'keep'">
                  <input type="radio" name="auto_delete" value="keep" checked="1"/>
                </xsl:when>
                <xsl:otherwise>
                  <input type="radio" name="auto_delete" value="keep"/>
                </xsl:otherwise>
              </xsl:choose>
              <xsl:value-of select="gsa:i18n ('Automatically delete oldest reports but always keep newest ', 'Task|Auto Delete Reports')"/>
              <div class="form-item">
                <input class="spinner" data-type="int" min="0"
                  type="text" name="auto_delete_data" value="5"
                  size="4" maxlength="5"/>
              </div>
              <xsl:value-of select="gsa:i18n (' reports', 'Task|Auto Delete Reports')"/>
            </label>
          </div>
        </div>
      </div>

      <div class="form-group">
        <label class="col-2 control-label">
          <xsl:value-of select="gsa:i18n ('Scanner')"/>
        </label>
        <div class="col-10">
          <div class="form-item">
            <xsl:variable name="scanner_id">
              <xsl:choose>
                <xsl:when test="string-length (scanner_id) &gt; 0">
                  <xsl:value-of select="scanner_id"/>
                </xsl:when>
                <xsl:otherwise>
                  <!-- use default scanner -->
                  <xsl:value-of select="'08b69003-5fc2-4037-a479-93b440211c73'"/>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:variable>
            <select name="scanner_id" class="form-selection-control"
              id="scanner">
              <xsl:for-each select="get_scanners_response/scanner">
                <xsl:choose>
                  <xsl:when test="@id = $scanner_id">
                    <option value="{@id}" selected="1" data-select="{type}"><xsl:value-of select="name"/></option>
                  </xsl:when>
                  <xsl:otherwise>
                    <option value="{@id}" data-select="{type}"><xsl:value-of select="name"/></option>
                  </xsl:otherwise>
                </xsl:choose>
              </xsl:for-each>
            </select>
          </div>
        </div>
      </div>

      <xsl:if test="count(get_scanners_response/scanner[type = 2])">
        <xsl:call-template name="new-task-scanner-fields">
          <xsl:with-param name="scanner-type" select="2"/>
        </xsl:call-template>
      </xsl:if>

      <xsl:if test="count(get_scanners_response/scanner[type = 4])">
        <xsl:call-template name="new-task-scanner-fields">
          <xsl:with-param name="scanner-type" select="4"/>
        </xsl:call-template>
      </xsl:if>

      <xsl:if test="count(get_scanners_response/scanner[type = 1]) and count(get_configs_response/config[type = 1])">
        <div class="form-group offset-container offset-2 form-selection-item-scanner form-selection-item-scanner--1">
          <input type="hidden" name="scanner_type" value="1" class="form-selection-input-scanner form-selection-input-scanner--1"/>
          <div class="form-group">
            <label class="col-4 control-label">
              <xsl:value-of select="gsa:i18n ('Scan Config')"/>
            </label>
            <div class="col-8">
              <xsl:variable name="osp_config_id" select="osp_config_id"/>
              <select name="config_id" class="form-selection-input-scanner form-selection-input-scanner--1">
                <!-- Skip the "empty" config. -->
                <xsl:for-each select="get_configs_response/config[type = 1]">
                  <xsl:choose>
                    <xsl:when test="@id = $osp_config_id">
                      <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
                    </xsl:when>
                    <xsl:otherwise>
                      <option value="{@id}"><xsl:value-of select="name"/></option>
                    </xsl:otherwise>
                  </xsl:choose>
                </xsl:for-each>
              </select>
            </div>
          </div>
        </div>
      </xsl:if>

      <xsl:if test="count(get_scanners_response/scanner[type = 3])">
        <div class="form-group offset-container offset-2 form-selection-item-scanner form-selection-item-scanner--3">
          <input type="hidden" name="scanner_type" value="3" class="form-selection-input-scanner form-selection-input-scanner--3"/>
        </div>
      </xsl:if>

      <xsl:if test="gsa:may-op ('get_tags') and gsa:may-op ('create_task') and count(get_tags_response/tag) != 0">
        <div class="form-group">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Tag')"/>
          </label>
          <div class="col-10">
            <div class="form-item">
              <div class="checkbox">
                <label>
                  <xsl:choose>
                    <xsl:when test="/envelope/params/add_tag != 0">
                      <input type="checkbox" name="add_tag" value="1" checked="1"/>
                    </xsl:when>
                    <xsl:otherwise>
                      <input type="checkbox" name="add_tag" value="1"/>
                    </xsl:otherwise>
                  </xsl:choose>
                  <xsl:value-of select="gsa:i18n ('Add Tag')"/>:
                </label>
              </div>
            </div>
            <div class="form-item">
              <select name="tag_name">
                <xsl:for-each select="get_tags_response/tag">
                  <xsl:call-template name="opt">
                    <xsl:with-param name="value" select="name"/>
                    <xsl:with-param name="select-value" select="/envelope/params/tag_name"/>
                  </xsl:call-template>
                </xsl:for-each>
              </select>
            </div>
            <div class="form-item">
              <xsl:value-of select="gsa:i18n ('with Value', 'Tag')"/>
            </div>
            <div class="form-item">
              <input name="tag_value" type="text" value="{/envelope/params/tag_value}"
                class="form-control"/>
            </div>
          </div>
        </div>
      </xsl:if>
    </form>
   </div>
  </div>
</xsl:template>

<!-- LAST_REPORT -->

<xsl:template match="last_report">
  <xsl:apply-templates/>
</xsl:template>

<!-- REPORT -->

<xsl:template match="report" name="report">
  <xsl:param name="container">0</xsl:param>
  <xsl:param name="observed" select="0"/>
  <xsl:param name="apply_overrides" select="../../../../apply_overrides"/>
  <xsl:param name="min_qod" select="../../filters/keywords/keyword[column='min_qod']/value"/>
  <xsl:param name="delta" select="/envelope/params/delta_report_id"/>
  <xsl:param name="task_id" select="/envelope/params/task_id"/>

  <tr class="{gsa:table-row-class(position())}">
    <td>
      <b>
        <a href="/omp?cmd=get_report&amp;report_id={@id}&amp;notes=1&amp;overrides={$apply_overrides}&amp;min_qod={$min_qod}&amp;result_hosts_only=1&amp;token={/envelope/token}"
           title="{gsa:view_details_title ('Report', @id)}">
          <xsl:value-of select="concat (date:day-abbreviation (timestamp), ' ', date:month-abbreviation (timestamp), ' ', date:day-in-month (timestamp), ' ', format-number(date:hour-in-day(timestamp), '00'), ':', format-number(date:minute-in-hour(timestamp), '00'), ':', format-number(date:second-in-minute(timestamp), '00'), ' ', date:year(timestamp))"/>
        </a>
      </b>
    </td>
    <td>
      <xsl:call-template name="status_bar">
        <xsl:with-param name="status">
          <xsl:choose>
            <xsl:when test="task/target/@id='' and scan_run_status='Running'">
              <xsl:text>Uploading</xsl:text>
            </xsl:when>
            <xsl:when test="task/target/@id=''">
              <xsl:text>Container</xsl:text>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="scan_run_status"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:with-param>
        <xsl:with-param name="progress">
          <xsl:value-of select="task/progress/text()"/>
        </xsl:with-param>
      </xsl:call-template>
    </td>
    <td style="max-width: 100px; overflow: hidden;">
      <a href="/omp?cmd=get_task&amp;task_id={task/@id}&amp;overrides={../../filters/keywords/keyword[column='apply_overrides']/value}&amp;min_qod={$min_qod}&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/task_filt_id}&amp;token={/envelope/token}"
         title="{gsa:view_details_title ('Task', task/@id)}"
         style="margin-left:3px;">
        <xsl:value-of select="task/name"/>
      </a>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="severity/filtered &lt; 0.0">
          <xsl:call-template name="severity-bar">
            <xsl:with-param name="cvss" select="''"/>
            <xsl:with-param name="extra_text" select="gsa:i18n (gsa:result-cvss-risk-factor (severity/filtered), 'Severity')"/>
            <xsl:with-param name="title" select="gsa:i18n (gsa:result-cvss-risk-factor (severity/filtered), 'Severity')"/>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="severity-bar">
            <xsl:with-param name="cvss" select="severity/filtered"/>
            <xsl:with-param name="extra_text" select="concat (' (', gsa:i18n (gsa:result-cvss-risk-factor (severity/filtered), 'Severity'), ')')"/>
            <xsl:with-param name="title" select="gsa:i18n (gsa:result-cvss-risk-factor (severity/filtered), 'Severity')"/>
          </xsl:call-template>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td class="threat_info_table">
      <xsl:value-of select="result_count/hole/filtered"/>
    </td>
    <td class="threat_info_table">
      <xsl:value-of select="result_count/warning/filtered"/>
    </td>
    <td class="threat_info_table">
      <xsl:value-of select="result_count/info/filtered"/>
    </td>
    <td class="threat_info_table">
      <xsl:value-of select="result_count/log/filtered"/>
    </td>
    <td class="threat_info_table">
      <xsl:value-of select="result_count/false_positive/filtered"/>
    </td>
    <xsl:choose>
      <xsl:when test="/envelope/params/bulk_select = 1">
        <td style="text-align:center">
          <label style="width:100%">
            <input name="bulk_selected:{@id}" type="checkbox" style="width:100%; height:100%" title="{gsa:i18n ('Select for bulk action')}"/>
          </label>
        </td>
      </xsl:when>
      <xsl:otherwise>
        <td class="table-actions">
          <xsl:choose>
            <xsl:when test="$delta = @id">
              <img src="/img/delta_inactive.svg" alt="{gsa:i18n ('Compare')}"
                  title="{gsa:i18n ('Report is already selected for delta')}"
                  class="icon icon-sm"/>
            </xsl:when>
            <xsl:when test="string-length (../../filters/keywords/keyword[column='task_id']) = 0">
              <img src="/img/delta_inactive.svg" alt="{gsa:i18n ('Compare')}"
                  title="{gsa:i18n ('Filter must be limited to a single Task to allow delta reports')}"
                  class="icon icon-sm"/>
            </xsl:when>
            <xsl:when test="string-length ($delta) &gt; 0">
              <a href="/omp?cmd=get_report&amp;report_id={$delta}&amp;delta_report_id={@id}&amp;notes=1&amp;overrides={$apply_overrides}&amp;result_hosts_only=1&amp;token={/envelope/token}"
                title="{gsa:i18n ('Compare')}"
                class="icon icon-sm">
                <img src="/img/delta_second.svg" alt="{gsa:i18n ('Compare')}"/>
              </a>
            </xsl:when>
            <xsl:otherwise>
              <a href="/omp?cmd=get_reports&amp;replace_task_id=1&amp;task_id={$task_id}&amp;delta_report_id={@id}&amp;overrides={$apply_overrides}&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;task_filter={str:encode-uri (/envelope/params/task_filter, true ())}&amp;task_filt_id={/envelope/params/task_filt_id}&amp;token={/envelope/token}"
                title="{gsa:i18n ('Compare')}"
                class="icon icon-sm">
                <img src="/img/delta.svg" alt="{gsa:i18n ('Compare')}"/>
              </a>
            </xsl:otherwise>
          </xsl:choose>
          <xsl:choose>
            <xsl:when test="boolean ($observed)">
              <img src="/img/delete_inactive.svg"
                  alt="{gsa:i18n ('Delete')}"
                  title="{gsa:i18n ('Report is observed')}"
                  class="icon icon-sm"/>
            </xsl:when>
            <xsl:when test="scan_run_status='Running' or scan_run_status='Requested' or scan_run_status='Stop Requested' or scan_run_status='Resume Requested'">
              <img src="/img/delete_inactive.svg"
                  alt="{gsa:i18n ('Delete')}"
                  title="{gsa:i18n ('Scan is active')}"
                  class="icon icon-sm"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:call-template name="delete-icon">
                <xsl:with-param name="type">report</xsl:with-param>
                <xsl:with-param name="id" select="@id"/>
                <xsl:with-param name="params">
                  <input type="hidden" name="task_id" value="{$task_id}"/>
                  <input type="hidden" name="overrides" value="{/envelope/params/overrides}"/>
                  <input type="hidden" name="next" value="get_reports"/>
                  <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
                  <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
                  <input type="hidden" name="task_filter" value="{/envelope/params/task_filter}"/>
                  <input type="hidden" name="task_filt_id" value="{/envelope/params/task_filt_id}"/>
                </xsl:with-param>
              </xsl:call-template>
            </xsl:otherwise>
          </xsl:choose>
        </td>
      </xsl:otherwise>
    </xsl:choose>
  </tr>
</xsl:template>

<!-- LAST_REPORT -->

<xsl:template match="last_report">
  <xsl:choose>
    <xsl:when test="report/severity &lt; 0.0">
      <xsl:call-template name="severity-bar">
        <xsl:with-param name="cvss" select="''"/>
        <xsl:with-param name="extra_text" select="gsa:i18n (gsa:result-cvss-risk-factor (report/severity), 'Severity')"/>
        <xsl:with-param name="title" select="gsa:i18n (gsa:result-cvss-risk-factor (report/severity), 'Severity')"/>
      </xsl:call-template>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="severity-bar">
        <xsl:with-param name="cvss" select="report/severity"/>
        <xsl:with-param name="extra_text" select="concat (' (', gsa:i18n (gsa:result-cvss-risk-factor (report/severity), 'Severity'), ')')"/>
        <xsl:with-param name="title" select="gsa:i18n (gsa:result-cvss-risk-factor (report/severity), 'Severity')"/>
      </xsl:call-template>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="html-edit-task-config">
  <xsl:param name="type"/>
  <xsl:param name="param_name"/>
  <xsl:param name="scanner_type"/>
  <div class="form-group">
    <label class="col-4 control-label">
      <xsl:value-of select="gsa:i18n ('Scan Config')"/>
    </label>
    <div class="col-8">
      <xsl:variable name="config_id" select="gsa:param-or ('config_id', commands_response/get_tasks_response/task/config/@id)"/>
      <div class="form-item">
        <select name="config_id" class="form-selection-input-scanner form-selection-input-scanner--{$scanner_type}">
          <xsl:choose>
            <xsl:when test="string-length (commands_response/get_configs_response/config/name) &gt; 0">
              <xsl:for-each select="commands_response/get_configs_response/config[type = $type]">
                <xsl:choose>
                  <xsl:when test="@id = $config_id">
                    <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
                  </xsl:when>
                  <xsl:otherwise>
                    <option value="{@id}"><xsl:value-of select="name"/></option>
                  </xsl:otherwise>
                </xsl:choose>
              </xsl:for-each>
            </xsl:when>
            <xsl:otherwise>
              <option value="0">--</option>
            </xsl:otherwise>
          </xsl:choose>
        </select>
      </div>
      <div class="form-item">
        <a href="#" title="{ gsa:i18n('Create a new scan config') }"
          class="new-action-icon icon icon-sm" data-type="config" data-done="select[name=config_id]">
          <img src="/img/new.svg"/>
        </a>
      </div>
    </div>
  </div>
</xsl:template>

<xsl:template name="html-edit-task-config-disabled">
  <div class="form-group">
    <label class="col-4 control-label">
      <xsl:value-of select="gsa:i18n ('Scan Config')"/> (<xsl:value-of select="gsa:i18n ('immutable')"/>)
    </label>
    <div class="col-8">
      <select name="dummy" disabled="0">
        <xsl:choose>
          <xsl:when test="string-length (commands_response/get_tasks_response/task/config/name) &gt; 0">
            <xsl:apply-templates select="commands_response/get_tasks_response/task/config" mode="newtask"/>
          </xsl:when>
          <xsl:otherwise>
            <option value="0">--</option>
          </xsl:otherwise>
        </xsl:choose>
      </select>
    </div>
  </div>
</xsl:template>

<xsl:template name="html-edit-task-scanner-disabled">
  <div class="form-group">
    <label class="col-2 control-label">
      <xsl:value-of select="gsa:i18n ('Scanner')"/>
    </label>
    <div class="col-10">
      <xsl:variable name="scanner_id">
        <xsl:choose>
          <xsl:when test="string-length (/envelope/params/scanner_id) &gt; 0">
            <xsl:value-of select="/envelope/params/scanner_id"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="commands_response/get_tasks_response/task/scanner/@id"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:variable>
      <select name="dummy" disabled="0">
        <xsl:choose>
          <xsl:when test="string-length (commands_response/get_tasks_response/task/scanner/name) &gt; 0">
            <xsl:for-each select="commands_response/get_scanners_response/scanner">
              <xsl:if test="@id = $scanner_id">
                <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
              </xsl:if>
            </xsl:for-each>
          </xsl:when>
          <xsl:otherwise>
            <option value="0">--</option>
          </xsl:otherwise>
        </xsl:choose>
      </select>
    </div>
  </div>
</xsl:template>

<xsl:template name="html-edit-task-target">
  <xsl:choose>
    <xsl:when test="commands_response/get_tasks_response/task/status = 'New' or commands_response/get_tasks_response/task/alterable != 0">
      <div class="form-group">
        <label class="col-2 control-label">
          <xsl:value-of select="gsa:i18n ('Scan Targets')"/>
        </label>
        <div class="col-10">
          <xsl:variable name="target_id" select="gsa:param-or ('target_id', commands_response/get_tasks_response/task/target/@id)"/>
          <div class="form-item">
            <select name="target_id">
              <xsl:choose>
                <xsl:when test="string-length (commands_response/get_targets_response/target/name) &gt; 0">
                  <xsl:for-each select="commands_response/get_targets_response/target">
                    <xsl:choose>
                      <xsl:when test="@id = $target_id">
                        <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
                      </xsl:when>
                      <xsl:otherwise>
                        <option value="{@id}"><xsl:value-of select="name"/></option>
                      </xsl:otherwise>
                    </xsl:choose>
                  </xsl:for-each>
                </xsl:when>
                <xsl:otherwise>
                  <option value="0">--</option>
                </xsl:otherwise>
              </xsl:choose>
            </select>
          </div>
          <div class="form-item">
            <a href="#" title="{ gsa:i18n('Create a new scan target') }"
              class="new-action-icon icon icon-sm" data-type="target" data-done="select[name=target_id]">
              <img src="/img/new.svg"/>
            </a>
          </div>
        </div>
      </div>
    </xsl:when>
    <xsl:otherwise>
      <div class="form-group">
        <label class="col-2 control-label">
          <xsl:value-of select="gsa:i18n ('Scan Targets')"/> (<xsl:value-of select="gsa:i18n ('immutable')"/>)
        </label>
        <div class="col-10">
          <input type="hidden" name="target_id" value="0"/>
          <select name="dummy2" disabled="0">
            <xsl:choose>
              <xsl:when test="string-length (commands_response/get_tasks_response/task/target/name) &gt; 0">
                <xsl:apply-templates select="commands_response/get_tasks_response/task/target" mode="newtask"/>
              </xsl:when>
              <xsl:otherwise>
                <option value="0">--</option>
              </xsl:otherwise>
            </xsl:choose>
          </select>
        </div>
      </div>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="html-edit-task-alert">
  <xsl:if test="gsa:may-op ('get_alerts')">
    <div class="form-group">
      <label class="col-2 control-label">
        <xsl:value-of select="gsa:i18n ('Alerts')"/>
      </label>
      <div class="col-10">
        <xsl:variable name="task_alerts" select="commands_response/get_tasks_response/task/alert"/>
        <div class="form-item">
          <select name="alert_ids:" multiple="multiple" id="alert_ids">
            <xsl:for-each select="commands_response/get_alerts_response/alert">
              <xsl:variable name="alert_id" select="@id"/>
              <xsl:choose>
                <xsl:when test="$task_alerts[@id = $alert_id]">
                  <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
                </xsl:when>
                <xsl:otherwise>
                  <option value="{@id}"><xsl:value-of select="name"/></option>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:for-each>
          </select>
        </div>
        <div class="form-item">
          <a href="#" title="{ gsa:i18n('Create a new alert') }"
            class="new-action-icon icon icon-sm" data-type="alert" data-done="#alert_ids">
            <img src="/img/new.svg"/>
          </a>
        </div>
      </div>
    </div>
  </xsl:if>
</xsl:template>

<xsl:template name="html-edit-task-scanner">
  <div class="form-group">
    <label class="col-2 control-label">
      <xsl:value-of select="gsa:i18n ('Scanner')"/>
    </label>
    <div class="col-10">
      <select name="scanner_id" class="form-selection-control"
        id="scanner">
        <xsl:variable name="scanner_id">
          <xsl:choose>
            <xsl:when test="string-length (/envelope/params/scanner_id) &gt; 0">
              <xsl:value-of select="/envelope/params/scanner_id"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="commands_response/get_tasks_response/task/scanner/@id"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:variable>
        <xsl:for-each select="commands_response/get_scanners_response/scanner">
          <xsl:choose>
            <xsl:when test="@id = $scanner_id">
              <option value="{@id}" selected="1" data-select="{type}"><xsl:value-of select="name"/></option>
            </xsl:when>
            <xsl:otherwise>
              <option value="{@id}" data-select="{type}"><xsl:value-of select="name"/></option>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:for-each>
      </select>
    </div>
  </div>
</xsl:template>

<xsl:template name="html-edit-task-schedule">
  <xsl:if test="gsa:may-op ('get_schedules')">
    <div class="form-group">
      <label class="col-2 control-label">
        <xsl:value-of select="gsa:i18n ('Schedule')"/>
      </label>
      <div class="col-10">
        <div class="form-item">
          <select name="schedule_id">
            <xsl:variable name="schedule_id">
              <xsl:choose>
                <xsl:when test="string-length (/envelope/params/schedule_id) &gt; 0">
                  <xsl:value-of select="/envelope/params/schedule_id"/>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:value-of select="commands_response/get_tasks_response/task/schedule/@id"/>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:variable>
            <xsl:choose>
              <xsl:when test="string-length ($schedule_id) &gt; 0">
                <option value="0">--</option>
              </xsl:when>
              <xsl:otherwise>
                <option value="0" selected="1">--</option>
              </xsl:otherwise>
            </xsl:choose>
            <xsl:for-each select="commands_response/get_schedules_response/schedule">
              <xsl:choose>
                <xsl:when test="@id = $schedule_id">
                  <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
                </xsl:when>
                <xsl:otherwise>
                  <option value="{@id}"><xsl:value-of select="name"/></option>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:for-each>
          </select>
        </div>
        <div class="form-item">
          <div class="checkbox">
            <label>
              <xsl:choose>
                <xsl:when test="commands_response/get_tasks_response/task/schedule_periods = 1">
                  <input name="schedule_periods" type="checkbox" value="1" checked="1"
                        title="{gsa:i18n ('Once')}"/>
                  <xsl:value-of select="gsa:i18n ('Once')"/>
                </xsl:when>
                <xsl:when test="commands_response/get_tasks_response/task/schedule_periods &gt; 1">
                  <input name="schedule_periods" type="text"
                        value="{commands_response/get_tasks_response/task/schedule_periods}"
                        style="width:40px" checked="1" title="{gsa:i18n ('Periods', 'Time')}"/>
                  <xsl:value-of select="gsa:i18n ('more times')"/>
                </xsl:when>
                <xsl:otherwise>
                  <input name="schedule_periods" type="checkbox" value="1"
                        title="{gsa:i18n ('Once')}"/>
                  <xsl:value-of select="gsa:i18n ('Once')"/>
                </xsl:otherwise>
              </xsl:choose>
            </label>
          </div>
        </div>
        <div class="form-item">
          <a href="#" title="{ gsa:i18n('Create a schedule') }"
            class="new-action-icon icon icon-sm" data-type="schedule" data-done="select[name=schedule_id]">
            <img src="/img/new.svg"/>
          </a>
        </div>
      </div>
    </div>
  </xsl:if>
</xsl:template>

<xsl:template name="html-edit-task-scan-options">
  <xsl:variable name="container_task">
    <xsl:choose>
      <xsl:when test="commands_response/get_tasks_response/task/target/@id = ''">1</xsl:when>
      <xsl:otherwise>0</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>

  <div class="form-group">
    <xsl:variable name="in_assets" select="commands_response/get_tasks_response/task/preferences/preference[scanner_name='in_assets']"/>
    <label class="col-2 control-label">
      <xsl:value-of select="gsa:i18n ($in_assets/name, 'Task')"/>
    </label>
    <div class="col-10">
      <xsl:choose>
        <xsl:when test="$in_assets/value='yes'">
          <div class="form-item">
            <div class="radio">
              <label>
                <input type="radio" name="in_assets" value="1" checked="1"
                disable-on="0"
                class="form-enable-control" id="in-assets"/>
                <xsl:value-of select="gsa:i18n ('yes')"/>
              </label>
            </div>
          </div>
          <div class="form-item">
            <div class="radio">
              <label>
                <input type="radio" name="in_assets" value="0"
                disable-on="0"
                class="form-enable-control" id="in-assets"/>
                <xsl:value-of select="gsa:i18n ('no')"/>
              </label>
            </div>
          </div>
        </xsl:when>
        <xsl:otherwise>
          <div class="form-item">
            <div class="radio">
              <label>
                <input type="radio" name="in_assets" value="1"
                  disable-on="0"
                  class="form-enable-control" id="in-assets"/>
                <xsl:value-of select="gsa:i18n ('yes')"/>
              </label>
            </div>
          </div>
          <div class="form-item">
            <div class="radio">
              <label>
                <input type="radio" name="in_assets" value="0" checked="1"
                  disable-on="0"
                  class="form-enable-control" id="in-assets"/>
                <xsl:value-of select="gsa:i18n ('no')"/>
              </label>
            </div>
          </div>
        </xsl:otherwise>
      </xsl:choose>
    </div>
  </div>
  <xsl:if test="$container_task = 0">
    <div class="offset-container offset-2 col-10 form-enable-item--in-assets">
      <div class="form-group">
        <xsl:variable name="apply_overrides"
          select="commands_response/get_tasks_response/task/preferences/preference[scanner_name='assets_apply_overrides']"/>
        <label class="col-3 control-label">
          <xsl:value-of select="gsa:i18n ('Apply Overrides')"/>
        </label>
        <div class="col-9">
          <xsl:choose>
            <xsl:when test="$apply_overrides/value='yes'">
              <div class="form-item">
                <div class="radio">
                  <label>
                    <input type="radio" name="apply_overrides" value="1" checked="1"
                      class="form-enable-item--in-assets" />
                    <xsl:value-of select="gsa:i18n ('yes')"/>
                  </label>
                </div>
              </div>
              <div class="form-item">
                <div class="radio">
                  <label>
                    <input type="radio" name="apply_overrides" value="0"
                      class="form-enable-item--in-assets" />
                    <xsl:value-of select="gsa:i18n ('no')"/>
                  </label>
                </div>
              </div>
            </xsl:when>
            <xsl:otherwise>
              <div class="form-item">
                <div class="radio">
                  <label>
                    <input type="radio" name="apply_overrides" value="1"
                      class="form-enable-item--in-assets" />
                    <xsl:value-of select="gsa:i18n ('yes')"/>
                  </label>
                </div>
              </div>
              <div class="form-item">
                <div class="radio">
                  <label>
                    <input type="radio" name="apply_overrides" value="0" checked="1"
                      class="form-enable-item--in-assets" />
                    <xsl:value-of select="gsa:i18n ('no')"/>
                  </label>
                </div>
              </div>
            </xsl:otherwise>
          </xsl:choose>
        </div>
      </div>
      <div class="form-group">
        <xsl:variable name="min_qod"
          select="commands_response/get_tasks_response/task/preferences/preference[scanner_name='assets_min_qod']"/>
        <label class="col-3 control-label">
          <xsl:value-of select="gsa:i18n ('Min QoD')"/>
        </label>
        <div class="col-9">
          <div class="form-item">
            <input type="text" name="min_qod" value="{$min_qod/value}" size="4"
              class="spinner form-enable-item--in-assets"
              data-type="int" min="0"/>
          </div>
          <div class="form-item">
            <xsl:text>%</xsl:text>
          </div>
        </div>
      </div>
    </div>
  </xsl:if>
  <xsl:if test="commands_response/get_tasks_response/task/status = 'New' and $container_task = 0">
    <div class="form-group">
      <label class="col-2 control-label">
        <xsl:value-of select="gsa:i18n ('Alterable Task')"/>
      </label>
      <div class="col-10">
        <xsl:variable name="yes" select="commands_response/get_tasks_response/task/alterable"/>
        <div class="form-item">
          <div class="radio">
            <label>
              <xsl:choose>
                <xsl:when test="string-length ($yes) = 0 or $yes = 0">
                  <input type="radio" name="alterable" value="1"/>
                </xsl:when>
                <xsl:otherwise>
                  <input type="radio" name="alterable" value="1" checked="1"/>
                </xsl:otherwise>
              </xsl:choose>
              <xsl:value-of select="gsa:i18n ('yes')"/>
            </label>
          </div>
        </div>
        <div class="form-item">
          <div class="radio">
            <label>
              <xsl:choose>
                <xsl:when test="string-length ($yes) = 0 or $yes = 0">
                  <input type="radio" name="alterable" value="0" checked="1"/>
                </xsl:when>
                <xsl:otherwise>
                  <input type="radio" name="alterable" value="0"/>
                </xsl:otherwise>
              </xsl:choose>
              <xsl:value-of select="gsa:i18n ('no')"/>
            </label>
          </div>
        </div>
      </div>
    </div>
  </xsl:if>
  <div class="form-group">
    <label class="col-2 control-label">
      <xsl:value-of select="gsa:i18n ('Auto Delete Reports')"/>
    </label>
    <div class="col-10">
      <xsl:variable name="auto_delete">
        <xsl:choose>
          <xsl:when test="string-length (/envelope/params/auto_delete) &gt; 0">
            <xsl:value-of select="/envelope/params/auto_delete"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="commands_response/get_tasks_response/task/preferences/preference[scanner_name='auto_delete']/value"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:variable>
      <xsl:variable name="auto_delete_data">
        <xsl:choose>
          <xsl:when test="string-length (/envelope/params/auto_delete_data) &gt; 0">
            <xsl:value-of select="/envelope/params/auto_delete_data"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="commands_response/get_tasks_response/task/preferences/preference[scanner_name='auto_delete_data']/value"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:variable>
      <div class="radio">
        <label>
          <xsl:choose>
            <xsl:when test="$auto_delete = 'keep'">
              <input type="radio" name="auto_delete" value="no"/>
            </xsl:when>
            <xsl:otherwise>
              <input type="radio" name="auto_delete" value="no" checked="1"/>
            </xsl:otherwise>
          </xsl:choose>
          <xsl:value-of select="gsa:i18n ('Do not automatically delete reports')"/>
        </label>
      </div>
      <div class="radio">
        <label>
          <xsl:choose>
            <xsl:when test="$auto_delete = 'keep'">
              <input type="radio" name="auto_delete" value="keep" checked="1"/>
            </xsl:when>
            <xsl:otherwise>
              <input type="radio" name="auto_delete" value="keep"/>
            </xsl:otherwise>
          </xsl:choose>
          <xsl:value-of select="gsa:i18n ('Automatically delete oldest reports but always keep newest ', 'Task|Auto Delete Reports')"/>
          <div class="form-item">
            <xsl:variable name="data">
              <xsl:choose>
                <xsl:when test="$auto_delete_data = 0">5</xsl:when>
                <xsl:otherwise><xsl:value-of select="$auto_delete_data"/></xsl:otherwise>
              </xsl:choose>
            </xsl:variable>
            <input class="spinner"
              data-type="int" min="0"
              type="text" name="auto_delete_data" value="{$data}"
              size="4" maxlength="5"/>
          </div>
          <xsl:value-of select="gsa:i18n (' reports', 'Task|Auto Delete Reports')"/>
        </label>
      </div>
    </div>
  </div>
</xsl:template>

<xsl:template name="html-edit-task-openvas-options">
  <div class="form-group">
    <label class="col-4 control-label">
      <xsl:value-of select="gsa:i18n ('Network Source Interface')"/>
    </label>
    <div class="col-8">
      <div class="form-item">
        <input type="text" name="source_iface"
          class="form-control"
          value="{commands_response/get_tasks_response/task/preferences/preference[scanner_name='source_iface']/value}"/>
      </div>
    </div>
  </div>
  <div class="form-group">
    <label class="col-4 control-label">
      <xsl:value-of select="gsa:i18n ('Order for target hosts')"/>
    </label>
    <div class="col-8">
      <xsl:variable name="hosts_ordering"
                    select="commands_response/get_tasks_response/task/hosts_ordering"/>
      <select name="hosts_ordering">
        <xsl:call-template name="opt">
          <xsl:with-param name="content" select="gsa:i18n ('Sequential', 'Task|Hosts Ordering')"/>
          <xsl:with-param name="value" select="'sequential'"/>
          <xsl:with-param name="select-value" select="$hosts_ordering"/>
        </xsl:call-template>
        <xsl:call-template name="opt">
          <xsl:with-param name="content" select="gsa:i18n ('Random', 'Task|Hosts Ordering')"/>
          <xsl:with-param name="value" select="'random'"/>
          <xsl:with-param name="select-value" select="$hosts_ordering"/>
        </xsl:call-template>
        <xsl:call-template name="opt">
          <xsl:with-param name="content" select="gsa:i18n ('Reverse', 'Task|Hosts Ordering')"/>
          <xsl:with-param name="value" select="'reverse'"/>
          <xsl:with-param name="select-value" select="$hosts_ordering"/>
        </xsl:call-template>
      </select>
    </div>
  </div>
  <xsl:choose>
    <xsl:when test="commands_response/get_tasks_response/task/target/@id = ''">
      <input type="hidden" name="target_id" value="0"/>
    </xsl:when>
    <xsl:otherwise>
      <div class="form-group">
        <label class="col-4 control-label">
          <xsl:value-of select="gsa:i18n (commands_response/get_tasks_response/task/preferences/preference[scanner_name='max_checks']/name, 'Task')"/>
        </label>
        <div class="col-8">
          <div class="form-item">
            <input type="text" name="max_checks"
              class="spinner" data-type="int" min="0"
              value="{gsa:param-or ('max_checks', commands_response/get_tasks_response/task/preferences/preference[scanner_name='max_checks']/value)}"
              size="10" maxlength="10"/>
          </div>
        </div>
      </div>
      <div class="form-group">
        <label class="col-4 control-label">
          <xsl:value-of select="gsa:i18n (commands_response/get_tasks_response/task/preferences/preference[scanner_name='max_hosts']/name, 'Task')"/>
        </label>
        <div class="col-8">
          <div class="form-item">
            <input type="text" name="max_hosts"
              class="spinner" data-type="int" min="0"
              value="{gsa:param-or ('max_hosts', commands_response/get_tasks_response/task/preferences/preference[scanner_name='max_hosts']/value)}"
              size="10" maxlength="10"/>
          </div>
        </div>
      </div>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="html-edit-task-name">
  <div class="form-group">
    <label class="col-2 control-label">
      <xsl:value-of select="gsa:i18n ('Name')"/>
    </label>
    <div class="col-10">
      <input type="text" name="name"
        class="form-control"
        value="{gsa:param-or ('name', commands_response/get_tasks_response/task/name)}"
        size="30" maxlength="80"/>
    </div>
  </div>
</xsl:template>

<xsl:template name="html-edit-task-comment">
  <div class="form-group">
    <label class="col-2 control-label">
      <xsl:value-of select="gsa:i18n ('Comment')"/>
    </label>
    <div class="col-10">
      <input type="text" name="comment" size="30" maxlength="400"
        class="form-control"
        value="{gsa:param-or ('comment', commands_response/get_tasks_response/task/comment)}"/>
    </div>
  </div>
</xsl:template>

<xsl:template name="html-edit-task-form">

  <xsl:variable name="container_task">
    <xsl:choose>
      <xsl:when test="commands_response/get_tasks_response/task/target/@id = ''">1</xsl:when>
      <xsl:otherwise>0</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>

  <xsl:variable name="new_task">
    <xsl:choose>
      <xsl:when test="commands_response/get_tasks_response/task/status = 'New' or commands_response/get_tasks_response/task/alterable != 0">1</xsl:when>
      <xsl:otherwise>0</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>

  <div class="edit-dialog">
    <div class="title">
      <xsl:choose>
        <xsl:when test="$container_task = 1">
          <xsl:value-of select="gsa:i18n ('Edit Container Task')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="gsa:i18n ('Edit Task')"/>
        </xsl:otherwise>
      </xsl:choose>
    </div>

    <div class="content">
      <form action="" method="post" enctype="multipart/form-data" class="form-horizontal">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden"
               name="task_id"
               value="{commands_response/get_tasks_response/task/@id}"/>
        <input type="hidden" name="next" value="{next}"/>
        <input type="hidden" name="sort_field" value="{sort_field}"/>
        <input type="hidden" name="sort_order" value="{sort_order}"/>
        <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
        <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
        <xsl:if test="not (gsa:may-op ('get_schedules'))">
          <input type="hidden" name="schedule_id" value="0"/>
        </xsl:if>

        <xsl:call-template name="html-edit-task-name"/>

        <xsl:call-template name="html-edit-task-comment"/>


        <xsl:choose>
          <xsl:when test="$container_task = 1">

            <!-- Container -->

            <input type="hidden" name="target_id" value="0"/>
            <input type="hidden" name="cmd" value="save_container_task"/>
            <xsl:call-template name="html-edit-task-scan-options"/>
          </xsl:when>
          <xsl:otherwise>
            <input type="hidden" name="cmd" value="save_task"/>

            <!-- Regular task.  Immutable. -->

            <xsl:call-template name="html-edit-task-target"/>
            <xsl:call-template name="html-edit-task-alert"/>
            <xsl:call-template name="html-edit-task-schedule"/>
            <xsl:call-template name="html-edit-task-scan-options"/>

            <xsl:choose>
              <xsl:when test="$new_task = 0">
                <input type="hidden" name="scanner_type" value="{commands_response/get_tasks_response/task/scanner/type}"/>
                <input type="hidden" name="scanner_id" value="0"/>
                <input type="hidden" name="osp_scanner_id" value="0"/>
                <input type="hidden" name="cve_scanner_id" value="0"/>
                <input type="hidden" name="cmd" value="save_task"/>
                <input type="hidden" name="config_id" value="0"/>
                <input type="hidden" name="osp_config_id" value="0"/>

                <xsl:call-template name="html-edit-task-scanner-disabled"/>

                <div class="form-group offset-container offset-2 form-selection-item-scanner form-selection-item-scanner--{commands_response/get_tasks_response/task/scanner/type}">
                  <xsl:if test="commands_response/get_tasks_response/task/scanner/type != 3">
                    <xsl:call-template name="html-edit-task-config-disabled"/>
                  </xsl:if>
                  <xsl:if test="commands_response/get_tasks_response/task/scanner/type = 2 or commands_response/get_tasks_response/task/scanner/type = 4">
                    <xsl:call-template name="html-edit-task-openvas-options"/>
                  </xsl:if>
                </div>
              </xsl:when>
              <xsl:otherwise>
                <!-- Regular task.  Alterable. -->
                <xsl:call-template name="html-edit-task-scanner">
                </xsl:call-template>

                <!-- OpenVAS Scanner. -->
                <xsl:if test="count(commands_response/get_scanners_response/scanner[type = 2])">
                  <div class="form-group offset-container offset-2 form-selection-item-scanner form-selection-item-scanner--2">
                    <input type="hidden" name="scanner_type" value="2" class="form-selection-input-scanner form-selection-input-scanner--2"/>
                    <xsl:call-template name="html-edit-task-config">
                      <xsl:with-param name="scanner_type">2</xsl:with-param>
                      <xsl:with-param name="type">0</xsl:with-param>
                    </xsl:call-template>
                    <xsl:call-template name="html-edit-task-openvas-options"/>
                  </div>
                </xsl:if>

                <!-- OMP Scanner. -->
                <xsl:if test="count(commands_response/get_scanners_response/scanner[type = 4])">
                  <div class="form-group offset-container offset-2 form-selection-item-scanner form-selection-item-scanner--4">
                    <input type="hidden" name="scanner_type" value="4" class="form-selection-input-scanner form-selection-input-scanner--4"/>
                    <xsl:call-template name="html-edit-task-config">
                      <xsl:with-param name="scanner_type">4</xsl:with-param>
                      <xsl:with-param name="type">0</xsl:with-param>
                    </xsl:call-template>
                    <xsl:call-template name="html-edit-task-openvas-options"/>
                  </div>
                </xsl:if>

                <!-- OSP Scanner. -->
                <xsl:if test="count(commands_response/get_scanners_response/scanner[type = 1]) and count(commands_response/get_configs_response/config[type = 1])">
                  <div class="form-group offset-container offset-2 form-selection-item-scanner form-selection-item-scanner--1">
                    <input type="hidden" name="scanner_type" value="1" class="form-selection-input-scanner form-selection-input-scanner--1"/>
                    <xsl:call-template name="html-edit-task-config">
                      <xsl:with-param name="type">1</xsl:with-param>
                      <xsl:with-param name="scanner_type">1</xsl:with-param>
                    </xsl:call-template>
                  </div>
                </xsl:if>

                <!-- CVE Scanner. -->
                <xsl:if test="count(commands_response/get_scanners_response/scanner[type = 3])">
                  <div class="form-group offset-container offset-2 form-selection-item-scanner form-selection-item-scanner--3">
                    <input type="hidden" name="scanner_type" value="3" class="form-selection-input-scanner form-selection-input-scanner--3"/>
                  </div>
                </xsl:if>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:otherwise>
        </xsl:choose>
      </form>
    </div>
  </div>

  <xsl:if test="commands_response/get_tasks_response/task/target/@id = '' and gsa:may-op ('create_report')">
  </xsl:if>
</xsl:template>

<xsl:template match="edit_task">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="modify_task_response"/>
  <xsl:apply-templates select="move_task_response"/>
  <xsl:call-template name="html-edit-task-form"/>
</xsl:template>

<xsl:template match="modify_task_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Save Task</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!-- TASK -->

<xsl:template match="task">
  <xsl:choose>
    <xsl:when test="report">
      <xsl:variable name="class">
        <xsl:choose>
          <xsl:when test="position() mod 2 = 0">even</xsl:when>
          <xsl:otherwise>odd</xsl:otherwise>
        </xsl:choose>
      </xsl:variable>
      <xsl:apply-templates select="report"/>
    </xsl:when>
    <xsl:otherwise>

      <tr class="{gsa:table-row-class(position())}">
        <td>
          <div class="pull-right">
            <xsl:choose>
              <xsl:when test="alterable = 0">
              </xsl:when>
              <xsl:otherwise>
                <img src="/img/alterable.svg"
                  class="icon icon-sm"
                  alt="{gsa:i18n ('Task is alterable')}"
                  title="{gsa:i18n ('Task is alterable')}"/>
              </xsl:otherwise>
            </xsl:choose>
            <xsl:choose>
              <xsl:when test="scanner/type = 4">
                <img src="/img/sensor.svg"
                  class="icon icon-sm"
                  alt="{gsa-i18n:strformat (gsa:i18n ('Task is configured to run on slave scanner %1'), scanner/name)}"
                  title="{gsa-i18n:strformat (gsa:i18n ('Task is configured to run on slave scanner %1'), scanner/name)}"/>
              </xsl:when>
              <xsl:otherwise>
              </xsl:otherwise>
            </xsl:choose>
            <xsl:choose>
              <xsl:when test="string-length (observers) &gt; 0 or count (observers/group) &gt; 0">
                <xsl:variable name="observer_groups">
                  <xsl:choose>
                    <xsl:when test="count (observers/group) &gt; 0">
                      <xsl:value-of select="concat ('&#10;', gsa:i18n ('Task made visible for Groups:'),' ', gsa:join (observers/group))"/>
                    </xsl:when>
                    <xsl:otherwise></xsl:otherwise>
                  </xsl:choose>
                </xsl:variable>
                <xsl:variable name="observer_roles">
                  <xsl:choose>
                    <xsl:when test="count (observers/role) &gt; 0">
                      <xsl:value-of select="concat ('&#10;', gsa:i18n ('Task made visible for Roles:'),' ', gsa:join (observers/role))"/>
                    </xsl:when>
                    <xsl:otherwise></xsl:otherwise>
                  </xsl:choose>
                </xsl:variable>
                <img src="/img/provide_view.svg"
                  class="icon icon-sm"
                  alt="{gsa:i18n ('Task made visible for:')} {observers/text()}{$observer_groups}{$observer_roles}"
                  title="{gsa:i18n ('Task made visible for:')} {observers/text()}{$observer_groups}{$observer_roles}"/>
              </xsl:when>
              <xsl:otherwise>
              </xsl:otherwise>
            </xsl:choose>
            <xsl:choose>
              <xsl:when test="owner/name=/envelope/login/text()">
              </xsl:when>
              <xsl:otherwise>
                <img src="/img/view_other.svg"
                  class="icon icon-sm"
                  alt="{gsa-i18n:strformat (gsa:i18n ('Observing task owned by %1'), owner/name)}"
                  title="{gsa-i18n:strformat (gsa:i18n ('Observing task owned by %1'), owner/name)}"/>
              </xsl:otherwise>
            </xsl:choose>
          </div>
          <b>
            <a href="/omp?cmd=get_task&amp;task_id={@id}&amp;overrides={../filters/keywords/keyword[column='apply_overrides']/value}&amp;min_qod={../filters/keywords/keyword[column='min_qod']/value}&amp;filter={str:encode-uri (../filters/term, true ())}&amp;filt_id={../filters/@id}{gsa:token ()}"
               title="{gsa:view_details_title ('Task', name)}">
              <xsl:value-of select="name"/>
            </a>
          </b>
          <xsl:choose>
            <xsl:when test="comment != ''">
              <div class="comment">(<xsl:value-of select="comment"/>)</div>
            </xsl:when>
            <xsl:otherwise></xsl:otherwise>
          </xsl:choose>
        </td>
        <td>
          <xsl:variable name="current_or_last_report_id">
            <xsl:choose>
              <xsl:when test="current_report/report/@id">
                <xsl:value-of select="current_report/report/@id"/>
              </xsl:when>
              <xsl:when test="last_report/report/@id">
                <xsl:value-of select="last_report/report/@id"/>
              </xsl:when>
              <xsl:otherwise></xsl:otherwise>
            </xsl:choose>
          </xsl:variable>
          <xsl:choose>
            <xsl:when test="$current_or_last_report_id != ''">
              <a href="/omp?cmd=get_report&amp;report_id={$current_or_last_report_id}&amp;notes=1&amp;overrides={../apply_overrides}&amp;min_qod={../filters/keywords/keyword[column='min_qod']/value}&amp;result_hosts_only=1&amp;token={/envelope/token}" title="{gsa-i18n:strformat (gsa:i18n ('View last report for Task %1'), name)}">
                <xsl:call-template name="status_bar">
                  <xsl:with-param name="status">
                    <xsl:choose>
                      <xsl:when test="target/@id='' and status='Running'">
                        <xsl:text>Uploading</xsl:text>
                      </xsl:when>
                      <xsl:when test="target/@id=''">
                        <xsl:text>Container</xsl:text>
                      </xsl:when>
                      <xsl:otherwise>
                        <xsl:value-of select="status"/>
                      </xsl:otherwise>
                    </xsl:choose>
                  </xsl:with-param>
                  <xsl:with-param name="progress">
                    <xsl:value-of select="progress/text()"/>
                  </xsl:with-param>
                  <xsl:with-param name="title_suffix">
                    <xsl:text> - </xsl:text><xsl:value-of select="gsa:i18n ('Go to the current report')"/>
                  </xsl:with-param>
                </xsl:call-template>
              </a>
            </xsl:when>
            <xsl:otherwise>
              <xsl:call-template name="status_bar">
                <xsl:with-param name="status">
                  <xsl:choose>
                    <xsl:when test="target/@id='' and status='Running'">
                      <xsl:text>Uploading</xsl:text>
                    </xsl:when>
                    <xsl:when test="target/@id=''">
                      <xsl:text>Container</xsl:text>
                    </xsl:when>
                    <xsl:otherwise>
                      <xsl:value-of select="status"/>
                    </xsl:otherwise>
                  </xsl:choose>
                </xsl:with-param>
                <xsl:with-param name="progress">
                  <xsl:value-of select="progress/text()"/>
                </xsl:with-param>
              </xsl:call-template>
            </xsl:otherwise>
          </xsl:choose>
        </td>
        <td>
          <xsl:choose>
            <xsl:when test="report_count &gt; 0">
              <a href="/omp?cmd=get_reports&amp;replace_task_id=1&amp;filt_id=-2&amp;filter=task_id={@id} and status=Done apply_overrides={../apply_overrides} min_qod={../filters/keywords/keyword[column='min_qod']/value} sort-reverse=date&amp;task_filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;task_filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
                title="{gsa-i18n:strformat (gsa:i18n ('View list of all finished reports for Task %1'), name)}">
                <xsl:value-of select="report_count/finished"/>
              </a>
              (<a href="/omp?cmd=get_reports&amp;replace_task_id=1&amp;filt_id=-2&amp;filter=task_id={@id} apply_overrides={../apply_overrides} min_qod={../filters/keywords/keyword[column='min_qod']/value} sort-reverse=date&amp;task_filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;task_filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
                title="{gsa-i18n:strformat (gsa:i18n ('View list of all reports for Task %1, including unfinished ones'), name)}">
                <xsl:value-of select="report_count/text()"/>
               </a>)
            </xsl:when>
            <xsl:otherwise>
            </xsl:otherwise>
          </xsl:choose>
        </td>
        <td>
          <a href="/omp?cmd=get_report&amp;report_id={last_report/report/@id}&amp;notes=1&amp;overrides={../apply_overrides}&amp;min_qod={../filters/keywords/keyword[column='min_qod']/value}&amp;result_hosts_only=1&amp;token={/envelope/token}" title="{gsa-i18n:strformat (gsa:i18n ('View last report for Task %1'), name)}">
            <xsl:call-template name="short_timestamp_last"/>
          </a>
        </td>
        <td style="text-align:center;">
          <xsl:choose>
            <xsl:when test="target/@id=''">
            </xsl:when>
            <xsl:when test="last_report">
              <xsl:apply-templates select="last_report"/>
            </xsl:when>
          </xsl:choose>
        </td>
        <td style="text-align:center;">
          <xsl:choose>
            <xsl:when test="target/@id=''">
            </xsl:when>
            <xsl:when test="alterable = 0">
              <!-- Trend -->
              <xsl:call-template name="trend_meter"/>
            </xsl:when>
            <xsl:otherwise>
            </xsl:otherwise>
          </xsl:choose>
        </td>
        <xsl:choose>
          <xsl:when test="/envelope/params/bulk_select = 1">
            <td style="text-align:center">
              <label style="width:100%">
                <input name="bulk_selected:{@id}" type="checkbox" style="width:100%; height:100%" title="{gsa:i18n ('Select for bulk action')}"/>
              </label>
            </td>
          </xsl:when>
          <xsl:otherwise>
            <td class="table-actions">
              <xsl:call-template name="task-icons"/>
              <xsl:call-template name="list-window-line-icons">
                <xsl:with-param name="cap-type" select="'Task'"/>
                <xsl:with-param name="type" select="'task'"/>
                <xsl:with-param name="id" select="@id"/>
              </xsl:call-template>
            </td>
          </xsl:otherwise>
        </xsl:choose>
      </tr>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="task" mode="trash">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <xsl:call-template name="observers-icon">
        <xsl:with-param name="type" select="'Task'"/>
      </xsl:call-template>
      <b><xsl:value-of select="name"/></b>
      <xsl:choose>
        <xsl:when test="comment != ''">
          <div class="comment">(<xsl:value-of select="comment"/>)</div>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:call-template name="status_bar">
        <xsl:with-param name="status">
          <xsl:choose>
            <xsl:when test="target/@id=''">
              <xsl:text>Container</xsl:text>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="status"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:with-param>
        <xsl:with-param name="progress">
          <xsl:value-of select="progress/text()"/>
        </xsl:with-param>
      </xsl:call-template>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="report_count &gt; 0">
          <xsl:value-of select="report_count/finished"/>
        </xsl:when>
        <xsl:otherwise>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="last_report/report/@id = first_report/report/@id">
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="short_timestamp_first"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td style="font-size:10px;">
      <xsl:call-template name="short_timestamp_last"/>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="last_report">
          <xsl:apply-templates select="last_report"/>
        </xsl:when>
      </xsl:choose>
    </td>
    <td>
      <!-- Trend -->
      <xsl:call-template name="trend_meter"/>
    </td>
    <td class="table-actions">
      <xsl:choose>
        <xsl:when test="not (gsa:may-op ('restore'))"/>
        <xsl:when test="(target/trash = '0') and (config/trash = '0') and (schedule/trash = '0') and (scanner/trash = '0') and (gsa:alert-in-trash () = 0)">
          <xsl:call-template name="restore-icon">
            <xsl:with-param name="id" select="@id"/>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <xsl:variable name="resources_list" select="target[trash!='0'] | config[trash!='0'] | schedule[trash!='0'] | scanner[trash!='0'] | (alert[trash!='0'])[0]"/>
          <xsl:variable name="resources_string">
            <xsl:for-each select="$resources_list">
              <xsl:value-of select="gsa:i18n (gsa:type-name (name (.)), gsa:type-name (name (.)))"/>
              <xsl:if test="position() &lt; last()-1">, </xsl:if>
              <xsl:if test="position() = last()-1">
                <xsl:text> </xsl:text><xsl:value-of select="gsa:i18n ('and')"/><xsl:text> </xsl:text>
              </xsl:if>
            </xsl:for-each>
          </xsl:variable>
          <img src="/img/restore_inactive.svg" alt="{gsa:i18n ('Restore')}"
               title="{$resources_string}{gsa:i18n (' must be restored first.', 'Trashcan')}"
               class="icon icon-sm"/>
        </xsl:otherwise>
      </xsl:choose>
      <xsl:call-template name="trash-delete-icon">
        <xsl:with-param name="type">task</xsl:with-param>
        <xsl:with-param name="id" select="@id"/>
      </xsl:call-template>
    </td>
  </tr>
</xsl:template>

<!-- GET_TASKS_RESPONSE -->

<xsl:template match="get_tasks_response">
  <xsl:choose>
    <xsl:when test="substring(@status, 1, 1) = '4' or substring(@status, 1, 1) = '5'">
      <xsl:call-template name="command_result_dialog">
        <xsl:with-param name="operation">Get Tasks</xsl:with-param>
        <xsl:with-param name="status">
          <xsl:value-of select="@status"/>
        </xsl:with-param>
        <xsl:with-param name="msg">
          <xsl:value-of select="@status_text"/>
        </xsl:with-param>
      </xsl:call-template>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="html-tasks-table"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- GET_TASK -->

<xsl:template match="get_task">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="commands_response/delete_task_response"/>
  <xsl:apply-templates select="delete_tag_response"/>
  <xsl:apply-templates select="create_tag_response"/>
  <xsl:apply-templates select="modify_tag_response"/>
  <xsl:apply-templates select="delete_task_response"/>
  <xsl:apply-templates select="create_report_response"/>
  <xsl:apply-templates select="create_task_response"/>
  <xsl:apply-templates select="start_task_response"/>
  <xsl:apply-templates select="stop_task_response"/>
  <xsl:apply-templates select="modify_task_response"/>
  <xsl:apply-templates select="resume_task_response"/>
  <xsl:apply-templates select="move_task_response"/>
  <xsl:apply-templates select="commands_response/get_tasks_response/task"
                       mode="details"/>
</xsl:template>

<!-- GET_TASKS -->

<xsl:template match="get_tasks">
  <xsl:apply-templates select="run_wizard_response"/>
  <xsl:apply-templates select="delete_task_response"/>
  <xsl:apply-templates select="create_filter_response"/>
  <xsl:apply-templates select="create_report_response"/>
  <xsl:apply-templates select="create_task_response"/>
  <xsl:apply-templates select="start_task_response"/>
  <xsl:apply-templates select="stop_task_response"/>
  <xsl:apply-templates select="modify_task_response"/>
  <xsl:apply-templates select="resume_task_response"/>
  <xsl:apply-templates select="get_tasks_response"/>
</xsl:template>

<xsl:template name="html-tasks-table">
  <xsl:call-template name="list-window">
    <xsl:with-param name="type" select="'task'"/>
    <xsl:with-param name="cap-type" select="'Task'"/>
    <xsl:with-param name="resources-summary" select="tasks"/>
    <xsl:with-param name="resources" select="task"/>
    <xsl:with-param name="count" select="count (task)"/>
    <xsl:with-param name="filtered-count" select="task_count/filtered"/>
    <xsl:with-param name="full-count" select="task_count/text ()"/>
    <xsl:with-param name="columns" xmlns="">
      <column>
        <name><xsl:value-of select="gsa:i18n('Name')"/></name>
        <field>name</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Status')"/></name>
        <field>status</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Reports')"/></name>
        <column>
          <name><xsl:value-of select="gsa:i18n('Total', 'Task|Reports')"/></name>
          <field>total</field>
          <sort-reverse/>
        </column>
        <column>
          <name><xsl:value-of select="gsa:i18n('Last', 'Task|Report')"/></name>
          <field>last</field>
          <sort-reverse/>
        </column>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Severity', 'Severity Short')"/></name>
        <field>severity</field>
        <sort-reverse/>
        <html>
          <before>
            <xsl:choose xmlns="http://www.w3.org/1999/xhtml">
              <xsl:when test="/envelope/params/bulk_select = 1">
                <div class="pull-right">
                  <xsl:choose>
                    <xsl:when test="filters/keywords/keyword[column='apply_overrides']/value = 0">
                      <img src="/img/overrides_disabled.svg"
                        alt="{gsa:i18n ('No Overrides')}"
                        title="{gsa:i18n ('No Overrides')}"
                        class="icon icon-sm"/>
                    </xsl:when>
                    <xsl:otherwise>
                      <img src="/img/overrides_enabled.svg"
                        alt="{gsa:i18n ('Overrides are Applied')}"
                        title="{gsa:i18n ('Overrides are Applied')}"
                        class="icon icon-sm"/>
                    </xsl:otherwise>
                  </xsl:choose>
                </div>
              </xsl:when>
              <xsl:otherwise>
                <div class="pull-right">
                  <form method="get" action="">
                    <input type="hidden" name="token" value="{/envelope/token}"/>
                    <input type="hidden" name="cmd" value="get_tasks"/>
                    <input type="hidden" name="filter" value="{filters/term}"/>
                    <xsl:choose>
                      <xsl:when test="filters/keywords/keyword[column='apply_overrides']/value = 0">
                        <input type="hidden" name="overrides" value="1"/>
                        <input type="image"
                          name="No Overrides"
                          src="/img/overrides_disabled.svg"
                          alt="{gsa:i18n ('No Overrides')}"
                          value="No Overrides"
                          title="{gsa:i18n ('No Overrides')}"
                          class="icon icon-sm"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <input type="hidden" name="overrides" value="0"/>
                        <input type="image"
                          name="Overrides are Applied"
                          src="/img/overrides_enabled.svg"
                          alt="{gsa:i18n ('Overrides are Applied')}"
                          value="Overrides are Applied"
                          title="{gsa:i18n ('Overrides are Applied')}"
                          class="icon icon-sm"/>
                      </xsl:otherwise>
                    </xsl:choose>
                  </form>
                </div>
              </xsl:otherwise>
            </xsl:choose>
          </before>
        </html>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Trend')"/></name>
        <field>trend</field>
      </column>
    </xsl:with-param>
    <xsl:with-param name="icon-count" select="6"/>
    <xsl:with-param name="top-visualization">
      <xsl:call-template name="init-d3charts"/>
      <xsl:call-template name="js-scan-management-top-visualization">
        <xsl:with-param name="type" select="'task'"/>
        <xsl:with-param name="auto_load_left_default" select="'left-by-class'"/>
        <xsl:with-param name="auto_load_right_default" select="'right-by-task-status'"/>
      </xsl:call-template>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!-- GET_TASKS_CHART -->

<xsl:template match="get_tasks_chart">
  <xsl:variable name="filter_term" select="/envelope/params/filter"/>
  <xsl:variable name="filt_id" select="/envelope/params/filt_id"/>

  <xsl:call-template name="init-d3charts"/>
  <xsl:choose>
    <xsl:when test="$filter_term != ''">
      <div id="applied_filter" class="footnote" style="padding: 5px 10px">
        <b><xsl:value-of select="gsa:i18n('Applied filter:')"/></b>
        <xsl:text> </xsl:text>
        <xsl:value-of select="$filter_term"/>
      </div>
    </xsl:when>
    <xsl:otherwise>
      <div id="applied_filter"/>
    </xsl:otherwise>
  </xsl:choose>

  <xsl:variable name="init_params_js">
    <xsl:if test="/envelope/params/_param[starts-with (name, 'chart_init:')]">{
      <xsl:for-each select="/envelope/params/_param[starts-with (name, 'chart_init:')]">
        "<xsl:value-of select="substring-after (name, 'chart_init:')"/>": "<xsl:value-of select="value"/>"<xsl:if test="position() &lt; count(exslt:node-set (/envelope/params/_param[starts-with (name, 'chart_init:')]))">, </xsl:if>
      </xsl:for-each>
      }</xsl:if>
  </xsl:variable>

  <xsl:variable name="gen_params_js">
    <xsl:if test="/envelope/params/_param[starts-with (name, 'chart_gen:')]">{
      <xsl:for-each select="/envelope/params/_param[starts-with (name, 'chart_gen:')]">
        "<xsl:value-of select="substring-after (name, 'chart_gen:')"/>": "<xsl:value-of select="value"/>"<xsl:if test="position() &lt; count(exslt:node-set (/envelope/params/_param[starts-with (name, 'chart_gen:')]))">, </xsl:if>
      </xsl:for-each>
      }</xsl:if>
  </xsl:variable>

  <div id="single-box-dashboard" class="dashboard"
    data-dashboard-name="single-box-dashboard"
    data-filter="{$filter_term}"
    data-default-controllers="tasks-chart"
    data-filter-id="{$filt_id}"
    data-hide-controller-select="1"
    data-hide-filter-select="1"
    data-detached="1"
    data-max-components="1">
    <div class="dashboard-data-source"
      data-soure-name="tasks-source"
      data-type="task"
      data-filter="{$filter_term}"
      data-filter-id="{$filt_id}">
      <span class="dashboard-chart"
        data-chart-name="tasks-chart"
        data-chart-type="{/envelope/params/chart_type}"
        data-chart-template="{/envelope/params/chart_template}"
        data-chart-title="{/envelope/params/chart_title}"
        data-init-params="{$init_params_js}"
        data-gen-params="{$gen_params_js}"
        />
    </div>
  </div>

  <xsl:call-template name="html-footer"/>
</xsl:template>


<!-- BEGIN CREDENTIALS MANAGEMENT -->

<xsl:template match="new_credential">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="create_credential_response"/>
  <xsl:apply-templates select="commands_response/delete_credential_response"/>
  <xsl:call-template name="html-create-credential-form"/>
</xsl:template>

<xsl:template name="html-create-credential-form">
  <div class="edit-dialog">
    <div class="title">
      <xsl:value-of select="gsa:i18n ('New Credential')"/>
    </div>
    <div class="content">
      <form action="/omp" method="post" enctype="multipart/form-data" class="form-horizontal">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="create_credential"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden" name="next" value="get_credential"/>
        <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
        <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
        <input type="hidden" name="restrict_credential_type" value="{/envelope/params/restrict_credential_type}"/>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Name')"/></label>
          <div class="col-10">
            <input type="text" name="name" value="unnamed" size="30"
              class="form-control"
              maxlength="80"/>
          </div>
        </div>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Comment')"/></label>
          <div class="col-10">
            <input type="text" name="comment" value="" size="30"
              class="form-control"
              maxlength="400"/>
          </div>
        </div>
        <div class="form-group">
          <xsl:variable name="allowed_types" select="str:tokenize (/envelope/params/restrict_credential_type, '|')"/>

          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Type')"/></label>
          <div class="col-10">
            <select class="col-2 control-label form-selection-control" id="credentials" name="base">
              <xsl:if test="(count($allowed_types) = 0) or (count($allowed_types[text()='up']))">
                <xsl:call-template name="opt">
                  <xsl:with-param name="value" select="'up'"/>
                  <xsl:with-param name="content" select="gsa:i18n ('Username + Password')"/>
                  <xsl:with-param name="select-value" select="/envelope/params/base"/>
                </xsl:call-template>
              </xsl:if>
              <xsl:if test="(count($allowed_types) = 0) or (count($allowed_types[text()='usk']))">
                <xsl:call-template name="opt">
                  <xsl:with-param name="value" select="'usk'"/>
                  <xsl:with-param name="content" select="gsa:i18n ('Username + SSH Key')"/>
                  <xsl:with-param name="select-value" select="/envelope/params/base"/>
                </xsl:call-template>
              </xsl:if>
              <xsl:if test="(count($allowed_types) = 0) or (count($allowed_types[text()='cc']))">
                <xsl:call-template name="opt">
                  <xsl:with-param name="value" select="'cc'"/>
                  <xsl:with-param name="content" select="gsa:i18n ('Client Certificate')"/>
                  <xsl:with-param name="select-value" select="/envelope/params/base"/>
                </xsl:call-template>
              </xsl:if>
              <xsl:if test="(count($allowed_types) = 0) or (count($allowed_types[text()='snmp']))">
                <xsl:call-template name="opt">
                  <xsl:with-param name="value" select="'snmp'"/>
                  <xsl:with-param name="content" select="gsa:i18n ('SNMP')"/>
                  <xsl:with-param name="select-value" select="/envelope/params/base"/>
                </xsl:call-template>
              </xsl:if>
            </select>
          </div>
        </div>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Allow insecure use')"/></label>
          <div class="col-10">
            <label class="radio-inline">
              <input name="allow_insecure" value="1" type="radio"/>
              <xsl:value-of select="gsa:i18n ('Yes')"/>
            </label>
            <label class="radio-inline">
              <input name="allow_insecure" value="0" type="radio" checked="1"/>
              <xsl:value-of select="gsa:i18n ('No')"/>
            </label>
          </div>
        </div>
        <div class="form-group form-selection-item-credentials form-selection-item-credentials--up form-selection-item-credentials--usk"
            id="autogenerate_row">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Auto-generate')"/></label>
          <div class="col-10">
            <label class="radio-inline">
              <input name="autogenerate" id="autogenerate" value="1" type="radio"
                class="form-enable-control"/>
              <xsl:value-of select="gsa:i18n ('Yes')"/>
            </label>
            <label class="radio-inline">
              <input name="autogenerate" id="autogenerate" value="0" type="radio" checked="1"
                class="form-enable-control"/>
              <xsl:value-of select="gsa:i18n ('No')"/>
            </label>
          </div>
        </div>
        <div class="form-group">
        </div>
        <div class="form-group form-selection-item-credentials form-selection-item-credentials--snmp"
            id="community_row" style="display:none;">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('SNMP Community')"/></label>
          <div class="col-10">
            <div class="form-item">
              <input type="password" name="community" size="30" class="form-control"/>
            </div>
          </div>
        </div>
        <div class="form-group form-selection-item-credentials form-selection-item-credentials--up form-selection-item-credentials--usk form-selection-item-credentials--snmp"
            id="login_row">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Username')"/></label>
          <div class="col-10">
            <div class="form-item">
              <input type="text" name="credential_login" size="30" class="form-control"/>
            </div>
          </div>
        </div>
        <div class="form-group form-selection-item-credentials form-selection-item-credentials--up form-selection-item-credentials--snmp"
            id="password_row">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Password')"/></label>
          <div class="col-10">
            <div class="form-item">
              <input type="password" name="lsc_password" size="30" class="form-control form-enable-item--autogenerate"/>
            </div>
          </div>
        </div>
        <div class="form-group form-selection-item-credentials form-selection-item-credentials--usk"
            id="passphrase_row" style="display:none;">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Passphrase')"/></label>
          <div class="col-10">
            <div class="form-item">
              <input type="password" name="passphrase" size="30" class="form-control form-enable-item--autogenerate"/>
            </div>
          </div>
        </div>
        <div class="form-group form-selection-item-credentials form-selection-item-credentials--snmp"
            id="priv_password_row" style="display:none;">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Privacy Password')"/></label>
          <div class="col-10">
            <div class="form-item">
              <input type="password" name="privacy_password" size="30" class="form-control"/>
            </div>
          </div>
        </div>
        <div class="form-group form-selection-item-credentials form-selection-item-credentials--cc"
            id="certificate_row" style="display:none;">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Certificate')"/></label>
          <div class="col-10">
            <input type="file" name="certificate" size="30"/>
          </div>
        </div>
        <div class="form-group form-selection-item-credentials form-selection-item-credentials--usk form-selection-item-credentials--cc"
            id="private_key_row" style="display:none;">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Private Key')"/></label>
          <div class="col-10">
            <input type="file" name="private_key" size="30"/>
          </div>
        </div>
        <div class="form-group form-selection-item-credentials form-selection-item-credentials--snmp"
            id="auth_algo_row" style="display:none;">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Auth Algorithm')"/></label>
          <div class="col-10">
            <label class="radio-inline">
              <input name="auth_algorithm" value="md5" type="radio"/>
              MD5
            </label>
            <label class="radio-inline">
              <input name="auth_algorithm" value="sha1" type="radio" checked="1"/>
              SHA1
            </label>
          </div>
        </div>
        <div class="form-group form-selection-item-credentials form-selection-item-credentials--snmp"
            id="priv_algo_row" style="display:none;">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Privacy Algorithm')"/></label>
          <div class="col-10">
            <label class="radio-inline">
              <input name="privacy_algorithm" value="aes" type="radio" checked="1"/>
              AES
            </label>
            <label class="radio-inline">
              <input name="privacy_algorithm" value="des" type="radio"/>
              DES
            </label>
            <label class="radio-inline">
              <input name="privacy_algorithm" value="" type="radio"/>
              <xsl:value-of select="gsa:i18n ('None', 'Privacy Algorithm')"/>
            </label>
          </div>
        </div>
      </form>
    </div>
  </div>
</xsl:template>

<xsl:template name="html-credentials-table">
  <xsl:call-template name="list-window">
    <xsl:with-param name="type" select="'credential'"/>
    <xsl:with-param name="cap-type" select="'Credential'"/>
    <xsl:with-param name="resources-summary" select="credentials"/>
    <xsl:with-param name="resources" select="credential"/>
    <xsl:with-param name="count" select="count (credential)"/>
    <xsl:with-param name="filtered-count" select="credential_count/filtered"/>
    <xsl:with-param name="full-count" select="credential_count/text ()"/>
    <xsl:with-param name="columns" xmlns="">
      <column>
        <name><xsl:value-of select="gsa:i18n('Name')"/></name>
        <field>name</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Type')"/></name>
        <field>type</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Allow insecure use')"/></name>
        <field>allow_insecure</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Login', 'Auth Data')"/></name>
        <field>login</field>
      </column>
    </xsl:with-param>
    <xsl:with-param name="icon-count" select="7"/>
  </xsl:call-template>
</xsl:template>

<!--     CREATE_CREDENTIAL_RESPONSE -->

<xsl:template match="create_credential_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Create Credential</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!--     DELETE_CREDENTIAL_RESPONSE -->

<xsl:template match="delete_credential_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">
      Delete Credential
    </xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!--     EDITING CREDENTIALS -->

<xsl:template name="html-edit-credential-form">
  <div class="edit-dialog">
    <div class="title"><xsl:value-of select="gsa:i18n ('Edit Credential')"/></div>
    <xsl:variable name="credential_type" select="commands_response/get_credentials_response/credential/type"/>
    <div class="content">
      <form action="" method="post" enctype="multipart/form-data" class="form-horizontal">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="save_credential"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden"
          name="credential_id"
          value="{commands_response/get_credentials_response/credential/@id}"/>
        <input type="hidden" name="next" value="{/envelope/params/next}"/>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Name')"/></label>
          <div class="col-10">
            <input type="text"
              name="name"
              value="{commands_response/get_credentials_response/credential/name}"
              size="30"
              class="form-control"
              maxlength="80"/>
          </div>
        </div>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Comment')"/></label>
          <div class="col-10">
            <input type="text" name="comment" size="30" maxlength="400" class="form-control"
              value="{commands_response/get_credentials_response/credential/comment}"/>
          </div>
        </div>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Type')"/></label>
          <div class="col-10">
            <select name="base" class="form-selection-control" disabled="1">
              <xsl:call-template name="opt">
                <xsl:with-param name="value" select="'up'"/>
                <xsl:with-param name="content" select="gsa:i18n ('Username + Password')"/>
                <xsl:with-param name="select-value" select="$credential_type"/>
              </xsl:call-template>
              <xsl:call-template name="opt">
                <xsl:with-param name="value" select="'usk'"/>
                <xsl:with-param name="content" select="gsa:i18n ('Username + SSH Key')"/>
                <xsl:with-param name="select-value" select="$credential_type"/>
              </xsl:call-template>
              <xsl:call-template name="opt">
                <xsl:with-param name="value" select="'cc'"/>
                <xsl:with-param name="content" select="gsa:i18n ('Client Certificate')"/>
                <xsl:with-param name="select-value" select="$credential_type"/>
              </xsl:call-template>
              <xsl:call-template name="opt">
                <xsl:with-param name="value" select="'snmp'"/>
                <xsl:with-param name="content" select="gsa:i18n ('SNMP')"/>
                <xsl:with-param name="select-value" select="$credential_type"/>
              </xsl:call-template>
            </select>
          </div>
        </div>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Allow insecure use')"/></label>
          <div class="col-10">
            <label class="radio-inline">
              <xsl:choose>
                <xsl:when test="commands_response/get_credentials_response/credential/allow_insecure != 0">
                  <input name="allow_insecure" value="1" type="radio" checked="1"/>
                </xsl:when>
                <xsl:otherwise>
                  <input name="allow_insecure" value="1" type="radio"/>
                </xsl:otherwise>
              </xsl:choose>
              <xsl:value-of select="gsa:i18n ('Yes')"/>
            </label>
            <label class="radio-inline">
              <xsl:choose>
                <xsl:when test="commands_response/get_credentials_response/credential/allow_insecure != 0">
                  <input name="allow_insecure" value="0" type="radio"/>
                </xsl:when>
                <xsl:otherwise>
                  <input name="allow_insecure" value="0" type="radio" checked="1"/>
                </xsl:otherwise>
              </xsl:choose>
              <xsl:value-of select="gsa:i18n ('No')"/>
            </label>
          </div>
        </div>
        <xsl:if test="$credential_type = 'snmp'">
          <div class="form-group">
            <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('SNMP Community')"/></label>
            <div class="col-10">
              <div class="form-item checkbox">
                <label>
                  <input type="checkbox" name="change_community" value="1"/>
                  <xsl:value-of select="gsa:i18n ('Replace existing value with', 'Auth Data|Password')"/>:
                </label>
              </div>
              <div class="form-item">
                <input type="password" autocomplete="off" name="community"
                  class="form-control"
                  size="30" maxlength="400" value=""/>
              </div>
            </div>
          </div>
        </xsl:if>
        <xsl:if test="$credential_type != 'cc'">
          <div class="form-group">
            <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Login', 'Auth Data')"/></label>
            <div class="col-10">
              <div class="form-item">
                <input type="text" name="credential_login" size="30" maxlength="400"
                  class="form-control"
                  value="{commands_response/get_credentials_response/credential/login}"/>
              </div>
            </div>
          </div>
        </xsl:if>
        <xsl:if test="$credential_type = 'up' or $credential_type = 'snmp'">
          <div class="form-group">
            <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Password')"/></label>
            <div class="col-10">
              <div class="form-item checkbox">
                <label>
                  <input type="checkbox" name="change_password" value="1"/>
                  <xsl:value-of select="gsa:i18n ('Replace existing value with', 'Auth Data|Password')"/>:
                </label>
              </div>
              <div class="form-item">
                <input type="password" autocomplete="off" name="password"
                  class="form-control"
                  size="30" maxlength="400" value=""/>
              </div>
            </div>
          </div>
        </xsl:if>
        <xsl:if test="$credential_type = 'snmp'">
          <div class="form-group">
            <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Privacy Password')"/></label>
            <div class="col-10">
              <div class="form-item checkbox">
                <label>
                  <input type="checkbox" name="change_privacy_password" value="1"/>
                  <xsl:value-of select="gsa:i18n ('Replace existing value with', 'Auth Data|Password')"/>:
                </label>
              </div>
              <div class="form-item">
                <input type="password" autocomplete="off" name="privacy_password"
                  class="form-control"
                  size="30" maxlength="400" value=""/>
              </div>
            </div>
          </div>
        </xsl:if>
        <xsl:if test="$credential_type = 'cc'">
          <div class="form-group">
            <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Certificate')"/></label>
            <div class="col-10">
              <input type="file" name="certificate" size="30"/>
            </div>
          </div>
        </xsl:if>
        <xsl:if test="$credential_type = 'cc' or $credential_type = 'usk'">
          <div class="form-group">
            <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Private key')"/></label>
            <div class="col-10">
              <input type="file" name="private_key" size="30"/>
            </div>
          </div>
        </xsl:if>
        <xsl:if test="$credential_type = 'usk'">
          <div class="form-group">
            <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Passphrase')"/></label>
            <div class="col-10">
              <div class="form-item checkbox">
                <label>
                  <input type="checkbox" name="change_passphrase" value="1"/>
                  <xsl:value-of select="gsa:i18n ('Replace existing value with', 'Auth Data|Password')"/>:
                </label>
              </div>
              <div class="form-item">
                <input type="password" autocomplete="off" name="passphrase"
                  class="form-control"
                  size="30" maxlength="400" value=""/>
              </div>
            </div>
          </div>
        </xsl:if>
        <xsl:if test="$credential_type = 'snmp'">
          <div class="form-group">
            <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Auth Algorithm')"/></label>
            <div class="col-10">
              <label class="radio-inline">
                <xsl:choose>
                  <xsl:when test="commands_response/get_credentials_response/credential/auth_algorithm = 'md5'">
                    <input name="auth_algorithm" value="md5" type="radio" checked="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input name="auth_algorithm" value="md5" type="radio"/>
                  </xsl:otherwise>
                </xsl:choose>
                MD5
              </label>
              <label class="radio-inline">
                <xsl:choose>
                  <xsl:when test="commands_response/get_credentials_response/credential/auth_algorithm = 'sha1'">
                    <input name="auth_algorithm" value="sha1" type="radio" checked="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input name="auth_algorithm" value="sha1" type="radio"/>
                  </xsl:otherwise>
                </xsl:choose>
                SHA1
              </label>
            </div>
          </div>
        </xsl:if>
        <xsl:if test="$credential_type = 'snmp'">
          <div class="form-group">
            <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Privacy Algorithm')"/></label>
            <div class="col-10">
              <label class="radio-inline">
                <xsl:choose>
                  <xsl:when test="commands_response/get_credentials_response/credential/privacy/algorithm = 'aes'">
                    <input name="privacy_algorithm" value="aes" type="radio" checked="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input name="privacy_algorithm" value="aes" type="radio"/>
                  </xsl:otherwise>
                </xsl:choose>
                AES
              </label>
              <label class="radio-inline">
                <xsl:choose>
                  <xsl:when test="commands_response/get_credentials_response/credential/privacy/algorithm = 'des'">
                    <input name="privacy_algorithm" value="des" type="radio" checked="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input name="privacy_algorithm" value="des" type="radio"/>
                  </xsl:otherwise>
                </xsl:choose>
                DES
              </label>
              <label class="radio-inline">
                <xsl:choose>
                  <xsl:when test="commands_response/get_credentials_response/credential/privacy/algorithm = ''">
                    <input name="privacy_algorithm" value="" type="radio" checked="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input name="privacy_algorithm" value="" type="radio"/>
                  </xsl:otherwise>
                </xsl:choose>
                <xsl:value-of select="gsa:i18n ('None', 'Privacy Algorithm')"/>
              </label>
            </div>
          </div>
        </xsl:if>
      </form>
    </div>
  </div>
</xsl:template>

<xsl:template match="edit_credential">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="modify_credential_response"/>
  <xsl:call-template name="html-edit-credential-form"/>
</xsl:template>

<xsl:template match="modify_credential_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Save Credential</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!--     CREDENTIAL -->

<xsl:template name="credential-download-icons">
  <xsl:param name="type" select="''"/>

  <xsl:if test="$type='usk'">
    <a href="/omp?cmd=download_credential&amp;credential_id={@id}&amp;package_format=rpm&amp;token={/envelope/token}"
      title="{gsa:i18n ('Download RPM package', 'Action Verb')}"
      class="icon icon-sm">
      <img src="/img/rpm.svg" alt="{gsa:i18n ('Download RPM', 'Action Verb')}"/>
    </a>
  </xsl:if>
  <xsl:if test="$type='usk'">
    <a href="/omp?cmd=download_credential&amp;credential_id={@id}&amp;package_format=deb&amp;token={/envelope/token}"
      title="{gsa:i18n ('Download Debian package', 'Action Verb')}"
      class="icon icon-sm">
      <img src="/img/deb.svg" alt="{gsa:i18n ('Download Deb', 'Action Verb')}"/>
    </a>
  </xsl:if>
  <xsl:if test="$type='up'">
    <a href="/omp?cmd=download_credential&amp;credential_id={@id}&amp;package_format=exe&amp;token={/envelope/token}"
      title="{gsa:i18n ('Download Exe package', 'Action Verb')}"
      class="icon icon-sm">
      <img src="/img/exe.svg" alt="{gsa:i18n ('Download Exe', 'Action Verb')}"/>
    </a>
  </xsl:if>
  <xsl:if test="$type='usk'">
    <a href="/omp?cmd=download_credential&amp;credential_id={@id}&amp;package_format=key&amp;token={/envelope/token}"
      title="{gsa:i18n ('Download Public Key', 'Action Verb')}"
      class="icon icon-sm">
      <img src="/img/key.svg" alt="{gsa:i18n ('Download Public Key', 'Action Verb')}"/>
    </a>
  </xsl:if>
  <xsl:if test="$type='cc'">
    <a href="/omp?cmd=download_credential&amp;credential_id={@id}&amp;package_format=pem&amp;token={/envelope/token}"
      title="{gsa:i18n ('Download Certificate', 'Action Verb')}"
      class="icon icon-sm">
      <img src="/img/key.svg" alt="{gsa:i18n ('Download Certificate', 'Action Verb')}"/>
    </a>
  </xsl:if>
</xsl:template>

<xsl:template match="credential">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <xsl:call-template name="observers-icon">
        <xsl:with-param name="type" select="'Credential'"/>
      </xsl:call-template>
      <b>
        <a href="/omp?cmd=get_credential&amp;credential_id={@id}&amp;filter={str:encode-uri (../filters/term, true ())}&amp;token={/envelope/token}"
           title="{gsa:view_details_title ('Credential', name)}">
          <xsl:value-of select="name"/>
        </a>
      </b>
      <xsl:choose>
        <xsl:when test="comment != ''">
          <div class="comment">(<xsl:value-of select="comment"/>)</div>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:value-of select="type"/>
      <span class="footnote"> (<xsl:value-of select="full_type"/>)</span>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="allow_insecure != 0">
          <xsl:value-of select="gsa:i18n ('Yes')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="gsa:i18n ('No')"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:value-of select="login"/>
    </td>
    <xsl:choose>
      <xsl:when test="/envelope/params/bulk_select = 1">
        <td style="text-align:center">
          <label style="width:100%">
            <input name="bulk_selected:{@id}" type="checkbox" style="width:100%; height:100%" title="{gsa:i18n ('Select for bulk action')}"/>
          </label>
        </td>
      </xsl:when>
      <xsl:otherwise>
        <td class="table-actions">
          <xsl:call-template name="list-window-line-icons">
            <xsl:with-param name="cap-type" select="'Credential'"/>
            <xsl:with-param name="type" select="'credential'"/>
            <xsl:with-param name="id" select="@id"/>
          </xsl:call-template>
          <xsl:call-template name="credential-download-icons">
            <xsl:with-param name="type" select="type"/>
          </xsl:call-template>
        </td>
      </xsl:otherwise>
    </xsl:choose>
  </tr>
</xsl:template>

<xsl:template match="credential" mode="trash">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <b><xsl:value-of select="name"/></b>
    </td>
    <td>
      <xsl:value-of select="type"/>
      <span class="footnote"> (<xsl:value-of select="full_type"/>)</span>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="allow_insecure != 0">
          <xsl:value-of select="gsa:i18n ('Yes')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="gsa:i18n ('No')"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:value-of select="login"/>
    </td>
    <td>
      <xsl:value-of select="comment"/>
    </td>
    <td class="table-actions">
      <xsl:call-template name="restore-icon">
        <xsl:with-param name="id" select="@id"/>
      </xsl:call-template>
      <xsl:choose>
        <xsl:when test="in_use='0'">
          <xsl:call-template name="trash-delete-icon">
            <xsl:with-param name="type">credential</xsl:with-param>
            <xsl:with-param name="id" select="@id"/>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <img src="/img/delete_inactive.svg" alt="{gsa:i18n ('Delete')}"
               title="{gsa:i18n ('Credential is still in use')}"
               class="icon icon-sm"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
  </tr>
</xsl:template>

<xsl:template match="credential" mode="details">
  <div class="toolbar">
    <xsl:call-template name="details-header-icons">
      <xsl:with-param name="cap-type" select="'Credential'"/>
      <xsl:with-param name="type" select="'credential'"/>
    </xsl:call-template>
    <xsl:call-template name="credential-download-icons">
      <xsl:with-param name="type" select="type"/>
    </xsl:call-template>
  </div>

  <div class="section-header">
    <xsl:call-template name="minor-details"/>
    <h1>
      <a href="/omp?cmd=get_credentials&amp;token={/envelope/token}"
         title="{gsa:i18n ('Credentials')}">
        <img class="icon icon-lg" src="/img/credential.svg" alt="Credentials"/>
      </a>
      <xsl:value-of select="gsa:i18n ('Credential')"/>:
      <xsl:value-of select="name"/>
      <xsl:text> </xsl:text>
    </h1>
  </div>

  <div class="section-box">
    <table>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Comment')"/>:</td>
        <td><xsl:value-of select="comment"/></td>
      </tr>
      <tr>
        <td style="vertical-align:top;"><xsl:value-of select="gsa:i18n ('Type')"/>:</td>
        <td>
          <xsl:value-of select="type"/><br/>
          <span class="footnote">(<xsl:value-of select="full_type"/>)</span>
        </td>
      </tr>
      <tr>
        <td style="vertical-align:top;"><xsl:value-of select="gsa:i18n ('Allow insecure use')"/>:</td>
        <td>
          <xsl:choose>
            <xsl:when test="allow_insecure != 0">
              <xsl:value-of select="gsa:i18n ('Yes')"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="gsa:i18n ('No')"/>
            </xsl:otherwise>
          </xsl:choose>
        </td>
      </tr>
      <xsl:if test="type != 'cc'">
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Login', 'Auth Data')"/>:</td>
          <td><xsl:value-of select="login"/></td>
        </tr>
      </xsl:if>
      <xsl:if test="type = 'snmp'">
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Auth Algorithm')"/>:</td>
          <td><xsl:value-of select="auth_algorithm"/></td>
        </tr>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Privacy Algorithm')"/>:</td>
          <td>
            <xsl:choose>
              <xsl:when test="privacy/algorithm != ''">
                <xsl:value-of select="privacy/algorithm"/>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="gsa:i18n ('None', 'Privacy Algorithm')"/>
              </xsl:otherwise>
            </xsl:choose>
          </td>
        </tr>
      </xsl:if>
    </table>

    <xsl:if test="certificate_info">
      <h3><xsl:value-of select="gsa:i18n ('Certificate')"/></h3>
      <xsl:call-template name="certificate-info-table">
        <xsl:with-param name="certificate_info" select="certificate_info"/>
      </xsl:call-template>
    </xsl:if>
  </div>

  <div class="section-header">
    <a href="#" class="toggle-action-icon icon icon-sm icon-action"
      data-target="#using-box2" data-name="Targets using this Credential"
      data-variable="using-box2--collapsed">
        <img src="/img/fold.svg"/>
    </a>
    <h2>
      <a href="/omp?cmd=get_targets&amp;token={/envelope/token}"
         title="{gsa:i18n ('Targets')}">
        <img class="icon icon-sm" src="/img/target.svg" alt="Targets"/>
      </a>
      <xsl:value-of select="gsa:i18n ('Targets using this Credential')"/>
      <xsl:text> </xsl:text>
      <xsl:choose>
        <xsl:when test="count(targets/target) != 0">
          (<xsl:value-of select="count(targets/target)"/>)
        </xsl:when>
        <xsl:otherwise>
          (<xsl:value-of select="gsa:i18n ('none')"/>)
        </xsl:otherwise>
      </xsl:choose>
    </h2>
  </div>

  <div class="section-box" id="using-box2">
    <table class="gbntable">
      <tr class="gbntablehead2">
        <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
      </tr>
      <xsl:for-each select="targets/target">
        <tr class="{gsa:table-row-class(position())}">
          <xsl:choose>
            <xsl:when test="boolean (permissions) and count (permissions/permission) = 0">
              <td><xsl:value-of select="name"/> (<xsl:value-of select="gsa:i18n('Unavailable')"/>, <xsl:value-of select="gsa:i18n('UUID')"/>: <xsl:value-of select="@id"/>)</td>
            </xsl:when>
            <xsl:otherwise>
              <td>
                <a href="/omp?cmd=get_target&amp;target_id={@id}&amp;token={/envelope/token}"
                   title="{gsa:i18n ('Target Details')}">
                  <xsl:value-of select="name"/>
                </a>
              </td>
            </xsl:otherwise>
          </xsl:choose>
        </tr>
      </xsl:for-each>
    </table>
  </div>

  <div class="section-header">
    <a href="#" class="toggle-action-icon icon icon-sm icon-action"
      data-target="#using-box3" data-name="Scanners using this Credential"
      data-variable="using-box3--collapsed">
        <img src="/img/fold.svg"/>
    </a>
    <h2>
      <a href="/omp?cmd=get_targets&amp;token={/envelope/token}"
         title="{gsa:i18n ('Targets')}">
        <img class="icon icon-sm" src="/img/scanner.svg" alt="Scanners"/>
      </a>
      <xsl:value-of select="gsa:i18n ('Scanners using this Credential')"/>
      <xsl:text> </xsl:text>
      <xsl:choose>
        <xsl:when test="count(scanners/scanner) != 0">
          (<xsl:value-of select="count(scanners/scanner)"/>)
        </xsl:when>
        <xsl:otherwise>
          (<xsl:value-of select="gsa:i18n ('none')"/>)
        </xsl:otherwise>
      </xsl:choose>
    </h2>
  </div>
  <div class="section-box" id="using-box3">
    <table class="gbntable">
      <thead>
        <tr class="gbntablehead2">
          <th><xsl:value-of select="gsa:i18n ('Name')"/></th>
        </tr>
      </thead>
      <tbody>
        <xsl:for-each select="scanners/scanner">
          <tr>
            <xsl:choose>
              <xsl:when test="boolean (permissions) and count (permissions/permission) = 0">
                <td><xsl:value-of select="name"/> (<xsl:value-of select="gsa:i18n('Unavailable')"/>, <xsl:value-of select="gsa:i18n('UUID')"/>: <xsl:value-of select="@id"/>)</td>
              </xsl:when>
              <xsl:otherwise>
                <td>
                  <a href="/omp?cmd=get_scanner&amp;scanner_id={@id}&amp;token={/envelope/token}"
                     title="{gsa:i18n ('Scanner Details')}">
                    <xsl:value-of select="name"/>
                  </a>
                </td>
              </xsl:otherwise>
            </xsl:choose>
          </tr>
        </xsl:for-each>
      </tbody>
    </table>
  </div>

  <xsl:call-template name="user-tags-window">
    <xsl:with-param name="resource_type" select="'credential'"/>
  </xsl:call-template>

  <xsl:call-template name="resource-permissions-window">
    <xsl:with-param name="resource_type" select="'credential'"/>
    <xsl:with-param name="permissions" select="../../permissions/get_permissions_response"/>
    <xsl:with-param name="related">
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!--     GET_CREDENTIAL -->

<xsl:template match="get_credential">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="create_credential_response"/>
  <xsl:apply-templates select="create_tag_response"/>
  <xsl:apply-templates select="delete_tag_response"/>
  <xsl:apply-templates select="modify_tag_response"/>
  <xsl:apply-templates select="commands_response/delete_credential_response"/>
  <xsl:apply-templates select="commands_response/modify_credential_response"/>
  <xsl:apply-templates select="get_credentials_response/credential"
                       mode="details"/>
</xsl:template>

<!--     GET_CREDENTIALS_RESPONSE -->

<xsl:template match="get_credentials_response">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="delete_credential_response"/>
  <xsl:call-template name="html-credentials-table"/>
</xsl:template>

<xsl:template match="credential" mode="select">
  <xsl:param name="select_id" select="''"/>
  <xsl:choose>
    <xsl:when test="@id = $select_id">
      <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
    </xsl:when>
    <xsl:otherwise>
      <option value="{@id}"><xsl:value-of select="name"/></option>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="credentials_response" mode="select">
  <xsl:apply-templates select="credential" mode="select"/>
</xsl:template>


<!-- BEGIN AGENTS MANAGEMENT -->

<xsl:template name="html-create-agent-form">
  <div class="gb_window">
    <div class="gb_window_part_left"></div>
    <div class="gb_window_part_right"></div>
    <div class="gb_window_part_center">
      <xsl:value-of select="gsa:i18n ('New Agent')"/>
      <a href="/help/new_agent.html?token={/envelope/token}"
        title="{concat(gsa:i18n('Help'),': ',gsa:i18n('New Agent'))}"
        class="icon icon-sm">
        <img src="/img/help.svg"/>
      </a>
      <a href="/omp?cmd=get_agents&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
        class="icon icon-sm"
        title="{gsa:i18n ('Agents')}">
        <img src="/img/list.svg" alt="{gsa:i18n ('Agents')}"/>
      </a>
    </div>
    <div class="gb_window_part_content">
      <form action="/omp" method="post" enctype="multipart/form-data">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="create_agent"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden" name="next" value="get_agent"/>
        <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
        <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
        <table class="table-form">
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
            <td>
              <input type="text" name="name" value="unnamed" size="30"
                     maxlength="80"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Comment')"/></td>
            <td>
              <input type="text" name="comment" size="30" maxlength="400"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Installer')"/></td>
            <td><input type="file" name="installer" size="30"/></td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Installer signature')"/></td>
            <td><input type="file" name="installer_sig" size="30"/></td>
          </tr>
          <!--
          <tr>
            <td valign="top" width="125">Howto Install</td>
            <td><input type="file" name="howto_install" size="30"/></td>
          </tr>
          <tr>
            <td valign="top" width="125">Howto Use</td>
            <td><input type="file" name="howto_use" size="30"/></td>
          </tr>
          -->
          <tr>
            <td>
              <input type="submit" name="submit" value="{gsa:i18n ('Create Agent')}"/>
            </td>
          </tr>
        </table>
      </form>
    </div>
  </div>
</xsl:template>

<xsl:template match="new_agent">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="create_agent_response"/>
  <xsl:apply-templates select="commands_response/delete_agent_response"/>
  <xsl:call-template name="html-create-agent-form"/>
</xsl:template>

<xsl:template name="html-agents-table">
  <xsl:call-template name="list-window">
    <xsl:with-param name="type" select="'agent'"/>
    <xsl:with-param name="cap-type" select="'Agent'"/>
    <xsl:with-param name="resources-summary" select="agents"/>
    <xsl:with-param name="resources" select="agent"/>
    <xsl:with-param name="count" select="count (agent)"/>
    <xsl:with-param name="filtered-count" select="agent_count/filtered"/>
    <xsl:with-param name="full-count" select="agent_count/text ()"/>
    <xsl:with-param name="columns" xmlns="">
      <column>
        <name><xsl:value-of select="gsa:i18n('Name')"/></name>
        <field>name</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Trust')"/></name>
        <field>trust</field>
      </column>
    </xsl:with-param>
    <xsl:with-param name="icon-count" select="6"/>
  </xsl:call-template>
</xsl:template>

<!--     EDIT_AGENT -->

<xsl:template name="html-edit-agent-form">
  <div class="gb_window">
    <div class="gb_window_part_left"></div>
    <div class="gb_window_part_right"></div>
    <div class="gb_window_part_center"><xsl:value-of select="gsa:i18n ('Edit Agent')"/>
      <xsl:call-template name="edit-header-icons">
        <xsl:with-param name="cap-type" select="'Agent'"/>
        <xsl:with-param name="type" select="'agent'"/>
        <xsl:with-param name="id"
                        select="commands_response/get_agents_response/agent/@id"/>
      </xsl:call-template>
    </div>
    <div class="gb_window_part_content">
      <form action="" method="post" enctype="multipart/form-data">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="save_agent"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden"
               name="agent_id"
               value="{commands_response/get_agents_response/agent/@id}"/>
        <input type="hidden" name="next" value="{/envelope/params/next}"/>
        <input type="hidden" name="agent" value="{/envelope/params/agent}"/>
        <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
        <table class="table-form">
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
            <td>
              <input type="text" name="name" size="30" maxlength="80"
                     value="{commands_response/get_agents_response/agent/name}"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Comment')"/></td>
            <td>
              <input type="text" name="comment" size="30" maxlength="400"
                     value="{commands_response/get_agents_response/agent/comment}"/>
            </td>
          </tr>
          <tr>
            <td>
              <input type="submit" name="submit" value="{gsa:i18n ('Save Agent')}"/>
            </td>
          </tr>
        </table>
      </form>
    </div>
  </div>
</xsl:template>

<xsl:template match="edit_agent">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="modify_agent_response"/>
  <xsl:call-template name="html-edit-agent-form"/>
</xsl:template>

<!--     RESPONSES -->

<xsl:template match="create_agent_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Create Agent</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="delete_agent_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">
      Delete Agent
    </xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="modify_agent_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Save Agent</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="verify_agent_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Verify Agent</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!--     AGENT -->

<xsl:template match="agent">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <xsl:call-template name="observers-icon">
        <xsl:with-param name="type" select="'Agent'"/>
      </xsl:call-template>
      <b>
        <a href="/omp?cmd=get_agent&amp;agent_id={@id}&amp;filter={str:encode-uri (../filters/term, true ())}&amp;token={/envelope/token}"
           title="{gsa:view_details_title ('Agent', name)}">
          <xsl:value-of select="name"/>
        </a>
      </b>
      <xsl:choose>
        <xsl:when test="comment != ''">
          <div class="comment">(<xsl:value-of select="comment"/>)</div>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:value-of select="gsa:i18n (installer/trust/text(), 'Trust')"/>
      <xsl:choose>
        <xsl:when test="installer/trust/time != ''">
          (<xsl:value-of select="concat (date:month-abbreviation (installer/trust/time), ' ', date:day-in-month (installer/trust/time), ' ', date:year (installer/trust/time))"/>)
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </td>
    <xsl:choose>
      <xsl:when test="/envelope/params/bulk_select = 1">
        <td style="text-align:center">
          <label style="width:100%">
            <input name="bulk_selected:{@id}" type="checkbox" style="width:100%; height:100%" title="{gsa:i18n ('Select for bulk action')}"/>
          </label>
        </td>
      </xsl:when>
      <xsl:otherwise>
        <td class="table-actions">
          <xsl:call-template name="list-window-line-icons">
            <xsl:with-param name="cap-type" select="'Agent'"/>
            <xsl:with-param name="type" select="'agent'"/>
            <xsl:with-param name="id" select="@id"/>
          </xsl:call-template>
          <a href="/omp?cmd=download_agent&amp;agent_id={@id}&amp;agent_format=installer&amp;token={/envelope/token}"
            title="{gsa:i18n ('Download installer package')}" class="icon icon-sm">
            <img src="/img/agent.svg" alt="{gsa:i18n ('Download Installer')}"/>
          </a>
          <div class="icon icon-sm ajax-post" data-reload="next" data-busy-text="{gsa:i18n ('Verifying Agent...')}">
            <img src="/img/verify.svg" alt="{gsa:i18n ('Verify Agent')}"/>
            <form>
              <input type="hidden" name="cmd" value="verify_agent"/>
              <input type="hidden" name="agent_id" value="{@id}"/>
              <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
              <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
              <input type="hidden" name="token" value="{/envelope/token}"/>
              <input type="hidden" name="next" value="get_agent"/>
            </form>
            <div class="success-dialog" data-title="{gsa:i18n ('Success')}">
              <div class="text-center">
                <xsl:value-of select="gsa:i18n ('Agent has been verified.')"/>
              </div>
            </div>
            <div class="error-dialog">
              <div class="text-center">
                <xsl:value-of select="gsa:i18n ('Agent could not be verified.')"/>
              </div>
            </div>
          </div>
        </td>
      </xsl:otherwise>
    </xsl:choose>
  </tr>
</xsl:template>

<xsl:template match="agent" mode="trash">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <xsl:call-template name="observers-icon">
        <xsl:with-param name="type" select="'Agent'"/>
      </xsl:call-template>
      <b><xsl:value-of select="name"/></b>
      <xsl:choose>
        <xsl:when test="comment != ''">
          <div class="comment">(<xsl:value-of select="comment"/>)</div>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:value-of select="gsa:i18n (installer/trust/text(), 'Trust')"/>
      <xsl:choose>
        <xsl:when test="installer/trust/time != ''">
          <div>(<xsl:value-of select="concat (date:month-abbreviation (installer/trust/time), ' ', date:day-in-month (installer/trust/time), ' ', date:year (installer/trust/time))"/>)</div>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </td>
    <td class="table-actions">
      <xsl:call-template name="restore-icon">
        <xsl:with-param name="id" select="@id"/>
      </xsl:call-template>
      <xsl:call-template name="trash-delete-icon">
        <xsl:with-param name="type" select="'agent'"/>
        <xsl:with-param name="id" select="@id"/>
      </xsl:call-template>
    </td>
  </tr>
</xsl:template>

<xsl:template match="agent" mode="details">
  <div class="gb_window">
    <div class="gb_window_part_left"></div>
    <div class="gb_window_part_right"></div>
    <div class="gb_window_part_center">
      <xsl:value-of select="gsa:i18n ('Agent Details')"/>
      <xsl:call-template name="details-header-icons">
        <xsl:with-param name="cap-type" select="'Agent'"/>
        <xsl:with-param name="type" select="'agent'"/>
      </xsl:call-template>
    </div>
    <div class="gb_window_part_content">
      <xsl:call-template name="minor-details"/>
      <table>
        <tr>
          <td><b><xsl:value-of select="gsa:i18n ('Name')"/>:</b></td>
          <td><b><xsl:value-of select="name"/></b></td>
        </tr>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Comment')"/>:</td>
          <td><xsl:value-of select="comment"/></td>
        </tr>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Trust')"/>:</td>
          <td>
            <xsl:value-of select="gsa:i18n (installer/trust/text(), 'Trust')"/>
            <xsl:choose>
              <xsl:when test="installer/trust/time != ''">
                (<xsl:value-of select="concat (date:month-abbreviation (installer/trust/time), ' ', date:day-in-month (installer/trust/time), ' ', date:year (installer/trust/time))"/>)
              </xsl:when>
              <xsl:otherwise></xsl:otherwise>
            </xsl:choose>
        </td>
        </tr>
      </table>
    </div>
  </div>
  <xsl:call-template name="user-tags-window">
    <xsl:with-param name="resource_type" select="'agent'"/>
  </xsl:call-template>
  <xsl:call-template name="resource-permissions-window">
    <xsl:with-param name="resource_type" select="'agent'"/>
    <xsl:with-param name="permissions" select="../../permissions/get_permissions_response"/>
    <xsl:with-param name="related">
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!--     GET_AGENT -->

<xsl:template match="get_agent">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="commands_response/delete_agent_response"/>
  <xsl:apply-templates select="modify_agent_response"/>
  <xsl:apply-templates select="delete_tag_response"/>
  <xsl:apply-templates select="create_tag_response"/>
  <xsl:apply-templates select="modify_tag_response"/>
  <xsl:apply-templates select="verify_agent_response"/>
  <xsl:apply-templates select="get_agents_response/agent" mode="details"/>
</xsl:template>

<!--     GET_AGENTS_RESPONSE -->

<xsl:template match="get_agents_response">
  <xsl:apply-templates select="modify_agent_response"/>
  <xsl:choose>
    <xsl:when test="substring(@status, 1, 1) = '4' or substring(@status, 1, 1) = '5'">
      <xsl:call-template name="command_result_dialog">
        <xsl:with-param name="operation">
          Get Agents
        </xsl:with-param>
        <xsl:with-param name="status">
          <xsl:value-of select="@status"/>
        </xsl:with-param>
        <xsl:with-param name="msg">
          <xsl:value-of select="@status_text"/>
        </xsl:with-param>
      </xsl:call-template>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="html-agents-table"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="agent" mode="select">
  <option value="{name}"><xsl:value-of select="name"/></option>
</xsl:template>

<xsl:template match="agents_response" mode="select">
  <xsl:apply-templates select="agent" mode="select"/>
</xsl:template>

<!-- BEGIN AGGREGATES MANAGEMENT -->

<xsl:template match="get_aggregate">
  <xsl:variable name="filter_term" select="/envelope/params/filter"/>
  <xsl:variable name="filt_id" select="/envelope/params/filt_id"/>

  <xsl:call-template name="init-d3charts"/>
  <xsl:choose>
    <xsl:when test="$filter_term != ''">
      <div id="applied_filter" class="footnote" style="padding: 5px 10px">
        <b><xsl:value-of select="gsa:i18n('Applied filter:')"/></b>
        <xsl:text> </xsl:text>
        <xsl:value-of select="$filter_term"/>
      </div>
    </xsl:when>
    <xsl:otherwise>
      <div id="applied_filter"/>
    </xsl:otherwise>
  </xsl:choose>

  <xsl:variable name="init_params_js">
    <xsl:if test="/envelope/params/_param[starts-with (name, 'chart_init:')]">{
      <xsl:for-each select="/envelope/params/_param[starts-with (name, 'chart_init:')]">
        "<xsl:value-of select="substring-after (name, 'chart_init:')"/>": "<xsl:value-of select="value"/>"<xsl:if test="position() &lt; count(exslt:node-set (/envelope/params/_param[starts-with (name, 'chart_init:')]))">, </xsl:if>
      </xsl:for-each>
      }</xsl:if>
  </xsl:variable>

  <xsl:variable name="gen_params_js">
    <xsl:if test="/envelope/params/_param[starts-with (name, 'chart_gen:')]">{
      <xsl:for-each select="/envelope/params/_param[starts-with (name, 'chart_gen:')]">
        "<xsl:value-of select="substring-after (name, 'chart_gen:')"/>": "<xsl:value-of select="value"/>"<xsl:if test="position() &lt; count(exslt:node-set (/envelope/params/_param[starts-with (name, 'chart_gen:')]))">, </xsl:if>
      </xsl:for-each>
      }</xsl:if>
  </xsl:variable>

  <xsl:variable name="data_columns">
    <xsl:if test="/envelope/params/_param[starts-with (name, 'data_columns:')]">
      <xsl:for-each select="/envelope/params/_param[starts-with (name, 'data_columns:')]">
       <xsl:value-of select="value"/><xsl:if test="position() &lt; count(exslt:node-set (/envelope/params/_param[starts-with (name, 'data_columns:')]))">,</xsl:if>
      </xsl:for-each>
    </xsl:if>
  </xsl:variable>

  <xsl:variable name="text_columns">
    <xsl:if test="/envelope/params/_param[starts-with (name, 'text_columns:')]">
      <xsl:for-each select="/envelope/params/_param[starts-with (name, 'text_columns:')]">
       <xsl:value-of select="value"/><xsl:if test="position() &lt; count(exslt:node-set (/envelope/params/_param[starts-with (name, 'text_columns:')]))">,</xsl:if>
      </xsl:for-each>
    </xsl:if>
  </xsl:variable>

  <xsl:variable name="y_fields">
    <xsl:if test="/envelope/params/_param[starts-with (name, 'y_fields:')]">
      <xsl:for-each select="/envelope/params/_param[starts-with (name, 'y_fields:')]">
       <xsl:value-of select="value"/><xsl:if test="position() &lt; count(exslt:node-set (/envelope/params/_param[starts-with (name, 'y_fields:')]))">,</xsl:if>
      </xsl:for-each>
    </xsl:if>
  </xsl:variable>

  <xsl:variable name="z_fields">
    <xsl:if test="/envelope/params/_param[starts-with (name, 'z_fields:')]">
      <xsl:for-each select="/envelope/params/_param[starts-with (name, 'z_fields:')]">
       <xsl:value-of select="value"/><xsl:if test="position() &lt; count(exslt:node-set (/envelope/params/_param[starts-with (name, 'z_fields:')]))">,</xsl:if>
      </xsl:for-each>
    </xsl:if>
  </xsl:variable>

  <div id="single-box-dashboard" class="dashboard"
    data-dashboard-name="single-box-dashboard"
    data-default-controllers="aggregate-chart"
    data-filter="{$filter_term}"
    data-filter-id="{$filt_id}"
    data-no-chart-links="{/envelope/params/no_chart_links}"
    data-max-components="1"
    data-detached="1"
    data-hide-controller-select="1"
    data-hide-filter-select="1">
    <div class="dashboard-data-source"
      data-soure-name="aggregate-source"
      data-type="aggregate"
      data-aggregate-type="{/envelope/params/aggregate_type}"
      data-group-column="{/envelope/params/group_column}"
      data-subgroup-column="{/envelope/params/subgroup_column}"
      data-column="{/envelope/params/data_column}"
      data-columns="{$data_columns}"
      data-text-columns="{$text_columns}"
      data-sort-field="{/envelope/params/sort_field}"
      data-sort-stat="{/envelope/params/sort_stat}"
      data-sort-order="{/envelope/params/sort_order}"
      data-first-group="{/envelope/params/first_group}"
      data-max-groups="{/envelope/params/max_groups}"
      data-aggregate-mode="{/envelope/params/aggregate_mode}"
      data-filter="{$filter_term}"
      data-filter-id="{$filt_id}">
      <span class="dashboard-chart"
        data-chart-name="aggregate-chart"
        data-chart-type="{/envelope/params/chart_type}"
        data-chart-template="{/envelope/params/chart_template}"
        data-chart-title="{/envelope/params/chart_title}"
        data-x-field="{/envelope/params/x_field}"
        data-y-fields="{$y_fields}"
        data-z-fields="{$z_fields}"
        data-init-params="{$init_params_js}"
        data-gen-params="{$gen_params_js}"
        />
    </div>
  </div>
  <xsl:call-template name="html-footer"/>
</xsl:template>


<!-- END AGGREGATES MANAGEMENT -->


<!-- BEGIN ALERTS MANAGEMENT -->

<xsl:variable name="include-message-default">
Task '$n': $e

After the event $e,
the following condition was met: $c

This email escalation is configured to apply report format '$r'.
Full details and other report formats are available on the scan engine.

$t
$i

Note:
This email was sent to you as a configured security scan escalation.
Please contact your local system administrator if you think you
should not have received it.
</xsl:variable>

<xsl:variable name="include-message-default-secinfo">
After the event $e,
the following condition was met: $c

$t
$i

Note:
This email was sent to you as a configured security information escalation.
Please contact your local system administrator if you think you
should not have received it.
</xsl:variable>

<xsl:variable name="attach-message-default">
Task '$n': $e

After the event $e,
the following condition was met: $c

This email escalation is configured to attach report format '$r'.
Full details and other report formats are available on the scan engine.

$t

Note:
This email was sent to you as a configured security scan escalation.
Please contact your local system administrator if you think you
should not have received it.
</xsl:variable>

<xsl:variable name="attach-message-default-secinfo">
After the event $e,
the following condition was met: $c

This email escalation is configured to attach the resource list.

$t

Note:
This email was sent to you as a configured security information escalation.
Please contact your local system administrator if you think you
should not have received it.
</xsl:variable>

<xsl:template name="condition-field">
  <xsl:param name="hide"/>
  <xsl:param name="show">
    <xsl:if test="string-length ($hide) = 0">
      <xsl:text>display: none</xsl:text>
    </xsl:if>
  </xsl:param>
  <xsl:param name="condition" select="false ()"/>
  <xsl:param name="condition-text">
    <xsl:choose>
      <xsl:when test="$condition">
        <xsl:value-of select="$condition/text()"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="''"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:param>
  <xsl:param name="filters"/>
    <div id="always_row">
      <div class="radio">
        <xsl:choose>
          <xsl:when test="not ($condition)">
            <xsl:call-template name="radio-button">
              <xsl:with-param name="value" select="'Always'"/>
              <xsl:with-param name="select-value" select="'Always'"/>
              <xsl:with-param name="text" select="gsa:i18n ('Always')"/>
              <xsl:with-param name="name" select="'condition'"/>
            </xsl:call-template>
          </xsl:when>
          <xsl:otherwise>
            <xsl:call-template name="radio-button">
              <xsl:with-param name="value" select="'Always'"/>
              <xsl:with-param name="select-value" select="$condition-text"/>
              <xsl:with-param name="text" select="gsa:i18n ('Always')"/>
              <xsl:with-param name="name" select="'condition'"/>
            </xsl:call-template>
          </xsl:otherwise>
        </xsl:choose>
      </div>
    </div>
    <div style="{$hide}" id="severity_at_least_row" class="form-selection-item-event form-selection-item-event--task">
      <div class="form-item radio">
        <xsl:call-template name="radio-button">
          <xsl:with-param name="value" select="'Severity at least'"/>
          <xsl:with-param name="select-value" select="$condition-text"/>
          <xsl:with-param name="name" select="'condition'"/>
          <xsl:with-param name="text">
            <xsl:value-of select="gsa:i18n ('Severity at least')"/>
          </xsl:with-param>
        </xsl:call-template>
      </div>
      <div class="form-item">
        <xsl:choose>
          <xsl:when test="$condition and $condition/text() = 'Severity at least'">
            <input name="condition_data:severity" value="{$condition/data/text()}" size="5"
              class="spinner"
              data-type="float" min="0" />
          </xsl:when>
          <xsl:otherwise>
            <input name="condition_data:severity" value="0.1" size="5"
              class="spinner"
              data-type="float" min="0" />
          </xsl:otherwise>
        </xsl:choose>
      </div>
    </div>
    <div style="{$hide}" id="severity_changed_row" class="form-selection-item-event form-selection-item-event--task">
      <div class="form-item radio">
        <xsl:call-template name="radio-button">
          <xsl:with-param name="name" select="'condition'"/>
          <xsl:with-param name="value" select="'Severity changed'"/>
          <xsl:with-param name="select-value" select="$condition-text"/>
          <xsl:with-param name="text">
            <xsl:value-of select="gsa:i18n ('Severity level')"/>
          </xsl:with-param>
        </xsl:call-template>
      </div>
      <div class="form-item">
        <select name="condition_data:direction">
          <xsl:call-template name="opt">
            <xsl:with-param name="value" select="'changed'"/>
            <xsl:with-param name="select-value" select="$condition-text"/>
            <xsl:with-param name="content" select="gsa:i18n ('changed', 'Alert Condition|Severity')"/>
          </xsl:call-template>
          <xsl:call-template name="opt">
            <xsl:with-param name="value" select="'increased'"/>
            <xsl:with-param name="select-value" select="$condition-text"/>
            <xsl:with-param name="content" select="gsa:i18n ('increased', 'Alert Condition|Severity')"/>
          </xsl:call-template>
          <xsl:call-template name="opt">
            <xsl:with-param name="value" select="'decreased'"/>
            <xsl:with-param name="select-value" select="$condition-text"/>
            <xsl:with-param name="content" select="gsa:i18n ('decreased', 'Alert Condition|Severity')"/>
          </xsl:call-template>
        </select>
      </div>
    </div>
    <div id="filter_count_at_least_row">
      <div class="form-item radio">
        <xsl:call-template name="radio-button">
          <xsl:with-param name="name" select="'condition'"/>
          <xsl:with-param name="value" select="'Filter count at least'"/>
          <xsl:with-param name="select-value" select="$condition-text"/>
          <xsl:with-param name="text">
            <xsl:value-of select="gsa:i18n ('Filter ', 'Alert Condition')"/>
          </xsl:with-param>
        </xsl:call-template>
      </div>
      <div class="form-item">
        <span id="filter_count_at_least_span_task"
            class="form-selection-item-event form-selection-item-event--task">
          <select id="filter_count_at_least_select_task" name="condition_data:at_least_filter_id"
            class="form-selection-input-event form-selection-input-event--task">
            <xsl:for-each select="$filters/filter[type='Result']">
              <xsl:choose>
                <xsl:when test="$condition and $condition/text() = 'Filter count at least' and $condition/data[name='filter_id']/text() = @id">
                  <option value="{@id}" selected="1">
                    <xsl:value-of select="name"/>
                  </option>
                </xsl:when>
                <xsl:otherwise>
                  <option value="{@id}"><xsl:value-of select="name"/></option>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:for-each>
          </select>
        </span>
        <span id="filter_count_at_least_span_nvts"
          class="form-selection-item-event form-selection-item-event--secinfo">
          <select id="filter_count_at_least_select_nvts" name="condition_data:at_least_filter_id"
            class="form-selection-input-event form-selection-input-event--secinfo">
            <xsl:for-each select="$filters/filter[type='SecInfo']">
              <xsl:choose>
                <xsl:when test="$condition and $condition/text() = 'Filter count at least' and $condition/data[name='filter_id']/text() = @id">
                  <option value="{@id}" selected="1">
                    <xsl:value-of select="name"/>
                  </option>
                </xsl:when>
                <xsl:otherwise>
                  <option value="{@id}"><xsl:value-of select="name"/></option>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:for-each>
          </select>
        </span>
        <xsl:value-of select="gsa:i18n (' matches at least ', 'Alert Condition')"/>
      </div>
      <div class="form-item">
        <xsl:choose>
          <xsl:when test="$condition and $condition/text() = 'Filter count at least'">
            <input name="condition_data:at_least_count"
              value="{$condition/data[name='count']/text()}"
              class="spinner"
              data-type="int" min="0"
              size="5"/>
          </xsl:when>
          <xsl:otherwise>
            <input name="condition_data:at_least_count" value="1"
              class="spinner"
              data-type="int" min="0"
              size="5"/>
          </xsl:otherwise>
        </xsl:choose>
        <span id="filter_count_at_least_results_span"
          class="form-selection-input-event form-selection-input-event--secinfo">
          <xsl:text> </xsl:text>
          <xsl:value-of select="gsa:i18n ('result(s)')"/>
        </span>
        <span id="filter_count_at_least_nvts_span"
          class="form-selection-input-event form-selection-input-event--secinfo">
          <xsl:text> </xsl:text>
          <xsl:value-of select="gsa:i18n ('NVT(s)')"/>
        </span>
      </div>
    </div>
    <div style="{$hide}" id="filter_count_changed_row" class="form-selection-item-event form-selection-item-event--task">
      <div class="form-item radio">
        <xsl:call-template name="radio-button">
          <xsl:with-param name="name" select="'condition'"/>
          <xsl:with-param name="value" select="'Filter count changed'"/>
          <xsl:with-param name="select-value" select="$condition-text"/>
          <xsl:with-param name="text">
            <xsl:value-of select="gsa:i18n ('Filter ', 'Alert Condition')"/>
          </xsl:with-param>
        </xsl:call-template>
      </div>
      <div class="form-item">
        <input type="hidden" name="condition_data:filter_direction" value="increased"/>
        <select name="condition_data:filter_id">
          <xsl:for-each select="$filters/filter[type='Result']">
            <xsl:choose>
              <xsl:when test="$condition and $condition/text() = 'Filter count changed' and $condition/data[name='filter_id']/text() = @id">
                <option value="{@id}" selected="1">
                  <xsl:value-of select="name"/>
                </option>
              </xsl:when>
              <xsl:otherwise>
                <option value="{@id}"><xsl:value-of select="name"/></option>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:for-each>
        </select>
        <xsl:value-of select="gsa:i18n (' matches at least ', 'Alert Condition')"/>
      </div>
      <div class="form-item">
        <xsl:choose>
          <xsl:when test="$condition and $condition/text() = 'Filter count changed'">
            <input name="condition_data:count"
              class="spinner"
              data-type="int" min="0"
              value="{$condition/data[name='count']/text()}"
              size="5"/>
          </xsl:when>
          <xsl:otherwise>
            <input name="condition_data:count" value="1" size="5"
              class="spinner"
              data-type="int" min="0" />
          </xsl:otherwise>
        </xsl:choose>
        <xsl:value-of select="gsa:i18n (' result(s) more than previous scan', 'Alert Condition')"/>
      </div>
    </div>
</xsl:template>

<xsl:template name="html-create-alert-form">
  <xsl:param name="credentials"/>
  <xsl:param name="report-formats"/>
  <xsl:param name="filters"/>
  <xsl:param name="tasks"/>
  <div class="edit-dialog">
    <div class="title"><xsl:value-of select="gsa:i18n ('New Alert')"/></div>
    <div class="content">
      <form action="/omp" method="post" enctype="multipart/form-data" class="form-horizontal">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="create_alert"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden" name="next" value="get_alert"/>
        <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
        <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
        <xsl:if test="not (gsa:may-op ('get_filters'))">
          <input type="hidden" name="filter_id" value="0"/>
        </xsl:if>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Name')"/></label>
          <div class="col-10">
            <input type="text" name="name" value="unnamed" size="30" class="form-control"
              maxlength="80"/>
          </div>
        </div>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Comment')"/></label>
          <div class="col-10">
            <input type="text" name="comment" size="30" maxlength="400" class="form-control"/>
          </div>
        </div>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Event')"/></label>
          <div class="col-10">
            <div class="radio">
              <div class="form-item">
                <label>
                  <input type="radio" name="event" data-select="task" value="Task run status changed" checked="1"
                    class="form-selection-control" id="event" />
                  <xsl:value-of select="gsa:i18n ('Task run status changed to')"/>
                </label>
              </div>
              <div class="form-item">
                <select name="event_data:status">
                  <option value="Done" selected="1"><xsl:value-of select="gsa:i18n ('Done', 'Status')"/></option>
                  <option value="New"><xsl:value-of select="gsa:i18n ('New', 'Status')"/></option>
                  <option value="Requested"><xsl:value-of select="gsa:i18n ('Requested', 'Status')"/></option>
                  <option value="Running"><xsl:value-of select="gsa:i18n ('Running', 'Status')"/></option>
                  <option value="Stop Requested"><xsl:value-of select="gsa:i18n ('Stop Requested', 'Status')"/></option>
                  <option value="Stopped"><xsl:value-of select="gsa:i18n ('Stopped', 'Status')"/></option>
                </select>
              </div>
            </div>
          </div>
          <div class="col-10 offset-2">
            <div class="form-item radio">
              <label>
                <input type="radio" name="event" data-select="secinfo" value="New SecInfo arrived"
                  class="form-selection-control" id="event" />
                &#160; <!-- we need none breakin space here to align the radio input element correctly -->
              </label>
              <select name="event_data:feed_event">
                <option value="new" selected="1"><xsl:value-of select="gsa:i18n ('New', 'Status')"/></option>
                <option value="updated"><xsl:value-of select="gsa:i18n ('Updated', 'Status')"/></option>
              </select>
            </div>
            <div class="form-item">
              <select name="event_data:secinfo_type">
                <option value="nvt" selected="1"><xsl:value-of select="gsa:i18n ('NVTs')"/></option>
                <option value="cve"><xsl:value-of select="gsa:i18n ('CVEs')"/></option>
                <option value="cpe"><xsl:value-of select="gsa:i18n ('CPEs')"/></option>
                <option value="cert_bund_adv"><xsl:value-of select="gsa:i18n ('CERT-Bund Advisories')"/></option>
                <option value="dfn_cert_adv"><xsl:value-of select="gsa:i18n ('DFN-CERT Advisories')"/></option>
                <option value="ovaldef"><xsl:value-of select="gsa:i18n ('OVAL Definition')"/></option>
              </select>
              <xsl:text> </xsl:text>
              <xsl:value-of select="gsa:i18n ('arrived', 'Alert Condition')"/>
            </div>
          </div>
        </div>

        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Condition')"/></label>
          <div class="col-10">
            <xsl:call-template name="condition-field">
              <xsl:with-param name="filters" select="$filters"/>
            </xsl:call-template>
          </div>
        </div>

        <!-- Fields that apply to all/multiple methods. -->

        <xsl:if test="gsa:may-op ('get_filters')">
          <div class="form-group form-selection-item-event form-selection-item-event--task" id="report_result_filter_row">
            <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Report Result Filter')"/></label>
            <div class="col-10">
              <select name="filter_id">
                <option value="0">--</option>
                <xsl:for-each select="$filters/filter[type='Result']">
                  <option value="{@id}"><xsl:value-of select="name"/></option>
                </xsl:for-each>
              </select>
            </div>
          </div>
        </xsl:if>

        <div class="form-group form-selection-item-event form-selection-item-event--secinfo"
          style="display: none" id="details_url_row">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Details URL')"/></label>
          <div class="col-10">
            <input type="text" name="method_data:details_url"
              class="form-control"
              size="30" maxlength="1000"
              value="https://secinfo.greenbone.net/omp?cmd=get_info&amp;info_type=$t&amp;info_id=$o&amp;details=1&amp;token=guest"/>
          </div>
        </div>

        <!-- Method -->

        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Method')"/></label>
          <div class="col-10">
            <select name="method" id="method" class="form-selection-control">
              <option data-select="email" value="Email" checked="1">
                <xsl:value-of select="gsa:i18n ('Email')"/>
              </option>
              <option data-select="http-get" value="HTTP Get"
                class="form-selection-input-event form-selection-input-event--task">
                <xsl:value-of select="gsa:i18n ('HTTP Get')"/>
              </option>
              <option data-select="scp" value="SCP">
                <xsl:value-of select="gsa:i18n ('SCP')"/>
              </option>
              <option data-select="send" value="Send">
                <xsl:value-of select="gsa:i18n ('Send to host ')"/>
              </option>
              <option data-select="snmp" value="SNMP">
                <xsl:value-of select="gsa:i18n ('SNMP')"/>
              </option>
              <option data-select="sourcefire" value="Sourcefire Connector"
                class="form-selection-input-event form-selection-input-event--task">
                <xsl:value-of select="gsa:i18n ('Sourcefire Connector')"/>
              </option>
              <option data-select="start_task" value="Start Task"
                class="form-selection-input-event form-selection-input-event--task">
                <xsl:value-of select="gsa:i18n ('Start Task')"/>
              </option>
              <option data-select="syslog" value="Syslog">
                <xsl:value-of select="gsa:i18n ('System Logger')"/>
              </option>
              <option data-select="verinice" value="verinice Connector"
                class="form-selection-input-event form-selection-input-event--task">
                <xsl:value-of select="gsa:i18n ('verinice.PRO Connector')"/>
              </option>
            </select>
          </div>
        </div>

        <!-- Method: Email. -->

        <div class="form-group form-selection-item-method form-selection-item-method--email">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('To Address', 'Alert|Email')"/></label>
          <div class="col-10">
            <div class="form-item">
              <input type="text" name="method_data:to_address" size="30" maxlength="301"
                class="form-control"/>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--email">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('From Address', 'Alert|Email')"/></label>
          <div class="col-10">
            <div class="form-item">
              <input type="text" name="method_data:from_address" size="30" maxlength="301"
                class="form-control"/>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--email">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Subject', 'Alert|Email')"/></label>
          <div id="email_subject_task"
            class="col-10 form-selection-item-event form-selection-item-event--task">
            <div class="form-item">
              <input id="email_subject_task_input" type="text" name="method_data:subject"
                size="30" maxlength="80"
                class="form-control form-selection-input-event form-selection-input-event--task"
                value="[OpenVAS-Manager] Task '$n': $e"/>
            </div>
          </div>
          <div  id="email_subject_secinfo"
            class="col-10 form-selection-item-event form-selection-item-event--secinfo">
            <div class="form-item">
              <input id="email_subject_secinfo_input" type="text" name="method_data:subject"
                size="30" maxlength="80"
                class="form-control form-selection-input-event form-selection-input-event--secinfo"
                value="[OpenVAS-Manager] $T $q $S since $d"/>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--email" id="email_content_row">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Content', 'Alert|Email')"/></label>
          <div class="col-10">
            <div class="radio">
              <label>
                <input type="radio" name="method_data:notice" value="1" checked="1"/>
                <xsl:value-of select="gsa:i18n ('Simple notice', 'Alert|Email')"/>
              </label>
            </div>

            <xsl:if test="gsa:may-op ('get_report_formats')">
              <div>
                <div class="form-item radio">
                  <label>
                    <input type="radio" name="method_data:notice" value="0"/>
                    <span class="form-selection-item-event form-selection-item-event--task">
                      <xsl:value-of select="gsa:i18n ('Include report', 'Alert|Email')"/>
                    </span>
                    <span id="email_content_include_secinfo"
                      class="form-selection-item-event form-selection-item-event--secinfo"
                      style="display: none">
                      <xsl:value-of select="gsa:i18n ('Include list of resources', 'Alert|Email')"/>
                      <xsl:text> with message:</xsl:text>
                    </span>
                  </label>
                </div>
                <div class="form-item form-selection-item-event form-selection-item-event--task"
                  style="display: inline">
                  <select name="method_data:notice_report_format">
                    <xsl:for-each select="$report-formats/report_format">
                      <xsl:if test="substring(content_type, 1, 5) = 'text/'">
                        <xsl:choose>
                          <xsl:when test="@id='19f6f1b3-7128-4433-888c-ccc764fe6ed5'">
                            <option value="{@id}" selected="1">
                              <xsl:value-of select="name"/>
                            </option>
                          </xsl:when>
                          <xsl:otherwise>
                            <option value="{@id}">
                              <xsl:value-of select="name"/>
                            </option>
                          </xsl:otherwise>
                        </xsl:choose>
                      </xsl:if>
                    </xsl:for-each>
                  </select>
                  <xsl:text> with message:</xsl:text>
                </div>
                <div id="email_content_include_message_task"
                  class="form-selection-item-event form-selection-item-event--task">
                  <textarea id="message_include_task"
                    name="method_data:message"
                    class="form-control form-selection-input-event form-selection-input-event--task"
                    rows="8" cols="50">
                    <xsl:value-of select="$include-message-default"/>
                  </textarea>
                </div>
                <div id="email_content_include_message_secinfo" style="display: none"
                  class="form-selection-item-event form-selection-item-event--secinfo">
                  <textarea id="message_include_secinfo"
                    name="method_data:message"
                    class="form-control form-selection-input-event form-selection-input-event--secinfo"
                    rows="8" cols="50">
                    <xsl:value-of select="$include-message-default-secinfo"/>
                  </textarea>
                </div>
              </div>

              <div>
                <div class="form-item radio">
                  <label>
                    <input type="radio" name="method_data:notice" value="2"/>
                    <span class="form-selection-item-event form-selection-item-event--task">
                      <xsl:value-of select="gsa:i18n ('Attach report', 'Alert|Email')"/>
                    </span>
                    <span class="form-selection-item-event form-selection-item-event--secinfo"
                      style="display: none">
                      <xsl:value-of select="gsa:i18n ('Attach list of resources', 'Alert|Email')"/>
                      <xsl:text> with message:</xsl:text>
                    </span>
                  </label>
                </div>
                <div id="email_content_attach_task"
                  style="display: inline"
                  class="form-item form-selection-item-event form-selection-item-event--task">
                  <select name="method_data:notice_attach_format">
                    <xsl:for-each select="$report-formats/report_format">
                      <xsl:choose>
                        <xsl:when test="@id='1a60a67e-97d0-4cbf-bc77-f71b08e7043d'">
                          <option value="{@id}" selected="1">
                            <xsl:value-of select="name"/>
                          </option>
                        </xsl:when>
                        <xsl:otherwise>
                          <option value="{@id}">
                            <xsl:value-of select="name"/>
                          </option>
                        </xsl:otherwise>
                      </xsl:choose>
                    </xsl:for-each>
                  </select>
                  <xsl:text> with message:</xsl:text>
                </div>
                <div id="email_content_attach_message_task"
                  class="form-selection-item-event form-selection-item-event--task">
                  <textarea id="message_attach_task"
                    name="method_data:message_attach"
                    class="form-control form-selection-input-event form-selection-input-event--task"
                    rows="9" cols="50">
                    <xsl:value-of select="$attach-message-default"/>
                  </textarea>
                </div>
                <div id="email_content_attach_message_secinfo" style="display: none"
                  class="form-selection-item-event form-selection-item-event--secinfo">
                  <textarea id="message_attach_secinfo"
                    class="form-control form-selection-input-event form-selection-input-event--secinfo"
                    name="method_data:message_attach"
                    rows="8" cols="50">
                    <xsl:value-of select="$attach-message-default-secinfo"/>
                  </textarea>
                </div>
              </div>
            </xsl:if>
          </div>
        </div>

        <!-- Method: System Logger. -->

        <input type="hidden" name="method_data:submethod" value="syslog"/>

        <!-- Method: HTTP Get. -->

        <div class="form-group form-selection-item-method form-selection-item-method--http-get"
          style="display: none" id="http_get_row">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('HTTP Get URL')"/>
          </label>
          <div class="col-10">
            <input type="text" name="method_data:URL" size="30" maxlength="301"
              class="form-control"/>
          </div>
        </div>

        <!-- Method: SNMP. -->

        <div class="form-group form-selection-item-method form-selection-item-method--snmp"
          style="display: none" id="snmp_row">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Community', 'SNMP')"/>
          </label>
          <div class="col-10">
            <input type="text" name="method_data:snmp_community" size="30" maxlength="301"
              class="form-control" value="public"/>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--snmp"
          style="display: none" id="snmp_row">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Agent')"/>
          </label>
          <div class="col-10">
            <input type="text" name="method_data:snmp_agent" size="30" maxlength="301"
              class="form-control" value="localhost"/>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--snmp"
          style="display: none" id="snmp_row">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Message')"/>
          </label>
          <div class="col-10">
            <input type="text" name="method_data:snmp_message" size="30" maxlength="301"
              class="form-control" value="$e"/>
          </div>
        </div>

        <!-- Method: Sourcefire Connector. -->

        <div class="form-group form-selection-item-method form-selection-item-method--sourcefire" id="sourcefire_row">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Defense Center IP')"/>
          </label>
          <div class="col-10">
            <div class="form-item">
                <input type="text" name="method_data:defense_center_ip"
                  class="form-control"
                  size="30" maxlength="40"/>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--sourcefire">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Defense Center Port')"/></label>
          <div class="col-10">
            <div class="form-item">
              <input type="text" name="method_data:defense_center_port"
                data-type="int" class="form-control spinner" min="0" max="65535"
                size="30" maxlength="400" value="8307"/>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--sourcefire">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('PKCS12 file')"/></label>
          <div class="col-10">
            <div class="form-item">
              <input type="file" name="method_data:pkcs12" size="30"/>
            </div>
          </div>
        </div>

        <!-- Method: Verinice Connector. -->

        <div class="form-group form-selection-item-method form-selection-item-method--verinice">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('verinice.PRO URL')"/></label>
          <div class="col-10">
            <div class="form-item">
              <input type="text" name="method_data:verinice_server_url"
                class="form-control"
                size="30" maxlength="256"/>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--verinice">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Credential')"/>
          </label>
          <div class="col-10">
            <select name="method_data:verinice_server_credential">
              <xsl:for-each select="$credentials/credential">
                <option value="{@id}">
                  <xsl:value-of select="name"/>
                </option>
              </xsl:for-each>
            </select>
            <div class="form-item">
              <a href="#" title="{ gsa:i18n('Create a new Credential') }"
                class="new-action-icon icon icon-sm" data-type="credential" data-done="select[name='method_data:verinice_server_credential']" data-extra="restrict_credential_type=up">
                <img src="/img/new.svg"/>
              </a>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--verinice">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('verinice.PRO Report')"/></label>
          <div class="col-10">
            <select name="method_data:verinice_server_report_format">
              <xsl:for-each select="$report-formats/report_format">
                <xsl:if test="extension = 'vna'">
                  <xsl:choose>
                    <xsl:when test="name='Verinice ISM'">
                      <option value="{@id}" selected="1">
                        <xsl:value-of select="name"/>
                      </option>
                    </xsl:when>
                    <xsl:otherwise>
                      <option value="{@id}">
                        <xsl:value-of select="name"/>
                      </option>
                    </xsl:otherwise>
                  </xsl:choose>
                </xsl:if>
              </xsl:for-each>
              <xsl:if test="count ($report-formats/report_format) = 0">
                <option value="''">--</option>
              </xsl:if>
            </select>
          </div>
        </div>

        <!-- Method: Start task. -->

        <div class="form-group form-selection-item-method form-selection-item-method--start_task" id="start_task_row">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Start Task')"/>
          </label>
          <div class="col-10">
            <select name="method_data:start_task_task">
              <xsl:for-each select="$tasks/task">
                <option value="{@id}">
                  <xsl:value-of select="name"/>
                </option>
              </xsl:for-each>
            </select>
          </div>
        </div>

        <!-- Method: Send. -->

        <div class="form-group form-selection-item-method form-selection-item-method--send" id="send_row">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Send to host ')"/>
          </label>
          <div class="col-10">
            <div class="form-item">
              <input type="text" name="method_data:send_host"
                class="form-control"
                size="30" maxlength="256"/>
            </div>
            <div class="form-item">
              <xsl:value-of select="gsa:i18n (' on port ')"/>
            </div>
            <div class="form-item">
              <input type="text" name="method_data:send_port"
                class="form-control"
                size="6" maxlength="6"/>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--send" id="send_to_host_report_row">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Report')"/></label>
          <div class="col-10">
            <select name="method_data:send_report_format">
              <xsl:for-each select="$report-formats/report_format">
                <option value="{@id}">
                  <xsl:value-of select="name"/>
                </option>
              </xsl:for-each>
              <xsl:if test="count ($report-formats/report_format) = 0">
                <option value="''">--</option>
              </xsl:if>
            </select>
          </div>
        </div>

        <!-- Method: SCP. -->

        <div class="form-group form-selection-item-method form-selection-item-method--scp">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Credential')"/>
          </label>
          <div class="col-10">
            <select name="method_data:scp_credential">
              <xsl:for-each select="$credentials/credential[not (contains (login, '@') or contains (login, ':'))]">
                <option value="{@id}">
                  <xsl:value-of select="name"/>
                </option>
              </xsl:for-each>
            </select>
            <div class="form-item">
              <a href="#" title="{ gsa:i18n('Create a new Credential') }"
                class="new-action-icon icon icon-sm" data-type="credential" data-done="select[name='method_data:scp_credential']" data-extra="restrict_credential_type=up">
                <img src="/img/new.svg"/>
              </a>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--scp">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Host')"/></label>
          <div class="col-10">
            <div class="form-item">
              <input type="text" name="method_data:scp_host"
                class="form-control"
                size="30" maxlength="256"/>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--scp">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Known Hosts')"/></label>
          <div class="col-10">
            <div class="form-item">
              <textarea name="method_data:scp_known_hosts"
                class="form-control"
                rows="3" cols="50">
              </textarea>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--scp">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Path')"/></label>
          <div class="col-10">
            <div class="form-item">
              <input type="text" name="method_data:scp_path"
                class="form-control"
                size="30" maxlength="256" value="report.xml"/>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--scp">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Report')"/></label>
          <div class="col-10">
            <select name="method_data:scp_report_format">
              <xsl:for-each select="$report-formats/report_format">
                <option value="{@id}">
                  <xsl:value-of select="name"/>
                </option>
              </xsl:for-each>
              <xsl:if test="count ($report-formats/report_format) = 0">
                <option value="''">--</option>
              </xsl:if>
            </select>
          </div>
        </div>

      </form>
    </div>
  </div>
</xsl:template>

<xsl:template match="new_alert">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="create_alert_response"/>
  <xsl:apply-templates select="commands_response/delete_alert_response"/>
  <xsl:call-template name="html-create-alert-form">
    <xsl:with-param
      name="credentials"
      select="get_credentials_response | commands_response/get_credentials_response"/>
    <xsl:with-param
      name="report-formats"
      select="get_report_formats_response | commands_response/get_report_formats_response"/>
    <xsl:with-param
      name="filters"
      select="get_filters_response | commands_response/get_filters_response"/>
    <xsl:with-param
      name="tasks"
      select="get_tasks_response | commands_response/get_tasks_response"/>
  </xsl:call-template>
</xsl:template>

<xsl:template name="html-alerts-table">
  <xsl:call-template name="list-window">
    <xsl:with-param name="type" select="'alert'"/>
    <xsl:with-param name="cap-type" select="'Alert'"/>
    <xsl:with-param name="resources-summary" select="alerts"/>
    <xsl:with-param name="resources" select="alert"/>
    <xsl:with-param name="count" select="count (alert)"/>
    <xsl:with-param name="filtered-count" select="alert_count/filtered"/>
    <xsl:with-param name="full-count" select="alert_count/text ()"/>
    <xsl:with-param name="columns" xmlns="">
      <column>
        <name><xsl:value-of select="gsa:i18n('Name')"/></name>
        <field>name</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Event')"/></name>
        <field>event</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Condition')"/></name>
        <field>condition</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Method')"/></name>
        <field>method</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Filter')"/></name>
        <field>filter</field>
      </column>
    </xsl:with-param>
    <xsl:with-param name="icon-count" select="5"/>
  </xsl:call-template>
</xsl:template>

<xsl:template name="radio-button">
  <xsl:param name="name"></xsl:param>
  <xsl:param name="value"></xsl:param>
  <xsl:param name="text"></xsl:param>
  <xsl:param name="select-value"></xsl:param>
  <label>
    <xsl:choose>
      <xsl:when test="$value = $select-value">
        <input type="radio" name="{$name}" checked="1" value="{$value}"/>
      </xsl:when>
      <xsl:otherwise>
        <input type="radio" name="{$name}" value="{$value}"/>
      </xsl:otherwise>
    </xsl:choose>
    <xsl:choose>
      <xsl:when test="$text">
        <xsl:copy-of select="$text"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$value"/>
      </xsl:otherwise>
    </xsl:choose>
  </label>
</xsl:template>

<xsl:template name="html-edit-alert-form">
  <xsl:param name="credentials"/>
  <xsl:param name="report-formats"></xsl:param>
  <xsl:param name="filters"/>
  <xsl:param name="tasks"/>
  <div class="edit-dialog">
    <div class="title">
      <xsl:value-of select="gsa:i18n ('Edit Alert')"/>
    </div>
    <div class="content">
      <form action="" method="post" enctype="multipart/form-data" class="form-horizontal">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="save_alert"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden"
               name="alert_id"
               value="{get_alerts_response/alert/@id}"/>
        <input type="hidden" name="next" value="{/envelope/params/next}"/>
        <input type="hidden" name="alert" value="{/envelope/params/alert}"/>
        <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>

        <xsl:if test="not (gsa:may-op ('get_filters'))">
          <input type="hidden" name="filter_id" value="0"/>
        </xsl:if>

        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Name')"/></label>
          <div class="col-10">
            <input type="text" name="name" size="30" maxlength="80"
              class="form-control"
              value="{get_alerts_response/alert/name}"/>
          </div>
        </div>

        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Comment')"/></label>
          <div class="col-10">
            <input type="text" name="comment" size="30" maxlength="400"
              class="form-control"
              value="{get_alerts_response/alert/comment}"/>
          </div>
        </div>

        <!-- Event. -->
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Event')"/></label>
          <div class="col-10">
            <div class="form-item radio">
              <label>
                <xsl:choose>
                  <xsl:when test="get_alerts_response/alert/event/text() = 'Task run status changed'">
                    <input type="radio" name="event" value="Task run status changed" checked="1"
                      class="form-selection-control" data-select="task" id="event"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="radio" name="event" value="Task run status changed"
                      class="form-selection-control" data-select="task" id="event"/>
                  </xsl:otherwise>
                </xsl:choose>
                <xsl:value-of select="gsa:i18n ('Task run status changed to')"/>
                <xsl:text> </xsl:text>
              </label>
            </div>

            <xsl:variable name="eventdata"
              select="get_alerts_response/alert/event/data[name='status']/text()"/>

            <div class="form-item">
              <select name="event_data:status">
                <xsl:if test="$eventdata = 'Delete Requested'">
                  <!-- In case the user has an alert with this state. -->
                  <xsl:call-template name="opt">
                    <xsl:with-param name="value" select="'Delete Requested'"/>
                    <xsl:with-param name="content" select="gsa:i18n ('Delete Requested', 'Status')"/>
                    <xsl:with-param name="select-value" select="$eventdata"/>
                  </xsl:call-template>
                </xsl:if>
                <xsl:call-template name="opt">
                  <xsl:with-param name="value" select="'Done'"/>
                  <xsl:with-param name="content" select="gsa:i18n ('Done', 'Status')"/>
                  <xsl:with-param name="select-value" select="$eventdata"/>
                </xsl:call-template>
                <xsl:call-template name="opt">
                  <xsl:with-param name="value" select="'New'"/>
                  <xsl:with-param name="content" select="gsa:i18n ('New', 'Status')"/>
                  <xsl:with-param name="select-value" select="$eventdata"/>
                </xsl:call-template>
                <xsl:call-template name="opt">
                  <xsl:with-param name="value" select="'Requested'"/>
                  <xsl:with-param name="content" select="gsa:i18n ('Requested', 'Status')"/>
                  <xsl:with-param name="select-value" select="$eventdata"/>
                </xsl:call-template>
                <xsl:call-template name="opt">
                  <xsl:with-param name="value" select="'Running'"/>
                  <xsl:with-param name="content" select="gsa:i18n ('Running', 'Status')"/>
                  <xsl:with-param name="select-value" select="$eventdata"/>
                </xsl:call-template>
                <xsl:call-template name="opt">
                  <xsl:with-param name="value" select="'Stop Requested'"/>
                  <xsl:with-param name="content" select="gsa:i18n ('Stop Requested', 'Status')"/>
                  <xsl:with-param name="select-value" select="$eventdata"/>
                </xsl:call-template>
                <xsl:call-template name="opt">
                  <xsl:with-param name="value" select="'Stopped'"/>
                  <xsl:with-param name="content" select="gsa:i18n ('Stopped', 'Status')"/>
                  <xsl:with-param name="select-value" select="$eventdata"/>
                </xsl:call-template>
              </select>
            </div>
          </div>

          <div class="col-10 offset-2">
            <div class="form-item radio">
              <xsl:choose>
                <xsl:when test="get_alerts_response/alert/event/text() = 'New SecInfo arrived'">
                  <label>
                    <input type="radio" name="event" value="New SecInfo arrived" checked="1"
                      class="form-selection-control" data-select="secinfo" id="event"/>
                    &#160; <!-- we need none breakin space here to align the radio input element correctly -->
                  </label>
                  <select name="event_data:feed_event">
                    <option value="new" selected="1"><xsl:value-of select="gsa:i18n ('New', 'Status')"/></option>
                    <option value="updated"><xsl:value-of select="gsa:i18n ('Updated', 'Status')"/></option>
                  </select>
                </xsl:when>
                <xsl:when test="get_alerts_response/alert/event/text() = 'Updated SecInfo arrived'">
                  <label>
                    <input type="radio" name="event" value="New SecInfo arrived" checked="1"
                      class="form-selection-control" data-select="secinfo" id="event"/>
                    &#160; <!-- we need none breakin space here to align the radio input element correctly -->
                  </label>
                  <select name="event_data:feed_event">
                    <option value="new"><xsl:value-of select="gsa:i18n ('New', 'Status')"/></option>
                    <option value="updated" selected="1"><xsl:value-of select="gsa:i18n ('Updated', 'Status')"/></option>
                  </select>
                </xsl:when>
                <xsl:otherwise>
                  <label>
                    <input type="radio" name="event" value="New SecInfo arrived"
                      class="form-selection-control" data-select="secinfo" id="event"/>
                    &#160; <!-- we need none breakin space here to align the radio input element correctly -->
                  </label>
                  <select name="event_data:feed_event">
                    <option value="new" selected="1"><xsl:value-of select="gsa:i18n ('New', 'Status')"/></option>
                    <option value="updated"><xsl:value-of select="gsa:i18n ('Updated', 'Status')"/></option>
                  </select>
                </xsl:otherwise>
              </xsl:choose>
            </div>

            <xsl:variable name="eventdata"
              select="get_alerts_response/alert/event/data[name='secinfo_type']/text()"/>

            <div class="form-item">
              <select name="event_data:secinfo_type">
                <xsl:call-template name="opt">
                  <xsl:with-param name="value" select="'nvt'"/>
                  <xsl:with-param name="content" select="gsa:i18n ('NVTs')"/>
                  <xsl:with-param name="select-value" select="$eventdata"/>
                </xsl:call-template>
                <xsl:call-template name="opt">
                  <xsl:with-param name="value" select="'cve'"/>
                  <xsl:with-param name="content" select="gsa:i18n ('CVEs')"/>
                  <xsl:with-param name="select-value" select="$eventdata"/>
                </xsl:call-template>
                <xsl:call-template name="opt">
                  <xsl:with-param name="value" select="'cpe'"/>
                  <xsl:with-param name="content" select="gsa:i18n ('CPEs')"/>
                  <xsl:with-param name="select-value" select="$eventdata"/>
                </xsl:call-template>
                <xsl:call-template name="opt">
                  <xsl:with-param name="value" select="'cert_bund_adv'"/>
                  <xsl:with-param name="content" select="gsa:i18n ('CERT-Bund Advisories')"/>
                  <xsl:with-param name="select-value" select="$eventdata"/>
                </xsl:call-template>
                <xsl:call-template name="opt">
                  <xsl:with-param name="value" select="'dfn_cert_adv'"/>
                  <xsl:with-param name="content" select="gsa:i18n ('DFN-CERT Advisories')"/>
                  <xsl:with-param name="select-value" select="$eventdata"/>
                </xsl:call-template>
                <xsl:call-template name="opt">
                  <xsl:with-param name="value" select="'ovaldef'"/>
                  <xsl:with-param name="content" select="gsa:i18n ('OVAL Definitions')"/>
                  <xsl:with-param name="select-value" select="$eventdata"/>
                </xsl:call-template>
              </select>
              <xsl:text> </xsl:text>
              <xsl:value-of select="gsa:i18n ('arrived', 'Alert Condition')"/>
            </div>
          </div>
        </div>

        <!-- Condition. -->
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Condition')"/></label>
          <div class="col-10">
            <xsl:variable name="condition"
              select="get_alerts_response/alert/condition"/>

            <xsl:call-template name="condition-field">
              <xsl:with-param name="condition"
                select="get_alerts_response/alert/condition"/>
              <xsl:with-param name="filters" select="$filters"/>
            </xsl:call-template>
          </div>
        </div>

        <!-- Report Result Filter. -->

        <xsl:variable name="method"
          select="get_alerts_response/alert/method"/>

        <xsl:if test="gsa:may-op ('get_filters')">
          <xsl:variable name="filtername"
            select="get_alerts_response/alert/filter/name"/>
          <div class="form-group form-selection-item-event form-selection-item-event--task" id="report_result_filter_row">
            <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Report Result Filter')"/></label>
            <div class="col-10">
              <select name="filter_id">
                <option value="0">--</option>
                <xsl:for-each select="$filters/filter[type='Result']">
                  <xsl:choose>
                    <xsl:when test="name = $filtername">
                      <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
                    </xsl:when>
                    <xsl:otherwise>
                      <option value="{@id}"><xsl:value-of select="name"/></option>
                    </xsl:otherwise>
                  </xsl:choose>
                </xsl:for-each>
              </select>
            </div>
          </div>
        </xsl:if>

        <div class="form-group form-selection-item-event form-selection-item-event--secinfo"
          style="display: none" id="details_url_row">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Details URL')"/></label>
          <div class="col-10">
            <xsl:choose>
              <xsl:when test="$method/data[name='details_url']/text()">
                <input type="text" name="method_data:details_url"
                  size="30" maxlength="1000"
                  class="form-control"
                  value="{$method/data[name='details_url']/text()}"/>
              </xsl:when>
              <xsl:otherwise>
                <input type="text" name="method_data:details_url"
                  class="form-control"
                  size="30" maxlength="1000"
                  value="https://secinfo.greenbone.net/omp?cmd=get_info&amp;info_type=$t&amp;info_id=$o&amp;details=1&amp;token=guest"/>
              </xsl:otherwise>
            </xsl:choose>
          </div>
        </div>

        <!-- Method. -->

        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Method')"/></label>
          <div class="col-10">
            <select name="method" id="method" class="form-selection-control">
              <xsl:choose>
                <xsl:when test="$method/text() = 'Email'">
                  <option data-select="email" value="Email" selected="1">
                    <xsl:value-of select="gsa:i18n ('Email')"/>
                  </option>
                </xsl:when>
                <xsl:otherwise>
                  <option data-select="email" value="Email">
                    <xsl:value-of select="gsa:i18n ('Email')"/>
                  </option>
                </xsl:otherwise>
              </xsl:choose>

              <xsl:choose>
                <xsl:when test="$method/text() = 'HTTP Get'">
                  <option data-select="http-get" value="HTTP Get" selected="1"
                    class="form-selection-input-event form-selection-input-event--task">
                    <xsl:value-of select="gsa:i18n ('HTTP Get')"/>
                  </option>
                </xsl:when>
                <xsl:otherwise>
                  <option data-select="http-get" value="HTTP Get"
                    class="form-selection-input-event form-selection-input-event--task">
                    <xsl:value-of select="gsa:i18n ('HTTP Get')"/>
                  </option>
                </xsl:otherwise>
              </xsl:choose>

              <xsl:choose>
                <xsl:when test="$method/text() = 'SCP'">
                  <option data-select="scp" value="SCP" selected="1">
                    <xsl:value-of select="gsa:i18n ('SCP')"/>
                  </option>
                </xsl:when>
                <xsl:otherwise>
                  <option data-select="scp" value="SCP">
                    <xsl:value-of select="gsa:i18n ('SCP')"/>
                  </option>
                </xsl:otherwise>
              </xsl:choose>

              <xsl:choose>
                <xsl:when test="$method/text() = 'Send'">
                  <option data-select="send" value="Send" selected="1">
                    <xsl:value-of select="gsa:i18n ('Send to host ')"/>
                  </option>
                </xsl:when>
                <xsl:otherwise>
                  <option data-select="send" value="Send">
                    <xsl:value-of select="gsa:i18n ('Send to host ')"/>
                  </option>
                </xsl:otherwise>
              </xsl:choose>

              <xsl:choose>
                <xsl:when test="$method/text() = 'SNMP'">
                  <option data-select="snmp" value="SNMP" selected="1">
                    <xsl:value-of select="gsa:i18n ('SNMP')"/>
                  </option>
                </xsl:when>
                <xsl:otherwise>
                  <option data-select="snmp" value="SNMP">
                    <xsl:value-of select="gsa:i18n ('SNMP')"/>
                  </option>
                </xsl:otherwise>
              </xsl:choose>

              <xsl:choose>
                <xsl:when test="$method/text() = 'Sourcefire Connector'">
                  <option data-select="sourcefire" value="Sourcefire Connector" selected="1"
                    class="form-selection-input-event form-selection-input-event--task">
                    <xsl:value-of select="gsa:i18n ('Sourcefire Connector')"/>
                  </option>
                </xsl:when>
                <xsl:otherwise>
                  <option data-select="sourcefire" value="Sourcefire Connector"
                    class="form-selection-input-event form-selection-input-event--task">
                    <xsl:value-of select="gsa:i18n ('Sourcefire Connector')"/>
                  </option>
                </xsl:otherwise>
              </xsl:choose>

              <xsl:choose>
                <xsl:when test="$method/text() = 'Start Task'">
                  <option data-select="start_task" value="Start Task" selected="1"
                    class="form-selection-input-event form-selection-input-event--task">
                    <xsl:value-of select="gsa:i18n ('Start Task')"/>
                  </option>
                </xsl:when>
                <xsl:otherwise>
                  <option data-select="start_task" value="Start Task"
                    class="form-selection-input-event form-selection-input-event--task">
                    <xsl:value-of select="gsa:i18n ('Start Task')"/>
                  </option>
                </xsl:otherwise>
              </xsl:choose>

              <xsl:choose>
                <xsl:when test="$method/text() = 'Syslog'">
                  <option data-select="syslog" value="Syslog" selected="1">
                    <xsl:value-of select="gsa:i18n ('System Logger')"/>
                  </option>
                </xsl:when>
                <xsl:otherwise>
                  <option data-select="syslog" value="Syslog">
                    <xsl:value-of select="gsa:i18n ('System Logger')"/>
                  </option>
                </xsl:otherwise>
              </xsl:choose>

              <xsl:choose>
                <xsl:when test="$method/text() = 'verinice Connector'">
                  <option data-select="verinice" value="verinice Connector" selected="1"
                    class="form-selection-input-event form-selection-input-event--task">
                    <xsl:value-of select="gsa:i18n ('verinice.PRO Connector')"/>
                  </option>
                </xsl:when>
                <xsl:otherwise>
                  <option data-select="verinice" value="verinice Connector"
                    class="form-selection-input-event form-selection-input-event--task">
                    <xsl:value-of select="gsa:i18n ('verinice.PRO Connector')"/>
                  </option>
                </xsl:otherwise>
              </xsl:choose>
            </select>
          </div>
        </div>

        <!-- Method: Email. -->

        <div class="form-group form-selection-item-method form-selection-item-method--email">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('To Address', 'Alert|Email')"/></label>
          <div class="col-10">
            <div class="form-item">
              <input type="text" name="method_data:to_address" size="30" maxlength="301"
                value="{$method/data[name='to_address']/text()}"
                class="form-control"/>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--email">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('From Address', 'Alert|Email')"/></label>
          <div class="col-10">
            <div class="form-item">
              <input type="text" name="method_data:from_address" size="30" maxlength="301"
                value="{$method/data[name='from_address']/text()}"
                class="form-control"/>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--email">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Subject', 'Alert|Email')"/></label>
          <div id="email_subject_task"
            class="col-10 form-selection-item-event form-selection-item-event--task">
            <div class="form-item">
              <xsl:choose>
                <xsl:when test="$method/data[name='subject']/text()">
                  <input id="email_subject_task_input" type="text" name="method_data:subject"
                    size="30" maxlength="80"
                    class="form-control form-selection-input-event form-selection-input-event--task"
                    value="{$method/data[name='subject']/text()}"/>
                </xsl:when>
                <xsl:otherwise>
                  <input id="email_subject_task_input" type="text" name="method_data:subject"
                    size="30" maxlength="80"
                    class="form-control form-selection-input-event form-selection-input-event--task"
                    value="[OpenVAS-Manager] Task '$n': $e"/>
                </xsl:otherwise>
              </xsl:choose>
            </div>
          </div>
          <div style="display: none" id="email_subject_secinfo"
            class="col-10 form-selection-item-event form-selection-item-event--secinfo">
            <div class="form-item">
              <xsl:choose>
                <xsl:when test="$method/data[name='subject']/text()">
                  <input id="email_subject_secinfo_input" type="text" name="method_data:subject"
                    size="30" maxlength="80"
                    class="form-control form-selection-input-event form-selection-input-event--secinfo"
                    value="{$method/data[name='subject']/text()}"/>
                </xsl:when>
                <xsl:otherwise>
                  <input id="email_subject_secinfo_input" type="text" name="method_data:subject"
                    size="30" maxlength="80"
                    class="form-control form-selection-input-event form-selection-input-event--secinfo"
                    value="[OpenVAS-Manager] $T $q $S since $d"/>
                </xsl:otherwise>
              </xsl:choose>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--email" id="email_content_row">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Content', 'Alert|Email')"/></label>
          <div class="col-10">
            <div class="radio">
              <xsl:call-template name="radio-button">
                <xsl:with-param name="name" select="'method_data:notice'"/>
                <xsl:with-param name="text" select="gsa:i18n ('Simple notice', 'Alert|Email')"/>
                <xsl:with-param name="value" select="'1'"/>
                <xsl:with-param name="select-value" select="$method/data[name='notice']/text()"/>
              </xsl:call-template>
            </div>

            <xsl:if test="gsa:may-op ('get_report_formats')">
              <div>
                <div class="form-item radio">
                  <label>
                    <xsl:choose>
                      <xsl:when test="$method/data[name='notice']/text() = '0'">
                        <input type="radio" name="method_data:notice" checked="1" value="0"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <input type="radio" name="method_data:notice" value="0"/>
                      </xsl:otherwise>
                    </xsl:choose>
                    <span class="form-selection-item-event form-selection-item-event--task">
                      <xsl:value-of select="gsa:i18n ('Include report', 'Alert|Email')"/>
                    </span>
                    <span id="email_content_include_secinfo"
                      class="form-selection-item-event form-selection-item-event--secinfo"
                      style="display: none">
                      <xsl:value-of select="gsa:i18n ('Include list of resources', 'Alert|Email')"/>
                      <xsl:text> with message:</xsl:text>
                    </span>
                  </label>
                </div>
                <div class="form-item form-selection-item-event form-selection-item-event--task"
                  style="display: inline">
                  <select name="method_data:notice_report_format">
                    <xsl:for-each select="$report-formats/report_format">
                      <xsl:if test="substring(content_type, 1, 5) = 'text/'">
                        <xsl:choose>
                          <xsl:when test="@id=$method/data[name='notice_report_format']/text()">
                            <option value="{@id}" selected="1">
                              <xsl:value-of select="name"/>
                            </option>
                          </xsl:when>
                          <xsl:otherwise>
                            <option value="{@id}">
                              <xsl:value-of select="name"/>
                            </option>
                          </xsl:otherwise>
                        </xsl:choose>
                      </xsl:if>
                    </xsl:for-each>
                  </select>
                  <xsl:text> with message:</xsl:text>
                </div>
                <div id="email_content_include_message_task"
                  class="form-selection-item-event form-selection-item-event--task">
                  <textarea id="message_include_task"
                    name="method_data:message"
                    class="form-control form-selection-input-event form-selection-input-event--task"
                    rows="8" cols="50">
                    <xsl:choose>
                      <xsl:when test="$method/data[name='notice']/text() = 0">
                        <xsl:value-of select="$method/data[name='message']/text()"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <xsl:value-of select="$include-message-default"/>
                      </xsl:otherwise>
                    </xsl:choose>
                  </textarea>
                </div>
                <div id="email_content_include_message_secinfo" style="display: none"
                  class="form-selection-item-event form-selection-item-event--secinfo">
                  <textarea id="message_include_secinfo"
                    name="method_data:message"
                    class="form-control form-selection-input-event form-selection-input-event--secinfo"
                    rows="8" cols="50">
                    <xsl:choose>
                      <xsl:when test="$method/data[name='notice']/text() = 0">
                        <xsl:value-of select="$method/data[name='message']/text()"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <xsl:value-of select="$include-message-default-secinfo"/>
                      </xsl:otherwise>
                    </xsl:choose>
                  </textarea>
                </div>
              </div>
              <div>
                <div class="form-item radio">
                  <label>
                    <xsl:choose>
                      <xsl:when test="$method/data[name='notice']/text() = '2'">
                        <input type="radio" name="method_data:notice" checked="1" value="2"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <input type="radio" name="method_data:notice" value="2"/>
                      </xsl:otherwise>
                    </xsl:choose>
                    <span class="form-selection-item-event form-selection-item-event--task">
                      <xsl:value-of select="gsa:i18n ('Attach report', 'Alert|Email')"/>
                    </span>
                    <span class="form-selection-item-event form-selection-item-event--secinfo"
                      style="display: none">
                      <xsl:value-of select="gsa:i18n ('Attach list of resources', 'Alert|Email')"/>
                      <xsl:text> with message:</xsl:text>
                    </span>
                  </label>
                </div>
                <div id="email_content_attach_task"
                  style="display: inline"
                  class="form-item form-selection-item-event form-selection-item-event--task">
                  <select name="method_data:notice_attach_format">
                    <xsl:for-each select="$report-formats/report_format">
                      <xsl:choose>
                        <xsl:when test="@id=$method/data[name='notice_attach_format']/text()">
                          <option value="{@id}" selected="1">
                            <xsl:value-of select="name"/>
                          </option>
                        </xsl:when>
                        <xsl:otherwise>
                          <option value="{@id}">
                            <xsl:value-of select="name"/>
                          </option>
                        </xsl:otherwise>
                      </xsl:choose>
                    </xsl:for-each>
                  </select>
                  <xsl:text> with message:</xsl:text>
                </div>
                <div id="email_content_attach_message_task"
                  class="form-selection-item-event form-selection-item-event--task">
                  <textarea id="message_attach_task"
                    name="method_data:message_attach"
                    class="form-control form-selection-input-event form-selection-input-event--task"
                    rows="9" cols="50">
                    <xsl:choose>
                      <xsl:when test="$method/data[name='notice']/text() = 2 and (get_alerts_response/alert/event/text() != 'New SecInfo arrived' and get_alerts_response/alert/event/text() != 'Updated SecInfo arrived')">
                        <xsl:value-of select="$method/data[name='message']/text()"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <xsl:value-of select="$attach-message-default"/>
                      </xsl:otherwise>
                    </xsl:choose>
                  </textarea>
                </div>
                <div id="email_content_attach_message_secinfo" style="display: none"
                  class="form-selection-item-event form-selection-item-event--secinfo">
                  <textarea id="message_attach_secinfo"
                    class="form-control form-selection-input-event form-selection-input-event--secinfo"
                    name="method_data:message_attach"
                    rows="8" cols="50">
                    <xsl:choose>
                      <xsl:when test="$method/data[name='notice']/text() = 2 and (get_alerts_response/alert/event/text() = 'New SecInfo arrived' or get_alerts_response/alert/event/text() = 'Updated SecInfo arrived')">
                        <xsl:value-of select="$method/data[name='message']/text()"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <xsl:value-of select="$attach-message-default-secinfo"/>
                      </xsl:otherwise>
                    </xsl:choose>
                  </textarea>
                </div>
              </div>
            </xsl:if>
          </div>
        </div>

        <!-- Method: System Logger. -->

        <xsl:variable name="submethod"
                      select="$method/data[name='submethod']/text()"/>
        <xsl:choose>
          <xsl:when test="string-length ($submethod) = 0 or $submethod = 'syslog'">
            <input type="hidden" name="method_data:submethod" value="syslog"/>
          </xsl:when>
          <xsl:otherwise>
            <div class="form-group form-selection-item-method form-selection-item-method--syslog"
              style="display: none">
              <label class="col-2 control-label">
                <xsl:value-of select="gsa:i18n ('Log Level')"/>
              </label>
              <div class="col-10">
                <label>
                  <select name="method_data:submethod">
                    <xsl:call-template name="opt">
                      <xsl:with-param name="value" select="'syslog'"/>
                      <xsl:with-param name="select-value" select="$submethod"/>
                    </xsl:call-template>
                    <option value="{$submethod}" selected="1">
                      <xsl:value-of select="$submethod"/>
                    </option>
                  </select>
                </label>
              </div>
            </div>
          </xsl:otherwise>
        </xsl:choose>

        <!-- Method: HTTP Get. -->

        <div class="form-group form-selection-item-method form-selection-item-method--http-get"
          style="display: none" id="http_get_row">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('HTTP Get URL')"/>
          </label>
          <div class="col-10">
            <input type="text" name="method_data:URL" size="30" maxlength="301"
              value="{$method/data[name='URL']/text()}"
              class="form-control"/>
          </div>
        </div>

        <!-- Method: Sourcefire Connector. -->

        <div class="form-group form-selection-item-method form-selection-item-method--sourcefire" id="sourcefire_row">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Defense Center IP')"/>
          </label>
          <div class="col-10">
            <div class="form-item">
                <input type="text" name="method_data:defense_center_ip"
                  class="form-control"
                  size="30" maxlength="40"
                  value="{$method/data[name='defense_center_ip']/text()}"/>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--sourcefire">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Defense Center Port')"/></label>
          <div class="col-10">
            <div class="form-item">
              <xsl:choose>
                <xsl:when test="$method/data[name='defense_center_port']/text()">
                  <input type="text" name="method_data:defense_center_port"
                    data-type="int" class="form-control spinner" min="0" max="65535"
                    size="30" maxlength="400"  value="{$method/data[name='defense_center_port']/text()}"/>
                </xsl:when>
                <xsl:otherwise>
                  <input type="text" name="method_data:defense_center_port"
                    data-type="int" class="form-control spinner" min="0" max="65535"
                    size="30" maxlength="400" value="8307"/>
                </xsl:otherwise>
              </xsl:choose>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--sourcefire" id="sourcefire_row">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('PKCS12 file')"/></label>
          <div class="col-10">
            <div class="form-item">
              <input type="file" name="method_data:pkcs12" size="30"/>
            </div>
          </div>
        </div>

        <!-- Method: verinice Connector. -->

        <div class="form-group form-selection-item-method form-selection-item-method--verinice">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('verinice.PRO URL')"/></label>
          <div class="col-10">
            <div class="form-item">
              <input type="text" name="method_data:verinice_server_url"
                class="form-control"
                size="30" maxlength="256" value="{$method/data[name='verinice_server_url']/text()}"/>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--verinice">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Credential')"/>
          </label>
          <div class="col-10">
            <select name="method_data:verinice_server_credential">
              <xsl:for-each select="$credentials/credential">
                <xsl:choose>
                  <xsl:when test="@id=$method/data[name='verinice_server_credential']/text()">
                    <option value="{@id}" selected="1">
                      <xsl:value-of select="name"/>
                    </option>
                  </xsl:when>
                  <xsl:otherwise>
                    <option value="{@id}">
                      <xsl:value-of select="name"/>
                    </option>
                  </xsl:otherwise>
                </xsl:choose>
              </xsl:for-each>
            </select>
            <div class="form-item">
              <a href="#" title="{ gsa:i18n('Create a new Credential') }"
                class="new-action-icon icon icon-sm" data-type="credential" data-done="select[name='method_data:verinice_server_credential']" data-extra="restrict_credential_type=up">
                <img src="/img/new.svg"/>
              </a>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--verinice">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('verinice.PRO Report')"/></label>
          <div class="col-10">
            <select name="method_data:verinice_server_report_format">
              <xsl:for-each select="$report-formats/report_format">
                <xsl:if test="extension = 'vna'">
                  <xsl:choose>
                    <xsl:when test="@id=$method/data[name='verinice_server_report_format']/text()">
                      <option value="{@id}" selected="1">
                        <xsl:value-of select="name"/>
                      </option>
                    </xsl:when>
                    <xsl:otherwise>
                      <option value="{@id}">
                        <xsl:value-of select="name"/>
                      </option>
                    </xsl:otherwise>
                  </xsl:choose>
                </xsl:if>
              </xsl:for-each>
              <xsl:if test="count ($report-formats/report_format) = 0">
                <option value="''">--</option>
              </xsl:if>
            </select>
          </div>
        </div>

        <!-- Method: Start Task. -->

        <div class="form-group form-selection-item-method form-selection-item-method--start_task" id="start_task_row">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Start Task')"/>
          </label>
          <div class="col-10">
            <select name="method_data:start_task_task">
              <xsl:for-each select="$tasks/task">
                <xsl:choose>
                  <xsl:when test="@id=$method/data[name='start_task_task']/text()">
                    <option value="{@id}" selected="1">
                      <xsl:value-of select="name"/>
                    </option>
                  </xsl:when>
                  <xsl:otherwise>
                    <option value="{@id}">
                      <xsl:value-of select="name"/>
                    </option>
                  </xsl:otherwise>
                </xsl:choose>
              </xsl:for-each>
            </select>
          </div>
        </div>

        <!-- Method: Send. -->

        <div class="form-group form-selection-item-method form-selection-item-method--send" id="send_row">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Send to host ')"/>
          </label>
          <div class="col-10">
            <div class="form-item">
              <input type="text" name="method_data:send_host"
                class="form-control"
                value="{$method/data[name='send_host']/text()}"
                size="30" maxlength="256"/>
            </div>
            <div class="form-item">
              <xsl:value-of select="gsa:i18n (' on port ')"/>
            </div>
            <div class="form-item">
              <input type="text" name="method_data:send_port"
                class="form-control"
                value="{$method/data[name='send_port']/text()}"
                size="6" maxlength="6"/>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--send" id="send_to_host_report_row">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Report')"/></label>
          <div class="col-10">
            <select name="method_data:send_report_format">
              <xsl:for-each select="$report-formats/report_format">
                <xsl:choose>
                  <xsl:when test="@id=$method/data[name='send_report_format']/text()">
                    <option value="{@id}" selected="1">
                      <xsl:value-of select="name"/>
                    </option>
                  </xsl:when>
                  <xsl:otherwise>
                    <option value="{@id}">
                      <xsl:value-of select="name"/>
                    </option>
                  </xsl:otherwise>
                </xsl:choose>
              </xsl:for-each>
              <xsl:if test="count ($report-formats/report_format) = 0">
                <option value="''">--</option>
              </xsl:if>
            </select>
          </div>
        </div>

        <!-- Method: SCP. -->

        <div class="form-group form-selection-item-method form-selection-item-method--scp">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Credential')"/>
          </label>
          <div class="col-10">
            <select name="method_data:scp_credential">
              <xsl:for-each select="$credentials/credential[not (contains (login, '@') or contains (login, ':'))]">
                <xsl:choose>
                  <xsl:when test="@id=$method/data[name='scp_credential']/text()">
                    <option value="{@id}" selected="1">
                      <xsl:value-of select="name"/>
                    </option>
                  </xsl:when>
                  <xsl:otherwise>
                    <option value="{@id}">
                      <xsl:value-of select="name"/>
                    </option>
                  </xsl:otherwise>
                </xsl:choose>
              </xsl:for-each>
            </select>
            <div class="form-item">
              <a href="#" title="{ gsa:i18n('Create a new Credential') }"
                class="new-action-icon icon icon-sm" data-type="credential" data-done="select[name='method_data:scp_credential']" data-extra="restrict_credential_type=up">
                <img src="/img/new.svg"/>
              </a>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--scp">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Host')"/></label>
          <div class="col-10">
            <div class="form-item">
              <input type="text" name="method_data:scp_host"
                class="form-control"
                value="{$method/data[name='scp_host']/text()}"
                size="30" maxlength="256"/>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--scp">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Known Hosts')"/></label>
          <div class="col-10">
            <div class="form-item">
              <textarea name="method_data:scp_known_hosts"
                class="form-control"
                rows="3" cols="50">
                <xsl:value-of select="$method/data[name='scp_known_hosts']/text()"/>
              </textarea>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--scp">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Path')"/></label>
          <div class="col-10">
            <div class="form-item">
              <xsl:choose>
                <xsl:when test="$method/data[name='scp_path']/text()">
                  <input type="text" name="method_data:scp_path"
                    class="form-control"
                    value="{$method/data[name='scp_path']/text()}"
                    size="30" maxlength="256"/>
                </xsl:when>
                <xsl:otherwise>
                  <input type="text" name="method_data:scp_path"
                    class="form-control"
                    value="report.xml"
                    size="30" maxlength="256"/>
                </xsl:otherwise>
              </xsl:choose>
            </div>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--scp">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Report')"/></label>
          <div class="col-10">
            <select name="method_data:scp_report_format">
              <xsl:for-each select="$report-formats/report_format">
                <xsl:call-template name="opt">
                  <xsl:with-param name="value" select="@id"/>
                  <xsl:with-param name="content" select="name"/>
                  <xsl:with-param name="select-value" select="$method/data[name='scp_report_format']/text()"/>
                </xsl:call-template>
              </xsl:for-each>
              <xsl:if test="count ($report-formats/report_format) = 0">
                <option value="''">--</option>
              </xsl:if>
            </select>
          </div>
        </div>

        <!-- Method: SNMP. -->

        <div class="form-group form-selection-item-method form-selection-item-method--snmp"
          style="display: none" id="snmp_row">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Community', 'SNMP')"/>
          </label>
          <div class="col-10">
            <input type="text" name="method_data:snmp_community" size="30" maxlength="301"
              value="{$method/data[name='snmp_community']/text()}"
              class="form-control"/>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--snmp"
          style="display: none" id="snmp_row">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Agent')"/>
          </label>
          <div class="col-10">
            <input type="text" name="method_data:snmp_agent" size="30" maxlength="301"
              value="{$method/data[name='snmp_agent']/text()}"
              class="form-control"/>
          </div>
        </div>

        <div class="form-group form-selection-item-method form-selection-item-method--snmp"
          style="display: none" id="snmp_row">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Message')"/>
          </label>
          <div class="col-10">
            <input type="text" name="method_data:snmp_message" size="30" maxlength="301"
              value="{$method/data[name='snmp_message']/text()}"
              class="form-control"/>
          </div>
        </div>

      </form>
    </div>
  </div>
</xsl:template>

<xsl:template match="edit_alert">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="modify_alert_response"/>
  <xsl:call-template name="html-edit-alert-form">
    <xsl:with-param
      name="credentials"
      select="get_credentials_response | commands_response/get_credentials_response"/>
    <xsl:with-param
      name="report-formats"
      select="get_report_formats_response | commands_response/get_report_formats_response"/>
    <xsl:with-param
      name="filters"
      select="get_filters_response | commands_response/get_filters_response"/>
    <xsl:with-param
      name="tasks"
      select="get_tasks_response | commands_response/get_tasks_response"/>
  </xsl:call-template>
</xsl:template>

<xsl:template match="get_alerts_response">
</xsl:template>

<!--     CREATE_ALERT_RESPONSE -->

<xsl:template match="create_alert_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Create Alert</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!--     DELETE_ALERT_RESPONSE -->

<xsl:template match="delete_alert_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">
      Delete Alert
    </xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!--     TEST_ALERT_RESPONSE -->

<xsl:template match="test_alert_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Test Alert</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!--     MODIFY_ALERT_RESPONSE -->

<xsl:template match="modify_alert_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Save Alert</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!--     ALERT -->

<xsl:template match="alert">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <xsl:call-template name="observers-icon">
        <xsl:with-param name="type" select="'Alert'"/>
      </xsl:call-template>
      <b>
        <a href="/omp?cmd=get_alert&amp;alert_id={@id}&amp;filter={str:encode-uri (../filters/term, true ())}&amp;token={/envelope/token}"
           title="{gsa:view_details_title ('Alert', name)}">
          <xsl:value-of select="name"/>
        </a>
      </b>
      <xsl:choose>
        <xsl:when test="comment != ''">
          <div class="comment">(<xsl:value-of select="comment"/>)</div>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="event/text() = 'New SecInfo arrived'">
          <xsl:text>New </xsl:text>
          <xsl:value-of select="gsa:type-name-plural (event/data[name='secinfo_type']/text())"/>
          <xsl:text> arrived</xsl:text>
        </xsl:when>
        <xsl:when test="event/text() = 'Updated SecInfo arrived'">
          <xsl:text>Updated </xsl:text>
          <xsl:value-of select="gsa:type-name-plural (event/data[name='secinfo_type']/text())"/>
          <xsl:text> arrived</xsl:text>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="gsa:i18n (event/text())"/>
        </xsl:otherwise>
      </xsl:choose>
      <xsl:choose>
        <xsl:when test="event/text()='Task run status changed' and string-length(event/data[name='status']/text()) &gt; 0">
          (<xsl:value-of select="gsa:i18n ('to', 'Status Change')"/>
           <xsl:text> </xsl:text>
           <xsl:value-of select="gsa:i18n (event/data[name='status']/text(), 'Status')"/>)
        </xsl:when>
      </xsl:choose>
    </td>
    <td>
      <xsl:call-template name="condition"/>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="method/text()='SCP'">
          <xsl:value-of select="gsa:i18n ('SCP to ')"/>
          <xsl:choose>
            <xsl:when test="method/data[name='scp_credential']/credential/login">
              <xsl:value-of select="method/data[name='scp_credential']/credential/login"/>
            </xsl:when>
            <xsl:otherwise>
              <i>(<xsl:value-of select="gsa:i18n ('Credential unavailable')"/>)</i>
            </xsl:otherwise>
          </xsl:choose>
          <xsl:text>@</xsl:text>
          <xsl:value-of select="method/data[name='scp_host']/text()"/>
          <xsl:text>:</xsl:text>
          <xsl:value-of select="method/data[name='scp_path']/text()"/>
        </xsl:when>
        <xsl:when test="method/text()='Send'">
          <xsl:value-of select="gsa:i18n ('Send to ')"/>
          <xsl:value-of select="method/data[name='send_host']/text()"/>
          <xsl:text>:</xsl:text>
          <xsl:value-of select="method/data[name='send_port']/text()"/>
        </xsl:when>
        <xsl:when test="method/text()='Syslog' and method/data[name='submethod']/text() = 'SNMP'">
          SNMP
        </xsl:when>
        <xsl:when test="method/text()='SNMP'">
          <xsl:value-of select="gsa:i18n ('SNMP to ')"/>
          <xsl:value-of select="method/data[name='snmp_agent']/text()"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="gsa:i18n (method/text(), 'Alert Method')"/>
          <xsl:choose>
            <xsl:when test="method/text()='Email' and string-length(method/data[name='to_address']/text()) &gt; 0">
              <div>(<xsl:value-of select="gsa:i18n ('To', 'Alert|Email')"/>
              <xsl:text> </xsl:text>
              <xsl:value-of select="method/data[name='to_address']/text()"/>)</div>
            </xsl:when>
            <xsl:when test="method/text()='Start Task'">
              <xsl:variable name="id" select="method/data[name='start_task_task']/text()"/>
              <xsl:variable name="name" select="../../get_tasks_response/task[@id = $id]/name"/>
              <xsl:text> </xsl:text>
              <xsl:choose>
                <xsl:when test="string-length ($name) = 0">
                  <xsl:value-of select="gsa:i18n('Unavailable')"/>
                  <br/>
                  (<xsl:value-of select="gsa:i18n('ID')"/>: <xsl:value-of select="$id"/>)
                </xsl:when>
                <xsl:otherwise>
                  <a href="/omp?cmd=get_task&amp;task_id={$id}{gsa:token ()}">
                    <xsl:value-of select="$name"/>
                  </a>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:when>
          </xsl:choose>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="boolean (filter/permissions) and count (filter/permissions/permission) = 0">
          <xsl:value-of select="filter/name"/>
        </xsl:when>
        <xsl:when test="gsa:may-op ('get_filters')">
          <a href="/omp?cmd=get_filter&amp;filter_id={filter/@id}&amp;token={/envelope/token}" title="{gsa:i18n ('Details')}">
            <xsl:value-of select="filter/name"/>
          </a>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="filter/name"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <xsl:choose>
      <xsl:when test="/envelope/params/bulk_select = 1">
        <td style="text-align:center">
          <label style="width:100%">
            <input name="bulk_selected:{@id}" type="checkbox" style="width:100%; height:100%" title="{gsa:i18n ('Select for bulk action')}"/>
          </label>
        </td>
      </xsl:when>
      <xsl:otherwise>
        <td class="table-actions">
          <xsl:call-template name="list-window-line-icons">
            <xsl:with-param name="cap-type" select="'Alert'"/>
            <xsl:with-param name="type" select="'alert'"/>
            <xsl:with-param name="id" select="@id"/>
          </xsl:call-template>
          <div class="icon icon-sm ajax-post" data-busy-text="{gsa:i18n ('Testing Alert...')}">
              <xsl:variable name="title">
                <xsl:value-of select="gsa:i18n ('Test Alert', 'Action Verb')"/>
              </xsl:variable>
              <img src="/img/start.svg" alt="{$title}" title="{$title}"/>
              <form action="/omp" method="post" enctype="multipart/form-data">
                  <input type="hidden" name="token" value="{/envelope/token}"/>
                  <input type="hidden" name="caller" value="{/envelope/current_page}"/>
                  <input type="hidden" name="cmd" value="test_alert"/>
                  <input type="hidden" name="alert_id" value="{@id}"/>
                  <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
                  <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
              </form>
              <div class="success-dialog" data-title="{gsa:i18n ('Success')}">
                <p class="text-center">
                  <xsl:value-of select="gsa-i18n:strformat (gsa:i18n ('Testing the alert %1 was successful.'), name)"/>
                </p>
              </div>
              <div class="error-dialog">
                  <p class="text-center">
                    <xsl:value-of select="gsa-i18n:strformat (gsa:i18n ('Testing the alert %1 failed.'), name)"/>
                  </p>
              </div>
          </div>
        </td>
      </xsl:otherwise>
    </xsl:choose>
  </tr>
</xsl:template>

<xsl:template match="alert" mode="trash">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <xsl:call-template name="observers-icon">
        <xsl:with-param name="type" select="'Alert'"/>
      </xsl:call-template>
      <b><xsl:value-of select="name"/></b>
      <xsl:choose>
        <xsl:when test="comment != ''">
          <div class="comment">(<xsl:value-of select="comment"/>)</div>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:value-of select="gsa:i18n (event/text(), 'Alert Event')"/>
      <xsl:choose>
        <xsl:when test="event/text()='Task run status changed' and string-length(event/data[name='status']/text()) &gt; 0">
          <div>(<xsl:value-of select="gsa:i18n ('to', 'Status Change')"/>
          <xsl:text> </xsl:text>
          <xsl:value-of select="gsa:i18n (event/data[name='status']/text(), 'Status')"/>)</div>
        </xsl:when>
      </xsl:choose>
    </td>
    <td>
      <xsl:value-of select="gsa:i18n (condition/text(), 'Alert Condition')"/>
      <xsl:choose>
        <xsl:when test="condition/text()='Severity at least' and string-length(condition/data[name='severity']/text()) &gt; 0">
          <div>(<xsl:value-of select="condition/data[name='severity']/text()"/>)</div>
        </xsl:when>
        <xsl:when test="condition/text()='Severity changed' and string-length(condition/data[name='direction']/text()) &gt; 0">
          <div>(<xsl:value-of select="gsa:i18n (condition/data[name='direction']/text())"/>)</div>
        </xsl:when>
      </xsl:choose>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="method/text()='Syslog' and method/data[name='submethod']/text() = 'SNMP'">
          SNMP
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="gsa:i18n (method/text(), 'Alert Method')"/>
          <xsl:choose>
            <xsl:when test="method/text()='Email' and string-length(method/data[name='to_address']/text()) &gt; 0">
              <div>(<xsl:value-of select="gsa:i18n ('To', 'Alert|Email')"/>
                    <xsl:text> </xsl:text>
                    <xsl:value-of select="method/data[name='to_address']/text()"/>)</div>
            </xsl:when>
          </xsl:choose>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="filter/trash = '1'">
          <xsl:value-of select="filter/name"/>
        </xsl:when>
        <xsl:otherwise>
          <a href="/omp?cmd=get_filter&amp;filter_id={filter/@id}&amp;token={/envelope/token}" title="{gsa:i18n ('Details')}">
            <xsl:value-of select="filter/name"/>
          </a>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td class="table-actions">
      <xsl:choose>
        <xsl:when test="not (gsa:may-op ('restore'))"/>
        <xsl:when test="filter/trash = '1'">
          <img src="/img/restore_inactive.svg" alt="{gsa:i18n ('Restore')}"
               title="{gsa:i18n ('Filter')}{gsa:i18n (' must be restored first.', 'Trashcan')}"
               class="icon icon-sm"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="restore-icon">
            <xsl:with-param name="id" select="@id"/>
          </xsl:call-template>
        </xsl:otherwise>
      </xsl:choose>
      <xsl:choose>
        <xsl:when test="in_use='0'">
          <xsl:call-template name="trash-delete-icon">
            <xsl:with-param name="type" select="'alert'"/>
            <xsl:with-param name="id" select="@id"/>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <img src="/img/delete_inactive.svg"
               alt="{gsa:i18n ('Delete')}"
               title="{gsa:i18n ('Alert is still in use')}"
               class="icon icon-sm"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
  </tr>
</xsl:template>

<xsl:template name="condition">
  <xsl:choose>
    <xsl:when test="condition/text()='Filter count at least'">
      <xsl:variable name="id" select="condition/data[name='filter_id']/text()"/>
      <xsl:text>Filter </xsl:text>
      <xsl:choose>
        <xsl:when test="gsa:may-op ('get_filters')">
          <a href="/omp?cmd=get_filter&amp;filter_id={$id}&amp;token={/envelope/token}"
             title="{gsa:i18n ('Details')}">
            <xsl:value-of select="../../get_filters_response/filter[@id=$id]/name"/>
          </a>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="../../get_filters_response/filter[@id=$id]/name"/>
        </xsl:otherwise>
      </xsl:choose>
      <xsl:text> matches at least </xsl:text>
      <xsl:value-of select="condition/data[name='count']/text()"/>
      <xsl:choose>
        <xsl:when test="event/text()='New SecInfo arrived' or event/text()='Updated SecInfo arrived'">
          <xsl:text> NVT</xsl:text>
        </xsl:when>
        <xsl:otherwise>
          <xsl:text> result</xsl:text>
        </xsl:otherwise>
      </xsl:choose>
      <xsl:if test="condition/data[name='count']/text() &gt; 1">
        <xsl:text>s</xsl:text>
      </xsl:if>
    </xsl:when>
    <xsl:when test="condition/text()='Filter count changed'">
      <xsl:variable name="id" select="condition/data[name='filter_id']/text()"/>
      <xsl:text>Filter </xsl:text>
      <xsl:choose>
        <xsl:when test="gsa:may-op ('get_filters')">
          <a href="/omp?cmd=get_filter&amp;filter_id={$id}&amp;token={/envelope/token}"
             title="{gsa:i18n ('Details')}">
            <xsl:value-of select="../../get_filters_response/filter[@id=$id]/name"/>
          </a>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="../../get_filters_response/filter[@id=$id]/name"/>
        </xsl:otherwise>
      </xsl:choose>
      <xsl:text> matches at least </xsl:text>
      <xsl:value-of select="condition/data[name='count']/text()"/>
      <xsl:choose>
        <xsl:when test="condition/data[name='direction']/text() = 'decreased'">
          <xsl:text> fewer</xsl:text>
        </xsl:when>
        <xsl:otherwise>
          <xsl:text> more</xsl:text>
        </xsl:otherwise>
      </xsl:choose>
      <xsl:choose>
        <xsl:when test="condition/data[name='count']/text() &gt; 1">
          <xsl:text> results than previous scan</xsl:text>
        </xsl:when>
        <xsl:otherwise>
          <xsl:text> result than previous scan</xsl:text>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>
    <xsl:when test="condition/text()='Severity at least' and string-length(condition/data[name='severity']/text()) &gt; 0">
      <xsl:value-of select="gsa:i18n (condition/text(), 'Alert Condition')"/>
      (<xsl:value-of select="condition/data[name='severity']/text()"/>)
    </xsl:when>
    <xsl:when test="condition/text()='Severity changed' and string-length(condition/data[name='direction']/text()) &gt; 0">
      <xsl:value-of select="gsa:i18n (condition/text(), 'Alert Condition')"/>
      (<xsl:value-of select="gsa:i18n (condition/data[name='direction']/text(), 'Alert Condition')"/>)
    </xsl:when>
    <xsl:otherwise>
      <xsl:value-of select="gsa:i18n (condition/text(), 'Alert Condition')"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="alert" mode="details">
  <div class="toolbar">
    <xsl:call-template name="details-header-icons">
      <xsl:with-param name="cap-type" select="'Alert'"/>
      <xsl:with-param name="type" select="'alert'"/>
    </xsl:call-template>
  </div>

  <div class="section-header">
    <xsl:call-template name="minor-details"/>
    <h1>
      <a href="/omp?cmd=get_alerts&amp;token={/envelope/token}"
         title="{gsa:i18n ('Alerts')}">
        <img class="icon icon-lg" src="/img/alert.svg" alt="Alerts"/>
      </a>
      <xsl:value-of select="gsa:i18n ('Alert')"/>:
      <xsl:value-of select="name"/>
      <xsl:text> </xsl:text>
    </h1>
  </div>

  <div class="section-box">
    <table>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Comment')"/>:</td>
        <td><xsl:value-of select="comment"/></td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Condition')"/>:</td>
        <td>
          <xsl:call-template name="condition"/>
        </td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Event')"/>:</td>
        <td>
          <xsl:choose>
            <xsl:when test="event/text() = 'New SecInfo arrived'">
              <xsl:text>New </xsl:text>
              <xsl:value-of select="gsa:type-name-plural (event/data[name='secinfo_type']/text())"/>
              <xsl:text> arrived</xsl:text>
            </xsl:when>
            <xsl:when test="event/text() = 'Updated SecInfo arrived'">
              <xsl:text>Updated </xsl:text>
              <xsl:value-of select="gsa:type-name-plural (event/data[name='secinfo_type']/text())"/>
              <xsl:text> arrived</xsl:text>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="gsa:i18n (event/text(), 'Alert Event')"/>
            </xsl:otherwise>
          </xsl:choose>
          <xsl:choose>
            <xsl:when test="event/text()='Task run status changed' and string-length(event/data[name='status']/text()) &gt; 0">
              (<xsl:value-of select="gsa:i18n ('to', 'Status Change')"/>
               <xsl:text> </xsl:text>
               <xsl:value-of select="gsa:i18n (event/data[name='status']/text(), 'Status')"/>)
            </xsl:when>
          </xsl:choose>
        </td>
      </tr>
      <tr>
        <td valign="top"><xsl:value-of select="gsa:i18n ('Method')"/>:</td>
        <td>
          <table>
            <tr>
              <td colspan="3">
                <xsl:choose>
                  <xsl:when test="(event/text() = 'New SecInfo arrived' or event/text() = 'Updated SecInfo arrived') and method/text()='Send'">
                    <xsl:value-of select="gsa:i18n ('Send list of resources to host')"/>
                  </xsl:when>
                  <xsl:when test="method/text()='Send'">
                    <xsl:value-of select="gsa:i18n ('Send report to host')"/>
                  </xsl:when>
                  <xsl:when test="method/text()='Syslog' and method/data[name='submethod']/text() = 'SNMP'">
                    SNMP
                  </xsl:when>
                  <xsl:otherwise>
                    <xsl:value-of select="gsa:i18n (method/text(), 'Alert Method')"/>
                  </xsl:otherwise>
                </xsl:choose>
              </td>
            </tr>
            <xsl:choose>
              <xsl:when test="method/text()='Email'">
                <tr>
                  <td width="45"></td>
                  <td><xsl:value-of select="gsa:i18n ('To Address', 'Alert|Email')"/>:</td>
                  <td>
                    <xsl:choose>
                      <xsl:when test="string-length(method/data[name='to_address']/text()) &gt; 0">
                        <xsl:value-of select="method/data[name='to_address']/text()"/>
                      </xsl:when>
                    </xsl:choose>
                  </td>
                </tr>
                <tr>
                  <td width="45"></td>
                  <td><xsl:value-of select="gsa:i18n ('From Address', 'Alert|Email')"/>:</td>
                  <td>
                    <xsl:choose>
                      <xsl:when test="string-length(method/data[name='from_address']/text()) &gt; 0">
                        <xsl:value-of select="method/data[name='from_address']/text()"/>
                      </xsl:when>
                    </xsl:choose>
                  </td>
                </tr>
                <xsl:choose>
                  <xsl:when test="event/text() = 'Updated SecInfo arrived' or event/text() = 'New SecInfo arrived'">
                    <tr>
                      <td width="45"></td>
                      <td><xsl:value-of select="gsa:i18n ('Content', 'Alert|Email')"/>:</td>
                      <td>
                        <xsl:choose>
                          <xsl:when test="method/data[name='notice']/text() = '0'">
                            <xsl:value-of select="gsa:i18n ('Include list of resources', 'Alert|Email')"/>
                          </xsl:when>
                          <xsl:when test="method/data[name='notice']/text() = '2'">
                            <xsl:value-of select="gsa:i18n ('Attach list of resources', 'Alert|Email')"/>
                          </xsl:when>
                          <xsl:otherwise>
                            <xsl:value-of select="gsa:i18n ('Simple notice', 'Alert|Email')"/>
                          </xsl:otherwise>
                        </xsl:choose>
                      </td>
                    </tr>
                  </xsl:when>
                  <xsl:otherwise>
                    <tr>
                      <td width="45"></td>
                      <td><xsl:value-of select="gsa:i18n ('Content', 'Alert|Email')"/>:</td>
                      <td>
                        <xsl:choose>
                          <xsl:when test="method/data[name='notice']/text() = '0'">
                            <xsl:value-of select="gsa:i18n ('Include report', 'Alert|Email')"/>
                            <xsl:variable name="id"
                                          select="method/data[name='notice_report_format']/text()"/>
                            <xsl:text> </xsl:text>
                            <xsl:choose>
                              <xsl:when test="boolean (../../get_report_formats_response/report_format[@id=$id])">
                                <xsl:value-of select="../../get_report_formats_response/report_format[@id=$id]/name"/>
                              </xsl:when>
                              <xsl:otherwise>
                                <xsl:text>(</xsl:text>
                                <xsl:value-of select="gsa:i18n('Unavailable')"/>
                                <xsl:text>, </xsl:text>
                                <xsl:value-of select="gsa:i18n('UUID')"/>
                                <xsl:text>: </xsl:text>
                                <xsl:value-of select="$id"/>
                                <xsl:text>)</xsl:text>
                              </xsl:otherwise>
                            </xsl:choose>
                          </xsl:when>
                          <xsl:when test="method/data[name='notice']/text() = '2'">
                            <xsl:value-of select="gsa:i18n ('Attach report', 'Alert|Email')"/>
                            <xsl:variable name="id"
                                          select="method/data[name='notice_attach_format']/text()"/>
                            <xsl:text> </xsl:text>
                            <xsl:choose>
                              <xsl:when test="boolean (../../get_report_formats_response/report_format[@id=$id])">
                                <xsl:value-of select="../../get_report_formats_response/report_format[@id=$id]/name"/>
                              </xsl:when>
                              <xsl:otherwise>
                                <xsl:text>(</xsl:text>
                                <xsl:value-of select="gsa:i18n('Unavailable')"/>
                                <xsl:text>, </xsl:text>
                                <xsl:value-of select="gsa:i18n('UUID')"/>
                                <xsl:text>: </xsl:text>
                                <xsl:value-of select="$id"/>
                                <xsl:text>)</xsl:text>
                              </xsl:otherwise>
                            </xsl:choose>
                          </xsl:when>
                          <xsl:otherwise>
                            <xsl:value-of select="gsa:i18n ('Simple notice', 'Alert|Email')"/>
                          </xsl:otherwise>
                        </xsl:choose>
                      </td>
                    </tr>
                  </xsl:otherwise>
                </xsl:choose>
                <tr>
                  <td width="45"></td>
                  <td><xsl:value-of select="gsa:i18n ('Subject', 'Alert|Email')"/>:</td>
                  <td>
                    <xsl:choose>
                      <xsl:when test="string-length(method/data[name='subject']/text()) &gt; 0">
                        <xsl:value-of select="method/data[name='subject']/text()"/>
                      </xsl:when>
                    </xsl:choose>
                  </td>
                </tr>
                <tr>
                  <td width="45"></td>
                  <td><xsl:value-of select="gsa:i18n ('Message')"/>:</td>
                  <td>
                    <xsl:choose>
                      <xsl:when test="string-length(method/data[name='message']/text()) &gt; 0">
                        <xsl:choose>
                          <xsl:when test="contains (method/data[name='message']/text(), '&#10;')">
                            <xsl:value-of select="substring-before (method/data[name='message']/text(), '&#10;')"/>
                            <xsl:text>...</xsl:text>
                          </xsl:when>
                          <xsl:otherwise>
                            <xsl:value-of select="method/data[name='message']/text()"/>
                          </xsl:otherwise>
                        </xsl:choose>
                      </xsl:when>
                    </xsl:choose>
                  </td>
                </tr>
              </xsl:when>
              <xsl:when test="method/text()='HTTP Get'">
                <tr>
                  <td width="45"></td>
                  <td>URL:</td>
                  <td>
                    <xsl:choose>
                      <xsl:when test="string-length(method/data[name='URL']/text()) &gt; 0">
                        <xsl:value-of select="method/data[name='URL']/text()"/>
                      </xsl:when>
                    </xsl:choose>
                  </td>
                </tr>
              </xsl:when>
              <xsl:when test="method/text()='Sourcefire Connector'">
                <tr>
                  <td width="45"></td>
                  <td><xsl:value-of select="gsa:i18n ('Defense Center IP')"/>:</td>
                  <td>
                    <xsl:choose>
                      <xsl:when test="string-length(method/data[name='defense_center_ip']/text()) &gt; 0">
                        <xsl:value-of select="method/data[name='defense_center_ip']/text()"/>
                      </xsl:when>
                    </xsl:choose>
                  </td>
                </tr>
                <tr>
                  <td width="45"></td>
                  <td><xsl:value-of select="gsa:i18n ('Defense Center Port')"/>:</td>
                  <td>
                    <xsl:choose>
                      <xsl:when test="string-length(method/data[name='defense_center_port']/text()) &gt; 0">
                        <xsl:value-of select="method/data[name='defense_center_port']/text()"/>
                      </xsl:when>
                    </xsl:choose>
                  </td>
                </tr>
              </xsl:when>
              <xsl:when test="method/text()='verinice Connector'">
                <tr>
                  <td width="45"></td>
                  <td>URL:</td>
                  <td>
                    <xsl:choose>
                      <xsl:when test="string-length(method/data[name='verinice_server_url']/text()) &gt; 0">
                        <xsl:value-of select="method/data[name='verinice_server_url']/text()"/>
                      </xsl:when>
                    </xsl:choose>
                  </td>
                </tr>
                <tr>
                  <td width="45"></td>
                  <td><xsl:value-of select="gsa:i18n ('Credential')"/>:</td>
                  <td>
                    <xsl:variable name="credential" select="method/data[name='verinice_server_credential']/credential"/>
                    <xsl:choose>
                      <xsl:when test="string-length($credential) &gt; 0">
                        <a href="/omp?cmd=get_credential&amp;credential_id={$credential/@id}&amp;details=1&amp;token={/envelope/token}">
                          <xsl:value-of select="$credential/name"/>
                        </a>
                        <xsl:text> (</xsl:text>
                        <xsl:value-of select="gsa:i18n ('Username')"/>
                        <xsl:text>: </xsl:text>
                        <xsl:value-of select="$credential/login"/>
                        <xsl:text>)</xsl:text>
                      </xsl:when>
                      <xsl:otherwise>
                        <xsl:value-of select="gsa:i18n ('Credential unavailable')"/>
                        <xsl:text> (ID: </xsl:text>
                        <xsl:value-of select="method/data[name='verinice_server_credential']/text()"/>
                        <xsl:text>)</xsl:text>
                      </xsl:otherwise>
                    </xsl:choose>
                  </td>
                </tr>
                <tr>
                  <td width="45"></td>
                  <td><xsl:value-of select="gsa:i18n ('Report')"/>:</td>
                  <td>
                    <xsl:choose>
                      <xsl:when test="string-length(method/data[name='verinice_server_report_format']/text()) &gt; 0">
                        <xsl:variable name="id"
                                      select="method/data[name='verinice_server_report_format']/text()"/>
                        <xsl:value-of select="../../get_report_formats_response/report_format[@id=$id]/name"/>
                      </xsl:when>
                      <xsl:otherwise>
                        Verinice ISM
                      </xsl:otherwise>
                    </xsl:choose>
                  </td>
                </tr>
              </xsl:when>
              <xsl:when test="method/text()='Start Task'">
                <tr>
                  <td width="45"></td>
                  <td><xsl:value-of select="gsa:i18n ('Task')"/>:</td>
                  <td>
                    <xsl:variable name="id" select="method/data[name='start_task_task']/text()"/>
                    <xsl:variable name="name" select="../../get_tasks_response/task[@id = $id]/name"/>
                    <xsl:text> </xsl:text>
                    <xsl:choose>
                      <xsl:when test="string-length ($name) = 0">
                        <xsl:value-of select="gsa:i18n('Unavailable')"/> (<xsl:value-of select="gsa:i18n('ID')"/>: <xsl:value-of select="$id"/>)
                      </xsl:when>
                      <xsl:otherwise>
                        <a href="/omp?cmd=get_task&amp;task_id={$id}{gsa:token ()}">
                          <xsl:value-of select="$name"/>
                        </a>
                      </xsl:otherwise>
                    </xsl:choose>
                  </td>
                </tr>
              </xsl:when>
              <xsl:when test="method/text()='Send'">
                <tr>
                  <td width="45"></td>
                  <td><xsl:value-of select="gsa:i18n ('Host')"/>:</td>
                  <td><xsl:value-of select="method/data[name='send_host']/text()"/></td>
                </tr>
                <tr>
                  <td width="45"></td>
                  <td><xsl:value-of select="gsa:i18n ('Port')"/>:</td>
                  <td><xsl:value-of select="method/data[name='send_port']/text()"/></td>
                </tr>
                <xsl:choose>
                  <xsl:when test="event/text() = 'New SecInfo arrived'"/>
                  <xsl:when test="event/text() = 'Updated SecInfo arrived'"/>
                  <xsl:otherwise>
                    <tr>
                      <td width="45"></td>
                      <td><xsl:value-of select="gsa:i18n ('Report')"/>:</td>
                      <td>
                        <xsl:variable name="id"
                                      select="method/data[name='send_report_format']/text()"/>
                        <xsl:choose>
                          <xsl:when test="string-length($id) &gt; 0 and boolean (../../get_report_formats_response/report_format[@id=$id])">
                            <a href="/omp?cmd=get_report_format&amp;report_format_id={$id}&amp;filter={str:encode-uri (../filters/term, true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
                               title="{gsa:view_details_title ('Report Format', name)}">
                              <xsl:value-of select="../../get_report_formats_response/report_format[@id=$id]/name"/>
                            </a>
                          </xsl:when>
                          <xsl:when test="string-length($id) &gt; 0">
                            <xsl:value-of select="$id"/>
                          </xsl:when>
                          <xsl:otherwise>
                            XML
                          </xsl:otherwise>
                        </xsl:choose>
                      </td>
                    </tr>
                  </xsl:otherwise>
                </xsl:choose>
              </xsl:when>
              <xsl:when test="method/text()='SCP'">
                <tr>
                  <td width="45"></td>
                  <td><xsl:value-of select="gsa:i18n ('Credential')"/>:</td>
                  <td>
                    <xsl:variable name="credential" select="method/data[name='scp_credential']/credential"/>
                    <xsl:choose>
                      <xsl:when test="string-length($credential) &gt; 0">
                        <a href="/omp?cmd=get_credential&amp;credential_id={$credential/@id}&amp;details=1&amp;token={/envelope/token}">
                          <xsl:value-of select="$credential/name"/>
                        </a>
                        <xsl:text> (</xsl:text>
                        <xsl:value-of select="gsa:i18n ('Username')"/>
                        <xsl:text>: </xsl:text>
                        <xsl:value-of select="$credential/login"/>
                        <xsl:text>)</xsl:text>
                      </xsl:when>
                      <xsl:otherwise>
                        <xsl:value-of select="gsa:i18n ('Credential unavailable')"/>
                        <xsl:text> (ID: </xsl:text>
                        <xsl:value-of select="method/data[name='scp_credential']/text()"/>
                        <xsl:text>)</xsl:text>
                      </xsl:otherwise>
                    </xsl:choose>
                  </td>
                </tr>
                <tr>
                  <td width="45"></td>
                  <td><xsl:value-of select="gsa:i18n ('Host')"/>:</td>
                  <td>
                    <xsl:choose>
                      <xsl:when test="string-length(method/data[name='scp_host']/text()) &gt; 0">
                        <xsl:value-of select="method/data[name='scp_host']/text()"/>
                      </xsl:when>
                    </xsl:choose>
                  </td>
                </tr>
                <tr>
                  <td width="45"></td>
                  <td><xsl:value-of select="gsa:i18n ('Known Hosts')"/>:</td>
                  <td>
                    <xsl:if test="string-length (method/data[name='scp_known_hosts']/text ()) &gt; 0">
                      <xsl:choose>
                        <xsl:when test="string-length (method/data[name='scp_known_hosts']/text ()) &gt; 40">
                          <xsl:choose>
                            <xsl:when test="contains (substring (method/data[name='scp_known_hosts']/text (), 0, 40), '&#10;')">
                              <xsl:value-of select="substring-before (substring (method/data[name='scp_known_hosts']/text (), 0, 40), '&#10;')"/>
                              <xsl:text>...</xsl:text>
                            </xsl:when>
                            <xsl:otherwise>
                              <xsl:value-of select="substring (method/data[name='scp_known_hosts']/text (), 0, 40)"/>
                              <xsl:text>...</xsl:text>
                            </xsl:otherwise>
                          </xsl:choose>
                        </xsl:when>
                        <xsl:when test="contains (method/data[name='scp_known_hosts']/text (), '&#10;')">
                          <xsl:value-of select="substring-before (method/data[name='scp_known_hosts']/text (), '&#10;')"/>
                          <xsl:text>...</xsl:text>
                        </xsl:when>
                        <xsl:otherwise>
                          <xsl:value-of select="method/data[name='scp_known_hosts']/text()"/>
                        </xsl:otherwise>
                      </xsl:choose>
                    </xsl:if>
                  </td>
                </tr>
                <tr>
                  <td width="45"></td>
                  <td><xsl:value-of select="gsa:i18n ('Path')"/>:</td>
                  <td>
                    <xsl:choose>
                      <xsl:when test="string-length(method/data[name='scp_path']/text()) &gt; 0">
                        <xsl:value-of select="method/data[name='scp_path']/text()"/>
                      </xsl:when>
                    </xsl:choose>
                  </td>
                </tr>
                <tr>
                  <td width="45"></td>
                  <td><xsl:value-of select="gsa:i18n ('Report')"/>:</td>
                  <td>
                    <xsl:variable name="id"
                                  select="method/data[name='scp_report_format']/text()"/>
                    <xsl:choose>
                      <xsl:when test="string-length($id) &gt; 0 and boolean (../../get_report_formats_response/report_format[@id=$id])">
                        <a href="/omp?cmd=get_report_format&amp;report_format_id={$id}&amp;filter={str:encode-uri (../filters/term, true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
                           title="{gsa:view_details_title ('Report Format', name)}">
                          <xsl:value-of select="../../get_report_formats_response/report_format[@id=$id]/name"/>
                        </a>
                      </xsl:when>
                      <xsl:when test="string-length($id) &gt; 0">
                        <xsl:value-of select="$id"/>
                      </xsl:when>
                      <xsl:otherwise>
                        XML
                      </xsl:otherwise>
                    </xsl:choose>
                  </td>
                </tr>
              </xsl:when>
              <xsl:when test="method/text()='SNMP'">
                <tr>
                  <td width="45"></td>
                  <td>Community:</td>
                  <td>
                    <xsl:choose>
                      <xsl:when test="string-length(method/data[name='snmp_community']/text()) &gt; 0">
                        <xsl:value-of select="method/data[name='snmp_community']/text()"/>
                      </xsl:when>
                    </xsl:choose>
                  </td>
                </tr>
                <tr>
                  <td width="45"></td>
                  <td>Agent:</td>
                  <td>
                    <xsl:choose>
                      <xsl:when test="string-length(method/data[name='snmp_agent']/text()) &gt; 0">
                        <xsl:value-of select="method/data[name='snmp_agent']/text()"/>
                      </xsl:when>
                    </xsl:choose>
                  </td>
                </tr>
                <tr>
                  <td width="45"></td>
                  <td>Message:</td>
                  <td>
                    <xsl:choose>
                      <xsl:when test="string-length(method/data[name='snmp_message']/text()) &gt; 0">
                        <xsl:value-of select="method/data[name='snmp_message']/text()"/>
                      </xsl:when>
                    </xsl:choose>
                  </td>
                </tr>
              </xsl:when>
            </xsl:choose>
          </table>
        </td>
      </tr>
      <xsl:if test="event/text() = 'New SecInfo arrived' or event/text() = 'Updated SecInfo arrived'">
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Details URL')"/>:</td>
          <td>
            <xsl:value-of select="method/data[name='details_url']/text()"/>
          </td>
        </tr>
      </xsl:if>
      <xsl:if test="gsa:may-op ('get_filters') or string-length (filter/name) &gt; 0">
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Filter')"/>:</td>
          <td>
            <xsl:choose>
              <xsl:when test="boolean (filter/permissions) and count (filter/permissions/permission) = 0">
                <xsl:value-of select="filter/name"/> (<xsl:value-of select="gsa:i18n('Unavailable')"/>, <xsl:value-of select="gsa:i18n('UUID')"/>: <xsl:value-of select="filter/@id"/>)
              </xsl:when>
              <xsl:when test="string-length (filter/name) &gt; 0">
                <xsl:choose>
                  <xsl:when test="gsa:may-op ('get_filters')">
                    <a href="/omp?cmd=get_filter&amp;filter_id={filter/@id}&amp;token={/envelope/token}"
                       title="{gsa:i18n ('Details')}">
                      <xsl:value-of select="filter/name"/>
                    </a>
                  </xsl:when>
                  <xsl:otherwise>
                    <xsl:value-of select="filter/name"/>
                  </xsl:otherwise>
                </xsl:choose>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="gsa:i18n ('None', 'Alert|Filter')"/>
              </xsl:otherwise>
            </xsl:choose>
          </td>
        </tr>
      </xsl:if>
    </table>
  </div>

  <div class="section-header">
    <a href="#" class="toggle-action-icon icon icon-sm icon-action"
      data-target="#using-box" data-name="Tasks using this Alert"
      data-variable="using-box--collapsed">
        <img src="/img/fold.svg"/>
    </a>
    <h2>
      <a href="/omp?cmd=get_tasks&amp;token={/envelope/token}"
         title="{gsa:i18n ('Tasks')}">
        <img class="icon icon-sm" src="/img/task.svg" alt="Tasks"/>
      </a>
      <xsl:value-of select="gsa:i18n ('Tasks using this Alert')"/>
      <xsl:text> </xsl:text>
      <xsl:choose>
        <xsl:when test="count(tasks/task) != 0">
          (<xsl:value-of select="count(tasks/task)"/>)
        </xsl:when>
        <xsl:otherwise>
          (<xsl:value-of select="gsa:i18n ('none')"/>)
        </xsl:otherwise>
      </xsl:choose>
    </h2>
  </div>

  <div class="section-box" id="using-box">
    <table class="gbntable">
      <tr class="gbntablehead2">
        <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
      </tr>
      <xsl:for-each select="tasks/task">
        <tr class="{gsa:table-row-class(position())}">
          <xsl:choose>
            <xsl:when test="boolean (permissions) and count (permissions/permission) = 0">
              <td><xsl:value-of select="name"/> (<xsl:value-of select="gsa:i18n('Unavailable')"/>, <xsl:value-of select="gsa:i18n('UUID')"/>: <xsl:value-of select="@id"/>)</td>
            </xsl:when>
            <xsl:otherwise>
              <td>
                <a href="/omp?cmd=get_task&amp;task_id={@id}&amp;token={/envelope/token}" title="{gsa:i18n ('Details')}">
                  <xsl:value-of select="name"/>
                </a>
              </td>
            </xsl:otherwise>
          </xsl:choose>
        </tr>
      </xsl:for-each>
    </table>
  </div>

  <xsl:call-template name="user-tags-window">
    <xsl:with-param name="resource_type" select="'alert'"/>
  </xsl:call-template>

  <xsl:call-template name="resource-permissions-window">
    <xsl:with-param name="resource_type" select="'alert'"/>
    <xsl:with-param name="permissions" select="../../permissions/get_permissions_response"/>
    <xsl:with-param name="related">
      <xsl:if test="filter/@id != ''">
        <filter id="{filter/@id}"/>
      </xsl:if>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!--     GET_ALERT -->

<xsl:template match="get_alert">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="create_alert_response"/>
  <xsl:apply-templates select="commands_response/delete_alert_response"/>
  <xsl:apply-templates select="modify_alert_response"/>
  <xsl:apply-templates select="delete_tag_response"/>
  <xsl:apply-templates select="create_tag_response"/>
  <xsl:apply-templates select="modify_tag_response"/>
  <xsl:apply-templates select="get_alerts_response/alert" mode="details"/>
</xsl:template>

<!--     GET_ALERTS_RESPONSE -->

<xsl:template match="get_alerts">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="commands_response/delete_alert_response"/>
  <xsl:apply-templates select="delete_alert_response"/>
  <xsl:apply-templates select="create_alert_response"/>
  <xsl:apply-templates select="create_filter_response"/>
  <xsl:apply-templates select="test_alert_response"/>
  <xsl:apply-templates select="modify_alert_response"/>
  <!-- The for-each makes the get_alerts_response the current node. -->
  <xsl:for-each select="get_alerts_response | commands_response/get_alerts_response">
    <xsl:call-template name="html-alerts-table"/>
  </xsl:for-each>
</xsl:template>


<!-- BEGIN FILTERS MANAGEMENT -->

<xsl:template match="filters">
</xsl:template>

<xsl:template match="create_filter_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Create Filter</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="delete_filter_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Delete Filter</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="modify_filter_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Save Filter</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template name="filter_resource_types">
  <xsl:param name="select_type"/>
  <xsl:if test="$select_type = 'agent' or gsa:may-op ('get_agents')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'Agent'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Agent')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'alert' or gsa:may-op ('get_alerts')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'Alert'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Alert')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'asset' or gsa:may-op ('get_assets')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'Asset'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Asset')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'credential' or gsa:may-op ('get_credentials')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'Credential'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Credential')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'filter' or gsa:may-op ('get_filters')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'Filter'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Filter')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'group' or gsa:may-op ('get_groups')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'Group'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Group')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'note' or gsa:may-op ('get_notes')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'Note'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Note')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'override' or gsa:may-op ('get_overrides')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'Override'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Override')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'permission' or gsa:may-op ('get_permissions')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'Permission'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Permission')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'port_list' or gsa:may-op ('get_port_lists')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'Port List'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Port List')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'report' or gsa:may-op ('get_reports')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'Report'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Report')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'report_format' or gsa:may-op ('get_report_formats')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'Report Format'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Report Format')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'result' or gsa:may-op ('get_results')">
      <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'Result'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Result')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'role' or gsa:may-op ('get_roles')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'Role'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Role')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'schedule' or gsa:may-op ('get_schedules')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'Schedule'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Schedule')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'secinfo' or gsa:may-op ('get_info')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'SecInfo'"/>
      <xsl:with-param name="content" select="gsa:i18n ('SecInfo')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'config' or gsa:may-op ('get_configs')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'Config'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Scan Config')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'tag' or gsa:may-op ('get_tags')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'Tag'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Tag')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'target' or gsa:may-op ('get_targets')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'Target'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Target')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'task' or gsa:may-op ('get_tasks')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'Task'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Task')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'user' or gsa:may-op ('get_users')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'User'"/>
      <xsl:with-param name="content" select="gsa:i18n ('User')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
</xsl:template>

<xsl:template match="filter">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <xsl:call-template name="observers-icon">
        <xsl:with-param name="type" select="'Filter'"/>
      </xsl:call-template>
      <b>
        <a href="/omp?cmd=get_filter&amp;filter_id={@id}&amp;filter={str:encode-uri (../filters/term, true ())}&amp;token={/envelope/token}"
           title="{gsa:view_details_title ('Filter', name)}">
          <xsl:value-of select="name"/>
        </a>
      </b>
      <xsl:choose>
        <xsl:when test="comment != ''">
          <div class="comment">(<xsl:value-of select="comment"/>)</div>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </td>
    <td><xsl:value-of select="term"/></td>
    <td><xsl:value-of select="gsa:i18n (type, 'Type')"/></td>
    <xsl:choose>
      <xsl:when test="/envelope/params/bulk_select = 1">
        <td style="text-align:center">
          <label style="width:100%">
            <input name="bulk_selected:{@id}" type="checkbox" style="width:100%; height:100%" title="{gsa:i18n ('Select for bulk action')}"/>
          </label>
        </td>
      </xsl:when>
      <xsl:otherwise>
        <td class="table-actions">
          <xsl:call-template name="list-window-line-icons">
            <xsl:with-param name="cap-type" select="'Filter'"/>
            <xsl:with-param name="type" select="'filter'"/>
            <xsl:with-param name="id" select="@id"/>
          </xsl:call-template>
        </td>
      </xsl:otherwise>
    </xsl:choose>
  </tr>
</xsl:template>

<xsl:template match="filter" mode="trash">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <xsl:call-template name="observers-icon">
        <xsl:with-param name="type" select="'Filter'"/>
      </xsl:call-template>
      <b>
        <xsl:value-of select="name"/>
      </b>
      <xsl:choose>
        <xsl:when test="comment != ''">
          <div class="comment">(<xsl:value-of select="comment"/>)</div>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </td>
    <td><xsl:value-of select="term"/></td>
    <td><xsl:value-of select="gsa:i18n (type, 'Type')"/></td>
    <td class="table-actions">
      <xsl:call-template name="restore-icon">
        <xsl:with-param name="id" select="@id"/>
      </xsl:call-template>
      <xsl:choose>
        <xsl:when test="in_use='0'">
          <xsl:call-template name="trash-delete-icon">
            <xsl:with-param name="type" select="'filter'"/>
            <xsl:with-param name="id" select="@id"/>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <img src="/img/delete_inactive.svg"
               alt="{gsa:i18n ('Delete')}"
               title="{gsa:i18n ('Filter is still in use')}"
               class="icon icon-sm"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
  </tr>
</xsl:template>

<xsl:template match="filter" mode="details">
  <div class="toolbar">
    <xsl:call-template name="details-header-icons">
      <xsl:with-param name="cap-type" select="'Filter'"/>
      <xsl:with-param name="type" select="'filter'"/>
    </xsl:call-template>
  </div>

  <div class="section-header">
    <xsl:call-template name="minor-details"/>
    <h1>
      <a href="/omp?cmd=get_filters&amp;token={/envelope/token}"
         title="{gsa:i18n ('Filters')}">
        <img class="icon icon-lg" src="/img/filter.svg" style="margin-right:5px" alt="Filters"/>
      </a>
      <xsl:value-of select="gsa:i18n ('Filter')"/>:
      <xsl:value-of select="name"/>
      <xsl:text> </xsl:text>
    </h1>
  </div>

  <div class="section-box">
    <table>
      <tr>
        <td><b><xsl:value-of select="gsa:i18n ('Name')"/>:</b></td>
        <td><b><xsl:value-of select="name"/></b></td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Comment')"/>:</td>
        <td><xsl:value-of select="comment"/></td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Term')"/>:</td>
        <td><xsl:value-of select="term"/></td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Type')"/>:</td>
        <td><xsl:value-of select="gsa:i18n (type)"/></td>
      </tr>
    </table>
  </div>

  <xsl:if test="gsa:may-op ('get_alerts')">
    <div class="section-header">
      <a href="#" class="toggle-action-icon icon icon-sm icon-action"
        data-target="#using-box" data-name="Alerts using this Filter"
        data-variable="using-box--collapsed">
          <img src="/img/fold.svg"/>
      </a>
      <h2>
        <a href="/omp?cmd=get_alerts&amp;token={/envelope/token}"
           title="{gsa:i18n ('Alerts')}">
          <img class="icon-sm icon" src="/img/alert.svg" style="margin-right:5px" alt="Alerts"/>
        </a>
        <xsl:value-of select="gsa:i18n ('Alerts using this Filter')"/>
        <xsl:text> </xsl:text>
        <xsl:choose>
          <xsl:when test="count(alerts/alert) != 0">
            (<xsl:value-of select="count(alerts/alert)"/>)
          </xsl:when>
          <xsl:otherwise>
            (<xsl:value-of select="gsa:i18n ('none')"/>)
          </xsl:otherwise>
        </xsl:choose>
      </h2>
    </div>

    <div class="section-box" id="using-box">
      <table class="gbntable">
        <tr class="gbntablehead2">
          <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
        </tr>
        <xsl:for-each select="alerts/alert">
          <tr class="{gsa:table-row-class(position())}">
            <xsl:choose>
              <xsl:when test="boolean (permissions) and count (permissions/permission) = 0">
                <td><xsl:value-of select="name"/> (<xsl:value-of select="gsa:i18n('Unavailable')"/>, <xsl:value-of select="gsa:i18n('UUID')"/>: <xsl:value-of select="@id"/>)</td>
              </xsl:when>
              <xsl:otherwise>
                <td>
                  <a href="/omp?cmd=get_alert&amp;alert_id={@id}&amp;token={/envelope/token}" title="{gsa:i18n ('Details')}">
                    <xsl:value-of select="name"/>
                  </a>
                </td>
              </xsl:otherwise>
            </xsl:choose>
          </tr>
        </xsl:for-each>
      </table>
    </div>
  </xsl:if>

  <xsl:call-template name="user-tags-window">
    <xsl:with-param name="resource_type" select="'filter'"/>
  </xsl:call-template>

  <xsl:call-template name="resource-permissions-window">
    <xsl:with-param name="resource_type" select="'filter'"/>
    <xsl:with-param name="permissions" select="../../permissions/get_permissions_response"/>
    <xsl:with-param name="related">
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template name="html-filters-trash-table">
  <div>
    <table class="gbntable">
      <tr class="gbntablehead2">
        <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
        <td><xsl:value-of select="gsa:i18n ('Term')"/></td>
        <td><xsl:value-of select="gsa:i18n ('Type')"/></td>
        <td style="width: {$trash-actions-width}px"><xsl:value-of select="gsa:i18n ('Actions')"/></td>
      </tr>
      <xsl:apply-templates select="filter" mode="trash"/>
    </table>
  </div>
</xsl:template>

<xsl:template name="html-filters-table">
  <xsl:call-template name="list-window">
    <xsl:with-param name="type" select="'filter'"/>
    <xsl:with-param name="cap-type" select="'Filter'"/>
    <xsl:with-param name="resources-summary" select="filters"/>
    <xsl:with-param name="resources" select="filter"/>
    <xsl:with-param name="count" select="count (filter)"/>
    <xsl:with-param name="filtered-count" select="filter_count/filtered"/>
    <xsl:with-param name="full-count" select="filter_count/text ()"/>
    <xsl:with-param name="columns" xmlns="">
      <column>
        <name><xsl:value-of select="gsa:i18n('Name')"/></name>
        <field>name</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Term')"/></name>
        <field>term</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Type')"/></name>
        <field>type</field>
      </column>
    </xsl:with-param>
    <xsl:with-param name="icon-count" select="4"/>
  </xsl:call-template>
</xsl:template>

<!-- NEW_FILTER -->

<xsl:template name="html-create-filter-form">
  <div class="gb_window">
    <div class="gb_window_part_left"></div>
    <div class="gb_window_part_right"></div>
    <div class="gb_window_part_center"><xsl:value-of select="gsa:i18n ('New Filter')"/>
      <a href="/help/new_filter.html?token={/envelope/token}"
        class="icon icon-sm"
        title="{concat(gsa:i18n('Help'),': ',gsa:i18n('New Filter'))}">
        <img src="/img/help.svg"/>
      </a>
      <a href="/omp?cmd=get_filters&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
        title="{gsa:i18n ('Filters')}"
        class="icon icon-sm">
        <img src="/img/list.svg" alt="{gsa:i18n ('Filters')}"/>
      </a>
    </div>
    <div class="gb_window_part_content">
      <form action="/omp" method="post" enctype="multipart/form-data">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="create_filter"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden" name="next" value="get_filter"/>
        <input type="hidden" name="filter_id" value="{/envelope/params/filter_id}"/>
        <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
        <table class="table-form">
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Name')"/>
            </td>
            <td>
              <input type="text" name="name" value="unnamed" size="30"
                     maxlength="80"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Comment')"/></td>
            <td>
              <input type="text" name="comment" size="30" maxlength="400"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Term')"/></td>
            <td>
              <input type="text" name="term" size="30" maxlength="1000"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Type')"/></td>
            <td>
              <select name="optional_resource_type">
                <xsl:call-template name="filter_resource_types"/>
              </select>
            </td>
          </tr>
          <tr>
            <td>
              <input type="submit" name="submit" value="{gsa:i18n ('Create Filter')}"/>
            </td>
          </tr>
        </table>
      </form>
    </div>
  </div>
</xsl:template>

<xsl:template match="new_filter">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="commands_response/delete_filter_response"/>
  <xsl:apply-templates select="create_filter_response"/>
  <xsl:call-template name="html-create-filter-form"/>
</xsl:template>

<!--     EDIT_FILTER -->

<xsl:template name="html-edit-filter-form">
  <div class="gb_window">
    <div class="gb_window_part_left"></div>
    <div class="gb_window_part_right"></div>
    <div class="gb_window_part_center"><xsl:value-of select="gsa:i18n ('Edit Filter')"/>
      <xsl:call-template name="edit-header-icons">
        <xsl:with-param name="cap-type" select="'Filter'"/>
        <xsl:with-param name="type" select="'filter'"/>
        <xsl:with-param name="id"
                        select="commands_response/get_filters_response/filter/@id"/>
      </xsl:call-template>
    </div>
    <div class="gb_window_part_content">
      <form action="" method="post" enctype="multipart/form-data">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="save_filter"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden"
               name="filter_id"
               value="{commands_response/get_filters_response/filter/@id}"/>
        <input type="hidden" name="next" value="{/envelope/params/next}"/>
        <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
        <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
        <table class="table-form">
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
            <td>
              <input type="text"
                     name="name"
                     value="{commands_response/get_filters_response/filter/name}"
                     size="30"
                     maxlength="80"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Comment')"/></td>
            <td>
              <input type="text" name="comment" size="30" maxlength="400"
                     value="{commands_response/get_filters_response/filter/comment}"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Term')"/></td>
            <td>
              <input type="text" name="term"
                     value="{commands_response/get_filters_response/filter/term}"
                     size="50"
                     maxlength="1000"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Type')"/></td>
            <td>
              <select name="optional_resource_type">
                <xsl:variable name="type">
                  <xsl:value-of select="commands_response/get_filters_response/filter/type"/>
                </xsl:variable>
                <option value="">--</option>
                <xsl:call-template name="filter_resource_types">
                  <xsl:with-param name="select_type" select="$type"/>
                </xsl:call-template>
              </select>
            </td>
          </tr>
          <tr>
            <td>
              <input type="submit" name="submit" value="{gsa:i18n ('Save Filter')}"/>
            </td>
          </tr>
        </table>
      </form>
    </div>
  </div>
</xsl:template>

<xsl:template match="edit_filter">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="modify_filter_response"/>
  <xsl:call-template name="html-edit-filter-form"/>
</xsl:template>

<!--     GET_FILTER -->

<xsl:template match="get_filter">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="delete_tag_response"/>
  <xsl:apply-templates select="create_tag_response"/>
  <xsl:apply-templates select="modify_tag_response"/>
  <xsl:apply-templates select="get_filters_response/filter" mode="details"/>
</xsl:template>

<!--     GET_FILTERS -->

<xsl:template match="get_filters">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="delete_filter_response"/>
  <xsl:apply-templates select="create_filter_response"/>
  <!-- The for-each makes the get_filters_response the current node. -->
  <xsl:for-each select="get_filters_response | commands_response/get_filters_response">
    <xsl:choose>
      <xsl:when test="substring(@status, 1, 1) = '4' or substring(@status, 1, 1) = '5'">
        <xsl:call-template name="command_result_dialog">
          <xsl:with-param name="operation">
            Get Filters
          </xsl:with-param>
          <xsl:with-param name="status">
            <xsl:value-of select="@status"/>
          </xsl:with-param>
          <xsl:with-param name="msg">
            <xsl:value-of select="@status_text"/>
          </xsl:with-param>
        </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
        <xsl:call-template name="html-filters-table"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:for-each>
</xsl:template>


<!-- BEGIN TAGS MANAGEMENT -->

<xsl:template name="tagged_resource_link">
  <xsl:param name="resource_type"/>
  <xsl:param name="resource_id"/>
  <xsl:param name="resource_name"/>
  <xsl:param name="resource_location" select="0"/>
  <xsl:param name="resource_permissions"/>
  <xsl:param name="token"/>
  <xsl:choose>
    <xsl:when test="boolean ($resource_permissions) and count ($resource_permissions/permission) = 0">
      <xsl:value-of select="$resource_name"/>
    </xsl:when>
    <xsl:when test="$resource_type='cve' or $resource_type='cpe' or $resource_type='ovaldef' or $resource_type='cert_bund_adv' or $resource_type='dfn_cert_adv'">
      <xsl:choose>
        <xsl:when test="gsa:may-op ('get_info') and $resource_location = '0'">
          <a href="/omp?cmd=get_info&amp;info_type={$resource_type}&amp;info_id={$resource_id}&amp;details=1&amp;token={$token}">
            <xsl:value-of select="$resource_name"/>
          </a>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="$resource_name"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>
    <xsl:when test="$resource_type='host' or $resource_type='os'">
      <a href="/omp?cmd=get_asset&amp;asset_type={$resource_type}&amp;asset_id={$resource_id}&amp;details=1&amp;token={$token}">
        <xsl:value-of select="$resource_name"/>
      </a>
    </xsl:when>
    <xsl:when test="$resource_type='nvt' and $resource_location = '0'">
      <a href="/omp?cmd=get_info&amp;info_type=nvt&amp;info_id={$resource_id}&amp;details=1&amp;token={$token}">
        <xsl:value-of select="$resource_name"/>
      </a>
    </xsl:when>
    <xsl:when test="$resource_type='nvt'">
      <xsl:value-of select="$resource_name"/>
    </xsl:when>
    <xsl:when test="gsa:may-op (concat ('get_', $resource_type, 's')) and $resource_location = '0'">
      <a href="/omp?cmd=get_{$resource_type}&amp;{$resource_type}_id={$resource_id}&amp;details=1&amp;token={$token}">
        <xsl:value-of select="$resource_name"/>
      </a>
    </xsl:when>
    <xsl:otherwise>
      <xsl:value-of select="$resource_name"/>
    </xsl:otherwise>
  </xsl:choose>
  <xsl:choose>
    <xsl:when test="$resource_location != 0 and boolean ($resource_permissions) and count ($resource_permissions/permission) = 0">
      <xsl:text> (</xsl:text>
      <xsl:value-of select="gsa:i18n ('in ', 'Trashcan')"/>
      <xsl:value-of select="gsa:i18n ('trashcan')"/>
      <xsl:text>)</xsl:text>
    </xsl:when>
    <xsl:when test="$resource_location != 0">
      <xsl:text> (</xsl:text>
      <xsl:value-of select="gsa:i18n ('in ', 'Trashcan')"/><a href="/omp?cmd=get_trash&amp;token={/envelope/token}"><xsl:value-of select="gsa:i18n ('trashcan')"/></a>
      <xsl:text>)</xsl:text>
    </xsl:when>
    <xsl:otherwise/>
  </xsl:choose>
</xsl:template>

<xsl:template match="get_tags">
  <xsl:apply-templates select="gsad_msg"/>

  <xsl:apply-templates select="delete_tag_response"/>
  <xsl:apply-templates select="create_filter_response"/>
  <xsl:apply-templates select="create_tag_response"/>
  <xsl:apply-templates select="modify_tag_response"/>

  <!-- The for-each makes the get_targets_response the current node. -->
  <xsl:for-each select="get_tags_response | commands_response/get_tags_response">
    <xsl:choose>
      <xsl:when test="substring(@status, 1, 1) = '4' or substring(@status, 1, 1) = '5'">
        <xsl:call-template name="command_result_dialog">
          <xsl:with-param name="operation">
            Get Tags
          </xsl:with-param>
          <xsl:with-param name="status">
            <xsl:value-of select="@status"/>
          </xsl:with-param>
          <xsl:with-param name="msg">
            <xsl:value-of select="@status_text"/>
          </xsl:with-param>
        </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
        <xsl:call-template name="html-tags-table"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:for-each>
</xsl:template>

<xsl:template name="html-tags-table">
  <xsl:call-template name="list-window">
    <xsl:with-param name="type" select="'tag'"/>
    <xsl:with-param name="cap-type" select="'Tag'"/>
    <xsl:with-param name="resources-summary" select="tags"/>
    <xsl:with-param name="resources" select="tag"/>
    <xsl:with-param name="count" select="count (tag)"/>
    <xsl:with-param name="filtered-count" select="tag_count/filtered"/>
    <xsl:with-param name="full-count" select="tag_count/text ()"/>
    <xsl:with-param name="columns" xmlns="">
      <column>
        <name><xsl:value-of select="gsa:i18n('Name')"/></name>
        <field>name</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Value')"/></name>
        <field>value</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Active', 'Tag')"/></name>
        <field>active</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Resource Type')"/></name>
        <field>resource_type</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Resource Name')"/></name>
        <field>resource_name</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Modified', 'Date')"/></name>
        <field>modified</field>
        <sort-reverse/>
      </column>
    </xsl:with-param>
    <xsl:with-param name="icon-count" select="5"/>
  </xsl:call-template>
</xsl:template>

<xsl:template match="tag">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <xsl:call-template name="observers-icon">
        <xsl:with-param name="type" select="'Tag'"/>
      </xsl:call-template>
      <a href="/omp?cmd=get_tag&amp;tag_id={@id}&amp;filter={str:encode-uri (../filters/term, true ())}&amp;first={../tags/@start}&amp;max={../tags/@max}&amp;token={/envelope/token}"
          title="{gsa:view_details_title ('Tag', name)}">
        <xsl:if test="orphan = 1"><b><xsl:value-of select="gsa:i18n ('Orphan', 'Tag')"/></b><br/></xsl:if>
        <xsl:value-of select="name"/>
      </a>
      <xsl:choose>
        <xsl:when test="comment != ''">
          <div class="comment">(<xsl:value-of select="comment"/>)</div>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:value-of select="value"/>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="active=0">
          <xsl:value-of select="gsa:i18n ('No')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="gsa:i18n ('Yes')"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:value-of select="gsa:i18n (gsa:type-name (resource/type), gsa:type-name (resource/type))"/>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="orphan='0'">
          <xsl:call-template name="tagged_resource_link">
            <xsl:with-param name="resource_type" select="resource/type"/>
            <xsl:with-param name="resource_id" select="resource/@id"/>
            <xsl:with-param name="resource_name" select="resource/name"/>
            <xsl:with-param name="resource_location" select="resource/trash"/>
            <xsl:with-param name="resource_permissions" select="resource/permissions"/>
            <xsl:with-param name="token" select="/envelope/token"/>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="gsa:i18n ('N/A')"/>
          <xsl:text> </xsl:text>
          <xsl:choose>
            <xsl:when test="resource/@id != ''">
              <i>(<xsl:value-of select="resource/@id"/>)</i>
            </xsl:when>
            <xsl:otherwise/>
          </xsl:choose>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:value-of select="gsa:date (modification_time)"/>
    </td>
    <xsl:choose>
      <xsl:when test="/envelope/params/bulk_select = 1">
        <td style="text-align:center">
          <label style="width:100%">
            <input name="bulk_selected:{@id}" type="checkbox" style="width:100%; height:100%" title="{gsa:i18n ('Select for bulk action')}"/>
          </label>
        </td>
      </xsl:when>
      <xsl:otherwise>
        <td class="table-actions">
          <xsl:call-template name="toggle-tag-icon">
            <xsl:with-param name="id" select="@id"/>
            <xsl:with-param name="enable" select="number(active=0)"/>
            <xsl:with-param name="params">
              <input type="hidden" name="next" value="get_tags"/>
              <input type="hidden" name="tag_id" value="{@id}"/>
              <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
              <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
            </xsl:with-param>
            <xsl:with-param name="fragment" select="'#user_tags'"/>
          </xsl:call-template>
          <xsl:call-template name="list-window-line-icons">
            <xsl:with-param name="cap-type" select="'Tag'"/>
            <xsl:with-param name="type" select="'tag'"/>
            <xsl:with-param name="id" select="@id"/>
          </xsl:call-template>
        </td>
      </xsl:otherwise>
    </xsl:choose>
  </tr>
</xsl:template>

<xsl:template match="get_tag">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="commands_response/delete_tag_response"/>
  <xsl:apply-templates select="get_tags_response/tag" mode="details"/>
  <xsl:apply-templates select="delete_tag_response"/>
  <xsl:apply-templates select="modify_tag_response"/>
  <xsl:apply-templates select="create_tag_response"/>
</xsl:template>

<xsl:template match="tag" mode="details">
  <div class="toolbar">
    <xsl:call-template name="details-header-icons">
      <xsl:with-param name="cap-type" select="'Tag'"/>
      <xsl:with-param name="type" select="'tag'"/>
    </xsl:call-template>
  </div>

  <div class="section-header">
    <xsl:call-template name="minor-details"/>
    <h1>
      <a href="/omp?cmd=get_tags&amp;token={/envelope/token}"
         title="{gsa:i18n ('Tags')}">
        <img class="icon icon-lg" src="/img/tag.svg" alt="Tags"/>
      </a>
      <xsl:value-of select="gsa:i18n ('Tag')"/>:
      <xsl:value-of select="name"/>
      <xsl:text> </xsl:text>
    </h1>
  </div>

  <div class="section-box">
    <table>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Comment')"/>:</td>
        <td><xsl:value-of select="comment"/></td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Value')"/>:</td>
        <td><xsl:value-of select="value"/></td>
      </tr>
      <xsl:choose>
        <xsl:when test="resource/name != '' and orphan='0'">
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Attached to Resource', 'Tag')"/>:</td>
            <td>
              <xsl:choose>
                <xsl:when test="boolean (resource/permissions) and count (resource/permissions/permission) = 0">
                  <xsl:value-of select="gsa:i18n('Unavailable')"/>
                  <xsl:text> (</xsl:text>
                  <xsl:value-of select="gsa:i18n ('Name')"/>
                  <xsl:text>: </xsl:text>
                  <xsl:value-of select="resource/name"/>
                  <xsl:text>, </xsl:text>
                  <xsl:value-of select="gsa:i18n ('ID')"/>: <xsl:value-of select="resource/@id"/>
                  <xsl:text>)</xsl:text>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:call-template name="tagged_resource_link">
                    <xsl:with-param name="resource_type" select="resource/type"/>
                    <xsl:with-param name="resource_id" select="resource/@id"/>
                    <xsl:with-param name="resource_name" select="resource/name"/>
                    <xsl:with-param name="resource_location" select="resource/trash"/>
                    <xsl:with-param name="resource_permissions" select="resource/permissions"/>
                    <xsl:with-param name="token" select="/envelope/token"/>
                  </xsl:call-template>
                </xsl:otherwise>
              </xsl:choose>
            </td>
          </tr>
        </xsl:when>
      </xsl:choose>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Resource Type')"/>:</td>
        <td><xsl:value-of select="gsa:i18n (gsa:type-name (resource/type), gsa:type-name (resource/type))"/></td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Resource ID')"/>:</td>
        <td><xsl:value-of select="resource/@id"/></td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Active', 'Tag')"/>:</td>
        <td>
          <xsl:choose>
            <xsl:when test="active = 0"><xsl:value-of select="gsa:i18n ('No')"/></xsl:when>
            <xsl:otherwise><xsl:value-of select="gsa:i18n ('Yes')"/></xsl:otherwise>
          </xsl:choose>
        </td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Orphan', 'Tag')"/>:</td>
        <td>
          <xsl:choose>
            <xsl:when test="orphan = 0"><xsl:value-of select="gsa:i18n ('No')"/></xsl:when>
            <xsl:otherwise><xsl:value-of select="gsa:i18n ('Yes')"/></xsl:otherwise>
          </xsl:choose>
        </td>
      </tr>
    </table>
  </div>

  <xsl:call-template name="resource-permissions-window">
    <xsl:with-param name="resource_type" select="'tag'"/>
    <xsl:with-param name="permissions" select="../../permissions/get_permissions_response"/>
    <xsl:with-param name="related">
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="tag" mode="trash">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <xsl:call-template name="observers-icon">
        <xsl:with-param name="type" select="'Tag'"/>
      </xsl:call-template>
      <b><xsl:value-of select="name"/></b>
      <xsl:choose>
        <xsl:when test="comment != ''">
          <div class="comment">(<xsl:value-of select="comment"/>)</div>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </td>
    <td><xsl:value-of select="value"/></td>
    <td><xsl:value-of select="gsa:i18n (gsa:type-name (resource/type), gsa:type-name (resource/type))"/></td>
    <td>
      <xsl:choose>
        <xsl:when test="orphan='0'">
          <xsl:call-template name="tagged_resource_link">
            <xsl:with-param name="resource_type" select="resource/type"/>
            <xsl:with-param name="resource_id" select="resource/@id"/>
            <xsl:with-param name="resource_name" select="resource/name"/>
            <xsl:with-param name="resource_location" select="resource/trash"/>
            <xsl:with-param name="token" select="/envelope/token"/>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="gsa:i18n ('N/A')"/>
          <xsl:text> </xsl:text>
          <xsl:choose>
            <xsl:when test="resource/@id != ''">
              <i>(<xsl:value-of select="resource/@id"/>)</i>
            </xsl:when>
            <xsl:otherwise/>
          </xsl:choose>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td class="table-actions">
      <xsl:call-template name="restore-icon">
        <xsl:with-param name="id" select="@id"/>
      </xsl:call-template>
      <xsl:choose>
        <xsl:when test="in_use='0'">
          <xsl:call-template name="trash-delete-icon">
            <xsl:with-param name="type" select="'tag'"/>
            <xsl:with-param name="id" select="@id"/>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <img src="/img/delete_inactive.svg"
               alt="{gsa:i18n ('Delete')}"
               title="{gsa:i18n ('Tag is still in use')}"
               class="icon icon-sm"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
  </tr>
</xsl:template>

<xsl:template name="tag_resource_types">
  <xsl:param name="select_type"/>
  <xsl:if test="$select_type = 'agent' or gsa:may-op ('get_agents')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'agent'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Agent')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'alert' or gsa:may-op ('get_alerts')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'alert'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Alert')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'host' or gsa:may-op ('get_assets')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'host'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Host')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'os' or gsa:may-op ('get_assets')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'os'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Operating System')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'cpe' or gsa:may-op ('get_info')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'cpe'"/>
      <xsl:with-param name="content" select="gsa:i18n ('CPE')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'credential' or gsa:may-op ('get_credentials')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'credential'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Credential')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'cve' or gsa:may-op ('get_info')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'cve'"/>
      <xsl:with-param name="content" select="gsa:i18n ('CVE')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'cert_bund_adv' or gsa:may-op ('get_info')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'cert_bund_adv'"/>
      <xsl:with-param name="content" select="gsa:i18n ('CERT-Bund Advisory')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'dfn_cert_adv' or gsa:may-op ('get_info')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'dfn_cert_adv'"/>
      <xsl:with-param name="content" select="gsa:i18n ('DFN-CERT Advisory')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'filter' or gsa:may-op ('get_filters')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'filter'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Filter')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'group' or gsa:may-op ('get_groups')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'group'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Group')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'note' or gsa:may-op ('get_notes')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'note'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Note')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'nvt' or gsa:may-op ('get_info')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'nvt'"/>
      <xsl:with-param name="content" select="gsa:i18n ('NVT')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'ovaldef' or gsa:may-op ('get_info')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'ovaldef'"/>
      <xsl:with-param name="content" select="gsa:i18n ('OVAL Definition')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'override' or gsa:may-op ('get_overrides')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'override'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Override')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'permission' or gsa:may-op ('get_permissions')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'permission'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Permission')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'port_list' or gsa:may-op ('get_port_lists')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'port_list'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Port List')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'report' or gsa:may-op ('get_reports')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'report'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Report')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'report_format' or gsa:may-op ('get_report_formats')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'report_format'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Report Format')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'result' or gsa:may-op ('get_results')">
      <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'result'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Result')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'role' or gsa:may-op ('get_roles')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'role'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Role')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'config' or gsa:may-op ('get_configs')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'config'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Scan Config')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'schedule' or gsa:may-op ('get_schedules')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'schedule'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Schedule')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'target' or gsa:may-op ('get_targets')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'target'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Target')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'task' or gsa:may-op ('get_tasks')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'task'"/>
      <xsl:with-param name="content" select="gsa:i18n ('Task')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:if test="$select_type = 'user' or gsa:may-op ('get_users')">
    <xsl:call-template name="opt">
      <xsl:with-param name="value" select="'user'"/>
      <xsl:with-param name="content" select="gsa:i18n ('User')"/>
      <xsl:with-param name="select-value" select="$select_type"/>
    </xsl:call-template>
  </xsl:if>
</xsl:template>

<xsl:template match="new_tag">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="create_tag_response"/>
  <xsl:apply-templates select="commands_response/delete_tag_response"/>
  <xsl:call-template name="html-create-tag-form"/>
  <!-- <xsl:call-template name="html-import-tag-form"/> -->
</xsl:template>

<xsl:template name="html-create-tag-form">
  <div class="gb_window">
    <div class="gb_window_part_left"></div>
    <div class="gb_window_part_right"></div>
    <div class="gb_window_part_center"><xsl:value-of select="gsa:i18n ('New Tag')"/>
      <a href="/help/new_tag.html?token={/envelope/token}"
        class="icon icon-sm"
        title="{concat(gsa:i18n('Help'),': ',gsa:i18n('New Tag'))}">
        <img src="/img/help.svg"/>
      </a>
      <a href="/omp?cmd=get_tags&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
        class="icon icon-sm"
        title="{gsa:i18n ('Tags')}">
        <img src="/img/list.svg" alt="{gsa:i18n ('Tags')}"/>
      </a>
    </div>
    <div class="gb_window_part_content">
      <form action="/omp" method="post" enctype="multipart/form-data">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="create_tag"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden" name="next" value="{/envelope/params/next}"/>
        <xsl:if test="/envelope/params/next_type!=''">
          <xsl:if test="/envelope/params/next_id != ''">
            <xsl:choose>
              <xsl:when test="/envelope/params/next_type='nvt'">
                <input type="hidden"
                       name="oid"
                       value="{/envelope/params/next_id}"/>
              </xsl:when>
              <xsl:otherwise>
                <input type="hidden"
                       name="{/envelope/params/next_type}_id"
                       value="{/envelope/params/next_id}"/>
              </xsl:otherwise>
            </xsl:choose>
            <xsl:if test="/envelope/params/next_type='info'">
              <input type="hidden"
                    name="details"
                    value="1"/>
            </xsl:if>
          </xsl:if>
          <xsl:if test="/envelope/params/next_subtype != ''">
            <input type="hidden"
                   name="{/envelope/params/next_type}_type"
                   value="{/envelope/params/next_subtype}"/>
          </xsl:if>
        </xsl:if>
        <xsl:if test="/envelope/params/report_section">
          <input type="hidden" name="report_section" value="{/envelope/params/report_section}"/>
        </xsl:if>
        <input type="hidden" name="filter" value="{filters/term}"/>
        <input type="hidden" name="first" value="{/envelope/params/start}"/>
        <input type="hidden" name="max" value="{/envelope/params/max}"/>
        <table class="table-form">
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Name')"/>
            </td>
            <td>
              <input type="text" name="tag_name" value="{tag_name}" size="30"
                     maxlength="80"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Comment')"/></td>
            <td>
              <input type="text" name="comment" value="{comment}" size="30"
                     maxlength="400"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Value')"/></td>
            <td>
              <input type="text" name="tag_value" value="{tag_value}" size="30"
                     maxlength="200"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Resource Type')"/></td>
            <td>
              <select name="resource_type">
                <xsl:call-template name="tag_resource_types">
                  <xsl:with-param name="select_type">
                    <xsl:value-of select="resource_type"/>
                  </xsl:with-param>
                </xsl:call-template>
              </select>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Resource ID')"/></td>
            <td>
              <input type="text" name="resource_id" value="{resource_id}" size="30"
                         maxlength="80"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Active', 'Tag')"/></td>
            <td>
              <label>
                <xsl:choose>
                  <xsl:when test="active!=0">
                    <input type="radio" name="active" value="1" checked="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="radio" name="active" value="1"/>
                  </xsl:otherwise>
                </xsl:choose>
                <xsl:value-of select="gsa:i18n ('Yes')"/>
              </label>
              <label>
                <xsl:choose>
                  <xsl:when test="active=0">
                    <input type="radio" name="active" value="0" checked="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="radio" name="active" value="0"/>
                  </xsl:otherwise>
                </xsl:choose>
                <xsl:value-of select="gsa:i18n ('No')"/>
              </label>
            </td>
          </tr>
          <tr>
            <td>
              <input type="submit" name="submit" value="{gsa:i18n ('Create Tag')}"/>
            </td>
          </tr>
        </table>
      </form>
    </div>
  </div>
</xsl:template>

<xsl:template match="create_tag_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Create Tag</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="edit_tag">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="modify_tag_response"/>
  <xsl:call-template name="html-edit-tag-form"/>
</xsl:template>

<xsl:template name="html-edit-tag-form">
  <div class="gb_window">
    <div class="gb_window_part_left"></div>
    <div class="gb_window_part_right"></div>
    <div class="gb_window_part_center"><xsl:value-of select="gsa:i18n ('Edit Tag')"/>
      <xsl:call-template name="edit-header-icons">
        <xsl:with-param name="cap-type" select="'Tag'"/>
        <xsl:with-param name="type" select="'tag'"/>
        <xsl:with-param name="id"
                        select="get_tags_response/tag/@id"/>
      </xsl:call-template>
    </div>
    <div class="gb_window_part_content">
      <form action="" method="post" enctype="multipart/form-data">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="save_tag"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden"
               name="tag_id"
               value="{get_tags_response/tag/@id}"/>
        <xsl:if test="/envelope/params/next_type!=''">
          <xsl:if test="/envelope/params/next_id != ''">
            <xsl:choose>
              <xsl:when test="/envelope/params/next_type='nvt'">
                <input type="hidden"
                       name="oid"
                       value="{/envelope/params/next_id}"/>
              </xsl:when>
              <xsl:otherwise>
                <input type="hidden"
                       name="{/envelope/params/next_type}_id"
                       value="{/envelope/params/next_id}"/>
              </xsl:otherwise>
            </xsl:choose>
            <xsl:if test="/envelope/params/next_type='info'">
              <input type="hidden"
                    name="details"
                    value="1"/>
            </xsl:if>
          </xsl:if>
          <xsl:if test="/envelope/params/next_subtype != ''">
            <input type="hidden"
                   name="{/envelope/params/next_type}_type"
                   value="{/envelope/params/next_subtype}"/>
          </xsl:if>
        </xsl:if>
        <xsl:if test="/envelope/params/report_section != ''">
          <input type="hidden"
                  name="report_section"
                  value="{/envelope/params/report_section}"/>
        </xsl:if>
        <input type="hidden" name="next" value="{/envelope/params/next}"/>
        <input type="hidden" name="sort_field" value="{sort_field}"/>
        <input type="hidden" name="sort_order" value="{sort_order}"/>
        <input type="hidden" name="filter" value="{filters/term}"/>
        <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
        <input type="hidden" name="first" value="{limits/@start}"/>
        <input type="hidden" name="max" value="{limits/@max}"/>
        <table class="table-form">
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Name')"/>
            </td>
            <td>
              <input type="text" name="tag_name" value="{get_tags_response/tag/name}" size="30"
                     maxlength="80"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Comment')"/></td>
            <td>
              <input type="text" name="comment" value="{get_tags_response/tag/comment}" size="30"
                     maxlength="400"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Value')"/></td>
            <td>
              <input type="text" name="tag_value" value="{get_tags_response/tag/value}" size="30"
                     maxlength="200"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Resource Type')"/></td>
            <td>
              <select name="resource_type">
                <xsl:call-template name="tag_resource_types">
                  <xsl:with-param name="select_type">
                    <xsl:value-of select="get_tags_response/tag/resource/type"/>
                  </xsl:with-param>
                </xsl:call-template>
              </select>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Resource ID')"/></td>
            <td>
              <input type="text" name="resource_id" value="{get_tags_response/tag/resource/@id}" size="30"
                         maxlength="80"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Active', 'Tag')"/></td>
            <td>
              <label>
                <xsl:choose>
                  <xsl:when test="get_tags_response/tag/active!=0">
                    <input type="radio" name="active" value="1" checked="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="radio" name="active" value="1"/>
                  </xsl:otherwise>
                </xsl:choose>
                <xsl:value-of select="gsa:i18n ('Yes')"/>
              </label>
              <label>
                <xsl:choose>
                  <xsl:when test="get_tags_response/tag/active=0">
                    <input type="radio" name="active" value="0" checked="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="radio" name="active" value="0"/>
                  </xsl:otherwise>
                </xsl:choose>
                <xsl:value-of select="gsa:i18n ('No')"/>
              </label>
            </td>
          </tr>
          <tr>
            <td>
              <input type="submit" name="submit" value="{gsa:i18n ('Save Tag')}"/>
            </td>
          </tr>
        </table>
      </form>
    </div>
  </div>
</xsl:template>

<xsl:template match="modify_tag_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Save Tag</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="delete_tag_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Delete Tag</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="get_tags_response"/>


<!-- BEGIN TARGETS MANAGEMENT -->

<xsl:template match="modify_target_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Save Target</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="port_list" mode="select">
  <xsl:param name="select_id" select="''"/>
  <xsl:choose>
    <xsl:when test="@id = $select_id">
      <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
    </xsl:when>
    <xsl:otherwise>
      <option value="{@id}"><xsl:value-of select="name"/></option>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="html-create-target-form">
  <xsl:param name="credentials"></xsl:param>
  <xsl:param name="port-lists"></xsl:param>
  <div class="gb_window">
    <div class="gb_window_part_left"></div>
    <div class="gb_window_part_right"></div>
    <div class="gb_window_part_center"><xsl:value-of select="gsa:i18n ('New Target')"/>
      <a href="/help/new_target.html?token={/envelope/token}"
        class="icon icon-sm"
        title="{concat(gsa:i18n('Help'),': ',gsa:i18n('New Target'))}">
        <img src="/img/help.svg"/>
      </a>
      <a href="/omp?cmd=get_targets&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
        class="icon icon-sm"
        title="{gsa:i18n ('Targets')}">
        <img src="/img/list.svg" alt="{gsa:i18n ('Targets')}"/>
      </a>
    </div>
    <div class="gb_window_part_content">
      <form action="/omp" method="post" enctype="multipart/form-data">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="create_target"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden" name="next" value="get_target"/>
        <input type="hidden" name="filter" value="{filters/term}"/>
        <input type="hidden" name="first" value="{targets/@start}"/>
        <input type="hidden" name="max" value="{targets/@max}"/>
        <xsl:if test="not (gsa:may-op ('get_credentials'))">
          <input type="hidden" name="ssh_credential_id" value="0"/>
          <input type="hidden" name="smb_credential_id" value="0"/>
          <input type="hidden" name="esxi_credential_id" value="0"/>
          <input type="hidden" name="snmp_credential_id" value="0"/>
        </xsl:if>
        <xsl:if test="not (gsa:may-op ('get_port_lists'))">
          <!-- Use port list "OpenVAS Default". -->
          <input type="hidden"
                 name="port_list_id"
                 value="c7e03b6c-3bbe-11e1-a057-406186ea4fc5"/>
        </xsl:if>
        <table class="table-form">
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Name')"/>
            </td>
            <td>
              <input type="text" name="name" value="unnamed" size="30"
                     maxlength="80"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Comment')"/></td>
            <td>
              <input type="text" name="comment" size="30" maxlength="400"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Hosts')"/></td>
            <td>
              <table class="table-form">
                <tr>
                  <td>
                    <label>
                      <input type="radio" name="target_source" value="manual"
                             checked="1"/>
                      <xsl:value-of select="gsa:i18n ('Manual', 'Target Source')"/>
                    </label>
                  </td>
                  <td>
                    <xsl:choose>
                      <xsl:when test="string-length (/envelope/params/hosts)">
                        <input type="text" name="hosts" value="{/envelope/params/hosts}" size="30"
                                maxlength="2000"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <input type="text" name="hosts" value="{/envelope/client_address}" size="30"
                                maxlength="2000"/>
                      </xsl:otherwise>
                    </xsl:choose>
                  </td>
                </tr>
                <tr>
                  <td>
                    <label>
                      <input type="radio" name="target_source" value="file"/>
                      <xsl:value-of select="gsa:i18n ('From file', 'Target Source')"/>
                    </label>
                  </td>
                  <td>
                    <input type="file" name="file" size="30"/>
                  </td>
                </tr>
                <xsl:variable name="host_count">
                  <xsl:choose>
                    <xsl:when test="/envelope/params/host_count = 0">
                      <xsl:value-of select="count (/envelope/params/_param[substring (name, 1, 13) = 'bulk_selected'])"/>
                    </xsl:when>
                    <xsl:otherwise>
                      <xsl:value-of select="/envelope/params/host_count"/>
                    </xsl:otherwise>
                  </xsl:choose>
                </xsl:variable>
                <xsl:choose>
                  <xsl:when test="$host_count &gt; 0">
                    <tr>
                      <td>
                        <label>
                          <xsl:choose>
                            <xsl:when test="$host_count > 0">
                              <input type="radio" name="target_source" value="asset_hosts"
                                     checked="1"/>
                            </xsl:when>
                            <xsl:otherwise>
                              <input type="radio" name="target_source" value="asset_hosts"/>
                            </xsl:otherwise>
                          </xsl:choose>
                          <xsl:value-of select="gsa:i18n ('From host assets', 'Target Source')"/>
                          <xsl:text> (</xsl:text>
                          <input type="hidden" name="hosts_filter" value="{/envelope/params/hosts_filter}"/>
                          <xsl:value-of select="$host_count"/>
<!--
                          <a href="?cmd=get_assets&amp;type=host&amp;filter={/envelope/params/hosts_filter}&amp;token={/envelope/token}">
                            <xsl:value-of select="$host_count"/>
                            <xsl:text> hosts</xsl:text>
                          </a>
-->
                          <xsl:choose>
                            <xsl:when test="$host_count = 1">
                              <xsl:text> host</xsl:text>
                            </xsl:when>
                            <xsl:otherwise>
                            <xsl:text> hosts</xsl:text>
                            </xsl:otherwise>
                          </xsl:choose>
                          <xsl:text>)</xsl:text>
                        </label>
                      </td>
                    </tr>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="hidden" name="hosts_filter" value=""/>
                    <tr>
                      <td>
                        <label style="color:#ccc;">
                          <input type="radio" name="target_source" value="asset_hosts" disabled="1"/>
                          <xsl:value-of select="gsa:i18n ('From host assets', 'Target Source')"/>
                          <xsl:text> (0 hosts)</xsl:text>
                        </label>
                      </td>
                    </tr>
                  </xsl:otherwise>
                </xsl:choose>
              </table>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Exclude Hosts')"/></td>
            <td>
              <input type="text" name="exclude_hosts" value="" size="30"
                      maxlength="2000"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Reverse Lookup Only')"/></td>
            <td>
              <label>
                <input type="radio" name="reverse_lookup_only" value="1"/><xsl:value-of select="gsa:i18n ('Yes')"/>
              </label>
              <label>
                <input type="radio" name="reverse_lookup_only" value="0"
                       checked="1"/><xsl:value-of select="gsa:i18n ('No')"/>
              </label>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Reverse Lookup Unify')"/></td>
            <td>
              <label>
                <input type="radio" name="reverse_lookup_unify" value="1"/><xsl:value-of select="gsa:i18n ('Yes')"/>
              </label>
              <label>
                <input type="radio" name="reverse_lookup_unify" value="0"
                       checked="1"/><xsl:value-of select="gsa:i18n ('No')"/>
              </label>
            </td>
          </tr>
          <xsl:if test="gsa:may-op ('get_port_lists')">
            <tr>
              <td><xsl:value-of select="gsa:i18n ('Port List')"/></td>
              <td>
                <select name="port_list_id">
                  <xsl:apply-templates select="$port-lists" mode="select">
                    <xsl:with-param name="select_id" select="port_list_id"/>
                  </xsl:apply-templates>
                </select>
                <a href="#" title="{ gsa:i18n('Create a new port list') }"
                   class="new-action-icon icon icon-sm" data-type="port_list" data-done="select[name=port_list_id]">
                  <img class="valign-middle" src="/img/new.svg"/>
                </a>
              </td>
            </tr>
          </xsl:if>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Alive Test')"/></td>
            <td>
              <select name="alive_tests">
                <option value="Scan Config Default"><xsl:value-of select="gsa:i18n ('Scan Config Default')"/></option>
                <option value="ICMP Ping">ICMP Ping</option>
                <option value="TCP-ACK Service Ping">TCP-ACK Service Ping</option>
                <option value="TCP-SYN Service Ping">TCP-SYN Service Ping</option>
                <option value="ARP Ping">ARP Ping</option>
                <option value="ICMP &amp; TCP-ACK Service Ping">ICMP &amp; TCP-ACK Service Ping</option>
                <option value="ICMP &amp; ARP Ping">ICMP &amp; ARP Ping</option>
                <option value="TCP-ACK Service &amp; ARP Ping">TCP-ACK Service &amp; ARP Ping</option>
                <option value="ICMP, TCP-ACK Service &amp; ARP Ping">ICMP, TCP-ACK Service &amp; ARP Ping</option>
                <option value="Consider Alive">Consider Alive</option>
              </select>
            </td>
          </tr>
          <xsl:if test="gsa:may-op ('get_credentials')">
            <tr>
              <td>
                <xsl:value-of select="gsa:i18n ('Credentials for authenticated checks')"/>
              </td>
            </tr>
            <tr>
              <td><xsl:value-of select="gsa:i18n ('SSH')"/></td>
              <td>
                <select name="ssh_credential_id">
                  <option value="0">--</option>
                  <xsl:apply-templates select="$credentials/credential [type = 'up' or type = 'usk']" mode="select">
                    <xsl:with-param name="select_id" select="ssh_credential_id"/>
                  </xsl:apply-templates>
                </select>
                <xsl:text> </xsl:text>
                <xsl:value-of select="gsa:i18n ('on port', 'Target|Credential')"/>
                <xsl:text> </xsl:text>
                <input type="text" name="port" value="22" size="6"
                       maxlength="400"/>
                <a href="#" title="{ gsa:i18n('Create a credential') }"
                  class="new-action-icon icon icon-sm" data-type="credential" data-done="select[name=ssh_credential_id]" data-extra="restrict_credential_type=up|usk">
                  <img class="valign-middle" src="/img/new.svg"/>
                </a>
              </td>
            </tr>
            <tr>
              <td><xsl:value-of select="gsa:i18n ('SMB')"/></td>
              <td>
                <select name="smb_credential_id">
                  <option value="0">--</option>
                  <xsl:apply-templates select="$credentials/credential [type = 'up']" mode="select">
                    <xsl:with-param name="select_id" select="smb_credential_id"/>
                  </xsl:apply-templates>
                </select>
                <a href="#" title="{ gsa:i18n('Create a credential') }"
                  class="new-action-icon icon icon-sm" data-type="credential" data-done="select[name=smb_credential_id]" data-extra="restrict_credential_type=up">
                  <img class="valign-middle" src="/img/new.svg"/>
                </a>
              </td>
            </tr>
            <tr>
              <td><xsl:value-of select="gsa:i18n ('ESXi')"/></td>
              <td>
                <select name="esxi_credential_id">
                  <option value="0">--</option>
                  <xsl:apply-templates select="$credentials/credential [type = 'up']" mode="select">
                    <xsl:with-param name="select_id" select="esxi_credential_id"/>
                  </xsl:apply-templates>
                </select>
                <a href="#" title="{ gsa:i18n('Create a credential') }"
                  class="new-action-icon icon icon-sm" data-type="credential" data-done="select[name=esxi_credential_id]" data-extra="restrict_credential_type=up">
                  <img class="valign-middle" src="/img/new.svg"/>
                </a>
              </td>
            </tr>
            <tr>
              <td><xsl:value-of select="gsa:i18n ('SNMP')"/></td>
              <td>
                <select name="snmp_credential_id">
                  <option value="0">--</option>
                  <xsl:apply-templates select="$credentials/credential [type = 'snmp']" mode="select">
                    <xsl:with-param name="select_id" select="snmp_credential_id"/>
                  </xsl:apply-templates>
                </select>
                <a href="#" title="{ gsa:i18n('Create a credential') }"
                  class="new-action-icon icon icon-sm" data-type="credential" data-done="select[name=snmp_credential_id]" data-extra="restrict_credential_type=snmp">
                  <img class="valign-middle" src="/img/new.svg"/>
                </a>
              </td>
            </tr>
          </xsl:if>
          <tr>
            <td>
              <input type="submit" name="submit" value="{gsa:i18n ('Create Target')}"/>
            </td>
          </tr>
        </table>
      </form>
    </div>
  </div>
</xsl:template>

<xsl:template name="html-edit-target-form">
  <div class="gb_window">
    <div class="gb_window_part_left"></div>
    <div class="gb_window_part_right"></div>
    <div class="gb_window_part_center"><xsl:value-of select="gsa:i18n ('Edit Target')"/>
      <xsl:call-template name="edit-header-icons">
        <xsl:with-param name="cap-type" select="'Target'"/>
        <xsl:with-param name="type" select="'target'"/>
        <xsl:with-param name="id"
                        select="get_targets_response/target/@id"/>
      </xsl:call-template>
    </div>
    <div class="gb_window_part_content">
      <form action="" method="post" enctype="multipart/form-data">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="save_target"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden"
               name="target_id"
               value="{get_targets_response/target/@id}"/>
        <input type="hidden" name="next" value="{next}"/>
        <input type="hidden" name="sort_field" value="{sort_field}"/>
        <input type="hidden" name="sort_order" value="{sort_order}"/>
        <input type="hidden" name="filter" value="{filters/term}"/>
        <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
        <input type="hidden" name="first" value="{targets/@start}"/>
        <input type="hidden" name="max" value="{targets/@max}"/>
        <input type="hidden" name="in_use" value="{get_targets_response/target/in_use}"/>
        <xsl:if test="not (gsa:may-op ('get_credentials'))">
          <input type="hidden" name="ssh_credential_id" value="--"/>
          <input type="hidden" name="smb_credential_id" value="--"/>
          <input type="hidden" name="esxi_credential_id" value="--"/>
          <input type="hidden" name="snmp_credential_id" value="--"/>
        </xsl:if>
        <xsl:if test="not (gsa:may-op ('get_port_lists'))">
          <!-- Use port list "OpenVAS Default". -->
          <input type="hidden"
                 name="port_list_id"
                 value="c7e03b6c-3bbe-11e1-a057-406186ea4fc5"/>
        </xsl:if>
        <table class="table-form">
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
            <td>
              <input type="text"
                     name="name"
                     value="{get_targets_response/target/name}"
                     size="30"
                     maxlength="80"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Comment')"/></td>
            <td>
              <input type="text" name="comment" size="30" maxlength="400"
                     value="{get_targets_response/target/comment}"/>
            </td>
          </tr>
          <xsl:choose>
            <xsl:when test="get_targets_response/target/in_use = '0'">
              <tr>
                <td><xsl:value-of select="gsa:i18n ('Hosts')"/></td>
                <td>
                  <table>
                    <tr>
                      <td>
                        <label>
                          <input type="radio" name="target_source" value="manual"
                                 checked="1"/>
                          <xsl:value-of select="gsa:i18n ('Manual', 'Target Source')"/>
                        </label>
                      </td>
                      <td>
                        <input type="text" name="hosts"
                               value="{get_targets_response/target/hosts}"
                               size="30"
                               maxlength="2000"/>
                      </td>
                    </tr>
                    <tr>
                      <td>
                        <label>
                          <input type="radio" name="target_source" value="file"/>
                          <xsl:value-of select="gsa:i18n ('From file', 'Target Source')"/>
                        </label>
                      </td>
                      <td>
                        <input type="file" name="file" size="30"/>
                      </td>
                    </tr>
                  </table>
                </td>
              </tr>
              <tr>
                <td><xsl:value-of select="gsa:i18n ('Exclude Hosts')"/></td>
                <td>
                  <input type="text" name="exclude_hosts"
                         value="{get_targets_response/target/exclude_hosts}"
                         size="30" maxlength="2000"/>
                </td>
              </tr>
              <tr>
                <td><xsl:value-of select="gsa:i18n ('Reverse Lookup Only')"/></td>
                <td>
                  <label>
                    <xsl:choose>
                      <xsl:when test="get_targets_response/target/reverse_lookup_only = '1'">
                        <input type="radio" name="reverse_lookup_only" value="1" checked="1"/><xsl:value-of select="gsa:i18n ('Yes')"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <input type="radio" name="reverse_lookup_only" value="1"/><xsl:value-of select="gsa:i18n ('Yes')"/>
                      </xsl:otherwise>
                    </xsl:choose>
                  </label>
                  <label>
                    <xsl:choose>
                      <xsl:when test="get_targets_response/target/reverse_lookup_only = '0'">
                        <input type="radio" name="reverse_lookup_only" value="0" checked="1"/><xsl:value-of select="gsa:i18n ('No')"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <input type="radio" name="reverse_lookup_only" value="0"/><xsl:value-of select="gsa:i18n ('No')"/>
                      </xsl:otherwise>
                    </xsl:choose>
                  </label>
                </td>
              </tr>
              <tr>
                <td><xsl:value-of select="gsa:i18n ('Reverse Lookup Unify')"/></td>
                <td>
                  <label>
                    <xsl:choose>
                      <xsl:when test="get_targets_response/target/reverse_lookup_unify = '1'">
                        <input type="radio" name="reverse_lookup_unify" value="1" checked="1"/><xsl:value-of select="gsa:i18n ('Yes')"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <input type="radio" name="reverse_lookup_unify" value="1"/><xsl:value-of select="gsa:i18n ('Yes')"/>
                      </xsl:otherwise>
                    </xsl:choose>
                  </label>
                  <label>
                    <xsl:choose>
                      <xsl:when test="get_targets_response/target/reverse_lookup_unify = '0'">
                        <input type="radio" name="reverse_lookup_unify" value="0" checked="1"/><xsl:value-of select="gsa:i18n ('No')"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <input type="radio" name="reverse_lookup_unify" value="0"/><xsl:value-of select="gsa:i18n ('No')"/>
                      </xsl:otherwise>
                    </xsl:choose>
                  </label>
                </td>
              </tr>
              <xsl:if test="gsa:may-op ('get_port_lists')">
                <tr>
                  <td><xsl:value-of select="gsa:i18n ('Port List')"/></td>
                  <td>
                    <select name="port_list_id">
                      <xsl:variable name="port_list_id">
                        <xsl:value-of select="get_targets_response/target/port_list/@id"/>
                      </xsl:variable>
                      <xsl:for-each select="get_port_lists_response/port_list">
                        <xsl:choose>
                          <xsl:when test="@id = $port_list_id">
                            <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
                          </xsl:when>
                          <xsl:otherwise>
                            <option value="{@id}"><xsl:value-of select="name"/></option>
                          </xsl:otherwise>
                        </xsl:choose>
                      </xsl:for-each>
                    </select>
                    <a href="#" title="{ gsa:i18n('Create a new port list') }"
                      class="new-action-icon icon icon-sm" data-type="port_list" data-done="select[name=port_list_id]">
                      <img class="valign-middle" src="/img/new.svg"/>
                    </a>
                  </td>
                </tr>
              </xsl:if>
              <tr>
                <td><xsl:value-of select="gsa:i18n ('Alive Test')"/></td>
                <td>
                  <xsl:variable name="alive_tests" select="get_targets_response/target/alive_tests/text()"/>
                  <select name="alive_tests">
                    <xsl:call-template name="opt">
                      <xsl:with-param name="value" select="'Scan Config Default'"/>
                      <xsl:with-param name="content" select="gsa:i18n ('Scan Config Default')"/>
                      <xsl:with-param name="select-value" select="$alive_tests"/>
                    </xsl:call-template>
                    <xsl:call-template name="opt">
                      <xsl:with-param name="value" select="'ICMP Ping'"/>
                      <xsl:with-param name="select-value" select="$alive_tests"/>
                    </xsl:call-template>
                    <xsl:call-template name="opt">
                      <xsl:with-param name="value" select="'TCP-ACK Service Ping'"/>
                      <xsl:with-param name="select-value" select="$alive_tests"/>
                    </xsl:call-template>
                    <xsl:call-template name="opt">
                      <xsl:with-param name="value" select="'TCP-SYN Service Ping'"/>
                      <xsl:with-param name="select-value" select="$alive_tests"/>
                    </xsl:call-template>
                    <xsl:call-template name="opt">
                      <xsl:with-param name="value" select="'ARP Ping'"/>
                      <xsl:with-param name="select-value" select="$alive_tests"/>
                    </xsl:call-template>
                    <xsl:call-template name="opt">
                      <xsl:with-param name="value" select="'ICMP &amp; TCP-ACK Service Ping'"/>
                      <xsl:with-param name="select-value" select="$alive_tests"/>
                    </xsl:call-template>
                    <xsl:call-template name="opt">
                      <xsl:with-param name="value" select="'ICMP &amp; ARP Ping'"/>
                      <xsl:with-param name="select-value" select="$alive_tests"/>
                    </xsl:call-template>
                    <xsl:call-template name="opt">
                      <xsl:with-param name="value" select="'TCP-ACK Service &amp; ARP Ping'"/>
                      <xsl:with-param name="select-value" select="$alive_tests"/>
                    </xsl:call-template>
                    <xsl:call-template name="opt">
                      <xsl:with-param name="value" select="'ICMP, TCP-ACK Service &amp; ARP Ping'"/>
                      <xsl:with-param name="select-value" select="$alive_tests"/>
                    </xsl:call-template>
                    <xsl:call-template name="opt">
                      <xsl:with-param name="value" select="'Consider Alive'"/>
                      <xsl:with-param name="select-value" select="$alive_tests"/>
                    </xsl:call-template>
                  </select>
                </td>
              </tr>
              <xsl:if test="gsa:may-op ('get_credentials')">
                <tr>
                  <td>
                    <xsl:value-of select="gsa:i18n ('Credentials for authenticated checks')"/>
                  </td>
                </tr>
                <tr>
                  <td><xsl:value-of select="gsa:i18n ('SSH')"/></td>
                  <td>
                    <select name="ssh_credential_id">
                      <xsl:variable name="credential_id">
                        <xsl:value-of select="get_targets_response/target/ssh_credential/@id"/>
                      </xsl:variable>
                      <xsl:choose>
                        <xsl:when test="string-length ($credential_id) &gt; 0">
                          <option value="0">--</option>
                        </xsl:when>
                        <xsl:otherwise>
                          <option value="0" selected="1">--</option>
                        </xsl:otherwise>
                      </xsl:choose>
                      <xsl:for-each select="get_credentials_response/credential [type = 'up' or type = 'usk']">
                        <xsl:choose>
                          <xsl:when test="@id = $credential_id">
                            <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
                          </xsl:when>
                          <xsl:otherwise>
                            <option value="{@id}"><xsl:value-of select="name"/></option>
                          </xsl:otherwise>
                        </xsl:choose>
                      </xsl:for-each>
                    </select>
                    <xsl:text> </xsl:text>
                    <xsl:value-of select="gsa:i18n ('on port', 'Target|Credential')"/>
                    <xsl:text> </xsl:text>
                    <xsl:variable name="credential"
                                  select="get_targets_response/target/ssh_credential"/>
                    <xsl:choose>
                      <xsl:when test="$credential and string-length ($credential/port)">
                        <input type="text"
                               name="port"
                               value="{get_targets_response/target/ssh_credential/port}"
                               size="6"
                               maxlength="400"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <input type="text" name="port" value="22" size="6" maxlength="400"/>
                      </xsl:otherwise>
                    </xsl:choose>
                    <a href="#" title="{ gsa:i18n('Create a credential') }"
                      class="new-action-icon icon icon-sm" data-type="credential" data-done="select[name=ssh_credential_id]" data-extra="restrict_credential_type=up|usk">
                      <img class="valign-middle" src="/img/new.svg"/>
                    </a>
                  </td>
                </tr>
                <tr>
                  <td><xsl:value-of select="gsa:i18n ('SMB')"/></td>
                  <td>
                    <select name="smb_credential_id">
                      <xsl:variable name="credential_id">
                        <xsl:value-of select="get_targets_response/target/smb_credential/@id"/>
                      </xsl:variable>
                      <xsl:choose>
                        <xsl:when test="string-length ($credential_id) &gt; 0">
                          <option value="0">--</option>
                        </xsl:when>
                        <xsl:otherwise>
                          <option value="0" selected="1">--</option>
                        </xsl:otherwise>
                      </xsl:choose>
                      <xsl:for-each select="get_credentials_response/credential [type = 'up']">
                        <xsl:choose>
                          <xsl:when test="@id = $credential_id">
                            <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
                          </xsl:when>
                          <xsl:otherwise>
                            <option value="{@id}"><xsl:value-of select="name"/></option>
                          </xsl:otherwise>
                        </xsl:choose>
                      </xsl:for-each>
                    </select>
                    <a href="#" title="{ gsa:i18n('Create a credential') }"
                      class="new-action-icon icon icon-sm" data-type="credential" data-done="select[name=smb_credential_id]" data-extra="restrict_credential_type=up">
                      <img class="valign-middle" src="/img/new.svg"/>
                    </a>
                  </td>
                </tr>
                <tr>
                  <td><xsl:value-of select="gsa:i18n ('ESXi')"/></td>
                  <td>
                    <select name="esxi_credential_id">
                      <xsl:variable name="credential_id">
                        <xsl:value-of select="get_targets_response/target/esxi_credential/@id"/>
                      </xsl:variable>
                      <xsl:choose>
                        <xsl:when test="string-length ($credential_id) &gt; 0">
                          <option value="0">--</option>
                        </xsl:when>
                        <xsl:otherwise>
                          <option value="0" selected="1">--</option>
                        </xsl:otherwise>
                      </xsl:choose>
                      <xsl:for-each select="get_credentials_response/credential [type = 'up']">
                        <xsl:choose>
                          <xsl:when test="@id = $credential_id">
                            <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
                          </xsl:when>
                          <xsl:otherwise>
                            <option value="{@id}"><xsl:value-of select="name"/></option>
                          </xsl:otherwise>
                        </xsl:choose>
                      </xsl:for-each>
                    </select>
                    <a href="#" title="{ gsa:i18n('Create a credential') }"
                      class="new-action-icon icon icon-sm" data-type="credential" data-done="select[name=esxi_credential_id]" data-extra="restrict_credential_type=up">
                      <img class="valign-middle" src="/img/new.svg"/>
                    </a>
                  </td>
                </tr>
                <tr>
                  <td><xsl:value-of select="gsa:i18n ('SNMP')"/></td>
                  <td>
                    <select name="snmp_credential_id">
                      <xsl:variable name="credential_id">
                        <xsl:value-of select="get_targets_response/target/snmp_credential/@id"/>
                      </xsl:variable>
                      <xsl:choose>
                        <xsl:when test="string-length ($credential_id) &gt; 0">
                          <option value="0">--</option>
                        </xsl:when>
                        <xsl:otherwise>
                          <option value="0" selected="1">--</option>
                        </xsl:otherwise>
                      </xsl:choose>
                      <xsl:for-each select="get_credentials_response/credential [type = 'snmp']">
                        <xsl:choose>
                          <xsl:when test="@id = $credential_id">
                            <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
                          </xsl:when>
                          <xsl:otherwise>
                            <option value="{@id}"><xsl:value-of select="name"/></option>
                          </xsl:otherwise>
                        </xsl:choose>
                      </xsl:for-each>
                    </select>
                    <a href="#" title="{ gsa:i18n('Create a credential') }"
                      class="new-action-icon icon icon-sm" data-type="credential" data-done="select[name=snmp_credential_id]" data-extra="restrict_credential_type=snmp">
                      <img class="valign-middle" src="/img/new.svg"/>
                    </a>
                  </td>
                </tr>
              </xsl:if>
            </xsl:when>
            <xsl:otherwise>
              <xsl:variable name="width" select="240"/>
              <!-- Target is in use. -->
              <tr>
                <td>
                  <xsl:value-of select="gsa:i18n ('Hosts')"/> (<xsl:value-of select="gsa:i18n ('immutable')"/>)
                </td>
                <td>
                  <table>
                    <tr>
                      <td>
                        <label>
                          <input type="radio" name="target_source" value="manual"
                                 checked="1" disabled="1"/>
                          <xsl:value-of select="gsa:i18n ('Manual', 'Target Source')"/>
                        </label>
                      </td>
                      <td>
                        <input type="text" name="hosts"
                               value="{get_targets_response/target/hosts}"
                               size="30"
                               maxlength="2000"
                               disabled="1"/>
                      </td>
                    </tr>
                    <tr>
                      <td>
                        <label>
                          <input type="radio" name="target_source" value="file" disabled="1"/>
                          <xsl:value-of select="gsa:i18n ('From file', 'Target Source')"/>
                        </label>
                      </td>
                      <td>
                        <input type="file" name="file" size="30" disabled="1"/>
                      </td>
                    </tr>
                  </table>
                </td>
              </tr>
              <tr>
                <td valign="top" width="{$width}">
                  <xsl:value-of select="gsa:i18n ('Exclude Hosts')"/> (<xsl:value-of select="gsa:i18n ('immutable')"/>)
                </td>
                <td>
                  <input type="text" name="exclude_hosts"
                         value="{get_targets_response/target/exclude_hosts}"
                         size="30" maxlength="2000"
                         disabled="1"/>
                </td>
              </tr>
              <tr>
                <td valign="top" width="{$width}">
                  <xsl:value-of select="gsa:i18n ('Reverse Lookup Only')"/> (<xsl:value-of select="gsa:i18n ('immutable')"/>)
                </td>
                <td>
                  <label>
                    <xsl:choose>
                      <xsl:when test="get_targets_response/target/reverse_lookup_only = '1'">
                        <input type="radio" name="reverse_lookup_only" value="1" checked="1" disabled="1"/><xsl:value-of select="gsa:i18n ('Yes')"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <input type="radio" name="reverse_lookup_only" value="1" disabled="1"/><xsl:value-of select="gsa:i18n ('Yes')"/>
                      </xsl:otherwise>
                    </xsl:choose>
                  </label>
                  <label>
                    <xsl:choose>
                      <xsl:when test="get_targets_response/target/reverse_lookup_only = '0'">
                        <input type="radio" name="reverse_lookup_only" value="0" checked="1" disabled="1"/><xsl:value-of select="gsa:i18n ('No')"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <input type="radio" name="reverse_lookup_only" value="0" disabled="1"/><xsl:value-of select="gsa:i18n ('No')"/>
                      </xsl:otherwise>
                    </xsl:choose>
                  </label>
                </td>
              </tr>
              <tr>
                <td valign="top" width="{$width}">
                  <xsl:value-of select="gsa:i18n ('Reverse Lookup Unify')"/> (<xsl:value-of select="gsa:i18n ('immutable')"/>)
                </td>
                <td>
                  <label>
                    <xsl:choose>
                      <xsl:when test="get_targets_response/target/reverse_lookup_unify = '1'">
                        <input type="radio" name="reverse_lookup_unify" value="1" checked="1" disabled="1"/><xsl:value-of select="gsa:i18n ('Yes')"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <input type="radio" name="reverse_lookup_unify" value="1" disabled="1"/><xsl:value-of select="gsa:i18n ('Yes')"/>
                      </xsl:otherwise>
                    </xsl:choose>
                  </label>
                  <label>
                    <xsl:choose>
                      <xsl:when test="get_targets_response/target/reverse_lookup_unify = '0'">
                        <input type="radio" name="reverse_lookup_unify" value="0" checked="1" disabled="1"/><xsl:value-of select="gsa:i18n ('No')"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <input type="radio" name="reverse_lookup_unify" value="0" disabled="1"/><xsl:value-of select="gsa:i18n ('No')"/>
                      </xsl:otherwise>
                    </xsl:choose>
                  </label>
                </td>
              </tr>
              <xsl:if test="gsa:may-op ('get_port_lists')">
                <tr>
                  <td valign="top" width="{$width}">
                    <xsl:value-of select="gsa:i18n ('Port List')"/> (<xsl:value-of select="gsa:i18n ('immutable')"/>)
                  </td>
                  <td>
                    <select name="port_list_id" disabled="1">
                      <option>
                        <xsl:value-of select="get_targets_response/target/port_list/name"/>
                      </option>
                    </select>
                  </td>
                </tr>
              </xsl:if>
              <tr>
                <td><xsl:value-of select="gsa:i18n ('Alive Test')"/></td>
                <td>
                  <xsl:variable name="alive_tests" select="get_targets_response/target/alive_tests/text()"/>
                  <select name="alive_tests">
                    <xsl:call-template name="opt">
                      <xsl:with-param name="value" select="'Scan Config Default'"/>
                      <xsl:with-param name="content" select="gsa:i18n ('Scan Config Default')"/>
                      <xsl:with-param name="select-value" select="$alive_tests"/>
                    </xsl:call-template>
                    <xsl:call-template name="opt">
                      <xsl:with-param name="value" select="'ICMP Ping'"/>
                      <xsl:with-param name="select-value" select="$alive_tests"/>
                    </xsl:call-template>
                    <xsl:call-template name="opt">
                      <xsl:with-param name="value" select="'TCP-ACK Service Ping'"/>
                      <xsl:with-param name="select-value" select="$alive_tests"/>
                    </xsl:call-template>
                    <xsl:call-template name="opt">
                      <xsl:with-param name="value" select="'TCP-SYN Service Ping'"/>
                      <xsl:with-param name="select-value" select="$alive_tests"/>
                    </xsl:call-template>
                    <xsl:call-template name="opt">
                      <xsl:with-param name="value" select="'ARP Ping'"/>
                      <xsl:with-param name="select-value" select="$alive_tests"/>
                    </xsl:call-template>
                    <xsl:call-template name="opt">
                      <xsl:with-param name="value" select="'ICMP &amp; TCP-ACK Service Ping'"/>
                      <xsl:with-param name="select-value" select="$alive_tests"/>
                    </xsl:call-template>
                    <xsl:call-template name="opt">
                      <xsl:with-param name="value" select="'ICMP &amp; ARP Ping'"/>
                      <xsl:with-param name="select-value" select="$alive_tests"/>
                    </xsl:call-template>
                    <xsl:call-template name="opt">
                      <xsl:with-param name="value" select="'TCP-ACK Service &amp; ARP Ping'"/>
                      <xsl:with-param name="select-value" select="$alive_tests"/>
                    </xsl:call-template>
                    <xsl:call-template name="opt">
                      <xsl:with-param name="value" select="'ICMP, TCP-ACK Service &amp; ARP Ping'"/>
                      <xsl:with-param name="select-value" select="$alive_tests"/>
                    </xsl:call-template>
                    <xsl:call-template name="opt">
                      <xsl:with-param name="value" select="'Consider Alive'"/>
                      <xsl:with-param name="select-value" select="$alive_tests"/>
                    </xsl:call-template>
                  </select>
                </td>
              </tr>
              <xsl:if test="gsa:may-op ('get_credentials')">
                <tr>
                  <td valign="top" width="175" colspan="2">
                    <xsl:value-of select="gsa:i18n ('Credentials for authenticated checks')"/>
                    (<xsl:value-of select="gsa:i18n ('immutable')"/>):
                  </td>
                </tr>
                <tr>
                  <td valign="top" width="{$width}">
                    <xsl:value-of select="gsa:i18n ('SSH')"/>
                  </td>
                  <td>
                    <select name="ssh_credential_id" disabled="1">
                      <xsl:variable name="credential_id">
                        <xsl:value-of select="get_targets_response/target/ssh_credential/@id"/>
                      </xsl:variable>
                      <xsl:choose>
                        <xsl:when test="string-length ($credential_id) &gt; 0">
                          <option value="0">--</option>
                        </xsl:when>
                        <xsl:otherwise>
                          <option value="0" selected="1">--</option>
                        </xsl:otherwise>
                      </xsl:choose>
                      <xsl:for-each select="get_credentials_response/credential">
                        <xsl:choose>
                          <xsl:when test="@id = $credential_id">
                            <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
                          </xsl:when>
                          <xsl:otherwise>
                            <option value="{@id}"><xsl:value-of select="name"/></option>
                          </xsl:otherwise>
                        </xsl:choose>
                      </xsl:for-each>
                    </select>
                    <xsl:text> </xsl:text>
                    <xsl:value-of select="gsa:i18n ('on port', 'Target|Credential')"/>
                    <xsl:text> </xsl:text>
                    <xsl:variable name="credential"
                                  select="get_targets_response/target/ssh_credential"/>
                    <xsl:choose>
                      <xsl:when test="$credential and string-length ($credential/port)">
                        <input type="text"
                               name="port"
                               value="{get_targets_response/target/ssh_credential/port}"
                               size="6"
                               maxlength="400"
                               disabled="1"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <input type="text" name="port" value="22" size="6" maxlength="400" disabled="1"/>
                      </xsl:otherwise>
                    </xsl:choose>
                  </td>
                </tr>
                <tr>
                  <td valign="top" width="{$width}">
                    <xsl:value-of select="gsa:i18n ('SMB')"/>
                  </td>
                  <td>
                    <select name="smb_credential_id" disabled="1">
                      <xsl:variable name="credential_id">
                        <xsl:value-of select="get_targets_response/target/smb_credential/@id"/>
                      </xsl:variable>
                      <xsl:choose>
                        <xsl:when test="string-length ($credential_id) &gt; 0">
                          <option value="0">--</option>
                        </xsl:when>
                        <xsl:otherwise>
                          <option value="0" selected="1">--</option>
                        </xsl:otherwise>
                      </xsl:choose>
                      <xsl:for-each select="get_credentials_response/credential">
                        <xsl:choose>
                          <xsl:when test="@id = $credential_id">
                            <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
                          </xsl:when>
                          <xsl:otherwise>
                            <option value="{@id}"><xsl:value-of select="name"/></option>
                          </xsl:otherwise>
                        </xsl:choose>
                      </xsl:for-each>
                    </select>
                  </td>
                </tr>
                <tr>
                  <td valign="top" width="175"><xsl:value-of select="gsa:i18n ('ESXi')"/></td>
                  <td>
                    <select name="esxi_credential_id" disabled="1">
                      <xsl:variable name="credential_id">
                        <xsl:value-of select="get_targets_response/target/esxi_credential/@id"/>
                      </xsl:variable>
                      <xsl:choose>
                        <xsl:when test="string-length ($credential_id) &gt; 0">
                          <option value="0">--</option>
                        </xsl:when>
                        <xsl:otherwise>
                          <option value="0" selected="1">--</option>
                        </xsl:otherwise>
                      </xsl:choose>
                      <xsl:for-each select="get_credentials_response/credential">
                        <xsl:choose>
                          <xsl:when test="@id = $credential_id">
                            <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
                          </xsl:when>
                          <xsl:otherwise>
                            <option value="{@id}"><xsl:value-of select="name"/></option>
                          </xsl:otherwise>
                        </xsl:choose>
                      </xsl:for-each>
                    </select>
                  </td>
                </tr>
                <tr>
                  <td valign="top" width="175"><xsl:value-of select="gsa:i18n ('SNMP')"/></td>
                  <td>
                    <select name="snmp_credential_id" disabled="1">
                      <xsl:variable name="credential_id">
                        <xsl:value-of select="get_targets_response/target/snmp_credential/@id"/>
                      </xsl:variable>
                      <xsl:choose>
                        <xsl:when test="string-length ($credential_id) &gt; 0">
                          <option value="0">--</option>
                        </xsl:when>
                        <xsl:otherwise>
                          <option value="0" selected="1">--</option>
                        </xsl:otherwise>
                      </xsl:choose>
                      <xsl:for-each select="get_credentials_response/credential">
                        <xsl:choose>
                          <xsl:when test="@id = $credential_id">
                            <option value="{@id}" selected="1"><xsl:value-of select="name"/></option>
                          </xsl:when>
                          <xsl:otherwise>
                            <option value="{@id}"><xsl:value-of select="name"/></option>
                          </xsl:otherwise>
                        </xsl:choose>
                      </xsl:for-each>
                    </select>
                  </td>
                </tr>
              </xsl:if>
            </xsl:otherwise>
          </xsl:choose>
          <tr>
            <td colspan="2" style="text-align:right;">
              <input type="submit" name="submit" value="{gsa:i18n ('Save Target')}"/>
            </td>
          </tr>
        </table>
      </form>
    </div>
  </div>
</xsl:template>

<xsl:template match="edit_target">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="modify_target_response"/>
  <xsl:call-template name="html-edit-target-form"/>
</xsl:template>

<xsl:template name="column-name">
  <xsl:param name="type">target</xsl:param>
  <xsl:param name="head"/>
  <xsl:param name="image"/>
  <xsl:param name="name"/>
  <xsl:param name="current" select="."/>
  <xsl:param name="extra_params"/>
  <xsl:param name="sort-reverse"/>
  <xsl:param name="token" select="/envelope/token"/>
  <xsl:variable name="heading">
    <xsl:choose>
      <xsl:when test="image">
        <img src="{$image}"
          alt="{$head}"
          class="icon icon-sm"
          title="{$head}"/>
      </xsl:when>
      <xsl:when test="contains ($head, '.png') or contains ($head, '.svg')">
        <img src="/img/{$head}"
          alt="{gsa:capitalise (str:replace (gsa:lower-case ($name), '_', ' '))}"
          class="icon icon-sm"
          title="{gsa:capitalise (str:replace (gsa:lower-case ($name), '_', ' '))}"/>
      </xsl:when>
      <xsl:when test="$head = 'High' or $head = 'Medium' or $head = 'Low' or $head = 'Log' or $head = 'False Positive'">
        <xsl:call-template name="severity-label">
          <xsl:with-param name="level" select="translate($head, '_', ' ')"/>
        </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$head"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:choose>
    <xsl:when test="$current/sort/field/text() = $name and $current/sort/field/order = 'descending'">
      <a class="gbntablehead2" href="/omp?cmd=get_{gsa:type-many($type)}{$extra_params}&amp;filter=sort={$name} first=1 {$current/filters/term}&amp;token={$token}"><xsl:copy-of select="$heading"/></a>
    </xsl:when>
    <xsl:when test="$current/sort/field/text() = $name and $current/sort/field/order = 'ascending'">
      <a class="gbntablehead2" href="/omp?cmd=get_{gsa:type-many($type)}{$extra_params}&amp;filter=sort-reverse={$name} first=1 {$current/filters/term}&amp;token={$token}"><xsl:copy-of select="$heading"/></a>
    </xsl:when>
    <xsl:when test="$name = ''">
      <!-- Starts with some other column. -->
      <xsl:copy-of select="$heading"/>
    </xsl:when>
    <xsl:when test="$sort-reverse">
      <!-- Starts with some other column.  Include reverse sorting link. -->
      <a class="gbntablehead2" href="/omp?cmd=get_{gsa:type-many($type)}{$extra_params}&amp;filter=sort-reverse={$name} first=1 {$current/filters/term}&amp;token={$token}"><xsl:copy-of select="$heading"/></a>
    </xsl:when>
    <xsl:otherwise>
      <!-- Starts with some other column.  Include sorting link. -->
      <a class="gbntablehead2" href="/omp?cmd=get_{gsa:type-many($type)}{$extra_params}&amp;filter=sort={$name} first=1 {$current/filters/term}&amp;token={$token}"><xsl:copy-of select="$heading"/></a>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="html-targets-table">
  <xsl:call-template name="list-window">
    <xsl:with-param name="type" select="'target'"/>
    <xsl:with-param name="cap-type" select="'Target'"/>
    <xsl:with-param name="resources-summary" select="targets"/>
    <xsl:with-param name="resources" select="target"/>
    <xsl:with-param name="count" select="count (target)"/>
    <xsl:with-param name="filtered-count" select="target_count/filtered"/>
    <xsl:with-param name="full-count" select="target_count/text ()"/>
    <xsl:with-param name="columns" xmlns="">
      <column>
        <name><xsl:value-of select="gsa:i18n('Name')"/></name>
        <field>name</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Hosts')"/></name>
        <field>hosts</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('IPs')"/></name>
        <field>ips</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Port List')"/></name>
        <field>port_list</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Credentials')"/></name>
        <hide_in_filter>1</hide_in_filter>
        <html>
          <after>
            <xsl:variable name="sort_keyword" select="/envelope/get_targets/get_targets_response/filters/keywords/keyword[column='sort' or column='sort-reverse']"/>
            <span style="font-size:80%; color:silver">
              <xsl:text> - </xsl:text>
              <xsl:value-of select="gsa:i18n('sort by', 'Target|Credential')"/>
              <xsl:text>: </xsl:text>
            </span>
            <span class="credential-type-sort"
              data-sort-colum="{$sort_keyword/column}"
              data-sort-value="{$sort_keyword/value}"
              data-filter="{/envelope/get_targets/get_targets_response/filters/term}"
              data-token="{/envelope/token}">
              <select name="sort_credential_type" onChange="select_credential_type (true)">
                <xsl:call-template name="opt">
                  <xsl:with-param name="value" select="'ssh_credential'"/>
                  <xsl:with-param name="content" select="'SSH'"/>
                  <xsl:with-param name="select-value" select="$sort_keyword/value"/>
                </xsl:call-template>
                <xsl:call-template name="opt">
                  <xsl:with-param name="value" select="'smb_credential'"/>
                  <xsl:with-param name="content" select="'SMB'"/>
                  <xsl:with-param name="select-value" select="$sort_keyword/value"/>
                </xsl:call-template>
                <xsl:call-template name="opt">
                  <xsl:with-param name="value" select="'esxi_credential'"/>
                  <xsl:with-param name="content" select="'ESXi'"/>
                  <xsl:with-param name="select-value" select="$sort_keyword/value"/>
                </xsl:call-template>
                <xsl:call-template name="opt">
                  <xsl:with-param name="value" select="'snmp_credential'"/>
                  <xsl:with-param name="content" select="'SNMP'"/>
                  <xsl:with-param name="select-value" select="$sort_keyword/value"/>
                </xsl:call-template>
              </select>
              <xsl:text> </xsl:text>
              <a class="credential-type-sort-asc icon icon-sm" style="position:relative; top:3px;"
                href="#" title="{gsa:i18n ('Ascending')}">
                <img src="/img/ascending.svg"/>
              </a>
              <a class="credential-type-sort-desc icon icon-sm" style="position:relative; top:3px;"
                href="#" title="{gsa:i18n ('Descending')}">
                <img src="/img/descending.svg"/>
              </a>
            </span>
          </after>
        </html>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('SSH Credential')"/></name>
        <hide_in_table>1</hide_in_table>
        <field>ssh_credential</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('SMB Credential')"/></name>
        <hide_in_table>1</hide_in_table>
        <field>smb_credential</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('ESXi Credential')"/></name>
        <hide_in_table>1</hide_in_table>
        <field>esxi_credential</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('SNMP Credential')"/></name>
        <hide_in_table>1</hide_in_table>
        <field>snmp_credential</field>
      </column>
    </xsl:with-param>
    <xsl:with-param name="icon-count" select="4"/>
  </xsl:call-template>
</xsl:template>

<!--     NEW_TARGET -->

<xsl:template match="new_target">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="commands_response/delete_target_response"/>
  <xsl:apply-templates select="create_target_response"/>
  <xsl:call-template name="html-create-target-form">
    <xsl:with-param
      name="credentials"
      select="get_credentials_response | commands_response/get_credentials_response"/>
    <xsl:with-param
      name="port-lists"
      select="get_port_lists_response | commands_response/get_port_lists_response"/>
  </xsl:call-template>
</xsl:template>

<!--     CREATE_TARGET_RESPONSE -->

<xsl:template match="create_target_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Create Target</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!--     DELETE_TARGET_RESPONSE -->

<xsl:template match="delete_target_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">
      Delete Target
    </xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!--     TARGET -->

<xsl:template match="target">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <xsl:call-template name="observers-icon">
        <xsl:with-param name="type" select="'Target'"/>
      </xsl:call-template>
      <b>
        <a href="/omp?cmd=get_target&amp;target_id={@id}&amp;filter={str:encode-uri (../filters/term, true ())}&amp;first={../targets/@start}&amp;max={../targets/@max}&amp;token={/envelope/token}"
           title="{gsa:view_details_title ('Target', name)}">
          <xsl:value-of select="name"/>
        </a>
      </b>
      <xsl:choose>
        <xsl:when test="comment != ''">
          <div class="comment">(<xsl:value-of select="comment"/>)</div>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:variable name="max" select="500"/>
      <xsl:choose>
        <xsl:when test="string-length(hosts) &gt; $max">
          <xsl:value-of select="substring (hosts, 0, $max)"/>...
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="hosts"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td><xsl:value-of select="max_hosts"/></td>
    <td>
      <xsl:choose>
        <xsl:when test="gsa:may-op ('get_port_lists')">
          <a href="/omp?cmd=get_port_list&amp;port_list_id={port_list/@id}&amp;token={/envelope/token}">
            <xsl:value-of select="port_list/name"/>
          </a>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="port_list/name"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <table>
        <xsl:if test="ssh_credential/@id != ''">
          <tr>
            <td width="35">SSH:</td>
            <td>
              <xsl:choose>
                <xsl:when test="boolean (ssh_credential/permissions) and count (ssh_credential/permissions/permission) = 0">
                  <xsl:value-of select="ssh_credential/name"/>
                </xsl:when>
                <xsl:when test="gsa:may-op ('get_credentials')">
                  <a href="/omp?cmd=get_credential&amp;credential_id={ssh_credential/@id}&amp;token={/envelope/token}">
                    <xsl:value-of select="ssh_credential/name"/>
                  </a>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:value-of select="ssh_credential/name"/>
                </xsl:otherwise>
              </xsl:choose>
            </td>
          </tr>
        </xsl:if>
        <xsl:if test="smb_credential/@id != ''">
          <tr>
            <td>SMB:</td>
            <td>
              <xsl:choose>
                <xsl:when test="boolean (smb_credential/permissions) and count (smb_credential/permissions/permission) = 0">
                  <xsl:value-of select="smb_credential/name"/>
                </xsl:when>
                <xsl:when test="gsa:may-op ('get_credentials')">
                  <a href="/omp?cmd=get_credential&amp;credential_id={smb_credential/@id}&amp;token={/envelope/token}">
                    <xsl:value-of select="smb_credential/name"/>
                  </a>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:value-of select="smb_credential/name"/>
                </xsl:otherwise>
              </xsl:choose>
            </td>
          </tr>
        </xsl:if>
        <xsl:if test="esxi_credential/@id != ''">
          <tr>
            <td>ESXi:</td>
            <td>
              <xsl:choose>
                <xsl:when test="boolean (esxi_credential/permissions) and count (esxi_credential/permissions/permission) = 0">
                  <xsl:value-of select="esxi_credential/name"/>
                </xsl:when>
                <xsl:when test="gsa:may-op ('get_credentials')">
                  <a href="/omp?cmd=get_credential&amp;credential_id={esxi_credential/@id}&amp;token={/envelope/token}">
                    <xsl:value-of select="esxi_credential/name"/>
                  </a>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:value-of select="esxi_credential/name"/>
                </xsl:otherwise>
              </xsl:choose>
            </td>
          </tr>
        </xsl:if>
        <xsl:if test="snmp_credential/@id != ''">
          <tr>
            <td>SNMP:</td>
            <td>
              <xsl:choose>
                <xsl:when test="boolean (snmp_credential/permissions) and count (snmp_credential/permissions/permission) = 0">
                  <xsl:value-of select="snmp_credential/name"/>
                </xsl:when>
                <xsl:when test="gsa:may-op ('get_credentials')">
                  <a href="/omp?cmd=get_credential&amp;credential_id={snmp_credential/@id}&amp;token={/envelope/token}">
                    <xsl:value-of select="snmp_credential/name"/>
                  </a>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:value-of select="snmp_credential/name"/>
                </xsl:otherwise>
              </xsl:choose>
            </td>
          </tr>
        </xsl:if>
      </table>
    </td>
    <xsl:choose>
      <xsl:when test="/envelope/params/bulk_select = 1">
        <td style="text-align:center">
          <label style="width:100%">
            <input name="bulk_selected:{@id}" type="checkbox" style="width:100%; height:100%" title="{gsa:i18n ('Select for bulk action')}"/>
          </label>
        </td>
      </xsl:when>
      <xsl:otherwise>
        <td class="table-actions">
          <xsl:call-template name="list-window-line-icons">
            <xsl:with-param name="cap-type" select="'Target'"/>
            <xsl:with-param name="type" select="'target'"/>
            <xsl:with-param name="id" select="@id"/>
          </xsl:call-template>
        </td>
      </xsl:otherwise>
    </xsl:choose>
  </tr>
</xsl:template>

<xsl:template match="target" mode="trash">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <xsl:call-template name="observers-icon">
        <xsl:with-param name="type" select="'Target'"/>
      </xsl:call-template>
      <b><xsl:value-of select="name"/></b>
      <xsl:choose>
        <xsl:when test="comment != ''">
          <div class="comment">(<xsl:value-of select="comment"/>)</div>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </td>
    <td><xsl:value-of select="hosts"/></td>
    <td><xsl:value-of select="max_hosts"/></td>
    <td>
      <xsl:choose>
        <xsl:when test="port_list/trash = '1'">
          <xsl:value-of select="port_list/name"/>
          <div>(<xsl:value-of select="gsa:i18n ('in trashcan')"/>)</div>
        </xsl:when>
        <xsl:otherwise>
          <a href="/omp?cmd=get_port_list&amp;port_list_id={port_list/@id}&amp;token={/envelope/token}">
            <xsl:value-of select="port_list/name"/>
          </a>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <table>
        <xsl:if test="ssh_credential/@id != ''">
          <tr>
            <td width="35" valign="top">SSH:</td>
            <td>
              <xsl:choose>
                <xsl:when test="ssh_credential/trash = '1'">
                  <xsl:value-of select="ssh_credential/name"/>
                  <div>(<xsl:value-of select="gsa:i18n ('in trashcan')"/>)</div>
                </xsl:when>
                <xsl:when test="boolean (ssh_credential/permissions) and count (ssh_credential/permissions/permission) = 0">
                  <xsl:value-of select="ssh_credential/name"/>
                </xsl:when>
                <xsl:when test="gsa:may-op ('get_credentials')">
                  <a href="/omp?cmd=get_credential&amp;credential_id={ssh_credential/@id}&amp;token={/envelope/token}">
                    <xsl:value-of select="ssh_credential/name"/>
                  </a>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:value-of select="ssh_credential/name"/>
                </xsl:otherwise>
              </xsl:choose>
            </td>
          </tr>
        </xsl:if>
        <xsl:if test="smb_credential/@id != ''">
          <tr>
            <td width="35" valign="top">SMB:</td>
            <td>
              <xsl:choose>
                <xsl:when test="smb_credential/trash = '1'">
                  <xsl:value-of select="smb_credential/name"/>
                  <div>(<xsl:value-of select="gsa:i18n ('in trashcan')"/>)</div>
                </xsl:when>
                <xsl:when test="boolean (smb_credential/permissions) and count (smb_credential/permissions/permission) = 0">
                  <xsl:value-of select="smb_credential/name"/>
                </xsl:when>
                <xsl:when test="gsa:may-op ('get_credentials')">
                  <a href="/omp?cmd=get_credential&amp;credential_id={smb_credential/@id}&amp;token={/envelope/token}">
                    <xsl:value-of select="smb_credential/name"/>
                  </a>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:value-of select="smb_credential/name"/>
                </xsl:otherwise>
              </xsl:choose>
            </td>
          </tr>
        </xsl:if>
        <xsl:if test="esxi_credential/@id != ''">
          <tr>
            <td width="35" valign="top">ESXi:</td>
            <td>
              <xsl:choose>
                <xsl:when test="esxi_credential/trash = '1'">
                  <xsl:value-of select="esxi_credential/name"/>
                  <div>(<xsl:value-of select="gsa:i18n ('in trashcan')"/>)</div>
                </xsl:when>
                <xsl:when test="boolean (esxi_credential/permissions) and count (esxi_credential/permissions/permission) = 0">
                  <xsl:value-of select="esxi_credential/name"/>
                </xsl:when>
                <xsl:when test="gsa:may-op ('get_credentials')">
                  <a href="/omp?cmd=get_credential&amp;credential_id={esxi_credential/@id}&amp;token={/envelope/token}">
                    <xsl:value-of select="esxi_credential/name"/>
                  </a>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:value-of select="esxi_credential/name"/>
                </xsl:otherwise>
              </xsl:choose>
            </td>
          </tr>
        </xsl:if>
        <xsl:if test="snmp_credential/@id != ''">
          <tr>
            <td width="35" valign="top">SNMP:</td>
            <td>
              <xsl:choose>
                <xsl:when test="snmp_credential/trash = '1'">
                  <xsl:value-of select="snmp_credential/name"/>
                  <div>(<xsl:value-of select="gsa:i18n ('in trashcan')"/>)</div>
                </xsl:when>
                <xsl:when test="boolean (snmp_credential/permissions) and count (snmp_credential/permissions/permission) = 0">
                  <xsl:value-of select="snmp_credential/name"/>
                </xsl:when>
                <xsl:when test="gsa:may-op ('get_credentials')">
                  <a href="/omp?cmd=get_credential&amp;credential_id={snmp_credential/@id}&amp;token={/envelope/token}">
                    <xsl:value-of select="snmp_credential/name"/>
                  </a>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:value-of select="snmp_credential/name"/>
                </xsl:otherwise>
              </xsl:choose>
            </td>
          </tr>
        </xsl:if>
      </table>
    </td>
    <td class="table-actions">
      <xsl:choose>
        <xsl:when test="not (gsa:may-op ('restore'))"/>
        <xsl:when test="ssh_credential/trash = '1' or smb_credential/trash = '1' or esxi_credential/trash = '1' or snmp_credential/trash = '1' or port_list/trash = '1'">
          <xsl:variable name="resources_string">
            <xsl:if test="ssh_credential/trash = '1' or smb_credential/trash = '1' or esxi_credential/trash = '1' or snmp_credential/trash = '1'"><xsl:value-of select="gsa:i18n ('Credentials')"/></xsl:if>
            <xsl:if test="(ssh_credential/trash = '1' or smb_credential/trash = '1' or esxi_credential/trash = '1' or snmp_credential/trash = '1') and port_list/trash = '1'">
              <xsl:text> </xsl:text><xsl:value-of select="gsa:i18n ('and')"/><xsl:text> </xsl:text>
            </xsl:if>
            <xsl:if test="port_list/trash = '1'"><xsl:value-of select="gsa:i18n ('Port List')"/></xsl:if>
            <xsl:value-of select="gsa:i18n (' must be restored first.', 'Trashcan')"/>
          </xsl:variable>
          <img src="/img/restore_inactive.svg" alt="{gsa:i18n ('Restore')}" title="{$resources_string}"
               class="icon icon-sm"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="restore-icon">
            <xsl:with-param name="id" select="@id"/>
          </xsl:call-template>
        </xsl:otherwise>
      </xsl:choose>
      <xsl:choose>
        <xsl:when test="in_use='0'">
          <xsl:call-template name="trash-delete-icon">
            <xsl:with-param name="type" select="'target'"/>
            <xsl:with-param name="id" select="@id"/>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <img src="/img/delete_inactive.svg"
               alt="{gsa:i18n ('Delete')}"
               title="{gsa:i18n ('Target is still in use')}"
               class="icon icon-sm"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
  </tr>
</xsl:template>

<xsl:template match="target" mode="details">
  <div class="toolbar">
    <xsl:call-template name="details-header-icons">
      <xsl:with-param name="cap-type" select="'Target'"/>
      <xsl:with-param name="type" select="'target'"/>
    </xsl:call-template>
  </div>

  <div class="section-header">
    <xsl:call-template name="minor-details"/>
    <h1>
      <a href="/omp?cmd=get_targets&amp;token={/envelope/token}"
         title="{gsa:i18n ('Targets')}">
        <img class="icon icon-lg" src="/img/target.svg" alt="Targets"/>
      </a>
      <xsl:value-of select="gsa:i18n ('Target')"/>:
      <xsl:value-of select="name"/>
      <xsl:text> </xsl:text>
    </h1>
  </div>

  <div class="section-box">
    <table>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Comment')"/>:</td>
        <td><xsl:value-of select="comment"/></td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Hosts')"/>:</td>
        <td><xsl:value-of select="hosts"/></td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Exclude Hosts')"/>:</td>
        <td><xsl:value-of select="exclude_hosts"/></td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Reverse Lookup Only')"/>:</td>
        <td>
          <xsl:choose>
            <xsl:when test="reverse_lookup_only = 1"><xsl:value-of select="gsa:i18n ('Yes')"/></xsl:when>
            <xsl:otherwise><xsl:value-of select="gsa:i18n ('No')"/></xsl:otherwise>
          </xsl:choose>
        </td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Reverse Lookup Unify')"/>:</td>
        <td>
          <xsl:choose>
            <xsl:when test="reverse_lookup_unify = 1"><xsl:value-of select="gsa:i18n ('Yes')"/></xsl:when>
            <xsl:otherwise><xsl:value-of select="gsa:i18n ('No')"/></xsl:otherwise>
          </xsl:choose>
        </td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Maximum number of hosts')"/>:</td>
        <td><xsl:value-of select="max_hosts"/></td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Port List')"/>:</td>
        <td>
          <xsl:choose>
            <xsl:when test="gsa:may-op ('get_port_lists')">
              <xsl:choose>
                <xsl:when test="boolean (port_list/permissions) and count (port_list/permissions/permission) = 0">
                  <xsl:value-of select="gsa:i18n('Unavailable')"/>
                  <xsl:text> (</xsl:text>
                  <xsl:value-of select="gsa:i18n ('Name')"/>
                  <xsl:text>: </xsl:text>
                  <xsl:value-of select="port_list/name"/>
                  <xsl:text>, </xsl:text>
                  <xsl:value-of select="gsa:i18n ('ID')"/>: <xsl:value-of select="port_list/@id"/>
                  <xsl:text>)</xsl:text>
                </xsl:when>
                <xsl:otherwise>
                  <a href="/omp?cmd=get_port_list&amp;port_list_id={port_list/@id}&amp;token={/envelope/token}">
                    <xsl:value-of select="port_list/name"/>
                  </a>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="port_list/name"/>
            </xsl:otherwise>
          </xsl:choose>
        </td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Alive Test')"/>:</td>
        <td>
          <xsl:value-of select="gsa:i18n (alive_tests/text())"/>
        </td>
      </tr>
      <xsl:if test="gsa:may-op ('get_credentials') or string-length (ssh_credential/@id) &gt; 0">
        <tr>
          <td valign="top" width="175" colspan="2">
            <xsl:value-of select="gsa:i18n ('Credentials for authenticated checks')"/>:
          </td>
        </tr>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('SSH')"/>:</td>
          <td>
            <xsl:if test="string-length (ssh_credential/@id) &gt; 0">
              <xsl:choose>
                <xsl:when test="boolean (ssh_credential/permissions) and count (ssh_credential/permissions/permission) = 0">
                  <xsl:value-of select="gsa:i18n('Unavailable')"/>
                  <xsl:text> (</xsl:text>
                  <xsl:value-of select="gsa:i18n ('Name')"/>
                  <xsl:text>: </xsl:text>
                  <xsl:value-of select="ssh_credential/name"/>
                  <xsl:text>, </xsl:text>
                  <xsl:value-of select="gsa:i18n ('ID')"/>: <xsl:value-of select="ssh_credential/@id"/>
                  <xsl:text>)</xsl:text>
                </xsl:when>
                <xsl:when test="gsa:may-op ('get_credentials')">
                  <a href="/omp?cmd=get_credential&amp;credential_id={ssh_credential/@id}&amp;token={/envelope/token}">
                    <xsl:value-of select="ssh_credential/name"/>
                  </a>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:value-of select="ssh_credential/name"/>
                </xsl:otherwise>
              </xsl:choose>
              <xsl:text> </xsl:text>
              <xsl:value-of select="gsa:i18n ('on port', 'Target|Credential')"/>
              <xsl:text> </xsl:text>
              <xsl:value-of select="ssh_credential/port"/>
            </xsl:if>
          </td>
        </tr>
      </xsl:if>
      <xsl:if test="gsa:may-op ('get_credentials') or string-length (smb_credential/@id) &gt; 0">
        <tr>
          <td><xsl:value-of select="gsa:i18n ('SMB')"/>:</td>
          <td>
            <xsl:choose>
              <xsl:when test="boolean (smb_credential/permissions) and count (smb_credential/permissions/permission) = 0">
                <xsl:value-of select="gsa:i18n('Unavailable')"/>
                <xsl:text> (</xsl:text>
                <xsl:value-of select="gsa:i18n ('Name')"/>
                <xsl:text>: </xsl:text>
                <xsl:value-of select="smb_credential/name"/>
                <xsl:text>, </xsl:text>
                <xsl:value-of select="gsa:i18n ('ID')"/>: <xsl:value-of select="smb_credential/@id"/>
                <xsl:text>)</xsl:text>
              </xsl:when>
              <xsl:when test="gsa:may-op ('get_credentials')">
                <a href="/omp?cmd=get_credential&amp;credential_id={smb_credential/@id}&amp;token={/envelope/token}">
                  <xsl:value-of select="smb_credential/name"/>
                </a>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="smb_credential/name"/>
              </xsl:otherwise>
            </xsl:choose>
          </td>
        </tr>
      </xsl:if>
      <xsl:if test="gsa:may-op ('get_credentials') or string-length (esxi_credential/@id) &gt; 0">
        <tr>
          <td><xsl:value-of select="gsa:i18n ('ESXi')"/>:</td>
          <td>
            <xsl:choose>
              <xsl:when test="boolean (esxi_credential/permissions) and count (esxi_credential/permissions/permission) = 0">
                <xsl:value-of select="gsa:i18n('Unavailable')"/>
                <xsl:text> (</xsl:text>
                <xsl:value-of select="gsa:i18n ('Name')"/>
                <xsl:text>: </xsl:text>
                <xsl:value-of select="esxi_credential/name"/>
                <xsl:text>, </xsl:text>
                <xsl:value-of select="gsa:i18n ('ID')"/>: <xsl:value-of select="esxi_credential/@id"/>
                <xsl:text>)</xsl:text>
              </xsl:when>
              <xsl:when test="gsa:may-op ('get_credentials')">
                <a href="/omp?cmd=get_credential&amp;credential_id={esxi_credential/@id}&amp;token={/envelope/token}">
                  <xsl:value-of select="esxi_credential/name"/>
                </a>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="esxi_credential/name"/>
              </xsl:otherwise>
            </xsl:choose>
          </td>
        </tr>
      </xsl:if>
      <xsl:if test="gsa:may-op ('get_credentials') or string-length (snmp_credential/@id) &gt; 0">
        <tr>
          <td><xsl:value-of select="gsa:i18n ('SNMP')"/>:</td>
          <td>
            <xsl:choose>
              <xsl:when test="boolean (snmp_credential/permissions) and count (snmp_credential/permissions/permission) = 0">
                <xsl:value-of select="gsa:i18n('Unavailable')"/>
                <xsl:text> (</xsl:text>
                <xsl:value-of select="gsa:i18n ('Name')"/>
                <xsl:text>: </xsl:text>
                <xsl:value-of select="snmp_credential/name"/>
                <xsl:text>, </xsl:text>
                <xsl:value-of select="gsa:i18n ('ID')"/>: <xsl:value-of select="snmp_credential/@id"/>
                <xsl:text>)</xsl:text>
              </xsl:when>
              <xsl:when test="gsa:may-op ('get_credentials')">
                <a href="/omp?cmd=get_credential&amp;credential_id={snmp_credential/@id}&amp;token={/envelope/token}">
                  <xsl:value-of select="snmp_credential/name"/>
                </a>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="snmp_credential/name"/>
              </xsl:otherwise>
            </xsl:choose>
          </td>
        </tr>
      </xsl:if>
    </table>
  </div>

  <div class="section-header">
    <a href="#" class="toggle-action-icon icon icon-sm icon-action"
      data-target="#using-box" data-name="Tasks using this Target"
      data-variable="using-box--collapsed">
        <img src="/img/fold.svg"/>
    </a>
    <h2>
      <a href="/omp?cmd=get_tasks&amp;token={/envelope/token}"
         title="{gsa:i18n ('Tasks')}">
        <img class="icon icon-sm" src="/img/task.svg" alt="Tasks"/>
      </a>
      <xsl:value-of select="gsa:i18n ('Tasks using this Target')"/>
      <xsl:text> </xsl:text>
      <xsl:choose>
        <xsl:when test="count(tasks/task) != 0">
          (<xsl:value-of select="count(tasks/task)"/>)
        </xsl:when>
        <xsl:otherwise>
          (<xsl:value-of select="gsa:i18n ('none')"/>)
        </xsl:otherwise>
      </xsl:choose>
    </h2>
  </div>

  <div class="section-box" id="using-box">
    <table class="gbntable">
      <tr class="gbntablehead2">
        <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
      </tr>
      <xsl:for-each select="tasks/task">
        <tr class="{gsa:table-row-class(position())}">
          <xsl:choose>
            <xsl:when test="boolean (permissions) and count (permissions/permission) = 0">
              <td><xsl:value-of select="name"/> (<xsl:value-of select="gsa:i18n('Unavailable')"/>, <xsl:value-of select="gsa:i18n('UUID')"/>: <xsl:value-of select="@id"/>)</td>
            </xsl:when>
            <xsl:otherwise>
              <td>
                <a href="/omp?cmd=get_task&amp;task_id={@id}&amp;token={/envelope/token}" title="{gsa:i18n ('Details')}">
                  <xsl:value-of select="name"/>
                </a>
              </td>
            </xsl:otherwise>
          </xsl:choose>
        </tr>
      </xsl:for-each>
    </table>
  </div>

  <xsl:call-template name="user-tags-window">
    <xsl:with-param name="resource_type" select="'target'"/>
  </xsl:call-template>

  <xsl:call-template name="resource-permissions-window">
    <xsl:with-param name="resource_type" select="'target'"/>
    <xsl:with-param name="permissions" select="../../permissions/get_permissions_response"/>
    <xsl:with-param name="related">
      <xsl:if test="port_list/@id != ''">
        <port_list id="{port_list/@id}"/>
      </xsl:if>
      <xsl:if test="ssh_credential/@id != ''">
        <credential id="{ssh_credential/@id}"/>
      </xsl:if>
      <xsl:if test="smb_credential/@id != '' and smb_credential/@id != ssh_credential/@id">
        <credential id="{smb_credential/@id}"/>
      </xsl:if>
      <xsl:if test="esxi_credential/@id != '' and esxi_credential/@id != ssh_credential/@id and esxi_credential/@id != smb_credential/@id">
        <credential id="{esxi_credential/@id}"/>
      </xsl:if>
      <xsl:if test="snmp_credential/@id != '' and snmp_credential/@id != ssh_credential/@id and snmp_credential/@id != smb_credential/@id and snmp_credential/@id != esxi_credential/@id">
        <credential id="{snmp_credential/@id}"/>
      </xsl:if>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!--     GET_TARGET -->

<xsl:template match="get_target">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="commands_response/delete_target_response"/>
  <xsl:apply-templates select="delete_tag_response"/>
  <xsl:apply-templates select="create_tag_response"/>
  <xsl:apply-templates select="modify_tag_response"/>
  <xsl:apply-templates select="get_targets_response/target" mode="details"/>
</xsl:template>

<!--     GET_TARGETS -->

<xsl:template match="get_targets">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="delete_target_response"/>
  <xsl:apply-templates select="create_filter_response"/>
  <xsl:apply-templates select="create_target_response"/>
  <!-- The for-each makes the get_targets_response the current node. -->
  <xsl:for-each select="get_targets_response | commands_response/get_targets_response">
    <xsl:choose>
      <xsl:when test="substring(@status, 1, 1) = '4' or substring(@status, 1, 1) = '5'">
        <xsl:call-template name="command_result_dialog">
          <xsl:with-param name="operation">
            Get Targets
          </xsl:with-param>
          <xsl:with-param name="status">
            <xsl:value-of select="@status"/>
          </xsl:with-param>
          <xsl:with-param name="msg">
            <xsl:value-of select="@status_text"/>
          </xsl:with-param>
        </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
        <xsl:call-template name="html-targets-table"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:for-each>
</xsl:template>


<!-- BEGIN CONFIGS MANAGEMENT -->

<xsl:template name="html-create-config-form">
  <div class="edit-dialog">
    <div class="title">
      <xsl:value-of select="gsa:i18n ('New Scan Config')"/>
    </div>
    <div class="content">
      <form action="/omp" method="post" enctype="multipart/form-data"
        class="form-horizontal">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="create_config"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden" name="next" value="get_config"/>
        <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
        <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Name')"/></label>
          <div class="col-10">
            <input type="text" name="name" value="unnamed" size="30"
              class="form-control" maxlength="80"/>
          </div>
        </div>
        <div class="form-group">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Comment')"/>
          </label>
          <div class="col-10">
            <input type="text" name="comment" size="30" maxlength="400"
              class="form-control"/>
          </div>
        </div>
        <div class="form-group">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Base', 'Scan Config')"/>
          </label>
          <div class="col-10">
            <div class="radio">
              <label>
                <input type="radio" name="base"
                        value="085569ce-73ed-11df-83c3-002264764cea"
                        checked="1"/>
                <xsl:value-of select="gsa:i18n ('Empty, static and fast', 'Scan Config')"/>
              </label>
            </div>
            <div class="radio">
              <label>
                <input type="radio" name="base"
                        value="daba56c8-73ec-11df-a475-002264764cea"/>
                <xsl:value-of select="gsa:i18n ('Full and fast', 'Scan Config')"/>
              </label>
            </div>
            <xsl:if test="get_scanners_response/scanner[type != '2' and type != '3']">
              <div class="radio">
                <label>
                  <input type="radio" name="base" value="0"/>
                  <select name="scanner_id">
                    <xsl:for-each select="get_scanners_response/scanner[type != '2' and type != '3']">
                      <option value="{@id}"><xsl:value-of select="name"/></option>
                    </xsl:for-each>
                  </select>
                </label>
              </div>
            </xsl:if>
          </div>
        </div>
      </form>
    </div>
  </div>
</xsl:template>

<xsl:template name="html-import-config-form">
  <div class="gb_window">
    <div class="gb_window_part_left"></div>
    <div class="gb_window_part_right"></div>
    <div class="gb_window_part_center">
      <xsl:value-of select="gsa:i18n ('Import Scan Config')"/>
      <a href="/help/new_config.html?token={/envelope/token}#importconfig"
        class="icon icon-sm"
        title="{concat(gsa:i18n('Help'),': ',gsa:i18n('Import Scan Config'))}">
        <img src="/img/help.svg"/>
      </a>
      <a href="/omp?cmd=get_configs&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
        class="icon icon-sm"
        title="{gsa:i18n ('Scan Configs')}">
        <img src="/img/list.svg" alt="{gsa:i18n ('Scan Configs')}"/>
      </a>
    </div>
    <div class="gb_window_part_content">
      <form action="/omp" method="post" enctype="multipart/form-data">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="import_config"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden" name="next" value="get_config"/>
        <table cellspacing="0" cellpadding="3" width="100%">
          <tr>
            <td valign="top" width="125">
              <xsl:value-of select="gsa:i18n ('Import XML config')"/>
            </td>
            <td><input type="file" name="xml_file" size="30"/></td>
          </tr>
          <tr>
            <td colspan="2" style="text-align:right;">
              <input type="submit" name="submit" value="{gsa:i18n('Import Scan Config')}"/>
            </td>
          </tr>
        </table>
      </form>
    </div>
  </div>
</xsl:template>

<xsl:template match="new_config">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="create_config_response"/>
  <xsl:apply-templates select="commands_response/delete_config_response"/>
  <xsl:call-template name="html-create-config-form"/>
</xsl:template>

<xsl:template match="upload_config">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="create_config_response"/>
  <xsl:apply-templates select="commands_response/delete_config_response"/>
  <xsl:call-template name="html-import-config-form"/>
</xsl:template>

<xsl:template name="edit-config-family">
  <div class="edit-dialog">
    <div class="title">
      <xsl:value-of select="gsa:i18n ('Edit Scan Config Family')"/>
    </div>
    <div class="content">
      <xsl:variable name="config_id" select="config/@id"/>
      <xsl:variable name="config_name" select="config/name"/>
      <xsl:variable name="family" select="config/family"/>

      <table>
        <tr><td><xsl:value-of select="gsa:i18n ('Config')"/>:</td><td><xsl:value-of select="$config_name"/></td></tr>
        <tr><td><b><xsl:value-of select="gsa:i18n ('Family')"/>:</b></td><td><b><xsl:value-of select="$family"/></b></td></tr>
      </table>

      <h1><xsl:value-of select="gsa:i18n ('Edit Network Vulnerability Tests')"/></h1>
      <form action="" method="post" enctype="multipart/form-data">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="save_config_family"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden" name="config_id" value="{$config_id}"/>
        <input type="hidden" name="name" value="{$config_name}"/>
        <input type="hidden" name="family" value="{$family}"/>
        <table class="gbntable">
          <tr class="gbntablehead2">
            <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
            <td><xsl:value-of select="gsa:i18n ('OID')"/></td>
            <td><xsl:value-of select="gsa:i18n ('Severity')"/></td>
            <td><xsl:value-of select="gsa:i18n ('Timeout')"/></td>
            <td><xsl:value-of select="gsa:i18n ('Prefs')"/></td>
            <td><xsl:value-of select="gsa:i18n ('Selected', 'NVTs')"/></td>
            <td><xsl:value-of select="gsa:i18n ('Actions')"/></td>
          </tr>
          <xsl:for-each select="all/get_nvts_response/nvt" >
            <xsl:variable name="current_name" select="name/text()"/>
            <xsl:variable name="id" select="@oid"/>

            <tr class="{gsa:table-row-class(position())}">
              <td>
                <xsl:value-of select="$current_name"/>
              </td>
              <td>
                <xsl:value-of select="@oid"/>
              </td>
              <td>
                <xsl:call-template name="severity-bar">
                  <xsl:with-param name="cvss" select="cvss_base"/>
                </xsl:call-template>
              </td>
              <td>
                <xsl:variable
                  name="timeout"
                  select="timeout"/>
                <xsl:choose>
                  <xsl:when test="string-length($timeout) &gt; 0">
                    <xsl:value-of select="$timeout"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <xsl:value-of select="gsa:i18n ('default', 'Timeout')"/>
                  </xsl:otherwise>
                </xsl:choose>
              </td>
              <td>
                <xsl:choose>
                  <xsl:when test="preference_count&gt;0">
                    <xsl:value-of select="preference_count"/>
                  </xsl:when>
                  <xsl:otherwise>
                  </xsl:otherwise>
                </xsl:choose>
              </td>
              <td style="text-align:center;">
                <xsl:choose>
                  <xsl:when test="../../../get_nvts_response/nvt[@oid=$id]">
                    <input type="checkbox" name="nvt:{@oid}" value="1"
                      checked="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="checkbox" name="nvt:{@oid}" value="1"/>
                  </xsl:otherwise>
                </xsl:choose>
              </td>
              <td>
                <a href="/omp?cmd=edit_config_nvt&amp;oid={@oid}&amp;config_id={$config_id}&amp;name={$config_name}&amp;family={$family}&amp;token={/envelope/token}"
                  title="{gsa:i18n ('Select and Edit NVT Details')}"
                  data-cmd="edit_config_nvt" data-type="config" data-id="{$config_id}"
                  data-extra="name={$config_name}&amp;family={$family}&amp;oid={@oid}" data-reload="parent"
                  class="edit-action-icon icon icon-sm">
                  <img src="/img/edit.svg" alt="{gsa:i18n ('Edit', 'Action Verb')}"/>
                </a>
              </td>
            </tr>
          </xsl:for-each>
        </table>
      </form>

      <xsl:value-of select="gsa-i18n:strformat (gsa:i18n ('Selected %1 of %2 total NVTs'), count(get_nvts_response/nvt), count(all/get_nvts_response/nvt))"/>
    </div>
  </div>
</xsl:template>

<xsl:template name="html-config-family-table">
  <div class="toolbar">
    <a href="/help/config_family_details.html?token={/envelope/token}"
      class="icon icon-sm"
      title="{concat(gsa:i18n('Help'),': ',gsa:i18n('Scan Configs'),' (',gsa:i18n('Scan Config Family Details'),')')}">
      <img src="/img/help.svg"/>
    </a>
    <a href="?cmd=get_config&amp;config_id={config/@id}&amp;token={/envelope/token}"
      title="{gsa:i18n ('Scan Config')}"
      class="icon icon-sm">
      <img src="/img/list.svg" alt="{gsa:i18n ('Scan Config')}"/>
    </a>
  </div>
  <div class="section-header">
    <div class="section-header-info">
      <table>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Config ID')"/>:</td>
          <td><xsl:value-of select="config/@id"/></td>
        </tr>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Created', 'Date')"/>:</td>
          <td><xsl:value-of select="gsa:long-time (get_nvts_response/nvt/creation_time)"/></td>
        </tr>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Modified', 'Date')"/>:</td>
          <td><xsl:value-of select="gsa:long-time (get_nvts_response/nvt/modification_time)"/></td>
        </tr>
      </table>
    </div>
    <h1>
      <a href="/omp?cmd=get_configs&amp;token={/envelope/token}"
         title="{gsa:i18n ('Scan Configs')}">
        <img class="icon icon-lg" src="/img/config.svg" alt="Scan Configs"/>
      </a>
      <xsl:value-of select="gsa:i18n ('Scan Config Family Details')"/>
    </h1>
  </div>
  <div class="section-box">
    <xsl:variable name="config_id" select="config/@id"/>
    <xsl:variable name="config_name" select="config/name"/>
    <xsl:variable name="family" select="config/family"/>

    <table>
      <tr><td><xsl:value-of select="gsa:i18n ('Config')"/>:</td><td><xsl:value-of select="$config_name"/></td></tr>
      <tr><td><b><xsl:value-of select="gsa:i18n ('Family')"/>:</b></td><td><b><xsl:value-of select="$family"/></b></td></tr>
    </table>

    <h1><xsl:value-of select="gsa:i18n ('Network Vulnerability Tests')"/></h1>

    <table class="gbntable">
      <tr class="gbntablehead2">
        <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
        <td><xsl:value-of select="gsa:i18n ('OID')"/></td>
        <td><xsl:value-of select="gsa:i18n ('Severity')"/></td>
        <td><xsl:value-of select="gsa:i18n ('Timeout')"/></td>
        <td><xsl:value-of select="gsa:i18n ('Prefs')"/></td>
      </tr>
      <xsl:for-each select="get_nvts_response/nvt" >
        <xsl:variable name="current_name" select="name/text()"/>

        <tr class="{gsa:table-row-class(position())}">
          <td>
            <a href="/omp?cmd=get_config_nvt&amp;oid={@oid}&amp;config_id={$config_id}&amp;name={$config_name}&amp;family={$family}&amp;token={/envelope/token}"
              title="{gsa:i18n ('NVT Details')}" style="margin-left:3px;">
              <xsl:value-of select="$current_name"/>
            </a>
          </td>
          <td>
            <xsl:value-of select="@oid"/>
          </td>
          <td>
            <xsl:call-template name="severity-bar">
              <xsl:with-param name="cvss" select="cvss_base"/>
            </xsl:call-template>
          </td>
          <td>
            <xsl:choose>
              <xsl:when test="string-length(timeout) &gt; 0">
                <xsl:value-of select="timeout"/>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="gsa:i18n ('default', 'Timeout')"/>
              </xsl:otherwise>
            </xsl:choose>
          </td>
          <td style="text-align:center;">
            <xsl:choose>
              <xsl:when test="preference_count&gt;0">
                <xsl:value-of select="preference_count"/>
              </xsl:when>
              <xsl:otherwise>
              </xsl:otherwise>
            </xsl:choose>
          </td>
        </tr>
      </xsl:for-each>
      <tr>
        <td>
          <xsl:value-of select="gsa:i18n ('Total', 'NVTs')"/>:
          <xsl:value-of select="count(get_nvts_response/nvt)"/>
        </td>
        <td></td>
        <td></td>
        <td></td>
        <td></td>
      </tr>
    </table>
  </div>
</xsl:template>

<!--     CONFIG PREFERENCES -->

<xsl:template name="preference" match="preference">
  <xsl:param name="config_id"></xsl:param>
  <xsl:param name="config_name"></xsl:param>
  <xsl:param name="edit"></xsl:param>

  <tr class="{gsa:table-row-class(position())}">
    <td>
      <xsl:choose>
        <xsl:when test="string-length($config_id) &gt; 0">
          <xsl:choose>
            <xsl:when test="string-length($edit) &gt; 0">
              <a href="/omp?cmd=get_config_nvt&amp;oid={nvt/@oid}&amp;config_id={$config_id}&amp;name={$config_name}&amp;family={nvt/family}&amp;token={/envelope/token}"
                target="_blank"
                title="{gsa:i18n ('Scan Config NVT Details')}">
                <xsl:value-of select="nvt/name"/>
              </a>
            </xsl:when>
            <xsl:otherwise>
              <a href="/omp?cmd=get_config_nvt&amp;oid={nvt/@oid}&amp;config_id={$config_id}&amp;name={$config_name}&amp;family={nvt/family}&amp;token={/envelope/token}"
                title="{gsa:i18n ('Scan Config NVT Details')}" style="margin-left:3px;">
                <xsl:value-of select="nvt/name"/>
              </a>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="nvt/name"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td><xsl:value-of select="name"/></td>
    <td>
      <xsl:choose>
        <xsl:when test="type='file' and string-length(value) &gt; 0">
          <i><xsl:value-of select="gsa:i18n ('File attached.')"/></i>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="value"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:if test="string-length($edit) &gt; 0">
        <a href="/omp?cmd=edit_config_nvt&amp;oid={nvt/@oid}&amp;config_id={$config_id}&amp;name={$config_name}&amp;family={nvt/family}&amp;token={/envelope/token}"
          class="edit-action-icon icon icon-sm" data-reload="parent" data-type="config_nvt"
          data-extra="oid={nvt/@oid}&amp;config_id={$config_id}&amp;name={$config_name}&amp;family={nvt/family}"
          title="{gsa:i18n ('Edit Scan Config NVT Details')}">
          <img src="/img/edit.svg" alt="{gsa:i18n ('Edit', 'Action Verb')}"/>
        </a>
      </xsl:if>
      <xsl:if test="type='file' and string-length(value) &gt; 0">
        <a href="/omp?cmd=export_preference_file&amp;config_id={$config_id}&amp;oid={nvt/@oid}&amp;preference_name={name}&amp;token={/envelope/token}"
          title="{gsa:i18n ('Export File')}"
          class="icon icon-sm">
          <img src="/img/download.svg" alt="{gsa:i18n ('Export File')}"/>
        </a>
      </xsl:if>
    </td>
  </tr>
</xsl:template>

<xsl:template name="preference-details" match="preference" mode="details">
  <xsl:param name="config" select=".."/>

  <tr class="{gsa:table-row-class(position())}">
    <td><xsl:value-of select="hr_name"/></td>
    <xsl:if test="$config != ''">
      <td>
        <xsl:choose>
          <xsl:when test="type='osp_boolean' and value = '0'">False</xsl:when>
          <xsl:when test="type='osp_boolean'">True</xsl:when>
          <xsl:when test="type='file' and string-length(value) &gt; 0">
            <i><xsl:value-of select="gsa:i18n ('File attached.')"/></i>
          </xsl:when>
          <xsl:when test="type='osp_credential_up'">
            <xsl:variable name="value">
              <xsl:value-of select="value"/>
            </xsl:variable>
            <xsl:choose>
              <xsl:when test="$value=0">
                <option value="0">
                  <xsl:value-of select="gsa:i18n ('Use target SSH credentials')"/>
                </option>
              </xsl:when>
              <xsl:otherwise>
                <a href="/omp?cmd=get_credential&amp;credential_id={$value}&amp;token={/envelope/token}"
                   title="{gsa:view_details_title ('Credential', $value)}">
                  <xsl:value-of select="../../../../get_credentials_response/credential[@id=$value]/name"/>
                </a>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="value"/>
          </xsl:otherwise>
        </xsl:choose>
      </td>
    </xsl:if>
    <td>
      <xsl:choose>
        <xsl:when test="type='file' and string-length(value) &gt; 0">
          <i><xsl:value-of select="gsa:i18n ('File attached.')"/></i>
        </xsl:when>
        <xsl:when test="type='osp_ovaldef_file'">OVAL Definitions files list.</xsl:when>
        <xsl:when test="type='osp_selection'">List.</xsl:when>
        <xsl:when test="type='osp_boolean' and default = '0'">False</xsl:when>
        <xsl:when test="type='osp_boolean'">True</xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="default"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:if test="type='file' and string-length(value) &gt; 0">
        <a href="/omp?cmd=export_preference_file&amp;config_id={$config/@id}&amp;oid={nvt/@oid}&amp;preference_name={name}&amp;token={/envelope/token}"
          title="{gsa:i18n ('Export File')}"
          class="icon icon-sm">
          <img src="/img/download.svg" alt="{gsa:i18n ('Export File')}"/>
        </a>
      </xsl:if>
    </td>
  </tr>
</xsl:template>

<xsl:template match="preference"
              name="edit-config-preference"
              mode="edit-details">
  <xsl:param name="config"></xsl:param>
  <xsl:param name="for_config_details"></xsl:param>
  <xsl:param name="family"></xsl:param>
  <xsl:param name="nvt"></xsl:param>

  <tr class="{gsa:table-row-class(position())}">
    <xsl:if test="$for_config_details">
      <td><xsl:value-of select="nvt/name"/></td>
    </xsl:if>
    <td><xsl:value-of select="hr_name"/></td>
    <td>
      <xsl:choose>

        <!-- OSP config types. -->
        <xsl:when test="type='osp_ovaldef_file' or type='osp_selection'">
          <xsl:variable name="value">
            <xsl:value-of select="value"/>
          </xsl:variable>
          <select name="osp_pref_{name}">
            <xsl:for-each select="str:split (default, '|')">
              <xsl:call-template name="opt">
                <xsl:with-param name="value" select="."/>
                <xsl:with-param name="select-value" select="$value"/>
              </xsl:call-template>
            </xsl:for-each>
          </select>
        </xsl:when>
        <xsl:when test="type='osp_integer'">
          <input type="text" name="osp_pref_{name}"
                 value="{value}" size="30" maxlength="400"/>
        </xsl:when>
        <xsl:when test="type='osp_string'">
          <input type="text" name="osp_pref_{name}"
                 value="{value}" size="30" maxlength="400"/>
        </xsl:when>
        <xsl:when test="type='osp_file'">
          <input type="file" name="osp_pref_{name}" value="{value}"/>
        </xsl:when>
        <xsl:when test="type='osp_boolean'">
          <select name="osp_pref_{name}">
            <xsl:call-template name="opt">
              <xsl:with-param name="value" select="'1'"/>
              <xsl:with-param name="content" select="'True'"/>
              <xsl:with-param name="select-value" select="value"/>
            </xsl:call-template>
            <xsl:call-template name="opt">
              <xsl:with-param name="value" select="'0'"/>
              <xsl:with-param name="content" select="'False'"/>
              <xsl:with-param name="select-value" select="value"/>
            </xsl:call-template>
          </select>
        </xsl:when>
        <xsl:when test="type='osp_credential_up'">
          <xsl:variable name="value" select="value"/>
          <select name="osp_pref_{name}">
            <option value="0">
              <xsl:value-of select="gsa:i18n ('Use target SSH credentials')"/>
            </option>
            <xsl:for-each select="../../../../get_credentials_response/credential[type='up']">
              <xsl:call-template name="opt">
                <xsl:with-param name="content" select="name"/>
                <xsl:with-param name="value" select="@id"/>
                <xsl:with-param name="select-value" select="$value"/>
              </xsl:call-template>
            </xsl:for-each>
          </select>
        </xsl:when>

        <!-- Classic config types. -->
        <xsl:when test="type='checkbox'">
          <xsl:choose>
            <xsl:when test="value='yes'">
              <label>
                <input type="radio" name="preference:{nvt/name}[checkbox]:{name}"
                       value="yes" checked="1"/>
                yes
              </label>
              <label>
                <input type="radio" name="preference:{nvt/name}[checkbox]:{name}"
                       value="no"/>
                no
              </label>
            </xsl:when>
            <xsl:otherwise>
              <label>
                <input type="radio" name="preference:{nvt/name}[checkbox]:{name}"
                       value="yes"/>
                yes
              </label>
              <label>
                <input type="radio" name="preference:{nvt/name}[checkbox]:{name}"
                       value="no" checked="1"/>
                no
              </label>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:when>
        <xsl:when test="type='password'">
          <label>
            <input type="checkbox" name="password:{nvt/name}[password]:{name}"
                   value="yes"/>
            <xsl:value-of select="gsa:i18n ('Replace existing value with', 'Auth Data|Password')"/>:
          </label>
          <input type="password" autocomplete="off"
                 name="preference:{nvt/name}[password]:{name}"
                 value="{value}" size="30" maxlength="40"/>
        </xsl:when>
        <xsl:when test="type='file'">
          <label>
            <input type="checkbox" name="file:{nvt/name}[file]:{name}"
                   value="yes"/>
            <xsl:choose>
              <xsl:when test="string-length(value) &gt; 0">
                <xsl:value-of select="gsa:i18n ('Replace existing file with')"/>:
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="gsa:i18n ('Upload file')"/>:
              </xsl:otherwise>
            </xsl:choose>
          </label>
          <input type="file" name="preference:{nvt/name}[file]:{name}" size="30"/>
        </xsl:when>
        <xsl:when test="type='entry'">
          <input type="text" name="preference:{nvt/name}[entry]:{name}"
                 value="{value}" size="30" maxlength="400"/>
        </xsl:when>
        <xsl:when test="type='radio'">
          <label>
            <input type="radio" name="preference:{nvt/name}[radio]:{name}"
                   value="{value}" checked="1"/>
            <xsl:value-of select="value"/>
          </label>
          <xsl:for-each select="alt">
            <div>
              <label>
                <input type="radio"
                      name="preference:{../nvt/name}[radio]:{../name}"
                      value="{text()}"/>
                <xsl:value-of select="."/>
              </label>
            </div>
          </xsl:for-each>
        </xsl:when>
        <xsl:when test="type=''">
          <xsl:choose>
            <xsl:when test="name='ping_hosts' or name='reverse_lookup' or name='unscanned_closed' or name='nasl_no_signature_check' or name='ping_hosts' or name='reverse_lookup' or name='unscanned_closed_udp' or name='auto_enable_dependencies' or name='kb_dont_replay_attacks' or name='kb_dont_replay_denials' or name='kb_dont_replay_info_gathering' or name='kb_dont_replay_scanners' or name='kb_restore' or name='log_whole_attack' or name='only_test_hosts_whose_kb_we_dont_have' or name='only_test_hosts_whose_kb_we_have' or name='optimize_test' or name='safe_checks' or name='save_knowledge_base' or name='silent_dependencies' or name='slice_network_addresses' or name='use_mac_addr' or name='drop_privileges' or name='network_scan' or name='report_host_details'">
              <xsl:choose>
                <xsl:when test="value='yes'">
                  <label>
                    <input type="radio" name="preference:scanner[scanner]:{name}"
                           value="yes" checked="1"/>
                    yes
                  </label>
                  <label>
                    <input type="radio" name="preference:scanner[scanner]:{name}"
                           value="no"/>
                    no
                  </label>
                </xsl:when>
                <xsl:otherwise>
                  <label>
                    <input type="radio" name="preference:scanner[scanner]:{name}"
                           value="yes"/>
                    yes
                  </label>
                  <label>
                    <input type="radio" name="preference:scanner[scanner]:{name}"
                           value="no" checked="1"/>
                    no
                  </label>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:when>
            <xsl:otherwise>
              <input type="text"
                     name="preference:scanner[scanner]:{name}"
                     value="{value}"
                     size="30"
                     maxlength="400"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:when>
        <xsl:otherwise>
          <input type="text"
                 name="preference:{nvt/name}[{type}]:{name}"
                 value="{value}"
                 size="30"
                 maxlength="400"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="type = 'osp_boolean' and default = '0'">False</xsl:when>
        <xsl:when test="type = 'osp_boolean'">True</xsl:when>
        <xsl:when test="type = 'osp_file'">
          <xsl:if test="string-length(default) &gt; 0">Uploaded file</xsl:if>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="default"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:if test="$for_config_details">
        <a href="/omp?cmd=edit_config_nvt&amp;oid={nvt/@oid}&amp;config_id={$config/@id}&amp;family={$family}&amp;token={/envelope/token}"
          title="{gsa:i18n ('Edit NVT Details')}"
          class="icon icon-sm">
          <img src="/img/edit.svg" alt="{gsa:i18n ('Edit', 'Action Verb')}"/>
        </a>
      </xsl:if>
      <xsl:choose>
        <xsl:when test="$config and type='file' and (string-length(value) &gt; 0)">
          <a href="/omp?cmd=export_preference_file&amp;config_id={$config/@id}&amp;oid={nvt/@oid}&amp;preference_name={name}&amp;token={/envelope/token}"
            title="{gsa:i18n ('Export File')}"
            class="icon icon-sm">
            <img src="/img/download.svg" alt="{gsa:i18n ('Export File')}"/>
          </a>
        </xsl:when>
        <xsl:otherwise>
        </xsl:otherwise>
      </xsl:choose>
    </td>
  </tr>
</xsl:template>

<xsl:template match="preferences" name="preferences">
  <xsl:param name="config_id"></xsl:param>
  <xsl:param name="config_name"></xsl:param>
  <xsl:param name="edit"></xsl:param>
  <div id="nvt-test-preferences">
    <table class="gbntable">
      <tr class="gbntablehead2">
        <td><xsl:value-of select="gsa:i18n ('NVT')"/></td>
        <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
        <td><xsl:value-of select="gsa:i18n ('Value')"/></td>
        <td style="width: {gsa:actions-width (1)}px"><xsl:value-of select="gsa:i18n ('Actions')"/></td>
      </tr>
      <xsl:for-each select="preference[string-length(./nvt)&gt;0]">
        <xsl:call-template name="preference">
          <xsl:with-param name="config_id" select="$config_id"/>
          <xsl:with-param name="config_name" select="$config_name"/>
          <xsl:with-param name="edit" select="$edit"/>
        </xsl:call-template>
      </xsl:for-each>
    </table>
  </div>
</xsl:template>

<xsl:template name="preferences-details" match="preferences" mode="details">
  <xsl:param name="config"></xsl:param>
  <div id="preferences">
    <table class="gbntable">
      <tr class="gbntablehead2">
        <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
        <xsl:if test="$config != ''">
          <td><xsl:value-of select="gsa:i18n ('Current Value')"/></td>
        </xsl:if>
        <td><xsl:value-of select="gsa:i18n ('Default Value')"/></td>
        <td><xsl:value-of select="gsa:i18n ('Actions')"/></td>
      </tr>

      <!-- Special case the NVT timeout. -->
      <tr class="even">
        <td><xsl:value-of select="gsa:i18n ('Timeout')"/></td>
        <td>
          <xsl:choose>
            <xsl:when test="string-length(timeout) &gt; 0">
              <xsl:value-of select="timeout"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="gsa:i18n ('default', 'Timeout')"/>
            </xsl:otherwise>
          </xsl:choose>
        </td>
        <xsl:if test="$config != ''">
          <td><xsl:value-of select="gsa:i18n ('default', 'Timeout')"/></td>
        </xsl:if>
        <td></td>
      </tr>

      <xsl:for-each select="preference">
        <xsl:call-template name="preference-details">
          <xsl:with-param name="config" select="$config"/>
        </xsl:call-template>
      </xsl:for-each>
    </table>
  </div>
</xsl:template>

<xsl:template name="preferences-edit-details">
  <xsl:param name="config"></xsl:param>
  <div id="preferences">
    <table class="gbntable">
      <tr class="gbntablehead2">
        <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
        <td><xsl:value-of select="gsa:i18n ('New Value')"/></td>
        <td><xsl:value-of select="gsa:i18n ('Default Value')"/></td>
        <td><xsl:value-of select="gsa:i18n ('Actions')"/></td>
      </tr>

      <!-- Special case the NVT timeout. -->
      <tr class="even">
        <td><xsl:value-of select="gsa:i18n ('Timeout')"/></td>
        <td>
          <label>
            <xsl:choose>
              <xsl:when test="string-length(timeout) &gt; 0">
                <input type="radio"
                       name="timeout"
                       value="0"/>
              </xsl:when>
              <xsl:otherwise>
                <input type="radio"
                       name="timeout"
                       value="0"
                       checked="1"/>
              </xsl:otherwise>
            </xsl:choose>
            <xsl:value-of select="gsa:i18n ('Apply default timeout')"/>
          </label>
          <div>
            <xsl:choose>
              <xsl:when test="string-length(timeout) &gt; 0">
                <input type="radio"
                      name="timeout"
                      value="1"
                      checked="1"/>
              </xsl:when>
              <xsl:otherwise>
                <input type="radio"
                      name="timeout"
                      value="1"/>
              </xsl:otherwise>
            </xsl:choose>
            <input type="text"
                  name="preference:scanner[scanner]:timeout.{../@oid}"
                  value="{timeout}"
                  size="30"
                  maxlength="400"/>
          </div>
        </td>
        <td></td>
        <td></td>
      </tr>

      <xsl:for-each select="preference">
        <xsl:call-template name="edit-config-preference">
          <xsl:with-param name="config" select="$config"/>
        </xsl:call-template>
      </xsl:for-each>
    </table>
  </div>
</xsl:template>

<xsl:template match="preferences" mode="scanner">
  <div id="preferences">
    <table class="gbntable">
      <tr class="gbntablehead2">
        <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
        <td><xsl:value-of select="gsa:i18n ('Current Value')"/></td>
        <td><xsl:value-of select="gsa:i18n ('Default Value')"/></td>
        <td><xsl:value-of select="gsa:i18n ('Actions')"/></td>
      </tr>
      <xsl:apply-templates
        select="preference[string-length(nvt)=0]"
        mode="details"/>
    </table>
  </div>
</xsl:template>

<xsl:template match="preferences" mode="edit-scanner-details">
  <div id="scanner-preferences">
    <table class="table-form">
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
        <td><xsl:value-of select="gsa:i18n ('New Value')"/></td>
        <td><xsl:value-of select="gsa:i18n ('Default Value')"/></td>
        <td><xsl:value-of select="gsa:i18n ('Actions')"/></td>
      </tr>
      <xsl:apply-templates
        select="preference[string-length(nvt)=0]"
        mode="edit-details"/>
      <tr>
        <td colspan="4" style="text-align:right;">
          <input type="submit"
                 name="submit"
                 value="{gsa:i18n ('Save Config')}"
                 title="{gsa:i18n ('Save Config')}"/>
        </td>
      </tr>
    </table>
  </div>
</xsl:template>

<!--     CONFIG NVTS -->
<xsl:template match="nvt" mode="edit">
  <xsl:param name="config"/>
  <table>
    <tr><td><b><xsl:value-of select="gsa:i18n ('Name')"/>:</b></td>
      <td>
        <b>
          <a href="/omp?cmd=get_config_nvt&amp;oid={@oid}&amp;config_id={$config/@id}&amp;name={$config/name}&amp;family={family}&amp;token={/envelope/token}"
            title="{gsa:i18n ('Scan Config NVT Details')}" target="_blank">
            <xsl:value-of select="name"/>
          </a>
        </b>
      </td>
    </tr>
    <tr><td><xsl:value-of select="gsa:i18n ('Config')"/>:</td><td><xsl:value-of select="$config/name"/></td></tr>
    <tr><td><xsl:value-of select="gsa:i18n ('Family')"/>:</td><td><xsl:value-of select="family"/></td></tr>
    <tr><td><xsl:value-of select="gsa:i18n ('OID')"/>:</td><td><xsl:value-of select="@oid"/></td></tr>
    <tr><td><xsl:value-of select="gsa:i18n ('Version')"/>:</td><td><xsl:value-of select="version"/></td></tr>
    <tr>
      <td><xsl:value-of select="gsa:i18n ('Notes')"/>:</td>
      <td>
        <xsl:value-of select="count (../../../get_notes_response/note)"/>
      </td>
    </tr>
    <tr>
      <td><xsl:value-of select="gsa:i18n ('Overrides')"/>:</td>
      <td>
        <xsl:value-of select="count (../../../get_overrides_response/override)"/>
      </td>
    </tr>
  </table>

  <xsl:choose>
    <xsl:when test="contains(tags, 'summary=')">
      <h2><xsl:value-of select="gsa:i18n ('Summary')"/></h2>
      <xsl:for-each select="str:split (tags, '|')">
        <xsl:if test="'summary' = substring-before (., '=')">
          <xsl:call-template name="structured-text">
            <xsl:with-param name="string" select="substring-after (., '=')"/>
          </xsl:call-template>
        </xsl:if>
      </xsl:for-each>
    </xsl:when>
    <xsl:otherwise>
    </xsl:otherwise>
  </xsl:choose>

  <xsl:choose>
    <xsl:when test="contains(tags, 'affected=')">
      <h2><xsl:value-of select="gsa:i18n ('Affected Software/OS')"/></h2>
      <xsl:for-each select="str:split (tags, '|')">
        <xsl:if test="'affected' = substring-before (., '=')">
          <xsl:call-template name="structured-text">
            <xsl:with-param name="string" select="substring-after (., '=')"/>
          </xsl:call-template>
        </xsl:if>
      </xsl:for-each>
    </xsl:when>
    <xsl:otherwise>
    </xsl:otherwise>
  </xsl:choose>

  <h2><xsl:value-of select="gsa:i18n ('Vulnerability Scoring')"/></h2>
  <table>
    <tr>
      <td><xsl:value-of select="gsa:i18n ('CVSS base')"/>:</td>
      <td>
        <xsl:choose>
          <xsl:when test="cvss_base &gt;= 0.0">
            <xsl:call-template name="severity-bar">
              <xsl:with-param name="cvss" select="cvss_base"/>
            </xsl:call-template>
          </xsl:when>
          <xsl:otherwise>
            <xsl:call-template name="severity-bar">
              <xsl:with-param name="extra_text" select="'N/A'"/>
            </xsl:call-template>
          </xsl:otherwise>
        </xsl:choose>
      </td>
    </tr>
    <xsl:variable name="token" select="/envelope/token"/>
    <xsl:for-each select="str:split (tags, '|')">
      <xsl:if test="'cvss_base_vector' = substring-before (., '=')">
        <xsl:variable name="cvss_vector" select="substring-after (., '=')"/>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('CVSS base vector')"/>:</td>
          <td>
            <a target="_blank" href="/omp?cmd=cvss_calculator&amp;cvss_vector={$cvss_vector}&amp;token={$token}">
              <xsl:value-of select="$cvss_vector"/>
            </a>
          </td>
        </tr>
      </xsl:if>
    </xsl:for-each>
  </table>
</xsl:template>

<xsl:template name="edit-config-nvt">
  <xsl:variable name="family">
    <xsl:value-of select="get_nvts_response/nvt/family"/>
  </xsl:variable>
  <div class="edit-dialog">
    <div class="title">
      <xsl:value-of select="gsa:i18n ('Edit Scan Config NVT')"/>
    </div>
    <div class="content">
      <xsl:apply-templates select="get_nvts_response/nvt" mode="edit">
        <xsl:with-param name="config" select="config"/>
      </xsl:apply-templates>

      <h2><xsl:value-of select="gsa:i18n ('Preferences')"/></h2>
      <form action="" method="post" enctype="multipart/form-data">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="save_config_nvt"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden" name="config_id" value="{config/@id}"/>
        <input type="hidden" name="name" value="{config/name}"/>
        <input type="hidden" name="family" value="{$family}"/>
        <input type="hidden"
          name="oid"
          value="{get_nvts_response/nvt/@oid}"/>
        <xsl:for-each select="get_nvts_response/nvt/preferences">
          <xsl:call-template name="preferences-edit-details">
            <xsl:with-param name="config" select="config"/>
          </xsl:call-template>
        </xsl:for-each>
      </form>
    </div>
  </div>
</xsl:template>

<xsl:template name="html-config-nvt-table">
  <xsl:variable name="family">
    <xsl:value-of select="get_nvts_response/nvt/family"/>
  </xsl:variable>
  <div class="toolbar">
    <a href="/help/config_nvt_details.html?token={/envelope/token}"
      class="icon icon-sm"
      title="{concat(gsa:i18n('Help'),': ',gsa:i18n('Scan Configs'),' (',gsa:i18n('Scan Config NVT Details'),')')}">
      <img src="/img/help.svg"/>
    </a>
    <a href="?cmd=get_config_family&amp;config_id={config/@id}&amp;name={config/name}&amp;family={$family}&amp;token={/envelope/token}"
      title="{gsa:i18n ('Scan Config Family')}"
      class="icon icon-sm">
      <img src="/img/list.svg" alt="{gsa:i18n ('Scan Config Family')}"/>
    </a>
  </div>
  <div class="section-header">
    <div class="section-header-info">
      <table>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Config ID')"/>:</td>
          <td><xsl:value-of select="config/@id"/></td>
        </tr>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Created', 'Date')"/>:</td>
          <td><xsl:value-of select="gsa:long-time (get_nvts_response/nvt/creation_time)"/></td>
        </tr>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Modified', 'Date')"/>:</td>
          <td><xsl:value-of select="gsa:long-time (get_nvts_response/nvt/modification_time)"/></td>
        </tr>
      </table>
    </div>
    <h1>
      <a href="/omp?cmd=get_configs&amp;token={/envelope/token}"
         title="{gsa:i18n ('Scan Configs')}">
        <img class="icon icon-lg" src="/img/config.svg" alt="Scan Configs"/>
      </a>
      <xsl:value-of select="gsa:i18n ('Scan Config NVT Details')"/>
    </h1>
  </div>
  <div class="section-box">
    <xsl:apply-templates select="get_nvts_response/nvt" mode="details">
      <xsl:with-param name="config" select="config/name"/>
    </xsl:apply-templates>

    <h2><xsl:value-of select="gsa:i18n ('Preferences')"/></h2>
    <xsl:variable name="config" select="config"/>
    <xsl:for-each select="get_nvts_response/nvt/preferences">
      <xsl:call-template name="preferences-details">
        <xsl:with-param name="config" select="$config"/>
      </xsl:call-template>
    </xsl:for-each>
  </div>
</xsl:template>

<!--     CONFIG FAMILIES -->

<xsl:template name="edit-families-family">
  <xsl:param name="config"></xsl:param>
  <xsl:param name="config-family"></xsl:param>
  <xsl:variable name="current_name" select="name/text()"/>
  <xsl:choose>
    <xsl:when test="name=''">
    </xsl:when>
    <xsl:otherwise>

      <tr class="{gsa:table-row-class(position())}">
        <td><xsl:value-of select="$current_name"/></td>
        <td>
          <xsl:choose>
            <xsl:when test="$config-family">
              <xsl:choose>
                <xsl:when test="$config-family/nvt_count='-1'">
                  N
                </xsl:when>
                <xsl:otherwise>
                  <xsl:value-of select="$config-family/nvt_count"/>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:when>
            <xsl:otherwise>
              0
            </xsl:otherwise>
          </xsl:choose>
          <xsl:choose>
            <xsl:when test="max_nvt_count='-1'">
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="gsa:i18n (' of ', 'Scan Config|NVTs')"/> <xsl:value-of select="max_nvt_count"/>
            </xsl:otherwise>
          </xsl:choose>
        </td>
        <td style="text-align:center;">
          <label>
            <xsl:choose>
              <xsl:when test="$config-family">
                <xsl:choose>
                  <xsl:when test="$config-family/growing=1">
                    <input type="radio" name="trend:{$current_name}" value="1"
                           checked="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="radio" name="trend:{$current_name}" value="1"/>
                  </xsl:otherwise>
                </xsl:choose>
              </xsl:when>
              <xsl:otherwise>
                <input type="radio" name="trend:{$current_name}" value="1"/>
              </xsl:otherwise>
            </xsl:choose>
            <img src="/img/trend_more.svg"
                 alt="{gsa:i18n ('Grows', 'Scan Config')}"
                 class="icon icon-sm"
                 title="{gsa:i18n ('The NVT selection is DYNAMIC. New NVTs will automatically be added and considered.')}"/>
          </label>
          <label>
            <xsl:choose>
              <xsl:when test="$config-family">
                <xsl:choose>
                  <xsl:when test="$config-family/growing=0">
                    <input type="radio" name="trend:{$current_name}" value="0"
                           checked="1"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="radio" name="trend:{$current_name}" value="0"/>
                  </xsl:otherwise>
                </xsl:choose>
              </xsl:when>
              <xsl:otherwise>
                <input type="radio" name="trend:{$current_name}" value="0"
                       checked="1"/>
              </xsl:otherwise>
            </xsl:choose>
            <img src="/img/trend_nochange.svg"
                 class="icon icon-sm"
                 alt="{gsa:i18n ('Static', 'Scan Config')}"
                 title="{gsa:i18n ('The NVT selection is STATIC. New NVTs will NOT automatically be added or considered.')}"/>
          </label>
        </td>
        <td style="text-align:center;">
          <xsl:choose>
            <xsl:when test="$config-family and ($config-family/nvt_count = max_nvt_count)">
              <input type="checkbox" name="select:{$current_name}"
                     value="1" checked="1"/>
            </xsl:when>
            <xsl:otherwise>
              <input type="checkbox" name="select:{$current_name}"
                     value="0"/>
            </xsl:otherwise>
          </xsl:choose>
        </td>
        <td>
          <a href="/omp?cmd=edit_config_family&amp;config_id={$config/@id}&amp;name={$config/name}&amp;family={$current_name}&amp;token={/envelope/token}"
            class="edit-action-icon icon icon-sm" data-cmd="edit_config_family" data-type="config" data-id="{$config/@id}"
            data-extra="name={$config/name}&amp;family={$current_name}" data-reload="parent"
            title="{gsa:i18n ('Edit Scan Config Family')}">
            <img src="/img/edit.svg" alt="{gsa:i18n ('Edit', 'Action Verb')}"/>
          </a>
        </td>
      </tr>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="family">
  <xsl:variable name="current_name" select="name/text()"/>
  <xsl:choose>
    <xsl:when test="name=''">
    </xsl:when>
    <xsl:otherwise>

      <tr class="{gsa:table-row-class(position())}">
        <td>
          <a href="/omp?cmd=get_config_family&amp;config_id={../../@id}&amp;name={../../name}&amp;family={$current_name}&amp;token={/envelope/token}"
             title="{gsa:i18n ('Scan Config Family Details')}" style="margin-left:3px;">
            <xsl:value-of select="$current_name"/>
          </a>
        </td>
        <td>
          <xsl:choose>
            <xsl:when test="nvt_count='-1'">
              N
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="nvt_count"/>
            </xsl:otherwise>
          </xsl:choose>
          <xsl:choose>
            <xsl:when test="max_nvt_count='-1'">
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="gsa:i18n (' of ', 'Scan Config|NVTs')"/>
              <xsl:value-of select="max_nvt_count"/>
            </xsl:otherwise>
          </xsl:choose>
        </td>
        <td>
          <xsl:choose>
            <xsl:when test="growing='1'">
              <img src="/img/trend_more.svg"
                   alt="{gsa:i18n ('Grows', 'Scan Config')}"
                   class="icon icon-sm"
                   title="{gsa:i18n ('The NVT selection is DYNAMIC. New NVTs will automatically be added and considered.')}"/>
            </xsl:when>
            <xsl:when test="growing='0'">
              <img src="/img/trend_nochange.svg"
                   alt="{gsa:i18n ('Static', 'Scan Config')}"
                   class="icon icon-sm"
                   title="{gsa:i18n ('The NVT selection is STATIC. New NVTs will NOT automatically be added or considered.')}"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="gsa:i18n ('N/A')"/>
            </xsl:otherwise>
          </xsl:choose>
        </td>
      </tr>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="config" mode="families">
  <div id="families">
    <table class="gbntable">
      <tr class="gbntablehead2">
        <td>
          <xsl:value-of select="gsa:i18n ('Family')"/>
          <xsl:choose>
            <xsl:when test="family_count/growing='1'">
              <img src="/img/trend_more.svg"
                   class="icon icon-sm"
                   alt="{gsa:i18n ('Grows', 'Scan Config')}"
                   title="{gsa:i18n ('The family selection is DYNAMIC. New families will automatically be added and considered.')}"/>
            </xsl:when>
            <xsl:when test="family_count/growing='0'">
              <img src="/img/trend_nochange.svg"
                   class="icon icon-sm"
                   alt="{gsa:i18n ('Static', 'Scan Config')}"
                   title="{gsa:i18n ('The family selection is STATIC. New families will NOT automatically be added or considered.')}"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="gsa:i18n ('N/A')"/>
            </xsl:otherwise>
          </xsl:choose>
        </td>
        <td><xsl:value-of select="gsa:i18n ('NVTs selected')"/></td>
        <td><xsl:value-of select="gsa:i18n ('Trend')"/></td>
      </tr>
      <xsl:apply-templates select="families/family"/>
      <xsl:if test="count(families/family) > 0">
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Total', 'Families')"/>: <xsl:value-of select="count(families/family)"/></td>
          <td>
            <table>
              <tr>
                <td style="margin-right:10px;">
                  <xsl:value-of select="known_nvt_count/text()"/>
                </td>
                <td>
                  <div style="margin-left:6px;">
                    <xsl:value-of select="gsa:i18n (' of ', 'Scan Config|NVTs')"/>
                    <xsl:value-of select="max_nvt_count/text()"/>
                    <xsl:value-of select="gsa:i18n (' in selected families', 'Scan Config|NVTs')"/><br/>
                    <xsl:value-of select="gsa:i18n (' of ', 'Scan Config|NVTs')"/>
                    <xsl:value-of select="sum(../../get_nvt_families_response/families/family/max_nvt_count)"/>
                    <xsl:value-of select="gsa:i18n (' in total', 'Scan Config|NVTs')"/>
                  </div>
                </td>
              </tr>
            </table>
          </td>
          <td>
            <xsl:choose>
              <xsl:when test="nvt_count/growing='1'">
                <img src="/img/trend_more.svg"
                     class="icon icon-sm"
                     alt="{gsa:i18n ('Grows', 'Scan Config')}"
                     title="{gsa:i18n ('The NVT selection is DYNAMIC. New NVTs will automatically be added and considered.')}"/>
              </xsl:when>
              <xsl:when test="nvt_count/growing='0'">
                <img src="/img/trend_nochange.svg"
                     class="icon icon-sm"
                     alt="{gsa:i18n ('Static', 'Scan Config')}"
                     title="{gsa:i18n ('The NVT selection is STATIC. New NVTs will NOT automatically be added or considered.')}"/>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="gsa:i18n ('N/A')"/>
              </xsl:otherwise>
            </xsl:choose>
          </td>
        </tr>
      </xsl:if>
    </table>
  </div>
</xsl:template>

<xsl:template name="edit-families">
  <xsl:param name="config"></xsl:param>
  <xsl:param name="families"></xsl:param>
  <div id="families">
    <table class="gbntable">
      <tr class="gbntablehead2">
        <td>
          <xsl:value-of select="gsa:i18n ('Family')"/>
          <xsl:choose>
            <xsl:when test="$config/family_count/growing=1">
              <label>
                <input type="radio" name="trend" value="1" checked="1"/>
                <img src="/img/trend_more.svg"
                     alt="{gsa:i18n ('Grows', 'Scan Config')}"
                     class="icon icon-sm"
                     title="{gsa:i18n ('The family selection is DYNAMIC. New families will automatically be added and considered.')}"/>
              </label>
              <label>
                <input type="radio" name="trend" value="0"/>
                <img src="/img/trend_nochange.svg"
                     alt="{gsa:i18n ('Static', 'Scan Config')}"
                     class="icon icon-sm"
                     title="{gsa:i18n ('The family selection is STATIC. New families will NOT automatically be added or considered.')}"/>
              </label>
            </xsl:when>
            <xsl:otherwise>
              <label>
                <input type="radio" name="trend" value="1"/>
                <img src="/img/trend_more.svg"
                     class="icon icon-sm"
                     alt="{gsa:i18n ('Grows', 'Scan Config')}"
                     title="{gsa:i18n ('The family selection is DYNAMIC. New families will automatically be added and considered.')}"/>
              </label>
              <label>
                <input type="radio" name="trend" value="0" checked="0"/>
                <img src="/img/trend_nochange.svg"
                     class="icon icon-sm"
                     alt="{gsa:i18n ('Static', 'Scan Config')}"
                     title="{gsa:i18n ('The family selection is STATIC. New families will NOT automatically be added or considered.')}"/>
              </label>
            </xsl:otherwise>
          </xsl:choose>
        </td>
        <td><xsl:value-of select="gsa:i18n ('NVTs selected')"/></td>
        <td><xsl:value-of select="gsa:i18n ('Trend')"/></td>
        <td><xsl:value-of select="gsa:i18n ('Select all NVTs')"/></td>
        <td><xsl:value-of select="gsa:i18n ('Actions')"/></td>
      </tr>
      <xsl:for-each select="$families/family">
        <xsl:variable name="family_name">
          <xsl:value-of select="name"/>
        </xsl:variable>
        <xsl:call-template name="edit-families-family">
          <xsl:with-param
            name="config-family"
            select="$config/families/family[name=$family_name]"/>
          <xsl:with-param name="config" select="$config"/>
        </xsl:call-template>
      </xsl:for-each>
      <tr>
        <td>
          <xsl:value-of select="gsa:i18n ('Total', 'Families')"/>: <xsl:value-of select="count($config/families/family)"/>
        </td>
        <td>
          <xsl:value-of select="$config/known_nvt_count/text()"/>
          <xsl:value-of select="gsa:i18n (' of ', 'Scan Config|NVTs')"/>
          <xsl:value-of select="$config/max_nvt_count/text()"/>
        </td>
        <td>
          <xsl:choose>
            <xsl:when test="$config/nvt_count/growing='1'">
              <img src="/img/trend_more.svg"
                   class="icon icon-sm"
                   alt="{gsa:i18n ('Grows', 'Scan Config')}"
                   title="{gsa:i18n ('The NVT selection is DYNAMIC. New NVTs will automatically be added and considered.')}"/>
            </xsl:when>
            <xsl:when test="$config/nvt_count/growing='0'">
              <img src="/img/trend_nochange.svg"
                   class="icon icon-sm"
                   alt="{gsa:i18n ('Static', 'Scan Config')}"
                   title="{gsa:i18n ('The NVT selection is STATIC. New NVTs will NOT automatically be added or considered.')}"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="gsa:i18n ('N/A')"/>
            </xsl:otherwise>
          </xsl:choose>
        </td>
        <td></td>
        <td></td>
      </tr>
      <tr>
        <td colspan="5" style="text-align:right;">
          <input type="submit"
                 name="submit"
                 value="{gsa:i18n ('Save Config')}"
                 title="{gsa:i18n ('Save Config')}"/>
        </td>
      </tr>
    </table>
  </div>
</xsl:template>

<!--     CONFIG OVERVIEW -->

<xsl:template name="html-config-table">
  <xsl:variable name="config" select="get_configs_response/config"/>

  <div class="toolbar">
    <a href="/help/config_details.html?token={/envelope/token}" class="icon icon-sm"
       title="{concat(gsa:i18n('Help'),': ',gsa:i18n('Scan Config Details'))}">
      <img src="/img/help.svg"/>
    </a>
    <a href="/omp?cmd=new_config&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;config_id={$config/@id}&amp;token={/envelope/token}"
       title="{gsa:i18n ('New Scan Config')}"
       class="new-action-icon icon icon-sm" data-type="config">
      <img src="/img/new.svg"/>
    </a>
    <a href="/omp?cmd=upload_config&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
       class="upload-action-icon icon icon-sm" data-type="config"
       title="{gsa:i18n ('Import Scan Config')}">
      <img src="/img/upload.svg"/>
    </a>
    <xsl:choose>
      <xsl:when test="gsa:may-clone ('config')">
        <div class="icon icon-sm ajax-post" data-reload="next" data-busy-text="{gsa:i18n ('Cloning...')}">
          <img src="/img/clone.svg"
            alt="{gsa:i18n ('Clone', 'Action Verb')}"
            name="Clone" title="{gsa:i18n ('Clone', 'Action Verb')}"/>
          <form action="/omp" method="post" enctype="multipart/form-data">
            <input type="hidden" name="token" value="{/envelope/token}"/>
            <input type="hidden" name="caller" value="{/envelope/current_page}"/>
            <input type="hidden" name="cmd" value="clone"/>
            <input type="hidden" name="resource_type" value="config"/>
            <input type="hidden" name="next" value="get_config"/>
            <input type="hidden" name="id" value="{$config/@id}"/>
            <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
            <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
          </form>
        </div>
      </xsl:when>
      <xsl:otherwise>
        <img src="/img/clone_inactive.svg"
             alt="{gsa:i18n ('Clone', 'Action Verb')}"
             value="Clone"
             title="{gsa:i18n ('Permission to clone denied')}"
             class="icon icon-sm"/>
      </xsl:otherwise>
    </xsl:choose>
    <a href="/omp?cmd=get_configs&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
       class="icon icon-sm"
       title="{gsa:i18n ('Scan Configs')}">
      <img src="/img/list.svg" alt="{gsa:i18n ('Scan Configs')}"/>
    </a>
    <span class="divider" />
    <xsl:choose>
      <xsl:when test="$config/writable!='0' and $config/in_use='0'">
        <xsl:call-template name="trashcan-icon">
          <xsl:with-param name="type" select="'config'"/>
          <xsl:with-param name="id" select="$config/@id"/>
          <xsl:with-param name="params">
            <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
            <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
          </xsl:with-param>
        </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
        <xsl:variable name="inactive_text">
          <xsl:choose>
            <xsl:when test="in_use != '0'"><xsl:value-of select="gsa:i18n ('Scan Config is not writable')"/></xsl:when>
            <xsl:when test="writable = '0'"><xsl:value-of select="gsa:i18n ('Scan Config is not writable')"/></xsl:when>
            <xsl:otherwise><xsl:value-of select="gsa:i18n ('Cannot move Scan Config to trashcan')"/></xsl:otherwise>
          </xsl:choose>
        </xsl:variable>
        <img src="/img/trashcan_inactive.svg" alt="{gsa:i18n ('To Trashcan', 'Action Verb')}"
              title="{$inactive_text}"
              class="icon icon-sm"/>
      </xsl:otherwise>
    </xsl:choose>
    <xsl:choose>
      <xsl:when test="$config/writable='0'">
        <img src="/img/edit_inactive.svg" alt="{gsa:i18n ('Edit', 'Action Verb')}"
              title="{gsa:i18n ('Scan Config is not writable')}"
              class="icon icon-sm"/>
      </xsl:when>
      <xsl:otherwise>
        <a href="/omp?cmd=edit_config&amp;config_id={$config/@id}&amp;next=get_config&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
            class="edit-action-icon icon icon-sm" data-type="config" data-id="{$config/@id}"
            title="{gsa:i18n ('Edit Scan Config')}">
          <img src="/img/edit.svg"/>
        </a>
      </xsl:otherwise>
    </xsl:choose>
    <a href="/omp?cmd=export_config&amp;config_id={$config/@id}&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
        title="{gsa:i18n ('Export Scan Config as XML')}"
        class="icon icon-sm">
      <img src="/img/download.svg" alt="{gsa:i18n ('Export XML', 'Action Verb')}"/>
    </a>
    <xsl:if test="$config/type = 1">
      <a href="/omp?cmd=sync_config&amp;config_id={$config/@id}&amp;next=get_config&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
        title="{gsa:i18n ('Sync Config')}" class="icon icon-sm">
        <img src="/img/refresh.svg" alt="{gsa:i18n ('Sync Config')}"/>
      </a>
    </xsl:if>
  </div>

  <div class="section-header">
    <div class="section-header-info">
      <table>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('ID')"/>:</td>
          <td><xsl:value-of select="$config/@id"/></td>
        </tr>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Created', 'Date')"/>:</td>
          <td><xsl:value-of select="gsa:long-time ($config/creation_time)"/></td>
        </tr>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Modified', 'Date')"/>:</td>
          <td><xsl:value-of select="gsa:long-time ($config/modification_time)"/></td>
        </tr>
      </table>
    </div>

    <h1>
      <a href="/omp?cmd=get_configs&amp;token={/envelope/token}"
         title="{gsa:i18n ('Scan Configs')}">
        <img class="icon icon-lg" src="/img/config.svg" alt="Scan Configs"/>
      </a>
      <xsl:value-of select="gsa:i18n ('Scan Config')"/>:
      <xsl:value-of select="$config/name"/>
      <xsl:text> </xsl:text>
    </h1>
  </div>

  <div class="section-box">
    <table>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Comment')"/>:</td><td><xsl:value-of select="$config/comment"/></td>
      </tr>
      <xsl:if test="$config/type = 1">
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Scanner')"/>:</td>
          <td>
            <a href="/omp?cmd=get_scanner&amp;scanner_id={$config/scanner/@id}&amp;token={/envelope/token}" title="{gsa:i18n ('Scanner Details')}">
              <xsl:value-of select="$config/scanner"/>
            </a>
          </td>
        </tr>
      </xsl:if>
    </table>
  </div>

  <div class="section-header">
    <a href="#" class="toggle-action-icon icon icon-sm icon-action"
      data-target="#nvtfamilies-box"
      data-name="Network Vulnerability Test Families"
      data-variable="nvtfamilies-box--collapsed">
        <img src="/img/fold.svg"/>
    </a>
    <h2>
      <xsl:value-of select="gsa:i18n ('Network Vulnerability Test Families')"/>
      <xsl:text> </xsl:text>
      <xsl:choose>
        <xsl:when test="count($config/families/family) != 0">
          (<xsl:value-of select="count($config/families/family)"/>)
        </xsl:when>
        <xsl:otherwise>
          (<xsl:value-of select="gsa:i18n ('none')"/>)
        </xsl:otherwise>
      </xsl:choose>
    </h2>
  </div>

  <div class="section-box" id="nvtfamilies-box">
    <xsl:apply-templates select="$config" mode="families"/>
  </div>

  <div class="section-header">
    <a href="#" class="toggle-action-icon icon icon-sm icon-action"
      data-target="#scannerprefs-box" data-name="Scanner Preferences"
      data-variable="scannerprefs-box--collapsed">
        <img src="/img/fold.svg"/>
    </a>
    <h2>
      <a href="/omp?cmd=get_tasks&amp;token={/envelope/token}"
         title="{gsa:i18n ('Scanner Preferences')}">
      </a>
      <xsl:value-of select="gsa:i18n ('Scanner Preferences')"/>
      <xsl:text> </xsl:text>
      <xsl:choose>
        <xsl:when test="count($config/preferences/preference[string-length(nvt)=0]) != 0">
          (<xsl:value-of select="count($config/preferences/preference[string-length(nvt)=0])"/>)
        </xsl:when>
        <xsl:otherwise>
          (<xsl:value-of select="gsa:i18n ('none')"/>)
        </xsl:otherwise>
      </xsl:choose>
    </h2>
  </div>

  <div class="section-box" id="scannerprefs-box">
    <xsl:apply-templates select="$config/preferences" mode="scanner"/>
  </div>

  <div class="section-header">
    <a href="#" class="toggle-action-icon icon icon-sm icon-action"
      data-target="#nvtprefs-box" data-name="NVT Preferences"
      data-variable="nvtprefs-box--collapsed">
        <img src="/img/fold.svg"/>
    </a>
    <h2>
      <a href="/omp?cmd=get_tasks&amp;token={/envelope/token}"
         title="{gsa:i18n ('Network Vulnerability Test Preferences')}">
      </a>
      <xsl:value-of select="gsa:i18n ('Network Vulnerability Test Preferences')"/>
      <xsl:text> </xsl:text>
      <xsl:choose>
        <xsl:when test="count($config/preferences/preference[string-length(nvt)>0]) != 0">
          (<xsl:value-of select="count($config/preferences/preference[string-length(nvt)>0])"/>)
        </xsl:when>
        <xsl:otherwise>
          (<xsl:value-of select="gsa:i18n ('none')"/>)
        </xsl:otherwise>
      </xsl:choose>
    </h2>
  </div>

  <div class="section-box" id="nvtprefs-box">
    <xsl:for-each select="$config/preferences">
      <xsl:call-template name="preferences">
        <xsl:with-param name="config_id" select="$config/@id"/>
        <xsl:with-param name="config_name" select="$config/name"/>
      </xsl:call-template>
    </xsl:for-each>
  </div>

  <div class="section-header">
    <a href="#" class="toggle-action-icon icon icon-sm icon-action"
      data-target="#using-box" data-name="Tasks using this Scan Config"
      data-variable="using-box--collapsed">
        <img src="/img/fold.svg"/>
    </a>
    <h2>
      <a href="/omp?cmd=get_tasks&amp;token={/envelope/token}"
         title="{gsa:i18n ('Tasks')}">
        <img class="icon icon-sm" src="/img/task.svg" alt="Tasks"/>
      </a>
      <xsl:value-of select="gsa:i18n ('Tasks using this Scan Config')"/>
      <xsl:text> </xsl:text>
      <xsl:choose>
        <xsl:when test="count($config/tasks/task) != 0">
          (<xsl:value-of select="count($config/tasks/task)"/>)
        </xsl:when>
        <xsl:otherwise>
          (<xsl:value-of select="gsa:i18n ('none')"/>)
        </xsl:otherwise>
      </xsl:choose>
    </h2>
  </div>

  <div class="section-box" id="using-box">
    <table class="gbntable">
      <tr class="gbntablehead2">
        <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
      </tr>
      <xsl:for-each select="$config/tasks/task">
        <tr class="{gsa:table-row-class(position())}">
          <xsl:choose>
            <xsl:when test="boolean (permissions) and count (permissions/permission) = 0">
              <td><xsl:value-of select="name"/> (<xsl:value-of select="gsa:i18n('Unavailable')"/>, <xsl:value-of select="gsa:i18n('UUID')"/>: <xsl:value-of select="@id"/>)</td>
            </xsl:when>
            <xsl:otherwise>
              <td>
                <a href="/omp?cmd=get_task&amp;task_id={@id}&amp;token={/envelope/token}" title="{gsa:i18n ('Details')}">
                  <xsl:value-of select="name"/>
                </a>
              </td>
            </xsl:otherwise>
          </xsl:choose>
        </tr>
      </xsl:for-each>
    </table>
  </div>

  <xsl:call-template name="user-tags-window">
    <xsl:with-param name="user_tags" select="$config/user_tags"/>
    <xsl:with-param name="title" select="gsa-i18n:strformat (gsa:i18n ('User Tags for &quot;%1&quot;'), $config/name)"/>
    <xsl:with-param name="resource_type" select="'config'"/>
    <xsl:with-param name="next" select="'get_config'"/>
    <xsl:with-param name="resource_id"   select="$config/@id"/>
  </xsl:call-template>

  <xsl:call-template name="resource-permissions-window">
    <xsl:with-param name="resource_type" select="'config'"/>
    <xsl:with-param name="title" select="gsa-i18n:strformat (gsa:i18n ('User Permissions for Config &quot;%1&quot;'), $config/name)"/>
    <xsl:with-param name="permissions" select="permissions/get_permissions_response"/>
    <xsl:with-param name="resource_id" select="$config/@id"/>
    <xsl:with-param name="related">
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template name="html-config-table-edit">
 <xsl:variable name="config" select="get_configs_response/config"/>

 <div class="gb_window">
  <div class="gb_window_part_left"></div>
  <div class="gb_window_part_right"></div>
  <div class="gb_window_part_center">
    <xsl:value-of select="gsa:i18n ('Edit Scan Config')"/>
    <xsl:call-template name="edit-header-icons">
      <xsl:with-param name="cap-type" select="'Scan Config'"/>
      <xsl:with-param name="type" select="'config'"/>
      <xsl:with-param name="id"
                      select="$config/@id"/>
    </xsl:call-template>
  </div>
  <div class="gb_window_part_content">
    <form action="" method="post" enctype="multipart/form-data">
      <input type="hidden" name="token" value="{/envelope/token}"/>
      <input type="hidden" name="cmd" value="save_config"/>
      <input type="hidden" name="caller" value="{/envelope/current_page}"/>
      <input type="hidden" name="config_id" value="{$config/@id}"/>
      <input type="hidden" name="name" value="{$config/name}"/>

      <table class="table-form">
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
          <td>
            <input type="text" name="name" value="{$config/name}" size="30"
                   maxlength="80"/>
          </td>
        </tr>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Comment')"/></td>
          <td>
            <input type="text" name="comment" size="30" maxlength="400"
                   value="{$config/comment}"/>
          </td>
        </tr>
        <xsl:if test="not($config/in_use != 0) and $config/type = 1">
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Scanner')"/></td>
            <td>
              <select name="scanner_id">
                <xsl:for-each select="get_scanners_response/scanner">
                  <xsl:call-template name="opt">
                    <xsl:with-param name="content" select="name"/>
                    <xsl:with-param name="value" select="@id"/>
                    <xsl:with-param name="select-value" select="$config/scanner/@id"/>
                  </xsl:call-template>
                </xsl:for-each>
              </select>
            </td>
          </tr>
        </xsl:if>
      </table>

      <xsl:if test="not($config/in_use != 0) and $config/type = 0">
        <div class="section-header">
          <a href="#" class="toggle-action-icon icon icon-sm icon-action"
            data-target="#families"
            data-name="Network Vulnerability Test Families"
            data-variable="families--collapsed">
              <img src="/img/fold.svg"/>
          </a>
          <h1><xsl:value-of select="gsa:i18n ('Edit Network Vulnerability Test Families')"/></h1>
        </div>

        <xsl:call-template name="edit-families">
          <xsl:with-param name="config" select="$config"/>
          <xsl:with-param
            name="families"
            select="get_nvt_families_response/families"/>
        </xsl:call-template>
      </xsl:if>

      <xsl:choose>
        <xsl:when test="$config/in_use != 0"/>
        <xsl:when test="count($config/preferences/preference[string-length(nvt)=0]) = 0">
          <h1><xsl:value-of select="gsa:i18n ('Edit Scanner Preferences')"/>: <xsl:value-of select="gsa:i18n ('None', 'Scanner|Preferences')"/></h1>
          <xsl:if test="$config/type = 0">
            <h1><xsl:value-of select="gsa:i18n ('Network Vulnerability Test Preferences')"/>: <xsl:value-of select="gsa:i18n ('None', 'NVT|Preferences')"/></h1>
          </xsl:if>
        </xsl:when>
        <xsl:otherwise>
          <div class="section-header">
            <a href="#" class="toggle-action-icon icon icon-sm icon-action"
              data-target="#scanner-preferences"
              data-name="Scanner Preferences" data-collapsed="true"
              data-variable="scanner-preferences--collapsed">
              <img src="/img/fold.svg"/>
            </a>
            <h1><xsl:value-of select="gsa:i18n ('Edit Scanner Preferences')"/></h1>
          </div>

          <xsl:apply-templates select="$config/preferences" mode="edit-scanner-details"/>

          <xsl:if test="$config/type = 0">
            <div class="section-header">
              <a href="#" class="toggle-action-icon icon icon-sm icon-action"
                data-target="#nvt-test-preferences"
                data-name="Scanner Preferences" data-collapsed="true"
                data-variable="nvt-test-preferences--collapsed">
                <img src="/img/fold.svg"/>
              </a>
              <h1><xsl:value-of select="gsa:i18n ('Network Vulnerability Test Preferences')"/></h1>
            </div>
              <xsl:for-each select="$config/preferences">
                <xsl:call-template name="preferences">
                  <xsl:with-param name="config_id" select="$config/@id"/>
                  <xsl:with-param name="config_name" select="$config/name"/>
                  <xsl:with-param name="edit">yes</xsl:with-param>
                </xsl:call-template>
              </xsl:for-each>
          </xsl:if>
        </xsl:otherwise>
      </xsl:choose>
    </form>
  </div>
 </div>
</xsl:template>


<xsl:template name="html-configs-table">
  <xsl:call-template name="list-window">
    <xsl:with-param name="type" select="'config'"/>
    <xsl:with-param name="cap-type" select="'Scan Config'"/>
    <xsl:with-param name="resources-summary" select="configs"/>
    <xsl:with-param name="resources" select="config"/>
    <xsl:with-param name="count" select="count (config)"/>
    <xsl:with-param name="filtered-count" select="config_count/filtered"/>
    <xsl:with-param name="full-count" select="config_count/text ()"/>
    <xsl:with-param name="columns" xmlns="">
      <column>
        <name><xsl:value-of select="gsa:i18n('Name')"/></name>
        <field>name</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Families')"/></name>
        <column>
          <name><xsl:value-of select="gsa:i18n('Total', 'Scan Config|Families')"/></name>
          <field>families_total</field>
          <sort-reverse/>
        </column>
        <column>
          <name><xsl:value-of select="gsa:i18n('Trend')"/></name>
          <field>families_trend</field>
        </column>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('NVTs')"/></name>
        <column>
          <name><xsl:value-of select="gsa:i18n('Total', 'Scan Config|NVTs')"/></name>
          <field>nvts_total</field>
          <sort-reverse/>
        </column>
        <column>
          <name><xsl:value-of select="gsa:i18n('Trend')"/></name>
          <field>nvts_total</field>
          <sort-reverse/>
        </column>
      </column>
    </xsl:with-param>
    <xsl:with-param name="default-filter" select="'apply_overrides=1 sort-reverse=date'"/>
    <xsl:with-param name="upload-icon" select="true ()"/>
    <xsl:with-param name="icon-count" select="5"/>
  </xsl:call-template>
</xsl:template>

<!--     CREATE_CONFIG_RESPONSE -->

<xsl:template match="create_config_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Create Config</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
    <xsl:with-param name="details">
      <xsl:if test="@status = '201' and config/name">
        Name of new config is '<xsl:value-of select="config/name"/>'.
      </xsl:if>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!--     DELETE_CONFIG_RESPONSE -->

<xsl:template match="delete_config_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Delete Config</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="sync_config_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Sync Config</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!-- CONFIG -->

<xsl:template match="config">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <xsl:call-template name="observers-icon">
        <xsl:with-param name="type" select="'Config'"/>
      </xsl:call-template>
      <b>
        <a href="/omp?cmd=get_config&amp;config_id={@id}&amp;filter={str:encode-uri (../filters/term, true ())}&amp;token={/envelope/token}"
           title="{gsa:view_details_title ('Scan Config', name)}">
          <xsl:value-of select="name"/>
        </a>
      </b>
      <xsl:choose>
        <xsl:when test="comment != ''">
          <div class="comment">(<xsl:value-of select="comment"/>)</div>
        </xsl:when>
        <xsl:otherwise>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="family_count/text()='-1'">
          <xsl:value-of select="gsa:i18n ('N/A')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="family_count/text()"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td style="text-align:center;">
      <xsl:choose>
        <xsl:when test="family_count/growing='1'">
          <img src="/img/trend_more.svg"
               class="icon icon-sm"
               alt="{gsa:i18n ('Grows', 'Scan Config')}"
               title="{gsa:i18n ('The family selection is DYNAMIC. New families will automatically be added and considered.')}"/>
        </xsl:when>
        <xsl:when test="family_count/growing='0'">
          <img src="/img/trend_nochange.svg"
               class="icon icon-sm"
               alt="{gsa:i18n ('Static', 'Scan Config')}"
               title="{gsa:i18n ('The family selection is STATIC. New families will NOT automatically be added or considered.')}"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="gsa:i18n ('N/A')"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="nvt_count/text()='-1'">
          <xsl:value-of select="gsa:i18n ('N/A')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="nvt_count/text()"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td style="text-align:center;">
      <xsl:choose>
        <xsl:when test="nvt_count/growing='1'">
          <img src="/img/trend_more.svg"
               class="icon icon-sm"
               alt="{gsa:i18n ('Dynamic', 'Scan Config')}"
               title="{gsa:i18n ('The NVT selection is DYNAMIC. New NVTs of selected families will automatically be added and considered.')}"/>
        </xsl:when>
        <xsl:when test="nvt_count/growing='0'">
          <img src="/img/trend_nochange.svg"
               class="icon icon-sm"
               alt="{gsa:i18n ('Static', 'Scan Config')}"
               title="{gsa:i18n ('The NVT selection is STATIC. New NVTs will NOT automatically be added or considered.')}"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="gsa:i18n ('N/A')"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <xsl:choose>
      <xsl:when test="/envelope/params/bulk_select = 1">
        <td style="text-align:center">
          <label style="width:100%">
            <input name="bulk_selected:{@id}" type="checkbox" style="width:100%; height:100%" title="{gsa:i18n ('Select for bulk action')}"/>
          </label>
        </td>
      </xsl:when>
      <xsl:otherwise>
        <td class="table-actions">
          <xsl:call-template name="list-window-line-icons">
            <xsl:with-param name="cap-type" select="'Scan Config'"/>
            <xsl:with-param name="type" select="'config'"/>
            <xsl:with-param name="id" select="@id"/>
          </xsl:call-template>
        </td>
      </xsl:otherwise>
    </xsl:choose>
  </tr>
</xsl:template>

<xsl:template match="config" mode="trash">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <xsl:call-template name="observers-icon">
        <xsl:with-param name="type" select="'Config'"/>
      </xsl:call-template>
      <b><xsl:value-of select="name"/></b>
      <xsl:choose>
        <xsl:when test="comment != ''">
          <div class="comment">(<xsl:value-of select="comment"/>)</div>
        </xsl:when>
        <xsl:otherwise>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="family_count/text()='-1'">
          <xsl:value-of select="gsa:i18n ('N/A')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="family_count/text()"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td style="text-align:center;">
      <xsl:choose>
        <xsl:when test="family_count/growing='1'">
          <img src="/img/trend_more.svg"
               class="icon icon-sm"
               alt="{gsa:i18n ('Grows', 'Scan Config')}"
               title="{gsa:i18n ('The family selection is DYNAMIC. New families will automatically be added and considered.')}"/>
        </xsl:when>
        <xsl:when test="family_count/growing='0'">
          <img src="/img/trend_nochange.svg"
               class="icon icon-sm"
               alt="{gsa:i18n ('Static', 'Scan Config')}"
               title="{gsa:i18n ('The family selection is STATIC. New families will NOT automatically be added or considered.')}"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="gsa:i18n ('N/A')"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="nvt_count/text()='-1'">
          <xsl:value-of select="gsa:i18n ('N/A')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="nvt_count/text()"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td style="text-align:center;">
      <xsl:choose>
        <xsl:when test="nvt_count/growing='1'">
          <img src="/img/trend_more.svg"
               alt="{gsa:i18n ('Grows', 'Scan Config')}"
               class="icon icon-sm"
               title="{gsa:i18n ('The NVT selection is DYNAMIC. New NVTs of selected families will automatically be added and considered.')}"/>
        </xsl:when>
        <xsl:when test="nvt_count/growing='0'">
          <img src="/img/trend_nochange.svg"
               class="icon icon-sm"
               alt="{gsa:i18n ('Static', 'Scan Config')}"
               title="{gsa:i18n ('The NVT selection is STATIC. New NVTs will NOT automatically be added or considered.')}"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="gsa:i18n ('N/A')"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td class="table-actions">
      <xsl:call-template name="restore-icon">
        <xsl:with-param name="id" select="@id"/>
      </xsl:call-template>
      <xsl:choose>
        <xsl:when test="in_use='0'">
          <xsl:call-template name="trash-delete-icon">
            <xsl:with-param name="type" select="'config'"/>
            <xsl:with-param name="id" select="@id"/>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <img src="/img/delete_inactive.svg" alt="{gsa:i18n ('Delete')}"
               title="{gsa:i18n ('Scan Config is still in use')}"
               class="icon icon-sm"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
  </tr>
</xsl:template>

<!-- GET_CONFIGS_RESPONSE -->

<xsl:template match="get_configs_response">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="create_filter_response"/>
  <xsl:apply-templates select="delete_config_response"/>
  <xsl:call-template name="html-configs-table"/>
</xsl:template>

<!-- GET_CONFIG_RESPONSE -->

<xsl:template match="get_config_response">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="create_config_response"/>
  <xsl:apply-templates select="create_tag_response"/>
  <xsl:apply-templates select="delete_tag_response"/>
  <xsl:apply-templates select="modify_tag_response"/>
  <xsl:choose>
    <xsl:when test="edit">
      <xsl:call-template name="html-config-table-edit"/>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="html-config-table"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- GET_CONFIG_FAMILY_RESPONSE -->

<xsl:template match="get_config_family_response">
  <xsl:choose>
    <xsl:when test="edit">
      <xsl:call-template name="edit-config-family"/>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="html-config-family-table"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- GET_CONFIG_NVT_RESPONSE -->

<xsl:template match="get_config_nvt_response">
  <xsl:choose>
    <xsl:when test="edit">
      <xsl:call-template name="edit-config-nvt"/>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="html-config-nvt-table"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>


<!-- BEGIN SCHEDULES MANAGEMENT -->

<xsl:template name="opt">
  <xsl:param name="value"></xsl:param>
  <xsl:param name="content"><xsl:value-of select="$value"/></xsl:param>
  <xsl:param name="select-value"></xsl:param>
  <xsl:param name="data-select" select="false ()"></xsl:param>
  <xsl:choose>
    <xsl:when test="$data-select and $value = $select-value">
      <option value="{$value}" selected="1" data-select="{$value}">
        <xsl:value-of select="$content"/>
      </option>
    </xsl:when>
    <xsl:when test="$data-select">
      <option value="{$value}" data-select="{$value}">
        <xsl:value-of select="$content"/>
      </option>
    </xsl:when>
    <xsl:when test="$value = $select-value">
      <option value="{$value}" selected="1"><xsl:value-of select="$content"/></option>
    </xsl:when>
    <xsl:otherwise>
      <option value="{$value}"><xsl:value-of select="$content"/></option>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="html-create-schedule-form">
  <div class="edit-dialog">
    <div class="title"><xsl:value-of select="gsa:i18n ('New Schedule')"/></div>
    <div class="content">
      <form action="/omp" method="post" enctype="multipart/form-data" class="form-horizontal">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="create_schedule"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden" name="next" value="get_schedule"/>
        <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
        <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Name')"/></label>
          <div class="col-10">
            <input type="text" name="name" value="unnamed" size="30"
              class="form-control"
              maxlength="80"/>
          </div>
        </div>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Comment')"/></label>
          <div class="col-10">
            <input type="text" name="comment" size="30" maxlength="400" class="form-control"/>
          </div>
        </div>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('First Time', 'Schedule')"/></label>
          <div class="col-10">
            <div class="datepicker form-item">
              <input class="datepicker-button" size="26"/>
              <input class="datepicker-day" name="day_of_month" type="hidden" value="{time/day_of_month}"/>
              <input class="datepicker-month" name="month" type="hidden" value="{time/month}"/>
              <input class="datepicker-year" name="year" type="hidden" value="{time/year}"/>
            </div>
            <div class="form-item">
              at
            </div>
            <div class="form-item">
              <input type="text"
                name="hour"
                value="{time/hour}"
                size="2"
                class="spinner"
                data-type="int"
                min="0"
                max="23"
                maxlength="2"/>
              h
            </div>
            <div class="form-item">
              <input type="text"
                name="minute"
                value="{time/minute - (time/minute mod 5)}"
                size="2"
                class="spinner"
                data-type="int"
                min="0"
                max="59"
                maxlength="2"/>
              m
            </div>
          </div>
        </div>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Timezone')"/></label>
          <div class="col-10">
            <xsl:call-template name="timezone-select">
              <xsl:with-param name="timezone" select="/envelope/timezone"/>
              <xsl:with-param name="input-name" select="'timezone'"/>
            </xsl:call-template>
          </div>
        </div>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Period', 'Schedule')"/></label>
          <div class="col-10">
            <div class="form-item">
              <input type="text"
                name="period"
                value="0"
                size="3"
                class="spinner"
                data-type="int"
                min="0"
                maxlength="80"/>
            </div>
            <div class="form-item">
              <select name="period_unit">
                <option value="hour" selected="1"><xsl:value-of select="gsa:i18n ('hour(s)')"/></option>
                <option value="day"><xsl:value-of select="gsa:i18n ('day(s)')"/></option>
                <option value="week"><xsl:value-of select="gsa:i18n ('week(s)')"/></option>
                <option value="month"><xsl:value-of select="gsa:i18n ('month(s)')"/></option>
              </select>
            </div>
          </div>
        </div>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Duration', 'Schedule')"/></label>
          <div class="col-10">
            <div class="form-item">
              <input type="text"
                name="duration"
                class="spinner"
                min="0"
                data-type="int"
                value="0"
                size="3"
                maxlength="80"/>
            </div>
            <div class="form-item">
              <select name="duration_unit">
                <option value="hour" selected="1"><xsl:value-of select="gsa:i18n ('hour(s)')"/></option>
                <option value="day"><xsl:value-of select="gsa:i18n ('day(s)')"/></option>
                <option value="week"><xsl:value-of select="gsa:i18n ('week(s)')"/></option>
              </select>
            </div>
          </div>
        </div>
      </form>
    </div>
  </div>
</xsl:template>

<xsl:template match="new_schedule">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="create_schedule_response"/>
  <xsl:apply-templates select="commands_response/delete_schedule_response"/>
  <xsl:call-template name="html-create-schedule-form"/>
</xsl:template>

<xsl:template name="html-schedules-table">
  <xsl:call-template name="list-window">
    <xsl:with-param name="type" select="'schedule'"/>
    <xsl:with-param name="cap-type" select="'Schedule'"/>
    <xsl:with-param name="resources-summary" select="schedules"/>
    <xsl:with-param name="resources" select="schedule"/>
    <xsl:with-param name="count" select="count (schedule)"/>
    <xsl:with-param name="filtered-count" select="schedule_count/filtered"/>
    <xsl:with-param name="full-count" select="schedule_count/text ()"/>
    <xsl:with-param name="columns" xmlns="">
      <column>
        <name><xsl:value-of select="gsa:i18n('Name')"/></name>
        <field>name</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('First Run', 'Schedule')"/></name>
        <field>first_run</field>
        <sort-reverse/>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Next Run', 'Schedule')"/></name>
        <field>next_run</field>
        <sort-reverse/>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Period', 'Schedule')"/></name>
        <field>period</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Duration', 'Schedule')"/></name>
        <field>duration</field>
      </column>
    </xsl:with-param>
    <xsl:with-param name="icon-count" select="4"/>
  </xsl:call-template>
</xsl:template>

<!--     CREATE_SCHEDULE_RESPONSE -->

<xsl:template match="create_schedule_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Create Schedule</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!--     DELETE_SCHEDULE_RESPONSE -->

<xsl:template match="delete_schedule_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">
      Delete Schedule
    </xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!--     MODIFY_SCHEDULE_RESPONSE -->

<xsl:template match="modify_schedule_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Save Schedules</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!--     EDIT_SCHEDULE -->

<xsl:template name="schedule-select-options">
  <xsl:param name="selected"/>
  <xsl:param name="max" select="24"/>
  <xsl:param name="current" select="0"/>
  <xsl:if test="$current &lt;= $max">
    <xsl:choose>
      <xsl:when test="$selected = $current">
        <option value="{format-number ($current, '00')}" selected="1">
          <xsl:value-of select="format-number ($current, '00')"/>
        </option>
      </xsl:when>
      <xsl:otherwise>
        <option value="{format-number ($current, '00')}">
          <xsl:value-of select="format-number ($current, '00')"/>
        </option>
      </xsl:otherwise>
    </xsl:choose>
    <xsl:call-template name="schedule-select-options">
      <xsl:with-param name="selected" select="$selected"/>
      <xsl:with-param name="current" select="$current + 1"/>
      <xsl:with-param name="max" select="$max"/>
    </xsl:call-template>
  </xsl:if>
</xsl:template>

<xsl:template name="html-edit-schedule-form">
  <div class="edit-dialog">
    <div class="title">
      <xsl:value-of select="gsa:i18n ('Edit Schedule')"/>
    </div>
    <div class="content">
      <xsl:variable name="task_count" select="count (commands_response/get_schedules_response/schedule/tasks/task)"/>
      <xsl:if test="$task_count &gt; 0">
        <p>
          <b><xsl:value-of select="gsa:i18n ('Notice')"/>:</b>
          <xsl:text> </xsl:text>
          <xsl:value-of select="gsa-i18n:strformat (gsa:n-i18n ('A Task using this schedule will be affected by changes made here.', '%1 Tasks using this schedule will be affected by changes made here.', $task_count), $task_count)"/>
        </p>
      </xsl:if>
      <form action="" method="post" enctype="multipart/form-data" class="form-horizontal">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="save_schedule"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden"
               name="schedule_id"
               value="{commands_response/get_schedules_response/schedule/@id}"/>
        <input type="hidden" name="next" value="{/envelope/params/next}"/>
        <input type="hidden" name="schedule" value="{/envelope/params/schedule}"/>
        <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Name')"/></label>
          <div class="col-10">
            <input type="text"
              name="name"
              class="form-control"
              value="{commands_response/get_schedules_response/schedule/name}"
              size="30"
              maxlength="80"/>
          </div>
        </div>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Comment')"/></label>
          <div class="col-10">
            <input type="text" name="comment" size="30" maxlength="400"
              class="form-control"
              value="{commands_response/get_schedules_response/schedule/comment}"/>
          </div>
        </div>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('First Time', 'Schedule')"/></label>
          <div class="col-10">
            <div class="datepicker form-item">
              <input class="datepicker-button" size="26"/>
              <input class="datepicker-day" name="day_of_month" type="hidden"
                value="{format-number (date:day-in-month (commands_response/get_schedules_response/schedule/first_time), '00')}"/>
              <input class="datepicker-month" name="month" type="hidden"
                value="{format-number (date:month-in-year (commands_response/get_schedules_response/schedule/first_time), '00')}"/>
              <input class="datepicker-year" name="year" type="hidden"
                value="{date:year (commands_response/get_schedules_response/schedule/first_time)}"/>
            </div>

            <div class="form-item">
              at
            </div>
            <xsl:variable name="hour"
              select="format-number (date:hour-in-day (commands_response/get_schedules_response/schedule/first_time), '00')"/>
            <div class="form-item">
              <input type="text"
                name="hour"
                value="{$hour}"
                size="2"
                class="spinner"
                data-type="int"
                min="0"
                max="23"
                maxlength="2"/>
              h
            </div>
            <xsl:variable name="minute"
              select="format-number (date:minute-in-hour (commands_response/get_schedules_response/schedule/first_time), '00')"/>
            <div class="form-item">
              <input type="text"
                name="minute"
                value="{$minute - ($minute mod 5)}"
                size="2"
                class="spinner"
                data-type="int"
                min="0"
                max="59"
                maxlength="2"/>
              m
            </div>
          </div>
        </div>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Timezone')"/></label>
          <div class="col-10">
            <xsl:call-template name="timezone-select">
              <xsl:with-param name="timezone" select="commands_response/get_schedules_response/schedule/timezone"/>
              <xsl:with-param name="input-name" select="'timezone'"/>
            </xsl:call-template>
          </div>
        </div>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Period', 'Schedule')"/></label>
          <div class="col-10">
            <xsl:choose>
              <xsl:when test="commands_response/get_schedules_response/schedule/simple_period/text() = 0 and commands_response/get_schedules_response/schedule/period/text() &gt; 0">
                <div class="form-item">
                  <input type="text"
                    name="period"
                    class="spinner"
                    data-type="int"
                    min="0"
                    value="{commands_response/get_schedules_response/schedule/period}"
                    size="10"
                    maxlength="80"/>
                  <input type="hidden" name="period_unit" value="second"/>
                </div>
                <xsl:value-of select="gsa:i18n ('seconds')"/>
              </xsl:when>
              <xsl:when test="commands_response/get_schedules_response/schedule/simple_period/text() = 0 and commands_response/get_schedules_response/schedule/period_months/text() &gt; 0">
                <div class="form-item">
                  <input type="text"
                    name="period"
                    value="{commands_response/get_schedules_response/schedule/period_months}"
                    class="spinner"
                    data-type="int"
                    min="0"
                    size="10"
                    maxlength="80"/>
                </div>
                <input type="hidden" name="period_unit" value="month"/>
                <xsl:value-of select="gsa:i18n ('months')"/>
              </xsl:when>
              <xsl:otherwise>
                <div class="form-item">
                  <input type="text"
                    name="period"
                    value="{commands_response/get_schedules_response/schedule/simple_period/text()}"
                    class="spinner"
                    data-type="int"
                    min="0"
                    size="4"
                    maxlength="80"/>
                </div>
                <div class="form-item">
                  <select name="period_unit">
                    <xsl:choose>
                      <xsl:when test="commands_response/get_schedules_response/schedule/simple_period/unit = 'hour'">
                        <option value="hour" selected="1"><xsl:value-of select="gsa:i18n ('hour(s)')"/></option>
                      </xsl:when>
                      <xsl:otherwise>
                        <option value="hour"><xsl:value-of select="gsa:i18n ('hour(s)')"/></option>
                      </xsl:otherwise>
                    </xsl:choose>
                    <xsl:choose>
                      <xsl:when test="commands_response/get_schedules_response/schedule/simple_period/unit = 'day'">
                        <option value="day" selected="1"><xsl:value-of select="gsa:i18n ('day(s)')"/></option>
                      </xsl:when>
                      <xsl:otherwise>
                        <option value="day"><xsl:value-of select="gsa:i18n ('day(s)')"/></option>
                      </xsl:otherwise>
                    </xsl:choose>
                    <xsl:choose>
                      <xsl:when test="commands_response/get_schedules_response/schedule/simple_period/unit = 'week'">
                        <option value="week" selected="1"><xsl:value-of select="gsa:i18n ('week(s)')"/></option>
                      </xsl:when>
                      <xsl:otherwise>
                        <option value="week"><xsl:value-of select="gsa:i18n ('week(s)')"/></option>
                      </xsl:otherwise>
                    </xsl:choose>
                    <xsl:choose>
                      <xsl:when test="commands_response/get_schedules_response/schedule/simple_period/unit = 'month'">
                        <option value="month" selected="1"><xsl:value-of select="gsa:i18n ('month(s)')"/></option>
                      </xsl:when>
                      <xsl:otherwise>
                        <option value="month"><xsl:value-of select="gsa:i18n ('month(s)')"/></option>
                      </xsl:otherwise>
                    </xsl:choose>
                  </select>
                </div>
              </xsl:otherwise>
            </xsl:choose>
          </div>
        </div>
        <div class="form-group">
          <label class="col-2 control-label"><xsl:value-of select="gsa:i18n ('Duration', 'Schedule')"/></label>
          <div class="col-10">
            <xsl:choose>
              <xsl:when test="commands_response/get_schedules_response/schedule/simple_duration/text() = 0 and commands_response/get_schedules_response/schedule/duration/text() &gt; 0">
                <div class="form-item">
                  <input type="text"
                    name="duration"
                    value="{commands_response/get_schedules_response/schedule/duration}"
                    size="10"
                    class="spinner"
                    min="0"
                    data-type="int"
                    maxlength="80"/>
                </div>
                <input type="hidden" name="duration_unit" value="second"/>
                <xsl:value-of select="gsa:i18n ('seconds')"/>
              </xsl:when>
              <xsl:otherwise>
                <div class="form-item">
                  <input type="text"
                    name="duration"
                    value="{commands_response/get_schedules_response/schedule/simple_duration/text()}"
                    size="4"
                    class="spinner"
                    min="0"
                    data-type="int"
                    maxlength="80"/>
                </div>
                <div class="form-item">
                  <select name="duration_unit">
                    <xsl:choose>
                      <xsl:when test="commands_response/get_schedules_response/schedule/simple_duration/unit = 'hour'">
                        <option value="hour" selected="1"><xsl:value-of select="gsa:i18n ('hour(s)')"/></option>
                      </xsl:when>
                      <xsl:otherwise>
                        <option value="hour"><xsl:value-of select="gsa:i18n ('hour(s)')"/></option>
                      </xsl:otherwise>
                    </xsl:choose>
                    <xsl:choose>
                      <xsl:when test="commands_response/get_schedules_response/schedule/simple_duration/unit = 'day'">
                        <option value="day" selected="1"><xsl:value-of select="gsa:i18n ('day(s)')"/></option>
                      </xsl:when>
                      <xsl:otherwise>
                        <option value="day"><xsl:value-of select="gsa:i18n ('day(s)')"/></option>
                      </xsl:otherwise>
                    </xsl:choose>
                    <xsl:choose>
                      <xsl:when test="commands_response/get_schedules_response/schedule/simple_duration/unit = 'week'">
                        <option value="week" selected="1"><xsl:value-of select="gsa:i18n ('week(s)')"/></option>
                      </xsl:when>
                      <xsl:otherwise>
                        <option value="week"><xsl:value-of select="gsa:i18n ('week(s)')"/></option>
                      </xsl:otherwise>
                    </xsl:choose>
                  </select>
                </div>
              </xsl:otherwise>
            </xsl:choose>
          </div>
        </div>
      </form>
    </div>
  </div>
</xsl:template>

<xsl:template match="edit_schedule">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="modify_schedule_response"/>
  <xsl:call-template name="html-edit-schedule-form"/>
</xsl:template>

<!--     SCHEDULE -->

<xsl:template name="interval-with-unit">
  <xsl:param name="interval">0</xsl:param>
  <xsl:choose>
    <xsl:when test="$interval mod (60 * 60 * 24 * 7) = 0">
      <xsl:value-of select="$interval div (60 * 60 * 24 * 7)"/>
      <xsl:text> </xsl:text>
      <xsl:choose>
        <xsl:when test="$interval = (60 * 60 * 24 * 7)">
          <xsl:value-of select="gsa:i18n ('week')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="gsa:i18n ('weeks')"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>
    <xsl:when test="$interval mod (60 * 60 * 24) = 0">
      <xsl:value-of select="$interval div (60 * 60 * 24)"/>
      <xsl:text> </xsl:text>
      <xsl:choose>
        <xsl:when test="$interval = (60 * 60 * 24)">
          <xsl:value-of select="gsa:i18n ('day')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="gsa:i18n ('days')"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>
    <xsl:when test="$interval mod (60 * 60) = 0">
      <xsl:value-of select="$interval div (60 * 60)"/>
      <xsl:text> </xsl:text>
      <xsl:choose>
        <xsl:when test="$interval = (60 * 60)">
          <xsl:value-of select="gsa:i18n ('hour')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="gsa:i18n ('hours')"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>
    <xsl:when test="$interval mod (60) = 0">
      <xsl:value-of select="$interval div (60)"/>
      <xsl:text> </xsl:text>
      <xsl:choose>
        <xsl:when test="$interval = 60">
          <xsl:value-of select="gsa:i18n ('min')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="gsa:i18n ('mins')"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>
    <xsl:when test="$interval = 1">
      <xsl:value-of select="$interval"/>
      <xsl:text> </xsl:text>
      <xsl:value-of select="gsa:i18n ('sec')"/>
    </xsl:when>
    <xsl:otherwise>
      <xsl:value-of select="$interval"/>
      <xsl:text> </xsl:text>
      <xsl:value-of select="gsa:i18n ('secs')"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="schedule">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <xsl:call-template name="observers-icon">
        <xsl:with-param name="type" select="'Schedule'"/>
      </xsl:call-template>
      <b>
        <a href="/omp?cmd=get_schedule&amp;schedule_id={@id}&amp;filter={str:encode-uri (../filters/term, true ())}&amp;token={/envelope/token}"
           title="{gsa:view_details_title ('Schedule', name)}">
          <xsl:value-of select="name"/>
        </a>
      </b>
      <xsl:choose>
        <xsl:when test="comment != ''">
          <div class="comment">(<xsl:value-of select="comment"/>)</div>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:value-of select="gsa:long-time (first_time)"/>
      <xsl:text> </xsl:text>
      <xsl:value-of select="timezone_abbrev"/>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="next_time = 'over'">
          -
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="gsa:long-time (next_time)"/>
          <xsl:text> </xsl:text>
          <xsl:value-of select="timezone_abbrev"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="period = 0 and period_months = 0">
          <xsl:value-of select="gsa:i18n ('Once')"/>
        </xsl:when>
        <xsl:when test="period = 0 and period_months = 1">
          1 <xsl:value-of select="gsa:i18n ('month')"/>
        </xsl:when>
        <xsl:when test="period = 0">
          <xsl:value-of select="period_months"/>
          <xsl:text> </xsl:text>
          <xsl:value-of select="gsa:i18n ('months')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="interval-with-unit">
            <xsl:with-param name="interval">
              <xsl:value-of select="period"/>
            </xsl:with-param>
          </xsl:call-template>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="duration = 0">
          <xsl:value-of select="gsa:i18n ('Entire Operation', 'Time')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="interval-with-unit">
            <xsl:with-param name="interval">
              <xsl:value-of select="duration"/>
            </xsl:with-param>
          </xsl:call-template>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <xsl:choose>
      <xsl:when test="/envelope/params/bulk_select = 1">
        <td style="text-align:center">
          <label style="width:100%">
            <input name="bulk_selected:{@id}" type="checkbox" style="width:100%; height:100%" title="{gsa:i18n ('Select for bulk action')}"/>
          </label>
        </td>
      </xsl:when>
      <xsl:otherwise>
        <td class="table-actions">
          <xsl:call-template name="list-window-line-icons">
            <xsl:with-param name="cap-type" select="'Schedule'"/>
            <xsl:with-param name="type" select="'schedule'"/>
            <xsl:with-param name="id" select="@id"/>
          </xsl:call-template>
        </td>
      </xsl:otherwise>
    </xsl:choose>
  </tr>
</xsl:template>

<xsl:template match="schedule" mode="trash">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <xsl:call-template name="observers-icon">
        <xsl:with-param name="type" select="'Schedule'"/>
      </xsl:call-template>
      <b><xsl:value-of select="name"/></b>
      <xsl:choose>
        <xsl:when test="comment != ''">
          <div class="comment">(<xsl:value-of select="comment"/>)</div>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:value-of select="first_time"/>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="next_time &gt; 0">
          <xsl:value-of select="next_time"/>
        </xsl:when>
        <xsl:otherwise>-</xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="period = 0 and period_months = 0">
          <xsl:value-of select="gsa:i18n ('Once')"/>
        </xsl:when>
        <xsl:when test="period = 0 and period_months = 1">
          1 <xsl:value-of select="gsa:i18n ('month')"/>
        </xsl:when>
        <xsl:when test="period = 0">
          <xsl:value-of select="period_months"/>
          <xsl:text> </xsl:text>
          <xsl:value-of select="gsa:i18n ('months')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="interval-with-unit">
            <xsl:with-param name="interval">
              <xsl:value-of select="period"/>
            </xsl:with-param>
          </xsl:call-template>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="duration = 0">
          <xsl:value-of select="gsa:i18n ('Entire Operation', 'Time')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="interval-with-unit">
            <xsl:with-param name="interval">
              <xsl:value-of select="duration"/>
            </xsl:with-param>
          </xsl:call-template>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td class="table-actions">
      <xsl:call-template name="restore-icon">
        <xsl:with-param name="id" select="@id"/>
      </xsl:call-template>
      <xsl:choose>
        <xsl:when test="in_use='0'">
          <xsl:call-template name="trash-delete-icon">
            <xsl:with-param name="type" select="'schedule'"/>
            <xsl:with-param name="id" select="@id"/>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <img src="/img/delete_inactive.svg"
               alt="{gsa:i18n ('Delete')}"
               title="{gsa:i18n ('Schedule is still in use')}"
               class="icon icon-sm"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
  </tr>
</xsl:template>

<xsl:template match="schedule" mode="details">
  <div class="toolbar">
    <xsl:call-template name="details-header-icons">
      <xsl:with-param name="cap-type" select="'Schedule'"/>
      <xsl:with-param name="type" select="'schedule'"/>
    </xsl:call-template>
  </div>

  <div class="section-header">
    <xsl:call-template name="minor-details"/>
    <h1>
      <a href="/omp?cmd=get_schedules&amp;token={/envelope/token}"
         title="{gsa:i18n ('Schedules')}">
        <img class="icon icon-lg" src="/img/schedule.svg" alt="Schedules"/>
      </a>
      <xsl:value-of select="gsa:i18n ('Schedule')"/>:
      <xsl:value-of select="name"/>
      <xsl:text> </xsl:text>
    </h1>
  </div>

  <div class="section-box">
    <table>
      <tr>
        <td><b><xsl:value-of select="gsa:i18n ('Name')"/>:</b></td>
        <td><b><xsl:value-of select="name"/></b></td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Comment')"/>:</td>
        <td><xsl:value-of select="comment"/></td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('First Run', 'Schedule')"/>:</td>
        <td>
          <xsl:value-of select="gsa:long-time (first_time)"/>
          <xsl:text> </xsl:text>
          <xsl:value-of select="timezone_abbrev"/>
        </td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Next Run', 'Schedule')"/>:</td>
        <td>
          <xsl:choose>
            <xsl:when test="next_time = 'over'">
              -
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="gsa:long-time (next_time)"/>
              <xsl:text> </xsl:text>
              <xsl:value-of select="timezone_abbrev"/>
            </xsl:otherwise>
          </xsl:choose>
        </td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Timezone')"/>:</td>
        <td>
          <xsl:value-of select="timezone"/>
          <xsl:if test="timezone != timezone_abbrev">
            (<xsl:value-of select="timezone_abbrev"/>)
          </xsl:if>
        </td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Period', 'Schedule')"/>:</td>
        <td>
          <xsl:choose>
            <xsl:when test="period = 0 and period_months = 0">
              <xsl:value-of select="gsa:i18n ('Once')"/>
            </xsl:when>
            <xsl:when test="period = 0 and period_months = 1">
              1 <xsl:value-of select="gsa:i18n ('month')"/>
            </xsl:when>
            <xsl:when test="period = 0">
              <xsl:value-of select="period_months"/>
              <xsl:text> </xsl:text>
              <xsl:value-of select="gsa:i18n ('months')"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:call-template name="interval-with-unit">
                <xsl:with-param name="interval">
                  <xsl:value-of select="period"/>
                </xsl:with-param>
              </xsl:call-template>
            </xsl:otherwise>
          </xsl:choose>
        </td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Duration', 'Schedule')"/>:</td>
        <td>
          <xsl:choose>
            <xsl:when test="duration = 0">
              <xsl:value-of select="gsa:i18n ('Entire Operation', 'Time')"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:call-template name="interval-with-unit">
                <xsl:with-param name="interval">
                  <xsl:value-of select="duration"/>
                </xsl:with-param>
              </xsl:call-template>
            </xsl:otherwise>
          </xsl:choose>
        </td>
      </tr>
    </table>
  </div>

  <div class="section-header">
    <a href="#" class="toggle-action-icon icon icon-sm icon-action"
      data-target="#using-box" data-name="Tasks using this Schedule"
      data-variable="using-box--collapsed">
        <img src="/img/fold.svg"/>
    </a>
    <h2>
      <a href="/omp?cmd=get_tasks&amp;token={/envelope/token}"
         title="{gsa:i18n ('Tasks')}">
        <img class="icon icon-sm" src="/img/task.svg" alt="Tasks"/>
      </a>
      <xsl:value-of select="gsa:i18n ('Tasks using this Schedule')"/>
      <xsl:text> </xsl:text>
      <xsl:choose>
        <xsl:when test="count(tasks/task) != 0">
          (<xsl:value-of select="count(tasks/task)"/>)
        </xsl:when>
        <xsl:otherwise>
          (<xsl:value-of select="gsa:i18n ('none')"/>)
        </xsl:otherwise>
      </xsl:choose>
    </h2>
  </div>

  <div class="section-box" id="using-box">
    <table class="gbntable">
      <tr class="gbntablehead2">
        <td><xsl:value-of select="gsa:i18n('Name')"/></td>
      </tr>
      <xsl:for-each select="tasks/task">
        <tr class="{gsa:table-row-class(position())}">
          <xsl:choose>
            <xsl:when test="boolean (permissions) and count (permissions/permission) = 0">
              <td><xsl:value-of select="name"/> (<xsl:value-of select="gsa:i18n('Unavailable')"/>, <xsl:value-of select="gsa:i18n('UUID')"/>: <xsl:value-of select="@id"/>)</td>
            </xsl:when>
            <xsl:otherwise>
              <td>
                <a href="/omp?cmd=get_task&amp;task_id={@id}&amp;token={/envelope/token}" title="{gsa:i18n ('Details')}">
                  <xsl:value-of select="name"/>
                </a>
              </td>
            </xsl:otherwise>
          </xsl:choose>
        </tr>
      </xsl:for-each>
    </table>
  </div>

  <xsl:call-template name="user-tags-window">
    <xsl:with-param name="resource_type" select="'schedule'"/>
  </xsl:call-template>

  <xsl:call-template name="resource-permissions-window">
    <xsl:with-param name="resource_type" select="'schedule'"/>
    <xsl:with-param name="permissions" select="../../permissions/get_permissions_response"/>
    <xsl:with-param name="related">
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!--     GET_SCHEDULE -->

<xsl:template match="get_schedule">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="commands_response/delete_schedule_response"/>
  <xsl:apply-templates select="modify_schedule_response"/>
  <xsl:apply-templates select="delete_tag_response"/>
  <xsl:apply-templates select="create_tag_response"/>
  <xsl:apply-templates select="modify_tag_response"/>
  <xsl:apply-templates select="get_schedules_response/schedule" mode="details"/>
</xsl:template>

<!--     GET_SCHEDULES -->

<xsl:template match="get_schedules">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="delete_schedule_response"/>
  <xsl:apply-templates select="create_filter_response"/>
  <xsl:apply-templates select="create_schedule_response"/>
  <xsl:apply-templates select="modify_schedule_response"/>
  <!-- The for-each makes the get_schedules_response the current node. -->
  <xsl:for-each select="get_schedules_response | commands_response/get_schedules_response">
    <xsl:choose>
      <xsl:when test="substring(@status, 1, 1) = '4' or substring(@status, 1, 1) = '5'">
        <xsl:call-template name="command_result_dialog">
          <xsl:with-param name="operation">
            Get Schedules
          </xsl:with-param>
          <xsl:with-param name="status">
            <xsl:value-of select="@status"/>
          </xsl:with-param>
          <xsl:with-param name="msg">
            <xsl:value-of select="@status_text"/>
          </xsl:with-param>
        </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
        <xsl:call-template name="html-schedules-table"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:for-each>
</xsl:template>

<!--    BEGIN SCANNERS MANAGEMENT -->

<xsl:template name="html-create-scanner-form-credentials">
  <xsl:param name="type" select="'cc'"/>
  <xsl:param name="input-classes"/>
  <td><xsl:value-of select="gsa:i18n ('Credential')"/></td>
  <td>
    <xsl:variable name="credential_id" select="/envelope/params/credential_id"/>
    <select name="credential_id" class="{$input-classes}">
      <xsl:for-each select="get_credentials_response/credential[type=$type]">
        <xsl:call-template name="opt">
          <xsl:with-param name="content" select="name"/>
          <xsl:with-param name="value" select="@id"/>
          <xsl:with-param name="select-value" select="$credential_id"/>
        </xsl:call-template>
      </xsl:for-each>
    </select>
    <a href="#" title="{ gsa:i18n('Create a new Credential') }"
       class="new-action-icon icon icon-sm" data-type="credential"
       data-done="select[name=credential_id]"
       data-extra="restrict_credential_type={$type}">
      <img class="valign-middle" src="/img/new.svg"/>
    </a>
  </td>
</xsl:template>

<xsl:template name="html-create-scanner-form">
  <div class="gb_window">
    <div class="gb_window_part_left"></div>
    <div class="gb_window_part_right"></div>
    <div class="gb_window_part_center">
      <xsl:value-of select="gsa:i18n ('New Scanner')"/>
      <a href="/help/new_scanner.html?token={/envelope/token}"
        class="icon icon-sm"
        title="{concat(gsa:i18n('Help'),': ',gsa:i18n('New Scanner'))}">
        <img src="/img/help.svg"/>
      </a>
      <a href="/omp?cmd=get_scanners&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
        class="icon icon-sm"
        title="{gsa:i18n ('Scanners')}">
        <img src="/img/list.svg" alt="{gsa:i18n ('Scanners')}"/>
      </a>
    </div>
    <div class="gb_window_part_content">
      <form action="/omp" method="post" enctype="multipart/form-data">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="create_scanner"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden" name="next" value="get_scanner"/>
        <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
        <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
        <table class="table-form">
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
            <td>
              <input type="text" name="name" value="unnamed" size="30"
                     maxlength="80"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Comment')"/></td>
            <td>
              <input type="text" name="comment" size="30" maxlength="400"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Host')"/></td>
            <td><input type="text" name="scanner_host" value="localhost" size="30"/></td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Port')"/></td>
            <td><input type="text" name="port" value="9391" size="30"/></td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Type')"/></td>
            <td>
              <select class="form-selection-control" id="scanner" name="scanner_type">
                <xsl:call-template name="scanner-type-list">
                  <xsl:with-param name="default" select="2"/>
                  <xsl:with-param name="data-select" select="true ()"/>
                </xsl:call-template>
              </select>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('CA Certificate')"/></td>
            <td>
              <input type="file" name="ca_pub"/>
            </td>
          </tr>
          <tr class="form-selection-item-scanner form-selection-item-scanner--1">
            <xsl:call-template name="html-create-scanner-form-credentials">
              <xsl:with-param name="type" select="'cc'"/>
              <xsl:with-param
                name="input-classes"
                select="'form-selection-input-scanner form-selection-input-scanner--1'"/>
            </xsl:call-template>
          </tr>
          <tr class="form-selection-item-scanner form-selection-item-scanner--2">
            <xsl:call-template name="html-create-scanner-form-credentials">
              <xsl:with-param name="type" select="'cc'"/>
              <xsl:with-param
                name="input-classes"
                select="'form-selection-input-scanner form-selection-input-scanner--2'"/>
            </xsl:call-template>
          </tr>
          <tr class="form-selection-item-scanner form-selection-item-scanner--4">
            <xsl:call-template name="html-create-scanner-form-credentials">
              <xsl:with-param name="type" select="'up'"/>
              <xsl:with-param
                name="input-classes"
                select="'form-selection-input-scanner form-selection-input-scanner--4'"/>
            </xsl:call-template>
          </tr>
          <tr>
            <td>
              <input type="submit" name="submit" value="{gsa:i18n ('Create Scanner')}"/>
            </td>
          </tr>
        </table>
      </form>
    </div>
  </div>
</xsl:template>

<xsl:template match="create_scanner_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Create Scanner</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="delete_scanner_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">
      Delete Scanner
    </xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="modify_scanner_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Save Scanner</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="new_scanner">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="create_scanner_response"/>
  <xsl:apply-templates select="commands_response/delete_scanner_response"/>
  <xsl:call-template name="html-create-scanner-form"/>
</xsl:template>

<xsl:template name="html-scanners-table">
  <xsl:call-template name="list-window">
    <xsl:with-param name="type" select="'scanner'"/>
    <xsl:with-param name="cap-type" select="'Scanner'"/>
    <xsl:with-param name="resources-summary" select="scanners"/>
    <xsl:with-param name="resources" select="scanner"/>
    <xsl:with-param name="count" select="count (scanner)"/>
    <xsl:with-param name="filtered-count" select="scanner_count/filtered"/>
    <xsl:with-param name="full-count" select="scanner_count/text ()"/>
    <xsl:with-param name="columns" xmlns="">
      <column>
        <name><xsl:value-of select="gsa:i18n('Name')"/></name>
        <field>name</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Host')"/></name>
        <field>host</field>
        <sort-reverse/>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Port')"/></name>
        <field>port</field>
        <sort-reverse/>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Type')"/></name>
        <field>type</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Credential')"/></name>
        <field>credential</field>
      </column>
    </xsl:with-param>
    <xsl:with-param name="icon-count" select="7"/>
  </xsl:call-template>
</xsl:template>

<xsl:template name="html-edit-scanner-form-credentials">
  <xsl:param name="type" select="'cc'"/>
  <xsl:param name="input-classes"/>

  <td><xsl:value-of select="gsa:i18n ('Credential')"/></td>
  <td>
    <xsl:variable name="credential_id" select="commands_response/get_scanners_response/scanner/credential/@id"/>
    <xsl:variable name="certificate_info" select="commands_response/get_scanners_response/scanner/credential/certificate_info"/>
    <select name="credential_id" class="{$input-classes}">
      <xsl:for-each select="commands_response/get_credentials_response/credential[type = $type]">
        <xsl:call-template name="opt">
          <xsl:with-param name="content" select="name"/>
          <xsl:with-param name="value" select="@id"/>
          <xsl:with-param name="select-value" select="$credential_id"/>
        </xsl:call-template>
      </xsl:for-each>
    </select>
    <a href="#" title="{ gsa:i18n('Create a new Credential') }"
       class="new-action-icon icon icon-sm"
       data-type="credential" data-done="select[name=credential_id]" data-extra="restrict_credential_type={$type}">
      <img src="/img/new.svg"/>
    </a>
    <xsl:if test="string-length ($credential_id) &gt; 0">
      <p class="footnote" style="margin-top:3px;">
        <a href="/omp?cmd=download_credential&amp;credential_id={$credential_id}&amp;package_format=pem&amp;token={/envelope/token}"
          title="{gsa:i18n ('Download Certificate currently in use', 'Action Verb')}"
          class="icon icon-sm">
          <img src="/img/key.svg" alt="{gsa:i18n ('Download Certificate currently in use', 'Action Verb')}"/>
        </a>
        <xsl:call-template name="certificate-status">
          <xsl:with-param name="certificate_info" select="$certificate_info"/>
        </xsl:call-template>
      </p>
    </xsl:if>
  </td>
</xsl:template>

<xsl:template name="html-edit-scanner-form-ca-cert">
  <td><xsl:value-of select="gsa:i18n ('CA Certificate')"/></td>
  <td>
    <xsl:variable name="ca_pub" select="commands_response/get_scanners_response/scanner/ca_pub"/>
    <xsl:variable name="ca_pub_info" select="commands_response/get_scanners_response/scanner/ca_pub_info"/>
    <xsl:choose>
      <xsl:when test="string-length ($ca_pub) &gt; 0">
        <label>
          <input type="radio" name="which_cert" value="existing" checked="1"/>
          <xsl:value-of select="gsa:i18n ('Existing', 'Certificate')"/>
        </label>
        <label>
          <input type="radio" name="which_cert" value="default"/>
          <xsl:value-of select="gsa:i18n ('Default', 'Certificate')"/>
        </label>
      </xsl:when>
      <xsl:otherwise>
        <label>
          <input type="radio" name="which_cert" value="default" checked="1"/>
          <xsl:value-of select="gsa:i18n ('Default', 'Certificate')"/>
        </label>
      </xsl:otherwise>
    </xsl:choose>
    <label>
      <input type="radio" name="which_cert" value="new"/>
      <xsl:value-of select="gsa:i18n ('New:', 'Certificate')"/>
      <input type="file" name="ca_pub"/>
    </label>
    <xsl:if test="string-length ($ca_pub) &gt; 0">
      <p class="footnote" style="margin-top:3px;">
        <a href="/omp?cmd=download_ca_pub&amp;scanner_id={commands_response/get_scanners_response/scanner/@id}&amp;ca_pub={str:encode-uri($ca_pub, true ())}&amp;token={/envelope/token}"
          title="{gsa:i18n ('Download CA Certificate currently in use', 'Action Verb')}"
          class="icon icon-sm">
          <img src="/img/key.svg"
              alt="{gsa:i18n ('Download CA Certificate currently in use', 'Action Verb')}"/>
        </a>
        <xsl:call-template name="certificate-status">
          <xsl:with-param name="certificate_info" select="$ca_pub_info"/>
        </xsl:call-template>
      </p>
    </xsl:if>
  </td>
</xsl:template>

<xsl:template name="html-edit-scanner-form">
  <xsl:variable name="in_use"
                select="commands_response/get_scanners_response/scanner/in_use = 1"/>
  <div class="gb_window">
    <div class="gb_window_part_left"></div>
    <div class="gb_window_part_right"></div>
    <div class="gb_window_part_center"><xsl:value-of select="gsa:i18n ('Edit Scanner')"/>
      <xsl:call-template name="edit-header-icons">
        <xsl:with-param name="cap-type" select="'Scanner'"/>
        <xsl:with-param name="type" select="'scanner'"/>
        <xsl:with-param name="id"
                        select="commands_response/get_scanners_response/scanner/@id"/>
      </xsl:call-template>
    </div>
    <div class="gb_window_part_content">
      <form action="" method="post" enctype="multipart/form-data">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="save_scanner"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden" name="scanner_id"
               value="{commands_response/get_scanners_response/scanner/@id}"/>
        <input type="hidden" name="next" value="{/envelope/params/next}"/>
        <input type="hidden" name="scanner" value="{/envelope/params/scanner}"/>
        <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>
        <table class="table-form">
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
            <td>
              <input type="text" name="name" size="30" maxlength="80"
                     value="{commands_response/get_scanners_response/scanner/name}"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Comment')"/></td>
            <td>
              <input type="text" name="comment" size="30" maxlength="400"
                     value="{commands_response/get_scanners_response/scanner/comment}"/>
            </td>
          </tr>
          <xsl:if test="not (gsa:is_absolute_path (commands_response/get_scanners_response/scanner/host))">
            <tr>
              <td><xsl:value-of select="gsa:i18n ('Host')"/></td>
              <td>
                <xsl:choose>
                  <xsl:when test="$in_use">
                    <input type="text" name="scanner_host" size="30" maxlength="400"
                           disabled="1"
                           value="{commands_response/get_scanners_response/scanner/host}"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="text" name="scanner_host" size="30" maxlength="400"
                           value="{commands_response/get_scanners_response/scanner/host}"/>
                  </xsl:otherwise>
                </xsl:choose>
              </td>
            </tr>
            <tr>
              <td><xsl:value-of select="gsa:i18n ('Port')"/></td>
              <td>
                <xsl:choose>
                  <xsl:when test="$in_use">
                    <input type="text" name="port" size="30" maxlength="400"
                          disabled="1"
                          value="{commands_response/get_scanners_response/scanner/port}"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="text" name="port" size="30" maxlength="400"
                          value="{commands_response/get_scanners_response/scanner/port}"/>
                  </xsl:otherwise>
                </xsl:choose>
              </td>
            </tr>
            <tr>
              <td><xsl:value-of select="gsa:i18n ('Type')"/></td>
              <td>
                <xsl:choose>
                  <xsl:when test="$in_use">
                    <select class="form-selection-control" id="scanner"
                            name="scanner_type" disabled="1">
                      <xsl:call-template name="scanner-type-list">
                        <xsl:with-param name="default" select="commands_response/get_scanners_response/scanner/type"/>
                      </xsl:call-template>
                    </select>
                  </xsl:when>
                  <xsl:otherwise>
                    <select class="form-selection-control" id="scanner"
                            name="scanner_type">
                      <xsl:call-template name="scanner-type-list">
                        <xsl:with-param name="default" select="commands_response/get_scanners_response/scanner/type"/>
                      </xsl:call-template>
                    </select>
                  </xsl:otherwise>
                </xsl:choose>
              </td>
            </tr>
          </xsl:if>
          <tr>
            <xsl:call-template name="html-edit-scanner-form-ca-cert"/>
          </tr>
          <tr class="form-selection-item-scanner form-selection-item-scanner--1">
            <xsl:call-template name="html-edit-scanner-form-credentials">
              <xsl:with-param name="type" select="'cc'"/>
              <xsl:with-param
                name="input-classes"
                select="'form-selection-input-scanner form-selection-input-scanner--1'"/>
            </xsl:call-template>
          </tr>
          <tr class="form-selection-item-scanner form-selection-item-scanner--2">
            <xsl:call-template name="html-edit-scanner-form-credentials">
              <xsl:with-param name="type" select="'cc'"/>
              <xsl:with-param
                name="input-classes"
                select="'form-selection-input-scanner form-selection-input-scanner--2'"/>
            </xsl:call-template>
          </tr>
          <tr class="form-selection-item-scanner form-selection-item-scanner--4">
            <xsl:call-template name="html-edit-scanner-form-credentials">
              <xsl:with-param name="type" select="'up'"/>
              <xsl:with-param
                name="input-classes"
                select="'form-selection-input-scanner form-selection-input-scanner--4'"/>
            </xsl:call-template>
          </tr>
          <tr>
            <td>
              <input type="submit" name="submit" value="{gsa:i18n ('Save Scanner')}"/>
            </td>
          </tr>
        </table>
      </form>
    </div>
  </div>
</xsl:template>

<xsl:template match="edit_scanner">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="modify_scanner_response"/>
  <xsl:call-template name="html-edit-scanner-form"/>
</xsl:template>

<xsl:template match="scanner" mode="details">
  <div class="toolbar">
    <xsl:call-template name="details-header-icons">
      <xsl:with-param name="cap-type" select="'Scanner'"/>
      <xsl:with-param name="type" select="'scanner'"/>
      <xsl:with-param name="grey-clone" select="type = 3"/>
    </xsl:call-template>
    <xsl:call-template name="scanner-icons">
      <xsl:with-param name="scanner_id" select="@id"/>
      <xsl:with-param name="ca_pub" select="ca_pub"/>
      <xsl:with-param name="credential" select="credential"/>
      <xsl:with-param name="next" select="'get_scanner'"/>
    </xsl:call-template>
  </div>

  <div class="section-header">
    <xsl:call-template name="minor-details"/>
    <h1>
      <a href="/omp?cmd=get_scanners&amp;token={/envelope/token}"
         title="{gsa:i18n ('Scanners')}">
        <img class="icon icon-lg" src="/img/scanner.svg" alt="Scanners"/>
      </a>
      <xsl:value-of select="gsa:i18n ('Scanner')"/>:
      <xsl:value-of select="name"/>
      <xsl:text> </xsl:text>
    </h1>
  </div>

  <div class="section-box">
    <table>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Comment')"/>:</td>
        <td><xsl:value-of select="comment"/></td>
      </tr>
      <xsl:if test="not (gsa:is_absolute_path (host))">
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Host')"/>:</td>
          <td>
            <xsl:choose>
              <xsl:when test="type = 3">
                <xsl:value-of select="gsa:i18n('N/A')"/> <i>(<xsl:value-of select="gsa:i18n('builtin scanner')"/>)</i>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="host"/>
              </xsl:otherwise>
            </xsl:choose>
          </td>
        </tr>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Port')"/>:</td>
          <td>
            <xsl:choose>
              <xsl:when test="type = 3">
                <xsl:value-of select="gsa:i18n('N/A')"/> <i>(<xsl:value-of select="gsa:i18n('builtin scanner')"/>)</i>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="port"/>
              </xsl:otherwise>
            </xsl:choose>
          </td>
        </tr>
      </xsl:if>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Type')"/>:</td>
        <td>
          <xsl:call-template name="scanner-type-name">
            <xsl:with-param name="type" select="type"/>
          </xsl:call-template>
        </td>
      </tr>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('Credential')"/>:</td>
        <td>
          <xsl:choose>
            <xsl:when test="gsa:may-op ('get_credentials')">
              <a href="/omp?cmd=get_credential&amp;credential_id={credential/@id}&amp;token={/envelope/token}">
                <xsl:value-of select="credential/name"/>
              </a>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="credential/name"/>
            </xsl:otherwise>
          </xsl:choose>
        </td>
      </tr>
    </table>

    <xsl:choose>
      <!-- OpenVAS Scanner case -->
      <xsl:when test="info/scanner/name = 'OpenVAS'">
      </xsl:when>
      <!-- OSP Scanner is online case -->
      <xsl:when test="string-length(info/scanner/name)">
        <h1><xsl:value-of select="gsa:i18n ('Online Response of Scanner')"/></h1>

        <table>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Scanner Name')"/>:</td>
            <td><xsl:value-of select="info/scanner/name"/></td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Scanner Version')"/>:</td>
            <td><xsl:value-of select="info/scanner/version"/></td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('OSP Daemon')"/>:</td>
            <td>
              <xsl:value-of select="info/daemon/name"/>
              <xsl:text> </xsl:text>
              <xsl:value-of select="info/daemon/version"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Protocol')"/>:</td>
            <td>
              <xsl:value-of select="info/protocol/name"/>
              <xsl:text> </xsl:text>
              <xsl:value-of select="info/protocol/version"/>
            </td>
          </tr>
        </table>

        <h2><xsl:value-of select="gsa:i18n ('Description')"/></h2>

        <p>
          <xsl:call-template name="structured-text">
            <xsl:with-param name="string" select="info/description"/>
          </xsl:call-template>
        </p>

        <h2><xsl:value-of select="gsa:i18n ('Scanner parameters')"/></h2>

        <table class="gbntable">
          <tr class="gbntablehead2">
            <td><xsl:value-of select="gsa:i18n('Name')"/></td>
            <td><xsl:value-of select="gsa:i18n('Description')"/></td>
            <td><xsl:value-of select="gsa:i18n('Type')"/></td>
            <td><xsl:value-of select="gsa:i18n('Default')"/></td>
            <td><xsl:value-of select="gsa:i18n('Mandatory')"/></td>
          </tr>
          <xsl:for-each select="info/params/param">
            <tr class="{gsa:table-row-class(position())}">
              <td><xsl:value-of select="name"/></td>
              <td><xsl:value-of select="description"/></td>
              <td><xsl:value-of select="substring-after(type, 'osp_')"/></td>
              <td>
                <xsl:choose>
                  <xsl:when test="type='osp_ovaldef_file'">OVAL Definitions files list.</xsl:when>
                  <xsl:when test="type='osp_selection'">List.</xsl:when>
                  <xsl:when test="type='osp_boolean' and default = '0'">False</xsl:when>
                  <xsl:when test="type='osp_boolean'">True</xsl:when>
                  <xsl:otherwise>
                    <xsl:value-of select="default"/>
                  </xsl:otherwise>
                </xsl:choose>
              </td>
              <td>
                <xsl:choose>
                  <xsl:when test="mandatory = '0'">False</xsl:when>
                  <xsl:otherwise>True</xsl:otherwise>
                </xsl:choose>
              </td>
            </tr>
          </xsl:for-each>
        </table>

      </xsl:when>
      <!-- OSP Scanner is offline case -->
      <xsl:otherwise>
        <h1><xsl:value-of select="gsa:i18n ('OSP Scanner is offline')"/></h1>
      </xsl:otherwise>
    </xsl:choose>

    <xsl:if test="credential/certificate_info">
      <h3>
        <xsl:value-of select="gsa:i18n ('Client Certificate')"/>
        <xsl:text> </xsl:text>
        (<xsl:value-of select="gsa:i18n ('from Credential')"/>)
      </h3>
      <xsl:call-template name="certificate-info-table">
        <xsl:with-param name="certificate_info" select="credential/certificate_info"/>
      </xsl:call-template>
    </xsl:if>

    <xsl:if test="ca_pub_info">
      <h3><xsl:value-of select="gsa:i18n ('CA Certificate')"/></h3>
      <xsl:call-template name="certificate-info-table">
        <xsl:with-param name="certificate_info" select="ca_pub_info"/>
      </xsl:call-template>
    </xsl:if>
  </div>

  <div class="section-header">
    <a href="#" class="toggle-action-icon icon icon-sm icon-action"
      data-target="#using-box" data-name="Tasks using this Scanner"
      data-variable="using-box--collapsed">
        <img src="/img/fold.svg"/>
    </a>
    <h2>
      <a href="/omp?cmd=get_tasks&amp;token={/envelope/token}"
         title="{gsa:i18n ('Tasks')}">
        <img class="icon icon-sm" src="/img/task.svg" alt="Tasks"/>
      </a>
      <xsl:value-of select="gsa:i18n ('Tasks using this Scanner')"/>
      <xsl:text> </xsl:text>
      <xsl:choose>
        <xsl:when test="count(tasks/task) != 0">
          (<xsl:value-of select="count(tasks/task)"/>)
        </xsl:when>
        <xsl:otherwise>
          (<xsl:value-of select="gsa:i18n ('none')"/>)
        </xsl:otherwise>
      </xsl:choose>
    </h2>
  </div>

  <div class="section-box" id="using-box">
    <table class="gbntable">
      <tr class="gbntablehead2">
        <td><xsl:value-of select="gsa:i18n('Name')"/></td>
      </tr>
      <xsl:for-each select="tasks/task">
        <tr class="{gsa:table-row-class(position())}">
          <xsl:choose>
            <xsl:when test="boolean (permissions) and count (permissions/permission) = 0">
              <td><xsl:value-of select="name"/> (<xsl:value-of select="gsa:i18n('Unavailable')"/>, <xsl:value-of select="gsa:i18n('UUID')"/>: <xsl:value-of select="@id"/>)</td>
            </xsl:when>
            <xsl:otherwise>
              <td>
                <a href="/omp?cmd=get_task&amp;task_id={@id}&amp;token={/envelope/token}" title="{gsa:i18n ('Details')}">
                  <xsl:value-of select="name"/>
                </a>
              </td>
            </xsl:otherwise>
          </xsl:choose>
        </tr>
      </xsl:for-each>
    </table>
  </div>

  <xsl:call-template name="user-tags-window">
    <xsl:with-param name="resource_type" select="'scanner'"/>
  </xsl:call-template>

  <xsl:call-template name="resource-permissions-window">
    <xsl:with-param name="resource_type" select="'scanner'"/>
    <xsl:with-param name="permissions" select="../../permissions/get_permissions_response"/>
    <xsl:with-param name="related">
      <xsl:if test="credential/@id != ''">
        <credential id="{credential/@id}"/>
      </xsl:if>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="get_scanner">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="commands_response/delete_scanner_response"/>
  <xsl:apply-templates select="modify_scanner_response"/>
  <xsl:apply-templates select="delete_tag_response"/>
  <xsl:apply-templates select="create_tag_response"/>
  <xsl:apply-templates select="modify_tag_response"/>
  <xsl:apply-templates select="get_scanners_response/scanner" mode="details"/>
</xsl:template>

<xsl:template match="get_scanners">
  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="delete_scanner_response"/>
  <xsl:apply-templates select="create_filter_response"/>
  <xsl:apply-templates select="create_scanner_response"/>
  <xsl:apply-templates select="modify_scanner_response"/>
  <!-- The for-each makes the get_scanners_response the current node. -->
  <xsl:for-each select="get_scanners_response | commands_response/get_scanners_response">
    <xsl:choose>
      <xsl:when test="substring(@status, 1, 1) = '4' or substring(@status, 1, 1) = '5'">
        <xsl:call-template name="command_result_dialog">
          <xsl:with-param name="operation">
            Get Scanners
          </xsl:with-param>
          <xsl:with-param name="status">
            <xsl:value-of select="@status"/>
          </xsl:with-param>
          <xsl:with-param name="msg">
            <xsl:value-of select="@status_text"/>
          </xsl:with-param>
        </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
        <xsl:call-template name="html-scanners-table"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:for-each>
</xsl:template>

<xsl:template match="verify_scanner_response">
  <xsl:call-template name="command_result_dialog">
    <xsl:with-param name="operation">Verify Scanner</xsl:with-param>
    <xsl:with-param name="status">
      <xsl:value-of select="@status"/>
    </xsl:with-param>
    <xsl:with-param name="msg">
      <xsl:value-of select="@status_text"/>
    </xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="scanner">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <xsl:call-template name="observers-icon">
        <xsl:with-param name="type" select="'Scanner'"/>
      </xsl:call-template>
      <b>
        <a href="/omp?cmd=get_scanner&amp;scanner_id={@id}&amp;filter={str:encode-uri (../filters/term, true ())}&amp;token={/envelope/token}"
           title="{gsa:view_details_title ('Scanner', name)}">
          <xsl:value-of select="name"/>
        </a>
      </b>
      <xsl:choose>
        <xsl:when test="comment != ''">
          <div class="comment">(<xsl:value-of select="comment"/>)</div>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:if test="type != 3 and not (gsa:is_absolute_path (host))">
        <xsl:value-of select="host"/>
      </xsl:if>
    </td>
    <td>
      <xsl:if test="type != 3 and not (gsa:is_absolute_path (host))">
        <xsl:value-of select="port"/>
      </xsl:if>
    </td>
    <td>
      <xsl:call-template name="scanner-type-name">
        <xsl:with-param name="type" select="type"/>
      </xsl:call-template>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="gsa:may-op ('get_credentials')">
          <a href="/omp?cmd=get_credential&amp;credential_id={credential/@id}&amp;token={/envelope/token}">
            <xsl:value-of select="credential/name"/>
          </a>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="credential/name"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <xsl:choose>
      <xsl:when test="/envelope/params/bulk_select = 1">
        <td style="text-align:center">
          <label style="width:100%">
            <input name="bulk_selected:{@id}" type="checkbox" style="width:100%; height:100%" title="{gsa:i18n ('Select for bulk action')}"/>
          </label>
        </td>
      </xsl:when>
      <xsl:otherwise>
        <td class="table-actions">
          <xsl:call-template name="list-window-line-icons">
            <xsl:with-param name="cap-type" select="'Scanner'"/>
            <xsl:with-param name="type" select="'scanner'"/>
            <xsl:with-param name="id" select="@id"/>
            <xsl:with-param name="grey-clone" select="type = 3"/>
          </xsl:call-template>
          <xsl:call-template name="scanner-icons">
            <xsl:with-param name="scanner_id" select="@id"/>
            <xsl:with-param name="ca_pub" select="ca_pub"/>
            <xsl:with-param name="credential" select="credential"/>
            <xsl:with-param name="next" select="'get_scanners'"/>
          </xsl:call-template>
        </td>
      </xsl:otherwise>
    </xsl:choose>
  </tr>
</xsl:template>

<xsl:template match="scanner" mode="trash">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <xsl:call-template name="observers-icon">
        <xsl:with-param name="type" select="'Scanner'"/>
      </xsl:call-template>
      <b><xsl:value-of select="name"/></b>
      <xsl:choose>
        <xsl:when test="comment != ''">
          <div class="comment">(<xsl:value-of select="comment"/>)</div>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:value-of select="host"/>
    </td>
    <td>
      <xsl:value-of select="port"/>
    </td>
    <td>
      <xsl:call-template name="scanner-type-name">
        <xsl:with-param name="type" select="type"/>
      </xsl:call-template>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="credential/trash != '0'">
          <xsl:value-of select="credential/name"/>
          <div>(<xsl:value-of select="gsa:i18n ('in trashcan')"/>)</div>
        </xsl:when>
        <xsl:otherwise>
          <a href="/omp?cmd=get_credential&amp;credential_id={credential/@id}&amp;token={/envelope/token}"
            title="{gsa:i18n ('Credential Details')}">
            <xsl:value-of select="credential/name"/>
          </a>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td class="table-actions">
      <xsl:choose>
        <xsl:when test="not (gsa:may-op ('restore'))"/>
        <xsl:when test="credential/trash = '0'">
          <xsl:call-template name="restore-icon">
            <xsl:with-param name="id" select="@id"/>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <img src="/img/restore_inactive.svg" alt="{gsa:i18n ('Restore')}"
               title="{gsa:i18n ('Credential')}{gsa:i18n (' must be restored first.', 'Trashcan')}"
               class="icon icon-sm"/>
        </xsl:otherwise>
      </xsl:choose>
      <xsl:choose>
        <xsl:when test="in_use='0'">
          <xsl:call-template name="trash-delete-icon">
            <xsl:with-param name="type" select="'scanner'"/>
            <xsl:with-param name="id" select="@id"/>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <img src="/img/delete_inactive.svg"
            alt="{gsa:i18n ('Delete')}"
            title="{gsa:i18n ('Scanner is still in use')}"
            class="icon icon-sm"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
  </tr>
</xsl:template>


<!-- BEGIN GET RAW INFO -->

<xsl:template name="ref_cve_list">
  <xsl:param name="cvelist"/>

  <xsl:variable name="token" select="/envelope/token"/>

  <xsl:variable name="cvecount" select="count(str:split($cvelist, ','))"/>
  <xsl:if test="$cvecount &gt; 0">
    <tr valign="top">
      <td><xsl:value-of select="gsa:i18n('CVE')"/>:</td>
      <td>
        <xsl:for-each select="str:split($cvelist, ',')">
          <xsl:call-template name="get_info_cve_lnk">
            <xsl:with-param name="cve" select="."/>
            <xsl:with-param name="gsa_token" select="$token"/>
          </xsl:call-template>
          <xsl:if test="position() &lt; $cvecount">
            <xsl:text>, </xsl:text>
          </xsl:if>
        </xsl:for-each>
      </td>
    </tr>
  </xsl:if>
</xsl:template>

<xsl:template name="ref_bid_list">
  <xsl:param name="bidlist"/>

  <xsl:variable name="token" select="/envelope/token"/>

  <xsl:variable name="bidcount" select="count(str:split($bidlist, ','))"/>
  <xsl:if test="$bidcount &gt; 0">
    <tr valign="top">
      <td><xsl:value-of select="gsa:i18n('BID')"/>:</td>
      <td>
        <xsl:for-each select="str:split($bidlist, ',')">
          <xsl:value-of select="."/>
          <xsl:if test="position() &lt; $bidcount">
            <xsl:text>, </xsl:text>
          </xsl:if>
        </xsl:for-each>
      </td>
    </tr>
  </xsl:if>
</xsl:template>

<xsl:template name="ref_cert_list">
  <xsl:param name="certlist"/>
  <xsl:variable name="token" select="/envelope/token"/>
  <xsl:variable name="certcount" select="count($certlist/cert_ref)"/>

  <xsl:if test="count($certlist/warning)">
    <xsl:for-each select="$certlist/warning">
      <tr valign="top">
        <td><xsl:value-of select="gsa:i18n('CERT')"/>:</td>
        <td><i><xsl:value-of select="gsa:i18n ('Warning')"/>: <xsl:value-of select="text()"/></i></td>
      </tr>
    </xsl:for-each>
  </xsl:if>

  <xsl:if test="$certcount &gt; 0">
    <tr valign="top">
      <td><xsl:value-of select="gsa:i18n('CERT')"/>:</td>
      <td>
        <xsl:for-each select="$certlist/cert_ref">
          <xsl:choose>
            <xsl:when test="@type='CERT-Bund'">
              <xsl:call-template name="get_info_cert_bund_adv_lnk">
                <xsl:with-param name="cert_bund_adv" select="@id"/>
                <xsl:with-param name="gsa_token" select="$token"/>
              </xsl:call-template>
            </xsl:when>
            <xsl:when test="@type='DFN-CERT'">
              <xsl:call-template name="get_info_dfn_cert_adv_lnk">
                <xsl:with-param name="dfn_cert_adv" select="@id"/>
                <xsl:with-param name="gsa_token" select="$token"/>
              </xsl:call-template>
            </xsl:when>
            <xsl:otherwise>
              <b>?</b><xsl:value-of select="./@id"/>
            </xsl:otherwise>
          </xsl:choose>
          <xsl:if test="position() &lt; $certcount">
            <xsl:text>, </xsl:text>
          </xsl:if>
        </xsl:for-each>
      </td>
    </tr>
  </xsl:if>
</xsl:template>

<xsl:template name="ref_xref_list">
  <xsl:param name="xreflist"/>

  <xsl:variable name="token" select="/envelope/token"/>

  <xsl:variable name="xrefcount" select="count(str:split($xreflist, ','))"/>
  <xsl:if test="$xrefcount &gt; 0">
    <xsl:for-each select="str:split($xreflist, ',')">
      <tr valign="top">
        <td><xsl:if test="position()=1"><xsl:value-of select="gsa:i18n ('Other', 'SecInfo|References')"/>:</xsl:if></td>
        <xsl:choose>
          <xsl:when test="contains(., 'URL:')">
            <td><a class="external" href="{substring-after(., 'URL:')}"><xsl:value-of select="substring-after(., 'URL:')"/></a></td>
          </xsl:when>
          <xsl:otherwise>
            <td><xsl:value-of select="."/></td>
          </xsl:otherwise>
        </xsl:choose>
      </tr>
    </xsl:for-each>
  </xsl:if>
</xsl:template>

<xsl:template match="info/cpe">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <b>
        <xsl:call-template name="get_info_cpe_lnk">
          <xsl:with-param name="cpe" select="../name"/>
          <xsl:with-param name="cpe_id" select="../@id"/>
        </xsl:call-template>
      </b>
      <xsl:choose>
        <xsl:when test="../comment != ''">
          <div class="comment">(<xsl:value-of select="../comment"/>)</div>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="title != ''">
          <xsl:value-of select="title"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="gsa:i18n ('N/A')"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="../modification_time != ''">
          <xsl:value-of select="gsa:date (../modification_time)"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="gsa:i18n ('N/A')"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:value-of select="cve_refs"/>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="max_cvss &gt;= 0.0">
          <xsl:call-template name="severity-bar">
            <xsl:with-param name="cvss" select="max_cvss"/>
            <xsl:with-param name="scale" select="7"/>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="severity-bar">
            <xsl:with-param name="extra_text" select="gsa:i18n ('N/A')"/>
            <xsl:with-param name="scale" select="7"/>
          </xsl:call-template>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <xsl:choose>
      <xsl:when test="/envelope/params/bulk_select = 1">
        <td style="text-align:center">
          <label style="width:100%">
            <input name="bulk_selected:{../@id}" type="checkbox" style="width:100%; height:100%" title="{gsa:i18n ('Select for bulk action')}"/>
          </label>
        </td>
      </xsl:when>
      <xsl:otherwise></xsl:otherwise>
    </xsl:choose>
  </tr>
</xsl:template>

<xsl:template match="info/cve">

  <tbody class="{gsa:table-row-class(position())}">
    <tr>
      <td rowspan="2">
        <b>
          <xsl:call-template name="get_info_cve_lnk">
            <xsl:with-param name="cve" select="../name"/>
            <xsl:with-param name="cve_id" select="../@id"/>
          </xsl:call-template>
        </b>
        <xsl:choose>
          <xsl:when test="../comment != ''">
            <div class="comment">(<xsl:value-of select="../comment"/>)</div>
          </xsl:when>
          <xsl:otherwise></xsl:otherwise>
        </xsl:choose>
      </td>
      <td>
        <xsl:choose>
          <xsl:when test="vector != ''">
            <xsl:value-of select="vector"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="gsa:i18n ('N/A')"/>
          </xsl:otherwise>
        </xsl:choose>
      </td>
      <td>
        <xsl:choose>
          <xsl:when test="complexity != ''">
            <xsl:value-of select="complexity"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="gsa:i18n ('N/A')"/>
          </xsl:otherwise>
        </xsl:choose>
      </td>
      <td>
        <xsl:choose>
          <xsl:when test="authentication != ''">
            <xsl:value-of select="authentication"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="gsa:i18n ('N/A')"/>
          </xsl:otherwise>
        </xsl:choose>
      </td>
      <td>
        <xsl:choose>
          <xsl:when test="confidentiality_impact != ''">
            <xsl:value-of select="confidentiality_impact"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="gsa:i18n ('N/A')"/>
          </xsl:otherwise>
        </xsl:choose>
      </td>
      <td>
        <xsl:choose>
          <xsl:when test="integrity_impact != ''">
            <xsl:value-of select="integrity_impact"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="gsa:i18n ('N/A')"/>
          </xsl:otherwise>
        </xsl:choose>
      </td>
      <td>
        <xsl:choose>
          <xsl:when test="availability_impact != ''">
            <xsl:value-of select="availability_impact"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="gsa:i18n ('N/A')"/>
          </xsl:otherwise>
        </xsl:choose>
      </td>
      <td>
        <xsl:choose>
          <xsl:when test="../creation_time != ''">
            <xsl:value-of select="gsa:date (../creation_time)"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="gsa:i18n ('N/A')"/>
          </xsl:otherwise>
        </xsl:choose>
      </td>
      <td>
        <xsl:choose>
          <xsl:when test="cvss &gt;= 0.0">
            <xsl:call-template name="severity-bar">
              <xsl:with-param name="cvss" select="cvss"/>
              <xsl:with-param name="scale" select="7"/>
            </xsl:call-template>
          </xsl:when>
          <xsl:otherwise>
            <xsl:call-template name="severity-bar">
              <xsl:with-param name="extra_text" select="gsa:i18n ('N/A')"/>
              <xsl:with-param name="scale" select="7"/>
            </xsl:call-template>
          </xsl:otherwise>
        </xsl:choose>
      </td>
      <xsl:choose>
        <xsl:when test="/envelope/params/bulk_select = 1">
          <td style="text-align:center" rowspan="2">
            <label style="width:100%">
              <input name="bulk_selected:{../@id}" type="checkbox" style="width:100%; height:100%" title="{gsa:i18n ('Select for bulk action')}"/>
            </label>
          </td>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </tr>
    <tr>
      <td colspan="8" style="font-size: 80%;">
        <xsl:variable name="truncate_length"
        select="string-length(description) - string-length(substring-after(substring(description, 250), ' ')) + 1"/>
        <xsl:value-of select="substring(description, 0, $truncate_length)"/>
        <xsl:if test="string-length(description) >= $truncate_length"><i><abbr title="[...] {substring(description, $truncate_length, string-length(description))}">[...]</abbr></i></xsl:if>
      </td>
    </tr>
  </tbody>
</xsl:template>

<xsl:template match="info/nvt">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <b>
        <xsl:call-template name="get_info_nvt_lnk">
          <xsl:with-param name="nvt" select="../name"/>
          <xsl:with-param name="oid" select="@oid"/>
        </xsl:call-template>
      </b>
    </td>
    <td>
      <a href="/omp?cmd=get_info&amp;info_type=nvt&amp;filter=family=&quot;{family}&quot;&amp;token={/envelope/token}">
        <xsl:value-of select="family"/>
      </a>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="../creation_time != ''">
          <xsl:value-of select="gsa:date (../creation_time)"/>
        </xsl:when>
        <xsl:otherwise>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="../modification_time != ''">
          <xsl:value-of select="gsa:date (../modification_time)"/>
        </xsl:when>
        <xsl:otherwise>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:value-of select="version"/>
    </td>
    <td>
      <!-- "NOCVE" means no CVE ID, skip. -->
      <xsl:choose>
        <xsl:when test="cve_id = 'NOCVE'">
        </xsl:when>
        <xsl:otherwise>
          <!-- get the GSA token before entering the for-each loop over the str:tokenize elements -->
          <xsl:variable name="gsa_token" select="/envelope/token"/>

          <xsl:for-each select="str:tokenize (cve_id, ', ')">
            <xsl:call-template name="get_info_cve_lnk">
              <xsl:with-param name="cve" select="text()"/>
              <xsl:with-param name="gsa_token" select="$gsa_token"/>
            </xsl:call-template>
            <xsl:text> </xsl:text>
          </xsl:for-each>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:variable name="solution_type">
        <xsl:for-each select="str:split (tags, '|')">
          <xsl:if test="'solution_type' = substring-before (., '=')">
            <xsl:value-of select="substring-after (., '=')"/>
          </xsl:if>
        </xsl:for-each>
      </xsl:variable>
      <xsl:call-template name="solution-icon">
        <xsl:with-param name="solution_type" select="$solution_type"/>
      </xsl:call-template>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="cvss_base &gt;= 0.0">
          <xsl:call-template name="severity-bar">
            <xsl:with-param name="cvss" select="cvss_base"/>
            <xsl:with-param name="scale" select="7"/>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="severity-bar">
            <xsl:with-param name="extra_text" select="gsa:i18n ('N/A')"/>
            <xsl:with-param name="scale" select="7"/>
          </xsl:call-template>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="qod/value != ''">
          <xsl:value-of select="qod/value"/>%
        </xsl:when>
        <xsl:otherwise/>
      </xsl:choose>
    </td>
    <xsl:choose>
      <xsl:when test="/envelope/params/bulk_select = 1">
        <td style="text-align:center">
          <label style="width:100%">
            <input name="bulk_selected:{../@id}" type="checkbox" style="width:100%; height:100%" title="{gsa:i18n ('Select for bulk action')}"/>
          </label>
        </td>
      </xsl:when>
      <xsl:otherwise></xsl:otherwise>
    </xsl:choose>
  </tr>
</xsl:template>

<xsl:template match="info/ovaldef">

  <tbody class="{gsa:table-row-class(position())}">
    <tr>
      <td rowspan="2">
        <div>
          <b>
            <xsl:call-template name="get_info_ovaldef_lnk">
              <xsl:with-param name="ovaldef" select="../name"/>
              <xsl:with-param name="ovaldef_id" select="../@id"/>
            </xsl:call-template>
          </b>
        </div>
        <span style="font-size:80%; color:grey">
          <xsl:choose>
            <xsl:when test="string-length(file) > 45">
              <abbr title="{file}"><i>[...]</i><xsl:value-of select="substring(file, string-length(file)-40, string-length(file))"/></abbr>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="file"/>
            </xsl:otherwise>
          </xsl:choose>
        </span>
        <xsl:choose>
          <xsl:when test="../comment != ''">
            <div class="comment">(<xsl:value-of select="../comment"/>)</div>
          </xsl:when>
          <xsl:otherwise></xsl:otherwise>
        </xsl:choose>
      </td>
      <td>
        <xsl:choose>
          <xsl:when test="version != ''">
            <xsl:value-of select="version"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="gsa:i18n ('N/A')"/>
          </xsl:otherwise>
        </xsl:choose>
      </td>
      <td>
        <xsl:choose>
          <xsl:when test="status != ''">
            <xsl:value-of select="status"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="gsa:i18n ('N/A')"/>
          </xsl:otherwise>
        </xsl:choose>
      </td>
      <td>
        <xsl:choose>
          <xsl:when test="class != ''">
            <xsl:value-of select="class"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="gsa:i18n ('N/A')"/>
          </xsl:otherwise>
        </xsl:choose>
      </td>
      <td>
        <xsl:choose>
          <xsl:when test="../creation_time != ''">
            <xsl:value-of select="gsa:date (../creation_time)"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="gsa:i18n ('N/A')"/>
          </xsl:otherwise>
        </xsl:choose>
      </td>
      <td>
        <xsl:choose>
          <xsl:when test="../modification_time != ''">
            <xsl:value-of select="gsa:date (../modification_time)"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="gsa:i18n ('N/A')"/>
          </xsl:otherwise>
        </xsl:choose>
      </td>
      <td>
        <xsl:value-of select="cve_refs"/>
      </td>
      <td>
        <xsl:choose>
          <xsl:when test="max_cvss &gt;= 0.0">
            <xsl:call-template name="severity-bar">
              <xsl:with-param name="cvss" select="max_cvss"/>
              <xsl:with-param name="scale" select="7"/>
            </xsl:call-template>
          </xsl:when>
          <xsl:otherwise>
            <xsl:call-template name="severity-bar">
              <xsl:with-param name="extra_text" select="gsa:i18n ('N/A')"/>
              <xsl:with-param name="scale" select="7"/>
            </xsl:call-template>
          </xsl:otherwise>
        </xsl:choose>
      </td>
      <xsl:choose>
        <xsl:when test="/envelope/params/bulk_select = 1">
          <td style="text-align:center" rowspan="2">
            <label style="width:100%">
              <input name="bulk_selected:{../@id}" type="checkbox" style="width:100%; height:100%" title="{gsa:i18n ('Select for bulk action')}"/>
            </label>
          </td>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </tr>
    <tr>
      <td colspan="7" style="font-size: 80%;">
        <xsl:choose>
          <xsl:when test="title != ''">
            <xsl:variable name="truncate_length"
              select="string-length(title) - string-length(substring-after(substring(title, 200), ' ')) + 1"/>
            <xsl:value-of select="substring(title, 0, $truncate_length)"/>
            <xsl:if test="string-length(title) >= $truncate_length"><i><abbr title="[...] {substring(title, $truncate_length, string-length(title))}">[...]</abbr></i></xsl:if>
          </xsl:when>
          <xsl:otherwise>
          </xsl:otherwise>
        </xsl:choose>
      </td>
    </tr>
  </tbody>
</xsl:template>

<xsl:template match="info/cert_bund_adv">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <b>
        <xsl:call-template name="get_info_cert_bund_adv_lnk">
          <xsl:with-param name="cert_bund_adv" select="../name"/>
          <xsl:with-param name="cert_bund_adv_id" select="../@id"/>
        </xsl:call-template>
      </b>
      <xsl:choose>
        <xsl:when test="../comment != ''">
          <div class="comment">(<xsl:value-of select="../comment"/>)</div>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:value-of select="title"/>
    </td>
    <td>
      <xsl:value-of select="gsa:date (../creation_time)"/>
    </td>
    <td>
      <xsl:value-of select="cve_refs"/>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="max_cvss &gt;= 0.0">
          <xsl:call-template name="severity-bar">
            <xsl:with-param name="cvss" select="max_cvss"/>
            <xsl:with-param name="scale" select="7"/>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="severity-bar">
            <xsl:with-param name="extra_text" select="gsa:i18n ('N/A')"/>
            <xsl:with-param name="scale" select="7"/>
          </xsl:call-template>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <xsl:choose>
      <xsl:when test="/envelope/params/bulk_select = 1">
        <td style="text-align:center">
          <label style="width:100%">
            <input name="bulk_selected:{../@id}" type="checkbox" style="width:100%; height:100%" title="{gsa:i18n ('Select for bulk action')}"/>
          </label>
        </td>
      </xsl:when>
      <xsl:otherwise></xsl:otherwise>
    </xsl:choose>
  </tr>
</xsl:template>

<xsl:template match="info/dfn_cert_adv">
  <tr class="{gsa:table-row-class(position())}">
    <td>
      <b>
        <xsl:call-template name="get_info_dfn_cert_adv_lnk">
          <xsl:with-param name="dfn_cert_adv" select="../name"/>
          <xsl:with-param name="dfn_cert_adv_id" select="../@id"/>
        </xsl:call-template>
      </b>
      <xsl:choose>
        <xsl:when test="../comment != ''">
          <div class="comment">(<xsl:value-of select="../comment"/>)</div>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </td>
    <td>
      <xsl:value-of select="title"/>
    </td>
    <td>
      <xsl:value-of select="gsa:date (../creation_time)"/>
    </td>
    <td>
      <xsl:value-of select="cve_refs"/>
    </td>
    <td>
      <xsl:choose>
        <xsl:when test="max_cvss &gt;= 0.0">
          <xsl:call-template name="severity-bar">
            <xsl:with-param name="cvss" select="max_cvss"/>
            <xsl:with-param name="scale" select="7"/>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="severity-bar">
            <xsl:with-param name="extra_text" select="gsa:i18n ('N/A')"/>
            <xsl:with-param name="scale" select="7"/>
          </xsl:call-template>
        </xsl:otherwise>
      </xsl:choose>
    </td>
    <xsl:choose>
      <xsl:when test="/envelope/params/bulk_select = 1">
        <td style="text-align:center">
          <label style="width:100%">
            <input name="bulk_selected:{../@id}" type="checkbox" style="width:100%; height:100%" title="{gsa:i18n ('Select for bulk action')}"/>
          </label>
        </td>
      </xsl:when>
      <xsl:otherwise></xsl:otherwise>
    </xsl:choose>
  </tr>
</xsl:template>

<xsl:template match="info/allinfo">

  <tbody class="{gsa:table-row-class(position())}">
    <tr>
      <td rowspan="2">
        <b>
          <xsl:call-template name="get_info_allinfo_lnk">
            <xsl:with-param name="name" select="../name"/>
            <xsl:with-param name="type" select="type"/>
            <xsl:with-param name="id" select="../@id"/>
          </xsl:call-template>
        </b>
        <xsl:choose>
          <xsl:when test="../comment != ''">
            <div class="comment">(<xsl:value-of select="../comment"/>)</div>
          </xsl:when>
          <xsl:otherwise></xsl:otherwise>
        </xsl:choose>
      </td>
      <td>
        <xsl:choose>
          <xsl:when test="type != ''">
            <xsl:value-of select="type"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="gsa:i18n ('N/A')"/>
          </xsl:otherwise>
        </xsl:choose>
      </td>
      <td>
        <xsl:choose>
          <xsl:when test="../creation_time != ''">
            <xsl:value-of select="gsa:date (../creation_time)"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="gsa:i18n ('N/A')"/>
          </xsl:otherwise>
        </xsl:choose>
      </td>
      <td>
        <xsl:choose>
          <xsl:when test="../modification_time != ''">
            <xsl:value-of select="gsa:date (../modification_time)"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="gsa:i18n ('N/A')"/>
          </xsl:otherwise>
        </xsl:choose>
      </td>
      <td>
        <xsl:choose>
          <xsl:when test="severity &gt;= 0.0">
            <xsl:call-template name="severity-bar">
              <xsl:with-param name="cvss" select="severity"/>
            </xsl:call-template>
          </xsl:when>
          <xsl:otherwise>
            <xsl:call-template name="severity-bar">
              <xsl:with-param name="extra_text" select="gsa:i18n ('N/A')"/>
            </xsl:call-template>
          </xsl:otherwise>
        </xsl:choose>
      </td>
      <xsl:choose>
        <xsl:when test="/envelope/params/bulk_select = 1">
          <td style="text-align:center" rowspan="2">
            <label style="width:100%">
              <input name="bulk_selected:{../@id}" type="checkbox" style="width:100%; height:100%" title="{gsa:i18n ('Select for bulk action')}"/>
            </label>
          </td>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
      </xsl:choose>
    </tr>
    <tr>
      <td colspan="4" style="font-size: 95%;">
        <xsl:variable name="summary">
          <xsl:choose>
            <xsl:when test="contains(extra, 'summary=')">
              <xsl:for-each select="str:split (extra, '|')">
                <xsl:if test="'summary' = substring-before (., '=')">
                  <xsl:value-of select="substring-after (., '=')"/><br />
                </xsl:if>
              </xsl:for-each>
            </xsl:when>
            <xsl:otherwise>
             <xsl:value-of select="extra"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:variable>
        <xsl:variable name="truncate_length" select="string-length($summary) - string-length(substring-after(substring($summary, 135), ' ')) + 1"/>
        <xsl:value-of select="substring($summary, 0, $truncate_length)"/>
        <xsl:if test="string-length($summary) >= $truncate_length"><i><abbr title="[...] {substring(extra, $truncate_length, string-length($summary))}">[...]</abbr></i></xsl:if>
      </td>
    </tr>
  </tbody>
</xsl:template>

<xsl:template name="get_info_allinfo_lnk">
 <xsl:param name="name"/>
 <xsl:param name="type"/>
 <xsl:param name="id"/>
  <xsl:choose>
    <xsl:when test="$type = 'cve'">
      <xsl:call-template name="get_info_cve_lnk">
        <xsl:with-param name="cve" select="$name"/>
        <xsl:with-param name="cve_id" select="$id"/>
      </xsl:call-template>
    </xsl:when>
    <xsl:when test="$type = 'cpe'">
      <xsl:call-template name="get_info_cpe_lnk">
        <xsl:with-param name="cpe" select="$name"/>
        <xsl:with-param name="cpe_id" select="$id"/>
        <xsl:with-param name="no_icon" select="1"/>
      </xsl:call-template>
    </xsl:when>
    <xsl:when test="$type = 'ovaldef'">
      <xsl:call-template name="get_info_ovaldef_lnk">
        <xsl:with-param name="ovaldef" select="$name"/>
        <xsl:with-param name="ovaldef_id" select="$id"/>
      </xsl:call-template>
    </xsl:when>
    <xsl:when test="$type = 'nvt'">
      <xsl:call-template name="get_info_nvt_lnk">
        <xsl:with-param name="nvt" select="$name"/>
        <xsl:with-param name="oid" select="$id"/>
      </xsl:call-template>
    </xsl:when>
    <xsl:when test="$type = 'cert_bund_adv'">
      <xsl:call-template name="get_info_cert_bund_adv_lnk">
        <xsl:with-param name="cert_bund_adv" select="$name"/>
        <xsl:with-param name="cert_bund_adv_id" select="$id"/>
      </xsl:call-template>
    </xsl:when>
    <xsl:when test="$type = 'dfn_cert_adv'">
      <xsl:call-template name="get_info_dfn_cert_adv_lnk">
        <xsl:with-param name="dfn_cert_adv" select="$name"/>
        <xsl:with-param name="dfn_cert_adv_id" select="$id"/>
      </xsl:call-template>
    </xsl:when>
  </xsl:choose>
</xsl:template>

<xsl:template name="get_info_cpe_lnk">
  <xsl:param name="cpe"/>
  <xsl:param name="cpe_id"/>
  <xsl:param name="no_icon"/>
  <xsl:param name="hide_other_icon"/>
  <xsl:variable name="cpe_select">
    <xsl:choose>
      <xsl:when test="$cpe_id">info_id=<xsl:value-of select="str:encode-uri (str:replace($cpe_id, '&amp;','&amp;amp;'), true())"/></xsl:when>
      <xsl:otherwise>info_name=<xsl:value-of select="str:encode-uri (str:replace($cpe, '&amp;','&amp;amp;'), true())"/></xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <div class="cpe">
    <xsl:if test="not($no_icon)">
      <xsl:call-template name="cpe-icon">
        <xsl:with-param name="cpe" select="$cpe"/>
        <xsl:with-param name="hide_other" select="$hide_other_icon"/>
      </xsl:call-template>
    </xsl:if>
    <a href="/omp?cmd=get_info&amp;info_type=cpe&amp;{$cpe_select}&amp;details=1&amp;filter={str:encode-uri (../../filters/term, true ())}&amp;filt_id={../../filters/@id}&amp;token={/envelope/token}"
       title="{gsa:view_details_title ('CPE', $cpe)}">
      <xsl:value-of select="$cpe"/>
    </a>
  </div>
</xsl:template>

<xsl:template name="get_info_cve_lnk">
  <xsl:param name="cve"/>
  <xsl:param name="cve_id"/>
  <xsl:param name="gsa_token"/>
  <xsl:variable name="cve_select">
    <xsl:choose>
      <xsl:when test="$cve_id">info_id=<xsl:value-of select="$cve_id"/></xsl:when>
      <xsl:otherwise>info_name=<xsl:value-of select="normalize-space($cve)"/></xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:choose>
    <xsl:when test="$gsa_token = ''">
      <a href="/omp?cmd=get_info&amp;info_type=cve&amp;{$cve_select}&amp;details=1&amp;filter={str:encode-uri (../../filters/term, true ())}&amp;filt_id={../../filters/@id}&amp;token={/envelope/token}"
         title="{gsa:view_details_title ('CVE', $cve)}"><xsl:value-of select="normalize-space($cve)"/></a>
    </xsl:when>
    <xsl:otherwise>
      <a href="/omp?cmd=get_info&amp;info_type=cve&amp;{$cve_select}&amp;details=1&amp;filter={str:encode-uri (../../filters/term, true ())}&amp;filt_id={../../filters/@id}&amp;token={$gsa_token}"
         title="{gsa:i18n ('Details')}"><xsl:value-of select="normalize-space($cve)"/></a>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="get_info_nvt_lnk">
  <xsl:param name="nvt"/>
  <xsl:param name="oid"/>
  <xsl:param name="gsa_token"/>
  <xsl:variable name="nvt_select">
    <xsl:choose>
      <xsl:when test="$oid">info_id=<xsl:value-of select="$oid"/></xsl:when>
      <xsl:otherwise>info_name=<xsl:value-of select="normalize-space($nvt)"/></xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:choose>
    <xsl:when test="$gsa_token = ''">
      <a href="/omp?cmd=get_info&amp;info_type=nvt&amp;{$nvt_select}&amp;details=1&amp;filter={str:encode-uri (../../filters/term, true ())}&amp;filt_id={../../filters/@id}&amp;token={/envelope/token}"
         title="{gsa:view_details_title ('NVT', $oid)}"><xsl:value-of select="normalize-space($nvt)"/></a>
    </xsl:when>
    <xsl:otherwise>
      <a href="/omp?cmd=get_info&amp;info_type=nvt&amp;{$nvt_select}&amp;details=1&amp;filter={str:encode-uri (../../filters/term, true ())}&amp;filt_id={../../filters/@id}&amp;token={$gsa_token}"
         title="{gsa:view_details_title ('NVT', $oid)}"><xsl:value-of select="normalize-space($nvt)"/></a>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="get_info_ovaldef_lnk">
  <xsl:param name="ovaldef"/>
  <xsl:param name="ovaldef_id"/>
  <xsl:variable name="ovaldef_select">
    <xsl:choose>
      <xsl:when test="$ovaldef_id">info_id=<xsl:value-of select="$ovaldef_id"/></xsl:when>
      <xsl:otherwise>info_name=<xsl:value-of select="$ovaldef"/></xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <a href="/omp?cmd=get_info&amp;info_type=ovaldef&amp;{$ovaldef_select}&amp;details=1&amp;filter={str:encode-uri (../../filters/term, true ())}&amp;token={/envelope/token}"
     title="{gsa:view_details_title ('OVAL Definition', $ovaldef)}">
     <xsl:value-of select="$ovaldef"/>
  </a>
</xsl:template>

<xsl:template name="get_info_cert_bund_adv_lnk">
  <xsl:param name="cert_bund_adv"/>
  <xsl:param name="cert_bund_adv_id"/>
  <xsl:variable name="cert_bund_adv_select">
    <xsl:choose>
      <xsl:when test="$cert_bund_adv_id">info_id=<xsl:value-of select="$cert_bund_adv_id"/></xsl:when>
      <xsl:otherwise>info_name=<xsl:value-of select="$cert_bund_adv"/></xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <a href="/omp?cmd=get_info&amp;info_type=cert_bund_adv&amp;{$cert_bund_adv_select}&amp;details=1&amp;filter={str:encode-uri (../../filters/term, true ())}&amp;token={/envelope/token}"
     title="{gsa:view_details_title ('CERT-Bund Advisory', $cert_bund_adv)}">
    <xsl:value-of select="$cert_bund_adv"/>
  </a>
</xsl:template>

<xsl:template name="get_info_dfn_cert_adv_lnk">
  <xsl:param name="dfn_cert_adv"/>
  <xsl:param name="dfn_cert_adv_id"/>
  <xsl:variable name="dfn_cert_adv_select">
    <xsl:choose>
      <xsl:when test="$dfn_cert_adv_id">info_id=<xsl:value-of select="$dfn_cert_adv_id"/></xsl:when>
      <xsl:otherwise>info_name=<xsl:value-of select="$dfn_cert_adv"/></xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <a href="/omp?cmd=get_info&amp;info_type=dfn_cert_adv&amp;{$dfn_cert_adv_select}&amp;details=1&amp;filter={str:encode-uri (../../filters/term, true ())}&amp;token={/envelope/token}"
     title="{gsa:view_details_title ('DFN-CERT Advisory', $dfn_cert_adv)}">
    <xsl:value-of select="$dfn_cert_adv"/>
  </a>
</xsl:template>

<xsl:template name="html-cpe-table">
  <xsl:if test="@status = 400">
    <div>
      <xsl:call-template name="error_window">
        <xsl:with-param name="heading">Warning: SecInfo Database Missing</xsl:with-param>
        <xsl:with-param name="message">
          SCAP and/or CERT database missing on OMP server.
          <a href="/help/cpes.html?token={/envelope/token}#secinfo_missing"
            class="icon icon-sm"
            title="Help: SecInfo database missing">
            <img src="/img/help.svg"/>
          </a>
        </xsl:with-param>
      </xsl:call-template>
    </div>
  </xsl:if>
  <xsl:call-template name="list-window">
    <xsl:with-param name="type" select="'info'"/>
    <xsl:with-param name="subtype" select="'cpe'"/>
    <xsl:with-param name="cap-type" select="'CPE'"/>
    <xsl:with-param name="resources-summary" select="info"/>
    <xsl:with-param name="resources" select="info/cpe"/>
    <xsl:with-param name="count" select="count (info/cpe)"/>
    <xsl:with-param name="filtered-count" select="info_count/filtered"/>
    <xsl:with-param name="full-count" select="info_count/text ()"/>
    <xsl:with-param name="top-visualization">
      <xsl:call-template name="init-d3charts"/>
      <xsl:call-template name="js-secinfo-top-visualization">
        <xsl:with-param name="type" select="'cpe'"/>
      </xsl:call-template>
    </xsl:with-param>
    <xsl:with-param name="columns" xmlns="">
      <column>
        <name><xsl:value-of select="gsa:i18n('Name')"/></name>
        <field>name</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Title')"/></name>
        <field>title</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Modified', 'Date')"/></name>
        <field>modified</field>
        <sort-reverse/>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('CVEs')"/></name>
        <field>cves</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Severity', 'Severity Short')"/></name>
        <field>severity</field>
        <sort-reverse/>
      </column>
    </xsl:with-param>
    <xsl:with-param name="icon-count" select="/envelope/params/bulk_select = 1"/>
  </xsl:call-template>
</xsl:template>

<xsl:template name="html-cve-table">
  <xsl:if test="@status = 400">
    <div>
      <xsl:call-template name="error_window">
        <xsl:with-param name="heading">Warning: SecInfo Database Missing</xsl:with-param>
        <xsl:with-param name="message">
          SCAP and/or CERT database missing on OMP server.
          <a href="/help/cves.html?token={/envelope/token}#secinfo_missing"
            class="icon icon-sm"
            title="Help: SecInfo database missing">
            <img src="/img/help.svg"/>
          </a>
        </xsl:with-param>
      </xsl:call-template>
    </div>
  </xsl:if>
  <xsl:call-template name="list-window">
    <xsl:with-param name="type" select="'info'"/>
    <xsl:with-param name="subtype" select="'cve'"/>
    <xsl:with-param name="cap-type" select="'CVE'"/>
    <xsl:with-param name="resources-summary" select="info"/>
    <xsl:with-param name="resources" select="info/cve"/>
    <xsl:with-param name="count" select="count (info/cve)"/>
    <xsl:with-param name="filtered-count" select="info_count/filtered"/>
    <xsl:with-param name="full-count" select="info_count/text ()"/>

    <xsl:with-param name="top-visualization">
      <xsl:call-template name="init-d3charts"/>
      <xsl:call-template name="js-secinfo-top-visualization">
        <xsl:with-param name="type" select="'cve'"/>
      </xsl:call-template>
    </xsl:with-param>
    <xsl:with-param name="columns" xmlns="">
      <column>
        <name><xsl:value-of select="gsa:i18n('Name')"/></name>
        <field>name</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Vector', 'CVSS Vector Short')"/></name>
        <field>vector</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Complexity', 'CVSS Vector Short')"/></name>
        <field>complexity</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Authentication', 'CVSS Vector Short')"/></name>
        <field>authentication</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Confidentiality Impact', 'CVSS Vector Short')"/></name>
        <field>confidentiality_impact</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Integrity Impact', 'CVSS Vector Short')"/></name>
        <field>integrity_impact</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Availability Impact', 'CVSS Vector Short')"/></name>
        <field>availability_impact</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Published', 'Date')"/></name>
        <field>published</field>
        <sort-reverse/>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Severity', 'Severity Short')"/></name>
        <field>severity</field>
        <sort-reverse/>
      </column>
    </xsl:with-param>
    <xsl:with-param name="icon-count" select="/envelope/params/bulk_select = 1"/>
  </xsl:call-template>
</xsl:template>

<xsl:template name="html-nvt-table">
  <xsl:if test="@status = 400">
    <div>
      <xsl:call-template name="error_window">
        <xsl:with-param name="heading">Warning: SecInfo Database Missing</xsl:with-param>
        <xsl:with-param name="message">
          SCAP and/or CERT database missing on OMP server.
          <a href="/help/nvts.html?token={/envelope/token}#secinfo_missing"
            class="icon icon-sm"
            title="Help: SecInfo database missing">
            <img src="/img/help.svg"/>
          </a>
        </xsl:with-param>
      </xsl:call-template>
    </div>
  </xsl:if>
  <xsl:call-template name="list-window">
    <xsl:with-param name="type" select="'info'"/>
    <xsl:with-param name="subtype" select="'nvt'"/>
    <xsl:with-param name="cap-type" select="'NVT'"/>
    <xsl:with-param name="resources-summary" select="info"/>
    <xsl:with-param name="resources" select="info/nvt"/>
    <xsl:with-param name="count" select="count (info/nvt)"/>
    <xsl:with-param name="filtered-count" select="info_count/filtered"/>
    <xsl:with-param name="full-count" select="info_count/text ()"/>
    <xsl:with-param name="top-visualization">
      <xsl:call-template name="init-d3charts"/>
      <xsl:call-template name="js-secinfo-top-visualization">
        <xsl:with-param name="type" select="'nvt'"/>
      </xsl:call-template>
    </xsl:with-param>
    <xsl:with-param name="columns" xmlns="">
      <column>
        <name><xsl:value-of select="gsa:i18n('Name')"/></name>
        <field>name</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Family')"/></name>
        <field>family</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Created', 'Date')"/></name>
        <field>created</field>
        <sort-reverse/>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Modified', 'Date')"/></name>
        <field>modified</field>
        <sort-reverse/>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Version')"/></name>
        <field>version</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('CVE')"/></name>
        <field>cve</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Solution type')"/></name>
        <field>solution_type</field>
        <image>/img/solution_type.svg</image>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Severity', 'Severity Short')"/></name>
        <field>severity</field>
        <sort-reverse/>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('QoD')"/></name>
        <field>qod</field>
        <sort-reverse/>
      </column>
    </xsl:with-param>
    <xsl:with-param name="icon-count" select="/envelope/params/bulk_select = 1"/>
  </xsl:call-template>
</xsl:template>

<xsl:template name="html-ovaldef-table">
  <xsl:if test="@status = 400">
    <div>
      <xsl:call-template name="error_window">
        <xsl:with-param name="heading">Warning: SecInfo Database Missing</xsl:with-param>
        <xsl:with-param name="message">
          SCAP and/or CERT database missing on OMP server.
          <a href="/help/ovaldefs.html?token={/envelope/token}#secinfo_missing"
            class="icon icon-sm"
            title="Help: SecInfo database missing">
            <img src="/img/help.svg"/>
          </a>
        </xsl:with-param>
      </xsl:call-template>
    </div>
  </xsl:if>
  <xsl:call-template name="list-window">
    <xsl:with-param name="type" select="'info'"/>
    <xsl:with-param name="subtype" select="'ovaldef'"/>
    <xsl:with-param name="cap-type" select="'OVAL Definition'"/>
    <xsl:with-param name="resources-summary" select="info"/>
    <xsl:with-param name="resources" select="info/ovaldef"/>
    <xsl:with-param name="count" select="count (info/ovaldef)"/>
    <xsl:with-param name="filtered-count" select="info_count/filtered"/>
    <xsl:with-param name="full-count" select="info_count/text ()"/>
    <xsl:with-param name="top-visualization">
      <xsl:call-template name="init-d3charts"/>
      <xsl:call-template name="js-secinfo-top-visualization">
        <xsl:with-param name="type" select="'ovaldef'"/>
      </xsl:call-template>
    </xsl:with-param>
    <xsl:with-param name="columns" xmlns="">
      <column>
        <name><xsl:value-of select="gsa:i18n('Name')"/></name>
        <field>name</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Version')"/></name>
        <field>version</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Status')"/></name>
        <field>status</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Class')"/></name>
        <field>class</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Created', 'Date')"/></name>
        <field>created</field>
        <sort-reverse/>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Modified', 'Date')"/></name>
        <field>modified</field>
        <sort-reverse/>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('CVEs')"/></name>
        <field>cves</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Severity', 'Severity Short')"/></name>
        <field>severity</field>
        <sort-reverse/>
      </column>
    </xsl:with-param>
    <xsl:with-param name="icon-count" select="/envelope/params/bulk_select = 1"/>
  </xsl:call-template>
</xsl:template>

<xsl:template name="html-cert_bund_adv-table">
  <xsl:if test="@status = 400">
    <div>
      <xsl:call-template name="error_window">
        <xsl:with-param name="heading">Warning: SecInfo Database Missing</xsl:with-param>
        <xsl:with-param name="message">
          SCAP and/or CERT database missing on OMP server.
          <a href="/help/cert_bund_advs.html?token={/envelope/token}#secinfo_missing"
            class="icon icon-sm"
            title="Help: SecInfo database missing">
            <img src="/img/help.svg"/>
          </a>
        </xsl:with-param>
      </xsl:call-template>
    </div>
  </xsl:if>
  <xsl:call-template name="list-window">
    <xsl:with-param name="type" select="'info'"/>
    <xsl:with-param name="subtype" select="'cert_bund_adv'"/>
    <xsl:with-param name="cap-type" select="'CERT-Bund Advisory'"/>
    <xsl:with-param name="cap-type-plural" select="'CERT-Bund Advisories'"/>
    <xsl:with-param name="resources-summary" select="info"/>
    <xsl:with-param name="resources" select="info/cert_bund_adv"/>
    <xsl:with-param name="count" select="count (info/cert_bund_adv)"/>
    <xsl:with-param name="filtered-count" select="info_count/filtered"/>
    <xsl:with-param name="full-count" select="info_count/text ()"/>

    <xsl:with-param name="top-visualization">
      <xsl:call-template name="init-d3charts"/>
      <xsl:call-template name="js-secinfo-top-visualization">
        <xsl:with-param name="type" select="'cert_bund_adv'"/>
      </xsl:call-template>
    </xsl:with-param>

    <xsl:with-param name="columns" xmlns="">
      <column>
        <name><xsl:value-of select="gsa:i18n('Name')"/></name>
        <field>name</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Title')"/></name>
        <field>title</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Created', 'Date')"/></name>
        <field>created</field>
        <sort-reverse/>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('CVEs')"/></name>
        <field>cves</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Severity', 'Severity Short')"/></name>
        <field>severity</field>
        <sort-reverse/>
      </column>
    </xsl:with-param>
    <xsl:with-param name="icon-count" select="/envelope/params/bulk_select = 1"/>
  </xsl:call-template>
</xsl:template>

<xsl:template name="html-dfn_cert_adv-table">
  <xsl:if test="@status = 400">
    <div>
      <xsl:call-template name="error_window">
        <xsl:with-param name="heading">Warning: SecInfo Database Missing</xsl:with-param>
        <xsl:with-param name="message">
          SCAP and/or CERT database missing on OMP server.
          <a href="/help/dfn_cert_advs.html?token={/envelope/token}#secinfo_missing"
            class="icon icon-sm"
            title="Help: SecInfo database missing">
            <img src="/img/help.svg"/>
          </a>
        </xsl:with-param>
      </xsl:call-template>
    </div>
  </xsl:if>
  <xsl:call-template name="list-window">
    <xsl:with-param name="type" select="'info'"/>
    <xsl:with-param name="subtype" select="'dfn_cert_adv'"/>
    <xsl:with-param name="cap-type" select="'DFN-CERT Advisory'"/>
    <xsl:with-param name="cap-type-plural" select="'DFN-CERT Advisories'"/>
    <xsl:with-param name="resources-summary" select="info"/>
    <xsl:with-param name="resources" select="info/dfn_cert_adv"/>
    <xsl:with-param name="count" select="count (info/dfn_cert_adv)"/>
    <xsl:with-param name="filtered-count" select="info_count/filtered"/>
    <xsl:with-param name="full-count" select="info_count/text ()"/>

    <xsl:with-param name="top-visualization">
      <xsl:call-template name="init-d3charts"/>
      <xsl:call-template name="js-secinfo-top-visualization">
        <xsl:with-param name="type" select="'dfn_cert_adv'"/>
      </xsl:call-template>
    </xsl:with-param>

    <xsl:with-param name="columns" xmlns="">
      <column>
        <name><xsl:value-of select="gsa:i18n('Name')"/></name>
        <field>name</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Title')"/></name>
        <field>title</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Created', 'Date')"/></name>
        <field>created</field>
        <sort-reverse/>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('CVEs')"/></name>
        <field>cves</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Severity', 'Severity Short')"/></name>
        <field>severity</field>
        <sort-reverse/>
      </column>
    </xsl:with-param>
    <xsl:with-param name="icon-count" select="/envelope/params/bulk_select = 1"/>
  </xsl:call-template>
</xsl:template>

<xsl:template name="html-allinfo-table">
  <xsl:if test="@status = 400">
    <div>
      <xsl:call-template name="error_window">
        <xsl:with-param name="heading">Warning: SecInfo Database Missing</xsl:with-param>
        <xsl:with-param name="message">
          SCAP and/or CERT database missing on OMP server.
          <a href="/help/allinfo.html?token={/envelope/token}#secinfo_missing"
            class="icon icon-sm"
            title="Help: SecInfo database missing">
            <img src="/img/help.svg"/>
          </a>
        </xsl:with-param>
      </xsl:call-template>
    </div>
  </xsl:if>
  <xsl:call-template name="list-window">
    <xsl:with-param name="type" select="'info'"/>
    <xsl:with-param name="subtype" select="'allinfo'"/>
    <xsl:with-param name="cap-type" select="'All SecInfo Information'"/>
    <xsl:with-param name="cap-type-plural" select="'All SecInfo Information'"/>
    <xsl:with-param name="resources-summary" select="info"/>
    <xsl:with-param name="resources" select="info/allinfo"/>
    <xsl:with-param name="count" select="count (info/allinfo)"/>
    <xsl:with-param name="filtered-count" select="info_count/filtered"/>
    <xsl:with-param name="full-count" select="info_count/text ()"/>
    <xsl:with-param name="no_bulk" select="1"/>
    <xsl:with-param name="top-visualization">
      <xsl:call-template name="init-d3charts"/>
      <xsl:call-template name="js-secinfo-top-visualization">
        <xsl:with-param name="type" select="'allinfo'"/>
      </xsl:call-template>
    </xsl:with-param>
    <xsl:with-param name="columns" xmlns="">
      <column>
        <name><xsl:value-of select="gsa:i18n('Name')"/></name>
        <field>name</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Type')"/></name>
        <field>type</field>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Created', 'Date')"/></name>
        <field>created</field>
        <sort-reverse/>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Modified', 'Date')"/></name>
        <field>modified</field>
        <sort-reverse/>
      </column>
      <column>
        <name><xsl:value-of select="gsa:i18n('Severity', 'Severity Short')"/></name>
        <field>severity</field>
        <sort-reverse/>
      </column>
    </xsl:with-param>
    <xsl:with-param name="icon-count" select="/envelope/params/bulk_select = 1"/>
  </xsl:call-template>
</xsl:template>

<xsl:template match="get_info_response">
  <xsl:choose>
    <xsl:when test="(substring(@status, 1, 1) = '4' or substring(@status, 1, 1) = '5') and not (contains (@status_text, 'GET_INFO requires the'))">
      <xsl:call-template name="command_result_dialog">
        <xsl:with-param name="operation">
          Get SecInfo
        </xsl:with-param>
        <xsl:with-param name="status">
          <xsl:value-of select="@status"/>
        </xsl:with-param>
        <xsl:with-param name="msg">
          <xsl:value-of select="@status_text"/>
        </xsl:with-param>
      </xsl:call-template>
    </xsl:when>
  </xsl:choose>
  <xsl:choose>
    <xsl:when test="/envelope/params/info_type = 'CPE' or /envelope/params/info_type = 'cpe'">
      <xsl:choose>
        <xsl:when test="(/envelope/params/info_name and info_count/filtered &lt;= 1)
                        or /envelope/params/info_id">
          <xsl:call-template name="cpe-details"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="html-cpe-table"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>
    <xsl:when test="/envelope/params/info_type = 'CVE' or /envelope/params/info_type = 'cve'">
      <xsl:choose>
        <xsl:when test="(/envelope/params/info_name and info_count/filtered &lt;= 1)
                        or /envelope/params/info_id">
          <xsl:call-template name="cve-details"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="html-cve-table"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>
    <xsl:when test="/envelope/params/info_type = 'NVT' or /envelope/params/info_type = 'nvt'">
      <xsl:choose>
        <xsl:when test="(/envelope/params/info_name and info_count/filtered &lt;= 1)
                        or /envelope/params/info_id">
          <xsl:call-template name="nvt-details">
            <xsl:with-param name="nvts_response" select="info"/>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="html-nvt-table"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>
    <xsl:when test="/envelope/params/info_type = 'OVALDEF' or /envelope/params/info_type = 'ovaldef'">
      <xsl:choose>
        <xsl:when test="(/envelope/params/info_name and info_count/filtered &lt;= 1)
                        or /envelope/params/info_id">
          <xsl:call-template name="ovaldef-details"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="html-ovaldef-table"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>
    <xsl:when test="/envelope/params/info_type = 'CERT_BUND_ADV' or /envelope/params/info_type = 'cert_bund_adv'">
      <xsl:choose>
        <xsl:when test="(/envelope/params/info_name and info_count/filtered &lt;= 1)
                        or /envelope/params/info_id">
          <xsl:call-template name="cert_bund_adv-details"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="html-cert_bund_adv-table"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>
    <xsl:when test="/envelope/params/info_type = 'DFN_CERT_ADV' or /envelope/params/info_type = 'dfn_cert_adv'">
      <xsl:choose>
        <xsl:when test="(/envelope/params/info_name and info_count/filtered &lt;= 1)
                        or /envelope/params/info_id">
          <xsl:call-template name="dfn_cert_adv-details"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="html-dfn_cert_adv-table"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>
    <xsl:when test="/envelope/params/info_type = 'ALLINFO' or /envelope/params/info_type = 'allinfo'">
      <xsl:choose>
        <xsl:when test="(/envelope/params/info_name and info_count/filtered &lt;= 1)
                        or /envelope/params/info_id">
          <xsl:call-template name="allinfo-details"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="html-allinfo-table"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>
    <xsl:otherwise>
      <div class="gb_window">
        <div class="gb_window_part_left"></div>
        <div class="gb_window_part_right"></div>
        <div class="gb_window_part_center">SecInfo</div>
        <div class="gb_window_part_content">
          <xsl:choose>
            <xsl:when test="contains (@status_text, 'SCAP') and @status = '400'">
              <h1>SecInfo Database not available</h1>
              <p>
                Please ensure that your SCAP data is synced by either running openvas-scapdata-sync
                or greenbone-scapdata-sync on your system.
              </p>
            </xsl:when>
            <xsl:when test="contains (@status_text, 'CVE-')">
              <h1>Unknown vulnerability</h1>
              <p>
                <xsl:value-of select="@status_text"/>
              </p>
              <p>
                Please ensure that your SCAP data is up to date and that you entered
                a valid CVE. If the problem persists, the CVE is not available.
                In some cases, CVE references are reserved but did not
                enter the official CVE database yet. Some were reserved and used as
                a reference by vendors, but never entered the CVE database.
              </p>
            </xsl:when>
            <xsl:otherwise>
              <h1>Unknown element</h1>
              <p>
                <xsl:value-of select="@status_text"/>
              </p>
              <p>
                Unknown element type. Ensure that the URL is correct and
                especially that the <code>info_type</code> and
                <code>info_name</code> parameters are consistent.
              </p>
            </xsl:otherwise>
          </xsl:choose>
        </div>
      </div>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="cve-details">
  <div class="toolbar">
    <a href="/help/cve_details.html?token={/envelope/token}"
      class="icon icon-sm"
      title="{concat(gsa:i18n('Help'),': ',gsa:i18n('CVE'),' (',gsa:i18n('CVE Details'),')')}">
      <img src="/img/help.svg"/>
    </a>
    <a href="/omp?cmd=get_info&amp;info_type=cve&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
      class="icon icon-sm"
      title="{gsa:i18n ('CVEs')}">
      <img src="/img/list.svg" alt="{gsa:i18n ('CVEs')}"/>
    </a>
  </div>

  <div class="section-header">
    <div class="section-header-info">
      <table>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('ID')"/>:</td>
          <td>
            <xsl:choose>
              <xsl:when test="info/cve">
                <xsl:value-of select="info/cve/raw_data/cve:entry/@id"/>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="/envelope/params/info_name"/>
              </xsl:otherwise>
            </xsl:choose>
          </td>
        </tr>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Published', 'Date')"/>:</td>
          <td><xsl:value-of select="info/cve/raw_data/cve:entry/vuln:published-datetime"/></td>
        </tr>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Modified', 'Date')"/>:</td>
          <td><xsl:value-of select="info/cve/raw_data/cve:entry/vuln:last-modified-datetime"/></td>
        </tr>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Last updated', 'Date')"/>:</td>
          <td><xsl:value-of select="info/update_time"/></td>
        </tr>
      </table>
    </div>
    <h1>
      <a href="/omp?cmd=get_info&amp;info_type=cve&amp;token={/envelope/token}"
         title="{gsa:i18n ('CVEs')}">
        <img class="icon icon-lg" src="/img/cve.svg" alt="CVEs"/>
      </a>
      <xsl:value-of select="gsa:i18n ('CVE')"/>:
      <xsl:value-of select="info/name"/>
      <xsl:text> </xsl:text>
    </h1>
  </div>

  <div class="section-box">
    <table>
      <tr>
        <td><xsl:value-of select="gsa:i18n ('CWE ID')"/>:</td>
        <td><xsl:value-of select="info/cve/raw_data/cve:entry/vuln:cwe/@id"/></td>
      </tr>
    </table>

    <xsl:choose>
      <xsl:when test="info/cve">
        <h1><xsl:value-of select="gsa:i18n ('Description')"/></h1>
        <xsl:value-of select="info/cve/raw_data/cve:entry/vuln:summary/text()"/>

        <xsl:choose>
          <xsl:when test="info/cve/raw_data/cve:entry/vuln:cvss">
            <h1>CVSS</h1>
            <table>
              <tr>
                <td><xsl:value-of select="gsa:i18n ('Base score')"/></td>
                <td>
                  <div class="pull-left">
                    <xsl:call-template name="severity-bar">
                      <xsl:with-param name="cvss" select="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:score"/>
                    </xsl:call-template>
                  </div>
                  <xsl:variable name="vector">
                    <xsl:text>AV:</xsl:text>
                    <xsl:choose>
                      <xsl:when test="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:access-vector = 'LOCAL'">L</xsl:when>
                      <xsl:when test="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:access-vector = 'NETWORK'">N</xsl:when>
                      <xsl:when test="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:access-vector = 'ADJACENT_NETWORK'">A</xsl:when>
                      <xsl:otherwise>ERROR</xsl:otherwise>
                    </xsl:choose>/AC:<xsl:choose>
                      <xsl:when test="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:access-complexity = 'LOW'">L</xsl:when>
                      <xsl:when test="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:access-complexity = 'MEDIUM'">M</xsl:when>
                      <xsl:when test="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:access-complexity = 'HIGH'">H</xsl:when>
                      <xsl:otherwise>ERROR</xsl:otherwise>
                    </xsl:choose>/Au:<xsl:choose>
                      <xsl:when test="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:authentication = 'NONE'">N</xsl:when>
                      <xsl:when test="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:authentication = 'MULTIPLE_INSTANCES'">M</xsl:when>
                      <xsl:when test="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:authentication = 'SINGLE_INSTANCE'">S</xsl:when>
                      <xsl:otherwise>ERROR</xsl:otherwise>
                    </xsl:choose>/C:<xsl:choose>
                      <xsl:when test="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:confidentiality-impact = 'NONE'">N</xsl:when>
                      <xsl:when test="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:confidentiality-impact = 'PARTIAL'">P</xsl:when>
                      <xsl:when test="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:confidentiality-impact = 'COMPLETE'">C</xsl:when>
                      <xsl:otherwise>ERROR</xsl:otherwise>
                    </xsl:choose>/I:<xsl:choose>
                      <xsl:when test="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:integrity-impact = 'NONE'">N</xsl:when>
                      <xsl:when test="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:integrity-impact = 'PARTIAL'">P</xsl:when>
                      <xsl:when test="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:integrity-impact = 'COMPLETE'">C</xsl:when>
                      <xsl:otherwise>ERROR</xsl:otherwise>
                    </xsl:choose>/A:<xsl:choose>
                      <xsl:when test="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:availability-impact = 'NONE'">N</xsl:when>
                      <xsl:when test="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:availability-impact = 'PARTIAL'">P</xsl:when>
                      <xsl:when test="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:availability-impact = 'COMPLETE'">C</xsl:when>
                      <xsl:otherwise>ERROR</xsl:otherwise>
                    </xsl:choose>
                  </xsl:variable>
                  (<a href="/omp?cmd=cvss_calculator&amp;cvss_vector={$vector}&amp;token={/envelope/token}">
                    <xsl:value-of select="$vector"/>
                  </a>)
                </td>
              </tr>
              <tr>
                <td>Access vector</td>
                <td><xsl:value-of select="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:access-vector"/></td>
              </tr>
              <tr>
                <td>Access Complexity</td>
                <td><xsl:value-of select="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:access-complexity"/></td>
              </tr>
              <tr>
                <td>Authentication</td>
                <td><xsl:value-of select="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:authentication"/></td>
              </tr>
              <tr>
                <td>Confidentiality impact</td>
                <td><xsl:value-of select="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:confidentiality-impact"/></td>
              </tr>
              <tr>
                <td>Integrity impact</td>
                <td><xsl:value-of select="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:integrity-impact"/></td>
              </tr>
              <tr>
                <td>Availability impact</td>
                <td><xsl:value-of select="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:availability-impact"/></td>
              </tr>
              <tr>
                <td>Source</td>
                <td><xsl:value-of select="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:source"/></td>
              </tr>
              <tr>
                <td>Generated</td>
                <td><xsl:value-of select="info/cve/raw_data/cve:entry/vuln:cvss/cvss:base_metrics/cvss:generated-on-datetime"/></td>
              </tr>
            </table>
          </xsl:when>
          <xsl:otherwise>
            <h1>CVSS:
              <div style="display: inline-block; vertical-align: middle;">
                <xsl:call-template name="severity-bar">
                  <xsl:with-param name="extra_text" select="'N/A'"/>
                </xsl:call-template>
              </div>
            </h1>
          </xsl:otherwise>
        </xsl:choose>

        <xsl:choose>
          <xsl:when test="count(info/cve/raw_data/cve:entry/vuln:references) = 0">
            <h1><xsl:value-of select="gsa:i18n ('References')"/>: <xsl:value-of select="gsa:i18n ('None', 'SecInfo|References')"/></h1>
          </xsl:when>
          <xsl:otherwise>
            <h1><xsl:value-of select="gsa:i18n ('References')"/></h1>
            <table>
              <xsl:for-each select="info/cve/raw_data/cve:entry/vuln:references">
                <tr>
                  <td><xsl:value-of select="vuln:source/text()"/></td>
                </tr>
                <tr>
                  <td></td>
                  <td><xsl:value-of select="vuln:reference/text()"/></td>
                </tr>
                <tr>
                  <td></td>
                  <td><a class="external" href="{vuln:reference/@href}"><xsl:value-of select="vuln:reference/@href"/></a></td>
                </tr>
              </xsl:for-each>
            </table>
          </xsl:otherwise>
        </xsl:choose>

        <xsl:choose>
          <xsl:when test="count(info/cve/cert/cert_ref) = 0">
          </xsl:when>
          <xsl:otherwise>
            <h1><xsl:value-of select="gsa:i18n ('CERT Advisories referencing this CVE')"/></h1>
            <table class="gbntable">
              <tr class="gbntablehead2">
                <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
                <td><xsl:value-of select="gsa:i18n ('Title')"/></td>
              </tr>
              <xsl:for-each select="info/cve/cert/cert_ref">
                <tr class="{gsa:table-row-class(position())}">
                  <td>
                    <xsl:choose>
                      <xsl:when test="@type='CERT-Bund'">
                      <a href="?cmd=get_info&amp;info_type=cert_bund_adv&amp;info_name={name}&amp;details=1&amp;token={/envelope/token}" title="{gsa:i18n ('Details')}">
                        <xsl:value-of select="name"/>
                      </a>
                      </xsl:when>
                      <xsl:when test="@type='DFN-CERT'">
                      <a href="?cmd=get_info&amp;info_type=dfn_cert_adv&amp;info_name={name}&amp;details=1&amp;token={/envelope/token}" title="{gsa:i18n ('Details')}">
                        <xsl:value-of select="name"/>
                      </a>
                      </xsl:when>
                      <xsl:otherwise>
                        <xsl:value-of select="name"/>
                        <div class="error"><xsl:value-of select="gsa:i18n ('Unknown CERT type!')"/></div>
                      </xsl:otherwise>
                    </xsl:choose>
                  </td>
                  <td><xsl:value-of select="title"/></td>
                </tr>
              </xsl:for-each>
            </table>
          </xsl:otherwise>
        </xsl:choose>

        <xsl:choose>
          <xsl:when test="count(info/cve/raw_data/cve:entry/vuln:vulnerable-software-list/vuln:product) = 0">
            <h1><xsl:value-of select="gsa:i18n ('Vulnerable products')"/>: <xsl:value-of select="gsa:i18n ('None', 'CVE|Products')"/></h1>
          </xsl:when>
          <xsl:otherwise>
            <h1><xsl:value-of select="gsa:i18n ('Vulnerable products')"/></h1>
            <table class="gbntable">
              <tr class="gbntablehead2">
                <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
              </tr>
              <xsl:for-each select="info/cve/raw_data/cve:entry/vuln:vulnerable-software-list/vuln:product">
                <xsl:sort select="text()"/>

                <tr class="{gsa:table-row-class(position())}">
                  <td>
                    <xsl:call-template name="get_info_cpe_lnk">
                      <xsl:with-param name="cpe" select="str:decode-uri(text())"/>
                    </xsl:call-template>
                  </td>
                </tr>
              </xsl:for-each>
            </table>
          </xsl:otherwise>
        </xsl:choose>

        <xsl:choose>
          <xsl:when test="count(info/cve/nvts/nvt) = 0">
            <h1><xsl:value-of select="gsa:i18n ('NVTs addressing this CVE')"/>: <xsl:value-of select="gsa:i18n ('None', 'NVTs')"/></h1>
          </xsl:when>
          <xsl:otherwise>
            <h1><xsl:value-of select="gsa:i18n ('NVTs addressing this CVE')"/></h1>
            <table class="gbntable">
              <tr class="gbntablehead2">
                <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
              </tr>
              <xsl:for-each select="info/cve/nvts/nvt">
                <tr class="{gsa:table-row-class(position())}">
                  <td>
                    <a href="?cmd=get_info&amp;info_type=nvt&amp;info_id={@oid}&amp;token={/envelope/token}" title="{gsa:i18n ('Details')}">
                      <xsl:value-of select="name"/>
                    </a>
                  </td>
                </tr>
              </xsl:for-each>
            </table>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:when>
      <xsl:otherwise>
        <h1><xsl:value-of select="gsa:i18n ('Description')"/></h1>
        <p>
          <xsl:value-of select="gsa:i18n ('This CVE was not found in the database.  This is not necessarily an error, because the CVE number might have been assigned for the issue, but the CVE not yet published.  Eventually the CVE content will appear in the database.')"/>
        </p>
      </xsl:otherwise>
    </xsl:choose>
  </div>

  <xsl:call-template name="user-tags-window">
    <xsl:with-param name="title" select="gsa-i18n:strformat (gsa:i18n ('User Tags for &quot;%1&quot;'), info/name)"/>
    <xsl:with-param name="user_tags" select="info/user_tags"/>
    <xsl:with-param name="tag_names" select="../get_tags_response"/>
    <xsl:with-param name="resource_type" select="'info'"/>
    <xsl:with-param name="resource_id"   select="info/@id"/>
    <xsl:with-param name="resource_subtype" select="'cve'"/>
  </xsl:call-template>
</xsl:template>

<xsl:template name="cpe-details">
  <div class="toolbar">
    <a href="/help/cpe_details.html?token={/envelope/token}"
      class="icon icon-sm"
      title="{concat(gsa:i18n('Help'),': ',gsa:i18n('CPE'),' (',gsa:i18n('CPE Details'),')')}">
      <img src="/img/help.svg"/>
    </a>
    <a href="/omp?cmd=get_info&amp;info_type=cpe&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
      class="icon icon-sm"
      title="{gsa:i18n ('CPEs')}">
      <img src="/img/list.svg" alt="{gsa:i18n ('CPEs')}"/>
    </a>
  </div>

  <div class="section-header">
    <div class="section-header-info">
      <table>
        <xsl:if test="info/@id != ''">
          <tr>
            <td><xsl:value-of select="gsa:i18n ('ID')"/>:</td>
            <td><xsl:value-of select="info/@id"/></td>
          </tr>
        </xsl:if>
        <xsl:if test="info/modification_time != ''">
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Modified', 'Date')"/>:</td>
            <td><xsl:value-of select="gsa:long-time (info/modification_time)"/></td>
          </tr>
        </xsl:if>
        <xsl:if test="info/creation_time != ''">
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Created', 'Date')"/>:</td>
            <td><xsl:value-of select="gsa:long-time (info/creation_time)"/></td>
          </tr>
        </xsl:if>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Last updated', 'Date')"/>:</td>
          <td><xsl:value-of select="info/update_time"/></td>
        </tr>
      </table>
    </div>
    <h1>
      <a href="/omp?cmd=get_info&amp;info_type=cpe&amp;token={/envelope/token}"
         title="{gsa:i18n ('CPEs')}">
        <img class="icon icon-lg" src="/img/cpe.svg" alt="CPEs"/>
      </a>
      <xsl:value-of select="gsa:i18n ('CPE')"/>:
      <xsl:value-of select="info/name"/>
      <xsl:text> </xsl:text>
    </h1>
  </div>

  <div class="section-box">
    <table>
      <xsl:if test="info/cpe/title">
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Title')"/>:</td>
          <td><xsl:value-of select="info/cpe/title"/></td>
        </tr>
      </xsl:if>
      <xsl:if test="info/@id != ''">
        <tr>
          <td><xsl:value-of select="gsa:i18n ('NVD ID')"/>:</td>
          <td><xsl:value-of select="info/cpe/nvd_id"/></td>
        </tr>
      </xsl:if>
      <xsl:if test="info/cpe/deprecated_by">
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Deprecated by')"/>:</td>
          <td><xsl:value-of select="info/cpe/deprecated_by"/></td>
        </tr>
      </xsl:if>
      <xsl:if test="info/cpe/update_time">
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Last updated', 'Date')"/>:</td>
          <td><xsl:value-of select="info/cpe/update_time"/></td>
        </tr>
      </xsl:if>
      <xsl:if test="info/cpe/status != ''">
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Status')"/>:</td>
          <td><xsl:value-of select="info/cpe/status"/></td>
        </tr>
      </xsl:if>
      <xsl:if test="info/cpe != ''">
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Severity')"/>:</td>
          <td>
            <xsl:choose>
              <xsl:when test="info/cpe/max_cvss &gt;= 0.0">
                <xsl:call-template name="severity-bar">
                  <xsl:with-param name="cvss" select="info/cpe/max_cvss"/>
                </xsl:call-template>
              </xsl:when>
              <xsl:otherwise>
                <xsl:call-template name="severity-bar">
                  <xsl:with-param name="extra_text" select="gsa:i18n ('N/A')"/>
                </xsl:call-template>
              </xsl:otherwise>
            </xsl:choose>
          </td>
        </tr>
      </xsl:if>
    </table>
    <xsl:if test="count(info/cpe/title) = 0">
      <p>
        <xsl:value-of select="gsa:i18n ('This CPE does not appear in the CPE dictionary but is referenced by one or more CVE.')"/>
      </p>
    </xsl:if>
    <xsl:choose>
      <xsl:when test="count(details) = 0 or details = '0' or not(info/cpe)"/>
      <xsl:when test="count(info/cpe/cves/cve) = 0">
        <h1><xsl:value-of select="gsa:i18n ('Reported vulnerabilites')"/>: <xsl:value-of select="gsa:i18n ('None', 'CPE|Vulnerabilities')"/></h1>
      </xsl:when>
      <xsl:otherwise>
        <h1><xsl:value-of select="gsa:i18n ('Reported vulnerabilites')"/></h1>
        <table class="gbntable">
          <tr class="gbntablehead2">
            <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
            <td width="104"><xsl:value-of select="gsa:i18n ('Severity')"/></td>
          </tr>
          <xsl:for-each select="info/cpe/cves/cve">

            <tr class="{gsa:table-row-class(position())}">
              <td>
                <xsl:call-template name="get_info_cve_lnk">
                  <xsl:with-param name="cve" select="cve:entry/@id"/>
                </xsl:call-template>
              </td>
              <td>
                <xsl:call-template name="severity-bar">
                  <xsl:with-param name="cvss" select="cve:entry/vuln:cvss/cvss:base_metrics/cvss:score"/>
                </xsl:call-template>
              </td>
            </tr>
          </xsl:for-each>
        </table>
      </xsl:otherwise>
    </xsl:choose>
  </div>

  <xsl:call-template name="user-tags-window">
    <xsl:with-param name="title" select="gsa-i18n:strformat (gsa:i18n ('User Tags for &quot;%1&quot;'), info/name)"/>
    <xsl:with-param name="user_tags" select="info/user_tags"/>
    <xsl:with-param name="tag_names" select="../get_tags_response"/>
    <xsl:with-param name="resource_type" select="'info'"/>
    <xsl:with-param name="resource_id"   select="info/@id"/>
    <xsl:with-param name="resource_subtype" select="'cpe'"/>
  </xsl:call-template>
</xsl:template>

<xsl:template name="ovaldef-details">
  <div class="toolbar">
    <a href="/help/ovaldef_details.html?token={/envelope/token}"
      class="icon icon-sm"
      title="{concat(gsa:i18n('Help'),': OVALDEF (',gsa:i18n('OVAL Definition Details'),')')}">
      <img src="/img/help.svg"/>
    </a>
    <a href="/omp?cmd=get_info&amp;info_type=ovaldef&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;token={/envelope/token}"
      class="icon icon-sm"
      title="{gsa:i18n ('OVAL Definitions')}">
      <img src="/img/list.svg" alt="{gsa:i18n ('OVAL Definitions')}"/>
    </a>
  </div>

  <div class="section-header">
    <div class="section-header-info">
      <table>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('ID')"/>:</td>
          <td>
            <xsl:value-of select="info/name"/>
          </td>
        </tr>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Created', 'Date')"/>:</td>
          <td><xsl:value-of select="info/creation_time"/></td>
        </tr>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Modified', 'Date')"/>:</td>
          <td><xsl:value-of select="info/modification_time"/></td>
        </tr>
      </table>
    </div>
    <h1>
      <a href="/omp?cmd=get_info&amp;info_type=ovaldef&amp;token={/envelope/token}"
         title="{gsa:i18n ('OVAL Definitions')}">
        <img class="icon icon-lg" src="/img/ovaldef.svg" alt="OVAL Definitions"/>
      </a>
      <xsl:value-of select="gsa:i18n ('OVAL Definition')"/>:
      <xsl:value-of select="info/name"/>
      <xsl:text> </xsl:text>
    </h1>
  </div>

  <div class="section-box">
    <xsl:choose>
      <xsl:when test="info/ovaldef">
        <table>
          <tr>
            <td valign="top"><xsl:value-of select="gsa:i18n ('Title')"/>:</td>
            <td><xsl:value-of select="info/ovaldef/raw_data/oval_definitions:definition/oval_definitions:metadata/oval_definitions:title"/></td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Version')"/>:</td>
            <td><xsl:value-of select="info/ovaldef/raw_data/oval_definitions:definition/@version"/></td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Definition class')"/>:</td>
            <td><xsl:value-of select="info/ovaldef/raw_data/oval_definitions:definition/@class"/></td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Referenced CVEs')"/>:</td>
            <td>
              <xsl:value-of name="cvss" select="info/ovaldef/cve_refs"/>
            </td>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('Severity')"/>:</td>
            <td>
              <xsl:choose>
                <xsl:when test="info/ovaldef/max_cvss &gt;= 0.0">
                  <xsl:call-template name="severity-bar">
                    <xsl:with-param name="cvss" select="info/ovaldef/max_cvss"/>
                  </xsl:call-template>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:call-template name="severity-bar">
                    <xsl:with-param name="extra_text" select="gsa:i18n ('N/A')"/>
                  </xsl:call-template>
                </xsl:otherwise>
              </xsl:choose>
            </td>
          </tr>
          <tr>
            <xsl:choose>
            <xsl:when test="info/ovaldef/raw_data/oval_definitions:definition/@deprecated != ''">
            <td><xsl:value-of select="gsa:i18n ('Deprecated', 'OVAL Definition')"/>:</td>
            <td><xsl:value-of select="info/ovaldef/raw_data/oval_definitions:definition/@deprecated"/></td>
            </xsl:when>
            <xsl:otherwise />
            </xsl:choose>
          </tr>
          <tr>
            <td><xsl:value-of select="gsa:i18n ('File')"/>:</td>
            <td><xsl:value-of select="info/ovaldef/file"/></td>
          </tr>
        </table>

        <xsl:choose>
          <xsl:when test ="count(info/ovaldef/raw_data/oval_definitions:definition/oval_definitions:metadata/oval_definitions:description) > 0">
            <h2><xsl:value-of select="gsa:i18n ('Description')"/></h2>
            <xsl:value-of select="info/ovaldef/raw_data/oval_definitions:definition/oval_definitions:metadata/oval_definitions:description"/>
          </xsl:when>
          <xsl:otherwise>
            <h2><xsl:value-of select="gsa:i18n ('Description')"/>: <xsl:value-of select="gsa:i18n ('None', 'Resource Property|Description')"/></h2>
          </xsl:otherwise>
        </xsl:choose>

        <xsl:choose>
          <xsl:when test="count(info/ovaldef/raw_data/oval_definitions:definition/oval_definitions:metadata/oval_definitions:affected) > 0">
            <h2><xsl:value-of select="gsa:i18n ('Affected', 'OVAL Definition')"/></h2>
            <xsl:for-each select="info/ovaldef/raw_data/oval_definitions:definition/oval_definitions:metadata/oval_definitions:affected">
              <h3>
              <xsl:if test="count(.)>1"><xsl:value-of select="position()"/>) </xsl:if><xsl:value-of select="gsa:i18n ('Family')"/>: <xsl:value-of select="./@family"/>
              </h3>
              <table class="gbntable">
              <tr class="gbntablehead2">
                <td><xsl:value-of select="gsa:i18n ('Type')"/></td>
                <td><xsl:value-of select="gsa:i18n ('Name')"/></td>
              </tr>
              <xsl:for-each select="./*">

              <tr class="{gsa:table-row-class(position())}">
                <td><xsl:value-of select="name()"/></td>
                <td><xsl:value-of select="text()"/></td>
              </tr>
              </xsl:for-each>
              </table>
            </xsl:for-each>
          </xsl:when>
          <xsl:otherwise>
            <h2><xsl:value-of select="gsa:i18n ('Affected', 'OVAL Definition')"/>: <xsl:value-of select="gsa:i18n ('None', 'OVAL Definition|Affected')"/></h2>
          </xsl:otherwise>
        </xsl:choose>

        <xsl:choose>
          <xsl:when test="count(info/ovaldef/raw_data/oval_definitions:definition/oval_definitions:criteria) > 0">
            <h2><xsl:value-of select="gsa:i18n ('Criteria')"/></h2>
            <ul>
            <xsl:apply-templates select="info/ovaldef/raw_data/oval_definitions:definition/oval_definitions:criteria"/>
            </ul>
          </xsl:when>
          <xsl:otherwise>
            <h2><xsl:value-of select="gsa:i18n ('Criteria')"/>: <xsl:value-of select="gsa:i18n ('None', 'OVAL Definition|Criteria')"/></h2>
          </xsl:otherwise>
        </xsl:choose>

        <xsl:choose>
          <xsl:when test="count(info/ovaldef/raw_data/oval_definitions:definition/oval_definitions:metadata/oval_definitions:reference) > 0">
            <h2><xsl:value-of select="gsa:i18n ('References')"/></h2>
            <table class="gbntable">
              <tr class="gbntablehead2">
                <td><xsl:value-of select="gsa:i18n ('Source')"/></td>
                <td><xsl:value-of select="gsa:i18n ('Ref.ID')"/></td>
                <td>URL</td>
              </tr>
              <xsl:for-each select="info/ovaldef/raw_data/oval_definitions:definition/oval_definitions:metadata/oval_definitions:reference">

                <tr class="{gsa:table-row-class(position())}">
                <td><xsl:value-of select="./@source"/></td>
                <td>
                <xsl:choose>
                  <xsl:when test="translate(./@source,'ABCDEFGHIJKLMNOPQRSTUVWXYZ','abcdefghijklmnopqrstuvwxyz') = 'cve'">
                    <xsl:call-template name="get_info_cve_lnk">
                      <xsl:with-param name="cve" select="./@ref_id"/>
                      <xsl:with-param name="gsa_token" select="/envelope/token"/>
                    </xsl:call-template>
                  </xsl:when>
                  <xsl:when test="translate(./@source,'ABCDEFGHIJKLMNOPQRSTUVWXYZ','abcdefghijklmnopqrstuvwxyz') = 'cpe'">
                    <xsl:call-template name="get_info_cpe_lnk">
                      <xsl:with-param name="cpe" select="str:decode-uri(./@ref_id)"/>
                      <xsl:with-param name="gsa_token" select="/envelope/token"/>
                    </xsl:call-template>
                  </xsl:when>
                  <xsl:otherwise>
                    <xsl:value-of select="./@ref_id"/>
                  </xsl:otherwise>
                </xsl:choose></td>
                <td><xsl:value-of select="./@ref_url"/></td>
                </tr>
              </xsl:for-each>
            </table>
          </xsl:when>
          <xsl:otherwise>
            <h2><xsl:value-of select="gsa:i18n ('References')"/>: <xsl:value-of select="gsa:i18n ('None', 'SecInfo|References')"/></h2>
          </xsl:otherwise>
        </xsl:choose>

        <xsl:choose>
          <xsl:when test="count(info/ovaldef/raw_data/oval_definitions:definition/oval_definitions:metadata/oval_definitions:oval_repository) > 0">
          <h2><xsl:value-of select="gsa:i18n ('Repository history')"/></h2>
          <p><b><xsl:value-of select="gsa:i18n('Status')"/>: </b>
            <xsl:value-of select="info/ovaldef/raw_data/oval_definitions:definition/oval_definitions:metadata/oval_definitions:oval_repository/oval_definitions:status"/>
          </p>
          <table class="gbntable">
            <tr class="gbntablehead2">
              <td><xsl:value-of select="gsa:i18n ('Status')"/></td>
              <td><xsl:value-of select="gsa:i18n ('Date')"/></td>
              <td><xsl:value-of select="gsa:i18n ('Contributors')"/></td>
            </tr>
            <xsl:for-each select="info/ovaldef/raw_data/oval_definitions:definition/oval_definitions:metadata/oval_definitions:oval_repository/oval_definitions:dates/*">

            <tr class="{gsa:table-row-class(position())}">
              <td><xsl:value-of select="name()"/>
                <xsl:if test="name() = 'status_change'">
                  <i> (<xsl:value-of select="text()"/>)</i>
                </xsl:if>
              </td>
              <td><xsl:value-of select="./@date"/></td>
              <td>
                <xsl:for-each select="./oval_definitions:contributor">
                  <div>
                    <xsl:value-of select="./text()"/>
                    <i> (<xsl:value-of select="./@organization"/>)</i>
                  </div>
                </xsl:for-each>
              </td>
            </tr>
            </xsl:for-each>
          </table>
          </xsl:when>
          <xsl:otherwise>
            <h1><xsl:value-of select="gsa:i18n ('Repository history')"/>: <xsl:value-of select="gsa:i18n ('None', 'OVAL Definition|History')"/></h1>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:when>
      <xsl:otherwise>
        <h1><xsl:value-of select="gsa:i18n ('OVAL definition not found')"/></h1>
        <xsl:value-of select="gsa:i18n ('No OVAL definition with the requested ID could be found in the SCAP database.')"/>
      </xsl:otherwise>
    </xsl:choose>
  </div>

  <xsl:call-template name="user-tags-window">
    <xsl:with-param name="title" select="gsa-i18n:strformat (gsa:i18n ('User Tags for &quot;%1&quot;'), info/name)"/>
    <xsl:with-param name="user_tags" select="info/user_tags"/>
    <xsl:with-param name="tag_names" select="../get_tags_response"/>
    <xsl:with-param name="resource_type" select="'info'"/>
    <xsl:with-param name="resource_id"   select="info/@id"/>
    <xsl:with-param name="resource_subtype" select="'ovaldef'"/>
  </xsl:call-template>
</xsl:template>

<xsl:template match="oval_definitions:criteria">
  <li>
    <b><xsl:if test="translate(./@negate,'ABCDEFGHIJKLMNOPQRSTUVWXYZ','abcdefghijklmnopqrstuvwxyz') = 'true'">NOT </xsl:if><xsl:value-of select="./@operator"/></b> <xsl:if test="./@comment != ''"><i> (<xsl:value-of select="./@comment"/>)</i></xsl:if><xsl:if test="translate(./@applicability_check,'ABCDEFGHIJKLMNOPQRSTUVWXYZ','abcdefghijklmnopqrstuvwxyz') = 'true'"><b> [Applicability check]</b></xsl:if>
    <ul>
      <xsl:apply-templates select="./oval_definitions:criteria | ./oval_definitions:criterion | ./oval_definitions:extend_definition"/>
    </ul>
  </li>
</xsl:template>

<xsl:template match="oval_definitions:criterion">
  <li>
    <xsl:if test="translate(./@negate,'ABCDEFGHIJKLMNOPQRSTUVWXYZ','abcdefghijklmnopqrstuvwxyz') = 'true'"><b>NOT </b></xsl:if><xsl:value-of select="./@comment"/> <i> (<xsl:value-of select="./@test_ref"/>)</i><xsl:if test="translate(./@applicability_check,'ABCDEFGHIJKLMNOPQRSTUVWXYZ','abcdefghijklmnopqrstuvwxyz') = 'true'"><b> [Applicability check]</b></xsl:if>
  </li>
</xsl:template>

<xsl:template match="oval_definitions:extend_definition">
  <li>
    <xsl:if test="translate(./@negate,'ABCDEFGHIJKLMNOPQRSTUVWXYZ','abcdefghijklmnopqrstuvwxyz') = 'true'"><b>NOT </b></xsl:if><xsl:value-of select="./@comment"/><i> (<a href="/omp?cmd=get_info&amp;info_type=ovaldef&amp;info_name={./@definition_ref}&amp;details=1&amp;token={/envelope/token}"><xsl:value-of select="./@definition_ref"/></a>)</i><xsl:if test="translate(./@applicability_check,'ABCDEFGHIJKLMNOPQRSTUVWXYZ','abcdefghijklmnopqrstuvwxyz') = 'true'"><b> [Applicability check]</b></xsl:if>
  </li>
</xsl:template>


<xsl:template name="cert_bund_adv-details">
  <div class="toolbar">
    <a href="/help/cert_bund_adv_details.html?token={/envelope/token}"
      class="icon icon-sm"
      title="{concat(gsa:i18n('Help'),': DFN_CERT_ADV (',gsa:i18n('CERT-Bund Details'),')')}">
      <img src="/img/help.svg"/>
    </a>
    <a href="/omp?cmd=get_info&amp;info_type=cert_bund_adv&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;token={/envelope/token}"
      class="icon icon-sm"
      title="{gsa:i18n ('CERT-Bund Advisories')}">
      <img src="/img/list.svg" alt="{gsa:i18n ('CERT-Bund Advisories')}"/>
    </a>
  </div>

  <div class="section-header">
    <div class="section-header-info">
      <table>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('ID')"/>:</td>
          <td>
            <xsl:value-of select="info/name"/>
          </td>
        </tr>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Created', 'Date')"/>:</td>
          <td><xsl:value-of select="info/creation_time"/></td>
        </tr>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Modified', 'Date')"/>:</td>
          <td><xsl:value-of select="info/modification_time"/></td>
        </tr>
      </table>
    </div>
    <h1>
      <a href="/omp?cmd=get_info&amp;info_type=cert_bund_adv&amp;token={/envelope/token}"
         title="{gsa:i18n ('CERT-Bund Advisories')}">
        <img class="icon icon-lg" src="/img/cert_bund_adv.svg" alt="CERT-Bund Advisories"/>
      </a>
      <xsl:value-of select="gsa:i18n ('CERT-Bund Advisory')"/>:
      <xsl:value-of select="info/name"/>
      <xsl:text> </xsl:text>
    </h1>
  </div>

  <div class="section-box">
    <xsl:choose>
      <xsl:when test="info/cert_bund_adv">
        <table>
          <xsl:if test="info/cert_bund_adv/raw_data/Advisory/Version != ''">
            <tr>
              <td valign="top"><xsl:value-of select="gsa:i18n ('Version')"/>:</td>
              <td valign="top"><xsl:value-of select="info/cert_bund_adv/raw_data/Advisory/Version"/></td>
            </tr>
          </xsl:if>
          <tr>
            <td valign="top"><xsl:value-of select="gsa:i18n ('Title')"/>:</td>
            <td>
              <xsl:value-of select="info/cert_bund_adv/raw_data/Advisory/Title"/>
            </td>
          </tr>
          <tr>
            <td valign="top"><xsl:value-of select="gsa:i18n ('Software')"/>:</td>
            <td valign="top"><xsl:value-of select="info/cert_bund_adv/raw_data/Advisory/Software"/></td>
          </tr>
          <tr>
            <td valign="top"><xsl:value-of select="gsa:i18n ('Platform')"/>:</td>
            <td valign="top"><xsl:value-of select="info/cert_bund_adv/raw_data/Advisory/Platform"/></td>
          </tr>
          <tr>
            <td valign="top"><xsl:value-of select="gsa:i18n ('Effect')"/>:</td>
            <td valign="top"><xsl:value-of select="gsa:i18n (info/cert_bund_adv/raw_data/Advisory/Effect, 'CERT-Bund Advisory|Effect')"/></td>
          </tr>
          <tr>
            <td valign="top"><xsl:value-of select="gsa:i18n ('Remote Attack')"/>:</td>
            <td valign="top"><xsl:value-of select="gsa:i18n (info/cert_bund_adv/raw_data/Advisory/RemoteAttack, 'CERT-Bund Advisory|Remote Attack')"/></td>
          </tr>

          <tr>
            <td valign="top"><xsl:value-of select="gsa:i18n ('Severity')"/>:</td>
            <td valign="top">
              <xsl:choose>
                <xsl:when test="info/cert_bund_adv/max_cvss &gt;= 0.0">
                  <xsl:call-template name="severity-bar">
                    <xsl:with-param name="cvss" select="info/cert_bund_adv/max_cvss"/>
                  </xsl:call-template>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:call-template name="severity-bar">
                    <xsl:with-param name="extra_text" select="gsa:i18n ('N/A')"/>
                  </xsl:call-template>
                </xsl:otherwise>
              </xsl:choose>
            </td>
          </tr>
          <tr>
            <td valign="top"><xsl:value-of select="gsa:i18n ('CERT-Bund risk rating')"/>:</td>
            <td valign=""><xsl:value-of select="gsa:i18n (info/cert_bund_adv/raw_data/Advisory/Risk, 'CERT-Bund Advisory|Risk')"/></td>
          </tr>

          <xsl:if test="info/cert_bund_adv/raw_data/Advisory/Reference_Source">
            <tr>
              <td valign="top"><xsl:value-of select="gsa:i18n ('Reference')"/>:</td>
              <td valign="top"><xsl:value-of select="info/cert_bund_adv/raw_data/Advisory/Reference_Source"/></td>
            </tr>
          </xsl:if>
          <tr>
            <td valign="top"><xsl:value-of select="gsa:i18n ('Reference URL')"/>:</td>
            <td valign="top">
              <a class="external" href="{info/cert_bund_adv/raw_data/Advisory/Reference_URL}">
                <xsl:value-of select="info/cert_bund_adv/raw_data/Advisory/Reference_URL"/>
              </a>
            </td>
          </tr>
        </table>

        <xsl:choose>
          <xsl:when test="count(info/cert_bund_adv/raw_data/Advisory/CategoryTree) > 0">
            <h2><xsl:value-of select="gsa:i18n ('Categories')"/></h2>
            <ul>
              <xsl:for-each select="info/cert_bund_adv/raw_data/Advisory/CategoryTree">
                <li><xsl:value-of select="text()"/></li>
              </xsl:for-each>
            </ul>
          </xsl:when>
          <xsl:otherwise>
            <h2><xsl:value-of select="gsa:i18n ('Categories')"/>: <xsl:value-of select="gsa:i18n ('None', 'CERT-Bund Advisory|Categories')"/></h2>
          </xsl:otherwise>
        </xsl:choose>

        <xsl:choose>
          <xsl:when test="count(info/cert_bund_adv/raw_data/Advisory/Description/Element/TextBlock) > 0">
            <h2><xsl:value-of select="gsa:i18n ('Description')"/></h2>
            <xsl:for-each select="info/cert_bund_adv/raw_data/Advisory/Description/Element/TextBlock">
              <p><xsl:value-of select="text()"/></p>
            </xsl:for-each>
          </xsl:when>
          <xsl:otherwise>
            <h2><xsl:value-of select="gsa:i18n ('Description')"/>: <xsl:value-of select="gsa:i18n ('None', 'Resource Property|Description')"/></h2>
          </xsl:otherwise>
        </xsl:choose>

        <xsl:choose>
          <xsl:when test="count(info/cert_bund_adv/raw_data/Advisory/CVEList/CVE) > 0">
            <h2><xsl:value-of select="gsa:i18n ('Referenced CVEs')"/></h2>
            <ul>
            <xsl:for-each select="info/cert_bund_adv/raw_data/Advisory/CVEList/CVE">
              <li>
                <xsl:call-template name="get_info_cve_lnk">
                  <xsl:with-param name="cve" select="."/>
                  <xsl:with-param name="gsa_token" select="/envelope/token"/>
                </xsl:call-template>
              </li>
            </xsl:for-each>
            </ul>
          </xsl:when>
          <xsl:otherwise>
          <h2><xsl:value-of select="gsa:i18n ('Referenced CVEs')"/>: <xsl:value-of select="gsa:i18n ('None', 'CVEs')"/></h2>
          </xsl:otherwise>
        </xsl:choose>

        <xsl:choose>
          <xsl:when test="count(info/cert_bund_adv/raw_data/Advisory/Description/Element/Infos/Info) > 0">
            <h2><xsl:value-of select="gsa:i18n ('Other links')"/></h2>
            <ul>
            <xsl:for-each select="info/cert_bund_adv/raw_data/Advisory/Description/Element/Infos/Info">
              <li>
                <p><b><xsl:value-of select="@Info_Issuer"/>:</b><br/>
                  <a class="external" href="{@Info_URL}">
                    <xsl:value-of select="@Info_URL"/>
                  </a>
                </p>
              </li>
            </xsl:for-each>
            </ul>
          </xsl:when>
          <xsl:otherwise>
            <!-- hide because the feed is not expected to contain other links -->
          </xsl:otherwise>
        </xsl:choose>
      </xsl:when>
      <xsl:otherwise>
        <h1><xsl:value-of select="gsa:i18n ('CERT-Bund advisory not found')"/></h1>
        <xsl:value-of select="gsa:i18n ('No CERT-Bund advisory with the requested ID could be found in the CERT database.')"/>
      </xsl:otherwise>
    </xsl:choose>
  </div>

  <xsl:call-template name="user-tags-window">
    <xsl:with-param name="title" select="gsa-i18n:strformat (gsa:i18n ('User Tags for &quot;%1&quot;'), info/name)"/>
    <xsl:with-param name="user_tags" select="info/user_tags"/>
    <xsl:with-param name="tag_names" select="../get_tags_response"/>
    <xsl:with-param name="resource_type" select="'info'"/>
    <xsl:with-param name="resource_id"   select="info/@id"/>
    <xsl:with-param name="resource_subtype" select="'cert_bund_adv'"/>
  </xsl:call-template>
</xsl:template>

<xsl:template name="dfn_cert_adv-details">
  <xsl:variable name="token" select="/envelope/token"/>

  <div class="toolbar">
    <a href="/help/dfn_cert_adv_details.html?token={/envelope/token}"
      class="icon icon-sm"
      title="{concat(gsa:i18n('Help'),': DFN_CERT_ADV (',gsa:i18n('DFN-CERT Advisory Details'),')')}">
      <img src="/img/help.svg"/>
    </a>
    <a href="/omp?cmd=get_info&amp;info_type=dfn_cert_adv&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;token={/envelope/token}"
      class="icon icon-sm"
      title="{gsa:i18n ('DFN-CERT Advisories')}">
      <img src="/img/list.svg" alt="{gsa:i18n ('DFN-CERT Advisories')}"/>
    </a>
  </div>

  <div class="section-header">
    <div class="section-header-info">
      <table>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('ID')"/>:</td>
          <td>
            <xsl:value-of select="info/name"/>
          </td>
        </tr>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Created', 'Date')"/>:</td>
          <td><xsl:value-of select="info/creation_time"/></td>
        </tr>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('Modified', 'Date')"/>:</td>
          <td><xsl:value-of select="info/modification_time"/></td>
        </tr>
      </table>
    </div>
    <h1>
      <a href="/omp?cmd=get_info&amp;info_type=dfn_cert_adv&amp;token={/envelope/token}"
         title="{gsa:i18n ('DFN-CERT Advisories')}">
        <img class="icon icon-lg" src="/img/dfn_cert_adv.svg" alt="DFN-CERT Advisories"/>
      </a>
      <xsl:value-of select="gsa:i18n ('DFN-CERT Advisory')"/>:
      <xsl:value-of select="info/name"/>
      <xsl:text> </xsl:text>
    </h1>
  </div>

  <div class="section-box">
    <xsl:choose>
      <xsl:when test="info/dfn_cert_adv">
        <table>
          <tr>
            <td valign="top"><xsl:value-of select="gsa:i18n ('Title')"/>:</td>
            <td>
              <xsl:value-of select="info/dfn_cert_adv/raw_data/atom:entry/atom:title"/>
            </td>
          </tr>
          <tr>
            <td valign="top"><xsl:value-of select="gsa:i18n ('Advisory link')"/>:</td>
            <td valign="top">
              <a class="external" href="{info/dfn_cert_adv/raw_data/atom:entry/atom:link[@rel='alternate']/@href}">
                <xsl:value-of select="info/dfn_cert_adv/raw_data/atom:entry/atom:link[@rel='alternate']/@href"/>
              </a>
            </td>
          </tr>
          <tr>
            <td valign="top"><xsl:value-of select="gsa:i18n ('Severity')"/>:</td>
            <td valign="top">
              <xsl:choose>
                <xsl:when test="info/dfn_cert_adv/max_cvss &gt;= 0.0">
                  <xsl:call-template name="severity-bar">
                    <xsl:with-param name="cvss" select="info/dfn_cert_adv/max_cvss"/>
                  </xsl:call-template>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:call-template name="severity-bar">
                    <xsl:with-param name="extra_text" select="gsa:i18n ('N/A')"/>
                  </xsl:call-template>
                </xsl:otherwise>
              </xsl:choose>
            </td>
          </tr>
        </table>

        <xsl:choose>
          <xsl:when test="count(info/dfn_cert_adv/raw_data/atom:entry/atom:summary) > 0">
            <h2><xsl:value-of select="gsa:i18n ('Summary')"/></h2>
            <p><xsl:value-of select="info/dfn_cert_adv/raw_data/atom:entry/atom:summary"/></p>
          </xsl:when>
          <xsl:otherwise>
            <h2><xsl:value-of select="gsa:i18n ('Summary')"/>: <xsl:value-of select="gsa:i18n ('None', 'Resource Property|Summary')"/></h2>
          </xsl:otherwise>
        </xsl:choose>

        <xsl:choose>
          <xsl:when test="count(info/dfn_cert_adv/raw_data/atom:entry/atom:link[@rel!='alternate']) > 0">
            <h2><xsl:value-of select="gsa:i18n ('Other links')"/>:</h2>
            <ul>
            <xsl:for-each select="info/dfn_cert_adv/raw_data/atom:entry/atom:link[@rel!='alternate']">
              <li><b><xsl:value-of select="@rel"/>: </b> <xsl:value-of select="@href"/></li>
            </xsl:for-each>
            </ul>
          </xsl:when>
          <xsl:otherwise>
            <!-- hide because the feed is not expected to contain other links -->
          </xsl:otherwise>
        </xsl:choose>

        <xsl:choose>
          <xsl:when test="count(info/dfn_cert_adv/raw_data/atom:entry/dfncert:cve) > 0">
            <h2><xsl:value-of select="gsa:i18n ('Referenced CVEs')"/></h2>
            <ul>
            <xsl:for-each select="info/dfn_cert_adv/raw_data/atom:entry/dfncert:cve">
              <xsl:for-each select="str:tokenize (str:replace (text (), 'CVE ', 'CVE-'), ' ')">
                <xsl:if test="starts-with (text (), 'CVE-') and (string-length (text ()) &gt;= 13) and string (number(substring (text (), 4, 4))) != 'NaN'">
                  <li>
                    <xsl:call-template name="get_info_cve_lnk">
                      <xsl:with-param name="cve" select="."/>
                      <xsl:with-param name="gsa_token" select="$token"/>
                    </xsl:call-template>
                  </li>
                </xsl:if>
              </xsl:for-each>
            </xsl:for-each>
            </ul>
          </xsl:when>
          <xsl:otherwise>
          <h2><xsl:value-of select="gsa:i18n ('Referenced CVEs')"/>: None</h2>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:when>
      <xsl:otherwise>
        <h1><xsl:value-of select="gsa:i18n ('DFN-CERT advisory not found')"/></h1>
        <xsl:value-of select="gsa:i18n ('No DFN-CERT advisory with the requested ID could be found in the CERT database.')"/>
      </xsl:otherwise>
    </xsl:choose>
  </div>

  <xsl:call-template name="user-tags-window">
    <xsl:with-param name="title" select="gsa-i18n:strformat (gsa:i18n ('User Tags for &quot;%1&quot;'), info/name)"/>
    <xsl:with-param name="user_tags" select="info/user_tags"/>
    <xsl:with-param name="tag_names" select="../get_tags_response"/>
    <xsl:with-param name="resource_type" select="'info'"/>
    <xsl:with-param name="resource_id"   select="info/@id"/>
    <xsl:with-param name="resource_subtype" select="'dfn_cert_adv'"/>
  </xsl:call-template>
</xsl:template>


<!-- BEGIN NVT DETAILS -->

<xsl:template match="nvt" mode="details">
  <xsl:param name="config"/>
  <table>
    <tr><td><xsl:value-of select="gsa:i18n ('Config')"/>:</td><td><xsl:value-of select="$config"/></td></tr>
    <tr><td><xsl:value-of select="gsa:i18n ('Family')"/>:</td><td><xsl:value-of select="family"/></td></tr>
    <tr><td><xsl:value-of select="gsa:i18n ('OID')"/>:</td><td><xsl:value-of select="@oid"/></td></tr>
    <tr><td><xsl:value-of select="gsa:i18n ('Version')"/>:</td><td><xsl:value-of select="version"/></td></tr>
    <tr>
      <td><xsl:value-of select="gsa:i18n ('Notes')"/>:</td>
      <td>
        <a href="/omp?cmd=get_notes&amp;filter=nvt_id={@oid} sort=nvt&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
           title="{gsa-i18n:strformat (gsa:i18n ('Notes on NVT %1'), name)}">
          <xsl:value-of select="count (../../../get_notes_response/note)"/>
        </a>
      </td>
    </tr>
    <tr>
      <td><xsl:value-of select="gsa:i18n ('Overrides')"/>:</td>
      <td>
        <a href="/omp?cmd=get_overrides&amp;filter=nvt_id={@oid} sort=nvt&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
           title="{gsa-i18n:strformat (gsa:i18n ('Overrides on NVT %1'), name)}">
          <xsl:value-of select="count (../../../get_overrides_response/override)"/>
        </a>
      </td>
    </tr>
  </table>

  <p>
    <a href="/omp?cmd=get_results&amp;filter={@oid}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}">
      <xsl:value-of select="gsa:i18n ('Show scan results for this NVT')"/>
    </a>
  </p>

  <xsl:choose>
    <xsl:when test="contains(tags, 'summary=')">
      <h2><xsl:value-of select="gsa:i18n ('Summary')"/></h2>
      <xsl:for-each select="str:split (tags, '|')">
        <xsl:if test="'summary' = substring-before (., '=')">
          <xsl:call-template name="structured-text">
            <xsl:with-param name="string" select="substring-after (., '=')"/>
          </xsl:call-template>
        </xsl:if>
      </xsl:for-each>
    </xsl:when>
    <xsl:otherwise>
    </xsl:otherwise>
  </xsl:choose>

  <xsl:choose>
    <xsl:when test="contains(tags, 'affected=')">
      <h2><xsl:value-of select="gsa:i18n ('Affected Software/OS')"/></h2>
      <xsl:for-each select="str:split (tags, '|')">
        <xsl:if test="'affected' = substring-before (., '=')">
          <xsl:call-template name="structured-text">
            <xsl:with-param name="string" select="substring-after (., '=')"/>
          </xsl:call-template>
        </xsl:if>
      </xsl:for-each>
    </xsl:when>
    <xsl:otherwise>
    </xsl:otherwise>
  </xsl:choose>

  <h2><xsl:value-of select="gsa:i18n ('Vulnerability Scoring')"/></h2>
  <table>
    <tr>
      <td><xsl:value-of select="gsa:i18n ('CVSS base')"/>:</td>
      <td>
        <xsl:choose>
          <xsl:when test="cvss_base &gt;= 0.0">
            <xsl:call-template name="severity-bar">
              <xsl:with-param name="cvss" select="cvss_base"/>
            </xsl:call-template>
          </xsl:when>
          <xsl:otherwise>
            <xsl:call-template name="severity-bar">
              <xsl:with-param name="extra_text" select="'N/A'"/>
            </xsl:call-template>
          </xsl:otherwise>
        </xsl:choose>
      </td>
    </tr>
    <xsl:variable name="token" select="/envelope/token"/>
    <xsl:for-each select="str:split (tags, '|')">
      <xsl:if test="'cvss_base_vector' = substring-before (., '=')">
        <xsl:variable name="cvss_vector" select="substring-after (., '=')"/>
        <tr>
          <td><xsl:value-of select="gsa:i18n ('CVSS base vector')"/>:</td>
          <td>
            <a href="/omp?cmd=cvss_calculator&amp;cvss_vector={$cvss_vector}&amp;token={$token}">
              <xsl:value-of select="$cvss_vector"/>
            </a>
          </td>
        </tr>
      </xsl:if>
    </xsl:for-each>
  </table>

  <xsl:choose>
    <xsl:when test="contains(tags, 'insight=')">
      <xsl:if test="not (contains(tags, 'insight=N/A'))">
        <h2><xsl:value-of select="gsa:i18n ('Vulnerability Insight')"/></h2>
        <xsl:for-each select="str:split (tags, '|')">
          <xsl:if test="'insight' = substring-before (., '=')">
            <xsl:call-template name="structured-text">
              <xsl:with-param name="string" select="substring-after (., '=')"/>
            </xsl:call-template>
          </xsl:if>
        </xsl:for-each>
      </xsl:if>
    </xsl:when>
    <xsl:otherwise>
    </xsl:otherwise>
  </xsl:choose>

  <xsl:choose>
    <xsl:when test="contains(tags, 'vuldetect=') or (qod != '')">
      <h2><xsl:value-of select="gsa:i18n ('Vulnerability Detection Method')"/></h2>
      <xsl:for-each select="str:split (tags, '|')">
        <xsl:if test="'vuldetect' = substring-before (., '=')">
          <xsl:call-template name="structured-text">
            <xsl:with-param name="string" select="substring-after (., '=')"/>
          </xsl:call-template>
        </xsl:if>
      </xsl:for-each>
    </xsl:when>
    <xsl:otherwise>
    </xsl:otherwise>
  </xsl:choose>

  <xsl:choose>
    <xsl:when test="qod != ''">
      <p>
      <b><xsl:value-of select="gsa:i18n ('Quality of Detection')"/>: </b>
        <xsl:choose>
          <xsl:when test="qod/type != ''">
            <xsl:value-of select="qod/type"/>
          </xsl:when>
          <xsl:otherwise>
            <i>N/A</i>
          </xsl:otherwise>
        </xsl:choose>
        <xsl:if test="qod/value != ''">
          (<xsl:value-of select="qod/value"/>%)
        </xsl:if>
      </p>
    </xsl:when>
    <xsl:otherwise>
    </xsl:otherwise>
  </xsl:choose>

  <xsl:choose>
    <xsl:when  test="contains(tags, 'impact=')">
      <xsl:if test="not (contains(tags, 'impact=N/A'))">
        <h2><xsl:value-of select="gsa:i18n ('Impact')"/></h2>
        <xsl:for-each select="str:split (tags, '|')">
          <xsl:if test="'impact' = substring-before (., '=')">
            <xsl:call-template name="structured-text">
              <xsl:with-param name="string" select="substring-after (., '=')"/>
            </xsl:call-template>
          </xsl:if>
        </xsl:for-each>
      </xsl:if>
    </xsl:when>
    <xsl:otherwise>
    </xsl:otherwise>
  </xsl:choose>

  <xsl:choose>
    <xsl:when test="contains(tags, 'solution=') or contains(tags, 'solution_type=')">
      <xsl:if test="not (contains(tags, 'solution=N/A')) or contains(tags, 'solution_type=')">
        <h2><xsl:value-of select="gsa:i18n ('Solution')"/></h2>
        <xsl:for-each select="str:split (tags, '|')">
          <xsl:if test="'solution_type' = substring-before (., '=')">
            <p>
              <b><xsl:value-of select="gsa:i18n ('Solution type')"/>: </b>
              <xsl:call-template name="solution-icon">
                <xsl:with-param name="solution_type" select="substring-after (., '=')"/>
              </xsl:call-template>
              <xsl:text> </xsl:text>
              <xsl:value-of select="substring-after (., '=')"/>
            </p>
          </xsl:if>
        </xsl:for-each>
        <xsl:for-each select="str:split (tags, '|')">
          <xsl:if test="'solution' = substring-before (., '=')">
            <xsl:call-template name="structured-text">
              <xsl:with-param name="string" select="substring-after (., '=')"/>
            </xsl:call-template>
          </xsl:if>
        </xsl:for-each>
      </xsl:if>
    </xsl:when>
    <xsl:otherwise>
    </xsl:otherwise>
  </xsl:choose>

  <!-- "NOTAG" means no tags, skip. -->
  <xsl:choose>
    <xsl:when test="tags = 'NOTAG' or (contains(tags,'summary=') + contains(tags,'affected=') + contains(tags,'cvss_base_vector=') + contains(tags,'insight=') + contains(tags,'vuldetect=') + contains(tags,'impact=') + contains(tags,'solution=') + contains(tags,'solution_type=') + contains(tags,'qod_type=')= count(str:split (tags, '|')))">
    </xsl:when>
    <xsl:otherwise>
      <h2><xsl:value-of select="gsa:i18n ('Other tags')"/></h2>
      <table>
      <xsl:for-each select="str:split (tags, '|')">
        <xsl:if test="not(contains('summary|cvss_base_vector|affected|insight|vuldetect|impact|solution|solution_type|qod_type',substring-before (., '=')))">
          <tr>
            <td valign="top"><xsl:value-of select="substring-before (., '=')"/>:</td>
            <td>
              <xsl:call-template name="structured-text">
                <xsl:with-param name="string" select="substring-after (., '=')"/>
              </xsl:call-template>
            </td>
          </tr>
        </xsl:if>
      </xsl:for-each>
      </table>
    </xsl:otherwise>
  </xsl:choose>

  <xsl:variable name="cve_ref">
    <xsl:if test="cve_id != '' and cve_id != 'NOCVE'">
      <xsl:value-of select="cve_id/text()"/>
    </xsl:if>
  </xsl:variable>
  <xsl:variable name="bid_ref">
    <xsl:if test="bugtraq_id != '' and bugtraq_id != 'NOBID'">
      <xsl:value-of select="bugtraq_id/text()"/>
    </xsl:if>
  </xsl:variable>
  <xsl:variable name="cert_ref" select="cert_refs"/>
  <xsl:variable name="xref">
    <xsl:if test="xrefs != '' and xrefs != 'NOXREF'">
      <xsl:value-of select="xrefs/text()"/>
    </xsl:if>
  </xsl:variable>
  <xsl:choose>
    <xsl:when test="$cve_ref != '' or $bid_ref != '' or $xref != '' or count($cert_ref/cert_ref) > 0">
      <h2><xsl:value-of select="gsa:i18n ('References')"/></h2>
      <table>
        <xsl:call-template name="ref_cve_list">
          <xsl:with-param name="cvelist" select="$cve_ref"/>
        </xsl:call-template>
        <xsl:call-template name="ref_bid_list">
          <xsl:with-param name="bidlist" select="$bid_ref"/>
        </xsl:call-template>
        <xsl:call-template name="ref_cert_list">
          <xsl:with-param name="certlist" select="$cert_ref"/>
        </xsl:call-template>
        <xsl:call-template name="ref_xref_list">
          <xsl:with-param name="xreflist" select="$xref"/>
        </xsl:call-template>
      </table>
    </xsl:when>
    <xsl:otherwise>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="get_notes_response">
</xsl:template>

<xsl:template match="get_overrides_response">
</xsl:template>

<xsl:template name="nvt-details">
  <xsl:param name="nvts_response" select="commands_response/get_nvts_response"/>

  <xsl:apply-templates select="gsad_msg"/>
  <xsl:apply-templates select="commands_response/delete_note_response"/>
  <xsl:apply-templates select="commands_response/delete_override_response"/>
  <xsl:apply-templates select="commands_response/modify_note_response"/>
  <xsl:apply-templates select="commands_response/modify_override_response"/>
  <xsl:apply-templates select="delete_tag_response"/>
  <xsl:apply-templates select="create_tag_response"/>
  <xsl:apply-templates select="modify_tag_response"/>

  <xsl:choose>
      <xsl:when test="substring($nvts_response/@status, 1, 1) = '4' or substring($nvts_response/@status, 1, 1) = '5'">
      <xsl:call-template name="command_result_dialog">
        <xsl:with-param name="operation">
          Get NVTs
        </xsl:with-param>
        <xsl:with-param name="status">
          <xsl:value-of select="$nvts_response/@status"/>
        </xsl:with-param>
        <xsl:with-param name="msg">
          <xsl:value-of select="$nvts_response/@status_text"/>
        </xsl:with-param>
      </xsl:call-template>
    </xsl:when>
    <xsl:otherwise>

      <div class="toolbar">
        <a href="/help/nvt_details.html?token={/envelope/token}"
          class="icon icon-sm"
          title="{concat(gsa:i18n('Help'),': ',gsa:i18n('NVT Details'))}">
          <img src="/img/help.svg"/>
        </a>
        <a href="/omp?cmd=get_info&amp;info_type=nvt&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
          class="icon icon-sm"
          title="{gsa:i18n ('NVTs')}">
          <img src="/img/list.svg" alt="{gsa:i18n ('NVTs')}"/>
        </a>
        <div class="small_inline_form" style="display: inline; margin-left: 15px; font-weight: normal;">
          <a href="/omp?cmd=new_note&amp;next=get_info&amp;info_type=nvt&amp;info_id={$nvts_response/nvt/@oid}&amp;oid={$nvts_response/nvt/@oid}&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
             title="{gsa:i18n ('Add Note')}"
             class="new-action-icon icon icon-sm" data-type="note" data-extra="info_type=nvt&amp;info_id={$nvts_response/nvt/@oid}&amp;oid={$nvts_response/nvt/@oid}">
            <img src="/img/new_note.svg" alt="{gsa:i18n ('Add Note')}"/>
          </a>
          <a href="/omp?cmd=new_override&amp;next=get_info&amp;info_type=nvt&amp;info_id={$nvts_response/nvt/@oid}&amp;oid={$nvts_response/nvt/@oid}&amp;filter={str:encode-uri (gsa:envelope-filter (), true ())}&amp;filt_id={/envelope/params/filt_id}&amp;token={/envelope/token}"
             title="{gsa:i18n ('Add Override')}"
             class="new-action-icon icon icon-sm" data-type="override" data-extra="info_type=nvt&amp;info_id={$nvts_response/nvt/@oid}&amp;oid={$nvts_response/nvt/@oid}">
            <img src="/img/new_override.svg" alt="{gsa:i18n ('Add Override')}"/>
          </a>
        </div>
      </div>

      <div class="section-header">
        <div class="section-header-info">
          <table>
            <tr>
              <td><xsl:value-of select="gsa:i18n ('ID')"/>:</td>
              <td>
                <xsl:value-of select="info/nvt/@oid"/>
              </td>
            </tr>
            <xsl:if test="info/nvt/modification_time != ''">
              <tr>
                <td><xsl:value-of select="gsa:i18n ('Modified', 'Date')"/>:</td>
                <td><xsl:value-of select="gsa:long-time (info/nvt/modification_time)"/></td>
              </tr>
            </xsl:if>
            <xsl:if test="info/nvt/creation_time != ''">
              <tr>
                <td><xsl:value-of select="gsa:i18n ('Created', 'Date')"/>:</td>
                <td><xsl:value-of select="gsa:long-time (info/nvt/creation_time)"/></td>
              </tr>
            </xsl:if>
          </table>
        </div>
        <h1>
          <a href="/omp?cmd=get_info&amp;info_type=nvt&amp;token={/envelope/token}"
             title="{gsa:i18n ('NVTs')}">
            <img class="icon icon-lg" src="/img/nvt.svg" alt="NVTs"/>
          </a>
          <xsl:value-of select="gsa:i18n ('NVT')"/>:
          <xsl:value-of select="$nvts_response/nvt/name"/>
          <xsl:text> </xsl:text>
        </h1>
      </div>

      <div class="section-box">
        <xsl:apply-templates
          select="$nvts_response/nvt" mode="details"/>

        <h2><xsl:value-of select="gsa:i18n ('Preferences')"/></h2>
        <xsl:for-each select="$nvts_response/nvt/preferences">
          <xsl:call-template name="preferences-details">
          </xsl:call-template>
        </xsl:for-each>
      </div>

      <xsl:call-template name="user-tags-window">
        <xsl:with-param name="title" select="gsa-i18n:strformat (gsa:i18n ('User Tags for &quot;%1&quot;'), $nvts_response/nvt/name)"/>
        <xsl:with-param name="user_tags" select="$nvts_response/nvt/user_tags"/>
        <xsl:with-param name="tag_names" select="get_tags_response"/>
        <xsl:with-param name="resource_type" select="'info'"/>
        <xsl:with-param name="resource_id"   select="$nvts_response/nvt/@oid"/>
        <xsl:with-param name="resource_subtype" select="'nvt'"/>
      </xsl:call-template>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>


<!-- BEGIN NOTES MANAGEMENT -->

<xsl:template name="html-create-note-form">
  <div class="edit-dialog">
    <div class="title">
      <xsl:value-of select="gsa:i18n ('New Note')"/>
    </div>
    <div class="content">
      <form action="/omp#notes-{result/@id}"
            method="post"
            enctype="multipart/form-data"
            class="form-horizontal">
        <input type="hidden" name="token" value="{/envelope/token}"/>
        <input type="hidden" name="cmd" value="create_note"/>
        <input type="hidden" name="caller" value="{/envelope/current_page}"/>
        <input type="hidden" name="next" value="{/envelope/params/next}"/>
        <input type="hidden" name="filter" value="{gsa:envelope-filter ()}"/>
        <input type="hidden" name="filt_id" value="{/envelope/params/filt_id}"/>

        <xsl:choose>
          <xsl:when test="/envelope/params/next='get_result'">
            <!-- get_result params. -->
            <input type="hidden" name="result_id" value="{result/@id}"/>
            <input type="hidden" name="name" value="{task/name}"/>
            <input type="hidden" name="task_id" value="{task/@id}"/>
            <input type="hidden" name="overrides" value="{overrides}"/>
            <input type="hidden" name="apply_overrides" value="{/envelope/params/apply_overrides}"/>
            <input type="hidden" name="autofp" value="{/envelope/params/autofp}"/>
            <input type="hidden" name="report_result_id" value="{/envelope/params/report_result_id}"/>

            <!-- get_report passthrough params. -->
            <input type="hidden" name="report_id" value="{report/@id}"/>
            <input type="hidden" name="overrides" value="{overrides}"/>
          </xsl:when>
          <xsl:when test="/envelope/params/next='get_info'">
            <!-- get_info (nvt) params. -->
            <input type="hidden" name="info_id" value="{/envelope/params/oid}"/>
            <input type="hidden" name="info_type" value="nvt"/>
          </xsl:when>
          <xsl:otherwise>
            <!-- get_report params. -->
            <input type="hidden" name="report_id" value="{/envelope/params/report_id}"/>
            <input type="hidden" name="overrides" value="{/envelope/params/overrides}"/>
            <input type="hidden" name="details" value="{/envelope/params/details}"/>
          </xsl:otherwise>
        </xsl:choose>

        <xsl:choose>
          <xsl:when test="result/@id">
            <input type="hidden" name="oid" value="{nvt/@id}"/>
            <div class="form-group">
              <label class="col-2 control-label">
                <xsl:value-of select="gsa:i18n ('NVT Name')"/>
              </label>
              <div class="col-10">
                <xsl:variable name="nvt" select="get_results_response/result/nvt"/>
                <xsl:variable name="max" select="70"/>
                <xsl:choose>
                  <xsl:when test="$nvt/@oid = 0">
                    <xsl:value-of select="gsa:i18n ('None.  Result was an open port.', 'NVT')"/>
                  </xsl:when>
                  <xsl:when test="string-length($nvt/name) &gt; $max">
                    <a href="?cmd=get_info&amp;info_type={$nvt/type}&amp;details=1&amp;info_id={$nvt/@oid}&amp;token={/envelope/token}">
                      <abbr title="{$nvt/name} ({$nvt/@oid})"><xsl:value-of select="substring($nvt/name, 0, $max)"/>...</abbr>
                    </a>
                  </xsl:when>
                  <xsl:otherwise>
                    <a href="?cmd=get_info&amp;info_type={$nvt/type}&amp;details=1&amp;info_id={$nvt/@oid}&amp;token={/envelope/token}">
                      <xsl:value-of select="$nvt/name"/>
                    </a>
                  </xsl:otherwise>
                </xsl:choose>
              </div>
            </div>
          </xsl:when>
          <xsl:otherwise>
            <div class="form-group">
              <label class="col-2 control-label">
                <xsl:value-of select="gsa:i18n ('NVT OID')"/>
              </label>
              <div class="col-10">
                <xsl:choose>
                  <xsl:when test="nvt/@id = '0'">
                    <input type="text" name="oid" size="30" maxlength="80" value="1.3.6.1.4.1.25623.1.0."/>
                  </xsl:when>
                  <xsl:otherwise>
                    <input type="text" name="oid" size="30" maxlength="80" value="{nvt/@id}"/>
                  </xsl:otherwise>
                </xsl:choose>
              </div>
            </div>
          </xsl:otherwise>
        </xsl:choose>

        <div class="form-group">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Active', 'Note')"/>
          </label>
          <div class="col-10">
            <div class="radio">
              <label>
                <input type="radio" name="active" value="-1" checked="1"/>
                <xsl:value-of select="concat (gsa:i18n ('yes'), ', ', gsa:i18n ('always'))"/>
              </label>
            </div>
            <div>
              <div class="form-item">
                <div class="radio">
                  <label>
                    <input type="radio" name="active" value="1"/>
                    <xsl:value-of select="concat (gsa:i18n ('yes'), ', ', gsa:i18n ('for the next', 'Time'), ' ')"/>
                  </label>
                </div>
              </div>
              <div class="form-item">
                <input type="text" name="days" size="3" maxlength="7" value="30"/>
              </div>
              <div class="form-item">
                <xsl:value-of select="gsa:i18n ('days')"/>
              </div>
            </div>
            <div class="radio">
              <label>
                <input type="radio" name="active" value="0"/>
                <xsl:value-of select="gsa:i18n ('no')"/>
              </label>
            </div>
          </div>
        </div>
        <div class="form-group">
          <label class="col-2 control-label">
            <xsl:value-of select="gsa:i18n ('Hosts')"/>
          </label>
          <div class="col-10">
            <xsl:choose>
              <xsl:when test="result/@id">
                <div class="form-item">
                  <div class="radio">
                    <label>
                      <input type="radio" name="hosts" value=""/>
                      <xsl:value-of select="gsa:i18n ('Any', 'Hosts')"/>
                    </label>
                  </div>
                </div>
                <div class="form-item">
                  <div class="radio">
                    <label>
                      <input type="radio" name="hosts" value="{hosts}" checked="1"/>
                      <xsl:value-of select="hosts"/>
                    </label>
                  </div>
                </div>
              </xsl:when>
              <xsl:otherwise>
                <div class="form-item">
                  <div class="radio">
                    <label>
                      <input type="radio" name="hosts" value="" checked="1"/>
                      <xsl:value-of select="gsa:i18n ('Any', 'Hosts')"/>
                    </label>
                  </div>
                </div>
                <div class="form-item">
                  <div class="radio">
                    <label>
                      <input type="radio" name="hosts" value="--"