# yesod-test

Pragmatic integration tests for haskell web applications using WAI and optionally a database (Persistent).

It's main goal is to encourage integration and system testing of web applications by making everything /easy to test/. 

Your tests are like browser sessions that keep track of cookies and the last
visited page. You can perform assertions on the content of HTML responses
using css selectors.

You can also easily build requests using forms present in the current page.
This is very useful for testing web applications built in yesod for example,
were your forms may have field names generated by the framework or a randomly
generated "\_token" field.

Your database is also directly available so you can use runDB to set up
backend pre-conditions, or to assert that your session is having the desired effect.

The testing facilities behind the scenes are HSpec (on top of HUnit).

This is the helloworld and kitchen sink. In this case for testing a yesod app.

```haskell

  import Yesod
  import Yesod.Static
  import qualified MySite.Settings as Settings
  import MySite.Models

  main :: IO a
  main = do
    cfg <- (loadConfig Test) >>= either fail return
    st <-  static Settings.staticDir
    Settings.withConnectionPool (connStr cfg) $ \cnPool -> do
      -- ... Perhaps some code here to truncate your test database?
      app <- toWaiApp $ MyApp st cfg 
      runTests app cnPool $ mySuite

  mySuite = do
    describe "Basic navigation and assertions" $ do
      it "Gets a page that has a form, with auto generated fields and token" $ do
        get_ "url/of/page/with/form" -- Load a page
        statusIs 200 -- Assert the status was success

        bodyContains "Hello Person" -- Assert any part of the document contains some text.
        
        -- Perform a css queries and assertions.
        htmlCount "form .main" 1 -- It matches 1 element
        htmlAllContain "h1#mainTitle" "Sign Up Now!" -- All matches have some text

        -- Performs the post using the current page to extract field values:
        post "url/to/post/to" $ do
          addNonce -- Add the _nonce field with the currently shown value

          -- Lookup field by the text on the labels pointing to them.
          byLabel "Email:" "gustavo@cerati.com"
          byLabel "Password:" "secret"
          byLabel "Confirm:" "secret"

      it "Sends another form, this one has a file" $ do
        post "url/to/post/file/to" $ do
          -- You can add files this easy, you still have to provide the mime type manually though.
          addFile "file_field_name" "path/to/local/file" "image/jpeg"
          
          -- And of course you can add any field if you know it's name
          byName "answer" "42"

        statusIs 302

    describe "Db access, still very raw" $ do
      it "selects the list" $ do
        msgs <- testDB $ do (selectList [] [] :: SqlPersist IO [(Key SqlPersist Message, Message)])
        assertEqual "One Message in the DB" 1 (DL.length msgs)

```

