/*
 * ArrowHead.java
 */

package latexDraw.figures;


import java.awt.*;
import java.awt.geom.AffineTransform;
import java.awt.geom.Arc2D;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Line2D;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;

import javax.swing.JLabel;

import latexDraw.psTricks.PSTricksConstants;
import latexDraw.ui.components.LaTeXDrawComboBox;
import latexDraw.ui.components.LabelListCellRenderer;
import latexDraw.util.LaTeXDrawNumber;
import latexDraw.util.LaTeXDrawPoint2D;
import latexDraw.util.LaTeXDrawResources;



/** 
 * This class defines an arrowhead.<br>
 * <br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 01/20/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class ArrowHead implements Serializable, Cloneable
{
	private static final long serialVersionUID = 1L;
	
	public static final double DEFAULT_ARROW_SIZE_DIM = PSTricksConstants.DEFAULT_ARROW_SIZE_DIM*Figure.PPC;
	public static final double DEFAULT_ARROW_SIZE_NUM = PSTricksConstants.DEFAULT_ARROW_SIZE_NUM;
	public static final double DEFAULT_ARROW_LGTH     = PSTricksConstants.DEFAULT_ARROW_LENGTH;
	public static final double DEFAULT_ARROW_INSET    = PSTricksConstants.DEFAULT_ARROW_INSET;
	public static final double DEFAULT_DOT_SIZE_DIM   = PSTricksConstants.DEFAULT_ARROW_DOTSIZE_DIM*Figure.PPC;
	public static final double DEFAULT_DOT_SIZE_NUM   = PSTricksConstants.DEFAULT_ARROW_DOTSIZE_NUM;
	public static final double DEFAULT_TBAR_SIZE_DIM  = PSTricksConstants.DEFAULT_ARROW_TBARSIZE_DIM*Figure.PPC;
	public static final double DEFAULT_TBAR_SIZE_NUM  = PSTricksConstants.DEFAULT_ARROW_TBARSIZE_NUM;
	public static final double DEFAULT_BRACKET_NUM    = PSTricksConstants.DEFAULT_ARROW_BRACKET_LGTH;
	public static final double DEFAULT_RBRACKET_NUM   = PSTricksConstants.DEFAULT_ARROW_RBRACKET_LGTH;
	
	public static final String DEFAULT_STYLE = PSTricksConstants.NONEARROW_STYLE;
	
	private Line line;
	
	private String currentArrowStyle;
	private LaTeXDrawPoint2D position;
	
	private double arrowSizeDim;
	private double arrowSizeNum;
	private double arrowLength;
	private double arrowInset;
	private double dotSizeDim;
	private double dotSizeNum;
	private double tBarSizeDim;
	private double tBarSizeNum;
	private double bracketNum;
	private double rBracketNum;
	
	/** The figure containing the arrow. @since 2.0.0 */
	protected transient Figure figure;
	
	
	
	/**
	 * The constructor taking an figure.
	 * @param f The figure containing the arrow.
	 * @throws IllegalArgumentException If f is null.
	 */
	public ArrowHead(Figure f)
	{
		this(new LaTeXDrawPoint2D(), new Line(false), f);	
	}
	
	
	
	/**
	 * The constructor taking a point.
	 * @param position The position of the arrowhead. 
	 * @param l the line of the arrow.
	 * @param f The figure containing the arrow.
	 * @throws IllegalArgumentException If f is null.
	 */
	public ArrowHead(LaTeXDrawPoint2D position, Line l, Figure f)
	{
		if(f==null)
			throw new IllegalArgumentException();
		
		figure = f;
		line = l;
		this.position = position;
		currentArrowStyle = DEFAULT_STYLE;
		arrowInset   = DEFAULT_ARROW_INSET;
		arrowLength  = DEFAULT_ARROW_LGTH;
		arrowSizeDim = DEFAULT_ARROW_SIZE_DIM;
		arrowSizeNum = DEFAULT_ARROW_SIZE_NUM;
		dotSizeDim   = DEFAULT_DOT_SIZE_DIM;
		dotSizeNum   = DEFAULT_DOT_SIZE_NUM;
		tBarSizeDim  = DEFAULT_TBAR_SIZE_DIM;
		tBarSizeNum  = DEFAULT_TBAR_SIZE_NUM;
		bracketNum   = DEFAULT_BRACKET_NUM;
		rBracketNum  = DEFAULT_RBRACKET_NUM;
	}
	
	
	
	
	
	
	/**
	 * Allows to create a list of the different style of arrowhead (left)
	 * @return The list
	 */
	public static LaTeXDrawComboBox createLeftArrowStyleList()
	{
		LaTeXDrawComboBox lineArrowLChoice = new LaTeXDrawComboBox();
		lineArrowLChoice.setRenderer(new LabelListCellRenderer());
		JLabel label = new JLabel(PSTricksConstants.NONEARROW_STYLE);
		label.setName(PSTricksConstants.NONEARROW_STYLE);
		label.setIcon(LaTeXDrawResources.arrowStyleNoneLIcon);
     	lineArrowLChoice.addItem(label);
     	label = new JLabel(PSTricksConstants.BARIN_STYLE);
     	label.setName(PSTricksConstants.BARIN_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleBarInLIcon);
     	lineArrowLChoice.addItem(label);
		label = new JLabel(PSTricksConstants.BAREND_STYLE);
		label.setName(PSTricksConstants.BAREND_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleBarEndLIcon);
		lineArrowLChoice.addItem(label);
     	label = new JLabel(PSTricksConstants.CIRCLEEND_STYLE);
     	label.setName(PSTricksConstants.CIRCLEEND_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleCircleEndLIcon);
     	lineArrowLChoice.addItem(label);
     	label = new JLabel(PSTricksConstants.CIRCLEIN_STYLE);
     	label.setName(PSTricksConstants.CIRCLEIN_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleCircleInLIcon);
     	lineArrowLChoice.addItem(label);
     	label = new JLabel(PSTricksConstants.DISKEND_STYLE);
     	label.setName(PSTricksConstants.DISKEND_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleDiskEndLIcon);
     	lineArrowLChoice.addItem(label);
     	label = new JLabel(PSTricksConstants.DISKIN_STYLE);
     	label.setName(PSTricksConstants.DISKIN_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleDiskInLIcon);
		lineArrowLChoice.addItem(label);
		label = new JLabel(PSTricksConstants.LARROW_STYLE);
		label.setName(PSTricksConstants.LARROW_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleArrowLIcon);
     	lineArrowLChoice.addItem(label);
		label = new JLabel(PSTricksConstants.RARROW_STYLE);
		label.setName(PSTricksConstants.RARROW_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleRArrowLIcon);
		lineArrowLChoice.addItem(label);
		label = new JLabel(PSTricksConstants.LRBRACKET_STYLE);
		label.setName(PSTricksConstants.LRBRACKET_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleArcLIcon);
		lineArrowLChoice.addItem(label);
		label = new JLabel(PSTricksConstants.RRBRACKET_STYLE);
		label.setName(PSTricksConstants.RRBRACKET_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleArcLRIcon);
		lineArrowLChoice.addItem(label);
		label = new JLabel(PSTricksConstants.LSBRACKET_STYLE);
		label.setName(PSTricksConstants.LSBRACKET_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleBrackLIcon);
		lineArrowLChoice.addItem(label);
		label = new JLabel(PSTricksConstants.RSBRACKET_STYLE);
		label.setName(PSTricksConstants.RSBRACKET_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleBrackLRIcon);
		lineArrowLChoice.addItem(label);
		label = new JLabel(PSTricksConstants.DLARROW_STYLE);
		label.setName(PSTricksConstants.DLARROW_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleDbleArrowLIcon);
		lineArrowLChoice.addItem(label);
		label = new JLabel(PSTricksConstants.DRARROW_STYLE);
		label.setName(PSTricksConstants.DRARROW_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleRDbleArrowLIcon);
		lineArrowLChoice.addItem(label);
     	label = new JLabel(PSTricksConstants.ROUNDIN_STYLE);
     	label.setName(PSTricksConstants.ROUNDIN_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleRoundInLIcon);
     	lineArrowLChoice.addItem(label);
     	lineArrowLChoice.setPreferredSize(new Dimension(75, 30));
     	lineArrowLChoice.setSize(new Dimension(75, 30));
     	lineArrowLChoice.setMaximumSize(new Dimension(75, 30));
     	lineArrowLChoice.setMinimumSize(new Dimension(75, 30));
     	
		return lineArrowLChoice;
	}
	
	
	
	
	/**
	 * Allows to create a list of the different style of arrowhead (right)
	 * @return The list
	 */
	public static LaTeXDrawComboBox createRightArrowStyleList()
	{
		LaTeXDrawComboBox lineArrowRChoice = new LaTeXDrawComboBox();
		lineArrowRChoice.setRenderer(new LabelListCellRenderer());
		JLabel label = new JLabel(PSTricksConstants.NONEARROW_STYLE);
		label.setName(PSTricksConstants.NONEARROW_STYLE);
		label.setIcon(LaTeXDrawResources.arrowStyleNoneRIcon);
		lineArrowRChoice.addItem(label);
     	label = new JLabel(PSTricksConstants.BARIN_STYLE);
     	label.setName(PSTricksConstants.BARIN_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleBarInRIcon);
     	lineArrowRChoice.addItem(label);
		label = new JLabel(PSTricksConstants.BAREND_STYLE);
		label.setName(PSTricksConstants.BAREND_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleBarEndRIcon);
     	lineArrowRChoice.addItem(label);
     	label = new JLabel(PSTricksConstants.CIRCLEEND_STYLE);
     	label.setName(PSTricksConstants.CIRCLEEND_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleCircleEndRIcon);
     	lineArrowRChoice.addItem(label);
     	label = new JLabel(PSTricksConstants.CIRCLEIN_STYLE);
     	label.setName(PSTricksConstants.CIRCLEIN_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleCircleInRIcon);
     	lineArrowRChoice.addItem(label);
     	label = new JLabel(PSTricksConstants.DISKEND_STYLE);
     	label.setName(PSTricksConstants.DISKEND_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleDiskEndRIcon);
     	lineArrowRChoice.addItem(label);
     	label = new JLabel(PSTricksConstants.DISKIN_STYLE);
     	label.setName(PSTricksConstants.DISKIN_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleDiskInRIcon);
     	lineArrowRChoice.addItem(label);
		label = new JLabel(PSTricksConstants.RARROW_STYLE);
		label.setName(PSTricksConstants.RARROW_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleArrowRIcon);
     	lineArrowRChoice.addItem(label);
		label = new JLabel(PSTricksConstants.LARROW_STYLE);
		label.setName(PSTricksConstants.LARROW_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleRArrowRIcon);
     	lineArrowRChoice.addItem(label);
		label = new JLabel(PSTricksConstants.RRBRACKET_STYLE);
		label.setName(PSTricksConstants.RRBRACKET_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleArcRIcon);
     	lineArrowRChoice.addItem(label);
		label = new JLabel(PSTricksConstants.LRBRACKET_STYLE);
		label.setName(PSTricksConstants.LRBRACKET_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleArcRRIcon);
     	lineArrowRChoice.addItem(label);
		label = new JLabel(PSTricksConstants.RSBRACKET_STYLE);
		label.setName(PSTricksConstants.RSBRACKET_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleBrackRIcon);
     	lineArrowRChoice.addItem(label);
		label = new JLabel(PSTricksConstants.LSBRACKET_STYLE);
		label.setName(PSTricksConstants.LSBRACKET_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleBrackRRIcon);
     	lineArrowRChoice.addItem(label);
		label = new JLabel(PSTricksConstants.DRARROW_STYLE);
		label.setName(PSTricksConstants.DRARROW_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleDbleArrowRIcon);
     	lineArrowRChoice.addItem(label);
		label = new JLabel(PSTricksConstants.DLARROW_STYLE);
		label.setName(PSTricksConstants.DLARROW_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleRDbleArrowRIcon);
     	lineArrowRChoice.addItem(label);
     	label = new JLabel(PSTricksConstants.ROUNDIN_STYLE);
     	label.setName(PSTricksConstants.ROUNDIN_STYLE);
     	label.setIcon(LaTeXDrawResources.arrowStyleRoundInRIcon);
     	lineArrowRChoice.addItem(label);
     	lineArrowRChoice.setPreferredSize(new Dimension(75, 30));
     	lineArrowRChoice.setSize(new Dimension(75, 30));
     	lineArrowRChoice.setMaximumSize(new Dimension(75, 30));
     	lineArrowRChoice.setMinimumSize(new Dimension(75, 30));
     	
     	return lineArrowRChoice;
	}
	
	

	
	
	/**
	 * Allows to set the style of the right arrow of the line
	 * @param style The new style of the right arrow of the line
	 */
	public synchronized void setArrowStyle(String style)
	{
		if(PSTricksConstants.isValidArrowStyle(style))
			currentArrowStyle = style;
	}
	
	
	
	
	/**
	 * Allows to get the style of the left arrow of the line
	 * @return The style of the left arrow of the line
	 */
	public synchronized String getArrowStyle()
	{
		return currentArrowStyle;
	}
	
	
	
	/**
	 * Allows to draw the arrowhead
	 * @param g The graphics where the arrowhead is drawn
	 * @param fColor The colour used to fill the circles.
	 * @param asShadow If true, the arrow will be drawn as an arrow of the shadow figure. 
	 */
	public void draw(Graphics2D g, Color fColor, boolean asShadow)
	{
		if(currentArrowStyle.equals(PSTricksConstants.NONEARROW_STYLE) || !isDrawable())
			return;
	
		line.updateAandB();
		
		LaTeXDrawPoint2D pt1 = line.getPt1(), pt2 = line.getPt2();
		Color lineColor = asShadow ? figure.getShadowColor() : figure.getLinesColor();
		Color fillColor = asShadow ? figure.getShadowColor() : fColor;
		float lineWidth = getThickness();
		double x, y, lineAngle, b = line.getB();
		lineAngle = Math.atan(line.getA());
		double xRot,yRot, c2x, c2y, c3x, c3y;
		
		if(Math.abs(lineAngle)==(Math.PI/2.))
		{
			yRot = position.y;
			xRot = position.x;
			double cx = position.x, cy = yRot;
			c2x = Math.cos(lineAngle)*cx - Math.sin(lineAngle)*cy;
			c2y = Math.sin(lineAngle)*cx + Math.cos(lineAngle)*cy;
			c3x = Math.cos(-lineAngle)*(cx-c2x) - Math.sin(-lineAngle)*(cy-c2y);
			c3y = Math.sin(-lineAngle)*(cx-c2x) + Math.cos(-lineAngle)*(cy-c2y);
		}
		else
		{
			xRot = Math.cos(-lineAngle)*position.x-Math.sin(-lineAngle)*(position.y-b); 
			yRot = Math.sin(-lineAngle)*position.x+Math.cos(-lineAngle)*(position.y-b)+b;
			c2x = - Math.sin(lineAngle)*b;
			c2y = Math.cos(lineAngle)*b;
			c3x = Math.cos(-lineAngle)*(-c2x) - Math.sin(-lineAngle)*(b-c2y);
			c3y = Math.sin(-lineAngle)*(-c2x) + Math.cos(-lineAngle)*(b-c2y);
		}
		
		if(lineAngle%(Math.PI*2)!=0)
		{		
			g.rotate(lineAngle);
			g.translate(c3x,c3y);
		}

		if(currentArrowStyle.equals(PSTricksConstants.CIRCLEEND_STYLE) || currentArrowStyle.equals(PSTricksConstants.DISKEND_STYLE))
		{
			double arrowRadius = (dotSizeDim+dotSizeNum*lineWidth)/2.;
			x = xRot - arrowRadius+lineWidth/2.;
			y = yRot - arrowRadius+lineWidth/2.;

			if(currentArrowStyle.equals(PSTricksConstants.DISKEND_STYLE))
				 g.setColor(lineColor);
			else g.setColor(fillColor);
			
			g.fill(new Ellipse2D.Double(x, y, arrowRadius*2-lineWidth, arrowRadius*2-lineWidth));
			g.setColor(lineColor);
			g.draw(new Ellipse2D.Double(x, y, arrowRadius*2-lineWidth, arrowRadius*2-lineWidth));
	
		}
		else
		if(currentArrowStyle.equals(PSTricksConstants.CIRCLEIN_STYLE) || currentArrowStyle.equals(PSTricksConstants.DISKIN_STYLE))
		{
			double arrowRadius = (dotSizeDim+dotSizeNum*lineWidth)/2.;
			
			if(position==pt1)
			{
				if(pt1.x<pt2.x) x = xRot+lineWidth/2.;
				else 			x = xRot-2*arrowRadius+lineWidth/2.;
			}
			else
				if(pt1.x>=pt2.x) x = xRot+lineWidth/2.;
				else x = xRot-2*arrowRadius+lineWidth/2.;
			
			y = yRot - arrowRadius +lineWidth/2.;
			
			if(currentArrowStyle.equals(PSTricksConstants.DISKIN_STYLE))
				 g.setColor(lineColor);
			else g.setColor(fillColor);
			
			g.fill(new Ellipse2D.Double(x, y, arrowRadius*2-lineWidth, arrowRadius*2-lineWidth));
			g.setColor(lineColor);
			g.draw(new Ellipse2D.Double(x, y, arrowRadius*2-lineWidth, arrowRadius*2-lineWidth));
		}
		else
		if(currentArrowStyle.equals(PSTricksConstants.BARIN_STYLE) ||
		   currentArrowStyle.equals(PSTricksConstants.LSBRACKET_STYLE) ||
		   currentArrowStyle.equals(PSTricksConstants.RSBRACKET_STYLE))
		{
			double width = tBarSizeDim + tBarSizeNum*lineWidth;
			double x1, x2, y1, y2;
			
			if((position==pt1 && pt1.x<pt2.x) || (position==pt2 && pt2.x<=pt1.x))
				x1 = x2 = xRot+lineWidth/2.;
			else
				x1 = x2 = xRot-lineWidth/2.;
			
			y1 = yRot-width/2.;
			y2 = yRot+width/2.;

			g.setStroke(new BasicStroke(lineWidth, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
			g.setColor(lineColor);
			g.draw(new Line2D.Double(x1, y1, x2, y2));	
			
			if(currentArrowStyle.equals(PSTricksConstants.LSBRACKET_STYLE) || currentArrowStyle.equals(PSTricksConstants.RSBRACKET_STYLE))
			{
				double lgth = bracketNum*width, x3, x4;
				
				if( (position==pt1 && pt1.x<pt2.x && currentArrowStyle.equals(PSTricksConstants.LSBRACKET_STYLE)) ||
					(position==pt1 && pt1.x>=pt2.x && currentArrowStyle.equals(PSTricksConstants.RSBRACKET_STYLE)) ||
					(position==pt2 && pt2.x<=pt1.x && currentArrowStyle.equals(PSTricksConstants.RSBRACKET_STYLE)) ||
					(position==pt2 && pt2.x>pt1.x && currentArrowStyle.equals(PSTricksConstants.LSBRACKET_STYLE)))
				{
					x3 = x1+lgth;
					x4 = x2+lgth;
				}	
				else
				{
					x3 = x1-lgth;
					x4 = x2-lgth;
				}				
				Stroke s = g.getStroke();
				g.setStroke(new BasicStroke(lineWidth, BasicStroke.CAP_SQUARE, BasicStroke.JOIN_MITER));
				g.draw(new Line2D.Double(x1, y1+lineWidth/2., x3, y1+lineWidth/2.));
				g.draw(new Line2D.Double(x2, y2-lineWidth/2., x4, y2-lineWidth/2.));
				g.setStroke(s);
			}	
		}	
		else
		if(currentArrowStyle.equals(PSTricksConstants.BAREND_STYLE))
		{
			double width = tBarSizeDim + tBarSizeNum*lineWidth;
			x = xRot;
			y = yRot;
			
			g.setColor(lineColor);
			g.draw(new Line2D.Double(x,y-width/2. ,x ,y+width/2.));
		}
		else
		if(currentArrowStyle.equals(PSTricksConstants.RARROW_STYLE) || currentArrowStyle.equals(PSTricksConstants.LARROW_STYLE))
		{
			double width  = arrowSizeNum*lineWidth + arrowSizeDim, length = arrowLength*width, inset  = arrowInset*length;
			int nbPts = 4, xs[] = new int[nbPts], ys[] = new int[nbPts];
				
			if((position==pt1 && pt1.x<pt2.x && currentArrowStyle.equals(PSTricksConstants.LARROW_STYLE)) || 
				(position==pt2 && pt1.x>=pt2.x && currentArrowStyle.equals(PSTricksConstants.RARROW_STYLE)))
			{
				xs[0] = (int)xRot;				ys[0] = (int)yRot;
				xs[1] = (int)(xRot+length);		ys[1] = (int)(yRot-width/2.);
				xs[3] = (int)(xRot+length);		ys[3] = (int)(yRot+width/2.);
				xs[2] = (int)(xRot+length-inset); ys[2] = (int)yRot;
				
				g.setColor(lineColor);
				g.fillPolygon(xs, ys, nbPts);
			}else{
			if((position==pt1 && pt1.x>=pt2.x && currentArrowStyle.equals(PSTricksConstants.LARROW_STYLE)) || 
				(position==pt2 && pt1.x<pt2.x && currentArrowStyle.equals(PSTricksConstants.RARROW_STYLE)))
			{
				xs[0] = (int)xRot;				ys[0] = (int)yRot;
				xs[1] = (int)(xRot-length);		ys[1] = (int)(yRot-width/2.);
				xs[3] = (int)(xRot-length);		ys[3] = (int)(yRot+width/2.);
				xs[2] = (int)(xRot-length+inset); ys[2] = (int)yRot;
				
				g.setColor(lineColor);
				g.fillPolygon(xs, ys, nbPts);
			}else{
			if((position==pt1 && pt1.x<pt2.x && currentArrowStyle.equals(PSTricksConstants.RARROW_STYLE)) || 
				(position==pt2 && pt1.x>=pt2.x && currentArrowStyle.equals(PSTricksConstants.LARROW_STYLE)))
			{
				xs[0] = (int)(xRot+length);		ys[0] = (int)yRot;
				xs[1] = (int)xRot;				ys[1] = (int)(yRot-width/2.);
				xs[3] = (int)xRot;				ys[3] = (int)(yRot+width/2.);
				xs[2] = (int)(xRot+inset); 		ys[2] = (int)yRot;
				
				g.setColor(lineColor);
				g.fillPolygon(xs, ys, nbPts);
			}else{
			if((position==pt1 && pt1.x>=pt2.x && currentArrowStyle.equals(PSTricksConstants.RARROW_STYLE)) || 
				(position==pt2 && pt1.x<pt2.x && currentArrowStyle.equals(PSTricksConstants.LARROW_STYLE)))
			{
				xs[0] = (int)(xRot-length);		ys[0] = (int)yRot;
				xs[1] = (int)xRot;				ys[1] = (int)(yRot-width/2.);
				xs[3] = (int)xRot;				ys[3] = (int)(yRot+width/2.);
				xs[2] = (int)(xRot-inset); 		ys[2] = (int)yRot;

				g.setColor(lineColor);
				g.fillPolygon(xs, ys, nbPts);	
			}}}}}
			else
			if(currentArrowStyle.equals(PSTricksConstants.LRBRACKET_STYLE) || currentArrowStyle.equals(PSTricksConstants.RRBRACKET_STYLE))
			{
				double width = tBarSizeDim + tBarSizeNum*lineWidth;
				double lgth  = rBracketNum*width;
				double xarc  = isInverted() ? xRot : xRot+lineWidth/2.;
				double widtharc = lgth*2 + (isInverted() ? lineWidth/2. : 0.);
				
				Shape s = new Arc2D.Double(xarc, yRot-width/2., widtharc, width, 130, 100, Arc2D.OPEN);
				
				if( (position==pt1 && pt1.x<pt2.x && currentArrowStyle.equals(PSTricksConstants.RRBRACKET_STYLE)) ||
					(position==pt1 && pt1.x>=pt2.x && currentArrowStyle.equals(PSTricksConstants.LRBRACKET_STYLE)) ||
					(position==pt2 && pt2.x<=pt1.x && currentArrowStyle.equals(PSTricksConstants.LRBRACKET_STYLE)) ||
					(position==pt2 && pt2.x>pt1.x && currentArrowStyle.equals(PSTricksConstants.RRBRACKET_STYLE)))
				{
					double cx = xRot, cy = yRot;
					double rotX = Math.cos(Math.PI)*cx - Math.sin(Math.PI)*cy;
					double rotY = Math.sin(Math.PI)*cx + Math.cos(Math.PI)*cy;
					
					AffineTransform at = AffineTransform.getTranslateInstance(cx-rotX, cy-rotY);
					at.rotate(Math.PI);
					s = at.createTransformedShape(s);
				}
				
				g.setColor(lineColor);	
				Stroke stroke = g.getStroke();
				g.setStroke(new BasicStroke(lineWidth, BasicStroke.CAP_SQUARE, BasicStroke.JOIN_MITER));
				g.draw(s);
				g.setStroke(stroke);
			}
			else
			if(currentArrowStyle.equals(PSTricksConstants.DLARROW_STYLE) || currentArrowStyle.equals(PSTricksConstants.DRARROW_STYLE))
			{
				double width  = arrowSizeNum*lineWidth+arrowSizeDim, length = arrowLength*width, inset  = arrowInset*length;
				int nbPts = 4, xs[] = new int[nbPts], ys[] = new int[nbPts];
				int xs2[] = new int[nbPts];		
				
				if((position==pt1 && pt1.x<pt2.x && currentArrowStyle.equals(PSTricksConstants.DLARROW_STYLE)) || 
					(position==pt2 && pt1.x>=pt2.x && currentArrowStyle.equals(PSTricksConstants.DRARROW_STYLE)))
				{
					xs[0] = (int)xRot;				ys[0] = (int)yRot;
					xs[1] = (int)(xRot+length);		ys[1] = (int)(yRot-width/2.);
					xs[3] = (int)(xRot+length);		ys[3] = (int)(yRot+width/2.);
					xs[2] = (int)(xRot+length-inset); ys[2] = (int)yRot;
					xs2[0] = (int)(xRot+length);	
					xs2[1] = (int)(xRot+2*length);		
					xs2[3] = (int)(xRot+2*length);	
					xs2[2] = (int)(xRot+2*length-inset); 
				}else{
				if((position==pt1 && pt1.x>=pt2.x && currentArrowStyle.equals(PSTricksConstants.DLARROW_STYLE)) || 
					(position==pt2 && pt1.x<pt2.x && currentArrowStyle.equals(PSTricksConstants.DRARROW_STYLE)))
				{
					xs[0] = (int)xRot;				ys[0] = (int)yRot;
					xs[1] = (int)(xRot-length);		ys[1] = (int)(yRot-width/2.);
					xs[3] = (int)(xRot-length);		ys[3] = (int)(yRot+width/2.);
					xs[2] = (int)(xRot-length+inset); ys[2] = (int)yRot;
					
					xs2[0] = (int)(xRot-length);	
					xs2[1] = (int)(xRot-2*length);	
					xs2[3] = (int)(xRot-2*length);	
					xs2[2] = (int)(xRot-2*length+inset); 	

				}else{
				if((position==pt1 && pt1.x<pt2.x && currentArrowStyle.equals(PSTricksConstants.DRARROW_STYLE)) || 
					(position==pt2 && pt1.x>=pt2.x && currentArrowStyle.equals(PSTricksConstants.DLARROW_STYLE)))
				{
					xs[0] = (int)(xRot+length);		ys[0] = (int)yRot;
					xs[1] = (int)xRot;				ys[1] = (int)(yRot-width/2.);
					xs[3] = (int)xRot;				ys[3] = (int)(yRot+width/2.);
					xs[2] = (int)(xRot+inset); 		ys[2] = (int)yRot;
					
					xs2[0] = (int)(xRot+2*length);	
					xs2[1] = (int)(xRot+length);	
					xs2[3] = (int)(xRot+length);		
					xs2[2] = (int)(xRot+length+inset);	

				}else{
				if((position==pt1 && pt1.x>=pt2.x && currentArrowStyle.equals(PSTricksConstants.DRARROW_STYLE)) || 
					(position==pt2 && pt1.x<pt2.x && currentArrowStyle.equals(PSTricksConstants.DLARROW_STYLE)))
				{
					xs[0] = (int)(xRot-length);		ys[0] = (int)yRot;
					xs[1] = (int)xRot;				ys[1] = (int)(yRot-width/2.);
					xs[3] = (int)xRot;				ys[3] = (int)(yRot+width/2.);
					xs[2] = (int)(xRot-inset); 		ys[2] = (int)yRot;
					
					xs2[0] = (int)(xRot-2*length);
					xs2[1] = (int)(xRot-length);	
					xs2[3] = (int)(xRot-length);
					xs2[2] = (int)(xRot-length-inset); 
			}}}}
			
				g.setColor(lineColor);
				g.draw(new Line2D.Double(xs[2], ys[0], xs2[2], ys[0]));
				g.fillPolygon(xs, ys, nbPts);	
				g.fillPolygon(xs2, ys, nbPts);
				
			}else
			if(currentArrowStyle.equals(PSTricksConstants.SQUAREEND_STYLE) || currentArrowStyle.equals(PSTricksConstants.ROUNDEND_STYLE))
			{
				Stroke s = g.getStroke();
				g.setColor(lineColor);
				
				if(currentArrowStyle.equals(PSTricksConstants.ROUNDEND_STYLE))
					g.setStroke(new BasicStroke(lineWidth, BasicStroke.CAP_ROUND, BasicStroke.JOIN_MITER));
				else
					g.setStroke(new BasicStroke(lineWidth, BasicStroke.CAP_SQUARE, BasicStroke.JOIN_MITER));
				
				if((position==pt1 && pt1.x<pt2.x) || (position==pt2 && pt2.x<=pt2.x))
					g.draw(new Line2D.Double(xRot+1, yRot, xRot, yRot));
				else
					g.draw(new Line2D.Double(xRot-1, yRot, xRot, yRot));

				g.setStroke(s);
			}else
			if(currentArrowStyle.equals(PSTricksConstants.ROUNDIN_STYLE))
			{
				Stroke s = g.getStroke();
				g.setColor(lineColor);
				g.setStroke(new BasicStroke(lineWidth, BasicStroke.CAP_ROUND, BasicStroke.JOIN_MITER));
				
				if((position==pt1 && pt1.x<pt2.x) || (position==pt2 && pt2.x<=pt1.x))
					g.draw(new Line2D.Double(xRot+lineWidth/2., yRot, xRot+lineWidth/2., yRot));
				else
					g.draw(new Line2D.Double(xRot-lineWidth/2., yRot, xRot-lineWidth/2., yRot));
				
				g.setStroke(s);
			}
		
		if(lineAngle%(Math.PI*2)!=0)
		{
			g.translate(-c3x,-c3y);
			g.rotate(-lineAngle);
		}
		
	}
	
	
	
	
	
	public String getParametersCode()
	{
		double threshold = 0.001;
		
		if(currentArrowStyle.equals(PSTricksConstants.BARIN_STYLE) ||
		   currentArrowStyle.equals(PSTricksConstants.BAREND_STYLE) ||
		   currentArrowStyle.equals(PSTricksConstants.LSBRACKET_STYLE) ||
		   currentArrowStyle.equals(PSTricksConstants.RSBRACKET_STYLE) ||
		   currentArrowStyle.equals(PSTricksConstants.LRBRACKET_STYLE) ||
		   currentArrowStyle.equals(PSTricksConstants.RRBRACKET_STYLE))
		{
			String code = "tbarsize="+LaTeXDrawNumber.getCutNumber((float)(getTBarSizeDim()/Figure.PPC), threshold)+"cm " +(float)getTBarSizeNum(); //$NON-NLS-1$ //$NON-NLS-2$
			if(currentArrowStyle.equals(PSTricksConstants.LSBRACKET_STYLE) ||
			   currentArrowStyle.equals(PSTricksConstants.RSBRACKET_STYLE))
				code+=",bracketlength="+(float)getBracketNum(); //$NON-NLS-1$
			else
				if(currentArrowStyle.equals(PSTricksConstants.RRBRACKET_STYLE) ||
				   currentArrowStyle.equals(PSTricksConstants.LRBRACKET_STYLE))
					code+=",rbracketlength="+(float)getRBracketNum(); //$NON-NLS-1$
			
			return code;
		}
		
		if(currentArrowStyle.equals(PSTricksConstants.LARROW_STYLE) ||
				currentArrowStyle.equals(PSTricksConstants.RARROW_STYLE) ||
				currentArrowStyle.equals(PSTricksConstants.DLARROW_STYLE) ||
				currentArrowStyle.equals(PSTricksConstants.DRARROW_STYLE))
			return "arrowsize="+LaTeXDrawNumber.getCutNumber((float)(getArrowSizeDim()/Figure.PPC), threshold) +"cm " +//$NON-NLS-1$ //$NON-NLS-2$
				(float)getArrowSizeNum()+",arrowlength="+   //$NON-NLS-1$
				(float)getArrowLength()+",arrowinset="+(float)getArrowInset(); //$NON-NLS-1$
		
		if(currentArrowStyle.equals(PSTricksConstants.CIRCLEIN_STYLE) || 
			currentArrowStyle.equals(PSTricksConstants.DISKIN_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.CIRCLEEND_STYLE) || 
			currentArrowStyle.equals(PSTricksConstants.DISKEND_STYLE))
			return "dotsize="+LaTeXDrawNumber.getCutNumber((float)(getDotSizeDim()/Figure.PPC), threshold)+"cm "+(float)getDotSizeNum(); //$NON-NLS-1$ //$NON-NLS-2$
		
		return "";//$NON-NLS-1$
	}
	
	
	
	/**
	 * Allows to get the length of the arrow style
	 * @return The length of the arrow style
	 */
	public double getArrowHeadLength()
	{
		if(currentArrowStyle.equals(PSTricksConstants.LARROW_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.RARROW_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.DLARROW_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.DRARROW_STYLE))
			return (arrowLength*getArrowSizeNum()*getThickness()+getArrowSizeDim())/2.;
		
		if(currentArrowStyle.equals(PSTricksConstants.LRBRACKET_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.RRBRACKET_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.BARIN_STYLE) ||
		    currentArrowStyle.equals(PSTricksConstants.LSBRACKET_STYLE) ||
		    currentArrowStyle.equals(PSTricksConstants.RSBRACKET_STYLE))
			return 0;
		
		if(currentArrowStyle.equals(PSTricksConstants.ROUNDIN_STYLE))
			return getThickness()/2.;
		
		if(currentArrowStyle.equals(PSTricksConstants.CIRCLEIN_STYLE) || 
			currentArrowStyle.equals(PSTricksConstants.DISKIN_STYLE))
			return (getDotSizeDim()+getDotSizeNum()*getThickness())/2.;
		
		if(currentArrowStyle.equals(PSTricksConstants.CIRCLEEND_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.DISKEND_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.BAREND_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.SQUAREEND_STYLE))
			return 0.;
		
		return 0;
		
	}
	
	
	
	
	/**
	 * Allows to get a clone of the arrow : beware : the pointer
	 * on the line is the same like the pointer on the position
	 * @return A clone of the arrowhead
	 */
	public ArrowHead clone(boolean cloneLine) throws CloneNotSupportedException
	{
		ArrowHead a;
		
		if(cloneLine)
		{
			Line l = (Line)line.clone();
			
			if(line.getPt1().equals(position))
				 a = new ArrowHead(l.getPt1(), l, figure);
			else a = new ArrowHead(l.getPt2(), l, figure);
		}
		else
			a = new ArrowHead(position, line, figure);
			
		a.copyArrowParameters(this);
		return a;
	}
	
	
	
	
	
	/**
	 * Allows to get a clone of the arrow : beware : the pointer
	 * on the line is the same like the pointer on the position
	 * @return A clone of the arrowhead
	 */
	@Override
	public Object clone() throws CloneNotSupportedException
	{
		return clone(true);
	}



	
	/**
	 * Allows to get the position of the arrowhead
	 * @return The position of the arrowhead
	 */
	public synchronized LaTeXDrawPoint2D getPosition() 
	{
		return position;
	}



	/**
	 * Allows to know if the arrowhead is of the same type as the given arrowhead
	 * (for example : square bracket is of the same type of round bracket and
	 * bar).
	 * @param ah The arrow to compare
	 * @return True if the arrowhead is of the same type as the given arrowhead
	 */
	public boolean isOfTheSameTypeAs(ArrowHead ah)
	{
		if(currentArrowStyle.equals(ah.currentArrowStyle))
			return true;
		
		if( (currentArrowStyle.equals(PSTricksConstants.BARIN_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.BAREND_STYLE)) &&
			(ah.currentArrowStyle.equals(PSTricksConstants.BARIN_STYLE) ||
			ah.currentArrowStyle.equals(PSTricksConstants.BAREND_STYLE)))
				return true;
		
		if((currentArrowStyle.equals(PSTricksConstants.LRBRACKET_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.RRBRACKET_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.RSBRACKET_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.LSBRACKET_STYLE)) &&
			(ah.currentArrowStyle.equals(PSTricksConstants.RRBRACKET_STYLE) ||
			ah.currentArrowStyle.equals(PSTricksConstants.LRBRACKET_STYLE) ||
			ah.currentArrowStyle.equals(PSTricksConstants.RSBRACKET_STYLE) ||
			ah.currentArrowStyle.equals(PSTricksConstants.LSBRACKET_STYLE)))
			return true;
		
		if((currentArrowStyle.equals(PSTricksConstants.CIRCLEEND_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.CIRCLEIN_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.DISKEND_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.DISKIN_STYLE)) &&
			(ah.currentArrowStyle.equals(PSTricksConstants.CIRCLEEND_STYLE) ||
			ah.currentArrowStyle.equals(PSTricksConstants.CIRCLEIN_STYLE) ||
			ah.currentArrowStyle.equals(PSTricksConstants.DISKEND_STYLE) ||
			ah.currentArrowStyle.equals(PSTricksConstants.DISKIN_STYLE)))
				return true;
		
		if((currentArrowStyle.equals(PSTricksConstants.DLARROW_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.DRARROW_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.LARROW_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.RARROW_STYLE)) &&
			(ah.currentArrowStyle.equals(PSTricksConstants.DLARROW_STYLE) ||
			ah.currentArrowStyle.equals(PSTricksConstants.DRARROW_STYLE) ||
			ah.currentArrowStyle.equals(PSTricksConstants.LARROW_STYLE) ||
			ah.currentArrowStyle.equals(PSTricksConstants.RARROW_STYLE)))
				return true;
		
		if((currentArrowStyle.equals(PSTricksConstants.ROUNDEND_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.ROUNDIN_STYLE)) &&
			(ah.currentArrowStyle.equals(PSTricksConstants.ROUNDEND_STYLE) ||
			ah.currentArrowStyle.equals(PSTricksConstants.ROUNDIN_STYLE)))
				return true;
		
		return false;
	}
	
	
	
	
	/**
	 * Allows to set the position of the arrowhead
	 * @param pt The new position of the arrowhead
	 */
	public synchronized void setPosition(LaTeXDrawPoint2D pt) 
	{
		if(pt!=null)
			position = pt;	
	}
	
	

	private void readObject(ObjectInputStream ois) throws IOException, 
					ClassNotFoundException
	{
		line = (Line) ois.readObject();
		
		currentArrowStyle = (String) ois.readObject();
		position = (LaTeXDrawPoint2D) ois.readObject();

		arrowSizeDim = ois.readDouble();
		arrowSizeNum = ois.readDouble();
		arrowLength = ois.readDouble();
		arrowInset = ois.readDouble();
		dotSizeDim = ois.readDouble();
		dotSizeNum = ois.readDouble();
		tBarSizeDim = ois.readDouble();
		tBarSizeNum = ois.readDouble();
		bracketNum = ois.readDouble();
		rBracketNum = ois.readDouble();
	}



	/**
	 * @return Returns the arrowInset.
	 */
	public synchronized double getArrowInset()
	{
		return arrowInset;
	}



	/**
	 * @param arrowInset The arrowInset to set.
	 */
	public synchronized void setArrowInset(double arrowInset)
	{
		if(arrowInset>=0)
			this.arrowInset = arrowInset;
	}



	/**
	 * @return Returns the arrowSizeDim.
	 */
	public synchronized double getArrowSizeDim()
	{
		return arrowSizeDim;
	}



	/**
	 * @param arrowSizeDim The arrowSizeDim to set.
	 */
	public synchronized void setArrowSizeDim(double arrowSizeDim)
	{
		if(arrowSizeDim>=0)
			this.arrowSizeDim = arrowSizeDim;
	}



	/**
	 * @return Returns the arrowSizeNum.
	 */
	public synchronized double getArrowSizeNum()
	{
		return arrowSizeNum;
	}



	/**
	 * @param arrowSizeNum The arrowSizeNum to set.
	 */
	public synchronized void setArrowSizeNum(double arrowSizeNum)
	{
		if(arrowSizeNum>=0.001)
			this.arrowSizeNum = arrowSizeNum;
	}



	/**
	 * @return Returns the bracketNumRight.
	 */
	public synchronized double getBracketNum()
	{
		return bracketNum;
	}



	/**
	 * @param bracketNum The bracketNumRight to set.
	 */
	public synchronized void setBracketNum(double bracketNum)
	{
		if(bracketNum>0.001)
			this.bracketNum = bracketNum;
	}



	/**
	 * @return Returns the dotSizeDim.
	 */
	public synchronized double getDotSizeDim()
	{
		return dotSizeDim;
	}



	/**
	 * @param dotSizeDim The dotSizeDim to set.
	 */
	public synchronized void setDotSizeDim(double dotSizeDim)
	{
		if(dotSizeDim>=0)
			this.dotSizeDim = dotSizeDim;
	}



	/**
	 * @return Returns the dotSizeNum.
	 */
	public synchronized double getDotSizeNum()
	{
		return dotSizeNum;
	}



	/**
	 * @param dotSizeNum The dotSizeNum to set.
	 */
	public synchronized void setDotSizeNum(double dotSizeNum)
	{
		if(dotSizeNum>=0.1)
			this.dotSizeNum = dotSizeNum;
	}



	/**
	 * @return Returns the rBracketNum.
	 */
	public synchronized double getRBracketNum()
	{
		return rBracketNum;
	}



	/**
	 * @param bracketNum The rBracketNum to set.
	 */
	public synchronized void setRBracketNum(double bracketNum)
	{
		if(bracketNum>=0.001)
			rBracketNum = bracketNum;
	}



	/**
	 * @return Returns the tBarSizeDim.
	 */
	public synchronized double getTBarSizeDim()
	{
		return tBarSizeDim;
	}



	/**
	 * @param barSizeDim The tBarSizeDim to set.
	 */
	public synchronized void setTBarSizeDim(double barSizeDim)
	{
		if(barSizeDim>=0)
			tBarSizeDim = barSizeDim;
	}



	/**
	 * @return Returns the tBarSizeNum.
	 */
	public synchronized double getTBarSizeNum()
	{
		return tBarSizeNum;
	}



	/**
	 * @param barSizeNum The tBarSizeNum to set.
	 */
	public synchronized void setTBarSizeNum(double barSizeNum)
	{
		if(tBarSizeDim>=0.1)
			tBarSizeNum = barSizeNum;
	}



	/**
	 * @param arrowLength The arrowLength to set.
	 */
	public void setArrowLength(double arrowLength)
	{
		if(arrowLength>=0)
			this.arrowLength = arrowLength;
	}



	/**
	 * @return Returns the line.
	 */
	public synchronized Line getLine()
	{
		return line;
	}



	/**
	 * @param line The line to set.
	 */
	public synchronized void setLine(Line line)
	{
		if(line!=null)
			this.line = line;
	}



	/**
	 * @return the arrowLength
	 */
	public double getArrowLength()
	{
		return arrowLength;
	}

	
	
	
	/**
	 * @return True if the current style of the arrow is an arrow, a double arrow, ...
	 */
	public boolean needReduceLine()
	{
		return 	currentArrowStyle.equals(PSTricksConstants.RARROW_STYLE) ||
				currentArrowStyle.equals(PSTricksConstants.LARROW_STYLE) ||
				currentArrowStyle.equals(PSTricksConstants.DLARROW_STYLE) ||
				currentArrowStyle.equals(PSTricksConstants.DRARROW_STYLE) ||
				currentArrowStyle.equals(PSTricksConstants.RRBRACKET_STYLE) ||
				currentArrowStyle.equals(PSTricksConstants.LRBRACKET_STYLE) ||
				currentArrowStyle.equals(PSTricksConstants.RSBRACKET_STYLE) ||
				currentArrowStyle.equals(PSTricksConstants.LSBRACKET_STYLE) ||
				currentArrowStyle.equals(PSTricksConstants.ROUNDIN_STYLE);
	}



	@Override
	public boolean equals(Object o)
	{
		if(o instanceof ArrowHead)
		{
			ArrowHead ah2 = (ArrowHead)o;
			
			return ((float)ah2.getArrowInset())==((float)getArrowInset()) && 
				((float)ah2.getArrowLength())==((float)getArrowLength()) &&
				((float)ah2.getArrowSizeDim())==((float)getArrowSizeDim()) && 
				((float)ah2.getArrowSizeNum())==((float)getArrowSizeNum()) &&
				ah2.getArrowStyle().equals(getArrowStyle()) && 
				((float)ah2.getBracketNum())==((float)getBracketNum()) &&
				((float)ah2.getDotSizeDim())==((float)getDotSizeDim()) && 
				((float)ah2.getDotSizeNum())==((float)getDotSizeNum()) &&
				((float)ah2.getRBracketNum())==((float)getRBracketNum()) && 
				((float)ah2.getTBarSizeDim())==((float)getTBarSizeDim()) &&
				((float)ah2.getTBarSizeNum())==((float)getTBarSizeNum());
		}
		return false;
	}

	
	
	/**
	 * @return True if the arrow has no style.
	 * @since 1.9
	 */
	public boolean isWithoutStyle()
	{
		return currentArrowStyle.equals(PSTricksConstants.NONEARROW_STYLE);
	}
	
	
	
	@Override
	public int hashCode()
	{
		return (int)(currentArrowStyle.hashCode()+getArrowInset()+getArrowLength()+
				getRBracketNum()+getDotSizeDim()+getDotSizeDim());
	}
	
	
	/**
	 * @return The parameters of the dot shape (even if the arrow has not currently the dot shape).
	 * @since 1.9
	 */
	public String getDotParameters()
	{
		return "dotsize="+(float)(getDotSizeDim()/Figure.PPC)+"cm "+(float)getDotSizeNum(); //$NON-NLS-1$ //$NON-NLS-2$
	}
	
	
	
	/**
	 * @return True if the current shape of the arrow is a dot.
	 * @since 1.9
	 */
	public boolean isArrowShapeDot()
	{
		return currentArrowStyle.equals(PSTricksConstants.CIRCLEEND_STYLE) ||
				currentArrowStyle.equals(PSTricksConstants.CIRCLEIN_STYLE) ||
				currentArrowStyle.equals(PSTricksConstants.DISKEND_STYLE) ||
				currentArrowStyle.equals(PSTricksConstants.DISKIN_STYLE);
	}
	
	
	
	
	/**
	 * Copy the parameters of the arrowHead (style and parameters) <code>a</code> to the current arrowhead.
	 * @param a The copied arrowhead
	 * @since 1.9
	 */
	public void copyArrowParameters(ArrowHead a)
	{
		arrowInset   = a.arrowInset;
		arrowLength  = a.arrowLength;
		arrowSizeDim = a.arrowSizeDim;
		arrowSizeNum = a.arrowSizeNum;
		bracketNum   = a.bracketNum;
		currentArrowStyle = a.currentArrowStyle;
		dotSizeDim   = a.dotSizeDim;
		dotSizeNum   = a.dotSizeNum;
		rBracketNum  = a.rBracketNum;
		tBarSizeDim  = a.tBarSizeDim;
		tBarSizeNum  = a.tBarSizeNum;
		arrowInset   = a.arrowInset;
		arrowLength  = a.arrowLength;
		arrowSizeDim = a.arrowSizeDim;
		arrowSizeNum = a.arrowSizeNum;
	}
	
	
	
	/**
	 * Sometimes it is useful to have the inverse of an arrow style (e.g. when you have > or >> you
	 * would like to get < or <<).
	 * @param arrowS The arrow style to reverse.
	 * @return The opposite of the given arrow.
	 * @since 1.9.1
	 */
	public static String invertArrowStyle(String arrowS)
	{
		if(arrowS.equals(PSTricksConstants.DLARROW_STYLE))
			return PSTricksConstants.DRARROW_STYLE;
		if(arrowS.equals(PSTricksConstants.DRARROW_STYLE))
			return PSTricksConstants.DLARROW_STYLE;
		if(arrowS.equals(PSTricksConstants.RARROW_STYLE))
			return PSTricksConstants.LARROW_STYLE;
		if(arrowS.equals(PSTricksConstants.LARROW_STYLE))
			return PSTricksConstants.RARROW_STYLE;
		if(arrowS.equals(PSTricksConstants.LRBRACKET_STYLE))
			return PSTricksConstants.RRBRACKET_STYLE;
		if(arrowS.equals(PSTricksConstants.RRBRACKET_STYLE))
			return PSTricksConstants.LRBRACKET_STYLE;
		if(arrowS.equals(PSTricksConstants.RSBRACKET_STYLE))
			return PSTricksConstants.LSBRACKET_STYLE;
		if(arrowS.equals(PSTricksConstants.LSBRACKET_STYLE))
			return PSTricksConstants.RSBRACKET_STYLE;
		if(arrowS.equals(PSTricksConstants.NONEARROW_STYLE))
			return PSTricksConstants.NONEARROW_STYLE;
		
		return arrowS;
	}


	
	/**
	 * @return True of the arrow can be drawn (if the two points of its line are not at the same location).
	 * @since 1.9.4
	 */
	public boolean isDrawable()
	{
		LaTeXDrawPoint2D pt1 = line.getPt1(), pt2 = line.getPt2();
		return pt1!=pt2 && (pt1.x!=pt2.x || pt1.y!=pt2.y);
	}



	/**
	 * @return the figure.
	 * @since 2.0.0
	 */
	public synchronized Figure getFigure()
	{
		return figure;
	}



	/**
	 * @param figure the figure to set.
	 * @throws IllegalArgumentException If figure is null.
	 * @since 2.0.0
	 */
	public synchronized void setFigure(Figure figure)
	{
		if(figure==null)
			throw new IllegalArgumentException();
		
		this.figure = figure;
	}
	
	
	
	/**
	 * @return The thickness of the arrowhead.
	 * @since 2.0.0
	 */
	public float getThickness()
	{
		return (float)(figure.hasDoubleBoundary() ? figure.getThickness()*2 + figure.getDoubleSep() : figure.getThickness());
	}



	/**
	 * @return True if the arrow style is inverted.
	 * @since 2.0.0
	 */
	public boolean isInverted()
	{
		LaTeXDrawPoint2D p1  = line.getPt1();
		LaTeXDrawPoint2D p2  = line.getPt2();
		LaTeXDrawPoint2D pos = getPosition();
		
		if(currentArrowStyle.equals(PSTricksConstants.DLARROW_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.LARROW_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.LRBRACKET_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.LSBRACKET_STYLE))
			return pos==p1 ? false : true; 
		
		if(currentArrowStyle.equals(PSTricksConstants.DRARROW_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.RARROW_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.RRBRACKET_STYLE) ||
			currentArrowStyle.equals(PSTricksConstants.RSBRACKET_STYLE))
			return pos==p2 ? false : true;
		
		return false;
	}
	
	
	
	/**
	 * @return True if the arrow is a left arrow.
	 * @since 2.0.0
	 */
	public boolean isLeftArrow()
	{
		return line.getPt1()==position ? true : false;
	}
}
