package latexDraw.generators.svg;

import java.awt.geom.Arc2D;
import java.awt.geom.Point2D;

import latexDraw.figures.Arc;
import latexDraw.figures.ArrowHead;
import latexDraw.figures.LaTeXDrawRectangle;
import latexDraw.parsers.svg.SVGAttributes;
import latexDraw.parsers.svg.SVGDocument;
import latexDraw.parsers.svg.elements.*;
import latexDraw.parsers.svg.elements.path.*;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.util.LaTeXDrawNamespace;
import latexDraw.util.LaTeXDrawPoint2D;

/**
 * Defines a SVG generator for an arc.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 11/11/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 */
public class LArcSVGGenerator extends LCircleSVGGenerator
{
	public LArcSVGGenerator(Arc f)
	{
		super(f);
	}

	
	
	public LArcSVGGenerator(SVGGElement elt)
	{
		this(elt, true);
	}
	
	
	
	/**
	 * Creates an arc from a latexdraw-SVG element.
	 * @param elt The source element.
	 * @param withTransformation If true, the SVG transformations will be applied.
	 * @since 2.0.0
	 */
	public LArcSVGGenerator(SVGGElement elt, boolean withTransformation)
	{
		this(new Arc(false));

		Arc a = (Arc)getShape();
		SVGElement elt2 = getLaTeXDrawElement(elt, null);
		ArrowHead ah1 = a.getArrowHead1();
		ArrowHead ah2 = a.getArrowHead2();
		
		if(elt==null || elt2==null || !(elt2 instanceof SVGPathElement))
			throw new IllegalArgumentException();
		
		SVGPathElement main = (SVGPathElement)elt2;
		SVGPathSegList l = main.getSegList();
		double sx, sy;
		LaTeXDrawRectangle borders = a.getBorders();
		
		if(l.size()<2 && !(l.firstElement() instanceof SVGPathSegMoveto) && !(l.elementAt(1) instanceof SVGPathSegArc))
			throw new IllegalArgumentException();
		
		sx = ((SVGPathSegMoveto)l.firstElement()).getX();
		sy = ((SVGPathSegMoveto)l.firstElement()).getY();
		Arc2D arc = ((SVGPathSegArc)l.elementAt(1)).getArc2D(sx, sy);
		
		a.setStartAngle(Math.toRadians(arc.getAngleStart())%(Math.PI*2));
		a.setEndAngle(Math.toRadians(arc.getAngleExtent()+arc.getAngleStart())%(Math.PI*2));
		borders.getPoint(0).setLocation(arc.getMinX(), arc.getMinY());
		borders.getPoint(1).setLocation(arc.getMaxX(), arc.getMinY());
		borders.getPoint(2).setLocation(arc.getMinX(), arc.getMaxY());
		borders.getPoint(3).setLocation(arc.getMaxX(), arc.getMaxY());
		
		if(l.size()>2)
			if(l.elementAt(2) instanceof SVGPathSegClosePath)
				a.setType(Arc2D.CHORD);
			else
				if(l.size()==4 && l.elementAt(2) instanceof SVGPathSegLineto && l.elementAt(3) instanceof SVGPathSegClosePath)
					a.setType(Arc2D.PIE);
		
		a.setShowPoints(getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_SHOW_PTS)!=null);
		setNumber(elt);
		setSVGLatexdrawParameters(elt);
		setSVGParameters(main);
		setSVGShadowParameters(getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_SHADOW));
		a.updateCenterStartEndDelimitors();
		setSVGArrow(ah1, main.getAttribute(main.getUsablePrefix()+SVGAttributes.SVG_MARKER_START), main);
		setSVGArrow(ah2, main.getAttribute(main.getUsablePrefix()+SVGAttributes.SVG_MARKER_END), main);
		homogeniseArrows(ah1, ah2);
		a.updateShape();
		a.updateGravityCenter();
		
		if(withTransformation)
			applyTransformations(elt);
	}

	
	@Override
	public SVGElement toSVG(SVGDocument doc)
	{
		if(doc==null || doc.getFirstChild().getDefs()==null)
			return null;
		
		Arc arc = (Arc)shape;
		SVGDefsElement defs  = doc.getFirstChild().getDefs();
		double rotationAngle = shape.getRotationAngle();
		ArrowHead arrowHead1 = arc.getArrowHead1();
		ArrowHead arrowHead2 = arc.getArrowHead2();
		double startAngle    =  arc.getStartAngle();
		double endAngle      =  arc.getEndAngle();
		int type 			 =  arc.getType();
		Point2D.Double gravityCenter = shape.getGravityCenter();
        SVGElement root = new SVGGElement(doc), elt;
        int number 		= shape.getNumber();
        root.setAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_TYPE, LaTeXDrawNamespace.XML_TYPE_ARC);
        root.setAttribute(SVGAttributes.SVG_ID, getSVGID());
        LaTeXDrawPoint2D start = arc.getStartPoint(false), end = arc.getEndPoint(false);
        double radius = arc.getRadius();
        boolean largeArcFlag;
        SVGElement arrow1 = null, arrow2 = null, arrow1Shad = null, arrow2Shad = null;
        String arrow1Name = "arrow1-" + number; //$NON-NLS-1$
 	   	String arrow2Name = "arrow2-" + number;//$NON-NLS-1$
 	   	String arrow1ShadName = "arrow1Shad-" + number;//$NON-NLS-1$
 	   	String arrow2ShadName = "arrow2Shad-" + number;//$NON-NLS-1$
 	   
 	   	startAngle%=2.*Math.PI;
 	   	endAngle%=2.*Math.PI;
      
        if(!arrowHead1.isWithoutStyle())
        {
 			arrow1 		= new LArrowHeadSVGGenerator(arrowHead1).toSVG(doc, false);
 			arrow1Shad 	= new LArrowHeadSVGGenerator(arrowHead1).toSVG(doc, true);

 			arrow1.setAttribute(SVGAttributes.SVG_ID, arrow1Name);
 			defs.appendChild(arrow1);
 			
 			if(shape.hasShadow())
 			{
 				arrow1Shad.setAttribute(SVGAttributes.SVG_ID, arrow1ShadName);
 				defs.appendChild(arrow1Shad);
 			}
        }

        if(!arrowHead2.isWithoutStyle())
        {
     	   arrow2 		= new LArrowHeadSVGGenerator(arrowHead2).toSVG(doc, false);
     	   arrow2Shad 	= new LArrowHeadSVGGenerator(arrowHead2).toSVG(doc, true);

     	   arrow2.setAttribute(SVGAttributes.SVG_ID, arrow2Name);
     	   defs.appendChild(arrow2);
     	   
 			if(shape.hasShadow())
 			{
 	    	   arrow2Shad.setAttribute(SVGAttributes.SVG_ID, arrow2ShadName);
 	    	   defs.appendChild(arrow2Shad);
 			}
        }
        
        largeArcFlag = startAngle>endAngle ? 2.*Math.PI-startAngle+endAngle>=Math.PI : endAngle-startAngle>Math.PI;
        
        SVGPathSegList path = new SVGPathSegList();
        
        path.add(new SVGPathSegMoveto(start.x, start.y, false));
        path.add(new SVGPathSegArc(end.x, end.y, radius, radius, 0, largeArcFlag, false, false));
        
        if(type==Arc2D.CHORD)
        	path.add(new SVGPathSegClosePath());
        else
        	if(type==Arc2D.PIE)
        	{
        		path.add(new SVGPathSegLineto(gravityCenter.x, gravityCenter.y, false));
        		path.add(new SVGPathSegClosePath());
        	}
        
        if(shape.hasShadow())
        {
     	   SVGElement shad = new SVGPathElement(doc);
         	
     	   shad.setAttribute(SVGAttributes.SVG_D, path.toString());
     	   setSVGShadowAttributes(shad, true);
     	   root.appendChild(shad);
     	   
           if(arrow1Shad!=null)
        	   shad.setAttribute(SVGAttributes.SVG_MARKER_START, "url(#" + arrow1ShadName + ")");//$NON-NLS-1$ //$NON-NLS-2$
           
           if(arrow2Shad!=null)
        	   shad.setAttribute(SVGAttributes.SVG_MARKER_END, "url(#" + arrow2ShadName + ")");//$NON-NLS-1$ //$NON-NLS-2$
        }
        
        if(shape.hasShadow() && !shape.getLineStyle().equals(PSTricksConstants.LINE_NONE_STYLE))
        {// The background of the borders must be filled is there is a shadow.
            elt = new SVGPathElement(doc);
            elt.setAttribute(SVGAttributes.SVG_D, path.toString());
            setSVGBorderBackground(elt, root);
        }
        
        elt = new SVGPathElement(doc);
        elt.setAttribute(SVGAttributes.SVG_D, path.toString());
        root.appendChild(elt);
        
        setSVGRotationAttribute(root);
        setSVGAttributes(doc, elt, true);
        elt.setAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE +':'+ LaTeXDrawNamespace.XML_ROTATION, String.valueOf(rotationAngle));
        
        if(arrow1!=null)
     	   elt.setAttribute(SVGAttributes.SVG_MARKER_START, "url(#" + arrow1Name + ")");//$NON-NLS-1$ //$NON-NLS-2$
        
        if(arrow2!=null)
     	   elt.setAttribute(SVGAttributes.SVG_MARKER_END, "url(#" + arrow2Name + ")");//$NON-NLS-1$ //$NON-NLS-2$
        
        if(arc.isShowPoints())
        	root.appendChild(getShowPointsElement(doc));
        
        return root;
	}
	
	
	
	/**
	 * Creates an SVG g element that contains the 'show points' plotting.
	 * @param doc The owner document.
	 * @return The created g element or null if the shape has not the 'show points' option activated.
	 * @since 2.0.0
	 */
	protected SVGGElement getShowPointsElement(SVGDocument doc)
	{
		Arc arc = (Arc)shape;
		
		if(!arc.isShowPoints() || doc==null)
			return null;
		
		SVGGElement showPts = new SVGGElement(doc);
		float thickness	    = arc.getThickness()/2f;
		
		showPts.setAttribute(new StringBuffer(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE).append(':').append(
				LaTeXDrawNamespace.XML_TYPE).toString(), LaTeXDrawNamespace.XML_TYPE_SHOW_PTS);
		
		showPts.appendChild(getShowPointsLine(doc, thickness, arc.getLinesColor(), arc.getGravityCenter(), 
				arc.getNonRotatedStartPoint(), arc.getBlackDashLength(), arc.getWhiteDashLength(), false, 1, shape.getDoubleSep()));
		showPts.appendChild(getShowPointsLine(doc, thickness, arc.getLinesColor(), arc.getGravityCenter(), 
				arc.getNonRotatedEndPoint(), arc.getBlackDashLength(), arc.getWhiteDashLength(), false, 1, shape.getDoubleSep()));
		
		return showPts;
	}
}
