package latexDraw.ui.components;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.RenderingHints;
import java.awt.geom.Line2D;
import java.awt.image.BufferedImage;

import latexDraw.psTricks.PSTricksConstants;
import latexDraw.util.LaTeXDrawPoint2D;

/** 
 * This class defines a magnetic grid.<br>
 * <br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 * <br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 * <br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 * <br>
 * 01/21/08<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class MagneticGrid
{
	/** The max dimension of the grid. */
	public static final int MAX_WIDTH_HEIGHT = 2000;
	
	public static final BasicStroke STROKE = new BasicStroke(0, BasicStroke.CAP_SQUARE, BasicStroke.JOIN_MITER);
	
	/** The gap by default between two lines of the personal grid. */
	public static final int DEFAULT_PERSONAL_GRID_GAP = 10;
	
	public static final short TYPE_PERSONAL = 0;
	
	public static final short TYPE_INCH = 1;
	
	public static final short TYPE_CM = 2;
	
	public static final short DEFAULT_TYPE = TYPE_PERSONAL;
	
	/** Allows to know if the grid is magnetic or not. */
	protected boolean isMagnetic;
	
	protected int persoInterval;
	
	protected transient BufferedImage buffer;
	
	protected int width;
	
	protected int height;
	
	protected short type;
	
	protected double zoom;
	
	protected int pixelsPerCm;
	
	/** Allows to know if the grid must be displayed */
	protected boolean isGridDisplayed;
	
	
	
	
	public MagneticGrid(int width, int height, int ppc, double zoom)
	{
		if(width<0 || height<0 || ppc<1 || zoom<=0)
			throw new IllegalArgumentException();
		
		isMagnetic  = false;
		this.width  = width>MAX_WIDTH_HEIGHT ? MAX_WIDTH_HEIGHT : width;
		this.height = height>MAX_WIDTH_HEIGHT ? MAX_WIDTH_HEIGHT : height;
		buffer 		= null;
		this.zoom	= zoom;
		pixelsPerCm = ppc;
		isGridDisplayed = true;
		persoInterval = DEFAULT_PERSONAL_GRID_GAP;
		reInitGrid();
	}
	
	
	
	public void paint(Graphics2D g)
	{
		if(buffer!=null && isGridDisplayed())
			g.drawImage(buffer, 0, 0, null);
	}
	
	
	
	@SuppressWarnings("fallthrough")
	public void updateBuffer()
	{
		if(getHeight()==0 || getWidth()==0)
		{
			if(buffer!=null)
				buffer.flush();
			
			buffer = null;
		}
		else
		{
			if(buffer==null)
				buffer = new BufferedImage(getWidth(), getHeight(), BufferedImage.TYPE_INT_ARGB);
			
			Graphics2D g  = (Graphics2D)buffer.getGraphics();
			double ppc    = pixelsPerCm, i, j, gap = persoInterval;
			
			g.setColor(Color.WHITE);
			g.fillRect(0, 0, getWidth(), getHeight());
			g.setColor(Color.BLACK);	    	
			g.setStroke(STROKE);
			g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
			g.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
			g.setRenderingHint(RenderingHints.KEY_ALPHA_INTERPOLATION, RenderingHints.VALUE_ALPHA_INTERPOLATION_QUALITY);
			
			switch(type)
			{
				case TYPE_INCH:
					ppc*=PSTricksConstants.INCH_VAL_CM;
					
				case TYPE_CM:
					gap = ppc;
					
					if(ppc>20)
			    	{
			    		if(ppc/10.!=(int)(ppc/10.))
			    		{
			    			ppc = Math.rint(ppc/10.)*10.;
			    			gap = ppc;
			    		}
			    		
			    		double pixPerCm10 = ppc/10.;
			    		int cpt;
			    		ppc*=getZoom();
			    		pixPerCm10*=getZoom();
			    		
			    		for(i=pixPerCm10-1; i<width; i+=ppc)
			    			for(j=i, cpt=0; cpt<10; j+=pixPerCm10, cpt++)
			    				g.draw(new Line2D.Double(j, 0, j, height));
			    		
			    		for(i=pixPerCm10-1; i<height; i+=ppc)
			    			for(j=i, cpt=0; cpt<10; j+=pixPerCm10, cpt++)
			    				g.draw(new Line2D.Double(0, j, width, j));
			    	}
					
				case TYPE_PERSONAL:
					gap*=getZoom();
					
					for(i=gap-1; i<width; i+=gap)
	    				g.draw(new Line2D.Double(i, 0, i, height));
		    		
		    		for(j=gap-1; j<height; j+=gap)
	    				g.draw(new Line2D.Double(0, j, width, j));
		    		
					break;
			}
		}
	}

	
	
	
	/**
	 * Transform a point to another "stick" to the magnetic grid.
	 * @since 1.8
	 * @param pt The point to transform.
	 * @param considerGridIsMagnetic True if we must consider if the grid is magnetic or not;<br>
	 * for instance, if the grid is not magnetic, <code>considerGridIsMagnetic=false</code> and a grid is
	 * displayed, the returned point will be a transformed point according to the magnetic grid.<br>
	 * If the grid is not magnetic and <code>considerGridIsMagnetic=true</code>, the returned point will
	 * be a clone of the point given in parameter.
	 * @return The transformed point or if there is no magnetic grid, a clone of the given point.
	 */
	public Point getTransformedPointToGrid(Point pt, boolean considerGridIsMagnetic)
	{
	   	if(isMagnetic() || (!considerGridIsMagnetic && isGridDisplayed()))
    	{
	   		Point point = (Point)pt.clone();
    		double modulo = getMagneticGridGap();
    		double x = point.x;
    		double y = point.y;
    		int base = (int)(((int)(x/modulo))*modulo);
    		
    		if(x>modulo) x = x%((int)modulo);
    		
    		double res = modulo-x;
    		x = base;
    		
    		if(res<modulo/2.) x+=modulo;
    		
    		point.x = (int)x;
    		base = (int)(((int)(point.y/modulo))*modulo);
    		
    		if(y>modulo) y = y%((int)modulo);
    		
    		res = modulo-y;
    		y = base;
    		
    		if(res<modulo/2.) y+=modulo;
    		
    		point.y = (int)y;

    		return point;
    	}
	   	return (Point)pt.clone();
	}

	
	
	/**
	 * @see #getTransformedPointToGrid(Point, boolean)
	 * @since 1.9
	 */
	public Point getTransformedPointToGrid(LaTeXDrawPoint2D pt, boolean considerGridIsMagnetic)
	{
		return getTransformedPointToGrid(new Point((int)pt.x, (int)pt.y), considerGridIsMagnetic);
	}
	
	
	
	/**
	 * @return The gap between the lines of the magnetic grid.
	 * @since 1.9
	 */
	public double getMagneticGridGap()
	{
		double gap;
		
		if(isPersonalGrid())
			gap = getPersoInterval();
		else
		{
			gap = type==TYPE_CM? getPixelsPerCm()/10. : (getPixelsPerCm()*PSTricksConstants.INCH_VAL_CM)/10.;
			gap = (gap-(int)gap)>0.5? ((int)gap)+1 : (int)gap;
		}
		
		return gap;
	}
	
	
	
	public void reInitGrid()
	{
		type          = DEFAULT_TYPE;
		persoInterval = DEFAULT_PERSONAL_GRID_GAP;
		isMagnetic	  = false;
		updateBuffer();
	}
	

	/**
	 * @return the isMagnetic
	 * @since 2.0.0
	 */
	public boolean isMagnetic()
	{
		return isMagnetic;
	}


	/**
	 * @param isMagnetic the isMagnetic to set
	 * @since 2.0.0
	 */
	public void setMagnetic(boolean isMagnetic)
	{
		this.isMagnetic = isMagnetic;
	}


	/**
	 * @return the interval
	 * @since 2.0.0
	 */
	public int getPersoInterval()
	{
		return persoInterval;
	}


	/**
	 * @param interval the interval to set
	 * @since 2.0.0
	 */
	public void setPersoInterval(int interval)
	{
		if(interval>1)
		{
			int oldInter = this.persoInterval;
			this.persoInterval = interval;
			
			if(oldInter!=persoInterval && isGridDisplayed())
				updateBuffer();
		}
	}


	
	public boolean isPersonalGrid()
	{
		return getType()==TYPE_PERSONAL;
	}
	
	
	
	/**
	 * @return the width
	 * @since 2.0.0
	 */
	public int getWidth()
	{
		return width;
	}


	
	/**
	 * @param width the width to set
	 * @since 2.0.0
	 */
	public void setWidth(int width)
	{
		if(width>=0)
		{
			int oldW = this.width;
			this.width = width>MAX_WIDTH_HEIGHT ? MAX_WIDTH_HEIGHT : width;
			
			if(oldW!=getWidth() && getHeight()>0 && getWidth()>0)
			{
				if(buffer!=null)
					buffer.flush();
				
				buffer = new BufferedImage(getWidth(), getHeight(), BufferedImage.TYPE_INT_ARGB);
				updateBuffer();
			}
		}
	}

	

	/**
	 * @return the height
	 * @since 2.0.0
	 */
	public int getHeight()
	{
		return height;
	}


	
	/**
	 * @param height the height to set
	 * @since 2.0.0
	 */
	public void setHeight(int height)
	{
		if(height>=0)
		{
			int oldH = this.height;
			this.height = height>MAX_WIDTH_HEIGHT ? MAX_WIDTH_HEIGHT : height;
			
			if(oldH!=getHeight() && getWidth()>0 && getHeight()>0)
			{
				if(buffer!=null)
					buffer.flush();
				
				buffer = new BufferedImage(getWidth(), getHeight(), BufferedImage.TYPE_INT_ARGB);
				updateBuffer();
			}
		}
	}


	
	/**
	 * @return the buffer
	 * @since 2.0.0
	 */
	public BufferedImage getBuffer()
	{
		return buffer;
	}


	
	/**
	 * @return the type
	 * @since 2.0.0
	 */
	public short getType()
	{
		return type;
	}

	

	/**
	 * @param type the type to set
	 * @since 2.0.0
	 */
	public void setType(short type)
	{
		if(type==TYPE_CM || type==TYPE_INCH || type==TYPE_PERSONAL)
		{
			short oldType = this.type;
			this.type = type;

			if(oldType!=type && isGridDisplayed())
				updateBuffer();
		}
	}


	
	/**
	 * @return The zoom.
	 * @since 2.0.0
	 */
	public double getZoom()
	{
		return zoom;
	}


	
	/**
	 * @param zoom The zoom to set.
	 * @since 2.0.0
	 */
	public void setZoom(double zoom)
	{
		if(zoom>0)
		{
			double oldZ = this.zoom;
			this.zoom = zoom;
			
			if(oldZ!=zoom && isGridDisplayed())
				updateBuffer();
		}
	}



	/**
	 * @return the pixelsPerCm
	 * @since 2.0.0
	 */
	public int getPixelsPerCm()
	{
		return pixelsPerCm;
	}


	
	/**
	 * @param pixelsPerCm the pixelsPerCm to set
	 * @since 2.0.0
	 */
	public void setPixelsPerCm(int pixelsPerCm)
	{
		if(pixelsPerCm>0)
		{
			int oldPPC = this.pixelsPerCm;
			this.pixelsPerCm = pixelsPerCm;
			
			if(oldPPC!=pixelsPerCm && isGridDisplayed())
				updateBuffer();
		}
	}


	
	/**
	 * @return The isGridDisplayed.
	 * @since 2.0.0
	 */
	public boolean isGridDisplayed()
	{
		return isGridDisplayed;
	}


	
	/**
	 * @param isGridDisplayed The isGridDisplayed to set.
	 * @since 2.0.0
	 */
	public void setGridDisplayed(boolean isGridDisplayed)
	{
		this.isGridDisplayed = isGridDisplayed;
		
		if(isGridDisplayed)
			updateBuffer();
	}
}
