package latexDraw.ui.dialog;

import java.awt.*;

import javax.swing.*;
import javax.swing.border.CompoundBorder;
import javax.swing.border.EmptyBorder;
import javax.swing.border.TitledBorder;
import javax.swing.event.ChangeEvent;

import latexDraw.figures.Figure;
import latexDraw.figures.Triangle;
import latexDraw.lang.LaTeXDrawLang;
import latexDraw.ui.DrawPanel;
import latexDraw.util.LaTeXDrawPoint2D;
import latexDraw.util.LaTeXDrawResources;


/** 
 * This class defines the frame which allows to change the parameters of triangles.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 01/20/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class ParametersTriangleFrame extends AbstractParametersFrame
{
	private static final long serialVersionUID = 1L;

	/** Allows to change the x-coordinates of the centre of the figure */
	protected JSpinner baseCenterXField;
	
	/** Allows to change the y-coordinates of the centre of the figure */
	protected JSpinner baseCenterYField;
	
	/** Allows to change the width of the triangle */
	protected JSpinner widthField;
	
	/** Allows to change the height of the triangle */
	protected JSpinner heightField;
	

	
	/**
	 * The constructor by default
	 */
	public ParametersTriangleFrame(Frame parent, DrawPanel p, boolean attachToMainFrame)
	{
		super(parent, p, attachToMainFrame);
		
		try
		{
	  		Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
	  		JTabbedPane tabbedPane = new JTabbedPane();
	  		JPanel pButton  = createButtonsPanel(this);
	  		JPanel pGeneral = new JPanel();
	  		pGeneral.setLayout(new BoxLayout(pGeneral, BoxLayout.Y_AXIS));
	  		JPanel coordP = new JPanel(new GridBagLayout());
	  		GridBagConstraints constraint = new GridBagConstraints();
	    	constraint.gridx = 0;
	     	constraint.gridy = 0;
	     	constraint.gridwidth = 1;
	     	constraint.gridheight = 1;
	     	constraint.weightx = 0.1;
	     	constraint.weighty = 0.1;
	     	constraint.fill = GridBagConstraints.NONE;
	     	constraint.anchor = GridBagConstraints.CENTER;
	     	coordP.add(new JLabel(LaTeXDrawLang.getDialogFrameString("ParametersTriangleFrame.0")), constraint); //$NON-NLS-1$
	  		
	     	SpinnerModel model = new SpinnerNumberModel(0,0,10000,0.1);
	     	baseCenterXField = new JSpinner(model);
	     	baseCenterXField.setEditor(new JSpinner.NumberEditor(baseCenterXField, "0.0"));//$NON-NLS-1$
			baseCenterXField.addChangeListener(this);
	     	baseCenterXField.setName(LABEL_CENTER_X);
	     	constraint.gridx = 1;	
	     	coordP.add(baseCenterXField, constraint);

	    	constraint.gridx++;
	    	coordP.add(new JLabel(LaTeXDrawLang.getDialogFrameString("ParametersTriangleFrame.1")), constraint); //$NON-NLS-1$
	  		
	     	model = new SpinnerNumberModel(0,0,10000,0.1);
	     	baseCenterYField = new JSpinner(model);
	     	baseCenterYField.setEditor(new JSpinner.NumberEditor(baseCenterYField, "0.0"));//$NON-NLS-1$
	     	baseCenterYField.addChangeListener(this);
	     	baseCenterYField.setName(LABEL_CENTER_Y);
	     	constraint.gridx++;	
	     	coordP.add(baseCenterYField, constraint);
	     	
	     	model = new SpinnerNumberModel(Figure.DEFAULT_ROTATION_ANGLE, -360, 360,0.1);
	     	rotationField = new JSpinner(model);
	     	rotationField.setEditor(new JSpinner.NumberEditor(rotationField, "0.0"));	//$NON-NLS-1$     	
	     	rotationField.setName(LaTeXDrawResources.LABEL_ROTATE);
	     	rotationField.addChangeListener(this);
	     	rotationField.setName(LaTeXDrawResources.LABEL_ROTATE);
	     	constraint.gridx = 1;	constraint.gridwidth = 1;
	     	coordP.add(rotationField, constraint);
	     	
	    	constraint.gridx = 0;	constraint.gridy++;
	    	coordP.add(new JLabel(LABEL_WIDTH), constraint);
	  		
	     	model = new SpinnerNumberModel(0.1,0.1,10000,0.1);
	     	widthField = new JSpinner(model);
	     	widthField.setEditor(new JSpinner.NumberEditor(widthField, "0.0"));//$NON-NLS-1$
	     	widthField.addChangeListener(this);
	     	widthField.setName(LABEL_WIDTH);
	     	constraint.gridx++;
	     	coordP.add(widthField, constraint);
	   
	     	constraint.gridx++;
	     	coordP.add(new JLabel(LABEL_HEIGHT), constraint);
	  		
	     	model = new SpinnerNumberModel(0.1,0.1,10000,0.1);
	     	heightField = new JSpinner(model);
	     	heightField.setEditor(new JSpinner.NumberEditor(heightField, "0.0"));//$NON-NLS-1$
	     	heightField.addChangeListener(this);
	     	heightField.setName(LABEL_HEIGHT);
	     	constraint.gridx++;
	     	coordP.add(heightField, constraint);
	     	
	     	constraint.gridx = 0;	constraint.gridy++;
	     	coordP.add(new JLabel(LABEL_ROTATION), constraint);
	     	constraint.gridx++;
	     	coordP.add(rotationField, constraint);
	     	
	     	coordP.setBorder(new CompoundBorder(new TitledBorder(null, LABEL_POSITION_DIMENSIONS, 
							  TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(0,0,0,0)));
	     	pGeneral.add(coordP);
	     	
	     	pGeneral.add(createBordersPanel(true));
	     	pGeneral.add(createDoubleLinePanel());
	     	pGeneral.add(createFillPanel());
	     	
	     	JPanel pOther = new JPanel();
	     	pOther.setLayout(new BoxLayout(pOther, BoxLayout.Y_AXIS));
	     	pOther.add(createShadowPanel());
	     	pOther.add(createGradientPanel());
	     	pOther.add(createHatchingPanel());
	     	
	     	tabbedPane.addTab(TITLE_TABBED_PANE_GENERAL, pGeneral);
	     	tabbedPane.addTab(TITLE_TABBED_PANE_OTHERS, pOther);
	     	BoxLayout bl = new BoxLayout(getContentPane(), BoxLayout.Y_AXIS);
	     	getContentPane().setLayout(bl);
	     	getContentPane().add(glimpsePanelZoom);
	     	getContentPane().add(tabbedPane);
			getContentPane().add(pButton);
	  		pack();
	  		setLocation(dim.width/2-getWidth()/2, dim.height/2-getHeight()/2);
			setVisible(false);
		}
		catch(Exception e)
		{
			e.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(e);
		}
	}
	
	
	
	
	@Override
	public void saveParameters()
	{
		super.saveParameters();
		
		Triangle t = (Triangle)figure;
		LaTeXDrawPoint2D p1 = t.getPoint(0), p2 = t.getPoint(1), 
			  p3 = t.getPoint(2), p4 = t.getPoint(3);
		double width = Double.valueOf(widthField.getValue().toString()).doubleValue();
		double height = Double.valueOf(heightField.getValue().toString()).doubleValue();
		double x = Double.valueOf(baseCenterXField.getValue().toString()).doubleValue();
		double y = Double.valueOf(baseCenterYField.getValue().toString()).doubleValue();
		
		if(p1.y<p3.y)
		{
			p1.y = y - height;
			p2.y = y - height;
			p3.y = y;
			p4.y = y;
		}
		else
		{
			p1.y = y;
			p2.y = y;
			p3.y = y - height;
			p4.y = y - height;
		}
		
		p1.x = x - width/2.;
		p2.x = x + width/2.;
		p3.x = x - width/2.;
		p4.x = x + width/2.;

		t.updateShape();
		t.getBorders().updateNSEWDelimitors();
		
		drawPanel.getDraw().updateBorders();
		drawPanel.updateDraw(true);
	}

	
	

	@Override
	public void setFigureFrameField()
	{
		Triangle t = (Triangle)figure;
		Triangle t2 = (Triangle) glimpsePanel.getGlimpseFigure();
		LaTeXDrawPoint2D NW = t.getTheNWPoint(), SE = t.getTheSEPoint();
		
		baseCenterXField.removeChangeListener(this);
		baseCenterYField.removeChangeListener(this);
		widthField.removeChangeListener(this);
		heightField.removeChangeListener(this);
		
		baseCenterXField.setValue((NW.x+SE.x)/2.);
		baseCenterYField.setValue(SE.y);
		widthField.setValue((SE.x-NW.x));
		heightField.setValue(SE.y-NW.y);
		
		widthField.addChangeListener(this);
		heightField.addChangeListener(this);
		baseCenterXField.addChangeListener(this);
		baseCenterYField.addChangeListener(this);
		
		for(int i=0; i<Triangle.NB_PTS_TRIANGLE; i++)
			t2.getPoint(i).setLocation(t.getPoint(i));
		
		t2.updateGravityCenter();
		t2.updateShape();
		glimpsePanel.repaint();
	}

	
	
	@Override
	public void stateChanged(ChangeEvent e) 
	{
		try
		{
			Object o = e.getSource();
			
			Triangle f = (Triangle) glimpsePanel.getGlimpseFigure();
			
			if(o instanceof JSpinner)
			{
				String name = ((JSpinner)o).getName();
				LaTeXDrawPoint2D p1 = f.getPoint(0), p2 = f.getPoint(1), p3 = f.getPoint(2), p4 = f.getPoint(3);
				double width = Double.valueOf(widthField.getValue().toString()).doubleValue();
				double height = Double.valueOf(heightField.getValue().toString()).doubleValue();
				double x = Double.valueOf(baseCenterXField.getValue().toString()).doubleValue();
				double y = Double.valueOf(baseCenterYField.getValue().toString()).doubleValue();

				if(name.equals(LABEL_CENTER_X))
				{	
					setModified(true);
					f.shift(x-(p3.x+p4.x)/2., 0);
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_CENTER_Y))
				{
					if(p1.y<p3.y)
						f.shift(0, y-(p3.y+p4.y)/2.);
					else
						f.shift(0, y-(p1.y+p2.y)/2.);

					setModified(true);
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_HEIGHT))
				{
					if(p1.y<p3.y)
					{
						p1.y = y - height;
						p2.y = y - height;
						p3.y = y;
						p4.y = y;
					}
					else
					{
						p1.y = y;
						p2.y = y;
						p3.y = y - height;
						p4.y = y - height;
					}
					
					setModified(true);
					f.updateShape();
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_WIDTH))
				{					
					p1.x = x - width/2.;
					p2.x = x + width/2.;
					p3.x = x - width/2.;
					p4.x = x + width/2.;		
					
					setModified(true);
					f.updateShape();
					glimpsePanel.repaint();
					return ;
				}
			}			
				
			super.stateChanged(e);
		}
		catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}
	
	
	

	@Override
	public void setVisible(boolean visible, Figure selected, boolean deleteOnCancel, boolean isFramedBox) 
	{
		try
		{
			if(!(selected instanceof Triangle))
				throw new ClassCastException();
			
			if(isFramedBox)
			{
				baseCenterXField.setEnabled(false);
				baseCenterYField.setEnabled(false);
				widthField.setEnabled(false);
				heightField.setEnabled(false);
			}
			
			super.setVisible(visible, selected, deleteOnCancel, isFramedBox);
		}
		catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}	
	}
}
