function [valueType] = getValueType(varargin)
% [valueType] = getValueType(typecode, level, version, pkgversion(optional))
%
% Takes
%
% 1. typecode; a string representing the type of object being queried
% 2. level, an integer representing an SBML level
% 3. version, an integer representing an SBML version
% 4. pkgversion, an integer representing the SBML package version 
%       (defaults to 1)
%
% Returns
%
% 1. an array of the value types for an SBML structure of the given typecode, level and version
%
% *NOTE:* The corresponding fields present in an SBML  structure can be found using
%   the function `getStructureFieldnames`

%<!---------------------------------------------------------------------------
% This file is part of libSBML.  Please visit http://sbml.org for more
% information about SBML, and the latest version of libSBML.
%
% Copyright (C) 2013-2017 jointly by the following organizations:
%     1. California Institute of Technology, Pasadena, CA, USA
%     2. EMBL European Bioinformatics Institute (EMBL-EBI), Hinxton, UK
%     3. University of Heidelberg, Heidelberg, Germany
%
% Copyright (C) 2009-2013 jointly by the following organizations: 
%     1. California Institute of Technology, Pasadena, CA, USA
%     2. EMBL European Bioinformatics Institute (EMBL-EBI), Hinxton, UK
%  
% Copyright (C) 2006-2008 by the California Institute of Technology,
%     Pasadena, CA, USA 
%  
% Copyright (C) 2002-2005 jointly by the following organizations: 
%     1. California Institute of Technology, Pasadena, CA, USA
%     2. Japan Science and Technology Agency, Japan
% 
% This library is free software; you can redistribute it and/or modify it
% under the terms of the GNU Lesser General Public License as published by
% the Free Software Foundation.  A copy of the license agreement is provided
% in the file named "LICENSE.txt" included with this software distribution
% and also available online as http://sbml.org/software/libsbml/license.html
%----------------------------------------------------------------------- -->

typecode = varargin{1};
level = varargin{2};
version = varargin{3};

pkgCount = 0;

if (nargin > 3)
    if (nargin < 5)
        error('not enough arguments');
    end;
    pkgCount = length(varargin{4});
 %   packages = cell(1, pkgCount);
  %  pkgVersion = zeros(1, pkgCount);
    packages = varargin{4};
    pkgVersion = varargin{5};
end;


    valueType = [];




done = 1;

switch (typecode)
  case {'SBML_ALGEBRAIC_RULE', 'AlgebraicRule', 'algebraicRule', 'rule'}
    fhandle = str2func('getAlgebraicRuleValueType');
  case {'SBML_ASSIGNMENT_RULE', 'AssignmentRule', 'assignmentRule'}
    fhandle = str2func('getAssignmentRuleValueType');
  case {'SBML_COMPARTMENT', 'Compartment', 'compartment'}
    fhandle = str2func('getCompartmentValueType');
  case {'SBML_COMPARTMENT_TYPE', 'CompartmentType', 'compartmentType'}
    fhandle = str2func('getCompartmentTypeValueType');
  case {'SBML_COMPARTMENT_VOLUME_RULE', 'CompartmentVolumeRule', 'compartmentVolumeRule'}
    fhandle = str2func('getCompartmentVolumeRuleValueType');
  case {'SBML_CONSTRAINT', 'Constraint', 'constraint'}
    fhandle = str2func('getConstraintValueType');
  case {'SBML_DELAY', 'Delay', 'delay'}
    fhandle = str2func('getDelayValueType');
  case {'SBML_EVENT', 'Event', 'event'}
    fhandle = str2func('getEventValueType');
  case {'SBML_EVENT_ASSIGNMENT', 'EventAssignment', 'eventAssignment'}
    fhandle = str2func('getEventAssignmentValueType');
  case {'SBML_FUNCTION_DEFINITION', 'FunctionDefinition', 'functionDefinition'}
    fhandle = str2func('getFunctionDefinitionValueType');
  case {'SBML_INITIAL_ASSIGNMENT', 'InitialAssignment', 'initialAssignment'}
    fhandle = str2func('getInitialAssignmentValueType');
  case {'SBML_KINETIC_LAW', 'KineticLaw', 'kineticLaw'}
    fhandle = str2func('getKineticLawValueType');
  case {'SBML_LOCAL_PARAMETER', 'LocalParameter', 'localParameter'}
    fhandle = str2func('getLocalParameterValueType');
  case {'SBML_MODEL', 'Model', 'model'}
    fhandle = str2func('getModelValueType');
  case {'SBML_MODIFIER_SPECIES_REFERENCE', 'ModifierSpeciesReference', 'modifierSpeciesReference', 'modifier'}
    fhandle = str2func('getModifierSpeciesReferenceValueType');
  case {'SBML_PARAMETER', 'Parameter', 'parameter'}
    fhandle = str2func('getParameterValueType');
  case {'SBML_PARAMETER_RULE', 'ParameterRule', 'parameterRule'}
    fhandle = str2func('getParameterRuleValueType');
  case {'SBML_PRIORITY', 'Priority', 'priority'}
    fhandle = str2func('getPriorityValueType');
  case {'SBML_RATE_RULE', 'RateRule', 'ruleRule'}
    fhandle = str2func('getRateRuleValueType');
  case {'SBML_REACTION', 'Reaction', 'reaction'}
    fhandle = str2func('getReactionValueType');
  case {'SBML_SPECIES', 'Species', 'species'}
    fhandle = str2func('getSpeciesValueType');
  case {'SBML_SPECIES_CONCENTRATION_RULE', 'SpeciesConcentrationRule', 'speciesConcentrationRule'}
    fhandle = str2func('getSpeciesConcentrationRuleValueType');
  case {'SBML_SPECIES_REFERENCE', 'SpeciesReference', 'speciesReference', 'reactant', 'Reactant', 'product', 'Product'}
    fhandle = str2func('getSpeciesReferenceValueType');
  case {'SBML_SPECIES_TYPE', 'SpeciesType', 'speciesType'}
    fhandle = str2func('getSpeciesTypeValueType');
  case {'SBML_STOICHIOMETRY_MATH', 'StoichiometryMath', 'stoichiometryMath'}
    fhandle = str2func('getStoichiometryMathValueType');
  case {'SBML_TRIGGER', 'Trigger', 'trigger'}
    fhandle = str2func('getTriggerValueType');
  case {'SBML_UNIT', 'Unit', 'unit'}
    fhandle = str2func('getUnitValueType');
  case {'SBML_UNIT_DEFINITION', 'UnitDefinition', 'unitDefinition'}
    fhandle = str2func('getUnitDefinitionValueType');
  otherwise
    done = 0; 
end;

if done == 1
  [valueType] = feval(fhandle, level, version);
else
    i = 1;
    found = 0;
    while (i < pkgCount+1)
        [found, fhandle] = getValues(typecode, packages{i}, 0);
        if (found == 1)
            break;
        end;
        i = i + 1;
    end;
    if (found == 1)
        [valueType] = feval(fhandle, level, version, pkgVersion(i));
        done = 1;
    end;
end;
 
if (done == 1 && pkgCount > 0)
    % we may need to extend the fields 
    i = 1;
    while (i < pkgCount+1)
        [found, fhandle] = getValues(typecode, packages{i}, 1);
        if (found == 1)
            copySBML = valueType;
            [newValueType] = feval(fhandle, level, version, pkgVersion(i));
            valueType = [copySBML, newValueType];
        end;
        i = i + 1;
    end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [found, fhandle] = getValues(typecode, pkg, extension)
    fhandle = str2func('disp');
    found = 0;
    if (strcmp(pkg, 'fbc'))
        if (extension)
            if (isFBCExtension(typecode))
                [found, fhandle] = getFBCValueFunction(typecode);
            end;
        else
             [found, fhandle] = getFBCValueFunction(typecode);           
        end;
    elseif strcmp(pkg, 'qual')
      if (extension)
        if (isQUALExtension(typecode))
          [found, fhandle] = getQUALValueTypeFunction(typecode);
        end;
      else
        [found, fhandle] = getQUALValueTypeFunction(typecode);
      end;
    elseif strcmp(pkg, 'groups')
      if (extension)
        if (isGROUPSExtension(typecode))
          [found, fhandle] = getGROUPSValueTypeFunction(typecode);
        end;
      else
        [found, fhandle] = getGROUPSValueTypeFunction(typecode);
      end;
    end;
%%%%% REMOVE END
%%%%% ADD ADDITIONAL
	
	
function extend = isFBCExtension(typecode)
   extend = 0;
   switch (typecode)
        case {'SBML_FBC_MODEL', 'FBCModel', 'SBML_MODEL', 'Model', 'model'}
             extend = 1;
       case {'SBML_FBC_SPECIES', 'FBCSpecies', 'SBML_SPECIES', 'Species', 'species'}
            extend = 1;
        case {'SBML_FBC_REACTION', 'FBCReaction','SBML_REACTION', 'Reaction', 'reaction'}
            extend = 1;
   end;
   
function extend = isQUALExtension(typecode)
  extend = 0;
  switch (typecode)
    case {'SBML_QUAL_MODEL', 'QUALModel', 'SBML_MODEL', 'Model', 'model'}
      extend = 1;
  end;

function extend = isGROUPSExtension(typecode)
  extend = 0;
  switch (typecode)
    case {'SBML_GROUPS_MODEL', 'GROUPSModel', 'SBML_MODEL', 'Model', 'model'}
      extend = 1;
  end;
%%%% ADD isExtension
%%%% ADD isExtension
%%%% END isExtension

function [found, fhandle] = getFBCValueFunction(typecode)
    found = 1;
    switch (typecode)
        case {'SBML_FBC_FLUXBOUND', 'FluxBound', 'fluxBound', 'fbc_fluxBound'}
      fhandle = str2func('getFluxBoundValueType');
        case {'SBML_FBC_FLUXOBJECTIVE', 'FluxObjective', 'fluxObjective', 'fbc_fluxObjective'}
      fhandle = str2func('getFluxObjectiveValueType');
        case {'SBML_FBC_OBJECTIVE', 'Objective', 'objective', 'fbc_objective'}
      fhandle = str2func('getObjectiveValueType');
        case {'SBML_FBC_MODEL', 'FBCModel', 'SBML_MODEL', 'Model', 'model'}
      fhandle = str2func('getFBCModelValueType');
        case {'SBML_FBC_SPECIES', 'FBCSpecies', 'SBML_SPECIES', 'Species', 'species'}
      fhandle = str2func('getFBCSpeciesValueType');
        case {'SBML_FBC_GENE_PRODUCT', 'GeneProduct', 'geneProduct', 'fbc_geneProduct'}
      fhandle = str2func('getGeneProductValueType');
        case {'SBML_FBC_GENE_PRODUCT_ASSOCIATION', 'GeneProductAssociation', 'geneProductAssociation', 'fbc_geneProductAssociation'}
      fhandle = str2func('getGeneProductAssociationValueType');
        case {'SBML_FBC_REACTION', 'FBCReaction','SBML_REACTION', 'Reaction', 'reaction'}
      fhandle = str2func('getFBCReactionValueType');
    case {'SBML_FBC_GENE_PRODUCT_REF', 'Association', 'association', 'SBML_FBC_ASSOCIATION', 'fbc_association'}
      fhandle = str2func('getAssociationValueType');
        otherwise
          fhandle = str2func('disp');
          found = 0;
    end;

function [found, fhandle] = getQUALValueTypeFunction(typecode)
  found = 1;
  switch (typecode)
    case {'SBML_QUAL_QUALITATIVE_SPECIES', 'QualitativeSpecies', 'qualitativeSpecies', 'qual_qualitativeSpecies'}
      fhandle = str2func('getQualitativeSpeciesValueType');
    case {'SBML_QUAL_TRANSITION', 'Transition', 'transition', 'qual_transition'}
      fhandle = str2func('getTransitionValueType');
    case {'SBML_QUAL_INPUT', 'Input', 'input', 'qual_input'}
      fhandle = str2func('getInputValueType');
    case {'SBML_QUAL_OUTPUT', 'Output', 'output', 'qual_output'}
      fhandle = str2func('getOutputValueType');
    case {'SBML_QUAL_DEFAULT_TERM', 'DefaultTerm', 'defaultTerm', 'qual_defaultTerm'}
      fhandle = str2func('getDefaultTermValueType');
    case {'SBML_QUAL_FUNCTION_TERM', 'FunctionTerm', 'functionTerm', 'qual_functionTerm'}
      fhandle = str2func('getFunctionTermValueType');
    case {'SBML_QUAL_MODEL', 'QUALModel', 'SBML_MODEL', 'Model', 'model'}
      fhandle = str2func('getQUALModelValueType');
    otherwise
      fhandle = str2func('disp');
      found = 0;
  end;

function [found, fhandle] = getGROUPSValueTypeFunction(typecode)
  found = 1;
  switch (typecode)
    case {'SBML_GROUPS_GROUP', 'Group', 'group', 'groups_group'}
      fhandle = str2func('getGroupValueType');
    case {'SBML_GROUPS_MEMBER', 'Member', 'member', 'groups_member'}
      fhandle = str2func('getMemberValueType');
    case {'SBML_GROUPS_MODEL', 'GROUPSModel', 'SBML_MODEL', 'Model', 'model'}
      fhandle = str2func('getGROUPSModelValueType');
    otherwise
      fhandle = str2func('disp');
      found = 0;
  end;
%%%% ADD getFieldname
%%%% ADD getFieldname
%%%% END getFieldname

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getAlgebraicRuleValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = {
		               'SBML_ALGEBRAIC_RULE', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
	                };
elseif (level == 2)
	if (version == 1)
		valueType = {
			               'SBML_ALGEBRAIC_RULE', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
	elseif (version == 2)
		valueType = {
			               'SBML_ALGEBRAIC_RULE', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
	elseif (version == 3)
		valueType = {
			               'SBML_ALGEBRAIC_RULE', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_ALGEBRAIC_RULE', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = {
			               'SBML_ALGEBRAIC_RULE', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
        elseif (version == 2)
		valueType = {
			               'SBML_ALGEBRAIC_RULE', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
                                       'char', ...
                                       'char', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getAssignmentRuleValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = [];
elseif (level == 2)
	if (version == 1)
		valueType = {
			               'SBML_ASSIGNMENT_RULE', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
	elseif (version == 2)
		valueType = {
			               'SBML_ASSIGNMENT_RULE', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
	elseif (version == 3)
		valueType = {
			               'SBML_ASSIGNMENT_RULE', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_ASSIGNMENT_RULE', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = {
			               'SBML_ASSIGNMENT_RULE', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
        elseif (version == 2)
		valueType = {
			               'SBML_ASSIGNMENT_RULE', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
                                       'char', ...
                                       'char', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getCompartmentValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = {
		               'SBML_COMPARTMENT', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'double', ...
		               'char', ...
		               'char', ...
		               'bool', ...
	                };
elseif (level == 2)
	if (version == 1)
		valueType = {
			               'SBML_COMPARTMENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'uint', ...
			               'double', ...
			               'char', ...
			               'char', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
		                };
	elseif (version == 2)
		valueType = {
			               'SBML_COMPARTMENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'uint', ...
			               'double', ...
			               'char', ...
			               'char', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
		                };
	elseif (version == 3)
		valueType = {
			               'SBML_COMPARTMENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'uint', ...
			               'double', ...
			               'char', ...
			               'char', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_COMPARTMENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'uint', ...
			               'double', ...
			               'char', ...
			               'char', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = {
			               'SBML_COMPARTMENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'double', ...
			               'double', ...
			               'char', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
		                };
        elseif (version == 2)
		valueType = {
			               'SBML_COMPARTMENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'double', ...
			               'double', ...
			               'char', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getCompartmentTypeValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = [];
elseif (level == 2)
	if (version == 1)
		valueType = [];
	elseif (version == 2)
		valueType = {
			               'SBML_COMPARTMENT_TYPE', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
	elseif (version == 3)
		valueType = {
			               'SBML_COMPARTMENT_TYPE', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_COMPARTMENT_TYPE', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = [];
        elseif (version == 2)
		valueType = [];
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getCompartmentVolumeRuleValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = {
		               'SBML_COMPARTMENT_VOLUME_RULE', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
	                };
elseif (level == 2)
	if (version == 1)
		valueType = [];
	elseif (version == 2)
		valueType = [];
	elseif (version == 3)
		valueType = [];
    elseif (version == 4 || version == 5)
		valueType = [];
	end;
elseif (level == 3)
	if (version == 1)
		valueType = [];
        elseif (version == 2)
		valueType = [];
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getConstraintValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = [];
elseif (level == 2)
	if (version == 1)
		valueType = [];
	elseif (version == 2)
		valueType = {
			               'SBML_CONSTRAINT', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
		                };
	elseif (version == 3)
		valueType = {
			               'SBML_CONSTRAINT', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_CONSTRAINT', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = {
			               'SBML_CONSTRAINT', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
		                };
        elseif (version == 2)
		valueType = {
			               'SBML_CONSTRAINT', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
                                       'char', ...
                                       'char', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getDelayValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = [];
elseif (level == 2)
	if (version == 1)
		valueType = [];
	elseif (version == 2)
		valueType = [];
	elseif (version == 3)
		valueType = {
			               'SBML_DELAY', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_DELAY', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = {
			               'SBML_DELAY', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
		                };
        elseif (version == 2)
		valueType = {
			               'SBML_DELAY', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
                                       'char', ...
                                       'char', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getEventAssignmentValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = [];
elseif (level == 2)
	if (version == 1)
		valueType = {
			               'SBML_EVENT_ASSIGNMENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
	elseif (version == 2)
		valueType = {
			               'SBML_EVENT_ASSIGNMENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
		                };
	elseif (version == 3)
		valueType = {
			               'SBML_EVENT_ASSIGNMENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_EVENT_ASSIGNMENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = {
			               'SBML_EVENT_ASSIGNMENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
		                };
        elseif (version == 2)
		valueType = {
			               'SBML_EVENT_ASSIGNMENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
                                       'char', ...
                                       'char', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getEventValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = [];
elseif (level == 2)
	if (version == 1)
		valueType = {
			               'SBML_EVENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'structure', ...
		                };
	elseif (version == 2)
		valueType = {
			               'SBML_EVENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'structure', ...
		                };
	elseif (version == 3)
		valueType = {
			               'SBML_EVENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_EVENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'bool', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = {
			               'SBML_EVENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'bool', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
		                };
        elseif (version == 2)
		valueType = {
			               'SBML_EVENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'bool', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getFunctionDefinitionValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = [];
elseif (level == 2)
	if (version == 1)
		valueType = {
			               'SBML_FUNCTION_DEFINITION', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
	elseif (version == 2)
		valueType = {
			               'SBML_FUNCTION_DEFINITION', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
	elseif (version == 3)
		valueType = {
			               'SBML_FUNCTION_DEFINITION', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_FUNCTION_DEFINITION', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = {
			               'SBML_FUNCTION_DEFINITION', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
        elseif (version == 2)
		valueType = {
			               'SBML_FUNCTION_DEFINITION', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getInitialAssignmentValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = [];
elseif (level == 2)
	if (version == 1)
		valueType = [];
	elseif (version == 2)
		valueType = {
			               'SBML_INITIAL_ASSIGNMENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
		                };
	elseif (version == 3)
		valueType = {
			               'SBML_INITIAL_ASSIGNMENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_INITIAL_ASSIGNMENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = {
			               'SBML_INITIAL_ASSIGNMENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
		                };
        elseif (version == 2)
		valueType = {
			               'SBML_INITIAL_ASSIGNMENT', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
                                       'char', ...
                                       'char', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getKineticLawValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = {
		               'SBML_KINETIC_LAW', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'structure', ...
		               'char', ...
		               'char', ...
	                };
elseif (level == 2)
	if (version == 1)
		valueType = {
			               'SBML_KINETIC_LAW', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'structure', ...
			               'char', ...
			               'char', ...
		                };
	elseif (version == 2)
		valueType = {
			               'SBML_KINETIC_LAW', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'structure', ...
			               'int', ...
		                };
	elseif (version == 3)
		valueType = {
			               'SBML_KINETIC_LAW', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'structure', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_KINETIC_LAW', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'structure', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = {
			               'SBML_KINETIC_LAW', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'structure', ...
		                };
        elseif (version == 2)
		valueType = {
			               'SBML_KINETIC_LAW', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'structure', ...
                                       'char', ...
                                       'char', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getLocalParameterValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = [];
elseif (level == 2)
	if (version == 1)
		valueType = [];
	elseif (version == 2)
		valueType = [];
	elseif (version == 3)
		valueType = [];
    elseif (version == 4 || version == 5)
		valueType = [];
	end;
elseif (level == 3)
	if (version == 1)
		valueType = {
			               'SBML_LOCAL_PARAMETER', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'double', ...
			               'char', ...
			               'bool', ...
		                };
        elseif (version == 2)
		valueType = {
			               'SBML_LOCAL_PARAMETER', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'double', ...
			               'char', ...
			               'bool', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getModelValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = {
		               'SBML_MODEL', ...
		               'char', ...
		               'char', ...
		               'int', ...
		               'int', ...
		               'char', ...
		               'structure', ...
		               'structure', ...
		               'structure', ...
		               'structure', ...
		               'structure', ...
		               'structure', ...
		               'structure', ...
	                };
elseif (level == 2)
	if (version == 1)
		valueType = {
			               'SBML_MODEL', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                 'int', ...
		                 'int', ...
			               'char', ...
			               'char', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'char', ...
			               'char', ...
			               'structure', ...
		                };
	elseif (version == 2)
		valueType = {
			               'SBML_MODEL', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                 'int', ...
		                 'int', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'char', ...
			               'char', ...
			               'structure', ...
		                };
	elseif (version == 3)
		valueType = {
			               'SBML_MODEL', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                 'int', ...
		                 'int', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'char', ...
			               'char', ...
			               'structure', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_MODEL', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
		                 'int', ...
		                 'int', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'char', ...
			               'char', ...
			               'structure', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = {
			               'SBML_MODEL', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
		                 'int', ...
		                 'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'structure', ...
		                };
        elseif (version == 2)
		valueType = {
			               'SBML_MODEL', ...
			               'char', ...
			               'char', ...
			               'char', ...
                           'structure', ...
		                   'int', ...
		                   'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'char', ...
			               'char', ...
			               'char', ...
                           'char', ...
			               'structure', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getModifierSpeciesReferenceValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = [];
elseif (level == 2)
	if (version == 1)
		valueType = {
			               'SBML_MODIFIER_SPECIES_REFERENCE', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
	elseif (version == 2)
		valueType = {
			               'SBML_MODIFIER_SPECIES_REFERENCE', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
		                };
	elseif (version == 3)
		valueType = {
			               'SBML_MODIFIER_SPECIES_REFERENCE', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_MODIFIER_SPECIES_REFERENCE', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = {
			               'SBML_MODIFIER_SPECIES_REFERENCE', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
        elseif (version == 2)
		valueType = {
			               'SBML_MODIFIER_SPECIES_REFERENCE', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
                                       'char', ...
                                       'char', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getParameterValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = {
		               'SBML_PARAMETER', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'double', ...
		               'char', ...
		               'bool', ...
	                };
elseif (level == 2)
	if (version == 1)
		valueType = {
			               'SBML_PARAMETER', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'double', ...
			               'char', ...
			               'bool', ...
			               'bool', ...
		                };
	elseif (version == 2)
		valueType = {
			               'SBML_PARAMETER', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'double', ...
			               'char', ...
			               'bool', ...
			               'int', ...
			               'bool', ...
		                };
	elseif (version == 3)
		valueType = {
			               'SBML_PARAMETER', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'double', ...
			               'char', ...
			               'bool', ...
			               'bool', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_PARAMETER', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'double', ...
			               'char', ...
			               'bool', ...
			               'bool', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = {
			               'SBML_PARAMETER', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'double', ...
			               'char', ...
			               'bool', ...
			               'bool', ...
		                };
        elseif (version == 2)
		valueType = {
			               'SBML_PARAMETER', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'double', ...
			               'char', ...
			               'bool', ...
			               'bool', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getParameterRuleValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = {
		               'SBML_PARAMETER_RULE', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
	                };
elseif (level == 2)
	if (version == 1)
		valueType = [];
	elseif (version == 2)
		valueType = [];
	elseif (version == 3)
		valueType = [];
    elseif (version == 4 || version == 5)
		valueType = [];
	end;
elseif (level == 3)
	if (version == 1)
		valueType = [];
        elseif (version == 2)
		valueType = [];
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getPriorityValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = [];
elseif (level == 2)
	if (version == 1)
		valueType = [];
	elseif (version == 2)
		valueType = [];
	elseif (version == 3)
		valueType = [];
    elseif (version == 4 || version == 5)
		valueType = [];
	end;
elseif (level == 3)
	if (version == 1)
		valueType = {
			               'SBML_PRIORITY', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
		                };
        elseif (version == 2)
		valueType = {
			               'SBML_PRIORITY', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
                                       'char', ...
                                       'char', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getRateRuleValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = [];
elseif (level == 2)
	if (version == 1)
		valueType = {
			               'SBML_RATE_RULE', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
	elseif (version == 2)
		valueType = {
			               'SBML_RATE_RULE', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
	elseif (version == 3)
		valueType = {
			               'SBML_RATE_RULE', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_RATE_RULE', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = {
			               'SBML_RATE_RULE', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
        elseif (version == 2)
		valueType = {
			               'SBML_RATE_RULE', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
                                       'char', ...
                                       'char', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getReactionValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = {
		               'SBML_REACTION', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'structure', ...
		               'structure', ...
		               'structure', ...
		               'bool', ...
		               'bool', ...
	                };
elseif (level == 2)
	if (version == 1)
		valueType = {
			               'SBML_REACTION', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
		                };
	elseif (version == 2)
		valueType = {
			               'SBML_REACTION', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'bool', ...
			               'bool', ...
			               'int', ...
			               'bool', ...
		                };
	elseif (version == 3)
		valueType = {
			               'SBML_REACTION', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_REACTION', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = {
			               'SBML_REACTION', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
			               'char', ...
		                };
        elseif (version == 2)
		valueType = {
			               'SBML_REACTION', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'structure', ...
			               'bool', ...
			               'char', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getSpeciesConcentrationRuleValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = {
		               'SBML_SPECIES_CONCENTRATION_RULE', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
	                };
elseif (level == 2)
	if (version == 1)
		valueType = [];
	elseif (version == 2)
		valueType = [];
	elseif (version == 3)
		valueType = [];
    elseif (version == 4 || version == 5)
		valueType = [];
	end;
elseif (level == 3)
	if (version == 1)
		valueType = [];
        elseif (version == 2)
		valueType = [];
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getSpeciesValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = {
		               'SBML_SPECIES', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'double', ...
		               'char', ...
		               'bool', ...
			       'int', ...
		               'bool', ...
		               'bool', ...
	                };
elseif (level == 2)
	if (version == 1)
		valueType = {
			               'SBML_SPECIES', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'double', ...
			               'double', ...
			               'char', ...
			               'char', ...
			               'bool', ...
			               'bool', ...
			               'int', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
		                };
	elseif (version == 2)
		valueType = {
			               'SBML_SPECIES', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'double', ...
			               'double', ...
			               'char', ...
			               'char', ...
			               'bool', ...
			               'bool', ...
			               'int', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
		                };
	elseif (version == 3)
		valueType = {
			               'SBML_SPECIES', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'double', ...
			               'double', ...
			               'char', ...
			               'bool', ...
			               'bool', ...
			               'int', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_SPECIES', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'double', ...
			               'double', ...
			               'char', ...
			               'bool', ...
			               'bool', ...
			               'int', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = {
			               'SBML_SPECIES', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'double', ...
			               'double', ...
			               'char', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
			               'char', ...
		                };
        elseif (version == 2)
		valueType = {
			               'SBML_SPECIES', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'double', ...
			               'double', ...
			               'char', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
			               'bool', ...
			               'char', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getSpeciesReferenceValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = {
		               'SBML_SPECIES_REFERENCE', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'int', ...
		               'int', ...
	                };
elseif (level == 2)
	if (version == 1)
		valueType = {
			               'SBML_SPECIES_REFERENCE', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'double', ...
			               'int', ...
			               'char', ...
		                };
	elseif (version == 2)
		valueType = {
			               'SBML_SPECIES_REFERENCE', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'double', ...
			               'char', ...
		                };
	elseif (version == 3)
		valueType = {
			               'SBML_SPECIES_REFERENCE', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'double', ...
			               'structure', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_SPECIES_REFERENCE', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'double', ...
			               'structure', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = {
			               'SBML_SPECIES_REFERENCE', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'double', ...
			               'bool', ...
			               'bool', ...
		                };
        elseif (version == 2)
		valueType = {
			               'SBML_SPECIES_REFERENCE', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'double', ...
			               'bool', ...
			               'bool', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getSpeciesTypeValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = [];
elseif (level == 2)
	if (version == 1)
		valueType = [];
	elseif (version == 2)
		valueType = {
			               'SBML_SPECIES_TYPE', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
		                };
	elseif (version == 3)
		valueType = {
			               'SBML_SPECIES_TYPE', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_SPECIES_TYPE', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = [];
        elseif (version == 2)
		valueType = [];
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getStoichiometryMathValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = [];
elseif (level == 2)
	if (version == 1)
		valueType = [];
	elseif (version == 2)
		valueType = [];
	elseif (version == 3)
		valueType = {
			               'SBML_STOICHIOMETRY_MATH', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_STOICHIOMETRY_MATH', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = [];
        elseif (version == 2)
		valueType = [];
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getTriggerValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = [];
elseif (level == 2)
	if (version == 1)
		valueType = [];
	elseif (version == 2)
		valueType = [];
	elseif (version == 3)
		valueType = {
			               'SBML_TRIGGER', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_TRIGGER', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = {
			               'SBML_TRIGGER', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'bool', ...
			               'bool', ...
			               'char', ...
		                };
        elseif (version == 2)
		valueType = {
			               'SBML_TRIGGER', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'bool', ...
			               'bool', ...
			               'char', ...
                                       'char', ...
                                       'char', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getUnitValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = {
		               'SBML_UNIT', ...
		               'char', ...
		               'char', ...
		               'char', ...
			       'int', ...
			       'int', ...
	                };
elseif (level == 2)
	if (version == 1)
		valueType = {
			               'SBML_UNIT', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'int', ...
			               'double', ...
			               'double', ...
		                };
	elseif (version == 2)
		valueType = {
			               'SBML_UNIT', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'int', ...
			               'double', ...
		                };
	elseif (version == 3)
		valueType = {
			               'SBML_UNIT', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'int', ...
			               'int', ...
			               'double', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_UNIT', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'int', ...
			               'int', ...
			               'double', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = {
			               'SBML_UNIT', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'double', ...
			               'int', ...
			               'double', ...
		                };
        elseif (version == 2)
		valueType = {
			               'SBML_UNIT', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'double', ...
			               'int', ...
			               'double', ...
                                       'char', ...
                                       'char', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [valueType] = getUnitDefinitionValueType(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	valueType = {
		               'SBML_UNIT_DEFINITION', ...
		               'char', ...
		               'char', ...
		               'char', ...
		               'structure', ...
	                };
elseif (level == 2)
	if (version == 1)
		valueType = {
			               'SBML_UNIT_DEFINITION', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'structure', ...
		                };
	elseif (version == 2)
		valueType = {
			               'SBML_UNIT_DEFINITION', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'structure', ...
		                };
	elseif (version == 3)
		valueType = {
			               'SBML_UNIT_DEFINITION', ...
			               'char', ...
			               'char', ...
			               'char', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'structure', ...
		                };
    elseif (version == 4 || version == 5)
		valueType = {
			               'SBML_UNIT_DEFINITION', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'structure', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		valueType = {
			               'SBML_UNIT_DEFINITION', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'structure', ...
		                };
        elseif (version == 2)
		valueType = {
			               'SBML_UNIT_DEFINITION', ...
			               'char', ...
			               'char', ...
			               'char', ...
                          'structure', ...
			               'int', ...
			               'char', ...
			               'char', ...
			               'structure', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getFBCModelValueType(level, ...
                                                         version, pkgVersion)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion == 1)
          SBMLfieldnames = { 'uint', ...
                             'structure', ...
                             'structure', ...
                             'char', ...
                           };
          nNumberFields = 4;
    elseif(pkgVersion ==2)
          SBMLfieldnames = { 'uint', ...
                             'structure', ...
                             'structure', ...
                             'char', ...
                             'bool'
                           };
          nNumberFields = 5;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getFBCSpeciesValueType(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion < 3)
      SBMLfieldnames = { 'int', ...
                         'char', ...
                         'bool', ...
                         'uint', ...
                         'uint', ...
                         'uint', ...
                       };
      nNumberFields = 6;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getFluxBoundValueType(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version <3)
    if (pkgVersion == 1)
      SBMLfieldnames = { 'SBML_FBC_FLUXBOUND', ...
                         'char', ...
                         'char', ...
                         'char', ...
                          'structure', ...
                         'int', ...
                         'char', ...
                         'char', ...
                         'char', ...
                         'double', ...
                         'bool', ...
                         'uint', ...
                         'uint', ...
                         'uint', ...
                       };
      nNumberFields = 14;
    else
        SBMLfieldnames = [];
        nNumberFields = 0;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getFluxObjectiveValueType(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion < 3)
      SBMLfieldnames = { 'SBML_FBC_FLUXOBJECTIVE', ...
                         'char', ...
                         'char', ...
                         'char', ...
                          'structure', ...
                         'int', ...
                         'char', ...
                         'double', ...
                         'bool', ...
                         'uint', ...
                         'uint', ...
                         'uint', ...
                       };
      nNumberFields = 12;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getObjectiveValueType(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion < 3)
      SBMLfieldnames = { 'SBML_FBC_OBJECTIVE', ...
                         'char', ...
                         'char', ...
                         'char', ...
                          'structure', ...
                         'int', ...
                         'char', ...
                         'char', ...
                         'structure', ...
                         'uint', ...
                         'uint', ...
                         'uint', ...
                       };
      nNumberFields = 12;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getGeneProductValueType(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion  == 2)
      SBMLfieldnames = { 'SBML_FBC_GENE_PRODUCT', ...
                         'char', ...
                         'char', ...
                         'char', ...
                          'structure', ...
                         'int', ...
                         'char', ...
                         'char', ...
                         'char', ...
                         'char', ...
                         'uint', ...
                         'uint', ...
                         'uint', ...
                       };
      nNumberFields = 13;
    else
        SBMLfieldnames = [];
        nNumberFields = 0;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getFBCReactionValueType(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion == 2)
      SBMLfieldnames = { 'char', ...
                         'char', ...
                         'structure', ...
                         'uint', ...
                         'uint', ...
                         'uint', ...
                       };
      nNumberFields = 6;
    else
       SBMLfieldnames = [];
	   nNumberFields = 0;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getGeneProductAssociationValueType(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion == 2)

      SBMLfieldnames = { 'SBML_FBC_GENE_PRODUCT_ASSOCIATION', ...
                         'char', ...
                         'char', ...
                         'char', ...
                          'structure', ...
                         'int', ...
                         'char', ...
                         'char', ...
                         'structure', ...
                         'uint', ...
                         'uint', ...
                         'uint', ...
                       };
                   nNumberFields = 12;
    else
       SBMLfieldnames = [];
	   nNumberFields = 0;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getAssociationValueType(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion == 2)

      SBMLfieldnames = { 'SBML_FBC_ASSOCIATION', ...
                         'char', ...
                         'char', ...
                         'char', ...
                          'structure', ...
                         'int', ...
                         'char', ...
                         'uint', ...
                         'uint', ...
                         'uint', ...
                       };
                   nNumberFields = 9;
    else
       SBMLfieldnames = [];
	   nNumberFields = 0;
    end;
	end;
end;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getQUALModelValueType(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end;

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        'uint', ...
        'structure', ...
        'structure', ...
        };
        nNumberFields = 3;
      end;
    end;
  end;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getQualitativeSpeciesValueType(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end;

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        'SBML_QUAL_QUALITATIVE_SPECIES', ...
        'char', ...
        'char', ...
        'char', ...
        'structure', ...
        'int', ...
        'char', ...
        'char', ...
        'char', ...
        'bool', ...
        'uint', ...
        'uint', ...
        'uint', ...
        'uint', ...
        'uint', ...
        };
        nNumberFields = 15;
      end;
    end;
  end;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getTransitionValueType(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end;

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        'SBML_QUAL_TRANSITION', ...
        'char', ...
        'char', ...
        'char', ...
        'structure', ...
        'int', ...
        'char', ...
        'char', ...
        'structure', ...
        'structure', ...
        'structure', ...
        'structure', ...
        'uint', ...
        'uint', ...
        'uint', ...
        };
        nNumberFields = 14;
      end;
    end;
  end;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getInputValueType(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end;

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        'SBML_QUAL_INPUT', ...
        'char', ...
        'char', ...
        'char', ...
        'structure', ...
        'int', ...
        'char', ...
        'char', ...
        'char', ...
        'char', ...
        'char', ...
        'uint', ...
        'uint', ...
        'uint', ...
        'uint', ...
        };
        nNumberFields = 15;
      end;
    end;
  end;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getOutputValueType(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end;

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        'SBML_QUAL_OUTPUT', ...
        'char', ...
        'char', ...
        'char', ...
        'structure', ...
        'int', ...
        'char', ...
        'char', ...
        'char', ...
        'char', ...
        'int', ...
        'uint', ...
        'uint', ...
        'uint', ...
        };
        nNumberFields = 14;
      end;
    end;
  end;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getDefaultTermValueType(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end;

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        'SBML_QUAL_DEFAULT_TERM', ...
        'char', ...
        'char', ...
        'char', ...
        'structure', ...
        'int', ...
        'uint', ...
        'uint', ...
        'uint', ...
        'uint', ...
        };
        nNumberFields = 10;
      end;
    end;
  end;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getFunctionTermValueType(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end;

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        'SBML_QUAL_FUNCTION_TERM', ...
        'char', ...
        'char', ...
        'char', ...
        'structure', ...
        'int', ...
        'uint', ...
        'char', ...
        'uint', ...
        'uint', ...
        'uint', ...
        };
        nNumberFields = 11;
      end;
    end;
  end;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getGROUPSModelValueType(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end;

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        'uint', ...
        'structure', ...
        };
        nNumberFields = 2;
      end;
    end;
  end;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getGroupValueType(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end;

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        'SBML_GROUPS_GROUP', ...
        'char', ...
        'char', ...
        'char', ...
        'structure', ...
        'int', ...
        'char', ...
        'char', ...
        'char', ...
        'structure', ...
        'uint', ...
        'uint', ...
        'uint', ...
        };
        nNumberFields = 13;
      end;
    end;
  end;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getMemberValueType(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end;

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        'SBML_GROUPS_MEMBER', ...
        'char', ...
        'char', ...
        'char', ...
        'structure', ...
        'int', ...
        'char', ...
        'char', ...
        'char', ...
        'char', ...
        'uint', ...
        'uint', ...
        'uint', ...
        };
        nNumberFields = 13;
      end;
    end;
  end;


%%%% ADD functions
%%%% ADD functions
%%%% END functions



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function valid = isValidLevelVersionCombination(level, version)
valid = 1;

if ~isIntegralNumber(level)
	error('level must be an integer');
elseif ~isIntegralNumber(version)
	error('version must be an integer');
end;

if (level < 1 || level > 3)
	error('current SBML levels are 1, 2 or 3');
end;

if (level == 1)
	if (version < 1 || version > 2)
		error('SBMLToolbox supports versions 1-2 of SBML Level 1');
	end;

elseif (level == 2)
	if (version < 1 || version > 5)
		error('SBMLToolbox supports versions 1-5 of SBML Level 2');
	end;

elseif (level == 3)
	if (version < 1 || version > 2)
		error('SBMLToolbox supports version 1-2 of SBML Level 3');
	end;

end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function value = isIntegralNumber(number)
value = 0;

integerClasses = {'int8', 'uint8', 'int16', 'uint16', 'int32', 'uint32', 'int64', 'uint64'};

% since the function isinteger does not exist in MATLAB Rel 13
% this is not used
%if (isinteger(number))
if (ismember(class(number), integerClasses))
    value = 1;
elseif (isnumeric(number))
    % if it is an integer 
    if (number == fix(number))
        value = 1;
    end;
end;



