/*
 * Picviz - Parallel coordinates ploter
 * Copyright (C) 2008 Sebastien Tricaud <toady@gscore.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $Id: filter.c 370 2008-12-22 00:25:13Z toady $
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <picviz.h>

#include "../props/color.h"


static int test_filter_criterion(PicvizImage *image, picviz_filter_criterion_t *filter, PicvizAxisPlot *axisplot)
{
        int ret = 0;
        PcvHeight compare_val;
	char *color = NULL;
	float rf; /* red from filter */
	float gf;
	float bf;
	float rc; /* red from axisplot color */
	float gc;
	float bc;

        if (filter->type == PF_PLOT_FILTER) {
                compare_val = (filter->options == PF_OPTIONS_PLOTPERCENT) ? axisplot->y * 100 / image->height : axisplot->y;

                switch(filter->relation) {
                        case PF_RELATION_ERROR:
                                fprintf(stderr, "ERROR: Filter relation!\n");
                                break;

                        case PF_RELATION_EQUAL:
                                if ( filter->value.plot == compare_val)
                                        ret = 1;
                                break;
                        case PF_RELATION_NOTEQUAL:
                                if ( filter->value.plot != compare_val)
                                        ret = 1;
                                break;
                        case PF_RELATION_GREATER:
                                if (compare_val > filter->value.plot)
                                        ret = 1;
                                break;
                        case PF_RELATION_LESS:
                                if (compare_val < filter->value.plot)
                                        ret = 1;
                                break;
                        case PF_RELATION_LESS_OR_EQUAL:
                                if (compare_val <= filter->value.plot)
                                        ret = 1;
                                break;
                        case PF_RELATION_GREATER_OR_EQUAL:
                                if (compare_val >= filter->value.plot)
                                        ret = 1;
                                break;
                        default:
                                fprintf(stderr, "ERROR: Filter relation!\n");
                                break;
                }
        }
        if (filter->type == PF_VALUE_FILTER) {
                switch(filter->relation) {
                        case PF_RELATION_ERROR:
                                fprintf(stderr, "ERROR: Filter relation!\n");
                                break;

                        case PF_RELATION_EQUAL:
				if (engine.use_pcre) {
					if (picviz_regex_match(axisplot->strval, filter->value.data))
						ret = 1;
				} else {
					if ( !strcmp(filter->value.data, axisplot->strval) )
						ret = 1;
				}
                                break;

                        default:
                                fprintf(stderr, "ERROR: Filter relation!\n");
                                break;
                }
        }

        if (filter->type == PF_FREQ_FILTER) {

		color = picviz_properties_get(axisplot->props, "color");
		if (!color) {
			fprintf(stderr,"[E] Picviz filter cannot extract axis plot color\n");
			return 0;
		}
		rf = picviz_color_extract_r(strdup(filter->value.data));
		gf = picviz_color_extract_g(strdup(filter->value.data));
		rc = picviz_color_extract_r(color);
		gc = picviz_color_extract_g(color);

                switch(filter->relation) {
                        case PF_RELATION_ERROR:
                                fprintf(stderr, "ERROR: Filter relation!\n");
                                break;

                        case PF_RELATION_GREATER_OR_EQUAL:
				if ((rc >= rf) && (rf >= rc))
					ret = 1;
                                break;

                        default:
                                fprintf(stderr, "ERROR: Filter relation!\n");
                                break;
		}
	}
        return ret;
}



static int filter_match(PicvizImage *image, picviz_filter_criterion_t *criterion, PicvizAxisPlot **axisplot, int axis_max)
{
        int ret;

        if ( criterion->axis >= axis_max || ! axisplot[criterion->axis] ) {
                fprintf(stderr, "ERROR: axis '%d' does not exist!.\n", criterion->axis);
                return -1;
        }

        ret = test_filter_criterion(image, criterion, axisplot[criterion->axis]);
        if ( ret < 0 )
                return ret;

        if ( ret == 1 && criterion->and )
                ret = filter_match(image, criterion->and, axisplot, axis_max);

        if ( ret == 0 && criterion->or )
                ret = filter_match(image, criterion->or, axisplot, axis_max);

        if ( ret < 0 )
                return ret;

        return ret;
}



int picviz_filter_display(picviz_filter_t *filter, PicvizImage *image, PicvizAxisPlot **axisplot, int axis_max)
{
        int ret;

        ret = filter_match(image, filter->criterion, axisplot, axis_max);
        if ( ret < 0 )
                return ret;

        if ( (filter->display == PF_SHOW && ret == 0) ||
             (filter->display == PF_HIDE && ret == 1) )
                return 1;
        else
                return 0;
}

PicvizFilter *picviz_filter_new(void)
{
        PicvizFilter *filter;

        filter = malloc(sizeof(PicvizFilter));
        if (!filter) {
                fprintf(stderr,"Cannot allocate a new filter\n");
                return NULL;
        }
        filter->display  = PF_TYPE_ERROR;
        /* Defaults: safer not to disturb the language parser */

        return filter;
}


picviz_filter_criterion_t *picviz_filter_criterion_new(void)
{
        picviz_filter_criterion_t *c;

        c = malloc(sizeof(*c));
        if ( ! c ) {
                fprintf(stderr,"Cannot allocate a new criterion\n");
                return NULL;
        }

        /* Defaults: safer not to disturb the language parser */
        c->type     = PF_POST_LINE_FILTER;
        c->relation = PF_RELATION_ERROR;
        c->options  = PF_OPTIONS_NONE;
        c->and = c->or = NULL;

        return c;
}


picviz_filter_criterion_t *picviz_filter_criterion_clone(picviz_filter_criterion_t *src)
{
        picviz_filter_criterion_t *dst;

        dst = picviz_filter_criterion_new();
        if ( ! dst )
                return NULL;

        memcpy(dst, src, sizeof(*dst));
        if ( src->and )
                dst->and = picviz_filter_criterion_clone(src->and);

        if ( src->or )
                dst->or = picviz_filter_criterion_clone(src->or);

        return dst;
}

picviz_filter_criterion_t *picviz_filter_and_criterion(picviz_filter_criterion_t *c1, picviz_filter_criterion_t *c2)
{
        picviz_filter_criterion_t *cs = c1, *last, *new;

        while ( c1 ) {
                last = c1;

                if ( c1->or ) {
                        new = picviz_filter_criterion_clone(c2);
                        if (! new )
                                return NULL;

                        picviz_filter_and_criterion(c1->or, new);
                }

                c1 = c1->and;
        }

        last->and = c2;
        return cs;
}


picviz_filter_criterion_t *picviz_filter_or_criterion(picviz_filter_criterion_t *c1, picviz_filter_criterion_t *c2)
{
        while ( c1->or )
                c1 = c1->or;

        c1->or = c2;

        return c1;
}



#ifdef _UNIT_TEST_
int main(void)
{
        PicvizFilter *filter;

        filter = picviz_filter_new();
        picviz_filter_set_string(filter, "show only plotmin 5 on axes");

}
#endif

