% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/h5read.R
\name{h5_read}
\alias{h5_read}
\alias{h5read}
\title{Reads and write object in HDF5 files}
\usage{
h5read(
  file,
  name,
  index = NULL,
  start = NULL,
  stride = NULL,
  block = NULL,
  count = NULL,
  compoundAsDataFrame = TRUE,
  callGeneric = TRUE,
  read.attributes = FALSE,
  drop = FALSE,
  ...,
  native = FALSE,
  s3 = FALSE,
  s3credentials = NULL
)
}
\arguments{
\item{file}{The filename (character) of the file in which the dataset is
be located. It is possible to provide an object of
class \linkS4class{H5IdComponent} representing a H5 location identifier
(file or group). See \code{\link{H5Fcreate}}, \code{\link{H5Fopen}},
\code{\link{H5Gcreate}}, \code{\link{H5Gopen}} to create an object of this
kind.}

\item{name}{The name of the dataset in the HDF5 file.}

\item{index}{List of indices for subsetting. The length of the list has to
agree with the dimensional extension of the HDF5 array. Each list element is
an integer vector of indices. A list element equal to NULL choses all
indices in this dimension. Counting is R-style 1-based.}

\item{start}{The start coordinate of a hyperslab (similar to subsetting in
R). Counting is R-style 1-based. This argument is ignored, if index is not
NULL.}

\item{stride}{The stride of the hypercube. Read the introduction
\url{http://ftp.hdfgroup.org/HDF5/Tutor/phypecont.html} before using this
argument. R behaves like Fortran in this example. This argument is ignored,
if index is not NULL.}

\item{block}{The block size of the hyperslab. Read the introduction
\url{http://ftp.hdfgroup.org/HDF5/Tutor/phypecont.html} before using this
argument. R behaves like Fortran in this example. This argument is ignored,
if index is not NULL.}

\item{count}{The number of blocks to be read. This argument is ignored,
if index is not NULL.}

\item{compoundAsDataFrame}{If true, a compound datatype will be coerced to a
data.frame. This is not possible, if the dataset is multi-dimensional.
Otherwise the compound datatype will be returned as a list. Nested compound
data types will be returned as a nested list.}

\item{callGeneric}{If TRUE a generic function h5read.classname will be
called if it exists depending on the dataset's class attribute within the
HDF5 file. This function can be used to convert the standard output of
h5read depending on the class attribute. Note that h5read is not a S3
generic function. Dispatching is done based on the HDF5 attribute after the
standard h5read function.}

\item{read.attributes}{(logical) If \code{TRUE}, the HDF5 attributes are read and
attached to the respective R object.}

\item{drop}{(logical) If TRUE, the HDF5 object is read as a vector with \code{NULL}
dim attributes.}

\item{\dots}{Further arguments passed to \code{\link{H5Dread}}.}

\item{native}{An object of class \code{logical}. If TRUE, array-like objects
are treated as stored in HDF5 row-major rather than R column-major
orientation. Using \code{native = TRUE} increases HDF5 file portability
between programming languages. A file written with \code{native = TRUE}
should also be read with \code{native = TRUE}}

\item{s3}{Logical value indicating whether the file argument should be
treated as a URL to an Amazon S3 bucket, rather than a local file path.}

\item{s3credentials}{A list of length three, providing the credentials for
accessing files in a private Amazon S3 bucket.}
}
\value{
\code{h5read} returns an array with the data read.
}
\description{
Reads objects in HDF5 files. This function can be used to read
either full arrays/vectors or subarrays (hyperslabs) from an
existing dataset.
}
\details{
Read an R object from an HDF5 file. If none of the arguments
\code{start, stride, block, count} are specified, the dataset has the same
dimension in the HDF5 file and in memory. If the dataset already exists in
the HDF5 file, one can read subarrays, so called hyperslabs from
the HDF5 file. The arguments \code{start, stride, block, count} define the
subset of the dataset in the HDF5 file that is to be read/written. See these
introductions to hyperslabs:
\url{https://support.hdfgroup.org/HDF5/Tutor/selectsimple.html},
\url{https://support.hdfgroup.org/HDF5/Tutor/select.html} and
\url{http://ftp.hdfgroup.org/HDF5/Tutor/phypecont.html}. Please note that in
R the first dimension is the fastest changing dimension.

When viewing the HDF5 datasets with any C-program (e.g. HDFView), the order
of dimensions is inverted. In the R interface counting starts with 1,
whereas in the C-programs (e.g. HDFView) counting starts with 0.
}
\examples{

h5createFile("ex_hdf5file.h5")

# write a matrix
B = array(seq(0.1,2.0,by=0.1),dim=c(5,2,2))
h5write(B, "ex_hdf5file.h5","B")

# read a matrix
E = h5read("ex_hdf5file.h5","B")

# write and read submatrix
h5createDataset("ex_hdf5file.h5", "S", c(5,8), storage.mode = "integer", chunk=c(5,1), level=7)
h5write(matrix(1:5,nr=5,nc=1), file="ex_hdf5file.h5", name="S", index=list(NULL,1))
h5read("ex_hdf5file.h5", "S")
h5read("ex_hdf5file.h5", "S", index=list(NULL,2:3))

# Read a subset of an hdf5 file in a public S3 bucket
\donttest{
h5read('https://rhdf5-public.s3.eu-central-1.amazonaws.com/rhdf5ex_t_float_3d.h5', 
      s3 = TRUE, name = "a1", index = list(NULL, 3, NULL))
}

}
\seealso{
\code{\link{h5ls}}
}
\author{
Bernd Fischer, Mike Smith
}
