#pragma once

/*******************************************************************************

In a PrimaryGraph, each vertex represents a primary edge of the marker graph.
Edges are generated by following the reads.

*******************************************************************************/

// Shasta.
#include "Base.hpp"
#include "MarkerGraphEdgePairInfo.hpp"
#include "MultithreadedObject.hpp"
#include "ReadId.hpp"
#include "shastaTypes.hpp"

// Boost libraries.
#include <boost/graph/adjacency_list.hpp>

// Standard library.
#include "iosfwd.hpp"
#include "memory.hpp"
#include "string.hpp"
#include "utility.hpp"
#include "vector.hpp"

namespace shasta {
    class Assembler;
    class MarkerGraph;
    namespace mode3 {

        // A connected component of the primary graph,
        // in which each vertex represents a primary edge of the marker graph.
        // Edges are created by following the reads on their journeys
        // over primary marker graph edges.
        class PrimaryGraphVertex;
        class PrimaryGraphEdge;
        class PrimaryGraph;
        using PrimaryGraphBaseClass = boost::adjacency_list<
            boost::listS,
            boost::vecS,
            boost::bidirectionalS,
            PrimaryGraphVertex,
            PrimaryGraphEdge>;

        class PrimaryGraphDisplayOptions;

    }
}



// Class to control Graphviz output of PrimaryGraph.
class shasta::mode3::PrimaryGraphDisplayOptions {
public:
    bool labels = true;
    bool tooltips = true;
    bool colorVertices = true;
    bool colorEdges = true;
    bool showNonTransitiveReductionEdges = true;

    // Thresholds for coloring by corrected Jaccard similarity J'.
    // If J' <= redJ, the edge is drawn red.
    // If J' >= greenJ, the edge is drawn green.
    // For values in between, the color is interpolated.
    double redJ;
    double greenJ;

    PrimaryGraphDisplayOptions(double redJ = 0., double greenJ = 1.) :
        redJ(redJ), greenJ(greenJ) {}

    void makeCompact()
    {
        labels = false;
        tooltips = false;
        colorVertices = false;
        colorEdges = false;
    }
};



class shasta::mode3::PrimaryGraphVertex {
public:

    // The corresponding marker graph edgeId.
    MarkerGraphEdgeId edgeId;
};



class shasta::mode3::PrimaryGraphEdge {
public:
    MarkerGraphEdgePairInfo info;
    uint64_t coverage;
    bool isNonTransitiveReductionEdge = false;
};



class shasta::mode3::PrimaryGraph : public PrimaryGraphBaseClass {
public:

    std::map<MarkerGraphEdgeId, vertex_descriptor> vertexMap;
    vertex_descriptor addVertex(MarkerGraphEdgeId);

    void addEdge(
        MarkerGraphEdgeId,
        MarkerGraphEdgeId,
        const MarkerGraphEdgePairInfo&,
        uint64_t coverage);
    void addEdgeFromVertexDescriptors(
        vertex_descriptor,
        vertex_descriptor,
        const MarkerGraphEdgePairInfo&,
        uint64_t coverage);

    void writeGraphviz(
        const string& name,
        const PrimaryGraphDisplayOptions&,
        const MarkerGraph&) const;

    void writeEdgeCoverageHistogram(const string& fileName) const;

    // Create the connected components of this PrimaryGraph,
    // without changing the PrimaryGraph itself.
    vector< shared_ptr<PrimaryGraph> > createConnectedComponents(uint64_t minComponentSize) const;

    void localTransitiveReduction(
        uint64_t distance,
        uint64_t maxCoverage);

    // Remove cross-edges.
    // This removes an edge v0->v1 if the following are all true:
    // - Its coverage is at most lowCoverageThreshold.
    // - Its estimated offset is at least minOffset.
    // - v0 has at least one out-edge with coverage at least highCoverageThreshold.
    // - v1 has at least one in-edge with coverage at least highCoverageThreshold.
    void removeCrossEdges(
        uint64_t lowCoverageThreshold,
        uint64_t highCoverageThreshold,
        uint64_t minOffset);

    // Remove edges for which loss = (commonCount - coverage) / commonCount > maxLoss
    void removeWeakEdges(double maxLoss);

};

