/*===========================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*
*/

#include <klib/extern.h>
#include <klib/text.h>
#include <klib/printf.h>
#include <klib/writer.h>
#include <klib/symbol.h>
#include <klib/time.h>
#include <klib/data-buffer.h>
#include <klib/rc.h>

#include "writer-priv.h"

#include <os-native.h>
#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include <assert.h>

#define USE_LIB_FLOAT 1

static
size_t string_flush ( char *dst, size_t bsize, const KWrtHandler *flush, rc_t *rc, size_t *total )
{
    size_t num_writ, flushed;
    for ( flushed = 0; flushed < bsize; flushed += num_writ )
    {
        * rc = ( * flush -> writer ) ( flush -> data, dst, bsize, & num_writ );
        if ( * rc != 0 )
            break;
        if ( num_writ == 0 )
            break;
    }

    if ( * rc == 0 )
    {
        * total += flushed;
        if ( flushed < bsize )
            memmove ( dst, dst + flushed, bsize - flushed );
    }

    return bsize - flushed;
}

static
rc_t CC string_flush_printf ( char *dst, size_t bsize,
    const KWrtHandler *flush, size_t *num_writ, const char *fmt, ... );

static
rc_t string_flush_vprintf ( char *dst, size_t bsize, const KWrtHandler *flush,
    size_t *num_writ, const char *fmt, va_list args )
{
    rc_t rc;
    size_t sidx, didx, total, sz;

    uint32_t *np;
    uint32_t u32;
    int64_t i64;
    uint64_t u64;
    double f64;
#if ! USE_LIB_FLOAT
    uint64_t frac;
    double ftmp;
    uint32_t exponent;
#endif
    const char *cp, *end;
    const String *str;
    const KSymbol *sym;
    const KTime *tm;

    char buff [ 256 ];
    char use_sign, padding;
    const char *hex_digits;
    uint32_t i, j, len, min_field_width, max_field_width, precision;
    bool left_align, comma_separate, have_precision, byte_size, half_size, long_size;
    bool alternate, trim_trailing_zeros, date_time_zone;

    if ( fmt == NULL )
    {
        rc = RC ( rcText, rcString, rcFormatting, rcParam, rcNull );
        if ( num_writ != NULL )
            * num_writ = 0;
        return rc;
    }

    if ( fmt [ 0 ] == 0 )
    {
        rc = RC ( rcText, rcString, rcFormatting, rcParam, rcEmpty );
        if ( num_writ != NULL )
            * num_writ = 0;
        return rc;
    }

    if ( dst == NULL )
    {
        flush = NULL;
        bsize = 0;
    }

    for ( rc = 0, sidx = didx = total = 0; fmt [ sidx ] != 0 && rc == 0; ++ sidx )
    {
        /* flush buffer */
        if ( didx == bsize && flush != NULL )
        {
            didx = string_flush ( dst, didx, flush, & rc, & total );
            if ( rc != 0 )
                break;
        }

        /* copy until ending NUL or '%' */
        if ( fmt [ sidx ] != '%' )
        {
            if ( didx < bsize )
                dst [ didx ] = fmt [ sidx ];
            ++ didx;
            continue;
        }

        /* process escaped '%' */
        if ( fmt [ ++ sidx ] == '%' )
        {
            if ( didx < bsize )
                dst [ didx ] = '%';
            ++ didx;
            continue;
        }

        /* process flags */
        use_sign = 0;
        left_align = comma_separate = alternate = false;
        padding = ' ';

        while ( 1 )
        {
            switch ( fmt [ sidx ] )
            {
            case '-':
                left_align = true;
                padding = ' ';
                ++ sidx;
                continue;
            case '+':
                use_sign = '+';
                ++ sidx;
                continue;
            case ' ':
                if ( use_sign != '+' )
                    use_sign = ' ';
                ++ sidx;
                continue;
            case '0':
                if ( ! left_align )
                    padding = '0';
                ++ sidx;
                continue;
            case ',':
            case '\'':
                comma_separate = true;
                ++ sidx;
                continue;
            case '#':
                alternate = true;
                ++ sidx;
                continue;
            }

            break;
        }

        /* field width */
        min_field_width = 0;
        if ( fmt [ sidx ] == '*' )
        {
            min_field_width = va_arg ( args, unsigned int );
            ++ sidx;
        }
        else while ( isdigit ( fmt [ sidx ] ) )
        {
            min_field_width *= 10;
            min_field_width += fmt [ sidx ++ ] - '0';
        }

        /* precision */
        precision = 0;
        have_precision = false;
        if ( fmt [ sidx ] == '.' )
        {
            padding = ' ';
            have_precision = true;
            if ( fmt [ ++ sidx ] == '*' ) {
                precision = va_arg ( args, unsigned int );
                sidx++;
            } else for ( ; isdigit ( fmt [ sidx ] ); ++ sidx )
            {
                precision *= 10;
                precision += fmt [ sidx ] - '0';
            }
        }

        /* storage size */
        byte_size = half_size = long_size = date_time_zone = false;
        switch ( fmt [ sidx ] )
        {
        case 'h':
            /* this should not be necessary */
            if ( fmt [  ++ sidx ] != 'h' )
                half_size = true;
            else
            {
                byte_size = true;
                ++ sidx;
            }
            break;
        case 'l':
            long_size = true;
            ++ sidx;
            break;
        case 'z':
            date_time_zone = true;
            if ( sizeof ( size_t ) == sizeof ( uint64_t ) )
                long_size = true;
            ++ sidx;
            break;
        }

        /* format */
        trim_trailing_zeros = false;
        switch ( fmt [  sidx  ] )
        {
        case 'd':
        case 'i':

            /* retrieve argument as signed integer */
            if ( byte_size )
                i64 = ( int8_t ) va_arg ( args, int );
            else if ( half_size )
                i64 = ( int16_t ) va_arg ( args, int );
            else if ( long_size )
                i64 = va_arg ( args, int64_t );
            else
                i64 = va_arg ( args, int32_t );

            /* detect zero */
            if ( i64 == 0 )
                use_sign = 0;

            /* detect negative */
            else if ( i64 < 0 )
            {
                use_sign = '-';
                i64 = - i64;
            }

            i = sizeof buff;

#if ! USE_LIB_FLOAT
        make_signed_integer:
#endif
            /* convert to numeral */
            if ( comma_separate ) for ( -- i, j = 0; ; -- i )
            {
                buff [ i ] =  ( i64 % 10 ) + '0';
                if ( ( i64 /= 10 ) == 0 )
                    break;
                if ( ++ j == 3 )
                {
                    buff [ -- i ] = ',';
                    j = 0;
                }
            }
            else for ( -- i; ; -- i )
            {
                buff [ i ] =  ( i64 % 10 ) + '0';
                if ( ( i64 /= 10 ) == 0 )
                    break;
            }

        insert_integer:

            /* the actual length */
            len = sizeof buff - i;

            /* test for buffer flush */
            if ( flush != NULL && didx < bsize )
            {
                max_field_width = len;
                if ( len < precision )
                    max_field_width = precision;
                max_field_width += ( use_sign != 0 );
                if ( max_field_width < min_field_width )
                    max_field_width = min_field_width;
                if ( didx + max_field_width > bsize )
                {
                    didx = string_flush ( dst, didx, flush, & rc, & total );
                    if ( rc != 0 )
                        break;
                }
            }

            /* insert left-aligned */
            if ( left_align )
            {
                /* sign */
                if ( use_sign != 0 )
                {
                    if ( didx < bsize )
                        dst [ didx ] = use_sign;
                    ++ didx;
                }

                /* precision */
                for ( ; len < precision; ++ didx, ++ len )
                {
                    if ( didx < bsize )
                        dst [ didx ] = '0';
                }

                /* value */
                for ( ; i < sizeof buff; ++ didx, ++ i )
                {
                    if ( didx < bsize )
                        dst [ didx ] = buff [ i ];
                }

                /* padding */
                for ( len += ( use_sign != 0 ); len < min_field_width; ++ didx, ++ len )
                {
                    if ( didx < bsize )
                        dst [ didx ] = ' ';
                }
            }
            /* insert right-aligned */
            else
            {
                /* zero padding means issue sign first */
                if ( use_sign != 0 && padding == '0' )
                {
                    if ( didx < bsize )
                        dst [ didx ] = use_sign;
                    ++ didx;
                    ++ len;
                }

                /* precision from length */
                if ( precision < len )
                    precision = len;

                /* apply padding */
                for ( ; precision < min_field_width; ++ didx, -- min_field_width )
                {
                    if ( didx < bsize )
                        dst [ didx ] = padding;
                }

                /* sign */
                if ( use_sign != 0 && padding != '0' )
                {
                    if ( didx < bsize )
                        dst [ didx ] = use_sign;
                    ++ didx;
                }

                /* precision */
                for ( ; len < precision; ++ didx, ++ len )
                {
                    if ( didx < bsize )
                        dst [ didx ] = '0';
                }

                /* value */
                for ( ; i < sizeof buff; ++ didx, ++ i )
                {
                    if ( didx < bsize )
                        dst [ didx ] = buff [ i ];
                }
            }

            break;

        case 'u':

            /* retrieve argument as unsigned integer */
            if ( byte_size )
                u64 = ( uint8_t ) va_arg ( args, int );
            else if ( half_size )
                u64 = ( uint16_t ) va_arg ( args, int );
            else if ( long_size )
                u64 = va_arg ( args, uint64_t );
            else
                u64 = va_arg ( args, uint32_t );

            /* no sign */
            use_sign = 0;

            /* convert to numeral */
            for ( i = sizeof buff - 1; ; -- i )
            {
                buff [ i ] =  ( u64 % 10 ) + '0';
                if ( ( u64 /= 10 ) == 0 )
                    break;
            }

            goto insert_integer;

        case 'p':

            /* retrieve argument as unsigned integer */
            if ( sizeof ( void* ) == sizeof ( uint32_t ) )
                u64 = va_arg ( args, uint32_t );
            else
                u64 = va_arg ( args, uint64_t );

            goto make_hex_integer;

        case 'x':
        case 'X':

            /* retrieve argument as unsigned integer */
            if ( byte_size )
                u64 = ( uint8_t ) va_arg ( args, int );
            else if ( half_size )
                u64 = ( uint16_t ) va_arg ( args, int );
            else if ( long_size )
                u64 = va_arg ( args, uint64_t );
            else
                u64 = va_arg ( args, uint32_t );

        make_hex_integer:

            /* choose numeric case */
            hex_digits = ( fmt [ sidx ] == 'x' ) ?
                "0123456789abcdefx" : "0123456789ABCDEFX";

            /* no sign */
            use_sign = 0;

            /* convert to numeral */
            for ( i = sizeof buff - 1; ; -- i )
            {
                buff [ i ] =  hex_digits [ u64 & 15 ];
                if ( ( u64 >>= 4 ) == 0 )
                    break;
            }

            if ( alternate )
            {
                buff [ -- i ] = hex_digits [ 16 ];
                buff [ -- i ] = '0';
            }

            goto insert_integer;

        case 'o':

            /* retrieve argument as unsigned integer */
            if ( byte_size )
                u64 = ( uint8_t ) va_arg ( args, int );
            else if ( half_size )
                u64 = ( uint16_t ) va_arg ( args, int );
            else if ( long_size )
                u64 = va_arg ( args, uint64_t );
            else
                u64 = va_arg ( args, uint32_t );

            /* no sign */
            use_sign = 0;

            /* convert to numeral */
            for ( i = sizeof buff - 1; ; -- i )
            {
                buff [ i ] =  ( u64 & 7 ) + '0';
                if ( ( u64 >>= 3 ) == 0 )
                    break;
            }

            if ( alternate )
                buff [ -- i ] = '0';

            goto insert_integer;

        case 'b':

            /* retrieve argument as unsigned integer */
            if ( byte_size )
                u64 = ( uint8_t ) va_arg ( args, int );
            else if ( half_size )
                u64 = ( uint16_t ) va_arg ( args, int );
            else if ( long_size )
                u64 = va_arg ( args, uint64_t );
            else
                u64 = va_arg ( args, uint32_t );

            /* no sign */
            use_sign = 0;

            /* convert to numeral */
            for ( i = sizeof buff - 1; ; -- i )
            {
                buff [ i ] =  ( u64 & 1 ) + '0';
                if ( ( u64 >>= 1 ) == 0 )
                    break;
            }

            if ( alternate )
            {
                buff [ -- i ] = 'b';
                buff [ -- i ] = '0';
            }

            goto insert_integer;

#if USE_LIB_FLOAT
        case 'g':
        case 'e':
        case 'f':
        {
            char subfmt [ 16 ];
            char * psubfmt = subfmt;

            *psubfmt++ = '%';

            if (alternate)
                *psubfmt++ = '#';

            if (use_sign)
                *psubfmt++ = use_sign;

            if (left_align)
                *psubfmt++ = '-';
            else if (padding == '0')
                *psubfmt++ = '0';

            *psubfmt++ = '*';
            *psubfmt++ = '.';
            *psubfmt++ = '*';
            *psubfmt++ = fmt [  sidx  ];
            *psubfmt = '\0';
            
            /* retrieve argument as double or long double */
        
            if ( long_size )
                f64 = ( double ) va_arg ( args, long double );
            else
                f64 = va_arg ( args, double );

            if ( ! have_precision )
                precision = 6;
            else if ( precision > 20 )
                precision = 20;

            i = snprintf (buff, sizeof buff, subfmt, min_field_width, precision, f64);

            if ( i >= sizeof buff )
            {
                i = sizeof buff - 1;
                buff [ i ] = '\0';
            }
            min_field_width = 0;
            have_precision = false;

            cp = buff;
            goto make_nul_term_string;
        }
#else
        case 'g':
            trim_trailing_zeros = true;
        case 'e':

#define HANDLE_NAN() \
            switch (fpclassify (f64))                   \
            {                                           \
            case FP_NAN:                                \
                cp = "nan";                             \
                goto make_nul_term_string;              \
            case FP_INFINITE:                           \
                cp = (f64 < 0) ? "-inf" : "inf";        \
                goto make_nul_term_string;              \
            }


            /* retrieve argument as double or long double */
        
            if ( long_size )
                f64 = ( double ) va_arg ( args, long double );
            else
                f64 = va_arg ( args, double );

            HANDLE_NAN();

            if ( f64 < 0 )
            {
                use_sign = '-';
                f64 = - f64;
            }

            exponent = 0;
            buff [ sizeof buff - 3 ] = '+';
            buff [ sizeof buff - 4 ] = 'e';

            i = len = sizeof buff;

            ftmp = f64;

            if ( f64 >= 10.0 )
            {
                for ( ftmp = f64 / 10, exponent = 1; ftmp >= 10.0; ++ exponent )
                    ftmp /= 10;

                if ( exponent < precision && fmt [ sidx ] == 'g' )
                    goto make_normal_float;
            }
            else if ( f64 < 1.0 && f64 > 0.0 )
            {
                buff [ sizeof buff - 3 ] = '-';
                for ( ftmp = f64 * 10, exponent = 1; ftmp < 1.0; ++ exponent )
                    ftmp *= 10;

                if ( exponent <= 4 && fmt [ sidx ] == 'g' )
                    goto make_normal_float;
            }
            else if ( fmt [ sidx ] == 'g' )
            {
                goto make_normal_float;
            }

            /* just for safety */
            exponent %= 100;

            trim_trailing_zeros = false;

            f64 = ftmp;

            buff [ sizeof buff - 1 ] = ( exponent % 10 ) + '0';
            buff [ sizeof buff - 2 ] = ( exponent / 10 ) + '0';

            i = len = sizeof buff - 4;
            goto make_normal_float;

        case 'f':

            /* retrieve argument as double or long double */
            if ( long_size )
                f64 = ( double ) va_arg ( args, long double );
            else
                f64 = va_arg ( args, double );

            HANDLE_NAN();

            if ( f64 < 0 )
            {
                use_sign = '-';
                f64 = - f64;
            }

            i = len = sizeof buff;

        make_normal_float:

            if ( ! have_precision )
                precision = 6;
            else if ( precision > 20 )
                precision = 20;

            ftemp = 0.5;
            for (j = 0; j < precision ; ++j)
                ftemp /= 10;

            f64 += ftemp;


            /* save off integral portion */
            i64 = ( int64_t ) f64;

            /* convert to fraction */
            f64 = f64 - i64;

            /* promote by precision */
            for ( j = 0; j < precision; ++ j )
                f64 *= 10;

            for ( frac = ( uint64_t ) f64; i + precision > len; frac /= 10 )
                buff [ -- i ] = frac % 10 + '0';

            if ( trim_trailing_zeros )
            {
                for ( j = len; -- j >= i; -- precision )
                {
                    if ( buff [ j ] != '0' )
                        break;
                }

                memmove ( & buff [ len - precision ], & buff [ i ], precision );
                i = len - precision;
            }

            if ( precision != 0 || alternate )
                buff [ -- i ] = '.';

            goto make_signed_integer;
#endif
        case 'c':

            /* retrieve as an int */
            u32 = va_arg ( args, unsigned int );
            if ( u32 < 128 )
                buff [ i = sizeof buff - 1 ] = ( char ) u32;
            else
            {
                int dbytes = utf32_utf8 ( buff, & buff [ sizeof buff ], u32 );
                if ( dbytes <= 0 )
                    buff [ i = sizeof buff - 1 ] = '?';
                else
                    memmove ( & buff [ i = sizeof buff - dbytes ], buff, dbytes );
            }

            /* precision makes no sense, but no error */
            precision = 0;

            /* sign makes no sense */
            use_sign = 0;

            /* padding is always with spaces */
            padding = ' ';

            /* other than that, we can treat it as an integer */
            goto insert_integer;

        case 's':

            /* retrieve as a NUL terminated ( or precision limited ) string */
            cp = va_arg ( args, const char* );
            if ( cp == NULL )
                cp = "NULL";

        make_nul_term_string:

            /* in our case, no precision field means unlimited */
            if ( ! have_precision )
                precision = -1;

            /* test for buffer flush */
            if ( flush != NULL && didx < bsize )
            {
                max_field_width = 0;
                if ( have_precision )
                    max_field_width = precision;
                if ( max_field_width < min_field_width )
                    max_field_width = min_field_width;
                if ( didx + max_field_width > bsize )
                {
                    didx = string_flush ( dst, didx, flush, & rc, & total );
                    if ( rc != 0 )
                        break;
                }
            }

            /* if right aligning with a minimum field width, measure string */
            if ( ! left_align && min_field_width != 0 )
            {
                for ( len = 0; cp [ len ] != 0 && len < precision; ++ len )
                    ( void ) 0;

                for ( ; len < min_field_width; ++ didx, ++ len )
                {
                    if ( didx < bsize )
                        dst [ didx ] = ' ';
                }
            }

            /* copy string */
            for ( i = 0; i < precision && cp [ i ] != 0; ++ didx, ++ i )
            {
                if ( flush != NULL && didx == bsize )
                {
                    didx = string_flush ( dst, didx, flush, & rc, & total );
                    if ( rc != 0 )
                        break;
                }

                if ( didx < bsize )
                    dst [ didx ] = cp [ i ];
            }

            if ( rc != 0 )
                break;

            /* apply right padding */
            if ( left_align ) for ( ; i < min_field_width; ++ didx, ++ i )
            {
                if ( flush != NULL && didx == bsize )
                {
                    didx = string_flush ( dst, didx, flush, & rc, & total );
                    if ( rc != 0 )
                        break;
                }

                if ( didx < bsize )
                    dst [ didx ] = ' ';
            }
            break;

            /* String object */
        case 'S':

            /* retrieve as a NUL terminated ( or precision limited ) string */
            str = va_arg ( args, const String* );
            if ( str == NULL )
            {
                cp = "NULL";
                goto make_nul_term_string;
            }

        make_String:

            /* in our case, no precision field means unlimited */
            if ( ! have_precision )
                precision = -1;

            /* test for buffer flush */
            if ( flush != NULL && didx < bsize )
            {
                /* buffer is measured in bytes, while printing
                   widths are measured in characters... */
                max_field_width = ( uint32_t ) str -> size;
                if ( str -> len < min_field_width )
                    max_field_width += min_field_width - str -> len;
                if ( didx + max_field_width > bsize )
                {
                    didx = string_flush ( dst, didx, flush, & rc, & total );
                    if ( rc != 0 )
                        break;
                }
            }

            /* if right aligning with a minimum field width, measure string */
            if ( ! left_align && min_field_width != 0 )
            {
                len = str -> len;
                if ( len > precision )
                    len = precision;

                for ( ; len < min_field_width; ++ didx, ++ len )
                {
                    if ( didx < bsize )
                        dst [ didx ] = ' ';
                }
            }

            cp = str -> addr;
            end = cp + str -> size;

            /* copy string */
            for ( i = 0; i < str -> len && i < precision; ++ i )
            {
                uint32_t ch;
                int sbytes = utf8_utf32 ( & ch, cp, end );
                if ( sbytes <= 0 )
                {
                    if ( sbytes == 0 )
                        rc = RC ( rcText, rcString, rcFormatting, rcData, rcInsufficient );
                    else
                        rc = RC ( rcText, rcString, rcFormatting, rcData, rcCorrupt );
                    break;
                }
                cp += sbytes;

                if ( didx < bsize )
                {
                    int dbytes = utf32_utf8 ( dst + didx, dst + bsize, ch );
                    if ( dbytes > 0 )
                    {
                        didx += dbytes;
                        continue;
                    }
                    if ( dbytes < 0 )
                    {
                        rc = RC ( rcText, rcString, rcFormatting, rcData, rcCorrupt );
                        break;
                    }
                }

                didx += sbytes;
            }

            /* apply right padding */
            if ( left_align ) for ( ; i < min_field_width; ++ didx, ++ i )
            {
                if ( didx < bsize )
                    dst [ didx ] = ' ';
            }
            break;

            /* version number */
        case 'V':

            u32 = va_arg ( args, uint32_t );

            if ( ! have_precision )
            {
                if ( ( u32 & 0xFFFF ) != 0 )
                    precision = 3;
                else if ( ( u32 & 0xFF0000 ) != 0 )
                    precision = 2;
                else
                    precision = 1;
            }

            switch ( precision )
            {
            case 0:
                cp = ""; break;
            case 1:
                cp ="%u"; break;
            case 2:
                cp ="%u.%u"; break;
            default:
                cp ="%u.%u.%u";
            }

            have_precision = false;
            precision = 0;

            rc = string_printf ( buff, sizeof buff, & sz,
                                 cp,
                                 VersionGetMajor ( u32 ),
                                 VersionGetMinor ( u32 ),
                                 VersionGetRelease ( u32 ) );
            if ( rc != 0 )
                break;

            use_sign = 0;
            padding = ' ';
            memmove ( & buff [ i = ( uint32_t ) ( sizeof buff - sz ) ], buff, sz );
            goto insert_integer;

        case 'R':

            rc = va_arg ( args, rc_t );
            sz = KWrtFmt_rc_t ( buff, sizeof buff, alternate ? "#" : "", rc );
            rc = 0; /* reset back to ok */
            assert ( sz < sizeof buff );

            use_sign = 0;
            padding = ' ';
            memmove ( & buff [ i = ( uint32_t ) ( sizeof buff - sz ) ], buff, sz );
            goto insert_integer;

        case 'N':

            /* THIS IS WRONG - FIELD WIDTH AND FRIENDS WILL NOT BE USED */
            sym = va_arg ( args, const KSymbol* );

            if ( sym -> dad != NULL )
            {
                if ( flush != NULL )
                    didx = string_flush ( dst, didx, flush, & rc, & total );

                sz = 0;
                rc = string_flush_printf ( & dst [ didx ], bsize - didx, flush, & sz, "%N:", sym -> dad );
                if ( rc != 0 )
                    break;

                didx += ( uint32_t ) sz;
                if ( flush != NULL )
                    didx = 0;
            }

            str = & sym -> name;
            goto make_String;

        case 'T': /* KTime */

            tm = va_arg ( args, const KTime* );

            sz = 0;

            /* LEGEND
             *  modifier 'h' means do date only
             *  modifier 'l' means date and time
             *  modifier 'z' means date, time and timezone
             *  no modifier means time
             *  precision affects time
             *  leading zero affects time
             */
            if ( date_time_zone || long_size || half_size )
            {
                static char const *months [ 12 ] =
                    { "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
                static char const *weekdays [ 7 ] =
                    { "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };
                rc = string_printf ( buff, sizeof buff, & sz, "%s %s %u %u"
                                     , weekdays [ tm -> weekday ]
                                     , months [ tm -> month ]
                                     , tm -> day + 1
                                     , tm -> year
                    );
                if ( rc != 0 )
                    break;
            }

            if ( ! half_size )
            {
                if ( sz != 0 )
                    buff [ sz ++ ] = ' ';

                u64 = sz;

                rc = string_printf ( & buff [ sz ], sizeof buff - sz, & sz,
                                     padding == '0' ? "%02u:%02u:%02u %cM" : "%u:%02u:%02u %cM"
                                     , ( tm -> hour + 11 ) % 12 + 1
                                     , tm -> minute
                                     , tm -> second
                                     , ( tm -> hour < 12 ) ? 'A' : 'P'
                    );
                if ( rc != 0 )
                    break;

                u64 += sz;

                if ( date_time_zone )
                {
                    rc = string_printf ( & buff [ u64 ], sizeof buff - ( size_t ) u64, & sz,
                                         " %+02d", tm -> tzoff / 60 );
                    if ( rc != 0 )
                        break;

                    u64 += sz;
                }

                sz = ( size_t ) u64;
            }

            padding = ' ';
            memmove ( & buff [ i = ( uint32_t ) ( sizeof buff - sz ) ], buff, sz );
            cp = buff;
            goto make_nul_term_string;

        case '!': /* operating system error code: e.g. errno or GetLastError() */
            u32 = va_arg ( args, uint32_t ); /* fetching it unsigned but it can be signed */

            sz = KWrtFmt_error_code ( buff, sizeof buff - 1, u32 );
            assert ( sz < sizeof buff );
            buff [ sz ] = '\0';

            padding  = ' ';
            memmove ( & buff [ i = ( uint32_t ) ( sizeof buff - sz ) ], buff, sz );
            cp = buff;
            goto make_nul_term_string;

        case 'n':

            np = va_arg ( args, uint32_t* );
            if ( np != NULL )
                * np = ( uint32_t ) ( total + didx );
            break;

        case 0:
            -- sidx;
            break;

        default:
            rc = RC ( rcText, rcString, rcFormatting, rcFormat, rcUnrecognized );
        }
    }

    if ( num_writ != NULL )
        * num_writ = total + didx;

    if ( flush != NULL )
    {
        if ( rc == 0 && didx != 0 )
        {
            didx = string_flush ( dst, didx, flush, & rc, & total );
            if ( rc == 0 && didx != 0 )
                rc = RC ( rcRuntime, rcString, rcFormatting, rcTransfer, rcIncomplete );
        }
    }
    else if ( didx < bsize )
        dst [ didx ] = 0;
    else if ( rc == 0 )
        rc = RC ( rcText, rcString, rcFormatting, rcBuffer, rcInsufficient );

    return rc;
}

static
rc_t CC string_flush_printf ( char *dst, size_t bsize,
    const KWrtHandler *flush, size_t *num_writ, const char *fmt, ... )
{
    rc_t rc;

    va_list args;
    va_start ( args, fmt );

    rc = string_flush_vprintf ( dst, bsize, flush, num_writ, fmt, args );

    va_end ( args );

    return rc;
}

LIB_EXPORT rc_t CC string_vprintf ( char *dst, size_t bsize,
    size_t *num_writ, const char *fmt, va_list args )
{
    return string_flush_vprintf ( dst, bsize, NULL, num_writ, fmt, args );
}

LIB_EXPORT rc_t CC string_printf ( char *dst, size_t bsize, size_t *num_writ, const char *fmt, ... )
{
    rc_t rc;

    va_list args;
    va_start ( args, fmt );

    rc = string_flush_vprintf ( dst, bsize, NULL, num_writ, fmt, args );

    va_end ( args );

    return rc;
}

LIB_EXPORT size_t CC vkfprintf ( const KWrtHandler *out,
    size_t *num_writ, const char * fmt, va_list args )
{
    rc_t rc;

    if ( out == NULL )
    {
        rc = RC ( rcRuntime, rcString, rcFormatting, rcFile, rcNull );
        if ( num_writ != NULL )
            * num_writ = 0;
    }
    else
    {
        char buff [ 4096 ];
        rc = string_flush_vprintf ( buff, sizeof buff, out, num_writ, fmt, args );
        if ( rc != 0 )
            rc = ResetRCContext ( rc, rcRuntime, rcString, rcFormatting );
    }

    return rc;
}

LIB_EXPORT size_t CC kfprintf ( const KWrtHandler *out,
    size_t *num_writ, const char * fmt, ... )
{
    rc_t rc;

    va_list args;
    va_start ( args, fmt );

    rc = vkfprintf ( out, num_writ, fmt, args );
        
    va_end ( args );

    return rc;
}
