#include <stdio.h>
#include <stdlib.h>

#include "inc.h"
#include "crypto.h"



struct testcase {
	const char* name;
	const struct iovec key;
	const struct iovec data;
	const struct iovec answer;
};

/* Test cases from RFC 4231 (https://tools.ietf.org/html/rfc4231) */
static const struct testcase tests[] = {
	{ "RFC 4231 Test Case 1",
	  {"\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
	   "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b", 20},
	  { "Hi There", 8},
	  {"\xb0\x34\x4c\x61\xd8\xdb\x38\x53\x5c\xa8\xaf\xce\xaf\x0b\xf1\x2b"
	   "\x88\x1d\xc2\x00\xc9\x83\x3d\xa7\x26\xe9\x37\x6c\x2e\x32\xcf\xf7", 32}
	},
	{ "RFC 4231 Test Case 2",
	  {"Jefe", 4},
	  {"what do ya want for nothing?", 28},
	  {"\x5b\xdc\xc1\x46\xbf\x60\x75\x4e\x6a\x04\x24\x26\x08\x95\x75\xc7"
	   "\x5a\x00\x3f\x08\x9d\x27\x39\x83\x9d\xec\x58\xb9\x64\xec\x38\x43", 32}
	},
	{ "RFC 4231 Test Case 3",
	  {"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
	   "\xaa\xaa\xaa\xaa", 20},
	  {"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
	   "\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
	   "\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
	   "\xdd\xdd", 50},
	  {"\x77\x3e\xa9\x1e\x36\x80\x0e\x46\x85\x4d\xb8\xeb\xd0\x91\x81\xa7"
	   "\x29\x59\x09\x8b\x3e\xf8\xc1\x22\xd9\x63\x55\x14\xce\xd5\x65\xfe", 32}
	},
	{ "RFC 4231 Test Case 4",
	  {"\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"
	   "\x11\x12\x13\x14\x15\x16\x17\x18\x19", 25},
	  {"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
	   "\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
	   "\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
	   "\xcd\xcd", 50},
	  {"\x82\x55\x8a\x38\x9a\x44\x3c\x0e\xa4\xcc\x81\x98\x99\xf2\x08\x3a"
	   "\x85\xf0\xfa\xa3\xe5\x78\xf8\x07\x7a\x2e\x3f\xf4\x67\x29\x66\x5b", 32}
	},
	{ "RFC 4231 Test Case 5",
	  {"\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c"
	   "\x0c\x0c\x0c\x0c", 20},
	  {"Test With Truncation", 20},
	  {"\xa3\xb6\x16\x74\x73\x10\x0e\xe0\x6e\x0c\x79\x6c\x29\x55\x55\x2b", 16}
	},
	{ "RFC 4231 Test Case 6",
	  {"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
	   "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
	   "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
	   "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
	   "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
	   "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
	   "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
	   "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
	   "\xaa\xaa\xaa\xaa\xaa\xaa", 131},
	  {"Test Using Larger Than Block-Size Key - Hash Key First", 54},
	  {"\x60\xe4\x31\x59\x1e\xe0\xb6\x7f\x0d\x8a\x26\xaa\xcb\xf5\xb7\x7f"
	   "\x8e\x0b\xc6\x21\x37\x28\xc5\x14\x05\x46\x04\x0f\x0e\xe3\x7f\x54", 32}
	},
	{ "RFC 4231 Test Case 7",
	  {"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
	   "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
	   "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
	   "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
	   "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
	   "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
	   "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
	   "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
	   "\xaa\xaa\xaa\xaa\xaa\xaa", 131},
	  {"This is a test using a larger than block-size key and a larger "
	   "than block-size data. The key needs to be hashed before being "
	   "used by the HMAC algorithm.", 152},
	  {"\x9b\x09\xff\xa7\x1b\x94\x2f\xcb\x27\x63\x5f\xbc\xd5\xb0\xe9\x44"
	   "\xbf\xdc\x63\x64\x4f\x07\x13\x93\x8a\x7f\x51\x53\x5c\x3a\x35\xe2", 32}
	}
};
int main() {
	struct crypt* c = NULL;
	int failures = 0;
	char mac[32];
	int maclen;
	int ix;
	int ret;

	c = crypt_HMAC_SHA256_new();
	for (ix = 0; ix < sizeof(tests)/sizeof(tests[0]); ix++) {
		ret = c->c_set_key(c, tests[ix].key.iov_base, tests[ix].key.iov_len);
		if (ret) {
			printf("%s: failed to set key (return: %d)\n", tests[ix].name, ret);
			failures++;
		}
		maclen = sizeof(mac);
		c->c_mac(c, &(tests[ix].data), 1, mac, &maclen);
		if (maclen != sizeof(mac)) {
			printf("%s: wrong HMAC output length %d\n", tests[ix].name, maclen);
			failures++;
		}
		if (memcmp(tests[ix].answer.iov_base, mac, tests[ix].answer.iov_len) != 0) {
			printf("%s: MAC did not match\n", tests[ix].name);
			failures++;
		}
	};
	c->c_destroy(c);
	if (!failures) {
		printf("OK\n");
		return 0;
	}
	return 1;
}
