/*
 * Copyright (C) 2016-2019 by the Widelands Development Team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 */

#include "ui_basic/dropdown.h"

#include <algorithm>

#include <boost/format.hpp>

#include "base/i18n.h"
#include "base/macros.h"
#include "graphic/align.h"
#include "graphic/font_handler.h"
#include "graphic/graphic.h"
#include "graphic/rendertarget.h"
#include "ui_basic/mouse_constants.h"
#include "ui_basic/tabpanel.h"
#include "ui_basic/window.h"

namespace {

int base_height(int button_dimension) {
	return std::max(
	   button_dimension,
	   UI::g_fh->render(as_uifont(UI::g_fh->fontset()->representative_character()))->height() + 2);
}

}  // namespace

namespace UI {

int BaseDropdown::next_id_ = 0;

BaseDropdown::BaseDropdown(UI::Panel* parent,
                           int32_t x,
                           int32_t y,
                           uint32_t w,
                           uint32_t h,
                           int button_dimension,
                           const std::string& label,
                           const DropdownType type,
                           UI::PanelStyle style)
   : UI::Panel(parent,
               x,
               y,
               type == DropdownType::kPictorial ? button_dimension : w,
               // Height only to fit the button, so we can use this in Box layout.
               base_height(button_dimension)),
     id_(next_id_++),
     max_list_height_(h - 2 * get_h()),
     list_width_(w),
     list_offset_x_(0),
     list_offset_y_(0),
     button_dimension_(button_dimension),
     mouse_tolerance_(50),
     button_box_(this, 0, 0, UI::Box::Horizontal, w, h),
     push_button_(type == DropdownType::kTextual ?
                     new UI::Button(&button_box_,
                                    "dropdown_select",
                                    0,
                                    0,
                                    button_dimension,
                                    get_h(),
                                    style == UI::PanelStyle::kFsMenu ?
                                       UI::ButtonStyle::kFsMenuMenu :
                                       UI::ButtonStyle::kWuiSecondary,
                                    g_gr->images().get("images/ui_basic/scrollbar_down.png")) :
                     nullptr),
     display_button_(&button_box_,
                     "dropdown_label",
                     0,
                     0,
                     type == DropdownType::kTextual ?
                        w - button_dimension :
                        type == DropdownType::kTextualNarrow ? w : button_dimension,
                     get_h(),
                     style == UI::PanelStyle::kFsMenu ? UI::ButtonStyle::kFsMenuSecondary :
                                                        UI::ButtonStyle::kWuiSecondary,
                     label),
     label_(label),
     type_(type),
     is_enabled_(true) {
	if (label.empty()) {
		set_tooltip(pgettext("dropdown", "Select Item"));
	} else {
		set_tooltip(label);
	}

	// Close whenever another dropdown is opened
	subscriber_ = Notifications::subscribe<NoteDropdown>([this](const NoteDropdown& note) {
		if (id_ != note.id) {
			close();
		}
	});

	assert(max_list_height_ > 0);
	// Hook into highest parent that we can get so that we can drop down outside the panel.
	// Positioning breaks down with TabPanels, so we exclude them.
	while (parent->get_parent() && !is_a(UI::TabPanel, parent->get_parent())) {
		parent = parent->get_parent();
	}
	list_ = new UI::Listselect<uintptr_t>(parent, 0, 0, w, 0, style, ListselectLayout::kDropdown);

	list_->set_visible(false);
	button_box_.add(&display_button_);
	display_button_.sigclicked.connect(boost::bind(&BaseDropdown::toggle_list, this));
	if (push_button_ != nullptr) {
		display_button_.set_perm_pressed(true);
		button_box_.add(push_button_);
		push_button_->sigclicked.connect(boost::bind(&BaseDropdown::toggle_list, this));
	}
	button_box_.set_size(w, get_h());
	list_->clicked.connect(boost::bind(&BaseDropdown::set_value, this));
	list_->clicked.connect(boost::bind(&BaseDropdown::toggle_list, this));
	set_can_focus(true);
	set_value();

	// Find parent windows so that we can move the list along with them
	UI::Panel* parent_window_candidate = get_parent();
	while (parent_window_candidate) {
		if (upcast(UI::Window, window, parent_window_candidate)) {
			window->position_changed.connect(boost::bind(&BaseDropdown::layout, this));
		}
		parent_window_candidate = parent_window_candidate->get_parent();
	}

	layout();
}

BaseDropdown::~BaseDropdown() {
	// The list needs to be able to drop outside of windows, so it won't close with the window.
	// Deleting here leads to conflict with who gets to delete it, so we hide it instead.
	// TODO(GunChleoc): Investigate whether we can find a better solution for this
	if (list_) {
		list_->clear();
		list_->set_visible(false);
	}
}

void BaseDropdown::set_height(int height) {
	max_list_height_ = height - base_height(button_dimension_);
	layout();
}

void BaseDropdown::set_max_items(int items) {
	set_height(list_->get_lineheight() * items + base_height(button_dimension_));
}

void BaseDropdown::layout() {
	const int base_h = base_height(button_dimension_);
	const int w = type_ == DropdownType::kPictorial ? button_dimension_ : get_w();
	button_box_.set_size(w, base_h);
	display_button_.set_desired_size(
	   type_ == DropdownType::kTextual ? w - button_dimension_ : w, base_h);
	int new_list_height =
	   std::min(static_cast<int>(list_->size()) * list_->get_lineheight(), max_list_height_);
	list_->set_size(type_ != DropdownType::kPictorial ? w : list_width_, new_list_height);
	set_desired_size(w, base_h);

	// Update list position. The list is hooked into the highest parent that we can get so that we
	// can drop down outside the panel. Positioning breaks down with TabPanels, so we exclude them.
	UI::Panel* parent = get_parent();
	int new_list_x = get_x() + parent->get_x() + parent->get_lborder();
	int new_list_y = get_y() + parent->get_y() + parent->get_tborder();
	while (parent->get_parent() && !is_a(UI::TabPanel, parent->get_parent())) {
		parent = parent->get_parent();
		new_list_x += parent->get_x() + parent->get_lborder();
		new_list_y += parent->get_y() + parent->get_tborder();
	}

	// Drop up instead of down if it doesn't fit
	if (new_list_y + list_->get_h() > g_gr->get_yres()) {
		list_offset_y_ = -list_->get_h();
	} else {
		list_offset_y_ = display_button_.get_h();
	}

	// Right align instead of left align if it doesn't fit
	if (new_list_x + list_->get_w() > g_gr->get_xres()) {
		list_offset_x_ = display_button_.get_w() - list_->get_w();
		if (push_button_ != nullptr) {
			list_offset_x_ += push_button_->get_w();
		}
	}

	list_->set_pos(Vector2i(new_list_x + list_offset_x_, new_list_y + list_offset_y_));

	// Keep open list on top while dragging
	// TODO(GunChleoc): It would be better to close the list if any other panel is clicked,
	// but we'd need a global "clicked" signal in the Panel class for that.
	// This will imply a complete overhaul of the signal names.
	if (list_->is_visible()) {
		list_->move_to_top();
	}
}

void BaseDropdown::add(const std::string& name,
                       const uint32_t value,
                       const Image* pic,
                       const bool select_this,
                       const std::string& tooltip_text) {
	assert(pic != nullptr || type_ != DropdownType::kPictorial);
	list_->add(name, value, pic, select_this, tooltip_text);
	if (select_this) {
		set_value();
	}
	layout();
}

bool BaseDropdown::has_selection() const {
	return list_->has_selection();
}

uint32_t BaseDropdown::get_selected() const {
	assert(has_selection());
	return list_->get_selected();
}

void BaseDropdown::select(uint32_t entry) {
	assert(entry < list_->size());
	list_->select(entry);
	current_selection_ = list_->selection_index();
	update();
}

void BaseDropdown::set_label(const std::string& text) {
	label_ = text;
	if (type_ != DropdownType::kPictorial) {
		display_button_.set_title(label_);
	}
}

void BaseDropdown::set_image(const Image* image) {
	display_button_.set_pic(image);
}

void BaseDropdown::set_tooltip(const std::string& text) {
	tooltip_ = text;
	display_button_.set_tooltip(tooltip_);
	if (push_button_) {
		push_button_->set_tooltip(push_button_->enabled() ? tooltip_ : "");
	}
}

void BaseDropdown::set_errored(const std::string& error_message) {
	set_tooltip((boost::format(_("%1%: %2%")) % _("Error") % error_message).str());
	if (type_ != DropdownType::kPictorial) {
		set_label(_("Error"));
	} else {
		set_image(g_gr->images().get("images/ui_basic/different.png"));
	}
}

void BaseDropdown::set_enabled(bool on) {
	is_enabled_ = on;
	set_can_focus(on);
	if (push_button_ != nullptr) {
		push_button_->set_enabled(on);
		push_button_->set_tooltip(on ? tooltip_ : "");
	}
	display_button_.set_enabled(on);
	list_->set_visible(false);
}

void BaseDropdown::set_disable_style(UI::ButtonDisableStyle disable_style) {
	display_button_.set_disable_style(disable_style);
}

bool BaseDropdown::is_expanded() const {
	return list_->is_visible();
}

void BaseDropdown::set_pos(Vector2i point) {
	UI::Panel::set_pos(point);
	list_->set_pos(Vector2i(point.x, point.y + get_h()));
}

void BaseDropdown::clear() {
	close();
	list_->clear();
	current_selection_ = list_->selection_index();
	list_->set_size(list_->get_w(), 0);
	list_->set_visible(false);
	set_layout_toplevel(false);
}

void BaseDropdown::think() {
	if (list_->is_visible()) {
		// Autocollapse with a bit of tolerance for the mouse movement to make it less fiddly.
		if (!(has_focus() || list_->has_focus()) || is_mouse_away()) {
			toggle_list();
		}
	}
}

uint32_t BaseDropdown::size() const {
	return list_->size();
}

void BaseDropdown::update() {
	const std::string name = list_->has_selection() ?
	                            list_->get_selected_name() :
	                            /** TRANSLATORS: Selection in Dropdown menus. */
	                            pgettext("dropdown", "Not Selected");

	if (type_ != DropdownType::kPictorial) {
		if (label_.empty()) {
			display_button_.set_title(name);
		} else {
			/** TRANSLATORS: Label: Value. */
			display_button_.set_title((boost::format(_("%1%: %2%")) % label_ % (name)).str());
		}
		display_button_.set_tooltip(list_->has_selection() ? list_->get_selected_tooltip() :
		                                                     tooltip_);
	} else {
		display_button_.set_pic(list_->has_selection() ?
		                           list_->get_selected_image() :
		                           g_gr->images().get("images/ui_basic/different.png"));
		display_button_.set_tooltip((boost::format(_("%1%: %2%")) % label_ % name).str());
	}
}

void BaseDropdown::set_value() {
	update();
	selected();
	current_selection_ = list_->selection_index();
}

void BaseDropdown::toggle_list() {
	if (!is_enabled_) {
		list_->set_visible(false);
		return;
	}
	list_->set_visible(!list_->is_visible());
	if (type_ != DropdownType::kTextual) {
		display_button_.set_perm_pressed(list_->is_visible());
	}
	if (list_->is_visible()) {
		list_->move_to_top();
		focus();
		Notifications::publish(NoteDropdown(id_));
	}
	// Make sure that the list covers and deactivates the elements below it
	set_layout_toplevel(list_->is_visible());
}

void BaseDropdown::close() {
	if (is_expanded()) {
		toggle_list();
	}
}

bool BaseDropdown::is_mouse_away() const {
	return (get_mouse_position().x + mouse_tolerance_) < list_offset_x_ ||
	       get_mouse_position().x > (list_offset_x_ + list_->get_w() + mouse_tolerance_) ||
	       (get_mouse_position().y + mouse_tolerance_) < list_offset_y_ ||
	       get_mouse_position().y > (list_offset_y_ + get_h() + list_->get_h() + mouse_tolerance_);
}

bool BaseDropdown::handle_key(bool down, SDL_Keysym code) {
	if (down) {
		switch (code.sym) {
		case SDLK_KP_ENTER:
		case SDLK_RETURN:
			if (list_->is_visible()) {
				set_value();
			}
			break;
		case SDLK_ESCAPE:
			if (list_->is_visible()) {
				list_->select(current_selection_);
				toggle_list();
				return true;
			}
			break;
		case SDLK_DOWN:
			if (!list_->is_visible() && !is_mouse_away()) {
				toggle_list();
				return true;
			}
			break;
		default:
			break;  // not handled
		}
	}
	if (list_->is_visible()) {
		return list_->handle_key(down, code);
	}
	return false;
}

}  // namespace UI
