/* go-cdiv.c -- complex division routines

   Copyright 2013 The Go Authors. All rights reserved.
   Use of this source code is governed by a BSD-style
   license that can be found in the LICENSE file.  */

/* Calls to these functions are generated by the Go frontend for
   division of complex64 or complex128.  We use these because Go's
   complex division expects slightly different results from the GCC
   default.  When dividing NaN+1.0i / 0+0i, Go expects NaN+NaNi but
   GCC generates NaN+Infi.  NaN+Infi seems wrong seems the rules of
   C99 Annex G specify that if either side of a complex number is Inf,
   the the whole number is Inf, but an operation involving NaN ought
   to result in NaN, not Inf.  */

__complex float
__go_complex64_div (__complex float a, __complex float b)
{
  if (__builtin_expect (b == 0+0i, 0))
    {
      if (!__builtin_isinff (__real__ a)
	  && !__builtin_isinff (__imag__ a)
	  && (__builtin_isnanf (__real__ a) || __builtin_isnanf (__imag__ a)))
	{
	  /* Pass "1" to nanf to match math/bits.go.  */
	  return __builtin_nanf("1") + __builtin_nanf("1")*1i;
	}
    }
  return a / b;
}

__complex double
__go_complex128_div (__complex double a, __complex double b)
{
  if (__builtin_expect (b == 0+0i, 0))
    {
      if (!__builtin_isinf (__real__ a)
	  && !__builtin_isinf (__imag__ a)
	  && (__builtin_isnan (__real__ a) || __builtin_isnan (__imag__ a)))
	{
	  /* Pass "1" to nan to match math/bits.go.  */
	  return __builtin_nan("1") + __builtin_nan("1")*1i;
	}
    }
  return a / b;
}
