/*
 * Unix SMB/CIFS implementation.
 *
 * Copyright (C) Catalyst.Net Ltd 2023
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdarg.h>
#include <stddef.h>
#include <stdint.h>
#include <setjmp.h>
#include <cmocka.h>

#include <talloc.h>

#include "lib/replace/replace.h"
#include "lib/util/genrand.h"
#include "lib/crypto/gnutls_helpers.h"

#include <gnutls/gnutls.h>
#include <gnutls/crypto.h>

#include "lib/crypto/gnutls_helpers.h"

struct test_vector {
	const uint8_t *KI; /* The key‐derivation key used as input. */
	size_t KI_len;
	const uint8_t *input; /* Other data used as input. */
	size_t input_len;
	const uint8_t *KO; /* The expected derived key. */
	size_t KO_len;
	gnutls_mac_algorithm_t prf_algorithm;
};

#define TEST_VECTOR_ARRAY(...) (const uint8_t[]){__VA_ARGS__}
#define TEST_VECTOR(key_in, data, key_out, algorithm) { \
	.KI = TEST_VECTOR_ARRAY key_in, \
	.KI_len = sizeof (TEST_VECTOR_ARRAY key_in), \
	.input = TEST_VECTOR_ARRAY data, \
	.input_len = sizeof (TEST_VECTOR_ARRAY data), \
	.KO = TEST_VECTOR_ARRAY key_out, \
	.KO_len = sizeof (TEST_VECTOR_ARRAY key_out), \
	.prf_algorithm = (algorithm), \
}

/*
 * A subset of the test vectors provided at
 * https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Algorithm-Validation-Program/documents/KBKDF800-108/CounterMode.zip,
 * and released into the public domain as per
 * https://www.nist.gov/oism/copyrights.
 *
 * These cover SHA1, SHA224, SHA256, SHA384, and SHA512; with the counter
 * located before the input data, an r length of 32, and various values for L.
 */
static const struct test_vector nist_vectors[] = {
	TEST_VECTOR((247, 89, 23, 51, 200, 86, 89, 53, 101, 19, 9, 117, 53, 25, 84, 208, 21, 90, 191, 60), (142, 52, 126, 245, 93, 95, 94, 153, 234, 182, 222, 112, 107, 81, 222, 124, 224, 4, 243, 136, 40, 137, 226, 89, 255, 78, 92, 255, 16, 33, 103, 165, 164, 189, 113, 21, 120, 212, 206, 23, 221, 154, 190, 86, 229, 28, 31, 45, 249, 80, 226, 252, 129, 46, 193, 178, 23, 202, 8, 214), (52, 254, 68, 176, 216, 196, 27, 147, 245, 250, 100, 251, 150, 240, 14, 91), GNUTLS_MAC_SHA1),
	TEST_VECTOR((71, 234, 93, 0, 130, 111, 17, 27, 250, 70, 142, 137, 155, 12, 163, 253, 167, 200, 51, 109), (219, 237, 20, 21, 110, 24, 38, 170, 29, 77, 102, 70, 156, 100, 163, 37, 113, 112, 17, 49, 178, 122, 140, 89, 94, 140, 5, 149, 102, 84, 120, 252, 241, 185, 92, 118, 174, 153, 125, 230, 24, 76, 145, 159, 75, 129, 13, 88, 133, 72, 80, 140, 212, 196, 87, 165, 65, 38, 7, 37), (125, 227, 116, 181, 239, 114, 64, 253, 66, 91, 124, 247, 197, 76, 41, 244), GNUTLS_MAC_SHA1),
	TEST_VECTOR((59, 22, 101, 78, 46, 137, 255, 74, 204, 87, 166, 197, 200, 175, 128, 100, 128, 120, 4, 166), (5, 225, 110, 209, 181, 96, 196, 201, 196, 50, 103, 241, 46, 228, 84, 218, 252, 1, 97, 190, 220, 225, 167, 211, 113, 236, 23, 66, 109, 8, 30, 254, 84, 175, 140, 47, 37, 102, 212, 238, 163, 138, 171, 171, 145, 251, 152, 253, 190, 22, 204, 182, 177, 20, 214, 119, 188, 176, 12, 102), (29, 41, 163, 210, 73, 158, 148, 115, 220, 98, 84, 77, 131, 11, 167, 115), GNUTLS_MAC_SHA1),
	TEST_VECTOR((87, 233, 253, 110, 12, 84, 201, 0, 101, 89, 59, 68, 72, 35, 66, 242, 234, 114, 193, 40), (187, 187, 239, 212, 124, 153, 118, 9, 196, 82, 47, 100, 188, 148, 160, 40, 165, 197, 92, 71, 17, 129, 199, 79, 239, 90, 83, 155, 163, 190, 119, 103, 75, 78, 237, 155, 181, 234, 9, 149, 177, 98, 2, 113, 100, 227, 141, 198, 60, 238, 176, 206, 111, 94, 119, 48, 235, 227, 71, 26), (150, 145, 249, 155, 148, 242, 15, 156, 84, 130, 118, 222, 78, 175, 6, 101), GNUTLS_MAC_SHA1),
	TEST_VECTOR((247, 202, 82, 65, 121, 75, 126, 107, 178, 66, 65, 68, 49, 176, 76, 112, 79, 0, 224, 24), (18, 70, 144, 235, 250, 211, 72, 105, 137, 160, 174, 6, 160, 128, 32, 130, 115, 176, 162, 63, 213, 183, 6, 180, 8, 74, 5, 154, 199, 105, 231, 102, 142, 108, 122, 101, 46, 21, 91, 94, 188, 74, 131, 37, 215, 12, 37, 199, 56, 5, 52, 111, 210, 105, 250, 236, 137, 55, 72, 59), (129, 54, 159, 109, 114, 114, 107, 229, 139, 230, 66, 157, 238, 216, 164, 203), GNUTLS_MAC_SHA1),
	TEST_VECTOR((178, 121, 242, 9, 138, 33, 241, 93, 52, 178, 237, 212, 82, 172, 236, 147, 152, 87, 81, 227), (150, 164, 202, 162, 36, 75, 85, 214, 84, 72, 193, 183, 1, 211, 161, 166, 182, 255, 125, 221, 228, 87, 119, 208, 75, 132, 148, 64, 53, 228, 166, 195, 123, 21, 107, 141, 80, 76, 139, 3, 159, 104, 231, 142, 185, 141, 46, 26, 231, 189, 154, 19, 126, 53, 118, 32, 104, 131, 228, 39), (79, 148, 67, 206, 124, 118, 184, 183, 117, 42, 203, 24, 178, 249, 176, 96), GNUTLS_MAC_SHA1),
	TEST_VECTOR((90, 114, 13, 158, 146, 231, 110, 71, 215, 166, 59, 148, 189, 204, 231, 10, 30, 46, 69, 173), (235, 123, 230, 244, 49, 196, 12, 158, 154, 74, 195, 178, 213, 17, 215, 187, 219, 155, 14, 66, 171, 105, 136, 162, 141, 35, 51, 38, 62, 184, 101, 135, 253, 232, 215, 229, 247, 51, 226, 4, 98, 122, 174, 197, 206, 92, 122, 131, 96, 54, 18, 52, 27, 112, 108, 15, 107, 30, 169, 95), (129, 100, 180, 82, 27, 201, 240, 155, 143, 70, 99, 118, 44, 245, 91, 126), GNUTLS_MAC_SHA1),
	TEST_VECTOR((67, 59, 126, 103, 66, 187, 216, 4, 15, 123, 170, 167, 136, 46, 179, 189, 158, 148, 8, 184), (103, 232, 74, 203, 220, 12, 188, 169, 193, 118, 161, 28, 230, 175, 95, 132, 154, 161, 126, 207, 151, 52, 157, 239, 107, 105, 247, 156, 20, 159, 164, 9, 70, 48, 218, 252, 118, 188, 105, 166, 111, 28, 104, 235, 36, 156, 29, 226, 103, 74, 104, 26, 192, 226, 82, 43, 245, 231, 96, 194), (132, 210, 201, 94, 238, 237, 154, 224, 152, 128, 15, 51, 115, 22, 226, 165), GNUTLS_MAC_SHA1),
	TEST_VECTOR((198, 123, 200, 96, 248, 215, 117, 219, 222, 72, 43, 2, 145, 143, 203, 169, 194, 29, 218, 208), (38, 215, 62, 11, 206, 251, 126, 30, 246, 43, 103, 92, 38, 59, 121, 5, 184, 107, 61, 232, 233, 35, 228, 13, 222, 155, 168, 160, 93, 222, 103, 201, 99, 167, 50, 164, 34, 129, 218, 169, 60, 187, 118, 174, 198, 254, 141, 88, 60, 10, 125, 240, 53, 248, 32, 126, 46, 114, 8, 47), (152, 229, 58, 236, 244, 45, 103, 233, 40, 97, 194, 149, 29, 157, 79, 103), GNUTLS_MAC_SHA1),
	TEST_VECTOR((136, 162, 179, 155, 67, 106, 69, 114, 136, 56, 1, 77, 68, 95, 22, 47, 169, 253, 123, 10), (246, 100, 120, 118, 191, 97, 113, 50, 160, 124, 92, 27, 73, 162, 142, 236, 60, 43, 106, 16, 116, 253, 81, 66, 170, 225, 50, 213, 51, 228, 161, 114, 203, 22, 110, 168, 155, 124, 177, 205, 209, 100, 198, 252, 127, 37, 74, 6, 3, 91, 43, 29, 40, 172, 69, 255, 8, 113, 174, 4), (55, 128, 142, 180, 10, 243, 68, 84, 173, 77, 214, 117, 11, 220, 142, 115), GNUTLS_MAC_SHA1),
	TEST_VECTOR((193, 239, 184, 210, 90, 255, 198, 30, 208, 96, 217, 148, 252, 213, 1, 124, 42, 223, 195, 136), (185, 47, 192, 85, 5, 127, 236, 113, 185, 197, 62, 124, 68, 135, 36, 35, 165, 126, 209, 134, 214, 186, 102, 217, 128, 254, 205, 18, 83, 191, 113, 71, 147, 32, 183, 191, 56, 213, 5, 239, 121, 202, 77, 98, 215, 140, 166, 98, 100, 44, 220, 237, 185, 149, 3, 234, 4, 193, 219, 232), (141, 183, 132, 207, 144, 181, 115, 176, 111, 155, 124, 125, 202, 99, 161, 234, 22, 217, 62, 231, 215, 15, 249, 216, 127, 162, 85, 142, 131, 220, 78, 170), GNUTLS_MAC_SHA1),
	TEST_VECTOR((126, 146, 92, 220, 188, 47, 213, 179, 33, 60, 186, 119, 20, 203, 113, 211, 89, 147, 97, 9), (31, 149, 165, 119, 149, 137, 129, 183, 71, 124, 162, 167, 29, 68, 230, 101, 100, 229, 165, 109, 129, 137, 84, 39, 24, 235, 26, 206, 129, 67, 110, 22, 244, 21, 135, 137, 164, 109, 25, 212, 56, 78, 21, 85, 91, 195, 180, 255, 161, 11, 84, 113, 172, 51, 191, 144, 73, 130, 244, 158), (85, 123, 94, 31, 231, 112, 180, 199, 110, 160, 204, 211, 31, 92, 29, 212, 255, 160, 70, 58, 140, 233, 250, 4, 247, 245, 209, 170, 83, 56, 0, 67), GNUTLS_MAC_SHA1),
	TEST_VECTOR((152, 182, 240, 180, 183, 168, 34, 232, 245, 103, 163, 45, 39, 254, 189, 169, 228, 98, 23, 211), (224, 138, 97, 35, 135, 25, 178, 31, 181, 246, 54, 31, 33, 94, 181, 165, 96, 50, 51, 191, 115, 161, 1, 243, 47, 9, 99, 151, 76, 1, 23, 252, 34, 208, 36, 216, 25, 63, 25, 254, 9, 92, 200, 115, 175, 199, 184, 152, 136, 203, 228, 23, 98, 231, 171, 110, 139, 120, 58, 0), (85, 4, 121, 58, 98, 151, 110, 119, 177, 243, 248, 187, 235, 197, 133, 78, 21, 56, 250, 0, 42, 152, 175, 91, 147, 93, 152, 80, 182, 95, 154, 97), GNUTLS_MAC_SHA1),
	TEST_VECTOR((186, 62, 97, 49, 207, 146, 76, 14, 103, 190, 1, 32, 93, 224, 124, 200, 199, 124, 185, 77), (157, 19, 178, 63, 237, 177, 139, 174, 77, 252, 248, 195, 246, 7, 14, 131, 234, 135, 161, 237, 32, 164, 1, 55, 55, 34, 158, 5, 246, 193, 248, 242, 107, 56, 74, 44, 237, 163, 125, 99, 37, 149, 21, 106, 188, 178, 18, 11, 222, 3, 204, 58, 233, 243, 229, 139, 205, 154, 229, 37), (90, 72, 65, 161, 74, 18, 240, 76, 101, 173, 94, 96, 201, 241, 14, 76, 237, 58, 254, 158, 225, 10, 168, 174, 214, 107, 166, 41, 216, 110, 33, 60), GNUTLS_MAC_SHA1),
	TEST_VECTOR((90, 59, 85, 212, 164, 137, 122, 31, 90, 212, 163, 165, 226, 37, 156, 205, 123, 47, 246, 47), (240, 46, 246, 25, 97, 190, 45, 239, 195, 203, 144, 100, 243, 15, 159, 215, 202, 128, 23, 74, 211, 246, 14, 2, 221, 123, 100, 82, 139, 111, 64, 24, 2, 152, 100, 85, 93, 58, 192, 62, 66, 21, 203, 221, 191, 180, 242, 88, 162, 107, 236, 70, 80, 122, 156, 254, 5, 84, 55, 247), (203, 9, 130, 105, 22, 233, 183, 59, 221, 182, 191, 37, 230, 202, 245, 182, 133, 111, 136, 213, 29, 179, 81, 253, 131, 21, 226, 144, 136, 152, 89, 33), GNUTLS_MAC_SHA1),
	TEST_VECTOR((123, 145, 103, 226, 246, 47, 55, 150, 89, 172, 4, 79, 69, 10, 239, 123, 213, 233, 56, 198), (244, 51, 232, 241, 119, 216, 253, 87, 67, 237, 89, 144, 168, 160, 237, 161, 154, 174, 8, 219, 17, 219, 111, 130, 228, 225, 222, 230, 125, 105, 159, 158, 17, 210, 157, 109, 125, 168, 240, 110, 183, 233, 115, 243, 161, 51, 253, 84, 192, 228, 10, 68, 91, 88, 3, 213, 171, 54, 101, 55), (68, 187, 182, 151, 134, 148, 177, 7, 113, 99, 71, 214, 198, 116, 166, 52, 89, 124, 252, 81, 168, 37, 177, 156, 35, 109, 246, 140, 63, 64, 152, 132), GNUTLS_MAC_SHA1),
	TEST_VECTOR((111, 157, 30, 146, 168, 241, 85, 194, 199, 28, 169, 214, 134, 15, 164, 222, 211, 181, 224, 28), (210, 235, 142, 103, 179, 17, 133, 105, 22, 40, 80, 205, 132, 151, 50, 36, 44, 41, 24, 85, 76, 76, 248, 97, 148, 129, 104, 184, 219, 120, 61, 69, 188, 253, 206, 219, 85, 186, 154, 201, 14, 57, 20, 50, 131, 51, 181, 241, 44, 213, 84, 142, 220, 177, 30, 139, 251, 63, 64, 95), (207, 78, 136, 187, 50, 49, 249, 5, 228, 3, 101, 63, 233, 102, 24, 54, 163, 37, 160, 10, 57, 20, 235, 225, 146, 53, 98, 80, 90, 31, 0, 174), GNUTLS_MAC_SHA1),
	TEST_VECTOR((139, 32, 209, 242, 89, 197, 44, 229, 174, 176, 106, 130, 73, 102, 149, 81, 16, 20, 90, 143), (67, 217, 219, 191, 8, 84, 20, 190, 187, 202, 232, 170, 248, 144, 171, 20, 169, 15, 217, 37, 165, 130, 176, 16, 249, 195, 160, 138, 62, 16, 168, 151, 203, 90, 232, 9, 152, 44, 85, 139, 101, 247, 193, 95, 231, 163, 22, 213, 65, 127, 38, 194, 115, 7, 92, 173, 87, 1, 209, 93), (78, 35, 56, 151, 110, 120, 50, 241, 111, 205, 240, 111, 211, 220, 75, 215, 119, 209, 112, 207, 86, 246, 168, 226, 17, 116, 53, 207, 5, 210, 181, 248), GNUTLS_MAC_SHA1),
	TEST_VECTOR((105, 59, 85, 124, 143, 212, 5, 74, 129, 118, 207, 195, 27, 104, 24, 21, 219, 224, 113, 128), (142, 148, 138, 52, 81, 175, 216, 59, 96, 65, 87, 109, 149, 148, 86, 186, 201, 123, 159, 147, 163, 209, 51, 25, 140, 218, 39, 61, 132, 111, 123, 139, 201, 175, 209, 155, 131, 32, 195, 197, 8, 89, 194, 146, 235, 249, 103, 216, 142, 160, 145, 36, 179, 51, 177, 68, 49, 99, 241, 140), (53, 155, 33, 56, 132, 176, 170, 86, 80, 53, 58, 83, 252, 191, 190, 91, 14, 0, 20, 243, 103, 204, 45, 67, 13, 124, 67, 122, 60, 9, 50, 108), GNUTLS_MAC_SHA1),
	TEST_VECTOR((255, 240, 254, 226, 54, 75, 85, 150, 247, 56, 142, 192, 2, 29, 219, 168, 202, 148, 251, 32), (94, 64, 206, 175, 237, 151, 71, 84, 84, 245, 129, 236, 184, 72, 186, 208, 16, 175, 13, 103, 131, 179, 44, 232, 107, 236, 87, 175, 110, 95, 5, 214, 199, 49, 145, 31, 82, 106, 186, 244, 149, 150, 213, 255, 11, 41, 194, 11, 191, 123, 132, 158, 202, 5, 139, 30, 171, 198, 13, 100), (75, 247, 73, 81, 46, 132, 147, 242, 186, 236, 96, 43, 209, 251, 29, 248, 65, 204, 146, 73, 107, 178, 140, 5, 54, 63, 119, 164, 242, 165, 75, 168), GNUTLS_MAC_SHA1),
	TEST_VECTOR((224, 43, 165, 213, 196, 16, 232, 85, 187, 209, 63, 132, 1, 36, 39, 62, 107, 134, 66, 55), (177, 78, 34, 123, 68, 56, 249, 115, 214, 113, 20, 28, 98, 70, 172, 220, 121, 78, 238, 145, 188, 126, 253, 29, 95, 240, 42, 123, 143, 176, 68, 0, 159, 182, 241, 240, 246, 79, 53, 54, 95, 177, 9, 142, 25, 149, 163, 79, 139, 112, 167, 30, 208, 38, 94, 209, 122, 231, 174, 64), (240, 119, 194, 213, 211, 106, 101, 128, 49, 199, 78, 245, 166, 106, 164, 139, 68, 86, 83, 10), GNUTLS_MAC_SHA1),
	TEST_VECTOR((209, 188, 223, 2, 137, 96, 25, 92, 236, 244, 120, 185, 232, 97, 234, 36, 57, 98, 235, 233), (236, 75, 18, 97, 15, 70, 247, 254, 68, 17, 227, 154, 178, 134, 25, 235, 255, 238, 109, 252, 99, 142, 78, 147, 142, 241, 224, 67, 66, 215, 93, 238, 18, 27, 55, 91, 226, 146, 41, 87, 4, 192, 144, 138, 166, 27, 72, 143, 106, 41, 171, 16, 100, 104, 19, 78, 113, 181, 67, 158), (9, 215, 29, 138, 239, 45, 52, 229, 82, 249, 190, 37, 252, 250, 96, 248, 127, 93, 182, 171), GNUTLS_MAC_SHA1),
	TEST_VECTOR((209, 210, 169, 16, 214, 137, 132, 143, 210, 32, 255, 48, 23, 124, 105, 3, 149, 132, 44, 105), (232, 206, 254, 230, 94, 184, 128, 149, 182, 252, 129, 217, 49, 62, 53, 180, 144, 148, 111, 159, 17, 74, 254, 35, 109, 93, 137, 39, 119, 64, 229, 0, 254, 80, 194, 185, 0, 182, 201, 19, 155, 10, 217, 63, 57, 136, 1, 112, 113, 137, 171, 243, 188, 115, 119, 157, 70, 251, 192, 18), (96, 216, 32, 108, 226, 142, 51, 198, 122, 136, 79, 13, 203, 138, 183, 12, 245, 93, 167, 95), GNUTLS_MAC_SHA1),
	TEST_VECTOR((136, 123, 116, 178, 200, 166, 235, 136, 143, 195, 137, 204, 45, 161, 135, 15, 4, 4, 116, 114), (201, 95, 228, 145, 183, 83, 138, 157, 105, 228, 206, 63, 255, 198, 199, 154, 216, 187, 20, 204, 181, 251, 191, 176, 163, 113, 237, 108, 1, 134, 64, 185, 99, 130, 165, 123, 35, 186, 174, 63, 77, 120, 58, 61, 33, 213, 185, 217, 225, 17, 162, 68, 201, 96, 17, 138, 34, 38, 156, 221), (246, 96, 45, 227, 223, 206, 7, 79, 227, 34, 210, 103, 201, 154, 80, 138, 70, 207, 37, 98), GNUTLS_MAC_SHA1),
	TEST_VECTOR((246, 175, 164, 69, 4, 28, 112, 115, 167, 76, 31, 185, 46, 145, 81, 67, 242, 248, 11, 188), (7, 81, 28, 137, 198, 138, 158, 208, 109, 82, 93, 250, 248, 78, 81, 21, 230, 7, 137, 32, 15, 134, 146, 235, 19, 178, 6, 153, 0, 194, 248, 138, 175, 203, 176, 183, 89, 161, 19, 231, 253, 101, 247, 239, 87, 145, 52, 25, 181, 77, 253, 252, 191, 180, 138, 157, 215, 177, 77, 25), (35, 38, 76, 67, 15, 238, 152, 198, 119, 150, 72, 58, 32, 56, 97, 173, 187, 156, 65, 143), GNUTLS_MAC_SHA1),
	TEST_VECTOR((238, 95, 35, 54, 62, 116, 167, 244, 104, 17, 189, 120, 227, 6, 223, 109, 13, 84, 106, 65), (39, 147, 181, 169, 202, 71, 35, 9, 57, 68, 185, 213, 133, 116, 74, 65, 212, 224, 55, 246, 183, 116, 169, 78, 141, 148, 234, 67, 201, 44, 255, 228, 225, 66, 190, 21, 57, 239, 155, 48, 118, 91, 227, 184, 172, 168, 84, 188, 196, 243, 0, 222, 64, 12, 79, 239, 52, 91, 95, 23), (154, 186, 74, 132, 122, 65, 20, 131, 190, 110, 208, 50, 177, 2, 19, 241, 5, 138, 46, 133), GNUTLS_MAC_SHA1),
	TEST_VECTOR((210, 15, 214, 92, 42, 21, 185, 158, 24, 146, 116, 222, 86, 57, 179, 13, 157, 157, 180, 183), (82, 211, 47, 193, 125, 195, 130, 123, 49, 111, 174, 219, 139, 143, 222, 144, 27, 53, 87, 111, 220, 135, 102, 154, 104, 95, 32, 177, 248, 178, 222, 5, 159, 117, 99, 78, 215, 176, 123, 253, 227, 144, 252, 151, 11, 109, 29, 104, 109, 251, 96, 227, 186, 150, 138, 132, 241, 97, 253, 184), (233, 181, 52, 197, 50, 96, 19, 120, 190, 70, 136, 7, 22, 247, 151, 21, 107, 148, 150, 242), GNUTLS_MAC_SHA1),
	TEST_VECTOR((128, 37, 179, 89, 12, 213, 20, 194, 249, 93, 233, 204, 144, 3, 255, 130, 86, 34, 238, 48), (23, 15, 8, 251, 116, 218, 99, 29, 42, 137, 53, 234, 194, 224, 205, 89, 255, 131, 247, 51, 210, 152, 147, 11, 108, 69, 143, 127, 136, 34, 253, 98, 19, 86, 134, 29, 11, 152, 208, 168, 177, 55, 161, 152, 124, 115, 17, 153, 255, 212, 239, 190, 171, 201, 156, 42, 101, 129, 15, 132), (88, 126, 88, 29, 158, 89, 65, 141, 87, 234, 192, 171, 249, 61, 148, 5, 207, 67, 53, 177), GNUTLS_MAC_SHA1),
	TEST_VECTOR((155, 233, 118, 48, 25, 63, 9, 77, 236, 38, 13, 76, 216, 124, 206, 114, 246, 109, 206, 88), (197, 223, 161, 77, 244, 114, 180, 114, 164, 188, 162, 8, 198, 216, 34, 231, 253, 70, 113, 110, 209, 174, 35, 44, 243, 53, 246, 103, 35, 53, 225, 252, 212, 2, 41, 248, 201, 14, 66, 228, 131, 61, 160, 172, 156, 163, 112, 46, 121, 87, 96, 121, 5, 14, 23, 98, 194, 169, 64, 167), (80, 225, 107, 193, 107, 84, 200, 105, 59, 145, 166, 173, 54, 43, 37, 92, 42, 124, 104, 74), GNUTLS_MAC_SHA1),
	TEST_VECTOR((168, 251, 21, 40, 47, 144, 244, 153, 55, 181, 237, 100, 39, 102, 12, 55, 179, 6, 149, 131), (101, 112, 172, 45, 192, 205, 120, 70, 88, 3, 52, 28, 41, 215, 63, 190, 90, 184, 4, 178, 31, 103, 127, 172, 83, 50, 24, 173, 150, 119, 73, 252, 221, 221, 131, 247, 74, 67, 131, 156, 100, 176, 115, 218, 211, 180, 48, 164, 209, 153, 82, 122, 121, 79, 15, 86, 152, 126, 169, 23), (125, 191, 38, 251, 58, 7, 170, 62, 196, 181, 115, 159, 62, 8, 253, 43, 45, 29, 130, 11), GNUTLS_MAC_SHA1),
	TEST_VECTOR((105, 58, 219, 144, 55, 24, 70, 39, 173, 48, 15, 23, 105, 133, 189, 55, 159, 56, 138, 149), (127, 9, 87, 12, 45, 147, 4, 236, 116, 58, 184, 69, 168, 118, 28, 18, 108, 24, 245, 207, 114, 53, 142, 173, 162, 181, 209, 222, 180, 61, 198, 160, 244, 255, 143, 147, 59, 239, 122, 240, 188, 250, 203, 51, 250, 7, 248, 202, 4, 160, 106, 254, 35, 24, 53, 213, 7, 89, 150, 190), (82, 245, 95, 81, 1, 14, 155, 215, 142, 79, 88, 202, 178, 116, 236, 175, 165, 97, 189, 78, 15, 32, 218, 132, 240, 48, 58, 30, 95, 249, 190, 188, 81, 67, 97, 236, 109, 245, 199, 126), GNUTLS_MAC_SHA1),
	TEST_VECTOR((112, 234, 15, 254, 28, 215, 109, 248, 107, 120, 21, 9, 147, 171, 207, 95, 134, 226, 222, 238), (175, 120, 11, 31, 76, 0, 181, 85, 118, 25, 55, 252, 86, 149, 51, 113, 13, 198, 198, 183, 108, 27, 149, 108, 155, 2, 172, 248, 114, 118, 119, 211, 108, 57, 110, 209, 243, 170, 102, 113, 16, 92, 143, 22, 200, 220, 117, 193, 247, 38, 43, 251, 61, 222, 184, 11, 150, 179, 223, 129), (120, 11, 180, 10, 56, 122, 191, 84, 169, 42, 185, 29, 36, 243, 21, 18, 54, 233, 175, 184, 232, 38, 134, 111, 68, 51, 195, 47, 12, 107, 228, 234, 52, 122, 120, 226, 51, 95, 40, 45), GNUTLS_MAC_SHA1),
	TEST_VECTOR((238, 167, 132, 195, 227, 24, 26, 248, 52, 131, 133, 69, 104, 120, 167, 117, 195, 164, 23, 8), (81, 214, 1, 236, 185, 202, 187, 76, 92, 198, 52, 137, 131, 161, 210, 67, 68, 131, 24, 18, 246, 211, 85, 147, 153, 57, 100, 24, 255, 136, 36, 183, 234, 227, 99, 80, 187, 64, 221, 102, 236, 6, 119, 244, 155, 95, 90, 185, 128, 92, 178, 114, 86, 46, 213, 199, 206, 11, 48, 201), (47, 214, 220, 33, 227, 28, 238, 129, 42, 227, 115, 140, 234, 159, 124, 42, 18, 28, 10, 15, 75, 152, 91, 29, 80, 110, 255, 114, 253, 231, 251, 111, 9, 72, 201, 42, 134, 180, 181, 37), GNUTLS_MAC_SHA1),
	TEST_VECTOR((59, 230, 232, 163, 238, 50, 249, 147, 19, 165, 184, 165, 30, 35, 227, 85, 129, 42, 71, 121), (48, 195, 246, 142, 230, 226, 250, 121, 232, 146, 104, 10, 105, 152, 49, 19, 78, 4, 9, 159, 153, 172, 183, 110, 53, 67, 124, 205, 241, 149, 14, 254, 243, 231, 189, 205, 209, 95, 93, 255, 223, 72, 37, 226, 136, 50, 63, 183, 120, 158, 177, 197, 75, 56, 166, 216, 181, 14, 229, 188), (0, 89, 254, 54, 80, 103, 60, 33, 146, 168, 215, 116, 107, 110, 252, 142, 6, 158, 66, 204, 106, 217, 161, 12, 70, 182, 183, 65, 204, 125, 29, 240, 75, 231, 224, 255, 128, 95, 12, 38), GNUTLS_MAC_SHA1),
	TEST_VECTOR((158, 202, 83, 24, 9, 22, 20, 182, 10, 62, 68, 246, 155, 162, 143, 65, 30, 169, 144, 232), (16, 121, 4, 63, 111, 109, 210, 31, 114, 105, 163, 196, 33, 74, 180, 215, 82, 130, 238, 100, 249, 19, 233, 236, 198, 104, 25, 43, 98, 20, 251, 18, 116, 62, 169, 182, 135, 236, 218, 121, 212, 205, 151, 194, 247, 141, 146, 103, 165, 8, 255, 70, 185, 91, 12, 143, 31, 49, 174, 251), (216, 87, 91, 129, 120, 135, 24, 61, 149, 233, 47, 244, 237, 136, 126, 82, 238, 21, 189, 117, 130, 13, 193, 169, 174, 117, 61, 166, 185, 127, 205, 109, 130, 232, 244, 87, 194, 3, 103, 112), GNUTLS_MAC_SHA1),
	TEST_VECTOR((249, 223, 193, 63, 4, 19, 196, 105, 240, 65, 53, 250, 198, 48, 131, 15, 49, 237, 130, 169), (182, 99, 198, 205, 224, 227, 181, 176, 52, 31, 25, 5, 209, 157, 88, 238, 46, 141, 169, 153, 215, 136, 211, 77, 75, 139, 76, 9, 29, 151, 220, 199, 60, 16, 147, 22, 156, 193, 211, 121, 72, 119, 79, 236, 44, 240, 137, 100, 210, 239, 216, 47, 113, 164, 64, 114, 136, 162, 156, 36), (76, 163, 144, 204, 175, 233, 214, 109, 75, 67, 160, 44, 195, 55, 195, 221, 116, 215, 190, 92, 73, 190, 255, 94, 196, 225, 57, 178, 221, 56, 202, 105, 90, 168, 123, 17, 185, 96, 49, 206), GNUTLS_MAC_SHA1),
	TEST_VECTOR((235, 10, 69, 152, 255, 9, 24, 154, 145, 54, 169, 123, 191, 135, 24, 160, 205, 103, 51, 134), (23, 211, 136, 174, 28, 161, 246, 3, 66, 46, 94, 33, 142, 26, 170, 158, 40, 191, 36, 167, 148, 240, 115, 102, 19, 39, 65, 34, 6, 242, 162, 166, 238, 94, 192, 218, 27, 255, 2, 222, 232, 118, 153, 26, 185, 219, 198, 158, 48, 231, 242, 89, 103, 80, 183, 217, 116, 226, 28, 178), (10, 185, 253, 97, 2, 144, 77, 2, 209, 204, 58, 49, 255, 216, 148, 116, 68, 209, 161, 86, 105, 132, 113, 132, 84, 11, 64, 140, 7, 32, 82, 80, 78, 6, 110, 76, 153, 90, 56, 140), GNUTLS_MAC_SHA1),
	TEST_VECTOR((24, 165, 83, 61, 243, 5, 224, 33, 123, 98, 44, 209, 219, 72, 133, 116, 216, 53, 255, 115), (255, 46, 44, 62, 40, 187, 249, 21, 177, 94, 88, 229, 158, 95, 194, 164, 151, 5, 215, 235, 23, 168, 77, 155, 65, 10, 132, 242, 191, 190, 73, 61, 231, 5, 236, 67, 73, 29, 87, 10, 203, 185, 88, 174, 252, 75, 235, 196, 122, 231, 178, 188, 24, 157, 66, 238, 163, 199, 24, 141), (40, 48, 128, 12, 9, 44, 64, 156, 220, 199, 52, 36, 160, 50, 69, 109, 5, 56, 111, 239, 161, 1, 240, 29, 105, 213, 65, 79, 210, 44, 170, 207, 178, 59, 156, 207, 119, 48, 80, 78), GNUTLS_MAC_SHA1),
	TEST_VECTOR((37, 201, 153, 194, 188, 101, 243, 174, 160, 58, 48, 157, 200, 204, 140, 0, 10, 17, 237, 150), (250, 124, 61, 34, 241, 188, 142, 165, 145, 143, 167, 26, 218, 122, 197, 46, 236, 13, 60, 25, 242, 93, 23, 122, 61, 187, 20, 251, 63, 54, 159, 82, 221, 53, 245, 67, 6, 0, 200, 119, 133, 8, 112, 162, 2, 27, 21, 228, 38, 116, 69, 105, 19, 110, 58, 7, 84, 25, 195, 223), (27, 164, 236, 50, 26, 220, 241, 147, 193, 11, 61, 9, 190, 201, 53, 224, 40, 102, 52, 198, 87, 154, 149, 239, 212, 146, 232, 18, 208, 133, 204, 147, 197, 58, 210, 149, 130, 36, 124, 69), GNUTLS_MAC_SHA1),
	TEST_VECTOR((90, 37, 74, 13, 173, 168, 75, 211, 170, 165, 17, 26, 17, 145, 230, 84, 219, 249, 231, 40), (53, 131, 208, 177, 208, 215, 15, 38, 219, 221, 170, 29, 234, 116, 228, 152, 98, 133, 24, 95, 242, 19, 51, 173, 68, 147, 91, 207, 104, 149, 63, 29, 109, 180, 104, 108, 83, 189, 101, 33, 197, 144, 206, 144, 130, 226, 130, 201, 73, 59, 44, 123, 193, 249, 157, 159, 137, 211, 238, 188), (38, 63, 230, 42, 4, 66, 223, 130, 48, 210, 158, 248, 96, 39, 109, 58, 26, 63, 126, 114, 22, 206, 235, 56, 19, 205, 82, 46, 162, 250, 129, 179, 228, 170, 75, 8, 215, 17, 121, 117), GNUTLS_MAC_SHA1),
	TEST_VECTOR((245, 203, 124, 198, 32, 127, 89, 32, 221, 96, 21, 93, 219, 104, 195, 251, 189, 245, 16, 67, 101, 48, 93, 44, 26, 188, 211, 17), (78, 90, 199, 83, 152, 3, 218, 137, 88, 30, 224, 136, 199, 209, 2, 53, 161, 5, 54, 54, 0, 84, 183, 43, 142, 159, 24, 247, 124, 37, 175, 1, 1, 155, 41, 6, 86, 182, 4, 40, 2, 76, 224, 31, 204, 244, 144, 34, 216, 49, 148, 20, 7, 230, 189, 39, 255, 158, 45, 40), (10, 219, 170, 180, 62, 221, 83, 43, 86, 10, 50, 44, 132, 172, 84, 14), GNUTLS_MAC_SHA224),
	TEST_VECTOR((58, 76, 49, 161, 87, 203, 181, 130, 233, 221, 7, 128, 217, 74, 187, 64, 12, 17, 44, 161, 225, 180, 17, 87, 33, 99, 239, 60), (153, 115, 23, 115, 114, 30, 152, 57, 244, 210, 179, 102, 75, 149, 187, 241, 166, 129, 81, 108, 42, 150, 89, 24, 247, 74, 53, 185, 132, 83, 182, 118, 250, 187, 125, 224, 59, 89, 192, 164, 115, 174, 203, 130, 177, 254, 183, 103, 210, 67, 72, 151, 96, 201, 199, 197, 140, 74, 48, 70), (6, 51, 149, 12, 245, 124, 15, 49, 113, 122, 98, 135, 135, 45, 29, 46), GNUTLS_MAC_SHA224),
	TEST_VECTOR((52, 11, 216, 187, 233, 14, 124, 28, 27, 77, 93, 130, 95, 152, 43, 89, 157, 163, 213, 170, 152, 132, 152, 171, 88, 39, 153, 137), (213, 210, 35, 130, 73, 18, 25, 117, 163, 252, 179, 112, 0, 75, 126, 60, 142, 50, 85, 80, 170, 14, 35, 153, 32, 170, 181, 63, 222, 131, 185, 229, 175, 195, 74, 229, 19, 74, 78, 253, 117, 25, 126, 163, 145, 72, 103, 165, 120, 27, 130, 252, 250, 183, 67, 127, 137, 95, 228, 239), (29, 19, 87, 60, 253, 241, 15, 59, 10, 49, 45, 151, 70, 247, 103, 14), GNUTLS_MAC_SHA224),
	TEST_VECTOR((243, 55, 29, 98, 56, 200, 137, 145, 129, 92, 138, 205, 228, 254, 109, 230, 33, 75, 224, 162, 255, 40, 57, 236, 189, 104, 23, 70), (55, 2, 63, 131, 130, 136, 17, 104, 91, 84, 142, 66, 128, 192, 198, 187, 248, 206, 181, 215, 135, 101, 239, 81, 73, 100, 92, 187, 169, 63, 104, 235, 29, 221, 4, 179, 95, 199, 122, 197, 120, 242, 231, 96, 198, 213, 101, 34, 8, 172, 204, 172, 230, 50, 45, 152, 217, 168, 66, 242), (233, 200, 119, 181, 163, 205, 1, 106, 211, 200, 77, 31, 248, 11, 48, 196), GNUTLS_MAC_SHA224),
	TEST_VECTOR((28, 14, 172, 219, 166, 71, 38, 243, 72, 100, 245, 187, 160, 22, 14, 202, 219, 182, 195, 88, 254, 56, 235, 61, 225, 137, 223, 253), (43, 209, 144, 87, 201, 204, 99, 43, 217, 221, 58, 42, 254, 6, 216, 18, 61, 106, 148, 90, 7, 83, 126, 142, 117, 89, 180, 88, 104, 32, 64, 236, 156, 129, 16, 237, 16, 46, 100, 190, 159, 156, 31, 235, 14, 108, 19, 201, 196, 60, 70, 8, 208, 215, 232, 17, 215, 40, 190, 245), (231, 245, 216, 251, 119, 143, 238, 148, 7, 128, 86, 69, 95, 121, 4, 134), GNUTLS_MAC_SHA224),
	TEST_VECTOR((95, 32, 17, 138, 134, 6, 231, 10, 235, 188, 250, 27, 192, 122, 57, 85, 229, 68, 204, 236, 21, 114, 87, 174, 51, 127, 221, 135), (184, 189, 171, 31, 17, 178, 136, 3, 148, 253, 217, 134, 0, 128, 110, 34, 133, 126, 148, 210, 38, 33, 29, 84, 242, 67, 151, 93, 200, 171, 233, 13, 215, 228, 151, 9, 9, 183, 90, 216, 46, 57, 182, 104, 142, 240, 108, 144, 0, 246, 122, 73, 140, 180, 5, 178, 25, 161, 84, 149), (36, 153, 127, 65, 62, 78, 220, 93, 109, 108, 10, 99, 23, 10, 96, 222), GNUTLS_MAC_SHA224),
	TEST_VECTOR((153, 152, 189, 102, 158, 4, 52, 158, 122, 18, 10, 42, 58, 63, 123, 197, 76, 212, 40, 159, 185, 249, 65, 46, 18, 15, 176, 27), (10, 146, 159, 60, 155, 130, 125, 81, 181, 118, 60, 253, 24, 6, 54, 125, 171, 173, 2, 52, 121, 252, 11, 174, 192, 60, 165, 116, 182, 253, 115, 210, 53, 242, 134, 9, 45, 202, 217, 9, 37, 236, 111, 254, 146, 239, 54, 54, 165, 196, 208, 13, 171, 237, 34, 114, 134, 33, 177, 40), (168, 230, 112, 255, 205, 38, 6, 169, 114, 47, 27, 67, 238, 130, 244, 122), GNUTLS_MAC_SHA224),
	TEST_VECTOR((49, 253, 25, 83, 148, 228, 8, 0, 136, 179, 132, 49, 58, 28, 122, 181, 209, 102, 10, 137, 124, 176, 48, 166, 226, 233, 191, 54), (242, 151, 30, 242, 125, 82, 46, 172, 83, 95, 172, 153, 89, 4, 255, 169, 58, 33, 39, 49, 122, 161, 141, 211, 108, 221, 29, 100, 52, 189, 103, 41, 209, 219, 225, 45, 120, 23, 164, 246, 172, 205, 192, 195, 77, 90, 82, 124, 154, 200, 89, 84, 163, 92, 120, 15, 204, 141, 190, 97), (242, 103, 161, 172, 39, 230, 0, 23, 64, 156, 131, 162, 100, 223, 81, 83), GNUTLS_MAC_SHA224),
	TEST_VECTOR((186, 109, 105, 219, 59, 202, 224, 221, 194, 72, 204, 160, 23, 182, 201, 239, 99, 165, 22, 241, 219, 77, 232, 66, 6, 145, 198, 43), (123, 109, 37, 156, 169, 74, 18, 237, 87, 239, 226, 181, 13, 42, 200, 242, 179, 30, 216, 75, 72, 97, 239, 115, 82, 175, 105, 218, 31, 117, 143, 186, 166, 151, 130, 59, 94, 102, 109, 221, 151, 107, 173, 170, 44, 179, 45, 128, 123, 81, 12, 185, 125, 254, 130, 204, 83, 160, 217, 191), (190, 150, 103, 242, 111, 141, 36, 155, 30, 59, 217, 144, 190, 159, 184, 82), GNUTLS_MAC_SHA224),
	TEST_VECTOR((161, 198, 96, 83, 219, 249, 100, 199, 91, 186, 102, 43, 169, 129, 103, 235, 17, 134, 184, 90, 111, 5, 135, 93, 247, 135, 12, 154), (213, 139, 117, 66, 209, 8, 64, 214, 159, 13, 127, 94, 38, 46, 17, 5, 96, 102, 213, 108, 141, 164, 100, 41, 180, 184, 202, 48, 109, 157, 149, 130, 151, 245, 21, 59, 222, 97, 80, 178, 190, 10, 29, 104, 184, 5, 93, 104, 254, 82, 215, 163, 95, 241, 218, 42, 211, 73, 32, 140), (224, 163, 62, 113, 67, 137, 177, 130, 143, 222, 238, 220, 250, 214, 205, 216), GNUTLS_MAC_SHA224),
	TEST_VECTOR((153, 40, 21, 18, 29, 136, 255, 178, 108, 51, 118, 6, 114, 60, 2, 239, 49, 119, 19, 8, 110, 44, 251, 189, 55, 225, 161, 103), (21, 45, 151, 78, 178, 113, 155, 144, 39, 211, 32, 84, 163, 39, 49, 35, 97, 18, 89, 89, 223, 157, 150, 161, 131, 46, 32, 86, 194, 87, 29, 79, 28, 244, 95, 110, 143, 101, 68, 200, 127, 21, 134, 28, 239, 98, 125, 47, 22, 233, 176, 180, 171, 121, 155, 179, 54, 47, 74, 174), (71, 94, 218, 58, 50, 213, 105, 147, 46, 4, 61, 182, 77, 191, 14, 155, 176, 148, 91, 84, 220, 223, 162, 3, 190, 26, 40, 82, 76, 20, 112, 117), GNUTLS_MAC_SHA224),
	TEST_VECTOR((204, 252, 133, 156, 17, 163, 224, 158, 27, 4, 11, 112, 31, 216, 185, 98, 169, 38, 160, 144, 17, 123, 170, 168, 12, 131, 226, 236), (137, 31, 7, 134, 138, 182, 166, 197, 73, 142, 123, 232, 91, 30, 61, 91, 175, 60, 65, 92, 245, 224, 174, 183, 201, 64, 57, 83, 175, 100, 146, 165, 215, 136, 96, 253, 192, 246, 27, 162, 196, 175, 85, 188, 61, 80, 79, 161, 199, 180, 10, 208, 119, 143, 99, 247, 155, 22, 75, 100), (173, 183, 216, 27, 163, 218, 100, 163, 196, 158, 48, 177, 232, 102, 228, 139, 13, 0, 9, 174, 143, 56, 80, 74, 74, 89, 37, 72, 76, 17, 86, 20), GNUTLS_MAC_SHA224),
	TEST_VECTOR((160, 99, 19, 24, 225, 141, 217, 102, 182, 236, 81, 231, 170, 121, 172, 107, 249, 182, 62, 50, 30, 166, 95, 87, 109, 224, 98, 86), (235, 241, 246, 22, 205, 113, 134, 232, 45, 229, 44, 132, 59, 98, 118, 147, 156, 104, 116, 212, 43, 129, 138, 203, 79, 242, 135, 53, 70, 200, 234, 252, 216, 69, 39, 55, 86, 243, 193, 36, 108, 21, 30, 156, 16, 229, 7, 172, 33, 103, 10, 81, 126, 10, 62, 14, 65, 168, 119, 3), (43, 142, 231, 61, 24, 51, 185, 60, 139, 254, 39, 248, 102, 225, 131, 107, 140, 67, 63, 79, 156, 238, 174, 34, 65, 78, 167, 18, 174, 38, 61, 126), GNUTLS_MAC_SHA224),
	TEST_VECTOR((50, 97, 88, 163, 77, 214, 152, 134, 134, 106, 238, 189, 223, 222, 1, 42, 150, 108, 237, 232, 88, 139, 106, 87, 238, 98, 222, 40), (106, 247, 213, 208, 115, 36, 102, 185, 125, 4, 110, 222, 189, 227, 20, 142, 131, 75, 25, 147, 128, 189, 111, 153, 240, 59, 222, 38, 240, 219, 141, 27, 54, 255, 74, 221, 141, 137, 7, 110, 229, 177, 246, 131, 56, 237, 83, 100, 188, 134, 45, 102, 72, 26, 190, 35, 213, 150, 195, 251), (167, 250, 155, 239, 18, 56, 75, 158, 66, 93, 153, 90, 193, 211, 197, 209, 170, 52, 168, 10, 141, 176, 222, 237, 103, 49, 108, 208, 52, 54, 37, 175), GNUTLS_MAC_SHA224),
	TEST_VECTOR((69, 97, 233, 66, 87, 40, 130, 253, 247, 131, 16, 96, 60, 62, 158, 166, 11, 243, 54, 246, 132, 24, 97, 169, 82, 198, 200, 203), (11, 165, 78, 112, 121, 185, 179, 239, 85, 203, 223, 220, 89, 222, 154, 182, 8, 55, 101, 65, 150, 193, 126, 202, 220, 60, 71, 180, 201, 255, 22, 26, 34, 108, 76, 222, 181, 173, 241, 65, 182, 113, 181, 10, 131, 88, 88, 182, 66, 59, 253, 201, 194, 228, 28, 81, 195, 207, 64, 125), (25, 111, 0, 168, 97, 198, 196, 15, 220, 188, 200, 78, 117, 148, 172, 227, 45, 145, 95, 126, 168, 55, 2, 236, 229, 99, 93, 134, 87, 212, 201, 57), GNUTLS_MAC_SHA224),
	TEST_VECTOR((116, 83, 172, 14, 157, 27, 46, 160, 84, 121, 230, 22, 241, 188, 135, 128, 100, 185, 212, 177, 237, 217, 151, 132, 28, 241, 40, 177), (234, 208, 197, 163, 67, 216, 222, 90, 98, 235, 87, 115, 27, 100, 176, 183, 233, 52, 215, 92, 91, 98, 236, 104, 60, 191, 97, 119, 107, 250, 119, 242, 13, 168, 220, 110, 40, 123, 31, 73, 24, 177, 223, 191, 64, 42, 249, 53, 37, 139, 17, 3, 30, 161, 7, 126, 90, 119, 91, 247), (74, 104, 200, 39, 103, 158, 181, 196, 91, 97, 161, 18, 128, 13, 156, 215, 91, 64, 232, 151, 94, 58, 54, 196, 81, 54, 65, 142, 32, 194, 254, 220), GNUTLS_MAC_SHA224),
	TEST_VECTOR((107, 63, 132, 194, 201, 169, 118, 34, 157, 83, 49, 114, 54, 11, 250, 74, 50, 130, 181, 123, 95, 97, 230, 76, 96, 1, 207, 55), (213, 242, 220, 142, 164, 227, 73, 219, 4, 7, 91, 147, 107, 5, 106, 153, 90, 246, 161, 11, 245, 108, 138, 16, 177, 255, 156, 191, 216, 32, 35, 253, 116, 30, 215, 228, 71, 20, 82, 107, 13, 140, 207, 78, 108, 87, 133, 73, 17, 135, 17, 84, 163, 173, 224, 109, 213, 168, 192, 254), (168, 171, 186, 169, 128, 201, 227, 194, 68, 119, 21, 94, 194, 167, 144, 87, 139, 30, 58, 92, 143, 7, 203, 178, 181, 44, 170, 96, 87, 212, 115, 119), GNUTLS_MAC_SHA224),
	TEST_VECTOR((1, 190, 172, 81, 27, 4, 117, 219, 238, 76, 209, 245, 229, 232, 223, 49, 17, 36, 196, 90, 218, 194, 23, 204, 242, 153, 109, 171), (148, 93, 168, 242, 142, 56, 203, 9, 231, 182, 19, 14, 140, 68, 219, 79, 196, 22, 30, 172, 51, 207, 42, 250, 32, 235, 123, 21, 224, 114, 65, 157, 250, 225, 182, 149, 120, 140, 3, 206, 133, 24, 185, 233, 215, 186, 62, 250, 210, 26, 169, 253, 51, 58, 109, 37, 233, 180, 219, 231), (162, 67, 76, 12, 10, 226, 30, 122, 191, 125, 86, 228, 117, 235, 73, 196, 59, 83, 254, 189, 32, 35, 24, 95, 133, 31, 196, 104, 204, 159, 108, 188), GNUTLS_MAC_SHA224),
	TEST_VECTOR((102, 10, 1, 196, 68, 13, 209, 201, 135, 88, 189, 0, 173, 157, 169, 91, 221, 162, 96, 106, 186, 3, 43, 208, 80, 235, 40, 70), (84, 172, 200, 120, 2, 153, 59, 176, 103, 207, 233, 147, 181, 235, 129, 23, 213, 214, 53, 42, 226, 12, 213, 135, 194, 84, 33, 93, 230, 73, 220, 16, 10, 138, 30, 194, 245, 141, 211, 94, 144, 116, 237, 89, 47, 73, 170, 246, 127, 151, 26, 73, 34, 64, 220, 218, 181, 210, 91, 131), (226, 126, 33, 235, 131, 63, 54, 83, 45, 22, 194, 112, 186, 50, 252, 213, 219, 90, 236, 215, 27, 155, 93, 116, 128, 93, 169, 65, 12, 189, 160, 162), GNUTLS_MAC_SHA224),
	TEST_VECTOR((7, 113, 55, 43, 169, 141, 231, 230, 214, 242, 11, 69, 194, 19, 107, 232, 248, 236, 100, 163, 109, 6, 77, 243, 72, 199, 109, 175), (222, 71, 118, 5, 141, 235, 142, 24, 40, 91, 63, 213, 116, 90, 206, 152, 238, 172, 74, 28, 248, 139, 195, 67, 132, 178, 153, 22, 138, 57, 71, 132, 184, 242, 205, 76, 146, 203, 92, 7, 234, 50, 105, 91, 145, 254, 64, 50, 132, 141, 104, 19, 27, 0, 166, 204, 9, 134, 206, 119), (181, 246, 182, 64, 69, 134, 91, 96, 253, 240, 28, 66, 157, 187, 50, 118, 54, 204, 62, 215, 42, 168, 191, 169, 160, 175, 162, 245, 138, 74, 127, 147), GNUTLS_MAC_SHA224),
	TEST_VECTOR((46, 171, 182, 185, 34, 194, 67, 38, 239, 154, 227, 193, 146, 223, 211, 65, 202, 245, 126, 254, 21, 221, 100, 151, 114, 162, 172, 59), (199, 95, 111, 90, 21, 97, 170, 179, 158, 160, 226, 39, 2, 166, 207, 125, 186, 60, 164, 221, 159, 4, 107, 176, 171, 234, 45, 50, 132, 22, 143, 217, 251, 57, 255, 114, 85, 35, 166, 96, 210, 31, 140, 42, 222, 3, 209, 141, 66, 115, 197, 47, 182, 242, 44, 158, 57, 214, 188, 46), (174, 80, 172, 235, 227, 8, 161, 207, 23, 71, 185, 177, 120, 160, 114, 7, 72, 250, 95, 229), GNUTLS_MAC_SHA224),
	TEST_VECTOR((90, 172, 206, 27, 112, 92, 73, 162, 14, 46, 51, 150, 177, 241, 143, 68, 33, 224, 119, 81, 117, 52, 221, 63, 81, 133, 246, 4), (136, 169, 98, 207, 214, 204, 128, 130, 142, 183, 16, 56, 134, 48, 160, 24, 2, 47, 223, 137, 21, 72, 227, 82, 137, 124, 114, 13, 166, 82, 35, 251, 42, 97, 78, 105, 44, 91, 208, 48, 218, 29, 0, 201, 198, 186, 209, 154, 186, 248, 121, 240, 93, 107, 92, 45, 220, 223, 206, 205), (169, 74, 159, 161, 238, 242, 248, 33, 28, 49, 38, 1, 91, 44, 185, 184, 15, 119, 239, 191), GNUTLS_MAC_SHA224),
	TEST_VECTOR((60, 22, 159, 185, 94, 242, 122, 65, 241, 159, 145, 64, 115, 2, 34, 251, 239, 57, 88, 72, 247, 171, 250, 24, 255, 208, 11, 230), (5, 79, 125, 25, 214, 11, 39, 197, 23, 113, 204, 210, 12, 62, 137, 65, 11, 84, 246, 163, 119, 243, 226, 235, 85, 60, 71, 199, 63, 98, 43, 72, 12, 96, 135, 51, 94, 153, 35, 133, 121, 210, 244, 32, 192, 109, 97, 236, 183, 205, 202, 49, 71, 218, 77, 241, 169, 199, 78, 10), (124, 135, 8, 211, 123, 107, 205, 143, 172, 230, 244, 60, 223, 169, 41, 72, 191, 15, 34, 163), GNUTLS_MAC_SHA224),
	TEST_VECTOR((155, 111, 157, 66, 186, 39, 28, 248, 8, 181, 233, 126, 169, 243, 159, 33, 194, 11, 162, 232, 45, 22, 134, 145, 172, 206, 59, 126), (187, 150, 215, 156, 62, 129, 32, 119, 172, 184, 91, 122, 174, 212, 31, 157, 33, 19, 224, 31, 251, 184, 10, 198, 211, 170, 162, 42, 144, 115, 5, 240, 236, 48, 158, 125, 210, 21, 91, 81, 21, 75, 138, 152, 95, 219, 65, 171, 40, 144, 205, 92, 140, 36, 145, 173, 94, 80, 78, 18), (155, 248, 92, 3, 89, 96, 153, 210, 60, 137, 249, 205, 223, 158, 130, 53, 77, 194, 101, 118), GNUTLS_MAC_SHA224),
	TEST_VECTOR((7, 84, 7, 192, 89, 224, 205, 73, 47, 26, 21, 119, 190, 166, 196, 233, 209, 164, 60, 232, 201, 103, 213, 47, 1, 239, 197, 12), (254, 120, 159, 253, 205, 132, 249, 56, 76, 208, 168, 137, 52, 98, 169, 15, 66, 53, 221, 188, 246, 83, 244, 91, 250, 218, 171, 199, 187, 95, 105, 234, 107, 228, 209, 175, 134, 114, 137, 8, 127, 61, 194, 225, 158, 197, 3, 245, 46, 217, 55, 181, 78, 158, 138, 245, 191, 111, 18, 142), (18, 182, 202, 77, 191, 133, 151, 31, 226, 8, 85, 235, 233, 88, 105, 254, 246, 19, 193, 172), GNUTLS_MAC_SHA224),
	TEST_VECTOR((80, 101, 183, 139, 253, 172, 84, 36, 107, 133, 15, 218, 189, 249, 43, 203, 249, 29, 70, 114, 134, 172, 179, 63, 158, 176, 129, 128), (244, 92, 16, 217, 57, 143, 0, 95, 135, 255, 28, 72, 63, 139, 158, 87, 40, 12, 192, 159, 100, 96, 59, 204, 94, 204, 112, 116, 237, 195, 76, 94, 188, 10, 211, 167, 236, 205, 81, 246, 94, 62, 196, 237, 158, 79, 176, 212, 203, 72, 204, 234, 152, 172, 27, 94, 39, 86, 167, 229), (66, 13, 81, 47, 134, 72, 121, 93, 47, 154, 168, 132, 76, 53, 33, 28, 94, 99, 153, 64), GNUTLS_MAC_SHA224),
	TEST_VECTOR((209, 166, 85, 147, 197, 225, 7, 162, 196, 79, 169, 167, 110, 223, 74, 30, 175, 139, 149, 70, 196, 0, 164, 173, 223, 31, 212, 78), (251, 220, 27, 69, 223, 204, 0, 238, 71, 184, 35, 234, 31, 175, 245, 105, 172, 129, 232, 157, 86, 165, 156, 153, 36, 94, 143, 195, 89, 75, 120, 64, 164, 68, 64, 168, 76, 251, 21, 112, 85, 33, 157, 152, 122, 27, 124, 151, 211, 207, 132, 99, 112, 80, 31, 216, 90, 86, 84, 130), (4, 188, 187, 160, 129, 164, 77, 160, 82, 96, 23, 29, 18, 164, 199, 11, 97, 151, 169, 37), GNUTLS_MAC_SHA224),
	TEST_VECTOR((140, 9, 255, 88, 185, 164, 80, 180, 185, 55, 122, 13, 197, 185, 37, 154, 153, 147, 226, 16, 187, 151, 223, 209, 176, 94, 201, 67), (76, 19, 176, 166, 238, 8, 224, 226, 136, 244, 25, 110, 153, 103, 41, 112, 28, 8, 223, 183, 232, 41, 27, 162, 234, 237, 247, 252, 254, 2, 26, 130, 198, 146, 56, 155, 25, 48, 183, 222, 15, 15, 131, 133, 242, 20, 216, 21, 176, 252, 122, 173, 54, 232, 9, 251, 174, 24, 28, 255), (73, 134, 6, 233, 153, 171, 237, 28, 172, 115, 36, 78, 28, 147, 88, 30, 225, 222, 123, 172), GNUTLS_MAC_SHA224),
	TEST_VECTOR((231, 220, 195, 208, 15, 63, 166, 138, 174, 99, 103, 57, 71, 249, 85, 25, 157, 21, 220, 60, 39, 209, 98, 202, 247, 161, 3, 79), (126, 200, 174, 219, 77, 110, 43, 116, 113, 49, 141, 115, 35, 22, 158, 20, 47, 208, 89, 25, 184, 126, 43, 198, 61, 255, 150, 30, 16, 60, 32, 253, 34, 34, 245, 58, 144, 215, 10, 227, 147, 122, 110, 113, 155, 10, 37, 137, 230, 190, 83, 84, 239, 40, 245, 60, 185, 39, 94, 178), (247, 200, 71, 187, 199, 240, 142, 233, 85, 38, 202, 208, 43, 170, 37, 111, 182, 231, 101, 245), GNUTLS_MAC_SHA224),
	TEST_VECTOR((198, 196, 25, 182, 68, 22, 44, 124, 79, 111, 66, 93, 193, 169, 152, 72, 111, 160, 187, 255, 160, 199, 158, 186, 253, 183, 27, 244), (21, 228, 118, 107, 113, 67, 102, 70, 238, 174, 45, 27, 192, 0, 182, 63, 192, 225, 93, 48, 184, 203, 148, 56, 64, 22, 196, 196, 124, 7, 46, 42, 236, 248, 120, 127, 21, 176, 112, 85, 255, 177, 240, 121, 193, 69, 11, 46, 71, 108, 140, 13, 77, 45, 22, 113, 137, 193, 128, 79), (52, 170, 203, 83, 46, 59, 128, 215, 101, 204, 80, 116, 107, 211, 158, 181, 196, 116, 186, 89), GNUTLS_MAC_SHA224),
	TEST_VECTOR((155, 117, 231, 250, 33, 108, 136, 64, 55, 199, 214, 149, 48, 146, 237, 51, 92, 78, 253, 136, 202, 87, 167, 66, 214, 172, 50, 33), (18, 190, 169, 120, 101, 223, 153, 49, 82, 89, 255, 98, 3, 2, 67, 46, 202, 252, 157, 206, 38, 25, 232, 125, 251, 73, 121, 65, 4, 86, 165, 36, 67, 67, 21, 221, 57, 32, 226, 177, 170, 28, 121, 213, 224, 113, 50, 167, 88, 167, 183, 183, 30, 241, 11, 207, 27, 184, 119, 243), (96, 7, 27, 208, 206, 234, 15, 224, 248, 121, 34, 59, 148, 13, 61, 231, 221, 224, 44, 166, 133, 143, 132, 80, 251, 156, 0, 50, 228, 159, 150, 142, 249, 205, 155, 87, 3, 22, 61, 188), GNUTLS_MAC_SHA224),
	TEST_VECTOR((218, 100, 168, 83, 66, 57, 113, 149, 216, 43, 128, 203, 151, 228, 206, 171, 191, 198, 157, 165, 149, 80, 192, 81, 12, 170, 44, 17), (68, 205, 12, 219, 105, 227, 174, 89, 36, 234, 122, 35, 55, 111, 53, 234, 147, 208, 237, 61, 71, 218, 189, 238, 30, 217, 135, 242, 88, 81, 174, 237, 44, 87, 239, 134, 99, 199, 89, 107, 174, 226, 24, 201, 199, 70, 91, 152, 6, 73, 184, 54, 117, 135, 103, 78, 56, 100, 68, 62), (18, 252, 194, 211, 108, 49, 171, 73, 85, 79, 127, 21, 0, 219, 16, 251, 228, 25, 50, 130, 226, 217, 124, 151, 103, 15, 245, 14, 59, 194, 41, 133, 7, 219, 147, 51, 60, 174, 122, 66), GNUTLS_MAC_SHA224),
	TEST_VECTOR((25, 129, 150, 17, 153, 78, 136, 169, 22, 132, 81, 109, 221, 161, 28, 142, 237, 179, 169, 105, 36, 8, 168, 80, 44, 225, 194, 96), (241, 217, 191, 254, 180, 161, 108, 164, 205, 144, 2, 242, 242, 113, 196, 59, 219, 66, 240, 251, 162, 81, 4, 93, 138, 119, 104, 173, 221, 227, 156, 109, 96, 82, 184, 125, 14, 225, 66, 217, 110, 82, 221, 61, 126, 205, 88, 155, 62, 78, 182, 61, 58, 173, 87, 144, 20, 165, 185, 145), (169, 74, 204, 195, 225, 196, 204, 56, 212, 197, 93, 201, 241, 135, 182, 149, 65, 26, 170, 50, 20, 107, 71, 88, 5, 134, 185, 214, 54, 172, 180, 227, 61, 136, 17, 167, 255, 77, 77, 147), GNUTLS_MAC_SHA224),
	TEST_VECTOR((91, 240, 222, 64, 119, 191, 133, 37, 99, 32, 60, 150, 166, 93, 223, 78, 100, 138, 223, 160, 15, 181, 33, 71, 18, 210, 245, 62), (70, 227, 218, 3, 68, 232, 61, 11, 255, 127, 67, 82, 126, 192, 158, 135, 63, 0, 176, 71, 129, 157, 18, 80, 240, 134, 92, 77, 249, 146, 236, 63, 226, 90, 0, 254, 169, 225, 210, 227, 1, 186, 143, 158, 70, 95, 20, 111, 9, 135, 73, 66, 123, 60, 137, 72, 139, 195, 245, 173), (128, 36, 114, 84, 119, 178, 69, 191, 203, 243, 174, 37, 135, 11, 162, 210, 10, 192, 36, 44, 215, 152, 198, 20, 158, 156, 215, 203, 81, 96, 200, 245, 87, 175, 192, 230, 181, 80, 40, 1), GNUTLS_MAC_SHA224),
	TEST_VECTOR((100, 176, 2, 126, 209, 151, 162, 124, 101, 182, 36, 86, 67, 174, 225, 40, 168, 59, 170, 155, 176, 173, 149, 72, 113, 11, 87, 29), (46, 26, 111, 88, 224, 58, 184, 83, 200, 211, 81, 156, 180, 30, 75, 195, 97, 56, 222, 133, 180, 223, 26, 48, 203, 153, 122, 36, 38, 17, 133, 150, 183, 194, 196, 1, 13, 168, 178, 41, 114, 159, 104, 170, 111, 166, 158, 135, 130, 148, 218, 142, 171, 3, 25, 140, 151, 193, 99, 90), (237, 86, 192, 73, 108, 205, 166, 15, 27, 107, 94, 167, 166, 25, 85, 164, 152, 161, 127, 14, 50, 104, 183, 83, 162, 43, 139, 219, 110, 142, 89, 87, 173, 164, 191, 163, 22, 223, 183, 4), GNUTLS_MAC_SHA224),
	TEST_VECTOR((14, 225, 111, 149, 202, 200, 49, 19, 210, 17, 232, 213, 105, 213, 177, 118, 66, 77, 192, 147, 251, 134, 81, 195, 151, 127, 226, 235), (160, 5, 215, 216, 222, 250, 246, 214, 195, 150, 211, 171, 103, 155, 57, 205, 79, 169, 42, 174, 133, 211, 114, 12, 25, 200, 101, 22, 102, 8, 179, 33, 141, 202, 201, 226, 85, 88, 65, 159, 150, 130, 128, 78, 240, 238, 134, 239, 63, 0, 44, 179, 203, 207, 197, 69, 122, 191, 16, 19), (170, 84, 238, 183, 204, 86, 79, 238, 91, 33, 24, 136, 34, 221, 58, 165, 25, 181, 53, 61, 209, 188, 215, 6, 216, 211, 184, 176, 94, 96, 13, 10, 88, 26, 64, 105, 88, 46, 187, 40), GNUTLS_MAC_SHA224),
	TEST_VECTOR((133, 54, 39, 116, 89, 236, 130, 21, 118, 70, 235, 147, 183, 234, 152, 12, 203, 152, 38, 145, 233, 168, 124, 205, 57, 65, 155, 31), (249, 87, 246, 245, 19, 198, 193, 18, 208, 47, 191, 171, 107, 142, 11, 122, 162, 246, 57, 71, 187, 206, 124, 220, 38, 22, 152, 200, 177, 75, 59, 188, 179, 5, 107, 215, 28, 158, 147, 194, 229, 31, 22, 207, 39, 93, 225, 21, 236, 82, 46, 123, 88, 226, 82, 24, 87, 69, 128, 242), (53, 205, 212, 154, 136, 180, 59, 205, 160, 101, 130, 239, 107, 40, 37, 88, 48, 112, 69, 92, 49, 81, 199, 228, 143, 202, 69, 121, 223, 76, 115, 254, 223, 115, 46, 245, 212, 39, 216, 205), GNUTLS_MAC_SHA224),
	TEST_VECTOR((36, 81, 170, 13, 77, 49, 35, 209, 228, 223, 86, 248, 242, 180, 105, 113, 198, 59, 73, 167, 181, 85, 117, 116, 69, 218, 175, 53), (20, 244, 92, 203, 141, 212, 22, 10, 241, 150, 205, 216, 57, 110, 126, 192, 28, 45, 188, 88, 149, 62, 155, 158, 196, 204, 133, 67, 181, 62, 1, 113, 201, 217, 79, 157, 89, 162, 14, 141, 113, 17, 100, 72, 246, 177, 49, 67, 23, 148, 94, 51, 134, 15, 169, 124, 160, 175, 153, 80), (80, 244, 183, 94, 145, 166, 57, 243, 99, 114, 249, 51, 41, 34, 6, 37, 59, 134, 134, 45, 216, 27, 174, 52, 152, 195, 179, 51, 202, 186, 114, 41, 174, 120, 205, 90, 26, 22, 141, 68), GNUTLS_MAC_SHA224),
	TEST_VECTOR((20, 86, 61, 152, 218, 182, 166, 90, 230, 215, 225, 192, 143, 133, 238, 21, 21, 213, 33, 57, 251, 164, 105, 157, 63, 159, 120, 57), (88, 73, 104, 105, 176, 241, 172, 13, 145, 186, 235, 65, 161, 233, 31, 73, 211, 78, 188, 215, 183, 127, 35, 177, 8, 44, 154, 92, 249, 184, 197, 118, 3, 133, 175, 45, 105, 70, 182, 19, 172, 59, 76, 9, 225, 220, 65, 232, 214, 135, 15, 237, 37, 226, 165, 137, 243, 45, 161, 246), (161, 249, 41, 115, 133, 122, 19, 218, 127, 214, 193, 246, 76, 135, 133, 126, 4, 46, 206, 197, 63, 225, 73, 98, 112, 199, 112, 253, 234, 167, 239, 235, 119, 238, 30, 130, 66, 174, 94, 32), GNUTLS_MAC_SHA224),
	TEST_VECTOR((109, 68, 17, 204, 254, 202, 120, 47, 252, 135, 237, 159, 201, 22, 57, 146, 217, 225, 202, 178, 77, 234, 105, 11, 150, 107, 50, 49), (238, 121, 103, 221, 202, 166, 221, 178, 66, 187, 178, 206, 225, 251, 247, 134, 50, 23, 45, 116, 249, 188, 11, 230, 69, 213, 44, 25, 199, 238, 91, 150, 31, 246, 118, 84, 50, 1, 51, 114, 246, 147, 183, 202, 127, 68, 137, 2, 95, 171, 110, 89, 153, 133, 198, 62, 85, 30, 55, 51), (34, 27, 177, 143, 8, 108, 115, 150, 183, 62, 203, 104, 178, 12, 232, 239, 97, 115, 154, 109, 185, 123, 42, 184, 163, 57, 103, 34, 169, 59, 225, 199, 89, 133, 90, 149, 222, 134, 212, 105), GNUTLS_MAC_SHA224),
	TEST_VECTOR((221, 29, 145, 183, 217, 11, 43, 211, 19, 133, 51, 206, 146, 178, 114, 251, 248, 163, 105, 49, 106, 239, 226, 66, 230, 89, 204, 10, 226, 56, 175, 224), (1, 50, 43, 150, 179, 10, 205, 25, 121, 121, 68, 78, 70, 142, 28, 92, 104, 89, 191, 27, 28, 249, 81, 183, 231, 37, 48, 62, 35, 126, 70, 184, 100, 161, 69, 250, 178, 94, 81, 123, 8, 248, 104, 61, 3, 21, 187, 41, 17, 216, 10, 14, 138, 186, 23, 243, 180, 19, 250, 172), (16, 98, 19, 66, 191, 176, 253, 64, 4, 108, 14, 41, 242, 207, 219, 240), GNUTLS_MAC_SHA256),
	TEST_VECTOR((50, 196, 0, 56, 114, 161, 70, 25, 64, 35, 234, 193, 189, 167, 77, 223, 43, 102, 151, 125, 173, 138, 85, 75, 151, 76, 162, 166, 47, 126, 79, 67), (51, 216, 207, 109, 12, 117, 159, 182, 34, 216, 103, 234, 140, 241, 40, 93, 228, 2, 10, 248, 28, 194, 135, 173, 223, 56, 204, 45, 164, 100, 62, 109, 179, 178, 21, 173, 62, 51, 191, 196, 120, 119, 195, 98, 14, 51, 104, 135, 195, 201, 173, 74, 28, 108, 4, 118, 176, 249, 10, 51), (245, 147, 175, 14, 26, 73, 42, 123, 144, 74, 38, 98, 137, 127, 161, 193), GNUTLS_MAC_SHA256),
	TEST_VECTOR((60, 135, 233, 204, 152, 87, 155, 39, 73, 255, 146, 200, 184, 35, 162, 173, 107, 54, 122, 194, 102, 34, 231, 181, 184, 10, 44, 230, 244, 80, 227, 97), (119, 125, 102, 162, 76, 45, 60, 195, 41, 156, 160, 113, 143, 79, 109, 205, 17, 97, 236, 190, 246, 235, 60, 113, 240, 188, 20, 91, 78, 118, 90, 110, 236, 232, 7, 167, 76, 167, 166, 152, 213, 91, 46, 176, 211, 13, 141, 62, 92, 215, 31, 210, 160, 43, 86, 8, 39, 76, 149, 195), (234, 100, 37, 240, 56, 3, 242, 240, 108, 66, 216, 186, 17, 206, 78, 233), GNUTLS_MAC_SHA256),
	TEST_VECTOR((222, 31, 74, 191, 231, 140, 77, 214, 240, 35, 49, 192, 87, 239, 169, 57, 174, 45, 177, 241, 231, 231, 198, 80, 224, 117, 55, 210, 89, 177, 234, 114), (76, 31, 0, 25, 141, 118, 243, 99, 13, 50, 96, 245, 109, 148, 245, 37, 7, 57, 79, 74, 152, 205, 194, 147, 125, 74, 186, 167, 110, 187, 63, 212, 9, 168, 118, 157, 240, 116, 220, 0, 41, 23, 239, 129, 138, 72, 82, 207, 0, 79, 2, 37, 239, 196, 102, 50, 17, 160, 140, 93), (116, 24, 42, 232, 30, 232, 140, 106, 22, 52, 255, 73, 145, 190, 185, 238), GNUTLS_MAC_SHA256),
	TEST_VECTOR((140, 41, 155, 238, 174, 160, 95, 68, 93, 89, 245, 195, 84, 219, 208, 200, 180, 205, 0, 159, 25, 122, 53, 54, 159, 179, 184, 97, 43, 117, 2, 107), (26, 115, 252, 225, 20, 203, 66, 125, 250, 166, 166, 153, 171, 39, 81, 191, 113, 54, 250, 3, 210, 56, 218, 73, 45, 154, 3, 97, 67, 20, 131, 52, 41, 77, 11, 219, 228, 133, 44, 143, 243, 119, 6, 187, 39, 215, 34, 221, 249, 9, 188, 139, 239, 145, 172, 114, 225, 132, 28, 173), (22, 97, 79, 62, 132, 133, 21, 203, 229, 38, 253, 43, 27, 90, 11, 195), GNUTLS_MAC_SHA256),
	TEST_VECTOR((250, 31, 76, 108, 164, 38, 132, 128, 82, 123, 55, 204, 22, 53, 182, 157, 74, 7, 17, 143, 114, 12, 96, 189, 19, 206, 220, 134, 125, 252, 39, 84), (70, 69, 22, 214, 248, 187, 101, 137, 146, 136, 73, 185, 132, 190, 166, 220, 58, 69, 163, 227, 203, 233, 178, 122, 149, 233, 72, 1, 199, 24, 144, 23, 100, 215, 137, 16, 231, 46, 95, 230, 152, 96, 231, 110, 143, 43, 187, 169, 41, 134, 118, 232, 168, 107, 61, 99, 86, 59, 69, 162), (103, 80, 123, 143, 188, 129, 61, 35, 135, 246, 155, 196, 211, 189, 164, 74), GNUTLS_MAC_SHA256),
	TEST_VECTOR((171, 18, 202, 71, 9, 202, 56, 53, 12, 175, 86, 2, 238, 229, 33, 142, 201, 80, 53, 61, 25, 230, 93, 233, 239, 196, 220, 45, 25, 254, 48, 23), (168, 218, 91, 37, 228, 242, 146, 193, 73, 200, 143, 146, 3, 197, 55, 8, 34, 25, 60, 218, 193, 53, 251, 205, 107, 3, 244, 35, 0, 184, 195, 114, 246, 133, 32, 221, 59, 82, 92, 121, 170, 37, 242, 80, 183, 134, 230, 222, 127, 93, 115, 181, 251, 70, 201, 135, 103, 28, 127, 118), (167, 244, 65, 135, 212, 235, 231, 89, 185, 163, 126, 72, 74, 132, 78, 43), GNUTLS_MAC_SHA256),
	TEST_VECTOR((18, 169, 196, 179, 146, 27, 78, 194, 117, 78, 28, 240, 147, 163, 138, 152, 112, 42, 207, 11, 142, 179, 15, 75, 246, 84, 184, 227, 161, 13, 57, 144), (197, 23, 14, 110, 103, 204, 239, 235, 232, 65, 94, 226, 251, 100, 41, 223, 55, 214, 194, 238, 143, 189, 214, 185, 112, 195, 169, 141, 72, 110, 135, 24, 194, 32, 47, 127, 192, 159, 228, 56, 213, 61, 253, 174, 235, 8, 116, 239, 15, 215, 180, 223, 162, 9, 204, 156, 92, 81, 43, 170), (157, 26, 49, 33, 118, 11, 23, 234, 120, 126, 15, 100, 201, 11, 241, 9), GNUTLS_MAC_SHA256),
	TEST_VECTOR((33, 139, 15, 118, 152, 9, 84, 204, 56, 31, 38, 70, 99, 105, 7, 231, 2, 7, 143, 121, 2, 161, 137, 73, 82, 150, 101, 71, 172, 233, 27, 25), (22, 203, 139, 237, 179, 99, 180, 121, 81, 83, 161, 5, 198, 4, 146, 145, 241, 151, 142, 123, 42, 171, 1, 171, 100, 226, 156, 155, 181, 98, 65, 140, 243, 171, 79, 30, 230, 17, 29, 94, 210, 229, 142, 190, 58, 217, 102, 85, 136, 224, 228, 217, 222, 174, 133, 36, 181, 183, 158, 216), (128, 23, 95, 213, 201, 202, 37, 44, 82, 189, 203, 83, 2, 222, 61, 177), GNUTLS_MAC_SHA256),
	TEST_VECTOR((52, 51, 242, 197, 56, 36, 214, 238, 190, 17, 225, 30, 182, 86, 218, 151, 64, 197, 163, 66, 245, 118, 157, 247, 254, 23, 196, 196, 128, 17, 50, 202), (7, 227, 248, 255, 3, 230, 175, 90, 173, 80, 60, 172, 177, 219, 17, 157, 49, 120, 187, 211, 226, 55, 120, 136, 214, 245, 230, 183, 191, 123, 143, 124, 86, 58, 136, 170, 138, 119, 136, 72, 244, 220, 1, 178, 156, 175, 133, 163, 178, 48, 126, 60, 223, 227, 222, 30, 112, 67, 221, 239), (174, 129, 145, 108, 211, 100, 28, 89, 137, 117, 18, 100, 155, 101, 114, 82), GNUTLS_MAC_SHA256),
	TEST_VECTOR((226, 4, 214, 212, 102, 170, 213, 7, 255, 175, 109, 109, 171, 10, 91, 38, 21, 44, 158, 33, 231, 100, 55, 4, 100, 227, 96, 200, 251, 199, 101, 198), (123, 3, 185, 141, 159, 148, 184, 153, 229, 145, 243, 239, 38, 75, 113, 177, 147, 251, 167, 4, 60, 126, 149, 60, 222, 35, 188, 83, 132, 188, 26, 98, 147, 88, 1, 21, 250, 227, 73, 95, 216, 69, 218, 219, 208, 43, 214, 69, 92, 244, 141, 15, 98, 179, 62, 98, 54, 74, 58, 128), (119, 13, 250, 182, 166, 164, 164, 190, 224, 37, 127, 243, 53, 33, 63, 120, 216, 40, 123, 79, 213, 55, 213, 193, 255, 250, 149, 105, 16, 231, 199, 121), GNUTLS_MAC_SHA256),
	TEST_VECTOR((174, 238, 202, 96, 246, 137, 164, 65, 177, 59, 12, 188, 212, 65, 216, 45, 240, 207, 135, 218, 194, 54, 41, 13, 236, 232, 147, 29, 248, 215, 3, 23), (88, 142, 192, 65, 229, 115, 59, 112, 49, 33, 44, 85, 56, 239, 228, 246, 170, 250, 76, 218, 139, 146, 93, 38, 31, 90, 38, 136, 240, 7, 179, 172, 36, 14, 225, 41, 145, 231, 123, 140, 184, 83, 134, 120, 97, 89, 102, 22, 74, 129, 135, 43, 209, 207, 203, 251, 57, 164, 244, 80), (62, 129, 214, 17, 60, 238, 60, 82, 158, 206, 223, 248, 154, 105, 153, 206, 37, 182, 24, 193, 94, 225, 209, 157, 69, 203, 55, 106, 28, 142, 35, 116), GNUTLS_MAC_SHA256),
	TEST_VECTOR((149, 200, 247, 110, 17, 54, 126, 181, 85, 38, 162, 179, 147, 174, 144, 101, 131, 209, 203, 221, 71, 150, 33, 70, 245, 6, 204, 124, 172, 18, 244, 100), (202, 214, 14, 144, 75, 158, 156, 139, 254, 180, 168, 26, 127, 103, 211, 189, 220, 192, 94, 100, 37, 88, 112, 64, 55, 112, 243, 83, 58, 230, 221, 99, 76, 234, 165, 108, 83, 230, 136, 189, 19, 122, 230, 1, 137, 53, 243, 75, 159, 176, 132, 234, 72, 228, 198, 136, 246, 187, 179, 136), (202, 250, 92, 160, 63, 95, 190, 42, 36, 32, 4, 171, 203, 211, 222, 16, 89, 199, 64, 123, 30, 229, 121, 37, 81, 36, 175, 24, 155, 224, 181, 86), GNUTLS_MAC_SHA256),
	TEST_VECTOR((77, 5, 57, 31, 214, 251, 30, 41, 46, 120, 171, 150, 25, 177, 183, 42, 125, 99, 238, 89, 215, 67, 93, 215, 24, 151, 185, 255, 126, 231, 174, 112), (240, 120, 230, 249, 183, 248, 45, 100, 85, 79, 166, 182, 4, 200, 8, 241, 155, 31, 106, 214, 114, 125, 183, 170, 111, 28, 134, 105, 78, 16, 75, 82, 86, 200, 180, 3, 153, 25, 100, 100, 129, 215, 234, 36, 82, 199, 44, 23, 163, 232, 215, 211, 145, 98, 133, 70, 10, 165, 235, 129), (107, 22, 232, 245, 59, 131, 26, 165, 232, 107, 249, 122, 92, 79, 163, 125, 8, 155, 193, 114, 218, 90, 30, 127, 102, 45, 212, 165, 149, 51, 154, 183), GNUTLS_MAC_SHA256),
	TEST_VECTOR((15, 104, 168, 47, 241, 103, 22, 52, 204, 145, 54, 197, 100, 169, 224, 42, 118, 118, 33, 221, 116, 161, 191, 92, 36, 18, 155, 128, 130, 20, 183, 82), (100, 133, 153, 128, 156, 44, 78, 124, 106, 94, 108, 68, 159, 0, 49, 235, 245, 92, 54, 97, 168, 149, 180, 77, 176, 87, 46, 232, 128, 131, 177, 244, 177, 38, 2, 170, 85, 252, 29, 241, 80, 166, 90, 109, 110, 237, 160, 170, 121, 164, 52, 161, 3, 155, 145, 181, 165, 143, 199, 241), (226, 151, 100, 15, 119, 104, 72, 93, 74, 110, 124, 254, 36, 95, 139, 250, 132, 112, 13, 153, 118, 38, 146, 234, 26, 66, 92, 204, 2, 117, 232, 245), GNUTLS_MAC_SHA256),
	TEST_VECTOR((67, 238, 246, 216, 36, 253, 130, 4, 5, 98, 106, 185, 182, 215, 159, 31, 208, 78, 18, 106, 184, 225, 119, 41, 227, 175, 199, 203, 90, 247, 148, 248), (94, 38, 155, 90, 123, 222, 220, 195, 232, 117, 226, 114, 86, 147, 162, 87, 252, 96, 1, 26, 247, 220, 214, 138, 51, 88, 80, 127, 226, 155, 6, 89, 202, 102, 149, 29, 170, 5, 161, 80, 50, 3, 54, 80, 188, 88, 162, 120, 64, 248, 251, 233, 244, 8, 139, 144, 48, 115, 143, 104), (240, 163, 57, 236, 188, 174, 106, 221, 26, 251, 39, 218, 59, 164, 10, 19, 32, 198, 66, 122, 88, 175, 185, 220, 54, 107, 33, 155, 126, 178, 158, 207), GNUTLS_MAC_SHA256),
	TEST_VECTOR((229, 243, 29, 152, 161, 63, 35, 144, 179, 84, 219, 160, 142, 30, 133, 17, 111, 153, 165, 108, 46, 135, 97, 211, 134, 149, 138, 13, 10, 136, 138, 41), (2, 17, 63, 69, 21, 27, 99, 243, 116, 207, 205, 177, 190, 222, 65, 206, 242, 34, 106, 66, 182, 192, 44, 159, 9, 15, 159, 61, 179, 157, 78, 152, 168, 37, 140, 66, 226, 114, 36, 39, 156, 212, 92, 37, 1, 202, 69, 160, 8, 216, 243, 137, 21, 229, 180, 91, 139, 153, 95, 91), (152, 231, 160, 35, 9, 42, 48, 100, 5, 9, 2, 200, 185, 12, 116, 157, 114, 0, 86, 38, 224, 41, 110, 29, 251, 40, 193, 14, 69, 11, 45, 211), GNUTLS_MAC_SHA256),
	TEST_VECTOR((230, 207, 164, 134, 77, 49, 254, 9, 150, 15, 255, 150, 138, 198, 47, 3, 182, 246, 59, 90, 34, 28, 201, 92, 58, 16, 88, 180, 182, 15, 233, 188), (70, 112, 167, 194, 200, 245, 100, 59, 117, 234, 76, 238, 216, 126, 37, 62, 88, 255, 170, 135, 71, 34, 153, 22, 13, 53, 36, 7, 83, 243, 22, 76, 8, 32, 55, 75, 31, 75, 237, 178, 220, 52, 105, 44, 139, 126, 6, 199, 149, 30, 231, 63, 22, 69, 177, 14, 63, 39, 45, 23), (79, 32, 142, 115, 6, 176, 118, 191, 6, 19, 63, 67, 154, 102, 23, 163, 214, 80, 37, 60, 248, 119, 117, 195, 214, 215, 254, 222, 50, 19, 159, 79), GNUTLS_MAC_SHA256),
	TEST_VECTOR((47, 24, 11, 26, 16, 68, 93, 61, 41, 104, 59, 63, 172, 184, 86, 128, 118, 137, 198, 222, 84, 199, 96, 246, 5, 12, 51, 41, 164, 161, 222, 78), (77, 72, 202, 73, 162, 121, 167, 155, 249, 179, 169, 227, 70, 195, 175, 116, 146, 111, 171, 106, 216, 129, 2, 125, 238, 106, 111, 64, 220, 246, 122, 221, 4, 239, 236, 77, 134, 223, 49, 187, 252, 25, 13, 67, 227, 167, 174, 234, 155, 171, 206, 186, 234, 211, 176, 123, 105, 221, 61, 110), (119, 101, 210, 69, 253, 241, 67, 182, 202, 67, 89, 162, 80, 58, 209, 216, 234, 140, 170, 125, 168, 213, 86, 177, 254, 142, 37, 196, 76, 112, 219, 225), GNUTLS_MAC_SHA256),
	TEST_VECTOR((46, 176, 164, 159, 211, 25, 154, 87, 38, 79, 116, 107, 28, 138, 203, 199, 111, 124, 229, 18, 35, 247, 33, 52, 89, 15, 187, 235, 49, 118, 38, 75), (152, 10, 144, 142, 211, 139, 110, 105, 157, 243, 244, 78, 101, 31, 10, 211, 169, 210, 9, 211, 134, 124, 73, 94, 82, 103, 56, 85, 208, 158, 79, 26, 88, 251, 71, 112, 103, 196, 0, 202, 204, 169, 171, 146, 96, 226, 5, 196, 85, 105, 5, 242, 114, 121, 37, 86, 18, 128, 99, 156), (6, 213, 173, 194, 213, 197, 23, 191, 64, 64, 108, 198, 187, 86, 85, 59, 34, 47, 112, 171, 242, 187, 80, 85, 132, 0, 14, 136, 98, 139, 175, 23), GNUTLS_MAC_SHA256),
	TEST_VECTOR((220, 96, 51, 141, 136, 78, 236, 183, 41, 117, 198, 3, 194, 123, 54, 6, 5, 1, 23, 86, 198, 151, 196, 252, 56, 143, 81, 118, 239, 129, 239, 177), (68, 215, 170, 8, 254, 186, 38, 9, 60, 20, 151, 156, 18, 44, 36, 55, 195, 17, 123, 99, 183, 136, 65, 205, 16, 164, 188, 94, 213, 92, 86, 88, 106, 216, 152, 109, 85, 48, 125, 202, 29, 25, 142, 220, 255, 188, 81, 106, 143, 190, 97, 82, 170, 66, 140, 221, 128, 12, 6, 45), (41, 172, 7, 220, 207, 31, 40, 213, 6, 205, 98, 62, 110, 63, 194, 250, 37, 91, 214, 11), GNUTLS_MAC_SHA256),
	TEST_VECTOR((122, 126, 206, 228, 240, 76, 31, 84, 83, 242, 155, 140, 101, 190, 233, 9, 246, 115, 196, 79, 101, 232, 249, 204, 24, 195, 28, 50, 233, 188, 252, 90), (14, 43, 83, 221, 99, 0, 142, 6, 99, 150, 42, 37, 218, 156, 213, 95, 194, 234, 55, 113, 72, 120, 61, 162, 41, 255, 126, 59, 214, 20, 42, 67, 200, 84, 182, 181, 208, 109, 135, 181, 53, 147, 111, 30, 220, 124, 208, 103, 232, 219, 186, 34, 10, 31, 154, 89, 50, 179, 42, 100), (150, 251, 142, 249, 56, 10, 201, 222, 39, 17, 239, 90, 131, 36, 158, 96, 141, 199, 191, 252), GNUTLS_MAC_SHA256),
	TEST_VECTOR((222, 113, 41, 93, 197, 10, 199, 110, 181, 87, 148, 16, 134, 158, 145, 139, 123, 231, 87, 175, 166, 6, 197, 9, 190, 67, 120, 189, 152, 237, 166, 134), (51, 202, 151, 79, 138, 26, 6, 91, 117, 9, 12, 52, 201, 72, 68, 153, 16, 73, 86, 17, 226, 142, 204, 98, 206, 210, 158, 91, 58, 231, 98, 23, 225, 57, 38, 112, 65, 186, 64, 190, 35, 93, 225, 48, 67, 140, 27, 20, 170, 131, 50, 150, 235, 142, 75, 171, 226, 16, 16, 16), (56, 93, 96, 83, 128, 144, 164, 90, 43, 37, 68, 39, 89, 5, 196, 193, 110, 159, 35, 226), GNUTLS_MAC_SHA256),
	TEST_VECTOR((56, 157, 224, 185, 20, 102, 28, 138, 201, 170, 241, 29, 38, 31, 98, 97, 186, 244, 101, 40, 134, 207, 32, 210, 177, 61, 246, 123, 226, 227, 177, 133), (146, 179, 212, 126, 160, 66, 89, 29, 181, 181, 49, 144, 126, 9, 164, 90, 96, 169, 197, 197, 254, 2, 81, 128, 107, 120, 5, 182, 65, 197, 179, 235, 205, 225, 77, 108, 181, 66, 180, 203, 36, 43, 4, 245, 169, 182, 11, 44, 102, 209, 162, 76, 102, 20, 31, 224, 184, 24, 233, 60), (177, 42, 78, 32, 1, 128, 210, 13, 164, 4, 180, 76, 149, 38, 57, 169, 85, 221, 131, 208), GNUTLS_MAC_SHA256),
	TEST_VECTOR((49, 26, 247, 56, 116, 225, 58, 138, 202, 217, 129, 73, 1, 19, 147, 64, 101, 179, 189, 93, 68, 142, 43, 184, 223, 166, 139, 112, 198, 157, 125, 69), (234, 32, 252, 157, 50, 205, 220, 120, 220, 188, 162, 234, 214, 197, 198, 103, 68, 218, 133, 217, 91, 100, 61, 63, 250, 178, 208, 226, 213, 103, 125, 211, 162, 115, 19, 21, 59, 1, 156, 252, 211, 59, 62, 48, 94, 214, 100, 4, 4, 43, 45, 176, 227, 222, 34, 103, 203, 85, 127, 216), (198, 232, 109, 16, 67, 51, 63, 182, 144, 173, 35, 39, 74, 144, 130, 4, 214, 188, 187, 172), GNUTLS_MAC_SHA256),
	TEST_VECTOR((117, 111, 153, 128, 203, 239, 234, 57, 131, 80, 184, 134, 202, 76, 27, 41, 16, 112, 139, 91, 129, 84, 160, 236, 75, 150, 72, 172, 119, 185, 215, 220), (63, 9, 170, 21, 210, 250, 118, 156, 110, 138, 51, 128, 188, 85, 132, 72, 39, 186, 62, 166, 76, 207, 23, 123, 235, 75, 251, 213, 20, 43, 57, 99, 191, 105, 104, 3, 168, 153, 116, 170, 125, 90, 240, 25, 44, 213, 37, 168, 59, 113, 205, 142, 231, 176, 188, 146, 240, 123, 149, 21), (227, 188, 98, 179, 138, 123, 60, 126, 127, 203, 158, 240, 7, 170, 74, 214, 169, 187, 81, 156), GNUTLS_MAC_SHA256),
	TEST_VECTOR((174, 129, 91, 254, 34, 4, 7, 188, 230, 99, 143, 32, 174, 250, 16, 155, 99, 199, 56, 46, 145, 215, 187, 139, 1, 14, 215, 198, 216, 211, 117, 125), (162, 33, 241, 163, 98, 62, 202, 95, 99, 133, 181, 126, 122, 254, 103, 209, 52, 1, 28, 96, 88, 227, 151, 125, 249, 119, 189, 240, 199, 171, 14, 20, 182, 213, 192, 89, 243, 153, 72, 152, 41, 18, 176, 71, 208, 1, 3, 220, 72, 54, 229, 155, 122, 71, 2, 34, 219, 174, 114, 202), (237, 94, 135, 109, 118, 34, 125, 10, 127, 26, 207, 92, 160, 140, 129, 41, 149, 48, 63, 178), GNUTLS_MAC_SHA256),
	TEST_VECTOR((213, 17, 12, 128, 138, 149, 28, 95, 227, 106, 164, 133, 47, 189, 231, 224, 188, 55, 42, 44, 105, 163, 90, 207, 200, 144, 204, 159, 247, 142, 64, 251), (244, 81, 135, 7, 42, 125, 120, 254, 145, 40, 47, 88, 37, 218, 235, 37, 106, 40, 168, 24, 199, 10, 40, 82, 98, 176, 128, 205, 62, 226, 236, 120, 81, 37, 178, 126, 64, 38, 172, 150, 136, 165, 234, 230, 87, 219, 87, 140, 210, 7, 149, 98, 73, 240, 74, 6, 72, 112, 214, 119), (14, 125, 226, 95, 197, 89, 150, 156, 8, 217, 115, 171, 64, 121, 93, 247, 78, 81, 150, 93), GNUTLS_MAC_SHA256),
	TEST_VECTOR((74, 162, 90, 97, 248, 179, 31, 6, 30, 15, 193, 213, 65, 222, 178, 14, 9, 118, 99, 204, 87, 5, 78, 31, 154, 52, 121, 137, 216, 23, 45, 98), (228, 115, 254, 88, 119, 250, 19, 124, 80, 190, 173, 194, 41, 81, 135, 241, 182, 110, 53, 200, 11, 120, 100, 191, 1, 193, 198, 32, 252, 9, 137, 57, 145, 80, 14, 154, 147, 133, 26, 225, 34, 23, 9, 17, 86, 43, 246, 238, 60, 117, 213, 221, 234, 222, 210, 120, 20, 98, 61, 44), (226, 53, 255, 114, 217, 192, 166, 74, 128, 205, 134, 253, 178, 111, 28, 216, 116, 14, 39, 4), GNUTLS_MAC_SHA256),
	TEST_VECTOR((38, 178, 149, 86, 16, 108, 6, 168, 92, 105, 80, 170, 242, 11, 94, 8, 165, 35, 232, 14, 25, 138, 114, 91, 105, 226, 63, 233, 59, 210, 225, 109), (189, 151, 63, 155, 198, 255, 2, 38, 178, 172, 198, 130, 224, 8, 75, 140, 103, 178, 133, 234, 155, 139, 131, 137, 56, 209, 143, 150, 222, 132, 82, 31, 228, 125, 86, 3, 55, 17, 95, 130, 50, 215, 101, 22, 103, 81, 241, 183, 2, 110, 96, 141, 37, 236, 101, 4, 52, 109, 16, 109), (226, 59, 25, 125, 77, 95, 216, 8, 28, 165, 77, 216, 106, 29, 69, 156, 202, 124, 105, 176), GNUTLS_MAC_SHA256),
	TEST_VECTOR((196, 190, 219, 221, 182, 100, 147, 231, 199, 37, 154, 59, 187, 194, 95, 140, 126, 12, 167, 254, 40, 77, 146, 212, 49, 217, 205, 153, 160, 210, 20, 172), (28, 105, 197, 71, 102, 121, 30, 49, 92, 44, 197, 196, 126, 205, 63, 250, 184, 125, 13, 39, 61, 217, 32, 231, 9, 85, 129, 76, 34, 14, 172, 172, 230, 165, 148, 101, 66, 218, 61, 254, 36, 255, 98, 107, 72, 151, 137, 140, 175, 183, 219, 131, 189, 255, 60, 20, 250, 70, 253, 75), (29, 164, 118, 56, 214, 201, 196, 208, 77, 116, 212, 100, 11, 189, 66, 171, 129, 77, 158, 140, 194, 47, 67, 38, 105, 82, 57, 249, 107, 6, 147, 241, 45, 13, 209, 21, 44, 244, 68, 48), GNUTLS_MAC_SHA256),
	TEST_VECTOR((34, 37, 108, 165, 113, 213, 200, 150, 219, 128, 168, 117, 143, 248, 28, 248, 99, 29, 43, 195, 140, 126, 118, 243, 186, 251, 12, 42, 245, 64, 163, 86), (157, 210, 220, 217, 123, 146, 98, 81, 181, 12, 97, 17, 217, 136, 226, 149, 27, 2, 172, 204, 20, 55, 2, 200, 137, 32, 207, 54, 132, 143, 124, 115, 23, 86, 171, 5, 55, 203, 38, 226, 39, 37, 241, 29, 224, 105, 229, 51, 88, 2, 176, 203, 86, 193, 88, 221, 117, 1, 71, 145), (161, 26, 163, 177, 169, 61, 44, 225, 23, 85, 8, 102, 194, 141, 105, 116, 207, 98, 103, 25, 56, 91, 136, 104, 16, 26, 113, 165, 210, 170, 121, 59, 194, 60, 60, 253, 235, 229, 46, 201), GNUTLS_MAC_SHA256),
	TEST_VECTOR((128, 102, 224, 87, 231, 50, 150, 21, 142, 213, 71, 152, 72, 49, 122, 213, 230, 78, 232, 251, 46, 84, 212, 239, 133, 183, 121, 47, 87, 246, 248, 135), (22, 218, 6, 231, 54, 14, 76, 39, 65, 155, 95, 105, 126, 76, 133, 72, 146, 92, 229, 91, 83, 173, 158, 94, 133, 185, 76, 127, 142, 87, 173, 20, 42, 26, 10, 3, 132, 51, 123, 26, 223, 100, 16, 237, 206, 206, 169, 33, 21, 43, 148, 214, 178, 58, 25, 44, 230, 246, 2, 215), (110, 190, 246, 75, 53, 128, 80, 237, 195, 200, 65, 245, 33, 136, 197, 228, 66, 203, 105, 99, 15, 236, 11, 229, 17, 72, 22, 175, 97, 106, 51, 63, 10, 172, 81, 83, 233, 38, 90, 166), GNUTLS_MAC_SHA256),
	TEST_VECTOR((86, 240, 108, 192, 188, 57, 42, 177, 8, 105, 108, 125, 247, 20, 149, 181, 165, 205, 54, 56, 224, 169, 32, 69, 175, 124, 189, 48, 118, 246, 221, 24), (79, 202, 232, 243, 43, 8, 184, 251, 116, 97, 33, 162, 219, 43, 201, 159, 187, 36, 185, 255, 17, 198, 10, 29, 201, 31, 20, 173, 154, 96, 198, 186, 254, 74, 219, 77, 193, 96, 233, 144, 30, 238, 235, 33, 42, 20, 126, 224, 167, 231, 109, 74, 239, 164, 39, 246, 106, 32, 92, 134), (204, 218, 130, 49, 250, 92, 7, 2, 186, 40, 42, 143, 24, 160, 193, 222, 198, 186, 239, 48, 134, 37, 251, 143, 80, 68, 16, 82, 44, 63, 59, 109, 100, 124, 23, 112, 84, 49, 122, 7), GNUTLS_MAC_SHA256),
	TEST_VECTOR((97, 127, 27, 104, 16, 197, 81, 170, 92, 33, 135, 139, 0, 101, 99, 81, 70, 96, 105, 212, 26, 220, 127, 238, 28, 237, 63, 47, 52, 50, 67, 92), (33, 19, 247, 0, 92, 88, 11, 119, 6, 9, 144, 194, 163, 249, 232, 200, 130, 95, 13, 126, 147, 163, 246, 159, 32, 142, 187, 91, 151, 196, 136, 218, 189, 13, 231, 199, 240, 14, 8, 180, 81, 93, 178, 192, 46, 24, 36, 249, 110, 113, 201, 162, 26, 24, 7, 156, 75, 100, 157, 129), (3, 37, 177, 200, 95, 61, 37, 173, 94, 188, 46, 123, 56, 12, 254, 222, 107, 60, 138, 218, 26, 240, 208, 213, 188, 217, 210, 179, 76, 41, 8, 49, 21, 22, 139, 138, 239, 84, 197, 132), GNUTLS_MAC_SHA256),
	TEST_VECTOR((101, 158, 219, 154, 13, 245, 26, 56, 103, 212, 170, 1, 247, 79, 96, 183, 177, 81, 176, 26, 52, 55, 194, 247, 116, 253, 55, 182, 136, 26, 156, 164), (63, 182, 8, 112, 165, 129, 59, 173, 253, 122, 249, 72, 195, 201, 36, 190, 192, 92, 146, 213, 64, 20, 11, 242, 143, 37, 70, 130, 92, 95, 189, 64, 241, 87, 20, 147, 161, 120, 70, 127, 218, 151, 147, 242, 247, 238, 212, 91, 64, 239, 104, 224, 16, 123, 141, 116, 192, 207, 50, 228), (92, 18, 199, 182, 236, 56, 245, 22, 247, 46, 118, 104, 156, 49, 6, 238, 0, 238, 140, 47, 80, 134, 44, 191, 127, 203, 116, 191, 135, 152, 235, 118, 26, 51, 131, 135, 136, 226, 118, 163), GNUTLS_MAC_SHA256),
	TEST_VECTOR((87, 3, 197, 86, 227, 165, 59, 141, 43, 243, 197, 202, 119, 61, 12, 110, 210, 193, 182, 106, 132, 230, 104, 4, 117, 168, 40, 105, 65, 178, 70, 179), (17, 158, 55, 214, 75, 90, 215, 2, 202, 89, 247, 149, 46, 88, 34, 205, 175, 183, 35, 192, 249, 44, 215, 3, 56, 161, 126, 36, 211, 175, 98, 103, 175, 121, 43, 24, 154, 1, 168, 163, 53, 58, 205, 122, 133, 180, 214, 59, 247, 228, 178, 47, 115, 215, 153, 46, 142, 78, 115, 137), (227, 2, 193, 193, 229, 200, 246, 136, 167, 88, 9, 151, 57, 148, 51, 251, 170, 228, 153, 64, 11, 138, 72, 144, 29, 128, 136, 57, 193, 235, 73, 223, 223, 99, 36, 20, 95, 30, 240, 30), GNUTLS_MAC_SHA256),
	TEST_VECTOR((154, 110, 131, 185, 27, 217, 153, 115, 126, 87, 126, 68, 145, 66, 218, 224, 89, 104, 231, 116, 178, 35, 193, 24, 93, 197, 116, 218, 120, 92, 147, 204), (75, 88, 69, 198, 115, 114, 2, 99, 43, 41, 70, 195, 87, 157, 157, 69, 130, 180, 117, 223, 163, 115, 148, 91, 10, 188, 104, 200, 240, 218, 163, 101, 32, 23, 148, 57, 8, 108, 104, 9, 170, 24, 32, 148, 69, 59, 192, 191, 254, 240, 220, 40, 136, 185, 98, 149, 252, 214, 228, 66), (233, 14, 62, 217, 2, 168, 235, 31, 198, 120, 35, 175, 83, 74, 43, 72, 70, 107, 242, 197, 135, 125, 173, 10, 173, 199, 214, 255, 116, 29, 143, 67, 123, 46, 109, 0, 49, 132, 105, 96), GNUTLS_MAC_SHA256),
	TEST_VECTOR((155, 235, 244, 101, 0, 58, 133, 188, 37, 237, 52, 12, 96, 149, 217, 99, 136, 85, 4, 211, 207, 2, 102, 175, 37, 46, 255, 210, 42, 211, 45, 111), (23, 80, 109, 180, 89, 220, 161, 72, 64, 145, 125, 173, 35, 38, 67, 3, 171, 28, 131, 227, 80, 24, 167, 34, 88, 9, 157, 32, 212, 248, 171, 133, 197, 34, 116, 4, 178, 58, 237, 106, 225, 8, 189, 18, 130, 229, 10, 0, 209, 96, 229, 52, 38, 71, 112, 161, 27, 79, 204, 117), (26, 50, 234, 48, 138, 166, 220, 111, 27, 124, 119, 241, 217, 170, 228, 6, 114, 252, 174, 34, 67, 139, 187, 5, 40, 226, 128, 7, 59, 49, 121, 120, 134, 182, 168, 0, 54, 160, 14, 25), GNUTLS_MAC_SHA256),
	TEST_VECTOR((29, 146, 9, 24, 62, 85, 125, 58, 172, 126, 42, 181, 61, 38, 236, 101, 157, 242, 167, 69, 254, 86, 165, 56, 24, 239, 88, 83, 164, 44, 225, 148), (192, 26, 67, 26, 50, 131, 57, 48, 162, 42, 190, 229, 198, 234, 52, 219, 69, 147, 22, 222, 243, 178, 65, 82, 158, 206, 126, 57, 226, 6, 154, 30, 107, 148, 41, 70, 19, 46, 235, 201, 103, 152, 1, 210, 206, 254, 244, 187, 182, 161, 184, 78, 248, 83, 50, 91, 123, 196, 152, 253), (218, 188, 255, 161, 106, 117, 137, 222, 238, 108, 118, 138, 175, 1, 224, 129, 61, 233, 9, 0, 85, 38, 218, 84, 112, 0, 131, 239, 6, 143, 133, 77, 73, 148, 18, 121, 104, 154, 23, 38), GNUTLS_MAC_SHA256),
	TEST_VECTOR((33, 110, 208, 68, 118, 156, 76, 57, 8, 24, 142, 206, 97, 96, 26, 248, 129, 156, 48, 245, 1, 209, 41, 149, 223, 96, 142, 6, 245, 224, 230, 7, 171, 84, 245, 66, 238, 45, 164, 25, 6, 223, 219, 73, 113, 242, 15, 157), (99, 142, 149, 6, 162, 199, 190, 105, 234, 52, 107, 132, 98, 154, 1, 12, 14, 34, 91, 117, 72, 245, 8, 22, 44, 137, 242, 156, 29, 219, 253, 112, 71, 44, 43, 88, 231, 220, 138, 166, 165, 176, 102, 2, 241, 200, 237, 73, 72, 205, 167, 156, 98, 112, 130, 24, 226, 106, 192, 226), (212, 177, 68, 187, 64, 199, 202, 190, 209, 57, 99, 215, 212, 49, 142, 114), GNUTLS_MAC_SHA384),
	TEST_VECTOR((145, 33, 65, 240, 78, 43, 207, 121, 254, 75, 175, 228, 111, 68, 220, 144, 130, 202, 57, 220, 249, 100, 217, 64, 156, 72, 97, 57, 120, 116, 103, 234, 200, 112, 149, 168, 242, 226, 86, 28, 25, 212, 24, 238, 111, 61, 131, 107), (203, 167, 40, 195, 203, 66, 246, 43, 159, 222, 101, 152, 200, 98, 142, 15, 136, 247, 99, 159, 214, 5, 179, 157, 129, 41, 106, 7, 73, 242, 124, 139, 117, 131, 6, 134, 222, 171, 148, 157, 225, 187, 208, 6, 46, 70, 82, 75, 31, 48, 116, 108, 28, 186, 2, 80, 143, 180, 194, 159), (21, 139, 49, 60, 109, 40, 176, 59, 40, 138, 226, 21, 78, 171, 33, 64), GNUTLS_MAC_SHA384),
	TEST_VECTOR((67, 200, 4, 38, 103, 113, 128, 188, 7, 61, 9, 58, 128, 148, 54, 225, 109, 86, 8, 38, 71, 206, 23, 148, 135, 101, 213, 96, 182, 204, 240, 68, 33, 41, 235, 85, 52, 19, 112, 118, 129, 151, 186, 220, 117, 75, 9, 93), (253, 113, 151, 76, 159, 45, 64, 192, 77, 98, 183, 58, 237, 182, 163, 128, 171, 101, 232, 71, 18, 231, 199, 220, 60, 16, 154, 227, 3, 17, 243, 237, 231, 124, 126, 206, 65, 61, 213, 118, 159, 215, 76, 188, 203, 2, 12, 146, 247, 184, 124, 55, 98, 5, 255, 148, 144, 182, 137, 195), (226, 65, 226, 197, 56, 253, 2, 147, 222, 29, 95, 110, 124, 213, 108, 124), GNUTLS_MAC_SHA384),
	TEST_VECTOR((6, 242, 154, 93, 104, 74, 46, 235, 98, 53, 50, 72, 74, 105, 26, 216, 80, 64, 217, 135, 36, 141, 200, 44, 81, 217, 176, 183, 231, 190, 81, 132, 126, 144, 118, 226, 111, 167, 211, 59, 230, 133, 98, 7, 254, 76, 224, 53), (86, 250, 255, 212, 145, 22, 1, 194, 23, 114, 95, 216, 22, 254, 12, 26, 31, 189, 143, 21, 83, 242, 13, 129, 1, 66, 47, 199, 32, 88, 202, 185, 194, 230, 58, 98, 102, 167, 38, 7, 92, 195, 30, 220, 10, 50, 127, 164, 68, 155, 94, 201, 129, 168, 109, 145, 65, 147, 42, 185), (162, 223, 3, 240, 33, 32, 56, 203, 131, 186, 70, 142, 158, 5, 165, 2), GNUTLS_MAC_SHA384),
	TEST_VECTOR((252, 245, 219, 154, 179, 33, 79, 211, 109, 159, 180, 36, 122, 178, 166, 245, 136, 98, 241, 56, 121, 23, 229, 133, 250, 101, 111, 224, 161, 152, 71, 179, 193, 17, 251, 238, 90, 134, 128, 146, 242, 36, 199, 150, 88, 235, 219, 82), (199, 146, 252, 139, 50, 168, 108, 148, 33, 43, 92, 132, 95, 199, 138, 154, 102, 217, 78, 249, 131, 120, 228, 216, 0, 179, 105, 245, 67, 240, 160, 119, 253, 224, 182, 177, 9, 239, 180, 198, 186, 207, 161, 80, 34, 73, 221, 193, 227, 156, 124, 86, 223, 190, 59, 40, 93, 142, 25, 112), (183, 91, 12, 149, 174, 96, 234, 207, 126, 215, 182, 162, 22, 192, 163, 149), GNUTLS_MAC_SHA384),
	TEST_VECTOR((65, 121, 163, 76, 172, 137, 222, 5, 60, 192, 225, 85, 125, 65, 186, 150, 194, 174, 164, 227, 90, 13, 48, 77, 254, 45, 69, 145, 223, 213, 222, 237, 206, 148, 255, 199, 186, 106, 186, 139, 23, 220, 218, 14, 156, 248, 196, 32), (59, 250, 185, 95, 34, 252, 68, 110, 14, 54, 130, 238, 25, 127, 52, 219, 8, 95, 242, 64, 136, 251, 106, 38, 41, 152, 120, 163, 9, 234, 225, 18, 238, 50, 191, 72, 130, 226, 24, 173, 10, 97, 139, 48, 98, 214, 133, 112, 126, 18, 97, 193, 93, 98, 209, 76, 92, 233, 88, 208), (73, 162, 89, 5, 193, 46, 249, 254, 206, 88, 192, 162, 134, 16, 120, 198), GNUTLS_MAC_SHA384),
	TEST_VECTOR((129, 110, 73, 134, 128, 229, 204, 57, 184, 38, 71, 9, 181, 8, 175, 101, 58, 80, 185, 39, 252, 189, 206, 255, 46, 190, 98, 186, 58, 37, 183, 153, 80, 38, 141, 49, 20, 172, 73, 22, 140, 87, 196, 25, 253, 119, 220, 80), (64, 72, 108, 178, 44, 172, 221, 222, 135, 11, 114, 125, 232, 30, 17, 102, 126, 249, 44, 180, 249, 32, 206, 127, 47, 128, 160, 239, 41, 173, 208, 22, 20, 44, 226, 105, 78, 134, 110, 29, 99, 27, 92, 28, 19, 186, 30, 63, 208, 221, 201, 161, 100, 71, 194, 1, 134, 250, 193, 59), (121, 168, 103, 192, 51, 61, 52, 111, 13, 248, 234, 44, 109, 76, 177, 75), GNUTLS_MAC_SHA384),
	TEST_VECTOR((108, 252, 87, 10, 74, 220, 235, 204, 184, 101, 28, 109, 181, 4, 231, 101, 54, 64, 121, 239, 52, 49, 92, 120, 53, 70, 96, 90, 196, 140, 194, 170, 147, 60, 111, 16, 232, 36, 172, 195, 199, 51, 138, 9, 192, 30, 26, 48), (213, 248, 238, 109, 227, 126, 61, 26, 84, 195, 112, 188, 42, 128, 200, 135, 73, 165, 70, 133, 124, 96, 198, 238, 109, 242, 47, 79, 22, 107, 225, 136, 189, 41, 153, 135, 15, 165, 252, 193, 215, 204, 69, 88, 40, 243, 68, 110, 106, 69, 10, 20, 166, 103, 231, 21, 211, 193, 102, 34), (124, 248, 68, 103, 127, 137, 112, 40, 90, 209, 72, 233, 144, 74, 177, 55), GNUTLS_MAC_SHA384),
	TEST_VECTOR((214, 212, 84, 151, 94, 223, 21, 152, 252, 85, 50, 197, 195, 217, 250, 14, 205, 53, 126, 165, 135, 176, 96, 28, 144, 199, 81, 87, 41, 168, 75, 226, 140, 12, 147, 19, 166, 236, 117, 123, 140, 42, 245, 193, 126, 153, 70, 117), (92, 62, 223, 77, 133, 105, 49, 216, 63, 137, 57, 190, 154, 36, 94, 243, 70, 56, 244, 6, 222, 228, 122, 174, 81, 98, 167, 31, 15, 104, 196, 108, 149, 206, 222, 70, 250, 69, 42, 131, 208, 105, 30, 125, 6, 218, 154, 178, 0, 253, 95, 178, 247, 197, 242, 140, 138, 18, 44, 65), (11, 20, 5, 92, 251, 132, 26, 70, 82, 89, 184, 13, 195, 232, 125, 144), GNUTLS_MAC_SHA384),
	TEST_VECTOR((214, 37, 105, 195, 102, 159, 16, 18, 201, 102, 245, 74, 197, 172, 18, 29, 138, 137, 135, 21, 130, 52, 123, 19, 178, 8, 222, 253, 80, 205, 59, 196, 144, 140, 152, 150, 37, 123, 16, 52, 91, 236, 45, 64, 223, 43, 108, 93), (145, 140, 2, 68, 163, 167, 135, 14, 58, 228, 248, 192, 118, 40, 187, 117, 74, 15, 111, 138, 82, 19, 125, 84, 156, 126, 158, 23, 16, 59, 66, 160, 41, 104, 136, 87, 24, 94, 96, 132, 112, 149, 63, 120, 124, 201, 122, 126, 215, 94, 122, 54, 9, 238, 210, 243, 177, 162, 14, 57), (45, 196, 82, 74, 236, 203, 90, 33, 172, 241, 240, 177, 48, 83, 245, 165), GNUTLS_MAC_SHA384),
	TEST_VECTOR((143, 202, 32, 20, 115, 67, 63, 45, 200, 246, 174, 81, 228, 141, 225, 165, 101, 76, 230, 135, 231, 17, 210, 214, 95, 13, 197, 218, 111, 238, 154, 106, 61, 185, 216, 83, 93, 62, 68, 85, 171, 83, 211, 88, 80, 200, 130, 114), (25, 91, 216, 138, 162, 212, 33, 25, 18, 51, 79, 226, 253, 155, 210, 69, 34, 247, 217, 251, 8, 224, 71, 71, 96, 155, 195, 79, 37, 56, 8, 154, 157, 40, 187, 199, 11, 46, 19, 54, 195, 100, 55, 83, 206, 198, 229, 205, 63, 36, 108, 170, 145, 94, 60, 58, 107, 148, 211, 182), (245, 26, 200, 107, 15, 70, 35, 136, 209, 137, 237, 1, 151, 239, 153, 194, 255, 58, 101, 129, 109, 132, 66, 229, 234, 48, 67, 151, 185, 141, 209, 31), GNUTLS_MAC_SHA384),
	TEST_VECTOR((150, 196, 93, 206, 121, 160, 45, 43, 252, 42, 16, 168, 231, 68, 201, 116, 129, 46, 106, 155, 131, 71, 76, 229, 55, 67, 252, 179, 52, 184, 125, 130, 111, 65, 27, 173, 131, 109, 224, 23, 121, 12, 254, 7, 8, 127, 139, 2), (128, 105, 140, 217, 136, 224, 43, 27, 187, 13, 2, 193, 187, 43, 218, 245, 68, 255, 219, 53, 39, 237, 230, 33, 210, 242, 245, 234, 180, 164, 150, 78, 245, 48, 55, 142, 148, 174, 154, 183, 72, 77, 30, 239, 133, 72, 50, 213, 187, 32, 74, 139, 255, 33, 101, 26, 158, 60, 231, 88), (142, 241, 224, 252, 38, 211, 153, 127, 152, 90, 181, 86, 112, 102, 57, 28, 13, 140, 237, 84, 241, 205, 171, 206, 87, 181, 172, 202, 190, 33, 239, 120), GNUTLS_MAC_SHA384),
	TEST_VECTOR((130, 44, 118, 74, 27, 17, 112, 133, 193, 15, 14, 104, 152, 20, 210, 191, 189, 155, 67, 40, 127, 26, 140, 117, 215, 149, 169, 131, 26, 40, 97, 132, 200, 88, 111, 53, 119, 182, 229, 187, 206, 22, 55, 146, 94, 4, 252, 71), (175, 52, 97, 16, 185, 65, 177, 29, 33, 137, 49, 108, 159, 194, 185, 244, 33, 55, 117, 165, 215, 54, 141, 53, 65, 38, 120, 162, 143, 205, 3, 176, 127, 5, 73, 102, 110, 253, 243, 12, 128, 240, 171, 85, 99, 114, 10, 86, 239, 97, 106, 19, 187, 143, 119, 128, 3, 111, 192, 142), (224, 174, 35, 92, 184, 35, 128, 82, 123, 231, 105, 52, 166, 150, 34, 57, 109, 144, 231, 191, 167, 226, 210, 149, 228, 55, 91, 206, 224, 209, 177, 1), GNUTLS_MAC_SHA384),
	TEST_VECTOR((52, 14, 33, 45, 117, 142, 131, 204, 91, 137, 228, 181, 106, 134, 238, 140, 150, 49, 174, 78, 75, 186, 236, 21, 172, 9, 94, 164, 64, 123, 199, 182, 52, 173, 99, 13, 208, 190, 133, 169, 28, 8, 168, 199, 225, 225, 3, 11), (60, 213, 86, 26, 209, 47, 173, 252, 228, 8, 224, 65, 128, 175, 206, 227, 139, 131, 21, 107, 158, 75, 224, 119, 156, 79, 13, 185, 226, 107, 254, 92, 205, 67, 225, 89, 33, 151, 124, 210, 107, 29, 184, 40, 139, 128, 8, 158, 183, 209, 187, 215, 245, 158, 16, 17, 179, 225, 139, 81), (5, 250, 87, 123, 112, 129, 33, 14, 124, 157, 230, 157, 176, 61, 124, 32, 38, 207, 68, 105, 169, 11, 250, 41, 241, 194, 193, 8, 24, 212, 99, 224), GNUTLS_MAC_SHA384),
	TEST_VECTOR((190, 183, 219, 222, 126, 10, 138, 204, 162, 243, 235, 248, 195, 122, 30, 156, 49, 142, 68, 123, 149, 146, 249, 63, 201, 2, 30, 105, 8, 136, 57, 26, 53, 72, 16, 185, 103, 62, 30, 36, 0, 240, 80, 96, 211, 170, 184, 81), (86, 11, 226, 156, 98, 236, 163, 56, 85, 189, 28, 199, 10, 223, 175, 78, 126, 171, 164, 254, 85, 7, 93, 70, 110, 145, 70, 122, 196, 244, 185, 190, 182, 120, 225, 178, 201, 97, 81, 26, 177, 42, 250, 40, 211, 74, 17, 205, 46, 226, 177, 81, 96, 155, 200, 81, 3, 109, 24, 69), (99, 249, 109, 51, 57, 227, 85, 72, 112, 145, 40, 112, 18, 216, 212, 98, 128, 56, 117, 103, 175, 182, 117, 23, 172, 97, 109, 52, 15, 163, 100, 102), GNUTLS_MAC_SHA384),
	TEST_VECTOR((163, 96, 240, 181, 22, 252, 53, 167, 109, 29, 74, 158, 198, 160, 142, 133, 107, 142, 250, 147, 16, 65, 19, 152, 191, 26, 198, 159, 40, 83, 147, 0, 188, 34, 86, 215, 99, 136, 229, 4, 193, 246, 12, 139, 96, 230, 6, 152), (2, 6, 250, 245, 201, 172, 64, 234, 172, 169, 161, 209, 106, 199, 31, 25, 226, 222, 132, 77, 178, 231, 62, 79, 183, 227, 69, 103, 226, 33, 213, 172, 188, 105, 49, 65, 39, 69, 220, 176, 176, 94, 17, 146, 132, 210, 28, 75, 179, 249, 169, 57, 39, 23, 80, 246, 135, 132, 253, 157), (136, 236, 238, 116, 42, 7, 31, 157, 36, 114, 5, 32, 232, 189, 191, 225, 64, 206, 87, 231, 254, 43, 104, 105, 102, 135, 30, 237, 10, 97, 101, 235), GNUTLS_MAC_SHA384),
	TEST_VECTOR((220, 135, 91, 47, 57, 60, 29, 64, 194, 65, 38, 27, 9, 24, 121, 6, 1, 200, 185, 8, 96, 132, 72, 247, 102, 230, 232, 174, 207, 165, 228, 146, 126, 117, 116, 175, 64, 48, 4, 125, 152, 177, 20, 104, 16, 130, 136, 227), (198, 58, 227, 185, 149, 175, 238, 74, 195, 21, 75, 249, 239, 198, 187, 16, 229, 55, 115, 106, 181, 188, 66, 127, 60, 55, 107, 232, 251, 129, 170, 94, 7, 100, 164, 176, 22, 94, 170, 176, 165, 25, 236, 255, 69, 230, 246, 199, 105, 234, 102, 75, 221, 105, 53, 201, 77, 138, 206, 194), (71, 61, 198, 169, 128, 214, 226, 70, 91, 250, 61, 2, 231, 229, 52, 27, 156, 234, 94, 9, 242, 119, 103, 165, 197, 143, 125, 182, 11, 95, 116, 75), GNUTLS_MAC_SHA384),
	TEST_VECTOR((73, 219, 118, 49, 163, 80, 127, 2, 214, 185, 4, 142, 38, 175, 206, 72, 176, 94, 145, 181, 15, 236, 59, 124, 207, 46, 139, 222, 154, 169, 215, 156, 11, 102, 115, 248, 20, 216, 206, 57, 121, 94, 121, 35, 156, 188, 57, 103), (201, 8, 58, 53, 41, 155, 37, 20, 28, 220, 96, 180, 204, 124, 36, 38, 54, 147, 241, 94, 122, 241, 168, 198, 209, 33, 215, 152, 254, 11, 153, 72, 168, 51, 56, 220, 44, 189, 199, 88, 118, 239, 200, 63, 211, 57, 140, 160, 223, 34, 59, 16, 91, 15, 19, 236, 32, 51, 186, 212), (179, 80, 66, 83, 71, 151, 156, 111, 28, 16, 13, 177, 246, 222, 145, 145, 0, 255, 113, 55, 101, 193, 222, 161, 70, 198, 42, 105, 137, 98, 53, 166), GNUTLS_MAC_SHA384),
	TEST_VECTOR((226, 22, 138, 176, 114, 193, 45, 164, 113, 252, 2, 79, 203, 197, 39, 120, 233, 189, 104, 244, 71, 98, 185, 216, 186, 171, 116, 109, 185, 193, 79, 254, 113, 129, 247, 28, 46, 230, 72, 129, 48, 131, 40, 48, 253, 11, 213, 240), (89, 86, 148, 84, 178, 245, 66, 74, 145, 74, 45, 22, 109, 168, 238, 18, 123, 14, 197, 171, 71, 136, 177, 21, 193, 217, 150, 106, 82, 151, 24, 28, 100, 84, 144, 115, 34, 154, 50, 229, 222, 64, 201, 74, 143, 158, 254, 113, 160, 201, 104, 246, 51, 4, 130, 135, 110, 55, 190, 221), (22, 203, 107, 201, 61, 154, 0, 20, 96, 184, 50, 89, 188, 154, 67, 68, 212, 180, 203, 66, 44, 97, 2, 86, 93, 197, 177, 144, 222, 28, 86, 115), GNUTLS_MAC_SHA384),
	TEST_VECTOR((0, 161, 45, 60, 228, 255, 117, 166, 227, 15, 65, 243, 85, 124, 130, 106, 241, 50, 107, 99, 2, 244, 206, 136, 123, 173, 61, 51, 23, 165, 72, 200, 192, 58, 5, 114, 132, 220, 195, 141, 139, 198, 144, 189, 74, 86, 95, 71), (36, 197, 192, 178, 200, 16, 223, 160, 142, 53, 215, 254, 235, 184, 199, 142, 12, 215, 38, 201, 46, 205, 66, 217, 23, 16, 19, 115, 140, 162, 83, 26, 148, 127, 82, 60, 55, 246, 76, 219, 4, 48, 91, 217, 105, 209, 214, 249, 236, 212, 100, 5, 210, 130, 128, 249, 104, 80, 11, 167), (174, 243, 213, 124, 141, 167, 217, 88, 44, 93, 28, 98, 214, 182, 72, 150, 218, 155, 27, 14, 64, 18, 164, 76, 220, 61, 207, 75, 112, 173, 108, 102), GNUTLS_MAC_SHA384),
	TEST_VECTOR((188, 49, 87, 184, 147, 46, 136, 209, 177, 207, 142, 70, 34, 19, 112, 16, 162, 66, 211, 82, 123, 29, 35, 214, 217, 192, 219, 156, 201, 237, 252, 32, 229, 19, 93, 232, 35, 151, 123, 244, 222, 250, 250, 228, 77, 108, 218, 182), (180, 42, 142, 67, 204, 45, 78, 92, 105, 238, 94, 79, 107, 25, 255, 107, 128, 113, 210, 107, 171, 77, 254, 69, 101, 11, 146, 177, 244, 118, 82, 210, 81, 98, 212, 182, 20, 65, 216, 68, 140, 84, 145, 138, 229, 104, 174, 47, 181, 48, 145, 198, 36, 219, 255, 250, 206, 229, 29, 136), (145, 49, 75, 223, 84, 33, 98, 3, 22, 67, 36, 125, 101, 7, 131, 142, 171, 165, 15, 26), GNUTLS_MAC_SHA384),
	TEST_VECTOR((35, 213, 243, 243, 76, 159, 231, 51, 232, 8, 148, 159, 64, 17, 186, 49, 113, 55, 110, 59, 238, 128, 126, 197, 178, 132, 150, 191, 78, 181, 29, 133, 170, 55, 196, 46, 30, 217, 60, 255, 186, 185, 108, 109, 202, 58, 251, 59), (110, 199, 182, 187, 216, 26, 49, 47, 247, 135, 220, 106, 247, 199, 199, 185, 205, 187, 125, 12, 25, 216, 8, 83, 107, 192, 153, 11, 215, 231, 158, 35, 43, 188, 20, 51, 202, 86, 124, 188, 196, 218, 247, 158, 141, 114, 36, 195, 1, 36, 166, 57, 133, 37, 135, 226, 113, 90, 230, 46), (195, 195, 87, 156, 215, 10, 247, 248, 193, 132, 197, 128, 34, 79, 39, 247, 102, 76, 159, 211), GNUTLS_MAC_SHA384),
	TEST_VECTOR((0, 0, 217, 183, 236, 111, 190, 253, 242, 86, 253, 104, 34, 11, 82, 5, 172, 101, 162, 0, 17, 69, 17, 140, 80, 186, 107, 101, 112, 50, 25, 139, 139, 124, 227, 178, 247, 6, 138, 120, 13, 193, 124, 34, 69, 154, 242, 183), (216, 87, 84, 28, 98, 184, 87, 86, 220, 115, 222, 125, 194, 216, 111, 93, 94, 139, 40, 51, 139, 176, 169, 69, 181, 196, 253, 124, 129, 247, 25, 97, 185, 112, 93, 61, 21, 59, 25, 25, 93, 0, 59, 116, 33, 32, 104, 237, 16, 249, 108, 83, 67, 134, 83, 8, 122, 1, 82, 207), (121, 62, 241, 19, 249, 99, 151, 171, 0, 49, 234, 160, 250, 167, 119, 193, 7, 231, 208, 60), GNUTLS_MAC_SHA384),
	TEST_VECTOR((79, 61, 116, 77, 62, 68, 158, 6, 39, 191, 68, 152, 116, 56, 40, 248, 110, 99, 143, 96, 98, 10, 126, 212, 167, 201, 181, 176, 115, 105, 28, 158, 201, 71, 40, 197, 136, 34, 232, 39, 240, 246, 204, 248, 109, 188, 28, 174), (48, 31, 238, 178, 94, 108, 168, 80, 62, 205, 130, 31, 29, 55, 135, 174, 191, 179, 208, 236, 81, 139, 179, 17, 116, 245, 32, 155, 42, 193, 242, 142, 211, 230, 152, 115, 107, 173, 16, 161, 142, 60, 189, 181, 220, 39, 187, 209, 45, 5, 139, 54, 219, 8, 146, 249, 207, 208, 131, 0), (133, 239, 149, 5, 178, 48, 86, 94, 204, 242, 166, 74, 179, 222, 83, 229, 169, 28, 123, 81), GNUTLS_MAC_SHA384),
	TEST_VECTOR((106, 243, 225, 101, 158, 231, 202, 241, 209, 10, 217, 19, 92, 151, 87, 53, 15, 105, 136, 108, 202, 177, 186, 79, 222, 80, 196, 97, 75, 49, 158, 43, 32, 236, 199, 99, 108, 95, 107, 186, 231, 28, 211, 85, 101, 164, 162, 200), (121, 186, 196, 71, 241, 47, 190, 150, 193, 151, 150, 59, 145, 185, 57, 172, 61, 191, 69, 65, 190, 222, 187, 34, 197, 103, 127, 84, 179, 119, 160, 204, 59, 85, 162, 191, 64, 51, 212, 34, 64, 26, 20, 158, 116, 168, 22, 21, 71, 45, 13, 79, 142, 79, 12, 229, 67, 132, 216, 68), (40, 171, 236, 32, 214, 240, 146, 208, 158, 50, 54, 9, 134, 53, 147, 3, 117, 60, 237, 190), GNUTLS_MAC_SHA384),
	TEST_VECTOR((2, 153, 209, 240, 100, 53, 45, 90, 131, 251, 97, 225, 224, 80, 152, 15, 101, 180, 180, 64, 129, 157, 44, 28, 41, 182, 173, 91, 223, 178, 192, 3, 71, 218, 153, 60, 201, 236, 0, 252, 196, 189, 1, 192, 63, 237, 155, 101), (111, 158, 229, 203, 56, 82, 238, 194, 172, 63, 9, 92, 37, 194, 3, 99, 208, 247, 244, 119, 8, 153, 191, 153, 24, 126, 42, 131, 12, 101, 113, 175, 248, 33, 206, 183, 232, 11, 108, 116, 79, 140, 94, 123, 192, 178, 184, 25, 111, 181, 64, 0, 43, 141, 183, 188, 96, 145, 57, 152), (41, 62, 15, 211, 228, 208, 110, 254, 158, 225, 186, 125, 139, 171, 29, 54, 189, 117, 9, 193), GNUTLS_MAC_SHA384),
	TEST_VECTOR((30, 220, 127, 233, 173, 51, 167, 246, 241, 47, 53, 12, 18, 142, 40, 147, 236, 27, 229, 156, 10, 119, 250, 88, 150, 242, 226, 118, 191, 158, 177, 156, 229, 177, 193, 204, 47, 242, 95, 147, 145, 218, 48, 77, 38, 3, 123, 227), (254, 52, 131, 84, 3, 65, 23, 79, 173, 214, 191, 169, 249, 100, 115, 234, 189, 150, 74, 213, 136, 174, 20, 96, 115, 157, 235, 59, 245, 18, 71, 92, 109, 87, 224, 23, 220, 70, 225, 17, 87, 51, 221, 129, 250, 106, 93, 170, 156, 31, 157, 35, 192, 175, 187, 179, 105, 88, 205, 172), (60, 25, 35, 71, 206, 115, 19, 233, 229, 66, 145, 195, 124, 65, 44, 249, 10, 245, 138, 101), GNUTLS_MAC_SHA384),
	TEST_VECTOR((39, 169, 165, 20, 230, 57, 67, 233, 185, 12, 173, 18, 145, 251, 42, 48, 84, 200, 91, 74, 116, 92, 195, 207, 49, 189, 193, 200, 147, 93, 21, 63, 180, 149, 231, 186, 74, 182, 65, 183, 201, 227, 181, 167, 196, 151, 177, 202), (197, 175, 102, 206, 108, 3, 251, 127, 216, 128, 23, 179, 129, 73, 163, 44, 140, 157, 202, 185, 102, 101, 239, 173, 40, 222, 183, 27, 97, 238, 117, 136, 126, 19, 36, 128, 11, 1, 177, 63, 221, 93, 176, 11, 139, 180, 190, 80, 204, 46, 174, 234, 67, 207, 38, 231, 125, 230, 220, 179), (136, 109, 183, 66, 7, 222, 59, 3, 79, 62, 112, 207, 155, 71, 50, 77, 87, 223, 81, 133), GNUTLS_MAC_SHA384),
	TEST_VECTOR((227, 88, 187, 4, 165, 170, 237, 151, 147, 154, 228, 21, 171, 63, 254, 157, 42, 177, 76, 134, 195, 195, 247, 43, 40, 91, 253, 251, 124, 20, 64, 248, 13, 117, 204, 242, 141, 44, 100, 213, 3, 66, 111, 85, 44, 180, 253, 186), (107, 147, 173, 217, 22, 63, 94, 20, 223, 1, 96, 176, 238, 4, 2, 179, 84, 47, 146, 15, 177, 44, 186, 113, 194, 199, 152, 136, 119, 190, 70, 190, 149, 17, 19, 218, 191, 72, 175, 242, 126, 140, 155, 33, 76, 191, 41, 59, 138, 150, 109, 46, 167, 255, 81, 91, 107, 218, 168, 225), (72, 188, 27, 193, 53, 213, 206, 74, 192, 75, 17, 231, 197, 12, 219, 191, 132, 196, 78, 215), GNUTLS_MAC_SHA384),
	TEST_VECTOR((110, 58, 183, 66, 34, 202, 163, 127, 115, 45, 227, 120, 153, 43, 167, 92, 55, 177, 193, 75, 63, 90, 44, 43, 180, 44, 39, 13, 111, 33, 62, 235, 173, 60, 53, 36, 164, 179, 84, 247, 83, 188, 229, 237, 124, 168, 101, 219), (121, 211, 144, 151, 177, 155, 198, 23, 42, 219, 68, 92, 249, 65, 137, 121, 185, 67, 130, 82, 40, 48, 127, 47, 58, 168, 165, 145, 224, 51, 28, 119, 186, 1, 174, 220, 162, 192, 148, 57, 12, 4, 105, 77, 172, 144, 123, 225, 237, 32, 193, 36, 192, 193, 229, 178, 120, 246, 58, 60), (98, 147, 84, 250, 218, 251, 7, 125, 115, 68, 245, 36, 184, 43, 125, 20, 165, 206, 171, 34), GNUTLS_MAC_SHA384),
	TEST_VECTOR((88, 47, 150, 138, 84, 184, 121, 123, 158, 168, 198, 85, 180, 46, 57, 122, 219, 115, 215, 115, 177, 152, 75, 30, 28, 66, 156, 213, 151, 184, 1, 93, 47, 145, 213, 158, 65, 54, 169, 213, 35, 191, 100, 145, 164, 115, 60, 122), (230, 211, 193, 147, 239, 243, 78, 52, 248, 183, 176, 14, 102, 86, 90, 235, 1, 246, 50, 6, 187, 39, 226, 122, 162, 129, 89, 42, 252, 6, 174, 30, 197, 183, 235, 151, 163, 150, 132, 206, 119, 61, 124, 53, 40, 242, 102, 124, 31, 93, 66, 132, 6, 231, 140, 228, 207, 57, 246, 82), (105, 23, 38, 193, 17, 229, 3, 11, 95, 150, 87, 6, 145, 7, 134, 30, 204, 24, 188, 88, 53, 168, 20, 195, 210, 229, 9, 44, 144, 28, 177, 251, 108, 26, 124, 211, 235, 11, 226, 167), GNUTLS_MAC_SHA384),
	TEST_VECTOR((189, 163, 46, 191, 107, 141, 108, 33, 180, 7, 140, 5, 88, 44, 202, 197, 125, 14, 9, 213, 152, 237, 81, 202, 128, 139, 186, 228, 49, 95, 243, 8, 32, 134, 231, 114, 165, 15, 130, 139, 163, 168, 164, 112, 137, 96, 76, 31), (114, 58, 106, 163, 226, 9, 63, 43, 58, 55, 126, 77, 113, 108, 250, 222, 247, 132, 235, 56, 209, 3, 2, 168, 188, 136, 41, 79, 250, 176, 46, 138, 180, 62, 108, 131, 167, 4, 137, 220, 145, 164, 4, 14, 28, 4, 247, 17, 169, 173, 246, 1, 212, 154, 42, 208, 120, 53, 198, 104), (146, 235, 161, 4, 64, 160, 242, 142, 202, 64, 199, 101, 204, 8, 3, 27, 251, 174, 197, 250, 42, 45, 63, 161, 144, 105, 203, 61, 93, 208, 142, 1, 112, 44, 213, 238, 22, 50, 141, 15), GNUTLS_MAC_SHA384),
	TEST_VECTOR((188, 141, 97, 5, 51, 231, 253, 129, 243, 246, 172, 47, 147, 135, 107, 226, 89, 232, 140, 111, 35, 74, 189, 8, 128, 118, 45, 18, 205, 125, 33, 52, 242, 82, 190, 204, 57, 92, 196, 139, 136, 235, 100, 93, 114, 37, 20, 199), (41, 165, 234, 13, 194, 98, 6, 38, 128, 149, 121, 26, 44, 11, 24, 25, 7, 156, 136, 248, 93, 208, 37, 155, 200, 234, 116, 214, 191, 80, 79, 252, 33, 198, 205, 95, 146, 247, 74, 2, 40, 61, 228, 227, 57, 69, 57, 62, 100, 202, 75, 15, 105, 24, 133, 228, 242, 118, 46, 39), (110, 58, 27, 98, 51, 216, 224, 138, 28, 254, 73, 201, 249, 232, 96, 95, 163, 102, 246, 16, 128, 197, 241, 167, 80, 178, 33, 19, 22, 141, 54, 0, 29, 170, 198, 215, 187, 140, 153, 212), GNUTLS_MAC_SHA384),
	TEST_VECTOR((214, 190, 117, 2, 143, 123, 215, 54, 32, 102, 126, 165, 12, 74, 103, 68, 113, 133, 75, 103, 127, 240, 182, 48, 38, 70, 12, 48, 0, 209, 27, 82, 16, 2, 159, 41, 121, 179, 238, 170, 97, 56, 18, 152, 62, 125, 118, 147), (92, 44, 226, 72, 192, 167, 58, 41, 166, 232, 231, 179, 253, 187, 175, 1, 152, 214, 54, 230, 217, 214, 165, 71, 104, 92, 39, 19, 77, 128, 65, 29, 18, 47, 150, 49, 20, 52, 121, 140, 177, 155, 61, 68, 111, 82, 215, 209, 215, 75, 243, 238, 63, 209, 254, 162, 163, 181, 38, 206), (230, 124, 153, 93, 164, 159, 78, 208, 246, 239, 192, 159, 131, 156, 121, 94, 132, 71, 225, 119, 144, 249, 128, 90, 168, 169, 58, 77, 185, 160, 57, 112, 194, 128, 233, 21, 154, 100, 254, 195), GNUTLS_MAC_SHA384),
	TEST_VECTOR((247, 103, 140, 192, 183, 174, 197, 244, 189, 228, 135, 243, 10, 90, 136, 201, 136, 253, 36, 196, 99, 185, 228, 107, 59, 42, 228, 199, 101, 202, 121, 196, 138, 190, 105, 28, 249, 59, 107, 67, 91, 43, 223, 242, 163, 212, 145, 242), (176, 76, 148, 91, 241, 73, 242, 246, 209, 116, 56, 149, 141, 26, 114, 92, 58, 138, 180, 108, 170, 21, 160, 94, 128, 149, 36, 185, 209, 119, 126, 251, 72, 231, 214, 186, 175, 93, 62, 251, 243, 11, 105, 179, 38, 147, 94, 89, 65, 18, 7, 185, 19, 243, 137, 136, 224, 106, 190, 93), (101, 227, 3, 173, 232, 213, 237, 85, 89, 196, 14, 99, 18, 177, 14, 228, 242, 244, 221, 238, 22, 97, 39, 84, 127, 241, 161, 76, 192, 49, 140, 220, 97, 209, 208, 195, 214, 84, 42, 217), GNUTLS_MAC_SHA384),
	TEST_VECTOR((213, 215, 142, 235, 209, 47, 130, 243, 152, 35, 92, 167, 42, 244, 39, 125, 180, 117, 168, 13, 168, 67, 243, 110, 30, 217, 90, 102, 30, 30, 97, 10, 44, 125, 204, 67, 62, 96, 2, 156, 102, 137, 152, 9, 97, 243, 89, 15), (9, 76, 93, 124, 116, 87, 197, 28, 135, 122, 11, 234, 3, 251, 196, 65, 101, 193, 6, 53, 164, 230, 75, 133, 86, 200, 236, 242, 9, 20, 230, 33, 239, 52, 34, 68, 150, 72, 157, 118, 246, 123, 38, 57, 96, 136, 32, 173, 167, 65, 55, 145, 107, 251, 49, 91, 213, 8, 114, 237), (161, 110, 158, 193, 190, 26, 227, 180, 10, 169, 6, 200, 225, 142, 154, 118, 151, 204, 52, 18, 75, 123, 4, 149, 124, 225, 73, 188, 57, 189, 242, 112, 22, 245, 131, 158, 33, 234, 121, 118), GNUTLS_MAC_SHA384),
	TEST_VECTOR((88, 241, 164, 193, 108, 179, 50, 99, 27, 33, 122, 142, 27, 92, 202, 132, 54, 101, 6, 76, 198, 28, 158, 130, 174, 65, 128, 213, 116, 222, 244, 226, 163, 171, 148, 3, 3, 110, 57, 21, 217, 80, 68, 153, 127, 27, 171, 129), (199, 122, 11, 201, 84, 56, 65, 220, 6, 44, 138, 187, 128, 73, 120, 163, 175, 113, 146, 36, 74, 172, 79, 115, 168, 33, 117, 11, 163, 141, 72, 109, 90, 43, 79, 141, 146, 127, 131, 7, 114, 196, 229, 32, 142, 145, 195, 127, 130, 183, 198, 88, 210, 64, 225, 78, 32, 213, 140, 238), (178, 208, 117, 41, 36, 137, 187, 209, 119, 247, 46, 94, 128, 6, 24, 208, 74, 48, 249, 182, 224, 168, 209, 252, 203, 67, 138, 230, 246, 70, 163, 62, 12, 71, 211, 98, 90, 184, 123, 108), GNUTLS_MAC_SHA384),
	TEST_VECTOR((170, 176, 224, 126, 40, 69, 206, 22, 100, 5, 122, 161, 90, 29, 80, 195, 92, 232, 85, 145, 197, 224, 216, 211, 170, 126, 140, 90, 88, 176, 191, 86, 194, 112, 149, 127, 55, 40, 201, 116, 90, 75, 198, 150, 248, 124, 160, 241), (60, 17, 91, 255, 194, 109, 65, 145, 130, 67, 59, 212, 53, 112, 251, 63, 111, 47, 254, 55, 133, 90, 85, 70, 219, 197, 66, 153, 15, 239, 30, 51, 44, 58, 7, 7, 67, 186, 49, 233, 132, 202, 24, 204, 148, 227, 63, 79, 44, 116, 79, 107, 112, 35, 74, 123, 168, 13, 126, 74), (0, 28, 218, 5, 145, 121, 172, 208, 103, 188, 145, 41, 1, 125, 36, 232, 185, 154, 45, 102, 4, 225, 71, 20, 247, 32, 53, 139, 247, 244, 60, 187, 106, 74, 173, 70, 35, 38, 126, 114), GNUTLS_MAC_SHA384),
	TEST_VECTOR((12, 86, 32, 243, 74, 167, 2, 159, 101, 90, 158, 185, 176, 81, 241, 50, 81, 214, 91, 223, 153, 211, 144, 184, 246, 120, 152, 235, 34, 22, 236, 16, 186, 203, 41, 53, 139, 137, 85, 41, 219, 100, 251, 252, 148, 47, 208, 255), (109, 139, 140, 223, 123, 105, 156, 2, 5, 198, 254, 180, 172, 24, 57, 211, 196, 54, 207, 150, 47, 133, 117, 238, 103, 255, 32, 214, 145, 3, 196, 170, 147, 187, 54, 157, 54, 9, 128, 24, 30, 56, 196, 66, 21, 6, 92, 153, 160, 102, 148, 103, 51, 237, 226, 49, 133, 24, 54, 23), (40, 87, 199, 180, 34, 26, 2, 182, 113, 122, 28, 103, 177, 238, 182, 77, 205, 168, 22, 34, 132, 250, 174, 136, 70, 100, 20, 179, 23, 228, 84, 87, 181, 170, 239, 91, 80, 137, 114, 47), GNUTLS_MAC_SHA384),
	TEST_VECTOR((181, 125, 197, 35, 84, 175, 238, 17, 237, 180, 201, 5, 42, 82, 131, 68, 52, 139, 44, 107, 108, 57, 243, 33, 51, 237, 59, 183, 32, 53, 164, 171, 85, 214, 100, 140, 21, 41, 239, 122, 145, 112, 254, 201, 239, 38, 168, 30), (23, 230, 65, 144, 157, 237, 254, 228, 150, 139, 185, 93, 127, 119, 14, 69, 87, 202, 52, 122, 70, 97, 76, 179, 113, 66, 63, 13, 145, 223, 59, 88, 181, 54, 237, 84, 83, 31, 210, 162, 235, 11, 139, 42, 22, 52, 194, 60, 136, 250, 217, 112, 108, 69, 219, 68, 17, 162, 59, 137), (89, 73, 172, 249, 99, 90, 119, 41, 121, 40, 193, 225, 85, 212, 58, 78, 75, 202, 97, 177, 54, 154, 94, 245, 5, 48, 136, 133, 80, 186, 39, 14, 38, 190, 74, 66, 28, 223, 128, 183), GNUTLS_MAC_SHA384),
	TEST_VECTOR((221, 93, 189, 69, 89, 62, 226, 172, 19, 151, 72, 231, 100, 91, 69, 15, 34, 61, 47, 242, 151, 183, 63, 215, 28, 188, 235, 231, 29, 65, 101, 60, 149, 11, 136, 80, 13, 229, 50, 45, 153, 239, 24, 223, 221, 48, 66, 130, 148, 196, 179, 9, 79, 76, 149, 67, 52, 229, 147, 189, 152, 46, 198, 20), (181, 11, 12, 150, 60, 107, 48, 52, 184, 207, 25, 205, 63, 92, 78, 190, 79, 73, 133, 175, 12, 3, 229, 117, 219, 98, 230, 253, 241, 236, 254, 79, 40, 185, 93, 124, 225, 109, 248, 88, 67, 36, 110, 21, 87, 206, 149, 187, 38, 204, 154, 33, 151, 75, 189, 46, 182, 158, 131, 85), (229, 153, 59, 249, 189, 42, 161, 196, 87, 70, 4, 46, 18, 89, 129, 85), GNUTLS_MAC_SHA512),
	TEST_VECTOR((96, 36, 189, 200, 36, 64, 71, 59, 175, 121, 134, 83, 188, 184, 70, 248, 80, 61, 115, 182, 237, 245, 206, 188, 17, 99, 116, 83, 139, 98, 86, 172, 138, 138, 213, 250, 140, 127, 173, 123, 63, 8, 153, 51, 185, 199, 50, 109, 107, 128, 87, 38, 53, 201, 245, 246, 179, 134, 67, 151, 29, 7, 91, 159), (20, 114, 169, 107, 200, 24, 129, 118, 127, 97, 84, 178, 187, 121, 244, 218, 133, 120, 212, 71, 172, 73, 93, 126, 222, 49, 69, 72, 52, 190, 61, 100, 48, 52, 178, 225, 96, 52, 186, 135, 122, 132, 110, 110, 110, 34, 178, 132, 182, 216, 148, 57, 95, 51, 180, 190, 165, 241, 205, 123), (172, 189, 118, 30, 151, 101, 118, 177, 137, 105, 109, 38, 231, 69, 166, 128), GNUTLS_MAC_SHA512),
	TEST_VECTOR((108, 125, 148, 98, 42, 45, 28, 67, 148, 118, 138, 57, 204, 52, 12, 104, 135, 224, 108, 74, 136, 213, 122, 167, 130, 47, 15, 43, 63, 172, 1, 146, 232, 81, 247, 221, 57, 203, 239, 230, 204, 215, 9, 146, 226, 126, 222, 164, 114, 155, 33, 92, 162, 218, 203, 5, 55, 58, 65, 22, 0, 35, 60, 202), (82, 209, 238, 139, 76, 10, 236, 119, 30, 35, 110, 134, 146, 139, 78, 148, 60, 236, 83, 64, 24, 72, 184, 163, 83, 251, 45, 192, 199, 77, 156, 255, 116, 232, 8, 110, 245, 84, 46, 63, 33, 2, 9, 255, 97, 77, 31, 211, 23, 123, 93, 244, 219, 248, 153, 120, 209, 171, 219, 170), (243, 4, 138, 234, 225, 27, 17, 106, 35, 70, 89, 212, 7, 17, 38, 125), GNUTLS_MAC_SHA512),
	TEST_VECTOR((224, 99, 5, 69, 235, 220, 68, 15, 55, 61, 25, 79, 198, 196, 98, 154, 98, 174, 240, 40, 219, 82, 163, 250, 220, 211, 45, 162, 76, 47, 247, 145, 6, 211, 73, 175, 216, 80, 13, 107, 79, 251, 254, 42, 157, 89, 128, 63, 185, 15, 37, 182, 15, 208, 41, 195, 182, 124, 33, 220, 7, 13, 133, 47), (24, 33, 203, 59, 28, 222, 130, 190, 183, 110, 85, 202, 200, 102, 227, 187, 247, 235, 84, 30, 10, 166, 108, 61, 251, 229, 9, 234, 120, 112, 103, 7, 69, 172, 213, 69, 29, 119, 84, 100, 174, 210, 214, 110, 92, 196, 54, 12, 107, 123, 117, 179, 90, 56, 32, 206, 194, 23, 160, 134), (144, 251, 250, 39, 17, 148, 80, 172, 200, 230, 215, 75, 3, 176, 85, 88), GNUTLS_MAC_SHA512),
	TEST_VECTOR((123, 48, 136, 240, 254, 134, 160, 199, 28, 192, 38, 96, 227, 171, 4, 211, 120, 149, 0, 171, 149, 81, 221, 89, 240, 48, 132, 125, 218, 14, 20, 221, 168, 214, 162, 5, 19, 121, 208, 245, 145, 38, 156, 61, 255, 48, 60, 191, 160, 132, 36, 244, 33, 145, 59, 59, 249, 195, 192, 118, 1, 42, 216, 252), (133, 36, 253, 62, 115, 235, 148, 118, 81, 62, 49, 0, 93, 41, 223, 230, 146, 125, 135, 123, 116, 115, 78, 221, 140, 43, 136, 111, 253, 63, 5, 48, 0, 210, 131, 68, 238, 159, 217, 52, 11, 68, 140, 41, 251, 23, 37, 101, 123, 49, 77, 185, 60, 99, 159, 152, 215, 86, 205, 157), (81, 34, 166, 26, 221, 88, 153, 147, 17, 23, 57, 244, 100, 62, 228, 6), GNUTLS_MAC_SHA512),
	TEST_VECTOR((244, 238, 53, 209, 87, 93, 39, 60, 228, 42, 102, 214, 166, 229, 33, 232, 127, 173, 100, 191, 129, 201, 159, 87, 115, 182, 12, 10, 159, 226, 212, 217, 124, 82, 230, 224, 193, 119, 91, 78, 247, 228, 30, 146, 161, 189, 50, 234, 191, 63, 186, 219, 147, 246, 160, 179, 231, 50, 75, 125, 160, 67, 214, 201), (146, 54, 79, 4, 78, 13, 100, 168, 251, 76, 83, 203, 27, 115, 166, 101, 170, 246, 111, 252, 157, 223, 65, 82, 174, 157, 163, 91, 71, 2, 86, 176, 11, 80, 35, 78, 38, 195, 20, 159, 28, 56, 13, 217, 62, 117, 172, 202, 204, 209, 103, 225, 214, 10, 138, 6, 211, 27, 178, 232), (226, 180, 203, 24, 10, 182, 115, 11, 176, 193, 97, 120, 178, 5, 72, 182), GNUTLS_MAC_SHA512),
	TEST_VECTOR((219, 61, 184, 118, 22, 103, 19, 167, 5, 42, 153, 84, 242, 211, 126, 243, 94, 68, 99, 104, 207, 132, 206, 92, 49, 91, 79, 89, 172, 0, 208, 217, 235, 34, 177, 25, 208, 64, 186, 181, 242, 95, 34, 74, 36, 217, 73, 1, 1, 90, 34, 35, 133, 98, 3, 218, 210, 38, 1, 100, 236, 230, 25, 53), (119, 213, 175, 193, 67, 100, 138, 36, 155, 23, 175, 16, 214, 136, 24, 138, 120, 5, 66, 219, 127, 46, 61, 66, 101, 209, 110, 79, 201, 76, 218, 177, 88, 197, 175, 185, 199, 10, 242, 209, 102, 138, 0, 138, 173, 74, 218, 100, 169, 219, 250, 151, 180, 57, 0, 46, 75, 211, 219, 246), (176, 28, 150, 186, 242, 140, 165, 132, 28, 182, 228, 224, 202, 227, 13, 89), GNUTLS_MAC_SHA512),
	TEST_VECTOR((124, 151, 123, 116, 205, 248, 119, 33, 163, 124, 111, 94, 157, 124, 145, 209, 225, 23, 68, 81, 77, 81, 166, 141, 18, 209, 131, 125, 38, 183, 150, 17, 232, 106, 5, 164, 104, 93, 190, 232, 235, 11, 48, 150, 43, 25, 185, 111, 55, 63, 90, 190, 146, 255, 2, 53, 164, 174, 124, 53, 56, 7, 148, 165), (141, 163, 114, 228, 213, 52, 65, 109, 156, 79, 236, 249, 210, 45, 197, 118, 248, 137, 188, 111, 39, 69, 60, 167, 84, 88, 204, 216, 248, 138, 134, 0, 186, 3, 212, 209, 148, 128, 210, 239, 137, 157, 199, 45, 240, 115, 47, 206, 232, 237, 153, 127, 234, 79, 45, 82, 107, 236, 68, 243), (178, 45, 182, 172, 58, 136, 170, 26, 160, 147, 192, 254, 85, 226, 99, 159), GNUTLS_MAC_SHA512),
	TEST_VECTOR((110, 110, 208, 31, 241, 45, 50, 81, 57, 107, 245, 96, 98, 177, 179, 199, 155, 85, 180, 55, 60, 3, 49, 177, 74, 165, 147, 163, 238, 0, 91, 95, 6, 143, 14, 252, 86, 2, 111, 199, 45, 102, 242, 116, 77, 209, 241, 104, 242, 71, 145, 47, 155, 38, 235, 197, 148, 99, 185, 107, 217, 241, 161, 161), (94, 202, 121, 107, 239, 3, 27, 135, 82, 30, 208, 144, 75, 241, 216, 85, 88, 151, 73, 176, 24, 62, 153, 61, 153, 247, 65, 97, 155, 98, 100, 79, 166, 134, 164, 201, 15, 127, 30, 110, 213, 52, 78, 183, 88, 128, 114, 76, 9, 183, 81, 97, 124, 49, 250, 85, 73, 130, 138, 37), (176, 59, 35, 95, 90, 249, 113, 159, 53, 242, 21, 194, 249, 74, 118, 165), GNUTLS_MAC_SHA512),
	TEST_VECTOR((244, 205, 79, 39, 154, 18, 140, 116, 229, 235, 130, 20, 114, 9, 130, 4, 237, 150, 237, 97, 234, 201, 40, 27, 205, 83, 253, 230, 137, 10, 44, 187, 28, 219, 151, 224, 227, 67, 252, 133, 136, 185, 50, 205, 112, 26, 248, 143, 10, 122, 247, 35, 213, 192, 133, 14, 62, 1, 228, 118, 18, 252, 236, 131), (239, 71, 13, 252, 213, 115, 89, 245, 88, 53, 79, 132, 95, 200, 202, 58, 26, 103, 65, 154, 109, 15, 109, 214, 152, 199, 143, 154, 87, 232, 32, 195, 216, 120, 111, 60, 88, 159, 150, 147, 184, 243, 251, 62, 18, 58, 72, 35, 134, 193, 192, 207, 226, 107, 197, 50, 46, 144, 253, 75), (250, 167, 32, 41, 149, 19, 183, 61, 226, 181, 72, 150, 217, 160, 107, 204), GNUTLS_MAC_SHA512),
	TEST_VECTOR((91, 226, 191, 127, 94, 37, 39, 225, 95, 230, 92, 222, 69, 7, 217, 139, 165, 84, 87, 0, 104, 103, 222, 158, 79, 54, 100, 91, 207, 244, 202, 56, 117, 79, 146, 137, 139, 28, 85, 68, 113, 129, 2, 89, 59, 140, 38, 212, 93, 31, 206, 174, 162, 125, 151, 237, 233, 222, 139, 158, 191, 232, 128, 147), (0, 75, 19, 193, 246, 40, 203, 122, 0, 217, 73, 137, 55, 191, 67, 123, 113, 254, 25, 108, 201, 22, 196, 125, 41, 143, 162, 150, 198, 184, 97, 136, 7, 53, 67, 187, 198, 107, 117, 53, 235, 23, 181, 207, 67, 195, 121, 68, 182, 202, 18, 37, 41, 138, 158, 86, 52, 19, 229, 187), (206, 224, 193, 27, 226, 216, 17, 11, 128, 143, 115, 133, 35, 231, 24, 68, 125, 120, 88, 120, 187, 183, 131, 251, 8, 26, 5, 81, 96, 89, 0, 114), GNUTLS_MAC_SHA512),
	TEST_VECTOR((176, 181, 89, 156, 41, 204, 231, 100, 67, 114, 178, 120, 163, 39, 95, 62, 7, 35, 115, 159, 66, 188, 3, 252, 154, 74, 202, 9, 217, 206, 207, 154, 253, 195, 3, 147, 221, 28, 247, 73, 146, 54, 35, 150, 8, 162, 88, 245, 196, 142, 102, 182, 99, 246, 82, 16, 73, 164, 130, 182, 61, 70, 138, 51), (240, 1, 1, 243, 142, 93, 96, 152, 207, 234, 76, 124, 132, 144, 235, 85, 5, 57, 0, 0, 71, 138, 100, 94, 102, 33, 177, 247, 17, 16, 91, 219, 72, 32, 220, 12, 120, 59, 249, 128, 192, 98, 137, 155, 112, 135, 94, 23, 204, 145, 22, 242, 250, 88, 138, 148, 36, 5, 187, 214), (118, 135, 150, 23, 156, 165, 17, 184, 65, 113, 120, 135, 129, 15, 135, 15, 163, 193, 38, 65, 201, 94, 205, 206, 168, 89, 13, 115, 14, 164, 59, 142), GNUTLS_MAC_SHA512),
	TEST_VECTOR((82, 190, 199, 3, 222, 201, 75, 128, 237, 47, 132, 73, 199, 82, 238, 75, 139, 139, 27, 128, 229, 91, 107, 14, 30, 157, 177, 119, 13, 173, 251, 178, 99, 29, 245, 84, 209, 186, 65, 187, 189, 14, 88, 115, 105, 172, 129, 180, 11, 166, 217, 65, 251, 98, 219, 167, 138, 118, 181, 84, 235, 3, 198, 67), (159, 81, 37, 107, 24, 150, 146, 88, 102, 242, 32, 83, 52, 142, 246, 232, 165, 46, 197, 128, 177, 243, 43, 235, 37, 116, 175, 210, 58, 84, 96, 195, 14, 140, 129, 29, 14, 122, 40, 135, 189, 92, 192, 101, 62, 200, 214, 141, 204, 164, 63, 113, 168, 223, 31, 1, 147, 32, 91, 117), (177, 154, 199, 212, 239, 222, 124, 98, 25, 30, 167, 101, 48, 211, 13, 250, 85, 16, 204, 116, 186, 250, 202, 234, 92, 190, 44, 237, 70, 127, 161, 165), GNUTLS_MAC_SHA512),
	TEST_VECTOR((140, 71, 40, 73, 193, 33, 83, 213, 164, 115, 96, 238, 208, 67, 134, 137, 208, 86, 192, 132, 150, 210, 133, 161, 121, 41, 122, 147, 223, 245, 0, 112, 253, 60, 44, 45, 136, 241, 196, 162, 91, 119, 193, 165, 239, 180, 123, 184, 248, 129, 200, 159, 36, 212, 100, 188, 35, 196, 242, 0, 41, 29, 98, 81), (242, 211, 23, 240, 32, 178, 18, 6, 110, 247, 200, 105, 201, 197, 102, 153, 193, 212, 223, 219, 116, 207, 103, 155, 151, 215, 24, 231, 1, 228, 66, 177, 23, 166, 209, 22, 43, 248, 37, 239, 73, 31, 166, 40, 235, 36, 154, 28, 226, 128, 24, 184, 165, 189, 11, 27, 41, 110, 77, 146), (255, 130, 180, 208, 252, 158, 23, 192, 83, 159, 162, 254, 102, 242, 225, 160, 123, 123, 89, 58, 19, 66, 205, 11, 72, 22, 47, 218, 104, 183, 216, 10), GNUTLS_MAC_SHA512),
	TEST_VECTOR((16, 66, 128, 200, 108, 121, 135, 158, 150, 207, 21, 222, 150, 65, 36, 146, 255, 245, 212, 219, 252, 103, 127, 125, 56, 76, 9, 164, 188, 238, 22, 15, 163, 201, 114, 150, 157, 18, 249, 177, 84, 45, 87, 11, 170, 235, 189, 128, 31, 99, 46, 23, 42, 138, 0, 71, 126, 220, 6, 249, 56, 234, 225, 152), (173, 241, 241, 103, 147, 144, 135, 215, 31, 232, 217, 221, 210, 201, 3, 197, 66, 119, 81, 41, 218, 178, 237, 194, 68, 108, 179, 148, 168, 8, 186, 73, 207, 71, 16, 234, 16, 203, 114, 164, 239, 232, 241, 163, 44, 3, 209, 70, 96, 241, 11, 10, 169, 37, 91, 208, 238, 134, 92, 16), (92, 237, 129, 66, 48, 215, 240, 125, 161, 22, 215, 147, 11, 77, 182, 177, 139, 57, 27, 184, 105, 209, 81, 150, 103, 123, 209, 209, 127, 47, 140, 124), GNUTLS_MAC_SHA512),
	TEST_VECTOR((218, 72, 152, 107, 140, 132, 208, 246, 178, 214, 186, 159, 44, 2, 165, 62, 216, 241, 115, 178, 56, 221, 143, 48, 32, 225, 94, 174, 66, 88, 113, 207, 241, 72, 241, 91, 176, 186, 23, 194, 174, 0, 244, 75, 44, 20, 114, 90, 123, 19, 5, 5, 70, 155, 192, 112, 107, 105, 158, 45, 144, 26, 209, 107), (8, 82, 134, 0, 35, 123, 109, 4, 146, 104, 183, 191, 62, 47, 205, 147, 115, 166, 178, 246, 115, 46, 200, 148, 55, 239, 142, 19, 79, 185, 71, 195, 243, 127, 37, 76, 112, 65, 211, 240, 111, 120, 153, 83, 186, 206, 212, 21, 233, 209, 220, 64, 15, 176, 2, 91, 225, 128, 66, 91), (80, 235, 41, 233, 205, 92, 167, 115, 15, 165, 101, 135, 11, 35, 164, 13, 183, 91, 94, 5, 27, 53, 133, 12, 152, 252, 150, 19, 162, 81, 189, 203), GNUTLS_MAC_SHA512),
	TEST_VECTOR((25, 94, 49, 179, 219, 70, 48, 169, 230, 108, 222, 52, 119, 22, 245, 2, 39, 248, 168, 28, 57, 122, 114, 227, 13, 39, 170, 194, 127, 139, 159, 47, 43, 69, 62, 104, 250, 188, 157, 32, 157, 68, 61, 149, 34, 99, 219, 12, 52, 28, 134, 109, 150, 110, 180, 219, 143, 138, 207, 142, 175, 231, 162, 39), (43, 40, 21, 38, 52, 83, 81, 215, 151, 190, 10, 125, 188, 81, 123, 185, 201, 194, 32, 181, 25, 76, 236, 56, 71, 246, 250, 196, 171, 16, 48, 134, 125, 24, 81, 203, 141, 70, 86, 29, 182, 5, 80, 101, 103, 220, 47, 57, 221, 219, 244, 113, 228, 167, 213, 27, 35, 116, 85, 146), (174, 255, 136, 77, 203, 220, 0, 42, 95, 122, 135, 122, 52, 247, 59, 66, 171, 173, 32, 137, 212, 227, 155, 114, 30, 179, 112, 179, 216, 118, 39, 244), GNUTLS_MAC_SHA512),
	TEST_VECTOR((122, 31, 18, 163, 250, 215, 224, 211, 226, 68, 252, 123, 211, 240, 145, 126, 71, 188, 118, 198, 2, 133, 222, 215, 57, 221, 249, 70, 44, 22, 12, 223, 204, 168, 135, 134, 109, 109, 174, 251, 241, 46, 116, 16, 251, 27, 80, 173, 108, 139, 33, 182, 229, 104, 119, 244, 126, 139, 20, 160, 172, 159, 147, 109), (119, 9, 112, 36, 201, 68, 227, 73, 240, 202, 42, 87, 156, 245, 19, 83, 245, 6, 100, 221, 180, 187, 189, 187, 224, 29, 253, 232, 5, 149, 218, 165, 208, 67, 217, 219, 150, 81, 251, 130, 28, 201, 172, 215, 47, 169, 27, 46, 233, 62, 35, 239, 179, 148, 9, 113, 31, 57, 21, 208), (210, 25, 27, 106, 172, 199, 102, 128, 56, 215, 98, 192, 205, 89, 21, 191, 104, 228, 144, 103, 30, 196, 3, 44, 32, 41, 122, 100, 2, 111, 224, 66), GNUTLS_MAC_SHA512),
	TEST_VECTOR((122, 65, 223, 232, 208, 201, 172, 146, 74, 42, 204, 214, 21, 173, 60, 11, 192, 215, 225, 36, 254, 214, 178, 183, 224, 91, 159, 142, 106, 222, 6, 120, 233, 141, 69, 118, 111, 150, 151, 205, 24, 123, 193, 3, 33, 25, 95, 161, 187, 132, 181, 91, 249, 75, 177, 18, 215, 60, 106, 187, 133, 127, 64, 161), (228, 122, 168, 26, 248, 209, 7, 76, 143, 127, 188, 181, 137, 114, 126, 194, 37, 22, 61, 7, 109, 78, 140, 239, 200, 39, 199, 142, 186, 123, 66, 213, 231, 3, 119, 215, 3, 61, 253, 245, 237, 217, 63, 96, 93, 217, 178, 93, 117, 84, 133, 24, 228, 5, 86, 61, 29, 75, 243, 19), (93, 191, 29, 61, 193, 40, 230, 127, 36, 82, 171, 168, 88, 183, 236, 251, 135, 239, 190, 102, 183, 20, 84, 28, 13, 76, 0, 156, 43, 146, 215, 249), GNUTLS_MAC_SHA512),
	TEST_VECTOR((178, 123, 227, 35, 238, 9, 68, 0, 252, 157, 172, 174, 238, 3, 101, 224, 192, 197, 40, 17, 166, 8, 21, 81, 221, 70, 221, 134, 48, 44, 99, 150, 252, 232, 11, 182, 14, 48, 108, 151, 133, 190, 161, 39, 3, 86, 161, 194, 180, 219, 118, 52, 213, 192, 72, 145, 232, 84, 23, 251, 114, 205, 250, 178), (110, 237, 234, 76, 124, 42, 210, 34, 137, 196, 230, 106, 150, 231, 167, 161, 202, 231, 211, 64, 250, 228, 18, 166, 231, 118, 151, 194, 8, 26, 39, 86, 10, 159, 12, 176, 228, 18, 41, 127, 153, 16, 58, 35, 58, 121, 152, 101, 5, 144, 197, 167, 226, 134, 23, 56, 190, 215, 201, 253), (208, 101, 98, 48, 184, 110, 77, 195, 234, 43, 219, 141, 98, 180, 82, 38, 36, 98, 238, 214, 18, 41, 209, 226, 185, 39, 88, 181, 124, 84, 196, 52), GNUTLS_MAC_SHA512),
	TEST_VECTOR((157, 208, 56, 100, 163, 26, 164, 21, 108, 167, 161, 32, 0, 245, 65, 104, 12, 224, 165, 244, 119, 94, 239, 16, 136, 172, 19, 54, 130, 0, 180, 71, 167, 141, 11, 241, 68, 22, 161, 213, 131, 197, 75, 15, 17, 32, 15, 244, 168, 152, 61, 215, 117, 206, 156, 3, 2, 210, 98, 72, 62, 48, 10, 230), (3, 115, 105, 241, 66, 214, 105, 252, 169, 232, 126, 159, 55, 174, 143, 44, 141, 80, 107, 117, 63, 223, 232, 163, 183, 47, 117, 202, 193, 197, 15, 161, 248, 98, 8, 131, 184, 220, 184, 220, 198, 122, 220, 201, 94, 112, 170, 98, 74, 219, 159, 225, 178, 203, 57, 102, 146, 176, 210, 232), (150, 232, 209, 188, 1, 220, 149, 192, 191, 66, 195, 195, 143, 197, 76, 9, 3, 115, 206, 212), GNUTLS_MAC_SHA512),
	TEST_VECTOR((106, 193, 155, 54, 175, 125, 14, 92, 101, 245, 141, 218, 47, 247, 108, 104, 86, 124, 216, 172, 22, 255, 36, 99, 252, 60, 134, 84, 222, 69, 160, 94, 57, 199, 93, 131, 117, 131, 81, 117, 61, 46, 20, 54, 41, 171, 144, 117, 115, 139, 3, 122, 30, 244, 31, 126, 253, 112, 241, 231, 0, 247, 246, 123), (75, 169, 249, 232, 227, 59, 64, 244, 57, 3, 51, 245, 151, 171, 86, 91, 39, 168, 65, 213, 250, 201, 48, 246, 113, 1, 230, 89, 115, 240, 113, 172, 107, 163, 148, 28, 157, 34, 244, 12, 164, 78, 145, 101, 122, 98, 155, 127, 86, 152, 14, 14, 204, 174, 203, 148, 59, 22, 238, 39), (62, 67, 183, 6, 51, 136, 68, 128, 221, 217, 61, 111, 32, 186, 81, 43, 85, 89, 11, 104), GNUTLS_MAC_SHA512),
	TEST_VECTOR((46, 206, 123, 201, 235, 82, 73, 245, 23, 159, 37, 40, 160, 98, 208, 220, 48, 73, 102, 4, 187, 118, 242, 110, 22, 13, 54, 51, 20, 15, 198, 237, 172, 113, 126, 181, 64, 45, 15, 206, 86, 183, 107, 124, 251, 10, 175, 71, 182, 175, 196, 16, 254, 6, 239, 176, 73, 250, 40, 149, 61, 126, 73, 66), (30, 225, 239, 227, 174, 36, 136, 25, 197, 179, 61, 235, 120, 105, 195, 84, 12, 216, 171, 183, 221, 82, 44, 95, 10, 183, 22, 24, 21, 156, 208, 210, 61, 216, 5, 241, 78, 226, 126, 1, 34, 64, 207, 248, 95, 87, 199, 47, 136, 213, 239, 110, 33, 150, 238, 222, 140, 247, 183, 134), (45, 244, 250, 82, 210, 133, 135, 0, 8, 85, 112, 144, 31, 95, 133, 115, 20, 225, 224, 45), GNUTLS_MAC_SHA512),
	TEST_VECTOR((40, 62, 55, 254, 24, 21, 22, 222, 43, 251, 20, 230, 41, 30, 219, 145, 219, 222, 74, 123, 20, 117, 53, 42, 204, 71, 239, 223, 36, 74, 43, 162, 230, 57, 217, 176, 40, 192, 45, 254, 191, 81, 221, 45, 55, 196, 218, 100, 170, 137, 230, 166, 73, 22, 213, 143, 28, 128, 178, 92, 192, 128, 169, 80), (87, 140, 254, 26, 112, 245, 47, 126, 69, 106, 34, 241, 72, 91, 162, 56, 48, 227, 220, 176, 87, 0, 237, 40, 18, 202, 220, 24, 29, 68, 143, 17, 25, 254, 205, 143, 203, 73, 102, 115, 9, 207, 14, 49, 100, 112, 89, 14, 217, 59, 175, 86, 127, 70, 157, 239, 93, 175, 195, 177), (36, 113, 156, 182, 246, 177, 14, 46, 181, 134, 160, 225, 103, 245, 104, 61, 242, 17, 55, 113), GNUTLS_MAC_SHA512),
	TEST_VECTOR((74, 102, 212, 30, 188, 177, 132, 195, 20, 138, 13, 235, 214, 122, 245, 200, 236, 227, 230, 161, 193, 60, 65, 55, 137, 121, 106, 153, 161, 237, 36, 173, 162, 245, 16, 124, 145, 21, 147, 6, 254, 143, 104, 194, 69, 73, 159, 191, 216, 248, 80, 9, 161, 0, 200, 83, 93, 34, 220, 113, 223, 9, 247, 79), (64, 234, 244, 41, 247, 222, 230, 245, 201, 31, 35, 81, 186, 81, 229, 60, 246, 247, 248, 54, 57, 141, 188, 195, 51, 129, 180, 234, 64, 70, 165, 242, 205, 84, 92, 147, 46, 88, 2, 122, 30, 114, 79, 207, 133, 82, 204, 121, 171, 188, 237, 237, 56, 90, 82, 70, 72, 193, 218, 228), (143, 27, 220, 185, 154, 240, 252, 81, 10, 212, 130, 94, 253, 215, 80, 211, 202, 183, 132, 111), GNUTLS_MAC_SHA512),
	TEST_VECTOR((149, 165, 59, 63, 149, 58, 185, 13, 79, 133, 231, 232, 67, 115, 37, 190, 26, 2, 53, 168, 147, 17, 250, 28, 116, 5, 195, 192, 188, 97, 241, 78, 246, 231, 202, 211, 33, 101, 174, 234, 42, 215, 24, 190, 241, 68, 214, 99, 2, 233, 137, 226, 241, 163, 35, 229, 18, 120, 13, 40, 182, 218, 116, 172), (55, 73, 82, 22, 193, 19, 239, 210, 204, 174, 243, 88, 58, 135, 176, 103, 214, 92, 22, 93, 186, 206, 169, 224, 78, 227, 68, 17, 234, 166, 97, 111, 206, 83, 72, 171, 189, 18, 17, 134, 242, 158, 7, 28, 70, 123, 96, 217, 41, 28, 162, 235, 102, 78, 188, 42, 126, 229, 14, 171), (113, 165, 177, 234, 180, 17, 175, 186, 101, 8, 29, 77, 4, 113, 213, 50, 191, 14, 198, 95), GNUTLS_MAC_SHA512),
	TEST_VECTOR((252, 62, 242, 144, 168, 155, 121, 233, 93, 232, 51, 38, 179, 66, 66, 119, 143, 210, 213, 66, 69, 115, 101, 226, 113, 157, 50, 143, 165, 88, 20, 1, 235, 254, 110, 38, 85, 218, 7, 10, 163, 226, 107, 54, 59, 89, 94, 26, 247, 186, 189, 23, 7, 93, 190, 241, 119, 92, 246, 21, 125, 60, 113, 107), (89, 46, 25, 132, 189, 251, 75, 241, 141, 232, 58, 107, 25, 205, 182, 63, 39, 177, 9, 133, 231, 202, 82, 11, 113, 236, 57, 44, 224, 158, 29, 45, 161, 76, 99, 37, 83, 87, 75, 135, 103, 82, 185, 213, 114, 236, 244, 27, 197, 175, 208, 87, 190, 153, 216, 1, 158, 25, 247, 232), (8, 60, 211, 176, 181, 100, 238, 124, 94, 146, 127, 50, 209, 1, 144, 116, 151, 21, 178, 28), GNUTLS_MAC_SHA512),
	TEST_VECTOR((56, 139, 140, 155, 185, 136, 231, 98, 237, 227, 204, 7, 43, 216, 139, 167, 41, 71, 133, 175, 176, 214, 80, 108, 35, 87, 165, 226, 51, 49, 165, 30, 150, 143, 123, 245, 5, 118, 239, 158, 100, 182, 155, 79, 72, 214, 229, 9, 99, 156, 233, 105, 62, 63, 32, 148, 250, 51, 185, 121, 99, 225, 204, 110), (91, 143, 224, 198, 88, 192, 253, 118, 152, 35, 76, 71, 120, 96, 83, 18, 33, 167, 244, 134, 23, 228, 64, 93, 92, 142, 141, 53, 80, 241, 5, 183, 132, 105, 164, 73, 204, 43, 133, 93, 74, 83, 87, 193, 198, 18, 35, 75, 42, 226, 184, 32, 240, 116, 135, 176, 148, 172, 124, 63), (115, 81, 2, 181, 155, 9, 208, 109, 62, 108, 148, 148, 67, 60, 176, 63, 91, 140, 163, 131), GNUTLS_MAC_SHA512),
	TEST_VECTOR((241, 227, 157, 157, 9, 132, 100, 31, 36, 112, 47, 139, 55, 251, 249, 168, 250, 251, 209, 127, 194, 53, 105, 233, 64, 195, 19, 132, 205, 165, 140, 2, 116, 177, 103, 177, 175, 201, 255, 68, 71, 61, 32, 214, 162, 60, 146, 37, 2, 227, 27, 61, 173, 19, 39, 75, 34, 2, 79, 244, 234, 237, 102, 142), (200, 128, 112, 120, 4, 167, 100, 80, 1, 150, 50, 195, 47, 3, 127, 151, 255, 44, 231, 251, 21, 65, 169, 173, 188, 9, 45, 150, 18, 44, 222, 25, 213, 254, 129, 205, 18, 166, 15, 73, 28, 88, 119, 182, 1, 228, 236, 190, 15, 186, 235, 103, 180, 70, 196, 187, 42, 158, 214, 14), (84, 64, 195, 69, 10, 195, 50, 208, 168, 245, 199, 36, 100, 121, 98, 172, 130, 246, 89, 225), GNUTLS_MAC_SHA512),
	TEST_VECTOR((158, 10, 121, 177, 214, 198, 114, 211, 35, 84, 100, 203, 43, 230, 163, 109, 2, 201, 130, 56, 125, 239, 126, 47, 227, 42, 93, 43, 231, 52, 63, 220, 3, 138, 244, 143, 185, 65, 151, 47, 172, 200, 227, 61, 98, 134, 225, 66, 133, 45, 196, 207, 38, 103, 163, 201, 125, 213, 87, 188, 111, 198, 110, 152), (205, 53, 109, 20, 227, 43, 249, 88, 193, 119, 17, 210, 106, 48, 65, 32, 136, 162, 226, 20, 230, 0, 230, 146, 237, 73, 102, 33, 37, 101, 234, 137, 28, 128, 126, 30, 11, 222, 192, 169, 219, 199, 31, 80, 47, 9, 151, 95, 206, 145, 12, 119, 35, 219, 77, 243, 128, 231, 255, 178), (244, 102, 147, 48, 11, 195, 191, 16, 109, 106, 133, 102, 72, 31, 177, 43, 69, 238, 31, 149), GNUTLS_MAC_SHA512),
	TEST_VECTOR((169, 244, 162, 197, 175, 131, 152, 103, 245, 219, 90, 30, 82, 10, 179, 204, 167, 42, 22, 108, 166, 13, 229, 18, 253, 127, 231, 230, 76, 249, 79, 146, 207, 29, 139, 99, 97, 117, 242, 147, 224, 3, 39, 94, 2, 16, 24, 195, 240, 237, 228, 149, 153, 122, 80, 94, 201, 162, 175, 235, 4, 149, 190, 87), (142, 157, 179, 51, 87, 121, 219, 104, 139, 207, 224, 150, 102, 141, 156, 59, 198, 78, 25, 62, 53, 41, 196, 48, 230, 141, 9, 213, 108, 131, 125, 214, 192, 249, 70, 120, 241, 33, 166, 142, 225, 254, 234, 71, 53, 218, 133, 164, 157, 52, 165, 41, 10, 163, 159, 123, 64, 222, 67, 95), (109, 184, 128, 218, 172, 152, 176, 120, 238, 56, 154, 33, 100, 37, 45, 237, 97, 50, 45, 102, 30, 43, 73, 36, 126, 169, 33, 229, 68, 103, 93, 143, 23, 175, 43, 246, 109, 212, 13, 129), GNUTLS_MAC_SHA512),
	TEST_VECTOR((36, 67, 86, 190, 155, 50, 121, 100, 115, 46, 180, 167, 192, 155, 4, 180, 32, 113, 35, 150, 235, 87, 247, 43, 201, 73, 36, 6, 108, 104, 126, 135, 142, 121, 142, 10, 3, 58, 30, 225, 164, 216, 205, 194, 218, 4, 67, 236, 215, 116, 1, 208, 70, 12, 217, 6, 234, 171, 2, 101, 108, 30, 220, 152), (216, 6, 226, 223, 140, 133, 211, 186, 245, 214, 126, 156, 151, 183, 70, 238, 107, 187, 27, 193, 13, 205, 246, 199, 166, 7, 92, 49, 28, 243, 71, 82, 172, 190, 96, 230, 143, 35, 199, 248, 144, 181, 234, 115, 0, 161, 173, 50, 23, 130, 84, 136, 82, 70, 240, 73, 57, 135, 166, 232), (240, 181, 188, 116, 158, 179, 0, 202, 33, 124, 168, 47, 223, 254, 216, 155, 27, 242, 200, 175, 194, 179, 110, 226, 180, 134, 149, 229, 8, 91, 137, 58, 109, 170, 213, 71, 79, 116, 239, 15), GNUTLS_MAC_SHA512),
	TEST_VECTOR((245, 86, 122, 45, 217, 35, 106, 153, 32, 12, 75, 213, 57, 7, 67, 226, 86, 11, 171, 75, 25, 110, 60, 115, 43, 1, 171, 249, 0, 199, 100, 156, 171, 91, 149, 125, 166, 174, 143, 208, 37, 96, 81, 71, 179, 101, 114, 193, 159, 16, 54, 112, 177, 111, 107, 181, 124, 19, 135, 84, 71, 154, 212, 93), (168, 204, 212, 189, 54, 251, 14, 208, 118, 94, 150, 98, 241, 2, 141, 96, 11, 214, 80, 228, 194, 205, 221, 249, 75, 39, 238, 136, 17, 32, 170, 247, 75, 114, 123, 2, 240, 3, 107, 70, 22, 32, 98, 227, 158, 212, 63, 168, 86, 135, 165, 141, 23, 122, 246, 245, 102, 129, 24, 137), (46, 85, 183, 61, 18, 109, 176, 249, 40, 16, 38, 108, 146, 228, 220, 122, 127, 45, 50, 203, 237, 158, 180, 237, 171, 81, 158, 92, 201, 19, 140, 100, 47, 212, 178, 41, 120, 12, 23, 191), GNUTLS_MAC_SHA512),
	TEST_VECTOR((70, 166, 95, 46, 67, 47, 226, 170, 178, 109, 226, 77, 27, 157, 57, 180, 77, 162, 112, 35, 15, 23, 132, 77, 68, 226, 73, 86, 90, 18, 95, 135, 160, 112, 180, 178, 238, 172, 11, 58, 108, 84, 172, 252, 73, 221, 198, 54, 15, 89, 254, 14, 51, 14, 6, 5, 198, 30, 133, 197, 194, 127, 231, 86), (176, 106, 35, 224, 89, 69, 249, 106, 41, 61, 229, 157, 195, 219, 89, 114, 202, 31, 160, 11, 70, 71, 172, 56, 247, 83, 121, 3, 53, 213, 218, 235, 47, 254, 9, 207, 137, 36, 172, 78, 128, 178, 117, 234, 77, 190, 197, 59, 158, 42, 175, 144, 223, 14, 111, 216, 42, 214, 159, 124), (4, 187, 196, 249, 63, 181, 195, 88, 150, 144, 121, 139, 247, 147, 193, 11, 183, 38, 248, 122, 74, 45, 233, 59, 143, 223, 107, 232, 1, 90, 190, 21, 101, 119, 17, 156, 86, 55, 174, 113), GNUTLS_MAC_SHA512),
	TEST_VECTOR((141, 55, 21, 82, 22, 75, 206, 230, 7, 113, 232, 118, 206, 84, 108, 75, 2, 119, 185, 120, 224, 150, 62, 35, 18, 102, 111, 237, 95, 170, 243, 64, 165, 94, 0, 100, 76, 179, 100, 116, 70, 74, 76, 39, 117, 244, 128, 216, 142, 45, 59, 65, 243, 30, 7, 6, 32, 158, 184, 229, 21, 149, 185, 46), (245, 112, 33, 45, 178, 211, 243, 20, 175, 174, 81, 248, 56, 117, 228, 218, 125, 188, 19, 7, 62, 206, 131, 166, 114, 126, 1, 29, 208, 247, 72, 245, 105, 113, 75, 89, 244, 73, 48, 6, 123, 137, 98, 184, 50, 137, 180, 65, 186, 101, 176, 226, 35, 122, 173, 163, 104, 0, 249, 85), (149, 101, 181, 66, 89, 252, 92, 97, 167, 247, 68, 142, 107, 223, 190, 72, 25, 28, 45, 214, 202, 97, 206, 53, 234, 243, 113, 127, 251, 254, 28, 70, 123, 61, 224, 128, 250, 238, 184, 180), GNUTLS_MAC_SHA512),
	TEST_VECTOR((7, 248, 231, 61, 71, 222, 95, 14, 226, 204, 248, 113, 199, 2, 169, 42, 31, 209, 90, 77, 225, 48, 97, 30, 70, 164, 36, 122, 4, 151, 70, 91, 63, 181, 215, 133, 192, 30, 209, 207, 6, 28, 115, 254, 31, 85, 223, 224, 17, 30, 199, 53, 211, 118, 187, 57, 172, 108, 248, 61, 122, 35, 113, 196), (193, 58, 173, 196, 77, 156, 58, 32, 155, 144, 129, 177, 155, 230, 85, 108, 159, 231, 180, 145, 224, 184, 217, 139, 24, 172, 130, 198, 180, 220, 62, 142, 226, 59, 130, 219, 43, 171, 194, 127, 5, 89, 121, 81, 24, 149, 226, 150, 182, 217, 247, 221, 163, 90, 14, 38, 95, 179, 198, 69), (158, 34, 46, 188, 210, 124, 124, 167, 33, 121, 18, 28, 108, 96, 251, 13, 170, 233, 16, 98, 12, 167, 176, 26, 141, 1, 84, 212, 141, 169, 199, 215, 170, 154, 86, 103, 55, 95, 245, 24), GNUTLS_MAC_SHA512),
	TEST_VECTOR((95, 0, 182, 117, 94, 216, 201, 211, 234, 140, 236, 221, 140, 23, 202, 58, 254, 221, 37, 135, 29, 28, 80, 179, 2, 193, 252, 134, 77, 4, 20, 115, 130, 49, 62, 106, 112, 29, 155, 108, 195, 222, 121, 31, 166, 90, 120, 43, 192, 215, 64, 151, 54, 27, 18, 165, 180, 84, 8, 241, 23, 45, 254, 165), (86, 105, 86, 155, 145, 17, 103, 232, 249, 198, 177, 168, 63, 222, 87, 182, 219, 65, 113, 174, 183, 101, 122, 169, 77, 135, 213, 66, 189, 157, 78, 137, 246, 156, 137, 224, 251, 75, 61, 53, 2, 112, 35, 39, 251, 254, 101, 58, 246, 10, 127, 225, 171, 96, 14, 38, 32, 215, 101, 166), (217, 55, 130, 46, 249, 32, 115, 207, 205, 181, 216, 219, 128, 26, 56, 117, 228, 95, 227, 77, 49, 111, 64, 238, 87, 20, 14, 17, 83, 228, 210, 130, 81, 208, 160, 90, 159, 73, 234, 102), GNUTLS_MAC_SHA512),
	TEST_VECTOR((190, 117, 90, 183, 111, 253, 143, 46, 179, 242, 70, 224, 54, 188, 192, 109, 243, 158, 131, 164, 113, 244, 104, 194, 158, 168, 188, 239, 244, 125, 40, 34, 38, 55, 193, 202, 207, 47, 219, 210, 51, 84, 92, 200, 136, 71, 74, 123, 33, 49, 204, 199, 53, 34, 80, 39, 197, 23, 196, 132, 32, 209, 10, 204), (78, 57, 209, 43, 179, 6, 245, 209, 225, 133, 148, 132, 33, 80, 72, 206, 111, 240, 3, 59, 49, 153, 207, 23, 50, 208, 8, 140, 65, 135, 212, 152, 132, 254, 48, 87, 18, 67, 178, 166, 67, 188, 86, 212, 152, 109, 218, 149, 142, 213, 194, 121, 191, 122, 155, 226, 227, 117, 103, 227), (48, 46, 95, 129, 16, 125, 24, 108, 5, 65, 115, 188, 112, 94, 17, 166, 72, 165, 220, 235, 132, 53, 81, 12, 60, 220, 128, 201, 152, 76, 185, 201, 195, 98, 132, 13, 68, 38, 94, 126), GNUTLS_MAC_SHA512),
	TEST_VECTOR((44, 148, 196, 135, 63, 175, 77, 227, 233, 49, 82, 180, 148, 40, 201, 44, 134, 93, 197, 142, 30, 51, 111, 251, 154, 165, 5, 35, 219, 99, 17, 219, 52, 213, 167, 49, 22, 64, 115, 215, 149, 133, 6, 80, 28, 219, 77, 206, 5, 184, 69, 25, 142, 111, 239, 230, 61, 184, 120, 55, 194, 67, 72, 80), (39, 254, 85, 155, 109, 146, 153, 83, 56, 40, 58, 119, 97, 89, 133, 98, 133, 71, 75, 116, 48, 45, 237, 203, 62, 127, 225, 145, 73, 14, 171, 127, 88, 2, 18, 207, 234, 111, 96, 19, 56, 142, 61, 178, 132, 67, 244, 84, 40, 5, 114, 185, 15, 13, 194, 32, 127, 220, 192, 239), (66, 136, 90, 197, 24, 201, 123, 218, 41, 11, 84, 139, 89, 123, 127, 4, 178, 11, 195, 74, 10, 109, 157, 252, 137, 164, 70, 217, 242, 58, 28, 204, 195, 118, 8, 254, 236, 167, 164, 195), GNUTLS_MAC_SHA512),
	TEST_VECTOR((15, 242, 194, 121, 101, 58, 123, 149, 74, 251, 0, 150, 194, 177, 110, 89, 31, 163, 46, 239, 57, 237, 216, 20, 28, 101, 19, 214, 220, 108, 8, 99, 175, 14, 148, 191, 213, 123, 23, 129, 124, 209, 3, 143, 55, 99, 156, 248, 214, 56, 113, 174, 244, 110, 232, 25, 71, 82, 107, 197, 69, 76, 19, 242), (22, 238, 29, 174, 246, 160, 49, 106, 160, 70, 118, 70, 197, 33, 250, 48, 22, 95, 227, 54, 178, 73, 96, 15, 30, 86, 93, 40, 123, 151, 1, 128, 51, 226, 186, 212, 93, 118, 198, 104, 93, 119, 51, 155, 39, 235, 221, 156, 206, 27, 52, 193, 228, 97, 154, 151, 119, 77, 148, 231), (188, 14, 60, 178, 183, 139, 217, 214, 226, 236, 84, 70, 114, 173, 190, 68, 57, 143, 219, 167, 239, 51, 47, 28, 66, 53, 193, 4, 202, 50, 236, 0, 251, 71, 212, 114, 9, 225, 81, 151), GNUTLS_MAC_SHA512),
};

static void test_sp800_108_sha256(void **state)
{
	NTSTATUS status;
	static const uint8_t key[] = {152, 203, 215, 84,  113, 216, 118, 177,
				      81,  128, 50,  160, 148, 132, 82,	 244,
				      65,  179, 164, 219, 209, 14,  33,	 131,
				      178, 193, 80,  248, 126, 23,  66,	 227,
				      45,  221, 171, 12,  247, 15,  62,	 179,
				      164, 217, 123, 179, 106, 118, 228, 74,
				      12,  2,	241, 229, 139, 55,  237, 155,
				      220, 122, 200, 245, 129, 222, 37,	 15};
	static const uint8_t context[] = {114, 233, 112, 1,   53,  160, 76,
					  175, 153, 59,	 224, 82,  213, 189,
					  18,  22,  106, 1,   0,   0,	255,
					  255, 255, 255, 255, 255, 255, 255};
	static const uint8_t label[] = {'K', 0, 'D', 0, 'S', 0, ' ', 0,
					's', 0, 'e', 0, 'r', 0, 'v', 0,
					'i', 0, 'c', 0, 'e', 0, 0,   0};
	static const uint8_t expected[] = {219, 94,  173, 243, 157, 13,	 49,
					   57,	54,  3,	  127, 239, 193, 4,
					   220, 218, 252, 33,  105, 76,	 18,
					   140, 166, 177, 95,  65,  164, 18,
					   52,	169, 9,	  194};
	uint8_t out[sizeof expected];

	status = samba_gnutls_sp800_108_derive_key(key,
						   sizeof key,
						   NULL,
						   0,
						   label,
						   sizeof label,
						   context,
						   sizeof context,
						   GNUTLS_MAC_SHA256,
						   out,
						   sizeof out);
	assert_true(NT_STATUS_IS_OK(status));
	assert_memory_equal(expected, out, sizeof out);
}

static void test_sp800_108_sha512(void **state)
{
	NTSTATUS status;
	static const uint8_t key[] = {152, 203, 215, 84,  113, 216, 118, 177,
				      81,  128, 50,  160, 148, 132, 82,	 244,
				      65,  179, 164, 219, 209, 14,  33,	 131,
				      178, 193, 80,  248, 126, 23,  66,	 227,
				      45,  221, 171, 12,  247, 15,  62,	 179,
				      164, 217, 123, 179, 106, 118, 228, 74,
				      12,  2,	241, 229, 139, 55,  237, 155,
				      220, 122, 200, 245, 129, 222, 37,	 15};
	static const uint8_t context[] = {114, 233, 112, 1,   53,  160, 76,
					  175, 153, 59,	 224, 82,  213, 189,
					  18,  22,  106, 1,   0,   0,	255,
					  255, 255, 255, 255, 255, 255, 255};
	static const uint8_t label[] = {'K', 0, 'D', 0, 'S', 0, ' ', 0,
					's', 0, 'e', 0, 'r', 0, 'v', 0,
					'i', 0, 'c', 0, 'e', 0, 0,   0};
	static const uint8_t expected[] = {
		7,   24,  223, 124, 39,	 199, 153, 162, 178, 37,  249, 182, 253,
		103, 255, 46,  60,  102, 61,  116, 186, 74,  221, 37,  242, 137,
		234, 58,  125, 105, 64,	 127, 42,  175, 82,  141, 104, 210, 231,
		17,  116, 215, 15,  144, 200, 234, 66,	162, 196, 216, 48,  111,
		239, 86,  93,  32,  81,	 206, 12,  145, 136, 185, 81,  56};
	uint8_t out[sizeof expected];

	status = samba_gnutls_sp800_108_derive_key(key,
						   sizeof key,
						   NULL,
						   0,
						   label,
						   sizeof label,
						   context,
						   sizeof context,
						   GNUTLS_MAC_SHA512,
						   out,
						   sizeof out);
	assert_true(NT_STATUS_IS_OK(status));
	assert_memory_equal(expected, out, sizeof out);
}

static void test_sp800_108_nist(void **state)
{
	size_t i;

	for (i = 0; i < ARRAY_SIZE(nist_vectors); ++i) {
		NTSTATUS status;
		const struct test_vector *test_vector = &nist_vectors[i];
		uint8_t out[test_vector->KO_len];

		status = samba_gnutls_sp800_108_derive_key(
			test_vector->KI,
			test_vector->KI_len,
			test_vector->input,
			test_vector->input_len,
			NULL,
			0,
			NULL,
			0,
			test_vector->prf_algorithm,
			out,
			sizeof out);
		assert_true(NT_STATUS_IS_OK(status));
		assert_memory_equal(test_vector->KO, out, sizeof out);
	}
}

int main(int argc, char *argv[])
{
	int rc;
	const struct CMUnitTest tests[] = {
		cmocka_unit_test(test_sp800_108_sha256),
		cmocka_unit_test(test_sp800_108_sha512),
		cmocka_unit_test(test_sp800_108_nist),
	};

	if (argc == 2) {
		cmocka_set_test_filter(argv[1]);
	}
	cmocka_set_message_output(CM_OUTPUT_SUBUNIT);

	rc = cmocka_run_group_tests(tests, NULL, NULL);

	return rc;
}
