/*
 * Copyright 2023 NVIDIA Corporation.  All rights reserved.
 *
 * NVIDIA CORPORATION and its licensors retain all intellectual property
 * and proprietary rights in and to this software, related documentation
 * and any modifications thereto.  Any use, reproduction, disclosure or
 * distribution of this software and related documentation without an express
 * license agreement from NVIDIA CORPORATION is strictly prohibited.
 *
 */
#ifndef NV_FM_NVL_AGENT_H
#define NV_FM_NVL_AGENT_H
#ifdef __cplusplus
extern "C" {
#endif
#include "nv_fm_types.h"
#include "nv_fm_nvl_types.h"

#define FM_LIB_API   __attribute__ ((visibility ("default")))

/************************************ Common APIs *************************************/


/**
 * @brief This method is used to query the NVLink Domain Information
 *
 * @param pFmHandle [IN]: Handle that came from \ref fmConnect
 *
 * @param pDomainInfo [OUT]
 *         On Success, the NVLink Domain Information will be populated.
 *
 * @return
 *         - \ref FM_ST_SUCCESS - successfully populated the NVLink Domain Information
 *         - \ref FM_ST_UNINITIALIZED - FM interface library has not been initialized
 *         - \ref FM_ST_BADPARAM - if pFmHandle is not a valid handle
 *         - \ref FM_ST_GENERIC_ERROR - an unspecified internal error occurred
 *         - \ref FM_ST_VERSION_MISMATCH - if the expected and provided versions do not match
 *         - \ref FM_ST_NOT_SUPPORTED - if requested operation/feature not supported/enabled
 */
fmReturn_t FM_LIB_API fmNvlGetDomainInfo(fmHandle_t pFmHandle, fmNvlGetDomainInfo_t *pDomainInfo);

/************************************ Compute Node APIs ************************************/

/**
 * @brief This method is used to query the number of Compute nodes
 * 
 * @param pFmHandle [IN] Handle that came from \ref fmConnect
 *
 * @param pComputeNodeCount  [IN|OUT]
 *         Number of Compute nodes that are matching with a given attribute and node
 *         health (optional). On success, the Compute node count will be populated.
 *
 * @return
 *         - \ref FM_ST_SUCCESS - successfully populated the Compute node count information
 *         - \ref FM_ST_UNINITIALIZED - FM interface library has not been initialized
 *         - \ref FM_ST_BADPARAM - if pFmHandle is not a valid handle
 *         - \ref FM_ST_GENERIC_ERROR - an unspecified internal error occurred
 *         - \ref FM_ST_VERSION_MISMATCH - if the expected and provided versions do not match
 *         - \ref FM_ST_NOT_SUPPORTED - if requested operation/feature not supported/enabled
 */
fmReturn_t FM_LIB_API fmNvlGetComputeNodeCount(fmHandle_t pFmHandle, fmNvlGetComputeNodeCount_t *pComputeNodeCount);

/**
 * @brief This method is used to query the list of Compute node IDs
 *
 * @param pFmHandle [IN] Handle that came from \ref fmConnect
 *
 * @param pNodeIdList [IN|OUT]
 *         Number of Compute node IDs that are matching with a given attribute and node
 *         health (optional). On success, the Compute node IDs will be populated.
 *
 * @return
 *         - \ref FM_ST_SUCCESS - successfully populated list of Compute node IDs
 *         - \ref FM_ST_UNINITIALIZED - FM interface library has not been initialized
 *         - \ref FM_ST_BADPARAM - if pFmHandle is not a valid handle
 *         - \ref FM_ST_GENERIC_ERROR - an unspecified internal error occurred
 *         - \ref FM_ST_VERSION_MISMATCH - if the expected and provided versions do not match
 *         - \ref FM_ST_NOT_SUPPORTED - if requested operation/feature not supported/enabled
 */
fmReturn_t FM_LIB_API fmNvlGetComputeNodeIdList(fmHandle_t pFmHandle, fmNvlGetComputeNodeIdList_t *pNodeIdList);

/**
 * @brief This method is used to query the list of Compute node information
 *
 * @param pFmHandle [IN] Handle that came from \ref fmConnect
 *
 * @param pNodeInfoList [IN|OUT]
 *         On success, list of Compute node information will be populated for a given list of node IDs.
 *
 * @return
 *         - \ref FM_ST_SUCCESS - successfully populated list of Compute node information
 *         - \ref FM_ST_UNINITIALIZED - FM interface library has not been initialized
 *         - \ref FM_ST_BADPARAM - if pFmHandle is not a valid handle
 *         - \ref FM_ST_GENERIC_ERROR - an unspecified internal error occurred
 *         - \ref FM_ST_VERSION_MISMATCH - if the expected and provided versions do not match
 *         - \ref FM_ST_NOT_SUPPORTED - if requested operation/feature not supported/enabled
 */
fmReturn_t FM_LIB_API fmNvlGetComputeNodeInfoList(fmHandle_t pFmHandle, fmNvlGetComputeNodeInfoList_t *pNodeInfoList);

/**
 * @brief This method is used to query the list of GPU information
 *
 * @param pFmHandle [IN] Handle that came from \ref fmConnect
 *
 * @param pSwitchInfo  [IN|OUT]
 *         On success, a list of GPU info for a given Compute node or Partition will be populated.
 *
 * @return
 *         - \ref FM_ST_SUCCESS - successfully populated list of GPU information
 *         - \ref FM_ST_UNINITIALIZED - FM interface library has not been initialized
 *         - \ref FM_ST_BADPARAM - if pFmHandle is not a valid handle
 *         - \ref FM_ST_GENERIC_ERROR - an unspecified internal error occurred
 *         - \ref FM_ST_VERSION_MISMATCH - if the expected and provided versions do not match
 *         - \ref FM_ST_NOT_SUPPORTED - if requested operation/feature not supported/enabled
 */
fmReturn_t FM_LIB_API fmNvlGetGpuInfoList(fmHandle_t pFmHandle, fmNvlGetGpuInfoList_t *pGpuInfoList);

/************************************ Switch Node APIs ************************************/

/**
 * @brief This method is used to query the number of Switch nodes
 *
 * @param pFmHandle [IN] Handle that came from \ref fmConnect
 *
 * @param pSwitchNodeCount [IN|OUT]
 *         Number of Switch nodes that are matching with a given attribute and node
 *         health (optional). On success, the Switch node count will be populated.
 *
 * @return
 *         - \ref FM_ST_SUCCESS - successfully populated the Switch node count information
 *         - \ref FM_ST_UNINITIALIZED - FM interface library has not been initialized
 *         - \ref FM_ST_BADPARAM - if pFmHandle is not a valid handle
 *         - \ref FM_ST_GENERIC_ERROR - an unspecified internal error occurred
 *         - \ref FM_ST_VERSION_MISMATCH - if the expected and provided versions do not match
 *         - \ref FM_ST_NOT_SUPPORTED - if requested operation/feature not supported/enabled
 */
fmReturn_t FM_LIB_API fmNvlGetSwitchNodeCount(fmHandle_t pFmHandle, fmNvlGetSwitchNodeCount_t *pSwitchNodeCount);

/**
 * @brief This method is used to query the list of Switch node IDs
 *
 * @param pFmHandle [IN] Handle that came from \ref fmConnect
 *
 * @param pNodeIdList [IN|OUT]
 *         Number of Switch node IDs that are matching with a given attribute and node
 *         health (optional). On success, the Switch node IDs will be populated.
 *
 * @return
 *         - \ref FM_ST_SUCCESS - successfully populated list of Switch node IDs
 *         - \ref FM_ST_UNINITIALIZED - FM interface library has not been initialized
 *         - \ref FM_ST_BADPARAM - if pFmHandle is not a valid handle
 *         - \ref FM_ST_GENERIC_ERROR - an unspecified internal error occurred
 *         - \ref FM_ST_VERSION_MISMATCH - if the expected and provided versions do not match
 *         - \ref FM_ST_NOT_SUPPORTED - if requested operation/feature not supported/enabled
 */
fmReturn_t FM_LIB_API fmNvlGetSwitchNodeIdList(fmHandle_t pFmHandle, fmNvlGetSwitchNodeIdList_t *pNodeIdList);

/**
 * @brief This method is used to query the list of Switch node information
 *
 * @param pFmHandle [IN] Handle that came from \ref fmConnect
 *
 * @param pNodeInfoList [IN|OUT]
 *         On success, list of Switch node information will be populated for a given list of node IDs.
 *
 * @return
 *         - \ref FM_ST_SUCCESS - successfully populated list of Switch node information
 *         - \ref FM_ST_UNINITIALIZED - FM interface library has not been initialized
 *         - \ref FM_ST_BADPARAM - if pFmHandle is not a valid handle
 *         - \ref FM_ST_GENERIC_ERROR - an unspecified internal error occurred
 *         - \ref FM_ST_VERSION_MISMATCH - if the expected and provided versions do not match
 *         - \ref FM_ST_NOT_SUPPORTED - if requested operation/feature not supported/enabled
 */
fmReturn_t FM_LIB_API fmNvlGetSwitchNodeInfoList(fmHandle_t pFmHandle, fmNvlGetSwitchNodeInfoList_t *pNodeInfoList);

/**
 * @brief This method is used to query the list of Switch information
 *
 * @param pFmHandle [IN] Handle that came from \ref fmConnect
 *
 * @param pSwitchInfo  [IN|OUT]
 *         On success, a list of Switch info for a given Switch Node will be populated.
 *
 * @return
 *         - \ref FM_ST_SUCCESS - successfully populated list of switch information
 *         - \ref FM_ST_UNINITIALIZED - FM interface library has not been initialized
 *         - \ref FM_ST_BADPARAM - if pFmHandle is not a valid handle
 *         - \ref FM_ST_GENERIC_ERROR - an unspecified internal error occurred
 *         - \ref FM_ST_VERSION_MISMATCH - if the expected and provided versions do not match
 *         - \ref FM_ST_NOT_SUPPORTED - if requested operation/feature not supported/enabled
 */
fmReturn_t FM_LIB_API fmNvlGetSwitchInfoList(fmHandle_t pFmHandle, fmNvlGetSwitchInfoList_t *pSwitchInfoList);

/**
 * @brief This method is used to query the list of Cage information
 *
 * @param pFmHandle [IN] Handle that came from \ref fmConnect
 *
 * @param pSwitchInfo  [IN|OUT]
 *         On success, a list of Cage info for a given Switch Node will be populated.
 *
 * @return
 *         - \ref FM_ST_SUCCESS - successfully populated list of Cage information
 *         - \ref FM_ST_UNINITIALIZED - FM interface library has not been initialized
 *         - \ref FM_ST_BADPARAM - if pFmHandle is not a valid handle
 *         - \ref FM_ST_GENERIC_ERROR - an unspecified internal error occurred
 *         - \ref FM_ST_VERSION_MISMATCH - if the expected and provided versions do not match
 *         - \ref FM_ST_NOT_SUPPORTED - if requested operation/feature not supported/enabled
 */
fmReturn_t FM_LIB_API fmNvlGetCageInfoList(fmHandle_t pFmHandle, fmNvlGetCageInfoList_t *pCageInfoList);

/********************************** Connection APIs ***********************************/

/**
 * @brief This method is used to query NVLink connection count
 *
 * @param pFmHandle [IN] Handle that came from \ref fmConnect
 *
 * @param pConnCount [OUT]
 *         Number of expected or discovered connections that are matching with a given attribute.
 *         On success, returns the number of connections in numConns.
 *
 * @return
 *         - \ref FM_ST_SUCCESS - successfully populated connection count information
 *         - \ref FM_ST_UNINITIALIZED - FM interface library has not been initialized
 *         - \ref FM_ST_BADPARAM - if pFmHandle is not a valid handle
 *         - \ref FM_ST_GENERIC_ERROR - an unspecified internal error occurred
 *         - \ref FM_ST_VERSION_MISMATCH - if the expected and provided versions do not match
 *         - \ref FM_ST_NOT_SUPPORTED - if requested operation/feature not supported/enabled
 */
fmReturn_t FM_LIB_API fmNvlGetConnCount(fmHandle_t pFmHandle, fmNvlGetConnCount_t  *pConnCount);

/**
 * @brief This method is used to query list of NVLink connection Information
 *
 * @param pFmHandle [IN] Handle that came from \ref fmConnect
 *
 * @param pConnInfo [IN|OUT]
 *         Number of expected or discovered connection information that are matching with a
 *         given attribute. On success, a list of NVLink connection information will be populated.
 *
 * @return
 *         - \ref FM_ST_SUCCESS - successfully populated list of connection information
 *         - \ref FM_ST_UNINITIALIZED - FM interface library has not been initialized
 *         - \ref FM_ST_BADPARAM - if pFmHandle is not a valid handle
 *         - \ref FM_ST_GENERIC_ERROR - an unspecified internal error occurred
 *         - \ref FM_ST_VERSION_MISMATCH - if the expected and provided versions do not match
 *         - \ref FM_ST_NOT_SUPPORTED - if requested operation/feature not supported/enabled
 */
fmReturn_t FM_LIB_API fmNvlGetConnInfoList(fmHandle_t pFmHandle, fmNvlGetConnInfoList_t  *pConnInfoList);

/********************************* Maintenance APIs ***********************************/

/*
 * This method is used to set maintenance state for one or more switch ports.
 *      - It disables routing if the corresponding switch connections are part of an active partition.
 *      - It sets the maintenance state to MAINTENANCE_ON
 *
 * @param pfmHandle IN: Handle that came form \ref fmConnect
 *
 * @param pConnList IN:
 *         On success, a list of ports and their cages will be set to MAINTENANCE_ON 
 *         state after disabling routing for corresponding switch connections if those 
 *         connections are part of active partition.
 *
 * @return
 *         - \ref FM_ST_SUCCESS - successfully set the maintenance state for requested ports.
 *         - \ref FM_ST_UNINITIALIZED - FM interface library has not been initialized
 *         - \ref FM_ST_BADPARAM - if pFmHandle is not a valid handle
 *         - \ref FM_ST_GENERIC_ERROR - an unspecified internal error occurred
 *         - \ref FM_ST_VERSION_MISMATCH - if the expected and provided versions do not match
 *         - \ref FM_ST_NOT_SUPPORTED - if requested operation/feature not supported/enabled
 */
fmReturn_t FM_LIB_API fmNvlSetMaintenanceOnPortList(fmHandle_t pFmHandle, fmNvlPortList_t *pPortList);

/**
 * This method is used to clear maintenance state for one or more switch ports.
 *      - It enable routing if the corresponding switch connections are part of an active partition.
 *      - It sets the maintenance state to MAINTENANCE_OFF
 *
 * @param pfmHandle IN: Handle that came form \ref fmConnect
 *
 * @param pConnList IN:
 *         On success, a list of ports and their cages will be set to MAINTENANCE_OFF 
 *         state after enabling routing for corresponding switch connections if those 
 *         connections are part of active partition.
 *
 * @return
 *         - \ref FM_ST_SUCCESS - successfully clear the maintenance state for requested ports.
 *         - \ref FM_ST_UNINITIALIZED - FM interface library has not been initialized
 *         - \ref FM_ST_BADPARAM - if pFmHandle is not a valid handle
 *         - \ref FM_ST_GENERIC_ERROR - an unspecified internal error occurred
 *         - \ref FM_ST_VERSION_MISMATCH - if the expected and provided versions do not match
 *         - \ref FM_ST_NOT_SUPPORTED - if requested operation/feature not supported/enabled
 */
fmReturn_t FM_LIB_API fmNvlClearMaintenanceOnPortList(fmHandle_t pFmHandle, fmNvlPortList_t *pPortList);

/**
 * This method is used to add Switch nodes to the NVLink Domain.
 *
 * @param pfmHandle IN: Handle that came form \ref fmConnect
 *
 * @param pConnList IN: Add list of Switch Nodes to the NVLink Domain
 *
 * @return
 *         - \ref FM_ST_SUCCESS - successfully added Switch nodes to the NVLink Domain
 *         - \ref FM_ST_UNINITIALIZED - FM interface library has not been initialized
 *         - \ref FM_ST_BADPARAM - if pFmHandle is not a valid handle
 *         - \ref FM_ST_GENERIC_ERROR - an unspecified internal error occurred
 *         - \ref FM_ST_VERSION_MISMATCH - if the expected and provided versions do not match
 *         - \ref FM_ST_NOT_SUPPORTED - if requested operation/feature not supported/enabled
 */
fmReturn_t FM_LIB_API fmNvlAddSwitchNodeList(fmHandle_t pFmHandle, fmNvlSwitchNodeList_t *pNodeList);

/**
 * This method is used to remove Switch nodes from the NVLink Domain.
 *
 * @param pfmHandle IN: Handle that came form \ref fmConnect
 *
 * @param pConnList IN: Remove list of Switch Nodes from the NVLink Domain
 *
 * @return
 *         - \ref FM_ST_SUCCESS - successfully removed Switch nodes from the NVLink Domain
 *         - \ref FM_ST_UNINITIALIZED - FM interface library has not been initialized
 *         - \ref FM_ST_BADPARAM - if pFmHandle is not a valid handle
 *         - \ref FM_ST_GENERIC_ERROR - an unspecified internal error occurred
 *         - \ref FM_ST_VERSION_MISMATCH - if the expected and provided versions do not match
 *         - \ref FM_ST_NOT_SUPPORTED - if requested operation/feature not supported/enabled
 */
fmReturn_t FM_LIB_API fmNvlRemoveSwitchNodeList(fmHandle_t pFmHandle, fmNvlSwitchNodeList_t *pNodeList);

#ifdef __cplusplus
}
#endif

#endif /* NV_FM_NVL_AGENT_H */
