/* isr.c: 
 *	Does device servicing work while irq was cleared

 * implemented as a thread polling the low level driver node representing device
 * it's ll driver's responsibility to disable the device interrupts 
 * before program serve it.

 * Author(s):
 *	A. Chentsov <chentsov@ukr.net>

 * evolution:
 *	11/23/05	first release to work
 *	11/09/05 	creation
 */

#include <stdio.h>
#include <pthread.h>

#include <unistd.h>
#include <sys/io.h>

#include <errno.h> 

#include <signal.h>

#include <asm/types.h>
#include "../martian.h"
#include "sysdep.h"

#define PREFIX	"isr"
#include "log.h"

int isr_tid;

struct _stats {
	int cring, cstream, cbit3;
	int dual_port_tx, dual_port_rx, io_dual_port_rx;
	int tx, rx, pdm_rx;
	int dcp;
} stats;

static __inline__ void atomic_dec(atomic_t *v)
{
	__asm__ __volatile__(
		LOCK "decl %0"
		:"=m" (v->counter)
		:"m" (v->counter));
}

static __inline__ void atomic_set (atomic_t *v, int i)  {      
	 v->counter = i;
}

extern struct martian_common *Mcb;


extern int isr_defered(void );
extern void pin_thread (void);

void *serve_interrupts (void *arg) {
	extern unsigned char *rd7_isr;
	int devfd = (int ) arg;


	isr_tid = mgettid();
	LOGDEBUG (2, "thread started (tid = %d)\n", isr_tid);

	{
		struct sched_param p;
		p.sched_priority = sched_get_priority_max (SCHED_FIFO);
		sched_setscheduler (0, SCHED_FIFO, &p);

		sched_getparam (0, &p);
		LOGDEBUG (2, "scheduler: policy %d, priority %d\n", 
			sched_getscheduler (0),
			p.sched_priority);
	}

	stats.cring = stats.cstream = stats.cbit3 = 0;
	stats.dual_port_tx = stats.dual_port_rx = 0;
	stats.dcp = 0;

	/* smp, pin isr thread to cpu */
	pin_thread();

	// polling loop
	while (1) { 
		int irqs[2], size;

		/*** number of interrupts to be returned */
		size = read (devfd, irqs, sizeof irqs);
		if (size == -1) {
			/* read might be interrupted by timer */
			if (errno == EINTR)
				continue;

			LOGSYSERR (Error, "dev read");
			LOGERR ("quitting");
			return NULL;
		}
		else if (size != sizeof (int) && size != sizeof irqs) {
			/* report wierd thing happened :*/
			LOGWARN ("unexpected dev read size %d\n", size);
			continue;
		}	

		ASSERT((*rd7_isr & ~6) == 0, *rd7_isr &= 6);
		if (*rd7_isr & 2)
			stats.cring++;
		if (*rd7_isr & 4)
			stats.cstream++;
		if (*rd7_isr & 8)
			stats.cbit3++;

		if (irqs[0] & 0x1000) {
			LOGDEBUG (Note, "read: %db of %x %x\n", size, irqs[0], irqs[1]);
			LOGERR ("fifo fault in kmodule is detected\n");
			atomic_dec (&Mcb->UnservedIrqs);
			atomic_set (&Mcb->UnservedIrqs, 0);

			/* exit graciously letting debugger intrude */
			kill (getpid(), SIGINT);
		}

		isr_defered();

		logdebugadd (5, "i");
	}

	return NULL;
}

void dump_isr (void) {
	LOGQ (Debug, "interrupts: ring=%d, stream=%d (dcp=%d, trx=%d), bit3=%d\n",
		stats.cring, stats.cstream, stats.dcp, stats.tx+stats.rx, stats.cbit3);
	LOGQ (Debug, "calls: tx: %d, rx: %d, pdm; bytes: tx: %d, rx: %d, pdm: %d\n",
		stats.dual_port_tx, stats.dual_port_rx, stats.io_dual_port_rx,
		stats.tx, stats.rx, stats.pdm_rx);

	LOGQ (Debug, "mcb: calls: tx: %d, rx: %d, pdm: %d;"
		     "bytes: tx: %d, rx: %d, pdm: %d\n",
		Mcb->dual_port_tx, Mcb->dual_port_rx, Mcb->io_dual_port_rx,
		Mcb->tx, Mcb->rx, Mcb->pdm_rx);
	LOGQ (Debug, "overrun: dce = %d, pdm = %d; "
		"incorrespondence: tx = %d, rx = %d, pdm = %d\n", 
		Mcb->rx_overrun, Mcb->pdm_rx_overrun, Mcb->tx_incorrespondence, 
		Mcb->rx_incorrespondence, Mcb->pdm_rx_incorrespondence);
}

