/****************************************************************************
 *    lib/b/Parser.y - This file is part of coala							*
 *																			*
 *    Copyright (C) 2009-2010  Torsten Grote								*
 *																			*
 *    This program is free software; you can redistribute it and/or modify	*
 *    it under the terms of the GNU General Public License as published by	*
 *    the Free Software Foundation; either version 3 of the License, or		*
 *    (at your option) any later version.									*
 *																			*
 *    This program is distributed in the hope that it will be useful,		*
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of		*
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the			*
 *    GNU General Public License for more details.							*
 *																			*
 *    You should have received a copy of the GNU General Public License		*
 *    along with this program; if not, see http://www.gnu.org/licenses		*
 ****************************************************************************/
%name BParser

%define LSP_NEEDED
%define ERROR_BODY = 0
%define LEX_BODY = 0
%define MEMBERS void preProcessing(Coala::CompilerOptions*); void postProcessing(Printer*); virtual ~BParser(){}\
	private: std::string to_string(int integer);\
	public: B::Program* program;

%header{
#include <stdio.h>
#include <iostream>
#include <string>
#include <vector>
#include <algorithm>

#include "../options.h"
#include "Program.h"

using namespace std;
using namespace C;
%}

%union {
	int number;
	string* str;
	Formula* form;
	Type* typ;
	Types* types;
	Arguments* args;
	Constraint* constr;
	Pool* var_pool;
	Interval* interval;
	LTLNode* ltl_node;
}

%token <str> IDENTIFIER
%token <str> NEG_IDENTI
%token <str> VARIABLE
%token <number> NUMBER
%token <number> NOT		// not
%token <number> ACT		// <action>
%token <number> FLU		// <fluent>
%token <number> DFLU	// <defined fluent>
%token <number> CAUS	// <caused>
%token <number> CAUSES	// <causes>
%token <number> IF		// <if>
%token <number> EXE		// <executable>
%token <number> NONEXE	// <impossible>
%token <number> INIT	// <initially>
%token <number> OC_AT	// <occurs at>
%token <number> HO_AT	// <holds at>
%token <number> GOAL	// <goal>
%token <number> WHERE	// <where>
%token <number> TRUE	// <true>
%token <number> FALSE	// <false>
%token <number> DEF		// <default>
%token <number> INERT	// <inertial>
%token <number> COMMA	// ,
%token <number> SEMIC	// ;
%token <number> COLON	// :
%token <number> DOT		// .
%token <number> DDOT	// ..
%token <number> LBRAC	// (
%token <number> RBRAC	// )
%token <number> EQ		// ==
%token <number> NEQ		// !=
%token <number> LT		// <
%token <number> GT		// >
%token <number> LE		// <=
%token <number> GE		// >=
%token <number> PLUS	// +
%token <number> MINUS	// -
%token <number> TIMES	// *
%token <number> DIV		// /
%token <number> MOD		// mod
%token <number> AND		// &
%token <number> OR		// ?
%token <number> XOR		// ^
%token <number> LTL		// LTL:
%token <number> LTLNOT	// !
%token <number> LTLOR	// |
%token <number> IMPL	// ->
%token <number> EQUIV	// <->
%token <number> X		// X
%token <number> F		// F
%token <number> G		// G
%token <number> U		// U
%token <number> R		// R

// LTL Operator Precedence
%left EQUIV IMPL LTLOR AND
%left U R
%nonassoc G F X
%nonassoc LTLNOT

%type <form>		formula
%type <types>		types_list
%type <typ>			type
%type <args>		term_list
%type <form>		identifier
%type <constr>		constraint
%type <var_pool>	pool
%type <interval>	interval
%type <constr>		operation
%type <str>			operator
%type <str>			eqoperator
%type <ltl_node>	ltl_expr

%start program

%%
/***************************************************************************************************************************/

program: /* empty */ | program rule DOT;

rule: fact | law | query;

fact: act_fact | flu_fact | dflu_fact;

law: static_rule | static_caused_rule | dyn_rule | exe_rule | nonexe_rule;

query: init_rule | oc_query | ho_query | goal_query | ltl_query;

// <action> a_1,...,a_n <where> bla.
act_fact:
  ACT formula					{ program->addActionDef($2); }
| ACT formula WHERE types_list	{ program->addActionDef($2, $4); }

// <fluent> f_1,...,f_n <where> bla.
flu_fact:
  FLU formula					{ program->addFluentDef($2); }
| FLU formula WHERE types_list	{ program->addFluentDef($2, false, $4); }

// <defined fluent> f_1,...,f_n <where> bla.
dflu_fact:
  DFLU formula					{ program->addFluentDef($2, true); }
| DFLU formula WHERE types_list	{ program->addFluentDef($2, true, $4); }

// f_1,...,f_n <if> g_1,...,g_m <where> bla.
static_rule:
  formula								{ program->addStaticRule($1, NULL); }
| formula WHERE types_list				{ program->addStaticRule($1, NULL, $3); }
| formula IF formula					{ program->addStaticRule($1, $3); }
| formula IF formula WHERE types_list	{ program->addStaticRule($1, $3, $5); }

// <caused> f_1,...,f_n <if> g_1,...,g_m <where> bla.
static_caused_rule:
  CAUS formula								{ program->addStaticRule($2, NULL); }
| CAUS formula WHERE types_list				{ program->addStaticRule($2, NULL, $4); }
| CAUS formula IF formula					{ program->addStaticRule($2, $4); }
| CAUS formula IF formula WHERE types_list	{ program->addStaticRule($2, $4, $6); }

// a <causes> f_1,...,f_n <if> g_1,...,g_m <where> bla.
dyn_rule:
  formula CAUSES formula								{ program->addDynamicRule($1, $3); }
| formula CAUSES formula WHERE types_list				{ program->addDynamicRule($1, $3, NULL, $5); }
| formula CAUSES formula IF formula						{ program->addDynamicRule($1, $3, $5); }
| formula CAUSES formula IF formula WHERE types_list	{ program->addDynamicRule($1, $3, $5, $7); }

// <executable> a_1,...,a_n <if> g_1,...,g_m <where> bla.
exe_rule:
  EXE formula								{ program->addExecutableRule($2, NULL); }
| EXE formula WHERE types_list				{ program->addExecutableRule($2, NULL, $4); }
| EXE formula IF formula					{ program->addExecutableRule($2, $4); }
| EXE formula IF formula WHERE types_list	{ program->addExecutableRule($2, $4, $6); }

// <impossible> a_1,..., a_n <if> f_1,..., f_m <where> bla.	=	<caused> <false> <after> a_1,...,a_n,f_1,...,f_m <where> bla.
nonexe_rule:
  NONEXE formula								{ program->addNonexecutableRule($2, NULL); }
| NONEXE formula WHERE types_list				{ program->addNonexecutableRule($2, NULL, $4); }
| NONEXE formula IF formula						{ program->addNonexecutableRule($2, $4); }
| NONEXE formula IF formula WHERE types_list	{ program->addNonexecutableRule($2, $4, $6); }

// <initially> f_1,..., f_n <where> bla.	=	f_1,...,f_n <holds at> 0 <where> bla.
init_rule:
  INIT formula								{ program->addInitiallyRule($2); }
| INIT formula WHERE types_list				{ program->addInitiallyRule($2, $4); }

// a_1,...,a_n <occurs at> t_i <where> bla.
oc_query:
  formula OC_AT NUMBER						{ program->addQQuery("occ", $1, to_string($3)); }
| formula OC_AT IDENTIFIER					{ program->addQQuery("occ", $1, *$3); }
| formula OC_AT NUMBER WHERE types_list		{ program->addQQuery("occ", $1, to_string($3), $5); }
| formula OC_AT IDENTIFIER WHERE types_list	{ program->addQQuery("occ", $1, *$3, $5); }

// f_1,...,f_n <holds at> t_i <where> bla.
ho_query:
  formula HO_AT NUMBER						{ program->addQQuery("holds", $1, to_string($3)); }
| formula HO_AT IDENTIFIER					{ program->addQQuery("holds", $1, *$3); }
| formula HO_AT NUMBER WHERE types_list		{ program->addQQuery("holds", $1, to_string($3), $5); }
| formula HO_AT IDENTIFIER WHERE types_list	{ program->addQQuery("holds", $1, *$3, $5); }

// <goal> f_1,...,f_n <where> bla.
goal_query:
  GOAL formula					{ program->addQQuery("holds", $2, "t"); }
| GOAL formula WHERE types_list	{ program->addQQuery("holds", $2, "t", $4); }

// f_1,...,f_n
formula:
  identifier					{ $$ = program->addFormula('c', $1); }
| TRUE							{ $$ = program->addFormula('c', new FluentAction(new string("0"))); }
| FALSE							{ $$ = program->addFormula('c', new Formula('n', new FluentAction(new string("0")))); }
| formula COMMA identifier		{ $1->addPart($3); $$ = $1; }

identifier:
  IDENTIFIER						{ $$ = new FluentAction($1); }
| IDENTIFIER LBRAC term_list RBRAC	{ $$ = new FluentAction($1, $3); }
| NEG_IDENTI						{ $$ = new Formula('n', new FluentAction($1)); }
| NEG_IDENTI LBRAC term_list RBRAC	{ $$ = new Formula('n', new FluentAction($1, $3)); }

// X, a, 3, X;Y, 1..Z
term_list:
  VARIABLE											{ $$ = new Arguments(program->addVariable($1)); }
| IDENTIFIER										{ $$ = new Arguments(new Constant($1)); }
| NUMBER											{ $$ = new Arguments(new Constant($1)); }
| pool												{ $$ = new Arguments($1); }
| interval											{ $$ = new Arguments($1); }
| IDENTIFIER LBRAC term_list RBRAC					{ $$ = new Arguments(new Function($1, $3->getArgs())); }
| term_list COMMA VARIABLE							{ $1->addVariable(program->addVariable($3)); $$ = $1; }
| term_list COMMA IDENTIFIER						{ $1->addConstant(new Constant($3)); $$ = $1; }
| term_list COMMA NUMBER							{ $1->addConstant(new Constant($3)); $$ = $1; }
| term_list COMMA pool								{ $1->addPool($3); $$ = $1; }
| term_list COMMA interval							{ $1->addInterval($3); $$ = $1; }
| term_list COMMA IDENTIFIER LBRAC term_list RBRAC	{ $1->addFunction(new Function($3, $5->getArgs())); $$ = $1; }

// X;Y
pool:
  VARIABLE SEMIC VARIABLE	{ $$ = new Pool(program->addVariable($1), program->addVariable($3)); }
| pool SEMIC VARIABLE		{ $1->addVariable(program->addVariable($3)); $$ = $1; }

// 1..X
interval:
  NUMBER DDOT NUMBER		{ $$ = new Interval(new Constant($1), new Constant($3)); }
| NUMBER DDOT VARIABLE		{ $$ = new Interval(new Constant($1), program->addVariable($3)); }
| VARIABLE DDOT NUMBER		{ $$ = new Interval(program->addVariable($1), new Constant($3)); }
| VARIABLE DDOT VARIABLE	{ $$ = new Interval(program->addVariable($1), program->addVariable($3)); }

// type_1(X),...,type_n(Y), X!=Y
types_list:
  constraint						{ $$ = new Types($1); }
| type								{ $$ = new Types($1); }
| NUMBER COLON type					{ $3->setMin($1); $$ = new Types($3); }
| type COLON NUMBER					{ $1->setMax($3); $$ = new Types($1); }
| NUMBER COLON type COLON NUMBER	{ $3->setMin($1); $3->setMax($5); $$ = new Types($3); }
| types_list COMMA constraint						{ $1->addType($3); }
| types_list COMMA type								{ $1->addType($3); }
| types_list COMMA NUMBER COLON type				{ $5->setMin($3); $1->addType($5); }
| types_list COMMA type COLON NUMBER				{ $3->setMax($5); $1->addType($3); }
| types_list COMMA NUMBER COLON type COLON NUMBER	{ $5->setMin($3); $5->setMax($7); $1->addType($5); }

// type_xyz(X,Y,Z)
type:
  IDENTIFIER LBRAC term_list RBRAC		{ $$ = new Type($1, $3, true); }
| NOT IDENTIFIER LBRAC term_list RBRAC	{ $$ = new Type($2, $4, false); }

// X!=Y etc.
constraint:
  operation eqoperator operation	{ $1->merge($3, $2); $$ = $1; }
| NUMBER eqoperator operation		{ $3->mergeLeft(new string(to_string($1)+*$2), $2); $$ = $3; }
| operation eqoperator NUMBER		{ $1->merge(new string(*$2+to_string($3)), $2); $$ = $1; }
| IDENTIFIER eqoperator operation	{ $3->mergeLeft(new string(*$1+*$2), $2); $$ = $3; }
| operation eqoperator IDENTIFIER	{ $1->merge(new string(*$2+*$3), $2); $$ = $1; }
| VARIABLE eqoperator operation		{ $3->mergeLeft(new string(*$1+*$2), program->addVariable($1), $2); $$ = $3; }
| operation eqoperator VARIABLE		{ $1->merge(new string(*$2+*$3), program->addVariable($3), $2); $$ = $1; }
| VARIABLE eqoperator VARIABLE		{ $$ = new Constraint(program->addVariable($1), program->addVariable($3), new string(*$1+*$2+*$3), $2); }
| NUMBER eqoperator VARIABLE		{ $$ = new Constraint(program->addVariable($3), new string(to_string($1)+*$2+*$3), $2); }
| VARIABLE eqoperator NUMBER		{ $$ = new Constraint(program->addVariable($1), new string(*$1+*$2+to_string($3)), $2); }
| IDENTIFIER eqoperator VARIABLE	{ $$ = new Constraint(program->addVariable($3), new string(*$1+*$2+*$3), $2); }
| VARIABLE eqoperator IDENTIFIER	{ $$ = new Constraint(program->addVariable($1), new string(*$1+*$2+*$3), $2); }
| NUMBER eqoperator IDENTIFIER		{ $$ = new Constraint(new string(to_string($1)+*$2+*$3), $2); }
| IDENTIFIER eqoperator NUMBER		{ $$ = new Constraint(new string(*$1+*$2+to_string($3)), $2); }

// X+1+Y
operation:
  NUMBER operator VARIABLE		{ $$ = new Constraint(program->addVariable($3), new string(to_string($1)+*$2+*$3)); }
| VARIABLE operator NUMBER		{ $$ = new Constraint(program->addVariable($1), new string(*$1+*$2+to_string($3))); }
| VARIABLE operator VARIABLE	{ $$ = new Constraint(program->addVariable($1), program->addVariable($3), new string(*$1+*$2+*$3)); }
| operation operator NUMBER		{ $1->merge(new string(*$2+to_string($3))); $$ = $1; }
| operation operator VARIABLE	{ $1->merge(new string(*$2+*$3), program->addVariable($3)); $$ = $1; }

eqoperator:
  EQ 	{ $$ = new string("=="); }
| NEQ	{ $$ = new string("!="); }
| LT	{ $$ = new string("<"); }
| GT	{ $$ = new string(">"); }
| LE	{ $$ = new string("<="); }
| GE	{ $$ = new string(">="); }

operator:
  PLUS	{ $$ = new string("+"); }
| MINUS	{ $$ = new string("-"); }
| TIMES	{ $$ = new string("*"); }
| DIV	{ $$ = new string("/"); }
| MOD	{ $$ = new string(" mod "); }
| AND	{ $$ = new string("&"); }
| OR	{ $$ = new string("?"); }
| XOR	{ $$ = new string("^"); }

ltl_query: LTL ltl_expr { program->addLTLQuery($2); }

ltl_expr:
  identifier				{ $$ = program->getLTLNode(IDENTIFIER, NULL, NULL, $1); }
| LBRAC ltl_expr RBRAC		{ $$ = $2; }
| LTLNOT ltl_expr			{ $$ = program->getLTLNode(LTLNOT, NULL, $2); }
| ltl_expr AND ltl_expr		{ $$ = program->getLTLNode(AND, $1, $3); }
| ltl_expr LTLOR ltl_expr	{ $$ = program->getLTLNode(LTLOR, $1, $3); }
| ltl_expr IMPL ltl_expr	{ $$ = program->getLTLNode(IMPL, $1, $3); }
| ltl_expr EQUIV ltl_expr	{ $$ = program->getLTLNode(EQUIV, $1, $3); }
| X ltl_expr				{ $$ = program->getLTLNode(X, NULL, $2); }
| G ltl_expr				{ $$ = program->getLTLNode(G, NULL, $2); }
| F ltl_expr				{ $$ = program->getLTLNode(F, NULL, $2); }
| ltl_expr U ltl_expr		{ $$ = program->getLTLNode(U, $1, $3); }
| ltl_expr R ltl_expr		{ $$ = program->getLTLNode(R, $1, $3); }


%%

/***************************************************************************************************************************/

void BParser::preProcessing(Coala::CompilerOptions* options) {
	program = new B::Program(options);
	program->setLine(&yylloc.first_line);
}

void BParser::postProcessing(Printer* p) {
	program->print(p);
//	delete program;
}

std::string BParser::to_string(int integer) {
	std::stringstream ss;
	std::string str;
	ss << integer;
	ss >> str;
	return str;
}
