-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with Unchecked_Conversion;

separate (Dictionary)
package body RawDict is
   --# hide RawDict;

   type OwnTaskInfo is record
      Variable : Variable_Info_Ref;
      Owner    : Package_Info_Ref;
      Next     : Symbol;
   end record;

   type OwnTaskRef is access OwnTaskInfo;

   type Pragmas is record
      Given : Boolean;                   -- true means has been declared
      Value : LexTokenManager.Lex_String; -- null string means no value, otherwise its a Maths StorageRep
   end record;

   type PragmaLists is array (Dictionary.RavenscarPragmas) of Pragmas;

   type ProtectedInfo is record
      Own_Variable      : Own_Variable_Info_Ref; -- abstract "own var" with same name as type
      ElementsHidden    : Boolean;
      Visible_Part      : Part_Info;
      Local_Part        : Part_Info;
      Private_Part      : Part_Info;
      TheEntry          : Symbol;
      FirstDiscriminant : Dictionary.Symbol;
      LastDiscriminant  : Dictionary.Symbol;
      Pragmas           : PragmaLists;
      Protected_Body    : Declaration_Info_Ref;
      Has_Proper_Body   : Boolean;
   end record;

   type ProtectedRef is access ProtectedInfo;

   type TaskInfo is record
      Signature_Is_Wellformed    : Booleans;
      Has_Second_Annotation      : Boolean;
      Has_Derives_Annotation     : Boolean;
      Local_Part                 : Part_Info;
      FirstDiscriminant          : Dictionary.Symbol;
      LastDiscriminant           : Dictionary.Symbol;
      Pragmas                    : PragmaLists;
      First_Global_Variable      : Global_Variables_T;
      Last_Global_Variable       : Global_Variables_T;
      Task_Body                  : Declaration_Info_Ref;
      Has_Proper_Body            : Boolean;
      Body_Is_Hidden             : Boolean;
      Suspends_List              : Symbol;
      First_Loop                 : Dictionary.Symbol;
      Last_Loop                  : Dictionary.Symbol;
      Uses_Unprotected_Variables : Boolean;
      Uses_Unchecked_Conversion  : Boolean;
      Assigns_From_External      : Boolean;
   end record;

   type TaskRef is access TaskInfo;

   type KnownDiscriminantInfo is record
      Name           : LexTokenManager.Lex_String;
      Protected_Type : Type_Info_Ref;  -- could also be a task type
      Type_Mark      : Type_Info_Ref;
      SetsPriority   : Boolean;
      Next           : Dictionary.Symbol;
   end record;

   type KnownDiscriminantRef is access KnownDiscriminantInfo;

   type SubtypeInfo is record -- record used for extra info needed by subtypes of TASK and PROTECTED types
      Priority        : LexTokenManager.Lex_String; -- StorageRep of number
      FirstConstraint : Dictionary.Symbol;         -- list of discriminant constraints
      LastConstraint  : Dictionary.Symbol;
   end record;

   type SubtypeRef is access SubtypeInfo;

   type DiscriminantConstraintInfo is record
      StaticValue    : LexTokenManager.Lex_String; -- only ONE of this and next field will have a value
      AccessedObject : Dictionary.Symbol;
      Next           : Dictionary.Symbol;
   end record;

   type DiscriminantConstraintRef is access DiscriminantConstraintInfo;

   type VirtualElementInfo is record
      The_Variable : RawDict.Variable_Info_Ref;
      TheOwner     : Dictionary.Symbol; -- Variable_Info
      SeenByOwner  : Boolean;
      Next         : Dictionary.Symbol;
   end record;

   type VirtualElementInfoRef is access VirtualElementInfo;

   type SuspendsListItemInfo is record
      ThePOorSO      : Dictionary.Symbol;
      IsAccountedFor : Boolean;
      Next           : Dictionary.Symbol;
   end record;

   type SuspendsListItemInfoRef is access SuspendsListItemInfo;

   type InterruptStreamMapping is record
      TheHandler         : LexTokenManager.Lex_String;
      TheInterruptStream : LexTokenManager.Lex_String;
      Next               : Dictionary.Symbol;
   end record;

   type InterruptStreamMappingRef is access InterruptStreamMapping;

   type LoopInfo is record
      Name          : LexTokenManager.Lex_String;
      Region        : Dictionary.Symbol;
      LoopParameter : Dictionary.Symbol;  -- LoopParameterInfor
      OnEntryVars   : Dictionary.Symbol;  -- LoopEntryVariableInfo;

      -- the following field stores a forced type conversion of a Cells.Cell created
      -- by the VCG and representing the exit expression of a for loop.  Storing a
      -- reference to this expression in the Dictionary makes it easy to plant a loop
      -- invariant asserting that the ExitBound variable = the exit expression.
      ExitExpn : Natural;
      -- And similarly for the entry expression...
      EntryExpn : Natural;

      HasExits : Boolean;
      Next     : Dictionary.Symbol;
   end record;

   type LoopRef is access LoopInfo;

   type LoopParameterInfo is record
      Name           : LexTokenManager.Lex_String;
      Type_Mark      : Type_Info_Ref;
      TheLoop        : Dictionary.Symbol;
      HasStaticRange : Boolean; -- true means bounds fixed, non-empty and for loop must be entered
      IsReverse      : Boolean; -- we assume for loops are forward unless this is set true
   end record;

   type LoopParameterRef is access LoopParameterInfo;

   -- Provides a linked list of variables used in the exit condition of a for loop.  Each
   -- variable is uniquely associated with the loop and the actual Ada variable used in the
   -- exit expression.  These special variables are used by teh VCG to store the value of variables
   -- on entry to the loop so that the loop bounds are properly frozen iaw Ada language definition.
   type LoopEntryVariableInfo is record
      OriginalVar : Dictionary.Symbol;
      TheLoop     : Dictionary.Symbol; -- we need this to construct the name of the var (<var>__entry__<loop>)
      Next        : Dictionary.Symbol;
   end record;

   type LoopEntryVariableRef is access LoopEntryVariableInfo;

   --------------------------------------------------------------------------------

   procedure DiscriminantDebug (CallerID        : in String;
                                Given, Expected : in Dictionary.SymbolDiscriminant) is
   begin
      SPARK_IO.Put_Line
        (SPARK_IO.Standard_Output,
         "*****************************************************************************",
         0);
      SPARK_IO.Put_String (SPARK_IO.Standard_Output, "* From: ", 0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output, CallerID, 0);
      SPARK_IO.Put_String (SPARK_IO.Standard_Output, "* Symbol of type ", 0);
      SPARK_IO.Put_String (SPARK_IO.Standard_Output, Dictionary.SymbolDiscriminant'Image (Given), 0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output, " found where ", 0);
      SPARK_IO.Put_String (SPARK_IO.Standard_Output, "* ", 0);
      SPARK_IO.Put_String (SPARK_IO.Standard_Output, Dictionary.SymbolDiscriminant'Image (Expected), 0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output, " was expected.", 0);
   end DiscriminantDebug;

   --------------------------------------------------------------------------------

   function GetSymbolDiscriminant (Item : in Dictionary.Symbol)
                                  return Dictionary.SymbolDiscriminant
   is
   begin
      return Dictionary.Dynamic_Symbol_Table.Get_Symbol_Discriminant (Dict.Symbols, Item);
   end GetSymbolDiscriminant;

   --------------------------------------------------------------------------------

   function Get_Symbol_Compilation_Unit (Item : in Dictionary.Symbol)
                                        return ContextManager.UnitDescriptors
   is
   begin
      return Dictionary.Dynamic_Symbol_Table.Get_Symbol_Compilation_Unit (Dict.Symbols, Item);
   end Get_Symbol_Compilation_Unit;

   --------------------------------------------------------------------------------

   function Get_Symbol_Location (Item : in Dictionary.Symbol)
                                return LexTokenManager.Token_Position
   is
   begin
      return Dictionary.Dynamic_Symbol_Table.Get_Symbol_Location (Dict.Symbols, Item);
   end Get_Symbol_Location;

   --------------------------------------------------------------------------------

   procedure Set_Symbol_Location (Item     : in Dictionary.Symbol;
                                  Location : in LexTokenManager.Token_Position) is
   begin
      Dictionary.Dynamic_Symbol_Table.Set_Symbol_Location (Dict.Symbols, Item, Location);
   end Set_Symbol_Location;

   --------------------------------------------------------------------------------

   function GetSymbolRef (Item : in Dictionary.Symbol)
                         return Dictionary.Ref_Type
   is
   begin
      return Dictionary.Dynamic_Symbol_Table.Get_Symbol_Ref (Dict.Symbols, Item);
   end GetSymbolRef;

   --------------------------------------------------------------------------------

   function GetProtectedRef (Item : Dictionary.Symbol) return ProtectedRef is
      function RefTypeToProtectedRef is new Unchecked_Conversion (Dictionary.Ref_Type, ProtectedRef);
   begin
      if GetSymbolDiscriminant (Item) /= Dictionary.ProtectedInfoSymbol then
         DiscriminantDebug ("GetProtectedRef", GetSymbolDiscriminant (Item), Dictionary.ProtectedInfoSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToProtectedRef (GetSymbolRef (Item));
   end GetProtectedRef;

   --------------------------------------------------------------------------------

   function GetTaskRef (Item : Dictionary.Symbol) return TaskRef is
      function RefTypeToTaskRef is new Unchecked_Conversion (Dictionary.Ref_Type, TaskRef);
   begin
      if GetSymbolDiscriminant (Item) /= Dictionary.TaskInfoSymbol then
         DiscriminantDebug ("GetTaskRef", GetSymbolDiscriminant (Item), Dictionary.TaskInfoSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToTaskRef (GetSymbolRef (Item));
   end GetTaskRef;

   --------------------------------------------------------------------------------

   procedure AddSymbol
     (Discriminant : in     Dictionary.SymbolDiscriminant;
      Ref          : in     Dictionary.Ref_Type;
      Comp_Unit    : in     ContextManager.UnitDescriptors;
      Loc          : in     LexTokenManager.Token_Position;
      Item         :    out Dictionary.Symbol)
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Discriminant,
   --#                   Ref &
   --#         Item from Dict;
   is
   begin
      Dictionary.Dynamic_Symbol_Table.Add_Symbol (The_Table    => Dictionary.Dict.Symbols,
                                                  Discriminant => Discriminant,
                                                  Ref          => Ref,
                                                  Comp_Unit    => Comp_Unit,
                                                  Loc          => Loc,
                                                  Item         => Item);
   end AddSymbol;

   --------------------------------------------------------------------------------
   --  Declaration_Info
   --------------------------------------------------------------------------------

   function Get_Declaration_Info_Ref (Item : Dictionary.Symbol) return Declaration_Info_Ref is
      function RefType_To_Declaration_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type, Declaration_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Declaration_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Declaration_Symbol then
            DiscriminantDebug ("Get_Declaration_Info_Ref", GetSymbolDiscriminant (Item), Dictionary.Declaration_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Declaration_Info_Ref");
         end if;
         return RefType_To_Declaration_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Declaration_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Declaration
     (Context         : in     Dictionary.Contexts;
      Scope           : in     Dictionary.Scopes;
      Comp_Unit       : in     ContextManager.UnitDescriptors;
      Loc             : in     LexTokenManager.Token_Position;
      The_Declaration :    out Declaration_Info_Ref) is
      The_Declaration_Symbol : Dictionary.Symbol;

      function Declaration_Info_Ref_To_RefType is new Unchecked_Conversion (Declaration_Info_Ref, Dictionary.Ref_Type);
   begin
      The_Declaration :=
        new Declaration_Info'
        (Self    => Dictionary.NullSymbol,
         Context => Context,
         Scope   => Scope,
         Item    => Dictionary.NullSymbol,
         Next    => Null_Declaration_Info_Ref);
      AddSymbol
        (Discriminant => Dictionary.Declaration_Symbol,
         Ref          => Declaration_Info_Ref_To_RefType (The_Declaration),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Declaration_Symbol);
      The_Declaration.Self := The_Declaration_Symbol;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Declaration");
   end Create_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Declaration_Context (The_Declaration : in Declaration_Info_Ref;
                                      Context         : in Contexts) is
   begin
      The_Declaration.Context := Context;
   end Set_Declaration_Context;

   --------------------------------------------------------------------------------

   procedure Set_Declaration_Item (The_Declaration : in Declaration_Info_Ref;
                                   Item            : in Dictionary.Symbol) is
   begin
      The_Declaration.Item := Item;
   end Set_Declaration_Item;

   --------------------------------------------------------------------------------

   procedure Set_Next_Declaration (The_Declaration, Next : in Declaration_Info_Ref) is
   begin
      The_Declaration.Next := Next;
   end Set_Next_Declaration;

   --------------------------------------------------------------------------------

   function Get_Declaration_Symbol (The_Declaration : Declaration_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Declaration = Null_Declaration_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Declaration.Self;
      end if;
   end Get_Declaration_Symbol;

   --------------------------------------------------------------------------------

   function Get_Declaration_Context (The_Declaration : Declaration_Info_Ref) return Dictionary.Contexts is
   begin
      return The_Declaration.Context;
   end Get_Declaration_Context;

   --------------------------------------------------------------------------------

   function Get_Declaration_Scope (The_Declaration : Declaration_Info_Ref) return Dictionary.Scopes is
   begin
      if The_Declaration = Null_Declaration_Info_Ref then
         return Dictionary.NullScope;
      else
         return The_Declaration.Scope;
      end if;
   end Get_Declaration_Scope;

   --------------------------------------------------------------------------------

   function Get_Declaration_Item (The_Declaration : Declaration_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Declaration = Null_Declaration_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Declaration.Item;
      end if;
   end Get_Declaration_Item;

   --------------------------------------------------------------------------------

   function Get_Next_Declaration (The_Declaration : Declaration_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Declaration = Null_Declaration_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return The_Declaration.Next;
      end if;
   end Get_Next_Declaration;

   --------------------------------------------------------------------------------
   --  Enumeration_Literal_Info
   --------------------------------------------------------------------------------

   function Get_Enumeration_Literal_Info_Ref (Item : Dictionary.Symbol) return Enumeration_Literal_Info_Ref is
      function RefType_To_Enumeration_Literal_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type,
                                                                                    Enumeration_Literal_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Enumeration_Literal_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Enumeration_Literal_Symbol then
            DiscriminantDebug
              ("Get_Enumeration_Literal_Info_Ref",
               GetSymbolDiscriminant (Item),
               Dictionary.Enumeration_Literal_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Enumeration_Literal_Info_Ref");
         end if;
         return RefType_To_Enumeration_Literal_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Enumeration_Literal_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Enumeration_Literal
     (Name                    : in     LexTokenManager.Lex_String;
      Position                : in     LexTokenManager.Lex_String;
      Enumeration_Type        : in     Type_Info_Ref;
      Comp_Unit               : in     ContextManager.UnitDescriptors;
      Loc                     : in     LexTokenManager.Token_Position;
      The_Enumeration_Literal :    out Enumeration_Literal_Info_Ref) is
      The_Enumeration_Literal_Symol : Dictionary.Symbol;

      function Enumeration_Literal_Info_Ref_To_RefType is new Unchecked_Conversion (Enumeration_Literal_Info_Ref,
                                                                                    Dictionary.Ref_Type);
   begin
      The_Enumeration_Literal :=
        new Enumeration_Literal_Info'
        (Self     => Dictionary.NullSymbol,
         Name     => Name,
         Position => Position,
         The_Type => Enumeration_Type,
         Next     => Null_Enumeration_Literal_Info_Ref);
      AddSymbol
        (Discriminant => Dictionary.Enumeration_Literal_Symbol,
         Ref          => Enumeration_Literal_Info_Ref_To_RefType (The_Enumeration_Literal),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Enumeration_Literal_Symol);
      The_Enumeration_Literal.Self := The_Enumeration_Literal_Symol;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Enumeration_Literal");
   end Create_Enumeration_Literal;

   --------------------------------------------------------------------------------

   procedure Set_Next_Enumeration_Literal (The_Enumeration_Literal, Next : in Enumeration_Literal_Info_Ref) is
   begin
      The_Enumeration_Literal.Next := Next;
   end Set_Next_Enumeration_Literal;

   --------------------------------------------------------------------------------

   function Get_Enumeration_Literal_Symbol (The_Enumeration_Literal : Enumeration_Literal_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Enumeration_Literal = Null_Enumeration_Literal_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Enumeration_Literal.Self;
      end if;
   end Get_Enumeration_Literal_Symbol;

   --------------------------------------------------------------------------------

   function Get_Enumeration_Literal_Name
     (The_Enumeration_Literal : Enumeration_Literal_Info_Ref)
     return                    LexTokenManager.Lex_String
   is
   begin
      if The_Enumeration_Literal = Null_Enumeration_Literal_Info_Ref then
         return LexTokenManager.Null_String;
      else
         return The_Enumeration_Literal.Name;
      end if;
   end Get_Enumeration_Literal_Name;

   --------------------------------------------------------------------------------

   function Get_Enumeration_Literal_Position
     (The_Enumeration_Literal : Enumeration_Literal_Info_Ref)
     return                    LexTokenManager.Lex_String
   is
   begin
      if The_Enumeration_Literal = Null_Enumeration_Literal_Info_Ref then
         return LexTokenManager.Null_String;
      else
         return The_Enumeration_Literal.Position;
      end if;
   end Get_Enumeration_Literal_Position;

   --------------------------------------------------------------------------------

   function Get_Enumeration_Literal_Type (The_Enumeration_Literal : Enumeration_Literal_Info_Ref) return Type_Info_Ref is
   begin
      if The_Enumeration_Literal = Null_Enumeration_Literal_Info_Ref then
         return Null_Type_Info_Ref;
      else
         return The_Enumeration_Literal.The_Type;
      end if;
   end Get_Enumeration_Literal_Type;

   --------------------------------------------------------------------------------

   function Get_Next_Enumeration_Literal
     (The_Enumeration_Literal : Enumeration_Literal_Info_Ref)
     return                    Enumeration_Literal_Info_Ref
   is
   begin
      if The_Enumeration_Literal = Null_Enumeration_Literal_Info_Ref then
         return Null_Enumeration_Literal_Info_Ref;
      else
         return The_Enumeration_Literal.Next;
      end if;
   end Get_Next_Enumeration_Literal;

   --------------------------------------------------------------------------------
   --  Array_Index_Info
   --------------------------------------------------------------------------------

   function Get_Array_Index_Info_Ref (Item : Dictionary.Symbol) return Array_Index_Info_Ref is
      function RefType_To_Array_Index_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type, Array_Index_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Array_Index_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Array_Index_Symbol then
            DiscriminantDebug ("Get_Array_Index_Info_Ref", GetSymbolDiscriminant (Item), Dictionary.Array_Index_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Array_Index_Info_Ref");
         end if;
         return RefType_To_Array_Index_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Array_Index_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Array_Index
     (Index_Type      : in     Type_Info_Ref;
      Comp_Unit       : in     ContextManager.UnitDescriptors;
      Loc             : in     LexTokenManager.Token_Position;
      The_Array_Index :    out Array_Index_Info_Ref) is
      The_Array_Index_Symbol : Dictionary.Symbol;

      function Array_Index_Info_Ref_To_RefType is new Unchecked_Conversion (Array_Index_Info_Ref, Dictionary.Ref_Type);
   begin
      The_Array_Index :=
        new Array_Index_Info'(Self       => Dictionary.NullSymbol,
                              Index_Type => Index_Type,
                              Next       => Null_Array_Index_Info_Ref);
      AddSymbol
        (Discriminant => Dictionary.Array_Index_Symbol,
         Ref          => Array_Index_Info_Ref_To_RefType (The_Array_Index),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Array_Index_Symbol);
      The_Array_Index.Self := The_Array_Index_Symbol;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Array_Index");
   end Create_Array_Index;

   --------------------------------------------------------------------------------

   procedure Set_Next_Array_Index (The_Array_Index, Next : in Array_Index_Info_Ref) is
   begin
      The_Array_Index.Next := Next;
   end Set_Next_Array_Index;

   --------------------------------------------------------------------------------

   function Get_Array_Index_Symbol (The_Array_Index : Array_Index_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Array_Index = Null_Array_Index_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Array_Index.Self;
      end if;
   end Get_Array_Index_Symbol;

   --------------------------------------------------------------------------------

   function Get_Array_Index_Type (The_Array_Index : Array_Index_Info_Ref) return Type_Info_Ref is
   begin
      if The_Array_Index = Null_Array_Index_Info_Ref then
         return Null_Type_Info_Ref;
      else
         return The_Array_Index.Index_Type;
      end if;
   end Get_Array_Index_Type;

   --------------------------------------------------------------------------------

   function Get_Next_Array_Index (The_Array_Index : Array_Index_Info_Ref) return Array_Index_Info_Ref is
   begin
      if The_Array_Index = Null_Array_Index_Info_Ref then
         return Null_Array_Index_Info_Ref;
      else
         return The_Array_Index.Next;
      end if;
   end Get_Next_Array_Index;

   --------------------------------------------------------------------------------
   --  Record_Component_Info
   --------------------------------------------------------------------------------

   function Get_Record_Component_Info_Ref (Item : Dictionary.Symbol) return Record_Component_Info_Ref is
      function RefType_To_Record_Component_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type, Record_Component_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Record_Component_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Record_Component_Symbol then
            DiscriminantDebug ("Get_Record_Component_Info_Ref", GetSymbolDiscriminant (Item), Dictionary.Record_Component_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Record_Component_Info_Ref");
         end if;
         return RefType_To_Record_Component_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Record_Component_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Record_Component
     (Name                 : in     LexTokenManager.Lex_String;
      Record_Type          : in     Type_Info_Ref;
      Component_Type       : in     Type_Info_Ref;
      Inherited_Field      : in     Boolean;
      Comp_Unit            : in     ContextManager.UnitDescriptors;
      Loc                  : in     LexTokenManager.Token_Position;
      The_Record_Component :    out Record_Component_Info_Ref) is
      The_Record_Component_Symbol : Dictionary.Symbol;

      function Record_Component_Info_Ref_To_RefType is new Unchecked_Conversion (Record_Component_Info_Ref, Dictionary.Ref_Type);
   begin
      The_Record_Component :=
        new Record_Component_Info'
        (Self            => Dictionary.NullSymbol,
         Name            => Name,
         Record_Type     => Record_Type,
         Component_Type  => Component_Type,
         Inherited_Field => Inherited_Field,
         Next            => Null_Record_Component_Info_Ref);
      AddSymbol
        (Discriminant => Dictionary.Record_Component_Symbol,
         Ref          => Record_Component_Info_Ref_To_RefType (The_Record_Component),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Record_Component_Symbol);
      The_Record_Component.Self := The_Record_Component_Symbol;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Record_Component");
   end Create_Record_Component;

   --------------------------------------------------------------------------------

   procedure Set_Next_Record_Component (The_Record_Component, Next : in Record_Component_Info_Ref) is
   begin
      The_Record_Component.Next := Next;
   end Set_Next_Record_Component;

   --------------------------------------------------------------------------------

   function Get_Record_Component_Symbol (The_Record_Component : Record_Component_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Record_Component = Null_Record_Component_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Record_Component.Self;
      end if;
   end Get_Record_Component_Symbol;

   --------------------------------------------------------------------------------

   function Get_Record_Component_Name (The_Record_Component : Record_Component_Info_Ref) return LexTokenManager.Lex_String is
   begin
      if The_Record_Component = Null_Record_Component_Info_Ref then
         return LexTokenManager.Null_String;
      else
         return The_Record_Component.Name;
      end if;
   end Get_Record_Component_Name;

   --------------------------------------------------------------------------------

   function Get_Record_Component_Record_Type (The_Record_Component : Record_Component_Info_Ref) return Type_Info_Ref is
   begin
      if The_Record_Component = Null_Record_Component_Info_Ref then
         return Null_Type_Info_Ref;
      else
         return The_Record_Component.Record_Type;
      end if;
   end Get_Record_Component_Record_Type;

   --------------------------------------------------------------------------------

   function Get_Record_Component_Type (The_Record_Component : Record_Component_Info_Ref) return Type_Info_Ref is
   begin
      if The_Record_Component = Null_Record_Component_Info_Ref then
         return Null_Type_Info_Ref;
      else
         return The_Record_Component.Component_Type;
      end if;
   end Get_Record_Component_Type;

   --------------------------------------------------------------------------------

   function Get_Record_Component_Inherited_Field (The_Record_Component : Record_Component_Info_Ref) return Boolean is
   begin
      if The_Record_Component = Null_Record_Component_Info_Ref then
         return False;
      else
         return The_Record_Component.Inherited_Field;
      end if;
   end Get_Record_Component_Inherited_Field;

   --------------------------------------------------------------------------------

   function Get_Next_Record_Component (The_Record_Component : Record_Component_Info_Ref) return Record_Component_Info_Ref is
   begin
      if The_Record_Component = Null_Record_Component_Info_Ref then
         return Null_Record_Component_Info_Ref;
      else
         return The_Record_Component.Next;
      end if;
   end Get_Next_Record_Component;

   --------------------------------------------------------------------------------
   --  Subcomponent_Info
   --------------------------------------------------------------------------------

   function Get_Subcomponent_Info_Ref (Item : Dictionary.Symbol) return Subcomponent_Info_Ref is
      function RefType_To_Subcomponent_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type, Subcomponent_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Subcomponent_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Subcomponent_Symbol then
            DiscriminantDebug ("Get_Subcomponent_Info_Ref", GetSymbolDiscriminant (Item), Dictionary.Subcomponent_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Subcomponent_Info_Ref");
         end if;
         return RefType_To_Subcomponent_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Subcomponent_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Subcomponent
     (Object           : in     Dictionary.Symbol;
      Record_Component : in     Record_Component_Info_Ref;
      Marked_Valid     : in     Boolean;
      Comp_Unit        : in     ContextManager.UnitDescriptors;
      Loc              : in     LexTokenManager.Token_Position;
      The_Subcomponent :    out Subcomponent_Info_Ref) is
      The_Subcomponent_Symbol : Dictionary.Symbol;

      function Subcomponent_Info_Ref_To_RefType is new Unchecked_Conversion (Subcomponent_Info_Ref, Dictionary.Ref_Type);
   begin
      The_Subcomponent :=
        new Subcomponent_Info'
        (Self             => Dictionary.NullSymbol,
         Object           => Object,
         Record_Component => Record_Component,
         Subcomponents    => Null_Subcomponent_Info_Ref,
         Marked_Valid     => Marked_Valid,
         Next             => Null_Subcomponent_Info_Ref);
      AddSymbol
        (Discriminant => Dictionary.Subcomponent_Symbol,
         Ref          => Subcomponent_Info_Ref_To_RefType (The_Subcomponent),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Subcomponent_Symbol);
      The_Subcomponent.Self := The_Subcomponent_Symbol;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Subcomponent");
   end Create_Subcomponent;

   --------------------------------------------------------------------------------

   procedure Set_Subcomponent_Subcomponents (The_Subcomponent : in Subcomponent_Info_Ref;
                                             Sibling          : in Subcomponent_Info_Ref) is
   begin
      The_Subcomponent.Subcomponents := Sibling;
   end Set_Subcomponent_Subcomponents;

   --------------------------------------------------------------------------------

   procedure Set_Subcomponent_Marked_Valid (The_Subcomponent : in Subcomponent_Info_Ref;
                                            Marked_Valid     : in Boolean) is
   begin
      The_Subcomponent.Marked_Valid := Marked_Valid;
   end Set_Subcomponent_Marked_Valid;

   --------------------------------------------------------------------------------

   procedure Set_Next_Subcomponent (The_Subcomponent, Next : in Subcomponent_Info_Ref) is
   begin
      The_Subcomponent.Next := Next;
   end Set_Next_Subcomponent;

   --------------------------------------------------------------------------------

   function Get_Subcomponent_Symbol (The_Subcomponent : Subcomponent_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Subcomponent = Null_Subcomponent_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Subcomponent.Self;
      end if;
   end Get_Subcomponent_Symbol;

   --------------------------------------------------------------------------------

   function Get_Subcomponent_Object (The_Subcomponent : Subcomponent_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Subcomponent = Null_Subcomponent_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Subcomponent.Object;
      end if;
   end Get_Subcomponent_Object;

   --------------------------------------------------------------------------------

   function Get_Subcomponent_Record_Component (The_Subcomponent : Subcomponent_Info_Ref) return Record_Component_Info_Ref is
   begin
      if The_Subcomponent = Null_Subcomponent_Info_Ref then
         return Null_Record_Component_Info_Ref;
      else
         return The_Subcomponent.Record_Component;
      end if;
   end Get_Subcomponent_Record_Component;

   --------------------------------------------------------------------------------

   function Get_Subcomponent_Subcomponents (The_Subcomponent : Subcomponent_Info_Ref) return Subcomponent_Info_Ref is
   begin
      if The_Subcomponent = Null_Subcomponent_Info_Ref then
         return Null_Subcomponent_Info_Ref;
      else
         return The_Subcomponent.Subcomponents;
      end if;
   end Get_Subcomponent_Subcomponents;

   --------------------------------------------------------------------------------

   function Get_Subcomponent_Marked_Valid (The_Subcomponent : Subcomponent_Info_Ref) return Boolean is
   begin
      if The_Subcomponent = Null_Subcomponent_Info_Ref then
         return False;
      else
         return The_Subcomponent.Marked_Valid;
      end if;
   end Get_Subcomponent_Marked_Valid;

   --------------------------------------------------------------------------------

   function Get_Next_Subcomponent (The_Subcomponent : Subcomponent_Info_Ref) return Subcomponent_Info_Ref is
   begin
      if The_Subcomponent = Null_Subcomponent_Info_Ref then
         return Null_Subcomponent_Info_Ref;
      else
         return The_Subcomponent.Next;
      end if;
   end Get_Next_Subcomponent;

   --------------------------------------------------------------------------------
   --  Type_Info
   --------------------------------------------------------------------------------

   function Get_Type_Info_Ref (Item : Dictionary.Symbol) return Type_Info_Ref is
      function RefType_To_Type_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type, Type_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Type_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Type_Symbol then
            DiscriminantDebug ("Get_Type_Info_Ref", GetSymbolDiscriminant (Item), Dictionary.Type_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Type_Info_Ref");
         end if;
         return RefType_To_Type_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Type_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Type
     (Name            : in     LexTokenManager.Lex_String;
      The_Declaration : in     Declaration_Info_Ref;
      Is_Private      : in     Boolean;
      Is_Announcement : in     Boolean;
      Comp_Unit       : in     ContextManager.UnitDescriptors;
      Loc             : in     LexTokenManager.Token_Position;
      Type_Mark       :    out Type_Info_Ref) is
      Type_Mark_Symbol : Dictionary.Symbol;

      function Type_Info_Ref_To_RefType is new Unchecked_Conversion (Type_Info_Ref, Dictionary.Ref_Type);
   begin
      if Is_Announcement then
         SystemErrors.RT_Assert
           (C       => not Is_Private,
            Sys_Err => SystemErrors.Other_Internal_Error,
            Msg     => "Dictionary.RawDict.Create_Type");
         Type_Mark :=
           new Type_Info'
           (Self                       => Dictionary.NullSymbol,
            Name                       => Name,
            Parent                     => Null_Type_Info_Ref,
            Declaration                => Null_Declaration_Info_Ref,
            Is_Full_Range_Subtype      => False,
            Discriminant               => Dictionary.Unknown_Type_Item,
            Is_Private                 => Dictionary.Never,
            Is_Limited                 => Dictionary.Never,
            Is_Limited_Private         => False,
            Is_Derived                 => False,
            Is_Tagged                  => False,
            Is_Own_Var_Type            => False,
            Is_Atomic                  => False,
            Equality_Defined           => True,
            Contains_Float             => False,
            Constrained                => False,
            Static                     => True,
            Wellformed                 => True,
            Base_Type                  => Null_Type_Info_Ref,
            Extends                    => Null_Type_Info_Ref,
            Accesses                   => Null_Type_Info_Ref,
            Lower                      => LexTokenManager.Null_String,
            Upper                      => LexTokenManager.Null_String,
            Modulus                    => LexTokenManager.Null_String,
            Error_Bound                => LexTokenManager.Null_String,
            Head                       => Dictionary.NullSymbol,
            Tail                       => Dictionary.NullSymbol,
            Component_Type             => Null_Type_Info_Ref,
            The_Virtual_Element_List   => Dictionary.NullSymbol,
            Ancillary_Fields           => Dictionary.NullSymbol,
            Size_Attribute             => LexTokenManager.Null_String,
            Is_Announcement_Or_Private => True,
            Is_Announcement            => True,
            Announcement_Declaration   => The_Declaration,
            Private_Declaration        => Null_Declaration_Info_Ref);
      elsif Is_Private then
         SystemErrors.RT_Assert
           (C       => not Is_Announcement,
            Sys_Err => SystemErrors.Other_Internal_Error,
            Msg     => "Dictionary.RawDict.Create_Type");
         Type_Mark :=
           new Type_Info'
           (Self                       => Dictionary.NullSymbol,
            Name                       => Name,
            Parent                     => Null_Type_Info_Ref,
            Declaration                => Null_Declaration_Info_Ref,
            Is_Full_Range_Subtype      => False,
            Discriminant               => Dictionary.Unknown_Type_Item,
            Is_Private                 => Dictionary.Sometimes,
            Is_Limited                 => Dictionary.Never,
            Is_Limited_Private         => False,
            Is_Derived                 => False,
            Is_Tagged                  => False,
            Is_Own_Var_Type            => False,
            Is_Atomic                  => False,
            Equality_Defined           => True,
            Contains_Float             => False,
            Constrained                => False,
            Static                     => True,
            Wellformed                 => True,
            Base_Type                  => Null_Type_Info_Ref,
            Extends                    => Null_Type_Info_Ref,
            Accesses                   => Null_Type_Info_Ref,
            Lower                      => LexTokenManager.Null_String,
            Upper                      => LexTokenManager.Null_String,
            Modulus                    => LexTokenManager.Null_String,
            Error_Bound                => LexTokenManager.Null_String,
            Head                       => Dictionary.NullSymbol,
            Tail                       => Dictionary.NullSymbol,
            Component_Type             => Null_Type_Info_Ref,
            The_Virtual_Element_List   => Dictionary.NullSymbol,
            Ancillary_Fields           => Dictionary.NullSymbol,
            Size_Attribute             => LexTokenManager.Null_String,
            Is_Announcement_Or_Private => True,
            Is_Announcement            => False,
            Announcement_Declaration   => Null_Declaration_Info_Ref,
            Private_Declaration        => The_Declaration);
      else
         Type_Mark :=
           new Type_Info'
           (Self                       => Dictionary.NullSymbol,
            Name                       => Name,
            Parent                     => Null_Type_Info_Ref,
            Declaration                => The_Declaration,
            Is_Full_Range_Subtype      => False,
            Discriminant               => Dictionary.Unknown_Type_Item,
            Is_Private                 => Dictionary.Never,
            Is_Limited                 => Dictionary.Never,
            Is_Limited_Private         => False,
            Is_Derived                 => False,
            Is_Tagged                  => False,
            Is_Own_Var_Type            => False,
            Is_Atomic                  => False,
            Equality_Defined           => True,
            Contains_Float             => False,
            Constrained                => False,
            Static                     => True,
            Wellformed                 => True,
            Base_Type                  => Null_Type_Info_Ref,
            Extends                    => Null_Type_Info_Ref,
            Accesses                   => Null_Type_Info_Ref,
            Lower                      => LexTokenManager.Null_String,
            Upper                      => LexTokenManager.Null_String,
            Modulus                    => LexTokenManager.Null_String,
            Error_Bound                => LexTokenManager.Null_String,
            Head                       => Dictionary.NullSymbol,
            Tail                       => Dictionary.NullSymbol,
            Component_Type             => Null_Type_Info_Ref,
            The_Virtual_Element_List   => Dictionary.NullSymbol,
            Ancillary_Fields           => Dictionary.NullSymbol,
            Size_Attribute             => LexTokenManager.Null_String,
            Is_Announcement_Or_Private => False,
            Kind_Of_Generic            => Dictionary.Invalid_Generic_Type);
      end if;
      AddSymbol
        (Discriminant => Dictionary.Type_Symbol,
         Ref          => Type_Info_Ref_To_RefType (Type_Mark),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => Type_Mark_Symbol);
      Type_Mark.Self := Type_Mark_Symbol;
      if The_Declaration /= Null_Declaration_Info_Ref then
         The_Declaration.Item := Type_Mark_Symbol;
      end if;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "Dictionary.RawDict.Create_Type");
   end Create_Type;

   --------------------------------------------------------------------------------

   procedure Set_Type_Parent (Type_Mark : in Type_Info_Ref;
                              Parent    : in Type_Info_Ref) is
   begin
      Type_Mark.Parent := Parent;
   end Set_Type_Parent;

   --------------------------------------------------------------------------------

   procedure Set_Type_Declaration (Type_Mark       : in Type_Info_Ref;
                                   The_Declaration : in Declaration_Info_Ref) is
   begin
      Type_Mark.Declaration := The_Declaration;
      The_Declaration.Item  := Type_Mark.Self; -- GAA
   end Set_Type_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Type_Is_Full_Range_Subtype (Type_Mark : in Type_Info_Ref) is
   begin
      Type_Mark.Is_Full_Range_Subtype := True;
   end Set_Type_Is_Full_Range_Subtype;

   --------------------------------------------------------------------------------

   procedure Set_Type_Discriminant (Type_Mark    : in Type_Info_Ref;
                                    Discriminant : in Dictionary.Type_Discriminant) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Discriminant");
      Type_Mark.Discriminant := Discriminant;
   end Set_Type_Discriminant;

   --------------------------------------------------------------------------------

   procedure Set_Type_Private (Type_Mark  : in Type_Info_Ref;
                               Is_Private : in Dictionary.TriState) is
   begin
      Type_Mark.Is_Private := Is_Private;
   end Set_Type_Private;

   --------------------------------------------------------------------------------

   procedure Set_Type_Limited (Type_Mark  : in Type_Info_Ref;
                               Is_Limited : in Dictionary.TriState) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Limited");
      Type_Mark.Is_Limited := Is_Limited;
   end Set_Type_Limited;

   --------------------------------------------------------------------------------

   procedure Set_Type_Is_Tagged (Type_Mark : in Type_Info_Ref;
                                 Is_Tagged : in Boolean) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Is_Tagged");
      Type_Mark.Is_Tagged := Is_Tagged;
   end Set_Type_Is_Tagged;

   --------------------------------------------------------------------------------

   procedure Set_Type_Is_Own_Var_Type (Type_Mark : in Type_Info_Ref) is
   begin
      Type_Mark.Is_Own_Var_Type := True;
   end Set_Type_Is_Own_Var_Type;

   --------------------------------------------------------------------------------

   procedure Set_Type_Extends (Type_Mark : in Type_Info_Ref;
                               Root_Type : in Type_Info_Ref) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Extends");
      Type_Mark.Extends := Root_Type;
   end Set_Type_Extends;

   --------------------------------------------------------------------------------

   procedure Set_Type_Accesses (Type_Mark  : in Type_Info_Ref;
                                The_Access : in Type_Info_Ref) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Accesses");
      Type_Mark.Accesses := The_Access;
   end Set_Type_Accesses;

   --------------------------------------------------------------------------------

   procedure Set_Type_Limited_Private (Type_Mark : in Type_Info_Ref) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Limited_Private");
      Type_Mark.Is_Limited_Private := True;
   end Set_Type_Limited_Private;

   --------------------------------------------------------------------------------

   procedure Set_Type_Derived (Type_Mark  : in Type_Info_Ref;
                               Is_Derived : in Boolean) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Derived");
      Type_Mark.Is_Derived := Is_Derived;
   end Set_Type_Derived;

   --------------------------------------------------------------------------------

   procedure Set_Type_Equality_Defined (Type_Mark        : in Type_Info_Ref;
                                        Equality_Defined : in Boolean) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Equality_Defined");
      Type_Mark.Equality_Defined := Equality_Defined;
   end Set_Type_Equality_Defined;

   --------------------------------------------------------------------------------

   procedure Set_Type_Contains_Float (Type_Mark      : in Type_Info_Ref;
                                      Contains_Float : in Boolean) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Contains_Float");
      Type_Mark.Contains_Float := Contains_Float;
   end Set_Type_Contains_Float;

   --------------------------------------------------------------------------------

   procedure Set_Type_Constrained (Type_Mark   : in Type_Info_Ref;
                                   Constrained : in Boolean) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Constrained");
      Type_Mark.Constrained := Constrained;
   end Set_Type_Constrained;

   --------------------------------------------------------------------------------

   procedure Set_Type_Static (Type_Mark : in Type_Info_Ref;
                              Static    : in Boolean) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Static");
      Type_Mark.Static := Static;
   end Set_Type_Static;

   --------------------------------------------------------------------------------

   procedure Set_Type_Wellformed (Type_Mark  : in Type_Info_Ref;
                                  Wellformed : in Boolean) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Wellformed");
      Type_Mark.Wellformed := Wellformed;
   end Set_Type_Wellformed;

   --------------------------------------------------------------------------------

   procedure Set_Type_Lower (Type_Mark : in Type_Info_Ref;
                             Lower     : in LexTokenManager.Lex_String) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Lower");
      Type_Mark.Lower := Lower;
   end Set_Type_Lower;

   --------------------------------------------------------------------------------

   procedure Set_Type_Upper (Type_Mark : in Type_Info_Ref;
                             Upper     : in LexTokenManager.Lex_String) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Upper");
      Type_Mark.Upper := Upper;
   end Set_Type_Upper;

   --------------------------------------------------------------------------------

   procedure Set_Type_Modulus (Type_Mark : in Type_Info_Ref;
                               Modulus   : in LexTokenManager.Lex_String) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Modulus");
      Type_Mark.Modulus := Modulus;
   end Set_Type_Modulus;

   --------------------------------------------------------------------------------

   procedure Set_Type_Error_Bound (Type_Mark   : in Type_Info_Ref;
                                   Error_Bound : in LexTokenManager.Lex_String) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Error_Bound");
      Type_Mark.Error_Bound := Error_Bound;
   end Set_Type_Error_Bound;

   --------------------------------------------------------------------------------

   procedure Set_Type_Base_Type (Type_Mark : in Type_Info_Ref;
                                 Base_Type : in Type_Info_Ref) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Base_Type");
      Type_Mark.Base_Type := Base_Type;
   end Set_Type_Base_Type;

   --------------------------------------------------------------------------------

   procedure Set_Type_First_Enumeration_Literal
     (Type_Mark           : in Type_Info_Ref;
      Enumeration_Literal : in Enumeration_Literal_Info_Ref) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_First_Enumeration_Literal");
      Type_Mark.Head := Get_Enumeration_Literal_Symbol (Enumeration_Literal); -- GAA
   end Set_Type_First_Enumeration_Literal;

   --------------------------------------------------------------------------------

   procedure Set_Type_Last_Enumeration_Literal
     (Type_Mark           : in Type_Info_Ref;
      Enumeration_Literal : in Enumeration_Literal_Info_Ref) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Last_Enumeration_Literal");
      Type_Mark.Tail := Get_Enumeration_Literal_Symbol (Enumeration_Literal); -- GAA
   end Set_Type_Last_Enumeration_Literal;

   --------------------------------------------------------------------------------

   procedure Set_Type_First_Array_Index (Type_Mark   : in Type_Info_Ref;
                                         Array_Index : in Array_Index_Info_Ref) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_First_Array_Index");
      Type_Mark.Head := Get_Array_Index_Symbol (Array_Index); -- GAA
   end Set_Type_First_Array_Index;

   --------------------------------------------------------------------------------

   procedure Set_Type_Last_Array_Index (Type_Mark   : in Type_Info_Ref;
                                        Array_Index : in Array_Index_Info_Ref) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Last_Array_Index");
      Type_Mark.Tail := Get_Array_Index_Symbol (Array_Index); -- GAA
   end Set_Type_Last_Array_Index;

   --------------------------------------------------------------------------------

   procedure Set_Type_Array_Component (Type_Mark      : in Type_Info_Ref;
                                       Component_Type : in Type_Info_Ref) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Array_Component");
      Type_Mark.Component_Type := Component_Type;
   end Set_Type_Array_Component;

   --------------------------------------------------------------------------------

   procedure Set_Type_First_Record_Component (Type_Mark        : in Type_Info_Ref;
                                              Record_Component : in Record_Component_Info_Ref) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_First_Record_Component");
      Type_Mark.Head := Get_Record_Component_Symbol (Record_Component); -- GAA
   end Set_Type_First_Record_Component;

   --------------------------------------------------------------------------------

   procedure Set_Type_Last_Record_Component (Type_Mark        : in Type_Info_Ref;
                                             Record_Component : in Record_Component_Info_Ref) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Last_Record_Component");
      Type_Mark.Tail := Get_Record_Component_Symbol (Record_Component); -- GAA
   end Set_Type_Last_Record_Component;

   --------------------------------------------------------------------------------

   procedure Set_Type_Ancillary_Fields (Type_Mark       : in Type_Info_Ref;
                                        The_Declaration : in Dictionary.Symbol) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Ancillary_Fields");
      Type_Mark.Ancillary_Fields := The_Declaration;
   end Set_Type_Ancillary_Fields;

   --------------------------------------------------------------------------------

   procedure Set_Type_Size_Attribute (Type_Mark : in Type_Info_Ref;
                                      Size_Val  : in LexTokenManager.Lex_String) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Size_Attribute");
      Type_Mark.Size_Attribute := Size_Val;
   end Set_Type_Size_Attribute;

   --------------------------------------------------------------------------------

   procedure Set_Type_Atomic (Type_Mark : in Type_Info_Ref) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Atomic");
      Type_Mark.Is_Atomic := True;
   end Set_Type_Atomic;

   --------------------------------------------------------------------------------

   procedure Set_Type_Virtual_Element_List (Type_Mark : in Type_Info_Ref;
                                            The_List  : in Dictionary.Symbol) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Full_Range_Subtype,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Virtual_Element_List");
      Type_Mark.The_Virtual_Element_List := The_List;
   end Set_Type_Virtual_Element_List;

   --------------------------------------------------------------------------------

   procedure Set_Type_Private_Type_Declaration (Type_Mark       : in Type_Info_Ref;
                                                The_Declaration : in Declaration_Info_Ref) is
   begin
      SystemErrors.RT_Assert
        (C       => Type_Mark.Is_Announcement_Or_Private,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Private_Type_Declaration");
      Type_Mark.Is_Private          := Dictionary.Sometimes;
      Type_Mark.Private_Declaration := The_Declaration;
   end Set_Type_Private_Type_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Type_Kind_Of_Generic (Type_Mark       : in Type_Info_Ref;
                                       Kind_Of_Generic : in Dictionary.Generic_Type_Discriminant) is
   begin
      SystemErrors.RT_Assert
        (C       => not Type_Mark.Is_Announcement_Or_Private
           and then Type_Mark.Discriminant = Dictionary.Generic_Type_Item,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Type_Kind_Of_Generic");
      Type_Mark.Kind_Of_Generic := Kind_Of_Generic;
   end Set_Type_Kind_Of_Generic;

   --------------------------------------------------------------------------------

   function Get_Type_Symbol (Type_Mark : Type_Info_Ref) return Dictionary.Symbol is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return Type_Mark.Self;
      end if;
   end Get_Type_Symbol;

   --------------------------------------------------------------------------------

   function Get_Type_Name (Type_Mark : Type_Info_Ref) return LexTokenManager.Lex_String is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return LexTokenManager.Null_String;
      else
         return Type_Mark.Name;
      end if;
   end Get_Type_Name;

   --------------------------------------------------------------------------------

   function Get_Type_Parent (Type_Mark : Type_Info_Ref) return Type_Info_Ref is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return Null_Type_Info_Ref;
      else
         return Type_Mark.Parent;
      end if;
   end Get_Type_Parent;

   --------------------------------------------------------------------------------

   function Get_Type_Declaration (Type_Mark : Type_Info_Ref) return Declaration_Info_Ref is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return Type_Mark.Declaration;
      end if;
   end Get_Type_Declaration;

   --------------------------------------------------------------------------------

   function Get_Type_Is_Full_Range_Subtype (Type_Mark : Type_Info_Ref) return Boolean is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return False;
      else
         return Type_Mark.Is_Full_Range_Subtype;
      end if;
   end Get_Type_Is_Full_Range_Subtype;

   --------------------------------------------------------------------------------

   function Get_First_Constrained_Subtype (Type_Mark : Type_Info_Ref) return Type_Info_Ref is
      Result : Type_Info_Ref;
   begin
      Result := Type_Mark;
      while Result /= Null_Type_Info_Ref
        and then Get_Type_Is_Full_Range_Subtype (Type_Mark => Result) loop
         Result := Get_Type_Parent (Type_Mark => Result);
      end loop;
      return Result;
   end Get_First_Constrained_Subtype;

   --------------------------------------------------------------------------------

   function Get_Type_Discriminant (Type_Mark : Type_Info_Ref) return Dictionary.Type_Discriminant is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return Dictionary.Unknown_Type_Item;
      else
         return Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Discriminant;
      end if;
   end Get_Type_Discriminant;

   --------------------------------------------------------------------------------

   function Get_Type_Private (Type_Mark : Type_Info_Ref) return Dictionary.TriState is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return Dictionary.Never;
      else
         return Type_Mark.Is_Private;
      end if;
   end Get_Type_Private;

   --------------------------------------------------------------------------------

   function Get_Type_Limited (Type_Mark : Type_Info_Ref) return Dictionary.TriState is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return Dictionary.Never;
      else
         return Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Is_Limited;
      end if;
   end Get_Type_Limited;

   --------------------------------------------------------------------------------

   function Get_Type_Limited_Private (Type_Mark : Type_Info_Ref) return Boolean is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return False;
      else
         return Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Is_Limited_Private;
      end if;
   end Get_Type_Limited_Private;

   --------------------------------------------------------------------------------

   function Get_Type_Derived (Type_Mark : Type_Info_Ref) return Boolean is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return False;
      else
         return Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Is_Derived;
      end if;
   end Get_Type_Derived;

   --------------------------------------------------------------------------------

   function Get_Type_Is_Tagged (Type_Mark : Type_Info_Ref) return Boolean is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return False;
      else
         return Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Is_Tagged;
      end if;
   end Get_Type_Is_Tagged;

   --------------------------------------------------------------------------------

   function Get_Type_Is_Own_Var_Type (Type_Mark : Type_Info_Ref) return Boolean is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return False;
      else
         return Type_Mark.Is_Own_Var_Type;
      end if;
   end Get_Type_Is_Own_Var_Type;

   --------------------------------------------------------------------------------

   function Get_Type_Extends (Type_Mark : Type_Info_Ref) return Type_Info_Ref is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return Null_Type_Info_Ref;
      else
         return Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Extends;
      end if;
   end Get_Type_Extends;

   --------------------------------------------------------------------------------

   function Get_Type_Accesses (Type_Mark : Type_Info_Ref) return Type_Info_Ref is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return Null_Type_Info_Ref;
      else
         return Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Accesses;
      end if;
   end Get_Type_Accesses;

   --------------------------------------------------------------------------------

   function Get_Type_Equality_Defined (Type_Mark : Type_Info_Ref) return Boolean is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return False;
      else
         return Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Equality_Defined;
      end if;
   end Get_Type_Equality_Defined;

   --------------------------------------------------------------------------------

   function Get_Type_Contains_Float (Type_Mark : Type_Info_Ref) return Boolean is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return False;
      else
         return Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Contains_Float;
      end if;
   end Get_Type_Contains_Float;

   --------------------------------------------------------------------------------

   function Get_Type_Constrained (Type_Mark : Type_Info_Ref) return Boolean is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return False;
      else
         return Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Constrained;
      end if;
   end Get_Type_Constrained;

   --------------------------------------------------------------------------------

   function Get_Type_Static (Type_Mark : Type_Info_Ref) return Boolean is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return False;
      else
         return Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Static;
      end if;
   end Get_Type_Static;

   --------------------------------------------------------------------------------

   function Get_Type_Wellformed (Type_Mark : Type_Info_Ref) return Boolean is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return False;
      else
         return Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Wellformed;
      end if;
   end Get_Type_Wellformed;

   --------------------------------------------------------------------------------

   function Get_Type_Base_Type (Type_Mark : Type_Info_Ref) return Type_Info_Ref is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return Null_Type_Info_Ref;
      else
         return Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Base_Type;
      end if;
   end Get_Type_Base_Type;

   --------------------------------------------------------------------------------

   function Get_Type_Lower (Type_Mark : Type_Info_Ref) return LexTokenManager.Lex_String is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return LexTokenManager.Null_String;
      else
         return Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Lower;
      end if;
   end Get_Type_Lower;

   --------------------------------------------------------------------------------

   function Get_Type_Upper (Type_Mark : Type_Info_Ref) return LexTokenManager.Lex_String is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return LexTokenManager.Null_String;
      else
         return Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Upper;
      end if;
   end Get_Type_Upper;

   --------------------------------------------------------------------------------

   function Get_Type_Modulus (Type_Mark : Type_Info_Ref) return LexTokenManager.Lex_String is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return LexTokenManager.Null_String;
      else
         return Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Modulus;
      end if;
   end Get_Type_Modulus;

   --------------------------------------------------------------------------------

   function Get_Type_Error_Bound (Type_Mark : Type_Info_Ref) return LexTokenManager.Lex_String is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return LexTokenManager.Null_String;
      else
         return Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Error_Bound;
      end if;
   end Get_Type_Error_Bound;

   --------------------------------------------------------------------------------

   function Get_Type_First_Enumeration_Literal (Type_Mark : Type_Info_Ref) return Enumeration_Literal_Info_Ref is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return Null_Enumeration_Literal_Info_Ref;
      else
         return Get_Enumeration_Literal_Info_Ref (Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Head); -- GAA
      end if;
   end Get_Type_First_Enumeration_Literal;

   --------------------------------------------------------------------------------

   function Get_Type_Last_Enumeration_Literal (Type_Mark : Type_Info_Ref) return Enumeration_Literal_Info_Ref is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return Null_Enumeration_Literal_Info_Ref;
      else
         return Get_Enumeration_Literal_Info_Ref (Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Tail); -- GAA
      end if;
   end Get_Type_Last_Enumeration_Literal;

   --------------------------------------------------------------------------------

   function Get_Type_First_Array_Index (Type_Mark : Type_Info_Ref) return Array_Index_Info_Ref is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return Null_Array_Index_Info_Ref;
      else
         return Get_Array_Index_Info_Ref (Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Head); -- GAA
      end if;
   end Get_Type_First_Array_Index;

   --------------------------------------------------------------------------------

   function Get_Type_Last_Array_Index (Type_Mark : Type_Info_Ref) return Array_Index_Info_Ref is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return Null_Array_Index_Info_Ref;
      else
         return Get_Array_Index_Info_Ref (Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Tail); -- GAA
      end if;
   end Get_Type_Last_Array_Index;

   --------------------------------------------------------------------------------

   function Get_Type_Array_Component (Type_Mark : Type_Info_Ref) return Type_Info_Ref is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return Null_Type_Info_Ref;
      else
         return Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Component_Type;
      end if;
   end Get_Type_Array_Component;

   --------------------------------------------------------------------------------

   function Get_Type_First_Record_Component (Type_Mark : Type_Info_Ref) return Record_Component_Info_Ref is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return Null_Record_Component_Info_Ref;
      else
         return Get_Record_Component_Info_Ref (Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Head); -- GAA
      end if;
   end Get_Type_First_Record_Component;

   --------------------------------------------------------------------------------

   function Get_Type_Last_Record_Component (Type_Mark : Type_Info_Ref) return Record_Component_Info_Ref is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return Null_Record_Component_Info_Ref;
      else
         return Get_Record_Component_Info_Ref (Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Tail); -- GAA
      end if;
   end Get_Type_Last_Record_Component;

   --------------------------------------------------------------------------------

   function Get_Type_Announcement (Type_Mark : Type_Info_Ref) return Declaration_Info_Ref is
   begin
      if Type_Mark /= Null_Type_Info_Ref
        and then Type_Mark.Is_Announcement_Or_Private
        and then Type_Mark.Is_Announcement then
         return Type_Mark.Announcement_Declaration;
      else
         return Null_Declaration_Info_Ref;
      end if;
   end Get_Type_Announcement;

   --------------------------------------------------------------------------------

   function Get_Type_Private_Type_Declaration (Type_Mark : Type_Info_Ref) return Declaration_Info_Ref is
   begin
      if Type_Mark /= Null_Type_Info_Ref and then Type_Mark.Is_Announcement_Or_Private then
         return Type_Mark.Private_Declaration;
      else
         return Null_Declaration_Info_Ref;
      end if;
   end Get_Type_Private_Type_Declaration;

   --------------------------------------------------------------------------------

   function Get_Type_Ancillary_Fields (Type_Mark : Type_Info_Ref) return Dictionary.Symbol is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Ancillary_Fields;
      end if;
   end Get_Type_Ancillary_Fields;

   --------------------------------------------------------------------------------

   function Get_Type_Size_Attribute (Type_Mark : Type_Info_Ref) return LexTokenManager.Lex_String is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return LexTokenManager.Null_String;
      else
         return Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Size_Attribute;
      end if;
   end Get_Type_Size_Attribute;

   --------------------------------------------------------------------------------

   function Get_Type_Atomic (Type_Mark : Type_Info_Ref) return Boolean is
   begin
      return Get_First_Constrained_Subtype (Type_Mark => Type_Mark).Is_Atomic;
   end Get_Type_Atomic;

   --------------------------------------------------------------------------------

   function Get_Type_Virtual_Element_List (Type_Mark : Type_Info_Ref) return Dictionary.Symbol is
   begin
      if Type_Mark = Null_Type_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return Get_First_Constrained_Subtype (Type_Mark => Type_Mark).The_Virtual_Element_List;
      end if;
   end Get_Type_Virtual_Element_List;

   --------------------------------------------------------------------------------

   function Get_Type_Kind_Of_Generic (Type_Mark : Type_Info_Ref) return Dictionary.Generic_Type_Discriminant is
      Constrained_Type_Mark : constant Type_Info_Ref := Get_First_Constrained_Subtype (Type_Mark => Type_Mark);
   begin
      if Constrained_Type_Mark = Null_Type_Info_Ref
        or else Constrained_Type_Mark.Is_Announcement_Or_Private
        or else Constrained_Type_Mark.Discriminant /= Dictionary.Generic_Type_Item then
         return Dictionary.Invalid_Generic_Type;
      else
         return Constrained_Type_Mark.Kind_Of_Generic;
      end if;
   end Get_Type_Kind_Of_Generic;

   --------------------------------------------------------------------------------
   --  Protected_Type_Info
   --------------------------------------------------------------------------------

   procedure Set_Protected_Type_Own_Variable (The_Protected_Type : in Type_Info_Ref;
                                              Own_Variable       : in Own_Variable_Info_Ref) is
   begin
      GetProtectedRef (The_Protected_Type.Ancillary_Fields).Own_Variable := Own_Variable;
   end Set_Protected_Type_Own_Variable;

   --------------------------------------------------------------------------------

   procedure Set_Protected_Type_Elements_Hidden (The_Protected_Type : in Type_Info_Ref) is
   begin
      GetProtectedRef (The_Protected_Type.Ancillary_Fields).ElementsHidden := True;
   end Set_Protected_Type_Elements_Hidden;

   --------------------------------------------------------------------------------

   procedure Set_Protected_Type_First_Visible_Declaration
     (The_Protected_Type : in Type_Info_Ref;
      The_Declaration    : in Declaration_Info_Ref) is
   begin
      GetProtectedRef (The_Protected_Type.Ancillary_Fields).Visible_Part.First_Declaration := The_Declaration;
   end Set_Protected_Type_First_Visible_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Protected_Type_Last_Visible_Declaration
     (The_Protected_Type : in Type_Info_Ref;
      The_Declaration    : in Declaration_Info_Ref) is
   begin
      GetProtectedRef (The_Protected_Type.Ancillary_Fields).Visible_Part.Last_Declaration := The_Declaration;
   end Set_Protected_Type_Last_Visible_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Protected_Type_First_Private_Declaration
     (The_Protected_Type : in Type_Info_Ref;
      The_Declaration    : in Declaration_Info_Ref) is
   begin
      GetProtectedRef (The_Protected_Type.Ancillary_Fields).Private_Part.First_Declaration := The_Declaration;
   end Set_Protected_Type_First_Private_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Protected_Type_Last_Private_Declaration
     (The_Protected_Type : in Type_Info_Ref;
      The_Declaration    : in Declaration_Info_Ref) is
   begin
      GetProtectedRef (The_Protected_Type.Ancillary_Fields).Private_Part.Last_Declaration := The_Declaration;
   end Set_Protected_Type_Last_Private_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Protected_Type_First_Local_Declaration
     (The_Protected_Type : in Type_Info_Ref;
      The_Declaration    : in Declaration_Info_Ref) is
   begin
      GetProtectedRef (The_Protected_Type.Ancillary_Fields).Local_Part.First_Declaration := The_Declaration;
   end Set_Protected_Type_First_Local_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Protected_Type_Last_Local_Declaration
     (The_Protected_Type : in Type_Info_Ref;
      The_Declaration    : in Declaration_Info_Ref) is
   begin
      GetProtectedRef (The_Protected_Type.Ancillary_Fields).Local_Part.Last_Declaration := The_Declaration;
   end Set_Protected_Type_Last_Local_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Protected_Type_The_Entry (The_Protected_Type : in Type_Info_Ref;
                                           The_Entry          : in Dictionary.Symbol) is
   begin
      GetProtectedRef (The_Protected_Type.Ancillary_Fields).TheEntry := The_Entry;
   end Set_Protected_Type_The_Entry;

   --------------------------------------------------------------------------------

   procedure Set_Protected_Type_First_Discriminant (The_Protected_Type : in Type_Info_Ref;
                                                    Discriminant       : in Dictionary.Symbol) is
   begin
      GetProtectedRef (The_Protected_Type.Ancillary_Fields).FirstDiscriminant := Discriminant;
   end Set_Protected_Type_First_Discriminant;

   --------------------------------------------------------------------------------

   procedure Set_Protected_Type_Last_Discriminant (The_Protected_Type : in Type_Info_Ref;
                                                   Discriminant       : in Dictionary.Symbol) is
   begin
      GetProtectedRef (The_Protected_Type.Ancillary_Fields).LastDiscriminant := Discriminant;
   end Set_Protected_Type_Last_Discriminant;

   --------------------------------------------------------------------------------

   procedure Set_Protected_Type_Has_Pragma (The_Protected_Type : in Type_Info_Ref;
                                            The_Pragma         : in Dictionary.RavenscarPragmas) is
   begin
      GetProtectedRef (The_Protected_Type.Ancillary_Fields).Pragmas (The_Pragma).Given := True;
   end Set_Protected_Type_Has_Pragma;

   --------------------------------------------------------------------------------

   procedure Set_Protected_Type_Pragma_Value
     (The_Protected_Type : in Type_Info_Ref;
      The_Pragma         : in Dictionary.RavenscarPragmasWithValue;
      The_Value          : in LexTokenManager.Lex_String) is
   begin
      GetProtectedRef (The_Protected_Type.Ancillary_Fields).Pragmas (The_Pragma).Value := The_Value;
   end Set_Protected_Type_Pragma_Value;

   --------------------------------------------------------------------------------

   procedure Set_Protected_Type_Body (The_Protected_Type : in Type_Info_Ref;
                                      The_Body           : in Declaration_Info_Ref) is
   begin
      GetProtectedRef (The_Protected_Type.Ancillary_Fields).Protected_Body := The_Body;
   end Set_Protected_Type_Body;

   --------------------------------------------------------------------------------

   procedure Set_Protected_Type_Has_Proper_Body (The_Protected_Type : in Type_Info_Ref) is
   begin
      GetProtectedRef (The_Protected_Type.Ancillary_Fields).Has_Proper_Body := True;
   end Set_Protected_Type_Has_Proper_Body;

   --------------------------------------------------------------------------------

   procedure Set_Protected_Type_With_Clauses
     (The_Protected_Type : in Type_Info_Ref;
      The_Context_Clause : in Context_Clause_Info_Ref) is
   begin
      GetProtectedRef (The_Protected_Type.Ancillary_Fields).Local_Part.With_Clauses := The_Context_Clause;
   end Set_Protected_Type_With_Clauses;

   --------------------------------------------------------------------------------

   procedure Set_Protected_Type_Use_Type_Clauses
     (The_Protected_Type  : in Type_Info_Ref;
      The_Use_Type_Clause : in Use_Type_Clause_Info_Ref) is
   begin
      GetProtectedRef (The_Protected_Type.Ancillary_Fields).Local_Part.Use_Type_Clauses := The_Use_Type_Clause;
   end Set_Protected_Type_Use_Type_Clauses;

   --------------------------------------------------------------------------------

   function Get_Protected_Type_Own_Variable (The_Protected_Type : Type_Info_Ref) return Own_Variable_Info_Ref is
   begin
      if The_Protected_Type = Null_Type_Info_Ref then
         return Null_Own_Variable_Info_Ref;
      else
         return GetProtectedRef (The_Protected_Type.Ancillary_Fields).Own_Variable;
      end if;
   end Get_Protected_Type_Own_Variable;

   --------------------------------------------------------------------------------

   function Get_Protected_Type_Elements_Hidden (The_Protected_Type : Type_Info_Ref) return Boolean is
   begin
      if The_Protected_Type = Null_Type_Info_Ref then
         return False;
      else
         return GetProtectedRef (The_Protected_Type.Ancillary_Fields).ElementsHidden;
      end if;
   end Get_Protected_Type_Elements_Hidden;

   --------------------------------------------------------------------------------

   function Get_Protected_Type_Has_Entry (The_Protected_Type : Type_Info_Ref) return Boolean is
   begin
      if The_Protected_Type = Null_Type_Info_Ref then
         return False;
      else
         return GetProtectedRef (The_Protected_Type.Ancillary_Fields).TheEntry /= Dictionary.NullSymbol;
      end if;
   end Get_Protected_Type_Has_Entry;

   --------------------------------------------------------------------------------

   function Get_Protected_Type_First_Visible_Declaration (The_Protected_Type : Type_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Protected_Type = Null_Type_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return GetProtectedRef (The_Protected_Type.Ancillary_Fields).Visible_Part.First_Declaration;
      end if;
   end Get_Protected_Type_First_Visible_Declaration;

   --------------------------------------------------------------------------------

   function Get_Protected_Type_Last_Visible_Declaration (The_Protected_Type : Type_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Protected_Type = Null_Type_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return GetProtectedRef (The_Protected_Type.Ancillary_Fields).Visible_Part.Last_Declaration;
      end if;
   end Get_Protected_Type_Last_Visible_Declaration;

   --------------------------------------------------------------------------------

   function Get_Protected_Type_First_Private_Declaration (The_Protected_Type : Type_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Protected_Type = Null_Type_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return GetProtectedRef (The_Protected_Type.Ancillary_Fields).Private_Part.First_Declaration;
      end if;
   end Get_Protected_Type_First_Private_Declaration;

   --------------------------------------------------------------------------------

   function Get_Protected_Type_Last_Private_Declaration (The_Protected_Type : Type_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Protected_Type = Null_Type_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return GetProtectedRef (The_Protected_Type.Ancillary_Fields).Private_Part.Last_Declaration;
      end if;
   end Get_Protected_Type_Last_Private_Declaration;

   --------------------------------------------------------------------------------

   function Get_Protected_Type_First_Local_Declaration (The_Protected_Type : Type_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Protected_Type = Null_Type_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return GetProtectedRef (The_Protected_Type.Ancillary_Fields).Local_Part.First_Declaration;
      end if;
   end Get_Protected_Type_First_Local_Declaration;

   --------------------------------------------------------------------------------

   function Get_Protected_Type_Last_Local_Declaration (The_Protected_Type : Type_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Protected_Type = Null_Type_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return GetProtectedRef (The_Protected_Type.Ancillary_Fields).Local_Part.Last_Declaration;
      end if;
   end Get_Protected_Type_Last_Local_Declaration;

   --------------------------------------------------------------------------------

   function Get_Protected_Type_First_Discriminant (The_Protected_Type : Type_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Protected_Type = Null_Type_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return GetProtectedRef (The_Protected_Type.Ancillary_Fields).FirstDiscriminant;
      end if;
   end Get_Protected_Type_First_Discriminant;

   --------------------------------------------------------------------------------

   function Get_Protected_Type_Last_Discriminant (The_Protected_Type : Type_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Protected_Type = Null_Type_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return GetProtectedRef (The_Protected_Type.Ancillary_Fields).LastDiscriminant;
      end if;
   end Get_Protected_Type_Last_Discriminant;

   --------------------------------------------------------------------------------

   function Get_Protected_Type_Has_Pragma
     (The_Protected_Type : Type_Info_Ref;
      The_Pragma         : Dictionary.RavenscarPragmas)
     return               Boolean
   is
   begin
      if The_Protected_Type = Null_Type_Info_Ref then
         return False;
      else
         return GetProtectedRef (The_Protected_Type.Ancillary_Fields).Pragmas (The_Pragma).Given;
      end if;
   end Get_Protected_Type_Has_Pragma;

   --------------------------------------------------------------------------------

   function Get_Protected_Type_Pragma_Value
     (The_Protected_Type : Type_Info_Ref;
      The_Pragma         : Dictionary.RavenscarPragmasWithValue)
     return               LexTokenManager.Lex_String
   is
   begin
      if The_Protected_Type = Null_Type_Info_Ref then
         return LexTokenManager.Null_String;
      else
         return GetProtectedRef (The_Protected_Type.Ancillary_Fields).Pragmas (The_Pragma).Value;
      end if;
   end Get_Protected_Type_Pragma_Value;

   --------------------------------------------------------------------------------

   function Get_Protected_Type_Body (The_Protected_Type : Type_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Protected_Type = Null_Type_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return GetProtectedRef (The_Protected_Type.Ancillary_Fields).Protected_Body;
      end if;
   end Get_Protected_Type_Body;

   --------------------------------------------------------------------------------

   function Get_Protected_Type_Has_Proper_Body (The_Protected_Type : Type_Info_Ref) return Boolean is
   begin
      if The_Protected_Type = Null_Type_Info_Ref then
         return False;
      else
         return GetProtectedRef (The_Protected_Type.Ancillary_Fields).Has_Proper_Body;
      end if;
   end Get_Protected_Type_Has_Proper_Body;

   --------------------------------------------------------------------------------

   function Get_Protected_Type_With_Clauses (The_Protected_Type : Type_Info_Ref) return Context_Clause_Info_Ref is
   begin
      if The_Protected_Type = Null_Type_Info_Ref then
         return Null_Context_Clause_Info_Ref;
      else
         return GetProtectedRef (The_Protected_Type.Ancillary_Fields).Local_Part.With_Clauses;
      end if;
   end Get_Protected_Type_With_Clauses;

   --------------------------------------------------------------------------------

   function Get_Protected_Type_Use_Type_Clauses (The_Protected_Type : Type_Info_Ref) return Use_Type_Clause_Info_Ref is
   begin
      if The_Protected_Type = Null_Type_Info_Ref then
         return Null_Use_Type_Clause_Info_Ref;
      else
         return GetProtectedRef (The_Protected_Type.Ancillary_Fields).Local_Part.Use_Type_Clauses;
      end if;
   end Get_Protected_Type_Use_Type_Clauses;

   --------------------------------------------------------------------------------
   --  Task_Type_Info
   --------------------------------------------------------------------------------

   procedure Set_Task_Type_Signature_Not_Wellformed (The_Task_Type : in Type_Info_Ref;
                                                     Abstraction   : in Dictionary.Abstractions) is
   begin
      GetTaskRef (The_Task_Type.Ancillary_Fields).Signature_Is_Wellformed (Abstraction) := False;
      --  we may be marking a missing second anno as malformed so we
      --  must also mark it as present, thus
      if Abstraction = Dictionary.IsRefined then
         GetTaskRef (The_Task_Type.Ancillary_Fields).Has_Second_Annotation := True;
      end if;
   end Set_Task_Type_Signature_Not_Wellformed;

   --------------------------------------------------------------------------------

   procedure Set_Task_Type_Has_Second_Annotation (The_Task_Type : in Type_Info_Ref) is
   begin
      GetTaskRef (The_Task_Type.Ancillary_Fields).Has_Second_Annotation := True;
   end Set_Task_Type_Has_Second_Annotation;

   --------------------------------------------------------------------------------

   procedure Set_Task_Type_Has_Derives_Annotation (The_Task_Type : in Type_Info_Ref) is
   begin
      GetTaskRef (The_Task_Type.Ancillary_Fields).Has_Derives_Annotation := True;
   end Set_Task_Type_Has_Derives_Annotation;

   --------------------------------------------------------------------------------

   procedure Set_Task_Type_First_Local_Declaration (The_Task_Type   : in Type_Info_Ref;
                                                    The_Declaration : in Declaration_Info_Ref) is
   begin
      GetTaskRef (The_Task_Type.Ancillary_Fields).Local_Part.First_Declaration := The_Declaration;
   end Set_Task_Type_First_Local_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Task_Type_Last_Local_Declaration (The_Task_Type   : in Type_Info_Ref;
                                                   The_Declaration : in Declaration_Info_Ref) is
   begin
      GetTaskRef (The_Task_Type.Ancillary_Fields).Local_Part.Last_Declaration := The_Declaration;
   end Set_Task_Type_Last_Local_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Task_Type_With_Clauses (The_Task_Type      : in Type_Info_Ref;
                                         The_Context_Clause : in Context_Clause_Info_Ref) is
   begin
      GetTaskRef (The_Task_Type.Ancillary_Fields).Local_Part.With_Clauses := The_Context_Clause;
   end Set_Task_Type_With_Clauses;

   --------------------------------------------------------------------------------

   procedure Set_Task_Type_Use_Type_Clauses
     (The_Task_Type       : in Type_Info_Ref;
      The_Use_Type_Clause : in Use_Type_Clause_Info_Ref) is
   begin
      GetTaskRef (The_Task_Type.Ancillary_Fields).Local_Part.Use_Type_Clauses := The_Use_Type_Clause;
   end Set_Task_Type_Use_Type_Clauses;

   --------------------------------------------------------------------------------

   procedure Set_Task_Type_First_Discriminant (The_Task_Type : in Type_Info_Ref;
                                               Discriminant  : in Dictionary.Symbol) is
   begin
      GetTaskRef (The_Task_Type.Ancillary_Fields).FirstDiscriminant := Discriminant;
   end Set_Task_Type_First_Discriminant;

   --------------------------------------------------------------------------------

   procedure Set_Task_Type_Last_Discriminant (The_Task_Type : in Type_Info_Ref;
                                              Discriminant  : in Dictionary.Symbol) is
   begin
      GetTaskRef (The_Task_Type.Ancillary_Fields).LastDiscriminant := Discriminant;
   end Set_Task_Type_Last_Discriminant;

   --------------------------------------------------------------------------------

   procedure Set_Task_Type_First_Global_Variable
     (The_Task_Type       : in Type_Info_Ref;
      Abstraction         : in Dictionary.Abstractions;
      The_Global_Variable : in Global_Variable_Info_Ref) is
   begin
      GetTaskRef (The_Task_Type.Ancillary_Fields).First_Global_Variable (Abstraction) := The_Global_Variable;
   end Set_Task_Type_First_Global_Variable;

   --------------------------------------------------------------------------------

   procedure Set_Task_Type_Last_Global_Variable
     (The_Task_Type       : in Type_Info_Ref;
      Abstraction         : in Dictionary.Abstractions;
      The_Global_Variable : in Global_Variable_Info_Ref) is
   begin
      GetTaskRef (The_Task_Type.Ancillary_Fields).Last_Global_Variable (Abstraction) := The_Global_Variable;
   end Set_Task_Type_Last_Global_Variable;

   --------------------------------------------------------------------------------

   procedure Set_Task_Type_Has_Pragma (The_Task_Type : in Type_Info_Ref;
                                       The_Pragma    : in Dictionary.RavenscarPragmas) is
   begin
      GetTaskRef (The_Task_Type.Ancillary_Fields).Pragmas (The_Pragma).Given := True;
   end Set_Task_Type_Has_Pragma;

   --------------------------------------------------------------------------------

   procedure Set_Task_Type_Pragma_Value
     (The_Task_Type : in Type_Info_Ref;
      The_Pragma    : in Dictionary.RavenscarPragmasWithValue;
      The_Value     : in LexTokenManager.Lex_String) is
   begin
      GetTaskRef (The_Task_Type.Ancillary_Fields).Pragmas (The_Pragma).Value := The_Value;
   end Set_Task_Type_Pragma_Value;

   --------------------------------------------------------------------------------

   procedure Set_Task_Type_First_Loop (The_Task_Type : in Type_Info_Ref;
                                       The_Loop      : in Dictionary.Symbol) is
   begin
      GetTaskRef (The_Task_Type.Ancillary_Fields).First_Loop := The_Loop;
   end Set_Task_Type_First_Loop;

   --------------------------------------------------------------------------------

   procedure Set_Task_Type_Last_Loop (The_Task_Type : in Type_Info_Ref;
                                      The_Loop      : in Dictionary.Symbol) is
   begin
      GetTaskRef (The_Task_Type.Ancillary_Fields).Last_Loop := The_Loop;
   end Set_Task_Type_Last_Loop;

   --------------------------------------------------------------------------------

   procedure Set_Task_Type_Suspends_List (The_Task_Type     : in Type_Info_Ref;
                                          The_Suspends_List : in Dictionary.Symbol) is
   begin
      GetTaskRef (The_Task_Type.Ancillary_Fields).Suspends_List := The_Suspends_List;
   end Set_Task_Type_Suspends_List;

   --------------------------------------------------------------------------------

   procedure Set_Task_Type_Body (The_Task_Type : in Type_Info_Ref;
                                 The_Body      : in Declaration_Info_Ref) is
   begin
      GetTaskRef (The_Task_Type.Ancillary_Fields).Task_Body := The_Body;
   end Set_Task_Type_Body;

   --------------------------------------------------------------------------------

   procedure Set_Task_Type_Has_Proper_Body (The_Task_Type : in Type_Info_Ref;
                                            Is_Hidden     : in Boolean) is
   begin
      GetTaskRef (The_Task_Type.Ancillary_Fields).Has_Proper_Body := True;
      GetTaskRef (The_Task_Type.Ancillary_Fields).Body_Is_Hidden  := Is_Hidden;
   end Set_Task_Type_Has_Proper_Body;

   --------------------------------------------------------------------------------

   procedure Set_Task_Type_Uses_Unprotected_Variables (The_Task_Type : in Type_Info_Ref) is
   begin
      GetTaskRef (The_Task_Type.Ancillary_Fields).Uses_Unprotected_Variables := True;
   end Set_Task_Type_Uses_Unprotected_Variables;

   --------------------------------------------------------------------------------

   procedure Set_Task_Type_Uses_Unchecked_Conversion (The_Task_Type : in Type_Info_Ref) is
   begin
      GetTaskRef (The_Task_Type.Ancillary_Fields).Uses_Unchecked_Conversion := True;
   end Set_Task_Type_Uses_Unchecked_Conversion;

   --------------------------------------------------------------------------------

   procedure Set_Task_Type_Assigns_From_External (The_Task_Type : in Type_Info_Ref) is
   begin
      GetTaskRef (The_Task_Type.Ancillary_Fields).Assigns_From_External := True;
   end Set_Task_Type_Assigns_From_External;

   --------------------------------------------------------------------------------

   function Get_Task_Type_Signature_Is_Wellformed
     (The_Task_Type : Type_Info_Ref;
      Abstraction   : Dictionary.Abstractions)
     return          Boolean
   is
   begin
      if The_Task_Type = Null_Type_Info_Ref then
         return False;
      else
         return GetTaskRef (The_Task_Type.Ancillary_Fields).Signature_Is_Wellformed (Abstraction);
      end if;
   end Get_Task_Type_Signature_Is_Wellformed;

   --------------------------------------------------------------------------------

   function Get_Task_Type_Has_Second_Annotation (The_Task_Type : Type_Info_Ref) return Boolean is
   begin
      if The_Task_Type = Null_Type_Info_Ref then
         return False;
      else
         return GetTaskRef (The_Task_Type.Ancillary_Fields).Has_Second_Annotation;
      end if;
   end Get_Task_Type_Has_Second_Annotation;

   --------------------------------------------------------------------------------

   function Get_Task_Type_Has_Derives_Annotation (The_Task_Type : Type_Info_Ref) return Boolean is
   begin
      if The_Task_Type = Null_Type_Info_Ref then
         return False;
      else
         return GetTaskRef (The_Task_Type.Ancillary_Fields).Has_Derives_Annotation;
      end if;
   end Get_Task_Type_Has_Derives_Annotation;

   --------------------------------------------------------------------------------

   function Get_Task_Type_First_Local_Declaration (The_Task_Type : Type_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Task_Type = Null_Type_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return GetTaskRef (The_Task_Type.Ancillary_Fields).Local_Part.First_Declaration;
      end if;
   end Get_Task_Type_First_Local_Declaration;

   --------------------------------------------------------------------------------

   function Get_Task_Type_Last_Local_Declaration (The_Task_Type : Type_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Task_Type = Null_Type_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return GetTaskRef (The_Task_Type.Ancillary_Fields).Local_Part.Last_Declaration;
      end if;
   end Get_Task_Type_Last_Local_Declaration;

   --------------------------------------------------------------------------------

   function Get_Task_Type_With_Clauses (The_Task_Type : Type_Info_Ref) return Context_Clause_Info_Ref is
   begin
      if The_Task_Type = Null_Type_Info_Ref then
         return Null_Context_Clause_Info_Ref;
      else
         return GetTaskRef (The_Task_Type.Ancillary_Fields).Local_Part.With_Clauses;
      end if;
   end Get_Task_Type_With_Clauses;

   --------------------------------------------------------------------------------

   function Get_Task_Type_Use_Type_Clauses (The_Task_Type : Type_Info_Ref) return Use_Type_Clause_Info_Ref is
   begin
      if The_Task_Type = Null_Type_Info_Ref then
         return Null_Use_Type_Clause_Info_Ref;
      else
         return GetTaskRef (The_Task_Type.Ancillary_Fields).Local_Part.Use_Type_Clauses;
      end if;
   end Get_Task_Type_Use_Type_Clauses;

   --------------------------------------------------------------------------------

   function Get_Task_Type_First_Discriminant (The_Task_Type : Type_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Task_Type = Null_Type_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return GetTaskRef (The_Task_Type.Ancillary_Fields).FirstDiscriminant;
      end if;
   end Get_Task_Type_First_Discriminant;

   --------------------------------------------------------------------------------

   function Get_Task_Type_Last_Discriminant (The_Task_Type : Type_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Task_Type = Null_Type_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return GetTaskRef (The_Task_Type.Ancillary_Fields).LastDiscriminant;
      end if;
   end Get_Task_Type_Last_Discriminant;

   --------------------------------------------------------------------------------

   function Get_Task_Type_First_Global_Variable
     (The_Task_Type : Type_Info_Ref;
      Abstraction   : Dictionary.Abstractions)
     return          Global_Variable_Info_Ref
   is
   begin
      if The_Task_Type = Null_Type_Info_Ref then
         return Null_Global_Variable_Info_Ref;
      else
         return GetTaskRef (The_Task_Type.Ancillary_Fields).First_Global_Variable (Abstraction);
      end if;
   end Get_Task_Type_First_Global_Variable;

   --------------------------------------------------------------------------------

   function Get_Task_Type_Last_Global_Variable
     (The_Task_Type : Type_Info_Ref;
      Abstraction   : Dictionary.Abstractions)
     return          Global_Variable_Info_Ref
   is
   begin
      if The_Task_Type = Null_Type_Info_Ref then
         return Null_Global_Variable_Info_Ref;
      else
         return GetTaskRef (The_Task_Type.Ancillary_Fields).Last_Global_Variable (Abstraction);
      end if;
   end Get_Task_Type_Last_Global_Variable;

   --------------------------------------------------------------------------------

   function Get_Task_Type_Has_Pragma (The_Task_Type : Type_Info_Ref;
                                      The_Pragma    : Dictionary.RavenscarPragmas) return Boolean is
   begin
      if The_Task_Type = Null_Type_Info_Ref then
         return False;
      else
         return GetTaskRef (The_Task_Type.Ancillary_Fields).Pragmas (The_Pragma).Given;
      end if;
   end Get_Task_Type_Has_Pragma;

   --------------------------------------------------------------------------------

   function Get_Task_Type_Pragma_Value
     (The_Task_Type : Type_Info_Ref;
      The_Pragma    : Dictionary.RavenscarPragmasWithValue)
     return          LexTokenManager.Lex_String
   is
   begin
      if The_Task_Type = Null_Type_Info_Ref then
         return LexTokenManager.Null_String;
      else
         return GetTaskRef (The_Task_Type.Ancillary_Fields).Pragmas (The_Pragma).Value;
      end if;
   end Get_Task_Type_Pragma_Value;

   --------------------------------------------------------------------------------

   function Get_Task_Type_First_Loop (The_Task_Type : Type_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Task_Type = Null_Type_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return GetTaskRef (The_Task_Type.Ancillary_Fields).First_Loop;
      end if;
   end Get_Task_Type_First_Loop;

   --------------------------------------------------------------------------------

   function Get_Task_Type_Last_Loop (The_Task_Type : Type_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Task_Type = Null_Type_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return GetTaskRef (The_Task_Type.Ancillary_Fields).Last_Loop;
      end if;
   end Get_Task_Type_Last_Loop;

   --------------------------------------------------------------------------------

   function Get_Task_Type_Suspends_List (The_Task_Type : Type_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Task_Type = Null_Type_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return GetTaskRef (The_Task_Type.Ancillary_Fields).Suspends_List;
      end if;
   end Get_Task_Type_Suspends_List;

   --------------------------------------------------------------------------------

   function Get_Task_Type_Body (The_Task_Type : Type_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Task_Type = Null_Type_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return GetTaskRef (The_Task_Type.Ancillary_Fields).Task_Body;
      end if;
   end Get_Task_Type_Body;

   --------------------------------------------------------------------------------

   function Get_Task_Type_Has_Proper_Body (The_Task_Type : Type_Info_Ref) return Boolean is
   begin
      if The_Task_Type = Null_Type_Info_Ref then
         return False;
      else
         return GetTaskRef (The_Task_Type.Ancillary_Fields).Has_Proper_Body;
      end if;
   end Get_Task_Type_Has_Proper_Body;

   --------------------------------------------------------------------------------

   function Get_Task_Type_Uses_Unprotected_Variables (The_Task_Type : Type_Info_Ref) return Boolean is
   begin
      if The_Task_Type = Null_Type_Info_Ref then
         return False;
      else
         return GetTaskRef (The_Task_Type.Ancillary_Fields).Uses_Unprotected_Variables;
      end if;
   end Get_Task_Type_Uses_Unprotected_Variables;

   --------------------------------------------------------------------------------

   function Get_Task_Type_Uses_Unchecked_Conversion (The_Task_Type : Type_Info_Ref) return Boolean is
   begin
      if The_Task_Type = Null_Type_Info_Ref then
         return False;
      else
         return GetTaskRef (The_Task_Type.Ancillary_Fields).Uses_Unchecked_Conversion;
      end if;
   end Get_Task_Type_Uses_Unchecked_Conversion;

   --------------------------------------------------------------------------------

   function Get_Task_Type_Assigns_From_External (The_Task_Type : Type_Info_Ref) return Boolean is
   begin
      if The_Task_Type = Null_Type_Info_Ref then
         return False;
      else
         return GetTaskRef (The_Task_Type.Ancillary_Fields).Assigns_From_External;
      end if;
   end Get_Task_Type_Assigns_From_External;

   --------------------------------------------------------------------------------

   function Get_Task_Type_Body_Is_Hidden (The_Task_Type : Type_Info_Ref) return Boolean is
   begin
      if The_Task_Type = Null_Type_Info_Ref then
         return False;
      else
         return GetTaskRef (The_Task_Type.Ancillary_Fields).Body_Is_Hidden;
      end if;
   end Get_Task_Type_Body_Is_Hidden;

   --------------------------------------------------------------------------------
   --  Constant_Info
   --------------------------------------------------------------------------------

   function Get_Constant_Info_Ref (Item : Dictionary.Symbol) return Constant_Info_Ref is
      function RefType_To_Constant_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type, Constant_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Constant_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Constant_Symbol then
            DiscriminantDebug ("Get_Constant_Info_Ref", GetSymbolDiscriminant (Item), Dictionary.Constant_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Constant_Info_Ref");
         end if;
         return RefType_To_Constant_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Constant_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Constant
     (Name            : in     LexTokenManager.Lex_String;
      Type_Mark       : in     Type_Info_Ref;
      Static          : in     Boolean;
      The_Declaration : in     Declaration_Info_Ref;
      Is_Deferred     : in     Boolean;
      Comp_Unit       : in     ContextManager.UnitDescriptors;
      Loc             : in     LexTokenManager.Token_Position;
      The_Constant    :    out Constant_Info_Ref) is
      The_Constant_Symbol : Dictionary.Symbol;

      function Constant_Info_Ref_To_RefType is new Unchecked_Conversion (Constant_Info_Ref, Dictionary.Ref_Type);
   begin
      if Is_Deferred then
         The_Constant :=
           new Constant_Info'
           (Self                         => Dictionary.NullSymbol,
            Name                         => Name,
            Static                       => Static,
            Type_Mark                    => Type_Mark,
            Value                        => LexTokenManager.Null_String,
            Exp_Is_Wellformed            => False,
            Exp_Node                     => 0,
            Associated_Generic_Parameter => Null_Generic_Parameter_Info_Ref,
            First_Rule_Policy            => Null_Rule_Policy_Info_Ref,
            Last_Rule_Policy             => Null_Rule_Policy_Info_Ref,
            Declaration                  => Null_Declaration_Info_Ref,
            Is_Deferred                  => True,
            Deferred_Declaration         => The_Declaration);
      else
         The_Constant :=
           new Constant_Info'
           (Self                         => Dictionary.NullSymbol,
            Name                         => Name,
            Static                       => Static,
            Type_Mark                    => Type_Mark,
            Value                        => LexTokenManager.Null_String,
            Exp_Is_Wellformed            => False,
            Exp_Node                     => 0,
            Associated_Generic_Parameter => Null_Generic_Parameter_Info_Ref,
            First_Rule_Policy            => Null_Rule_Policy_Info_Ref,
            Last_Rule_Policy             => Null_Rule_Policy_Info_Ref,
            Declaration                  => The_Declaration,
            Is_Deferred                  => False);
      end if;
      AddSymbol
        (Discriminant => Dictionary.Constant_Symbol,
         Ref          => Constant_Info_Ref_To_RefType (The_Constant),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Constant_Symbol);
      The_Constant.Self    := The_Constant_Symbol;
      if The_Declaration /= Null_Declaration_Info_Ref then
         The_Declaration.Item := The_Constant_Symbol;
      end if;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Constant");
   end Create_Constant;

   --------------------------------------------------------------------------------

   procedure Set_Constant_Value (The_Constant : in Constant_Info_Ref;
                                 Value        : in LexTokenManager.Lex_String) is
   begin
      The_Constant.Value := Value;
   end Set_Constant_Value;

   --------------------------------------------------------------------------------

   procedure Set_Constant_Exp_Node
     (The_Constant      : in Constant_Info_Ref;
      Exp_Is_Wellformed : in Boolean;
      Exp_Node          : in ExaminerConstants.RefType) is
   begin
      The_Constant.Exp_Node          := Exp_Node;
      The_Constant.Exp_Is_Wellformed := Exp_Is_Wellformed;
   end Set_Constant_Exp_Node;

   --------------------------------------------------------------------------------

   procedure Set_Constant_Static (The_Constant : in Constant_Info_Ref;
                                  Static       : in Boolean) is
   begin
      The_Constant.Static := Static;
   end Set_Constant_Static;

   --------------------------------------------------------------------------------

   procedure Set_Constant_Declaration (The_Constant    : in Constant_Info_Ref;
                                       The_Declaration : in Declaration_Info_Ref) is
   begin
      The_Constant.Declaration := The_Declaration;
      The_Declaration.Item     := The_Constant.Self; -- GAA
   end Set_Constant_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Constant_Deferred_Declaration (The_Constant    : in Constant_Info_Ref;
                                                The_Declaration : in Declaration_Info_Ref) is
   begin
      SystemErrors.RT_Assert
        (C       => The_Constant.Is_Deferred,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "Dictionary.RawDict.Set_Constant_Deferred_Declaration");
      The_Constant.Deferred_Declaration := The_Declaration;
   end Set_Constant_Deferred_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Constant_Associated_Generic_Parameter
     (The_Constant          : in Constant_Info_Ref;
      The_Generic_Parameter : in Generic_Parameter_Info_Ref) is
   begin
      The_Constant.Associated_Generic_Parameter := The_Generic_Parameter;
   end Set_Constant_Associated_Generic_Parameter;

   --------------------------------------------------------------------------------

   procedure Set_Constant_First_Rule_Policy (The_Constant    : in Constant_Info_Ref;
                                             The_Rule_Policy : in Rule_Policy_Info_Ref) is
   begin
      The_Constant.First_Rule_Policy := The_Rule_Policy;
   end Set_Constant_First_Rule_Policy;

   --------------------------------------------------------------------------------

   procedure Set_Constant_Last_Rule_Policy (The_Constant    : in Constant_Info_Ref;
                                            The_Rule_Policy : in Rule_Policy_Info_Ref) is
   begin
      The_Constant.Last_Rule_Policy := The_Rule_Policy;
   end Set_Constant_Last_Rule_Policy;

   --------------------------------------------------------------------------------

   function Get_Constant_Symbol (The_Constant : Constant_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Constant = Null_Constant_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Constant.Self;
      end if;
   end Get_Constant_Symbol;

   --------------------------------------------------------------------------------

   function Get_Constant_Name (The_Constant : Constant_Info_Ref) return LexTokenManager.Lex_String is
   begin
      if The_Constant = Null_Constant_Info_Ref then
         return LexTokenManager.Null_String;
      else
         return The_Constant.Name;
      end if;
   end Get_Constant_Name;

   --------------------------------------------------------------------------------

   function Get_Constant_Type (The_Constant : Constant_Info_Ref) return Type_Info_Ref is
   begin
      if The_Constant = Null_Constant_Info_Ref then
         return Null_Type_Info_Ref;
      else
         return The_Constant.Type_Mark;
      end if;
   end Get_Constant_Type;

   --------------------------------------------------------------------------------

   function Get_Constant_Value (The_Constant : Constant_Info_Ref) return LexTokenManager.Lex_String is
   begin
      if The_Constant = Null_Constant_Info_Ref then
         return LexTokenManager.Null_String;
      else
         return The_Constant.Value;
      end if;
   end Get_Constant_Value;

   --------------------------------------------------------------------------------

   function Get_Constant_First_Rule_Policy (The_Constant : Constant_Info_Ref) return Rule_Policy_Info_Ref is
   begin
      if The_Constant = Null_Constant_Info_Ref then
         return Null_Rule_Policy_Info_Ref;
      else
         return The_Constant.First_Rule_Policy;
      end if;
   end Get_Constant_First_Rule_Policy;

   --------------------------------------------------------------------------------

   function Get_Constant_Last_Rule_Policy (The_Constant : Constant_Info_Ref) return Rule_Policy_Info_Ref is
   begin
      if The_Constant = Null_Constant_Info_Ref then
         return Null_Rule_Policy_Info_Ref;
      else
         return The_Constant.Last_Rule_Policy;
      end if;
   end Get_Constant_Last_Rule_Policy;

   --------------------------------------------------------------------------------

   function Get_Constant_Exp_Node (The_Constant : Constant_Info_Ref) return ExaminerConstants.RefType is
   begin
      if The_Constant = Null_Constant_Info_Ref then
         return ExaminerConstants.RefType'First;
      else
         return The_Constant.Exp_Node;
      end if;
   end Get_Constant_Exp_Node;

   --------------------------------------------------------------------------------

   function Get_Constant_Exp_Is_Wellformed (The_Constant : Constant_Info_Ref) return Boolean is
   begin
      if The_Constant = Null_Constant_Info_Ref then
         return False;
      else
         return The_Constant.Exp_Is_Wellformed;
      end if;
   end Get_Constant_Exp_Is_Wellformed;

   --------------------------------------------------------------------------------

   function Get_Constant_Static (The_Constant : Constant_Info_Ref) return Boolean is
   begin
      if The_Constant = Null_Constant_Info_Ref then
         return False;
      else
         return The_Constant.Static;
      end if;
   end Get_Constant_Static;

   --------------------------------------------------------------------------------

   function Get_Constant_Declaration (The_Constant : Constant_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Constant = Null_Constant_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return The_Constant.Declaration;
      end if;
   end Get_Constant_Declaration;

   --------------------------------------------------------------------------------

   function Get_Constant_Deferred_Declaration (The_Constant : Constant_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Constant /= Null_Constant_Info_Ref and then The_Constant.Is_Deferred then
         return The_Constant.Deferred_Declaration;
      else
         return Null_Declaration_Info_Ref;
      end if;
   end Get_Constant_Deferred_Declaration;

   --------------------------------------------------------------------------------

   function Get_Constant_Associated_Generic_Parameter (The_Constant : Constant_Info_Ref) return Generic_Parameter_Info_Ref is
   begin
      if The_Constant = Null_Constant_Info_Ref then
         return Null_Generic_Parameter_Info_Ref;
      else
         return The_Constant.Associated_Generic_Parameter;
      end if;
   end Get_Constant_Associated_Generic_Parameter;

   --------------------------------------------------------------------------------
   --  Variable_Info
   --------------------------------------------------------------------------------

   function Get_Variable_Info_Ref (Item : Dictionary.Symbol) return Variable_Info_Ref is
      function RefType_To_Variable_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type, Variable_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Variable_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Variable_Symbol then
            DiscriminantDebug ("Get_Variable_Info_Ref", GetSymbolDiscriminant (Item), Dictionary.Variable_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Variable_Info_Ref");
         end if;
         return RefType_To_Variable_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Variable_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Variable
     (Name         : in     LexTokenManager.Lex_String;
      Comp_Unit    : in     ContextManager.UnitDescriptors;
      Loc          : in     LexTokenManager.Token_Position;
      The_Variable :    out Variable_Info_Ref) is
      The_Variable_Symbol : Dictionary.Symbol;

      function Variable_Info_Ref_To_RefType is new Unchecked_Conversion (Variable_Info_Ref, Dictionary.Ref_Type);
   begin
      The_Variable :=
        new Variable_Info'
        (Self               => Dictionary.NullSymbol,
         Name               => Name,
         Type_Mark          => Dictionary.Dict.Types.Unknown_Type_Mark,
         Abstract_Type_Mark => Dictionary.Dict.Types.Unknown_Type_Mark,
         Initialized        => False,
         Has_Address_Clause => False,
         Has_Pragma_Import  => False,
         Is_Aliased         => False,
         Marked_Valid       => True,
         Exp_Node           => 0,
         Declaration        => Null_Declaration_Info_Ref,
         Global_References  => Global_Variables_T'(Dictionary.Abstractions => Null_Global_Variable_Info_Ref),
         Own_Variable       => Null_Own_Variable_Info_Ref,
         Own_Task           => Dictionary.NullSymbol,
         Virtual_Element    => Dictionary.NullSymbol,
         Constituent        => Null_Constituent_Info_Ref,
         Subcomponents      => Null_Subcomponent_Info_Ref);
      AddSymbol
        (Discriminant => Dictionary.Variable_Symbol,
         Ref          => Variable_Info_Ref_To_RefType (The_Variable),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Variable_Symbol);
      The_Variable.Self := The_Variable_Symbol;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Variable");
   end Create_Variable;

   --------------------------------------------------------------------------------

   procedure Set_Variable_Type (The_Variable : in Variable_Info_Ref;
                                Type_Mark    : in Type_Info_Ref) is
   begin
      The_Variable.Type_Mark := Type_Mark;
   end Set_Variable_Type;

   --------------------------------------------------------------------------------

   procedure Set_Variable_Abstract_Type (The_Variable       : in Variable_Info_Ref;
                                         Abstract_Type_Mark : in Type_Info_Ref) is
   begin
      The_Variable.Abstract_Type_Mark := Abstract_Type_Mark;
   end Set_Variable_Abstract_Type;

   --------------------------------------------------------------------------------

   procedure Set_Variable_Initialized (The_Variable : in Variable_Info_Ref) is
   begin
      The_Variable.Initialized := True;
   end Set_Variable_Initialized;

   --------------------------------------------------------------------------------

   procedure Set_Variable_Has_Address_Clause (The_Variable : in Variable_Info_Ref) is
   begin
      The_Variable.Has_Address_Clause := True;
   end Set_Variable_Has_Address_Clause;

   --------------------------------------------------------------------------------

   procedure Set_Variable_Has_Pragma_Import (The_Variable : in Variable_Info_Ref) is
   begin
      The_Variable.Has_Pragma_Import := True;
   end Set_Variable_Has_Pragma_Import;

   --------------------------------------------------------------------------------

   procedure Set_Variable_Is_Aliased (The_Variable : in Variable_Info_Ref) is
   begin
      The_Variable.Is_Aliased := True;
   end Set_Variable_Is_Aliased;

   --------------------------------------------------------------------------------

   procedure Set_Variable_Marked_Valid (The_Variable : in Variable_Info_Ref;
                                        Val          : in Boolean) is
   begin
      The_Variable.Marked_Valid := Val;
   end Set_Variable_Marked_Valid;

   --------------------------------------------------------------------------------

   procedure Set_Variable_Declaration (The_Variable    : in Variable_Info_Ref;
                                       The_Declaration : in Declaration_Info_Ref) is
   begin
      The_Variable.Declaration := The_Declaration;
      The_Declaration.Item     := The_Variable.Self; -- GAA
   end Set_Variable_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Variable_Exp_Node (The_Variable : in Variable_Info_Ref;
                                    Exp_Node     : in ExaminerConstants.RefType) is
   begin
      The_Variable.Exp_Node := Exp_Node;
   end Set_Variable_Exp_Node;

   --------------------------------------------------------------------------------

   procedure Set_Variable_Own_Task (The_Variable : in Variable_Info_Ref;
                                    Own_Task     : in Dictionary.Symbol) is
   begin
      The_Variable.Own_Task := Own_Task;
   end Set_Variable_Own_Task;

   --------------------------------------------------------------------------------

   procedure Set_Variable_Virtual_Element (The_Variable    : in Variable_Info_Ref;
                                           Virtual_Element : in Dictionary.Symbol) is
   begin
      The_Variable.Virtual_Element := Virtual_Element;
   end Set_Variable_Virtual_Element;

   --------------------------------------------------------------------------------

   procedure Set_Variable_Global_References
     (The_Variable : in Variable_Info_Ref;
      Abstraction  : in Dictionary.Abstractions;
      Reference    : in Global_Variable_Info_Ref) is
   begin
      The_Variable.Global_References (Abstraction) := Reference;
   end Set_Variable_Global_References;

   --------------------------------------------------------------------------------

   procedure Set_Variable_Own_Variable (The_Variable : in Variable_Info_Ref;
                                        Own_Variable : in Own_Variable_Info_Ref) is
   begin
      The_Variable.Own_Variable := Own_Variable;
   end Set_Variable_Own_Variable;

   --------------------------------------------------------------------------------

   procedure Set_Variable_Constituent (The_Variable    : in Variable_Info_Ref;
                                       The_Constituent : in Constituent_Info_Ref) is
   begin
      The_Variable.Constituent := The_Constituent;
   end Set_Variable_Constituent;

   --------------------------------------------------------------------------------

   procedure Set_Variable_Subcomponents (The_Variable  : in Variable_Info_Ref;
                                         Subcomponents : in Subcomponent_Info_Ref) is
   begin
      The_Variable.Subcomponents := Subcomponents;
   end Set_Variable_Subcomponents;

   --------------------------------------------------------------------------------

   function Get_Variable_Symbol (The_Variable : Variable_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Variable = Null_Variable_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Variable.Self;
      end if;
   end Get_Variable_Symbol;

   --------------------------------------------------------------------------------

   function Get_Variable_Name (The_Variable : Variable_Info_Ref) return LexTokenManager.Lex_String is
   begin
      if The_Variable = Null_Variable_Info_Ref then
         return LexTokenManager.Null_String;
      else
         return The_Variable.Name;
      end if;
   end Get_Variable_Name;

   --------------------------------------------------------------------------------

   function Get_Variable_Type (The_Variable : Variable_Info_Ref) return Type_Info_Ref is
   begin
      if The_Variable = Null_Variable_Info_Ref then
         return Null_Type_Info_Ref;
      else
         return The_Variable.Type_Mark;
      end if;
   end Get_Variable_Type;

   --------------------------------------------------------------------------------

   function Get_Variable_Abstract_Type (The_Variable : Variable_Info_Ref) return Type_Info_Ref is
   begin
      if The_Variable = Null_Variable_Info_Ref then
         return Null_Type_Info_Ref;
      else
         return The_Variable.Abstract_Type_Mark;
      end if;
   end Get_Variable_Abstract_Type;

   --------------------------------------------------------------------------------

   function Get_Variable_Initialized (The_Variable : Variable_Info_Ref) return Boolean is
   begin
      if The_Variable = Null_Variable_Info_Ref then
         return False;
      else
         return The_Variable.Initialized;
      end if;
   end Get_Variable_Initialized;

   --------------------------------------------------------------------------------

   function Get_Variable_Has_Address_Clause (The_Variable : Variable_Info_Ref) return Boolean is
   begin
      if The_Variable = Null_Variable_Info_Ref then
         return False;
      else
         return The_Variable.Has_Address_Clause;
      end if;
   end Get_Variable_Has_Address_Clause;

   --------------------------------------------------------------------------------

   function Get_Variable_Has_Pragma_Import (The_Variable : Variable_Info_Ref) return Boolean is
   begin
      if The_Variable = Null_Variable_Info_Ref then
         return False;
      else
         return The_Variable.Has_Pragma_Import;
      end if;
   end Get_Variable_Has_Pragma_Import;

   --------------------------------------------------------------------------------

   function Get_Variable_Is_Aliased (The_Variable : Variable_Info_Ref) return Boolean is
   begin
      if The_Variable = Null_Variable_Info_Ref then
         return False;
      else
         return The_Variable.Is_Aliased;
      end if;
   end Get_Variable_Is_Aliased;

   --------------------------------------------------------------------------------

   function Get_Variable_Marked_Valid (The_Variable : Variable_Info_Ref) return Boolean is
   begin
      if The_Variable = Null_Variable_Info_Ref then
         return False;
      else
         return The_Variable.Marked_Valid;
      end if;
   end Get_Variable_Marked_Valid;

   --------------------------------------------------------------------------------

   function Get_Variable_Declaration (The_Variable : Variable_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Variable = Null_Variable_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return The_Variable.Declaration;
      end if;
   end Get_Variable_Declaration;

   --------------------------------------------------------------------------------

   function Get_Variable_Exp_Node (The_Variable : Variable_Info_Ref) return ExaminerConstants.RefType is
   begin
      if The_Variable = Null_Variable_Info_Ref then
         return ExaminerConstants.RefType'First;
      else
         return The_Variable.Exp_Node;
      end if;
   end Get_Variable_Exp_Node;

   --------------------------------------------------------------------------------

   function Get_Variable_Global_References
     (The_Variable : Variable_Info_Ref;
      Abstraction  : Dictionary.Abstractions)
     return         Global_Variable_Info_Ref
   is
   begin
      if The_Variable = Null_Variable_Info_Ref then
         return Null_Global_Variable_Info_Ref;
      else
         return The_Variable.Global_References (Abstraction);
      end if;
   end Get_Variable_Global_References;

   --------------------------------------------------------------------------------

   function Get_Variable_Own_Variable (The_Variable : Variable_Info_Ref) return Own_Variable_Info_Ref is
   begin
      if The_Variable = Null_Variable_Info_Ref then
         return Null_Own_Variable_Info_Ref;
      else
         return The_Variable.Own_Variable;
      end if;
   end Get_Variable_Own_Variable;

   --------------------------------------------------------------------------------

   function Get_Variable_Own_Task (The_Variable : Variable_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Variable = Null_Variable_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Variable.Own_Task;
      end if;
   end Get_Variable_Own_Task;

   --------------------------------------------------------------------------------

   function Get_Variable_Constituent (The_Variable : Variable_Info_Ref) return Constituent_Info_Ref is
   begin
      if The_Variable = Null_Variable_Info_Ref then
         return Null_Constituent_Info_Ref;
      else
         return The_Variable.Constituent;
      end if;
   end Get_Variable_Constituent;

   --------------------------------------------------------------------------------

   function Get_Variable_Subcomponents (The_Variable : Variable_Info_Ref) return Subcomponent_Info_Ref is
   begin
      if The_Variable = Null_Variable_Info_Ref then
         return Null_Subcomponent_Info_Ref;
      else
         return The_Variable.Subcomponents;
      end if;
   end Get_Variable_Subcomponents;

   --------------------------------------------------------------------------------

   function Get_Variable_Virtual_Element (The_Variable : Variable_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Variable = Null_Variable_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Variable.Virtual_Element;
      end if;
   end Get_Variable_Virtual_Element;

   --------------------------------------------------------------------------------
   --  Global_Variable_Info
   --------------------------------------------------------------------------------

   function Get_Global_Variable_Info_Ref (Item : Dictionary.Symbol) return Global_Variable_Info_Ref is
      function RefType_To_Global_Variable_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type, Global_Variable_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Global_Variable_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Global_Variable_Symbol then
            DiscriminantDebug ("Get_Global_Variable_info_Ref", GetSymbolDiscriminant (Item), Dictionary.Global_Variable_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Global_Variable_info_Ref");
         end if;
         return RefType_To_Global_Variable_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Global_Variable_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Global_Variable
     (Mode                 : in     Dictionary.Modes;
      Prefix_Needed        : in     Boolean;
      The_Subprogram       : in     Subprogram_Info_Ref;
      The_Task_Type        : in     Type_Info_Ref;
      Subprogram_Parameter : in     Subprogram_Parameter_Info_Ref;
      Variable             : in     Variable_Info_Ref;
      Next_Subprogram      : in     Global_Variable_Info_Ref;
      Comp_Unit            : in     ContextManager.UnitDescriptors;
      Loc                  : in     LexTokenManager.Token_Position;
      The_Global_Variable  :    out Global_Variable_Info_Ref) is
      The_Global_Variable_Symbol : Dictionary.Symbol;

      function Global_Variable_Info_Ref_To_RefType is new Unchecked_Conversion (Global_Variable_Info_Ref, Dictionary.Ref_Type);
   begin
      if The_Subprogram /= Null_Subprogram_Info_Ref then
         if Subprogram_Parameter /= Null_Subprogram_Parameter_Info_Ref then
            The_Global_Variable :=
              new Global_Variable_Info'
              (Self                      => Dictionary.NullSymbol,
               Mode                      => Mode,
               Exported                  => False,
               Imported                  => False,
               Prefix_Needed             => Prefix_Needed,
               Dependencies              => Dependencies_T'(Dictionary.Abstractions => Null_Dependency_Info_Ref),
               Next_Subprogram           => Next_Subprogram,
               Next_Variable             => Null_Global_Variable_Info_Ref,
               Kind_Of_Global_Variable   => Subprogram_Parameter_Item,
               Subprogram_With_Parameter => The_Subprogram,
               Subprogram_Parameter      => Subprogram_Parameter);
         elsif Variable /= Null_Variable_Info_Ref then
            The_Global_Variable :=
              new Global_Variable_Info'
              (Self                     => Dictionary.NullSymbol,
               Mode                     => Mode,
               Exported                 => False,
               Imported                 => False,
               Prefix_Needed            => Prefix_Needed,
               Dependencies             => Dependencies_T'(Dictionary.Abstractions => Null_Dependency_Info_Ref),
               Next_Subprogram          => Next_Subprogram,
               Next_Variable            => Null_Global_Variable_Info_Ref,
               Kind_Of_Global_Variable  => Subprogram_Variable_Item,
               Subprogram_With_Variable => The_Subprogram,
               Variable                 => Variable);
         else
            The_Global_Variable := Null_Global_Variable_Info_Ref;
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Create_Global_Variable");
         end if;
      elsif The_Task_Type /= Null_Type_Info_Ref and then Variable /= Null_Variable_Info_Ref then
         The_Global_Variable :=
           new Global_Variable_Info'
           (Self                    => Dictionary.NullSymbol,
            Mode                    => Mode,
            Exported                => False,
            Imported                => False,
            Prefix_Needed           => Prefix_Needed,
            Dependencies            => Dependencies_T'(Dictionary.Abstractions => Null_Dependency_Info_Ref),
            Next_Subprogram         => Next_Subprogram,
            Next_Variable           => Null_Global_Variable_Info_Ref,
            Kind_Of_Global_Variable => Task_Type_Variable_Item,
            Task_Type               => The_Task_Type,
            Task_Type_Variable      => Variable);
      else
         The_Global_Variable := Null_Global_Variable_Info_Ref;
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Symbol_Table,
            Msg     => "Dictionary.RawDict.Create_Global_Variable");
      end if;
      AddSymbol
        (Discriminant => Dictionary.Global_Variable_Symbol,
         Ref          => Global_Variable_Info_Ref_To_RefType (The_Global_Variable),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Global_Variable_Symbol);
      The_Global_Variable.Self := The_Global_Variable_Symbol;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Global_Variable");
   end Create_Global_Variable;

   --------------------------------------------------------------------------------

   procedure Set_Global_Variable_Exported (The_Global_Variable : in Global_Variable_Info_Ref) is
   begin
      The_Global_Variable.Exported := True;
   end Set_Global_Variable_Exported;

   --------------------------------------------------------------------------------

   procedure Set_Global_Variable_Imported (The_Global_Variable : in Global_Variable_Info_Ref) is
   begin
      The_Global_Variable.Imported := True;
   end Set_Global_Variable_Imported;

   --------------------------------------------------------------------------------

   procedure Set_Global_Variable_Dependencies
     (The_Global_Variable : in Global_Variable_Info_Ref;
      Abstraction         : in Dictionary.Abstractions;
      Dependency          : in Dependency_Info_Ref) is
   begin
      The_Global_Variable.Dependencies (Abstraction) := Dependency;
   end Set_Global_Variable_Dependencies;

   --------------------------------------------------------------------------------

   procedure Set_Next_Global_Variable (The_Global_Variable, Next : in Global_Variable_Info_Ref) is
   begin
      The_Global_Variable.Next_Variable := Next;
   end Set_Next_Global_Variable;

   --------------------------------------------------------------------------------

   function Get_Global_Variable_Symbol (The_Global_Variable : Global_Variable_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Global_Variable = Null_Global_Variable_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Global_Variable.Self;
      end if;
   end Get_Global_Variable_Symbol;

   --------------------------------------------------------------------------------

   function Get_Global_Variable_Mode (The_Global_Variable : Global_Variable_Info_Ref) return Dictionary.Modes is
   begin
      if The_Global_Variable = Null_Global_Variable_Info_Ref then
         return Dictionary.InvalidMode;
      else
         return The_Global_Variable.Mode;
      end if;
   end Get_Global_Variable_Mode;

   --------------------------------------------------------------------------------

   function Get_Global_Variable_Exported (The_Global_Variable : Global_Variable_Info_Ref) return Boolean is
   begin
      if The_Global_Variable = Null_Global_Variable_Info_Ref then
         return False;
      else
         return The_Global_Variable.Exported;
      end if;
   end Get_Global_Variable_Exported;

   --------------------------------------------------------------------------------

   function Get_Global_Variable_Imported (The_Global_Variable : Global_Variable_Info_Ref) return Boolean is
   begin
      if The_Global_Variable = Null_Global_Variable_Info_Ref then
         return False;
      else
         return The_Global_Variable.Imported;
      end if;
   end Get_Global_Variable_Imported;

   --------------------------------------------------------------------------------

   function Get_Global_Variable_Prefix_Needed (The_Global_Variable : Global_Variable_Info_Ref) return Boolean is
   begin
      if The_Global_Variable = Null_Global_Variable_Info_Ref then
         return False;
      else
         return The_Global_Variable.Prefix_Needed;
      end if;
   end Get_Global_Variable_Prefix_Needed;

   --------------------------------------------------------------------------------

   function Get_Global_Variable_Dependencies
     (The_Global_Variable : Global_Variable_Info_Ref;
      Abstraction         : Dictionary.Abstractions)
     return                Dependency_Info_Ref
   is
   begin
      if The_Global_Variable = Null_Global_Variable_Info_Ref then
         return Null_Dependency_Info_Ref;
      else
         return The_Global_Variable.Dependencies (Abstraction);
      end if;
   end Get_Global_Variable_Dependencies;

   --------------------------------------------------------------------------------

   function Get_Global_Variable_Next_Subprogram (The_Global_Variable : Global_Variable_Info_Ref) return Global_Variable_Info_Ref is
   begin
      if The_Global_Variable = Null_Global_Variable_Info_Ref then
         return Null_Global_Variable_Info_Ref;
      else
         return The_Global_Variable.Next_Subprogram;
      end if;
   end Get_Global_Variable_Next_Subprogram;

   --------------------------------------------------------------------------------

   function Get_Global_Variable_Subprogram (The_Global_Variable : Global_Variable_Info_Ref) return Subprogram_Info_Ref is
   begin
      if The_Global_Variable = Null_Global_Variable_Info_Ref then
         return Null_Subprogram_Info_Ref;
      else
         case The_Global_Variable.Kind_Of_Global_Variable is
            when Subprogram_Parameter_Item =>
               return The_Global_Variable.Subprogram_With_Parameter;
            when Subprogram_Variable_Item =>
               return The_Global_Variable.Subprogram_With_Variable;
            when others => -- non-exec code
               SystemErrors.Fatal_Error
                 (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                  Msg     => "Dictionary.RawDict.Get_Global_Variable_Subprogram");
               return Null_Subprogram_Info_Ref;
         end case;
      end if;
   end Get_Global_Variable_Subprogram;

   --------------------------------------------------------------------------------

   function Get_Global_Variable_Task_Type (The_Global_Variable : Global_Variable_Info_Ref) return Type_Info_Ref is
   begin
      if The_Global_Variable = Null_Global_Variable_Info_Ref then
         return Null_Type_Info_Ref;
      else
         case The_Global_Variable.Kind_Of_Global_Variable is
            when Task_Type_Variable_Item =>
               return The_Global_Variable.Task_Type;
            when others => -- non-exec code
               SystemErrors.Fatal_Error
                 (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                  Msg     => "Dictionary.RawDict.Get_Global_Variable_Task_Type");
               return Null_Type_Info_Ref;
         end case;
      end if;
   end Get_Global_Variable_Task_Type;

   --------------------------------------------------------------------------------

   function Get_Kind_Of_Global_Variable (The_Global_Variable : Global_Variable_Info_Ref) return Kind_Of_Global_Variable_T is
   begin
      return The_Global_Variable.Kind_Of_Global_Variable;
   end Get_Kind_Of_Global_Variable;

   --------------------------------------------------------------------------------

   function Get_Global_Variable_Variable (The_Global_Variable : Global_Variable_Info_Ref) return Variable_Info_Ref is
   begin
      if The_Global_Variable = Null_Global_Variable_Info_Ref then
         return Null_Variable_Info_Ref;
      else
         case The_Global_Variable.Kind_Of_Global_Variable is
            when Subprogram_Variable_Item =>
               return The_Global_Variable.Variable;
            when Task_Type_Variable_Item =>
               return The_Global_Variable.Task_Type_Variable;
            when others => -- non-exec code
               SystemErrors.Fatal_Error
                 (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                  Msg     => "Dictionary.RawDict.Get_Global_Variable_Variable");
               return Null_Variable_Info_Ref;
         end case;
      end if;
   end Get_Global_Variable_Variable;

   --------------------------------------------------------------------------------

   function Get_Global_Variable_Parameter (The_Global_Variable : Global_Variable_Info_Ref) return Subprogram_Parameter_Info_Ref is
   begin
      if The_Global_Variable = Null_Global_Variable_Info_Ref then
         return Null_Subprogram_Parameter_Info_Ref;
      else
         case The_Global_Variable.Kind_Of_Global_Variable is
            when Subprogram_Parameter_Item =>
               return The_Global_Variable.Subprogram_Parameter;
            when others => -- non-exec code
               SystemErrors.Fatal_Error
                 (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                  Msg     => "Dictionary.RawDict.Get_Global_Variable_Parameter");
               return Null_Subprogram_Parameter_Info_Ref;
         end case;
      end if;
   end Get_Global_Variable_Parameter;

   --------------------------------------------------------------------------------

   function Get_Next_Global_Variable (The_Global_Variable : Global_Variable_Info_Ref) return Global_Variable_Info_Ref is
   begin
      if The_Global_Variable = Null_Global_Variable_Info_Ref then
         return Null_Global_Variable_Info_Ref;
      else
         return The_Global_Variable.Next_Variable;
      end if;
   end Get_Next_Global_Variable;

   --------------------------------------------------------------------------------
   --  Own_Variable_Info
   --------------------------------------------------------------------------------

   function Get_Own_Variable_Info_Ref (Item : Dictionary.Symbol) return Own_Variable_Info_Ref is
      function RefType_To_Own_Variable_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type, Own_Variable_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Own_Variable_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Own_Variable_Symbol then
            DiscriminantDebug ("Get_Own_Variable_Info_Ref", GetSymbolDiscriminant (Item), Dictionary.Own_Variable_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Own_Variable_Info_Ref");
         end if;
         return RefType_To_Own_Variable_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Own_Variable_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Own_Variable
     (Variable         : in     Variable_Info_Ref;
      Owner            : in     Dictionary.Symbol;
      Comp_Unit        : in     ContextManager.UnitDescriptors;
      Loc              : in     LexTokenManager.Token_Position;
      The_Own_Variable :    out Own_Variable_Info_Ref) is
      The_Own_Variable_Symbol : Dictionary.Symbol;

      function Own_Variable_Info_Ref_To_RefType is new Unchecked_Conversion (Own_Variable_Info_Ref, Dictionary.Ref_Type);
   begin
      -- note use of InvalidMode in this aggregate.  The reasosn is that if we
      -- refine an own variable on to an embdedded package own variable then the
      -- own variable gets created by a call here when the refinement is processed.
      -- If there is an error when the embedded package own variable is eventually
      -- processed then the mode (set earlier to invalid) won't get overwitten so
      -- we are left with a nice detectable error case rather than an apaprently
      -- well ofrmed case of DefaultMode
      The_Own_Variable :=
        new Own_Variable_Info'
        (Self                      => Dictionary.NullSymbol,
         Variable                  => Variable,
         Owner                     => Owner,
         Announced                 => False,
         Typed                     => False,
         Initialized               => False,
         Mode                      => Dictionary.InvalidMode,
         Is_Protected              => False,
         Is_Interrupt_Stream       => False,
         Interrupt_Stream_Mappings => Dictionary.NullSymbol,
         Unprotected_Reference     => Dictionary.NullSymbol,
         Suspends_Reference        => Dictionary.NullSymbol,
         Implicit_In_Stream        => Null_Implicit_In_Stream_Info_Ref,
         Priority                  => LexTokenManager.Null_String,
         Integrity                 => LexTokenManager.Null_String,
         Suspendable               => False,
         Interruptable             => False,
         Constituents              => Null_Constituent_Info_Ref,
         Next                      => Null_Own_Variable_Info_Ref);
      AddSymbol
        (Discriminant => Dictionary.Own_Variable_Symbol,
         Ref          => Own_Variable_Info_Ref_To_RefType (The_Own_Variable),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Own_Variable_Symbol);
      The_Own_Variable.Self := The_Own_Variable_Symbol;
      Variable.Own_Variable := The_Own_Variable;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Own_Variable");
   end Create_Own_Variable;

   --------------------------------------------------------------------------------

   procedure Set_Own_Variable_Announced (The_Own_Variable : in Own_Variable_Info_Ref) is
   begin
      The_Own_Variable.Announced := True;
   end Set_Own_Variable_Announced;

   --------------------------------------------------------------------------------

   procedure Set_Own_Variable_Typed (The_Own_Variable : in Own_Variable_Info_Ref) is
   begin
      The_Own_Variable.Typed := True;
   end Set_Own_Variable_Typed;

   --------------------------------------------------------------------------------

   procedure Set_Own_Variable_Initialized (The_Own_Variable : in Own_Variable_Info_Ref) is
   begin
      The_Own_Variable.Initialized := True;
   end Set_Own_Variable_Initialized;

   --------------------------------------------------------------------------------

   procedure Set_Own_Variable_Constituents
     (The_Own_Variable : in Own_Variable_Info_Ref;
      The_Constituent  : in Constituent_Info_Ref) is
   begin
      The_Own_Variable.Constituents := The_Constituent;
   end Set_Own_Variable_Constituents;

   --------------------------------------------------------------------------------

   procedure Set_Own_Variable_Mode (The_Own_Variable : in Own_Variable_Info_Ref;
                                    Mode             : in Dictionary.Modes) is
   begin
      The_Own_Variable.Mode := Mode;
   end Set_Own_Variable_Mode;

   --------------------------------------------------------------------------------

   procedure Set_Own_Variable_Protected (The_Own_Variable : in Own_Variable_Info_Ref;
                                         Is_Protected     : in Boolean) is
   begin
      The_Own_Variable.Is_Protected := Is_Protected;
   end Set_Own_Variable_Protected;

   --------------------------------------------------------------------------------

   procedure Set_Next_Own_Variable (The_Own_Variable, Next : in Own_Variable_Info_Ref) is
   begin
      The_Own_Variable.Next := Next;
   end Set_Next_Own_Variable;

   --------------------------------------------------------------------------------

   procedure Set_Own_Variable_Interrupt_Stream_Mappings
     (The_Own_Variable              : in Own_Variable_Info_Ref;
      The_Interrupt_Stream_Mappings : in Dictionary.Symbol) is
   begin
      The_Own_Variable.Interrupt_Stream_Mappings := The_Interrupt_Stream_Mappings;
   end Set_Own_Variable_Interrupt_Stream_Mappings;

   --------------------------------------------------------------------------------

   procedure Set_Own_Variable_Implicit_In_Stream
     (The_Own_Variable       : in Own_Variable_Info_Ref;
      The_Implicit_In_Stream : in Implicit_In_Stream_Info_Ref) is
   begin
      The_Own_Variable.Implicit_In_Stream := The_Implicit_In_Stream;
   end Set_Own_Variable_Implicit_In_Stream;

   --------------------------------------------------------------------------------

   procedure Set_Own_Variable_Is_Interrupt_Stream
     (The_Own_Variable    : in Own_Variable_Info_Ref;
      Is_Interrupt_Stream : in Boolean) is
   begin
      The_Own_Variable.Is_Interrupt_Stream := Is_Interrupt_Stream;
   end Set_Own_Variable_Is_Interrupt_Stream;

   --------------------------------------------------------------------------------

   procedure Set_Own_Variable_Unprotected_Reference
     (The_Own_Variable : in Own_Variable_Info_Ref;
      By_Thread        : in Dictionary.Symbol) is
   begin
      The_Own_Variable.Unprotected_Reference := By_Thread;
   end Set_Own_Variable_Unprotected_Reference;

   --------------------------------------------------------------------------------

   procedure Set_Own_Variable_Suspends_Reference (The_Own_Variable : in Own_Variable_Info_Ref;
                                                  By_Thread        : in Dictionary.Symbol) is
   begin
      The_Own_Variable.Suspends_Reference := By_Thread;
   end Set_Own_Variable_Suspends_Reference;

   --------------------------------------------------------------------------------

   procedure Set_Own_Variable_Is_Suspendable (The_Own_Variable : in Own_Variable_Info_Ref) is
   begin
      The_Own_Variable.Suspendable := True;
   end Set_Own_Variable_Is_Suspendable;

   --------------------------------------------------------------------------------

   procedure Set_Own_Variable_Has_Interrupt_Property (The_Own_Variable : in Own_Variable_Info_Ref) is
   begin
      The_Own_Variable.Interruptable := True;
   end Set_Own_Variable_Has_Interrupt_Property;

   --------------------------------------------------------------------------------

   procedure Set_Own_Variable_Priority_Property
     (The_Own_Variable : in Own_Variable_Info_Ref;
      The_Value        : in LexTokenManager.Lex_String) is
   begin
      The_Own_Variable.Priority := The_Value;
   end Set_Own_Variable_Priority_Property;

   --------------------------------------------------------------------------------

   procedure Set_Own_Variable_Integrity_Property
     (The_Own_Variable : in Own_Variable_Info_Ref;
      The_Value        : in LexTokenManager.Lex_String) is
   begin
      The_Own_Variable.Integrity := The_Value;
   end Set_Own_Variable_Integrity_Property;

   --------------------------------------------------------------------------------

   function Get_Own_Variable_Symbol (The_Own_Variable : Own_Variable_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Own_Variable = Null_Own_Variable_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Own_Variable.Self;
      end if;
   end Get_Own_Variable_Symbol;

   --------------------------------------------------------------------------------

   function Get_Own_Variable_Variable (The_Own_Variable : Own_Variable_Info_Ref) return Variable_Info_Ref is
   begin
      if The_Own_Variable = Null_Own_Variable_Info_Ref then
         return Null_Variable_Info_Ref;
      else
         return The_Own_Variable.Variable;
      end if;
   end Get_Own_Variable_Variable;

   --------------------------------------------------------------------------------

   function Get_Own_Variable_Owner (The_Own_Variable : Own_Variable_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Own_Variable = Null_Own_Variable_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Own_Variable.Owner;
      end if;
   end Get_Own_Variable_Owner;

   --------------------------------------------------------------------------------

   function Get_Own_Variable_Announced (The_Own_Variable : Own_Variable_Info_Ref) return Boolean is
   begin
      if The_Own_Variable = Null_Own_Variable_Info_Ref then
         return False;
      else
         return The_Own_Variable.Announced;
      end if;
   end Get_Own_Variable_Announced;

   --------------------------------------------------------------------------------

   function Get_Own_Variable_Typed (The_Own_Variable : Own_Variable_Info_Ref) return Boolean is
   begin
      if The_Own_Variable = Null_Own_Variable_Info_Ref then
         return False;
      else
         return The_Own_Variable.Typed;
      end if;
   end Get_Own_Variable_Typed;

   --------------------------------------------------------------------------------

   function Get_Own_Variable_Initialized (The_Own_Variable : Own_Variable_Info_Ref) return Boolean is
   begin
      if The_Own_Variable = Null_Own_Variable_Info_Ref then
         return False;
      else
         return The_Own_Variable.Initialized;
      end if;
   end Get_Own_Variable_Initialized;

   --------------------------------------------------------------------------------

   function Get_Own_Variable_Constituents (The_Own_Variable : Own_Variable_Info_Ref) return Constituent_Info_Ref is
   begin
      if The_Own_Variable = Null_Own_Variable_Info_Ref then
         return Null_Constituent_Info_Ref;
      else
         return The_Own_Variable.Constituents;
      end if;
   end Get_Own_Variable_Constituents;

   --------------------------------------------------------------------------------

   function Get_Own_Variable_Mode (The_Own_Variable : Own_Variable_Info_Ref) return Dictionary.Modes is
   begin
      if The_Own_Variable = Null_Own_Variable_Info_Ref then
         return Dictionary.InvalidMode;
      else
         return The_Own_Variable.Mode;
      end if;
   end Get_Own_Variable_Mode;

   --------------------------------------------------------------------------------

   function Get_Own_Variable_Protected (The_Own_Variable : Own_Variable_Info_Ref) return Boolean is
   begin
      if The_Own_Variable = Null_Own_Variable_Info_Ref then
         return False;
      else
         return The_Own_Variable.Is_Protected;
      end if;
   end Get_Own_Variable_Protected;

   --------------------------------------------------------------------------------

   function Get_Own_Variable_Implicit_In_Stream (The_Own_Variable : Own_Variable_Info_Ref) return Implicit_In_Stream_Info_Ref is
   begin
      if The_Own_Variable = Null_Own_Variable_Info_Ref then
         return Null_Implicit_In_Stream_Info_Ref;
      else
         return The_Own_Variable.Implicit_In_Stream;
      end if;
   end Get_Own_Variable_Implicit_In_Stream;

   --------------------------------------------------------------------------------

   function Get_Next_Own_Variable (The_Own_Variable : Own_Variable_Info_Ref) return Own_Variable_Info_Ref is
   begin
      if The_Own_Variable = Null_Own_Variable_Info_Ref then
         return Null_Own_Variable_Info_Ref;
      else
         return The_Own_Variable.Next;
      end if;
   end Get_Next_Own_Variable;

   --------------------------------------------------------------------------------

   function Get_Own_Variable_Interrupt_Stream_Mappings (The_Own_Variable : Own_Variable_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Own_Variable = Null_Own_Variable_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Own_Variable.Interrupt_Stream_Mappings;
      end if;
   end Get_Own_Variable_Interrupt_Stream_Mappings;

   --------------------------------------------------------------------------------

   function Get_Own_Variable_Is_Interrupt_Stream (The_Own_Variable : Own_Variable_Info_Ref) return Boolean is
   begin
      if The_Own_Variable = Null_Own_Variable_Info_Ref then
         return False;
      else
         return The_Own_Variable.Is_Interrupt_Stream;
      end if;
   end Get_Own_Variable_Is_Interrupt_Stream;

   --------------------------------------------------------------------------------

   function Get_Own_Variable_Is_Suspendable (The_Own_Variable : Own_Variable_Info_Ref) return Boolean is
   begin
      if The_Own_Variable = Null_Own_Variable_Info_Ref then
         return False;
      else
         return The_Own_Variable.Suspendable;
      end if;
   end Get_Own_Variable_Is_Suspendable;

   --------------------------------------------------------------------------------

   function Get_Own_Variable_Has_Interrupt_Property (The_Own_Variable : Own_Variable_Info_Ref) return Boolean is
   begin
      if The_Own_Variable = Null_Own_Variable_Info_Ref then
         return False;
      else
         return The_Own_Variable.Interruptable;
      end if;
   end Get_Own_Variable_Has_Interrupt_Property;

   --------------------------------------------------------------------------------

   function Get_Own_Variable_Priority_Property (The_Own_Variable : Own_Variable_Info_Ref) return LexTokenManager.Lex_String is
   begin
      if The_Own_Variable = Null_Own_Variable_Info_Ref then
         return LexTokenManager.Null_String;
      else
         return The_Own_Variable.Priority;
      end if;
   end Get_Own_Variable_Priority_Property;

   --------------------------------------------------------------------------------

   function Get_Own_Variable_Integrity_Property (The_Own_Variable : Own_Variable_Info_Ref) return LexTokenManager.Lex_String is
   begin
      if The_Own_Variable = Null_Own_Variable_Info_Ref then
         return LexTokenManager.Null_String;
      else
         return The_Own_Variable.Integrity;
      end if;
   end Get_Own_Variable_Integrity_Property;

   --------------------------------------------------------------------------------

   function Get_Own_Variable_Unprotected_Reference (The_Own_Variable : Own_Variable_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Own_Variable = Null_Own_Variable_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Own_Variable.Unprotected_Reference;
      end if;
   end Get_Own_Variable_Unprotected_Reference;

   --------------------------------------------------------------------------------

   function Get_Own_Variable_Suspends_Reference (The_Own_Variable : Own_Variable_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Own_Variable = Null_Own_Variable_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Own_Variable.Suspends_Reference;
      end if;
   end Get_Own_Variable_Suspends_Reference;

   --------------------------------------------------------------------------------

   function Get_Own_Variable_Has_Valid_Priority_Property (The_Own_Variable : Own_Variable_Info_Ref) return Boolean is
   begin
      if The_Own_Variable = Null_Own_Variable_Info_Ref then
         return False;
      else
         return LexTokenManager.Lex_String_Case_Insensitive_Compare
           (Lex_Str1 => Get_Own_Variable_Priority_Property (The_Own_Variable => The_Own_Variable),
            Lex_Str2 => LexTokenManager.Null_String) /=
           LexTokenManager.Str_Eq;
      end if;
   end Get_Own_Variable_Has_Valid_Priority_Property;

   --------------------------------------------------------------------------------
   --  Quantified_Variable_Info
   --------------------------------------------------------------------------------

   function Get_Quantified_Variable_Info_Ref (Item : Dictionary.Symbol) return Quantified_Variable_Info_Ref is
      function RefType_To_Quantified_Variable_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type,
                                                                                    Quantified_Variable_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Quantified_Variable_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Quantified_Variable_Symbol then
            DiscriminantDebug
              ("Get_Quantified_Variable_Info_Ref",
               GetSymbolDiscriminant (Item),
               Dictionary.Quantified_Variable_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Quantified_Variable_Info_Ref");
         end if;
         return RefType_To_Quantified_Variable_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Quantified_Variable_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Quantified_Variable
     (Name                     : in     LexTokenManager.Lex_String;
      Type_Mark                : in     Type_Info_Ref;
      The_Parameter_Constraint : in     Parameter_Constraint_Info_Ref;
      Region                   : in     Dictionary.Symbol;
      Comp_Unit                : in     ContextManager.UnitDescriptors;
      Loc                      : in     LexTokenManager.Token_Position;
      The_Quantified_Variable  :    out Quantified_Variable_Info_Ref) is
      The_Quantified_Variable_Symbol : Dictionary.Symbol;

      function Quantified_Variable_Info_Ref_To_RefType is new Unchecked_Conversion (Quantified_Variable_Info_Ref,
                                                                                    Dictionary.Ref_Type);
   begin
      The_Quantified_Variable :=
        new Quantified_Variable_Info'
        (Self                 => Dictionary.NullSymbol,
         Name                 => Name,
         Type_Mark            => Type_Mark,
         Parameter_Constraint => The_Parameter_Constraint,
         Region               => Region);
      AddSymbol
        (Discriminant => Dictionary.Quantified_Variable_Symbol,
         Ref          => Quantified_Variable_Info_Ref_To_RefType (The_Quantified_Variable),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Quantified_Variable_Symbol);
      The_Quantified_Variable.Self := The_Quantified_Variable_Symbol;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Quantified_Variable");
   end Create_Quantified_Variable;

   --------------------------------------------------------------------------------

   function Get_Quantified_Variable_Symbol (The_Quantified_Variable : Quantified_Variable_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Quantified_Variable = Null_Quantified_Variable_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Quantified_Variable.Self;
      end if;
   end Get_Quantified_Variable_Symbol;

   --------------------------------------------------------------------------------

   function Get_Quantified_Variable_Name
     (The_Quantified_Variable : Quantified_Variable_Info_Ref)
     return                    LexTokenManager.Lex_String
   is
   begin
      if The_Quantified_Variable = Null_Quantified_Variable_Info_Ref then
         return LexTokenManager.Null_String;
      else
         return The_Quantified_Variable.Name;
      end if;
   end Get_Quantified_Variable_Name;

   --------------------------------------------------------------------------------

   function Get_Quantified_Variable_Type (The_Quantified_Variable : Quantified_Variable_Info_Ref) return Type_Info_Ref is
   begin
      if The_Quantified_Variable = Null_Quantified_Variable_Info_Ref then
         return Null_Type_Info_Ref;
      else
         return The_Quantified_Variable.Type_Mark;
      end if;
   end Get_Quantified_Variable_Type;

   --------------------------------------------------------------------------------

   function Get_Quantified_Variable_Constraint
     (The_Quantified_Variable : Quantified_Variable_Info_Ref)
     return                    Parameter_Constraint_Info_Ref
   is
   begin
      if The_Quantified_Variable = Null_Quantified_Variable_Info_Ref then
         return Null_Parameter_Constraint_Info_Ref;
      else
         return The_Quantified_Variable.Parameter_Constraint;
      end if;
   end Get_Quantified_Variable_Constraint;

   --------------------------------------------------------------------------------

   function Get_Quantified_Variable_Region (The_Quantified_Variable : Quantified_Variable_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Quantified_Variable = Null_Quantified_Variable_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Quantified_Variable.Region;
      end if;
   end Get_Quantified_Variable_Region;

   --------------------------------------------------------------------------------
   --  Implicit_Return_Variable_Info
   --------------------------------------------------------------------------------

   function Get_Implicit_Return_Variable_Info_Ref (Item : Dictionary.Symbol) return Implicit_Return_Variable_Info_Ref is
      function RefType_To_Implicit_Return_Variable_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type,
                                                                                         Implicit_Return_Variable_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Implicit_Return_Variable_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Implicit_Return_Variable_Symbol then
            DiscriminantDebug
              ("Get_Implicit_Return_Variable_Info_Ref",
               GetSymbolDiscriminant (Item),
               Dictionary.Implicit_Return_Variable_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Implicit_Return_Variable_Info_Ref");
         end if;
         return RefType_To_Implicit_Return_Variable_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Implicit_Return_Variable_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Implicit_Return_Variable
     (Name                         : in     LexTokenManager.Lex_String;
      The_Function                 : in     Subprogram_Info_Ref;
      Comp_Unit                    : in     ContextManager.UnitDescriptors;
      Loc                          : in     LexTokenManager.Token_Position;
      The_Implicit_Return_Variable :    out Implicit_Return_Variable_Info_Ref) is
      The_Implicit_Return_Variable_Symbol : Dictionary.Symbol;

      function Implicit_Return_Variable_Info_Ref_To_RefType is new Unchecked_Conversion (Implicit_Return_Variable_Info_Ref,
                                                                                         Dictionary.Ref_Type);
   begin
      The_Implicit_Return_Variable :=
        new Implicit_Return_Variable_Info'(Self         => Dictionary.NullSymbol,
                                           Name         => Name,
                                           The_Function => The_Function);
      AddSymbol
        (Discriminant => Dictionary.Implicit_Return_Variable_Symbol,
         Ref          => Implicit_Return_Variable_Info_Ref_To_RefType (The_Implicit_Return_Variable),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Implicit_Return_Variable_Symbol);
      The_Implicit_Return_Variable.Self := The_Implicit_Return_Variable_Symbol;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Implicit_Return_Variable");
   end Create_Implicit_Return_Variable;

   --------------------------------------------------------------------------------

   function Get_Implicit_Return_Variable_Symbol
     (The_Implicit_Return_Variable : Implicit_Return_Variable_Info_Ref)
     return                         Dictionary.Symbol
   is
   begin
      if The_Implicit_Return_Variable = Null_Implicit_Return_Variable_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Implicit_Return_Variable.Self;
      end if;
   end Get_Implicit_Return_Variable_Symbol;

   --------------------------------------------------------------------------------

   function Get_Implicit_Return_Variable_Name
     (The_Implicit_Return_Variable : Implicit_Return_Variable_Info_Ref)
     return                         LexTokenManager.Lex_String
   is
   begin
      if The_Implicit_Return_Variable = Null_Implicit_Return_Variable_Info_Ref then
         return LexTokenManager.Null_String;
      else
         return The_Implicit_Return_Variable.Name;
      end if;
   end Get_Implicit_Return_Variable_Name;

   --------------------------------------------------------------------------------

   function Get_Implicit_Return_Variable_Function
     (The_Implicit_Return_Variable : Implicit_Return_Variable_Info_Ref)
     return                         Subprogram_Info_Ref
   is
   begin
      if The_Implicit_Return_Variable = Null_Implicit_Return_Variable_Info_Ref then
         return Null_Subprogram_Info_Ref;
      else
         return The_Implicit_Return_Variable.The_Function;
      end if;
   end Get_Implicit_Return_Variable_Function;

   --------------------------------------------------------------------------------
   --  Implicit_In_Stream_Info
   --------------------------------------------------------------------------------

   function Get_Implicit_In_Stream_Info_Ref (Item : Dictionary.Symbol) return Implicit_In_Stream_Info_Ref is
      function RefType_To_Implicit_In_Stream_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type,
                                                                                   Implicit_In_Stream_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Implicit_In_Stream_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Implicit_In_Stream_Symbol then
            DiscriminantDebug
              ("Get_Implicit_In_Stream_Info_Ref",
               GetSymbolDiscriminant (Item),
               Dictionary.Implicit_In_Stream_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Implicit_In_Stream_Info_Ref");
         end if;
         return RefType_To_Implicit_In_Stream_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Implicit_In_Stream_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Implicit_In_Stream
     (The_Own_Variable       : in     Own_Variable_Info_Ref;
      Comp_Unit              : in     ContextManager.UnitDescriptors;
      Loc                    : in     LexTokenManager.Token_Position;
      The_Implicit_In_Stream :    out Implicit_In_Stream_Info_Ref) is
      The_Implicit_In_Stream_Symbol : Dictionary.Symbol;

      function Implicit_In_Stream_Info_Ref_To_RefType is new Unchecked_Conversion (Implicit_In_Stream_Info_Ref,
                                                                                   Dictionary.Ref_Type);
   begin
      The_Implicit_In_Stream := new Implicit_In_Stream_Info'(Self         => Dictionary.NullSymbol,
                                                             Own_Variable => The_Own_Variable);
      AddSymbol
        (Discriminant => Dictionary.Implicit_In_Stream_Symbol,
         Ref          => Implicit_In_Stream_Info_Ref_To_RefType (The_Implicit_In_Stream),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Implicit_In_Stream_Symbol);
      The_Implicit_In_Stream.Self := The_Implicit_In_Stream_Symbol;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Implicit_In_Stream");
   end Create_Implicit_In_Stream;

   --------------------------------------------------------------------------------

   function Get_Implicit_In_Stream_Symbol (The_Implicit_In_Stream : Implicit_In_Stream_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Implicit_In_Stream = Null_Implicit_In_Stream_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Implicit_In_Stream.Self;
      end if;
   end Get_Implicit_In_Stream_Symbol;

   --------------------------------------------------------------------------------

   function Get_Implicit_In_Stream_Own_Variable
     (The_Implicit_In_Stream : Implicit_In_Stream_Info_Ref)
     return                   Own_Variable_Info_Ref
   is
   begin
      if The_Implicit_In_Stream = Null_Implicit_In_Stream_Info_Ref then
         return Null_Own_Variable_Info_Ref;
      else
         return The_Implicit_In_Stream.Own_Variable;
      end if;
   end Get_Implicit_In_Stream_Own_Variable;

   --------------------------------------------------------------------------------
   --  Rule_Policy_Info
   --------------------------------------------------------------------------------

   procedure Create_Rule_Policy
     (Scope           : in     Dictionary.Scopes;
      Value           : in     Dictionary.Rule_Policies;
      Comp_Unit       : in     ContextManager.UnitDescriptors;
      Loc             : in     LexTokenManager.Token_Position;
      The_Rule_Policy :    out Rule_Policy_Info_Ref) is
      The_Rule_Policy_Symbol : Dictionary.Symbol;

      function Rule_Policy_Info_Ref_To_RefType is new Unchecked_Conversion (Rule_Policy_Info_Ref, Dictionary.Ref_Type);
   begin
      The_Rule_Policy := new Rule_Policy_Info'(Self  => Dictionary.NullSymbol,
                                               Scope => Scope,
                                               Value => Value,
                                               Next  => Null_Rule_Policy_Info_Ref);
      AddSymbol
        (Discriminant => Dictionary.Rule_Policy_Symbol,
         Ref          => Rule_Policy_Info_Ref_To_RefType (The_Rule_Policy),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Rule_Policy_Symbol);
      The_Rule_Policy.Self := The_Rule_Policy_Symbol;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Rule_Policy");
   end Create_Rule_Policy;

   --------------------------------------------------------------------------------

   procedure Set_Next_Rule_Policy (The_Rule_Policy, Next : in Rule_Policy_Info_Ref) is
   begin
      The_Rule_Policy.Next := Next;
   end Set_Next_Rule_Policy;

   --------------------------------------------------------------------------------

   function Get_Rule_Policy_Symbol (The_Rule_Policy : Rule_Policy_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Rule_Policy = Null_Rule_Policy_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Rule_Policy.Self;
      end if;
   end Get_Rule_Policy_Symbol;

   --------------------------------------------------------------------------------

   function Get_Rule_Policy_Scope (The_Rule_Policy : Rule_Policy_Info_Ref) return Dictionary.Scopes is
   begin
      if The_Rule_Policy = Null_Rule_Policy_Info_Ref then
         return Dictionary.NullScope;
      else
         return The_Rule_Policy.Scope;
      end if;
   end Get_Rule_Policy_Scope;

   --------------------------------------------------------------------------------

   function Get_Rule_Policy_Value (The_Rule_Policy : Rule_Policy_Info_Ref) return Dictionary.Rule_Policies is
   begin
      if The_Rule_Policy = Null_Rule_Policy_Info_Ref then
         return Dictionary.Unspecified;
      else
         return The_Rule_Policy.Value;
      end if;
   end Get_Rule_Policy_Value;

   --------------------------------------------------------------------------------

   function Get_Next_Rule_Policy (The_Rule_Policy : Rule_Policy_Info_Ref) return Rule_Policy_Info_Ref is
   begin
      if The_Rule_Policy = Null_Rule_Policy_Info_Ref then
         return Null_Rule_Policy_Info_Ref;
      else
         return The_Rule_Policy.Next;
      end if;
   end Get_Next_Rule_Policy;

   --------------------------------------------------------------------------------
   --  Constituent_Info
   --------------------------------------------------------------------------------

   function Get_Constituent_Info_Ref (Item : Dictionary.Symbol) return Constituent_Info_Ref is
      function RefType_To_Constituent_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type, Constituent_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Constituent_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Constituent_Symbol then
            DiscriminantDebug ("Get_Constituent_Info_Ref", GetSymbolDiscriminant (Item), Dictionary.Constituent_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Constituent_Info_Ref");
         end if;
         return RefType_To_Constituent_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Constituent_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Constituent
     (The_Own_Variable : in     Own_Variable_Info_Ref;
      The_Variable     : in     Variable_Info_Ref;
      Mode             : in     Dictionary.Modes;
      Comp_Unit        : in     ContextManager.UnitDescriptors;
      Loc              : in     LexTokenManager.Token_Position;
      The_Constituent  :    out Constituent_Info_Ref) is
      The_Constituent_Symbol : Dictionary.Symbol;

      function Constituent_Info_Ref_To_RefType is new Unchecked_Conversion (Constituent_Info_Ref, Dictionary.Ref_Type);
   begin
      The_Constituent :=
        new Constituent_Info'
        (Self         => Dictionary.NullSymbol,
         Own_Variable => The_Own_Variable,
         Variable     => The_Variable,
         Mode         => Mode,
         Next         => Null_Constituent_Info_Ref);
      AddSymbol
        (Discriminant => Dictionary.Constituent_Symbol,
         Ref          => Constituent_Info_Ref_To_RefType (The_Constituent),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Constituent_Symbol);
      The_Constituent.Self := The_Constituent_Symbol;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Constituent");
   end Create_Constituent;

   --------------------------------------------------------------------------------

   procedure Set_Next_Constituent (The_Constituent, Next : in Constituent_Info_Ref) is
   begin
      The_Constituent.Next := Next;
   end Set_Next_Constituent;

   --------------------------------------------------------------------------------

   function Get_Constituent_Symbol (The_Constituent : Constituent_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Constituent = Null_Constituent_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Constituent.Self;
      end if;
   end Get_Constituent_Symbol;

   --------------------------------------------------------------------------------

   function Get_Constituent_Own_Variable (The_Constituent : Constituent_Info_Ref) return Own_Variable_Info_Ref is
   begin
      if The_Constituent = Null_Constituent_Info_Ref then
         return Null_Own_Variable_Info_Ref;
      else
         return The_Constituent.Own_Variable;
      end if;
   end Get_Constituent_Own_Variable;

   --------------------------------------------------------------------------------

   function Get_Constituent_Variable (The_Constituent : Constituent_Info_Ref) return Variable_Info_Ref is
   begin
      if The_Constituent = Null_Constituent_Info_Ref then
         return Null_Variable_Info_Ref;
      else
         return The_Constituent.Variable;
      end if;
   end Get_Constituent_Variable;

   --------------------------------------------------------------------------------

   function Get_Constituent_Mode (The_Constituent : Constituent_Info_Ref) return Dictionary.Modes is
   begin
      if The_Constituent = Null_Constituent_Info_Ref then
         return Dictionary.InvalidMode;
      else
         return The_Constituent.Mode;
      end if;
   end Get_Constituent_Mode;

   --------------------------------------------------------------------------------

   function Get_Next_Constituent (The_Constituent : Constituent_Info_Ref) return Constituent_Info_Ref is
   begin
      if The_Constituent = Null_Constituent_Info_Ref then
         return Null_Constituent_Info_Ref;
      else
         return The_Constituent.Next;
      end if;
   end Get_Next_Constituent;

   --------------------------------------------------------------------------------
   --  Context_Clause_Info
   --------------------------------------------------------------------------------

   function Get_Context_Clause_Info_Ref (Item : Dictionary.Symbol) return Context_Clause_Info_Ref is
      function RefType_To_Context_Clause_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type, Context_Clause_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Context_Clause_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Context_Clause_Symbol then
            DiscriminantDebug ("Get_Context_Clause_Info_Ref", GetSymbolDiscriminant (Item), Dictionary.Context_Clause_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Context_Clause_Info_Ref");
         end if;
         return RefType_To_Context_Clause_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Context_Clause_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Context_Clause
     (The_Package        : in     Package_Info_Ref;
      The_Subprogram     : in     Subprogram_Info_Ref;
      Explicit           : in     Boolean;
      Comp_Unit          : in     ContextManager.UnitDescriptors;
      Loc                : in     LexTokenManager.Token_Position;
      The_Context_Clause :    out Context_Clause_Info_Ref) is
      The_Context_Clause_Symbol : Dictionary.Symbol;

      function Context_Clause_Info_Ref_To_RefType is new Unchecked_Conversion (Context_Clause_Info_Ref, Dictionary.Ref_Type);
   begin
      if The_Package /= Null_Package_Info_Ref
        and then The_Subprogram = Null_Subprogram_Info_Ref then
         The_Context_Clause :=
           new Context_Clause_Info'
           (Self          => Dictionary.NullSymbol,
            Is_Subprogram => False,
            The_Package   => The_Package,
            Explicit      => Explicit,
            Next          => Null_Context_Clause_Info_Ref);
      elsif The_Package = Null_Package_Info_Ref
        and then The_Subprogram /= Null_Subprogram_Info_Ref then
         The_Context_Clause :=
           new Context_Clause_Info'
           (Self           => Dictionary.NullSymbol,
            Is_Subprogram  => True,
            The_Subprogram => The_Subprogram,
            Explicit       => Explicit,
            Next           => Null_Context_Clause_Info_Ref);
      else
         The_Context_Clause := Null_Context_Clause_Info_Ref;
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Symbol_Table,
            Msg     => "Dictionary.RawDict.Create_Context_Clause");
      end if;
      AddSymbol
        (Discriminant => Dictionary.Context_Clause_Symbol,
         Ref          => Context_Clause_Info_Ref_To_RefType (The_Context_Clause),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Context_Clause_Symbol);
      The_Context_Clause.Self := The_Context_Clause_Symbol;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Context_Clause");
   end Create_Context_Clause;

   --------------------------------------------------------------------------------

   procedure Set_Context_Clause_Explicit (The_Context_Clause : in Context_Clause_Info_Ref) is
   begin
      The_Context_Clause.Explicit := True;
   end Set_Context_Clause_Explicit;

   --------------------------------------------------------------------------------

   procedure Set_Next_Context_Clause (The_Context_Clause, Next : in Context_Clause_Info_Ref) is
   begin
      The_Context_Clause.Next := Next;
   end Set_Next_Context_Clause;

   --------------------------------------------------------------------------------

   function Get_Context_Clause_Symbol (The_Context_Clause : Context_Clause_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Context_Clause = Null_Context_Clause_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Context_Clause.Self;
      end if;
   end Get_Context_Clause_Symbol;

   --------------------------------------------------------------------------------

   function Get_Context_Clause_Is_Subprogram (The_Context_Clause : Context_Clause_Info_Ref) return Boolean is
   begin
      if The_Context_Clause = Null_Context_Clause_Info_Ref then
         return False;
      else
         return The_Context_Clause.Is_Subprogram;
      end if;
   end Get_Context_Clause_Is_Subprogram;

   --------------------------------------------------------------------------------

   function Get_Context_Clause_Package (The_Context_Clause : Context_Clause_Info_Ref) return Package_Info_Ref is
   begin
      if The_Context_Clause = Null_Context_Clause_Info_Ref then
         return Null_Package_Info_Ref;
      else
         return The_Context_Clause.The_Package;
      end if;
   end Get_Context_Clause_Package;

   --------------------------------------------------------------------------------

   function Get_Context_Clause_Subprogram (The_Context_Clause : Context_Clause_Info_Ref) return Subprogram_Info_Ref is
   begin
      if The_Context_Clause = Null_Context_Clause_Info_Ref then
         return Null_Subprogram_Info_Ref;
      else
         return The_Context_Clause.The_Subprogram;
      end if;
   end Get_Context_Clause_Subprogram;

   --------------------------------------------------------------------------------

   function Get_Context_Clause_Explicit (The_Context_Clause : Context_Clause_Info_Ref) return Boolean is
   begin
      if The_Context_Clause = Null_Context_Clause_Info_Ref then
         return False;
      else
         return The_Context_Clause.Explicit;
      end if;
   end Get_Context_Clause_Explicit;

   --------------------------------------------------------------------------------

   function Get_Next_Context_Clause (The_Context_Clause : Context_Clause_Info_Ref) return Context_Clause_Info_Ref is
   begin
      if The_Context_Clause = Null_Context_Clause_Info_Ref then
         return Null_Context_Clause_Info_Ref;
      else
         return The_Context_Clause.Next;
      end if;
   end Get_Next_Context_Clause;

   --------------------------------------------------------------------------------
   --  Use_Type_Clause_Info
   --------------------------------------------------------------------------------

   procedure Create_Use_Type_Clause
     (Type_Mark           : in     Type_Info_Ref;
      Comp_Unit           : in     ContextManager.UnitDescriptors;
      Loc                 : in     LexTokenManager.Token_Position;
      The_Use_Type_Clause :    out Use_Type_Clause_Info_Ref) is
      The_Use_Type_Clause_Symbol : Dictionary.Symbol;
      pragma Unreferenced (The_Use_Type_Clause_Symbol);

      function Use_Type_Clause_Info_Ref_To_RefType is new Unchecked_Conversion (Use_Type_Clause_Info_Ref, Dictionary.Ref_Type);
   begin
      The_Use_Type_Clause := new Use_Type_Clause_Info'(The_Type => Type_Mark,
                                                       Next     => Null_Use_Type_Clause_Info_Ref);
      AddSymbol
        (Discriminant => Dictionary.Use_Type_Clause_Symbol,
         Ref          => Use_Type_Clause_Info_Ref_To_RefType (The_Use_Type_Clause),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Use_Type_Clause_Symbol);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Use_Type_Clause");
   end Create_Use_Type_Clause;

   --------------------------------------------------------------------------------

   procedure Set_Next_Use_Type_Clause (The_Use_Type_Clause, Next : in Use_Type_Clause_Info_Ref) is
   begin
      The_Use_Type_Clause.Next := Next;
   end Set_Next_Use_Type_Clause;

   --------------------------------------------------------------------------------

   function Get_Use_Type_Clause_Type (The_Use_Type_Clause : Use_Type_Clause_Info_Ref) return Type_Info_Ref is
   begin
      if The_Use_Type_Clause = Null_Use_Type_Clause_Info_Ref then
         return Null_Type_Info_Ref;
      else
         return The_Use_Type_Clause.The_Type;
      end if;
   end Get_Use_Type_Clause_Type;

   --------------------------------------------------------------------------------

   function Get_Next_Use_Type_Clause (The_Use_Type_Clause : Use_Type_Clause_Info_Ref) return Use_Type_Clause_Info_Ref is
   begin
      if The_Use_Type_Clause = Null_Use_Type_Clause_Info_Ref then
         return Null_Use_Type_Clause_Info_Ref;
      else
         return The_Use_Type_Clause.Next;
      end if;
   end Get_Next_Use_Type_Clause;

   --------------------------------------------------------------------------------
   --  Parameter_Constraint_Info_Ref
   --------------------------------------------------------------------------------

   function Get_Parameter_Constraint_Info_Ref (Item : Dictionary.Symbol) return Parameter_Constraint_Info_Ref is
      function RefType_To_Parameter_Constraint_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type,
                                                                                     Parameter_Constraint_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Parameter_Constraint_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Parameter_Constraint_Symbol then
            DiscriminantDebug
              ("Get_Parameter_Constraint_Info_Ref",
               GetSymbolDiscriminant (Item),
               Dictionary.Parameter_Constraint_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Parameter_Constraint_Info_Ref");
         end if;
         return RefType_To_Parameter_Constraint_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Parameter_Constraint_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Parameter_Constraint
     (The_Subprogram_Parameter : in     Subprogram_Parameter_Info_Ref;
      Dimension                : in     Positive;
      Comp_Unit                : in     ContextManager.UnitDescriptors;
      Loc                      : in     LexTokenManager.Token_Position;
      The_Parameter_Constraint :    out Parameter_Constraint_Info_Ref) is
      The_Parameter_Constraint_Symbol : Dictionary.Symbol;

      function Parameter_Constraint_Info_Ref_To_RefType is new Unchecked_Conversion (Parameter_Constraint_Info_Ref,
                                                                                     Dictionary.Ref_Type);
   begin
      The_Parameter_Constraint :=
        new Parameter_Constraint_Info'
        (Self                 => Dictionary.NullSymbol,
         Subprogram_Parameter => The_Subprogram_Parameter,
         Dimension            => Dimension,
         Next                 => Null_Parameter_Constraint_Info_Ref);
      AddSymbol
        (Discriminant => Dictionary.Parameter_Constraint_Symbol,
         Ref          => Parameter_Constraint_Info_Ref_To_RefType (The_Parameter_Constraint),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Parameter_Constraint_Symbol);
      The_Parameter_Constraint.Self := The_Parameter_Constraint_Symbol;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Parameter_Constraint");
   end Create_Parameter_Constraint;

   --------------------------------------------------------------------------------

   procedure Set_Next_Parameter_Constraint (The_Parameter_Constraint, Next : in Parameter_Constraint_Info_Ref) is
   begin
      The_Parameter_Constraint.Next := Next;
   end Set_Next_Parameter_Constraint;

   --------------------------------------------------------------------------------

   function Get_Parameter_Constraint_Symbol (The_Parameter_Constraint : Parameter_Constraint_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Parameter_Constraint = Null_Parameter_Constraint_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Parameter_Constraint.Self;
      end if;
   end Get_Parameter_Constraint_Symbol;

   --------------------------------------------------------------------------------

   function Get_Parameter_Constraint_Subprogram_Parameter
     (The_Parameter_Constraint : Parameter_Constraint_Info_Ref)
     return                     Subprogram_Parameter_Info_Ref
   is
   begin
      if The_Parameter_Constraint = Null_Parameter_Constraint_Info_Ref then
         return Null_Subprogram_Parameter_Info_Ref;
      else
         return The_Parameter_Constraint.Subprogram_Parameter;
      end if;
   end Get_Parameter_Constraint_Subprogram_Parameter;

   --------------------------------------------------------------------------------

   function Get_Parameter_Constraint_Dimension (The_Parameter_Constraint : Parameter_Constraint_Info_Ref) return Positive is
   begin
      if The_Parameter_Constraint = Null_Parameter_Constraint_Info_Ref then
         return 1;
      else
         return The_Parameter_Constraint.Dimension;
      end if;
   end Get_Parameter_Constraint_Dimension;

   --------------------------------------------------------------------------------

   function Get_Next_Parameter_Constraint
     (The_Parameter_Constraint : Parameter_Constraint_Info_Ref)
     return                     Parameter_Constraint_Info_Ref
   is
   begin
      if The_Parameter_Constraint = Null_Parameter_Constraint_Info_Ref then
         return Null_Parameter_Constraint_Info_Ref;
      else
         return The_Parameter_Constraint.Next;
      end if;
   end Get_Next_Parameter_Constraint;

   --------------------------------------------------------------------------------
   --  Subprogram_Parameter_Info
   --------------------------------------------------------------------------------

   function Get_Subprogram_Parameter_Info_Ref (Item : Dictionary.Symbol) return Subprogram_Parameter_Info_Ref is
      function RefType_To_Subprogram_Parameter_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type,
                                                                                     Subprogram_Parameter_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Subprogram_Parameter_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Subprogram_Parameter_Symbol then
            DiscriminantDebug
              ("Get_Subprogram_Parameter_Info_Ref",
               GetSymbolDiscriminant (Item),
               Dictionary.Subprogram_Parameter_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Subprogram_Parameter_Info_Ref");
         end if;
         return RefType_To_Subprogram_Parameter_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Subprogram_Parameter_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Subprogram_Parameter
     (Name                     : in     LexTokenManager.Lex_String;
      The_Subprogram           : in     Subprogram_Info_Ref;
      Type_Mark                : in     Type_Info_Ref;
      Mode                     : in     Modes;
      Comp_Unit                : in     ContextManager.UnitDescriptors;
      Loc                      : in     LexTokenManager.Token_Position;
      The_Subprogram_Parameter :    out Subprogram_Parameter_Info_Ref) is
      The_Subprogram_Parameter_Symbol : Dictionary.Symbol;

      function Subprogram_Parameter_Info_Ref_To_RefType is new Unchecked_Conversion (Subprogram_Parameter_Info_Ref,
                                                                                     Dictionary.Ref_Type);
   begin
      The_Subprogram_Parameter :=
        new Subprogram_Parameter_Info'
        (Self              => Dictionary.NullSymbol,
         Name              => Name,
         Subprogram        => The_Subprogram,
         Type_Mark         => Type_Mark,
         Mode              => Mode,
         Exported          => Booleans'(Dictionary.Abstractions => False),
         Imported          => Booleans'(Dictionary.Abstractions => False),
         Dependencies      => Dependencies_T'(Dictionary.Abstractions => Null_Dependency_Info_Ref),
         Global_References => Global_Variables_T'(Dictionary.Abstractions => Null_Global_Variable_Info_Ref),
         Subcomponents     => Null_Subcomponent_Info_Ref,
         Index_Constraints => Null_Parameter_Constraint_Info_Ref,
         Next              => Null_Subprogram_Parameter_Info_Ref);
      AddSymbol
        (Discriminant => Dictionary.Subprogram_Parameter_Symbol,
         Ref          => Subprogram_Parameter_Info_Ref_To_RefType (The_Subprogram_Parameter),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Subprogram_Parameter_Symbol);
      The_Subprogram_Parameter.Self := The_Subprogram_Parameter_Symbol;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Subprogram_Parameter");
   end Create_Subprogram_Parameter;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Parameter_Exported
     (The_Subprogram_Parameter : in Subprogram_Parameter_Info_Ref;
      Abstraction              : in Dictionary.Abstractions) is
   begin
      The_Subprogram_Parameter.Exported (Abstraction) := True;
   end Set_Subprogram_Parameter_Exported;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Parameter_Imported
     (The_Subprogram_Parameter : in Subprogram_Parameter_Info_Ref;
      Abstraction              : in Dictionary.Abstractions) is
   begin
      The_Subprogram_Parameter.Imported (Abstraction) := True;
   end Set_Subprogram_Parameter_Imported;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Parameter_Dependencies
     (The_Subprogram_Parameter : in Subprogram_Parameter_Info_Ref;
      Abstraction              : in Dictionary.Abstractions;
      Dependency               : in Dependency_Info_Ref) is
   begin
      The_Subprogram_Parameter.Dependencies (Abstraction) := Dependency;
   end Set_Subprogram_Parameter_Dependencies;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Parameter_Global_References
     (The_Subprogram_Parameter : in Subprogram_Parameter_Info_Ref;
      Abstraction              : in Dictionary.Abstractions;
      Reference                : in Global_Variable_Info_Ref) is
   begin
      The_Subprogram_Parameter.Global_References (Abstraction) := Reference;
   end Set_Subprogram_Parameter_Global_References;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Parameter_Subcomponents
     (The_Subprogram_Parameter : in Subprogram_Parameter_Info_Ref;
      Subcomponents            : in Subcomponent_Info_Ref) is
   begin
      The_Subprogram_Parameter.Subcomponents := Subcomponents;
   end Set_Subprogram_Parameter_Subcomponents;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Parameter_Index_Constraints
     (The_Subprogram_Parameter : in Subprogram_Parameter_Info_Ref;
      The_Index_Constraints    : in Parameter_Constraint_Info_Ref) is
   begin
      The_Subprogram_Parameter.Index_Constraints := The_Index_Constraints;
   end Set_Subprogram_Parameter_Index_Constraints;

   --------------------------------------------------------------------------------

   procedure Set_Next_Subprogram_Parameter (The_Subprogram_Parameter, Next : in Subprogram_Parameter_Info_Ref) is
   begin
      The_Subprogram_Parameter.Next := Next;
   end Set_Next_Subprogram_Parameter;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Parameter_Symbol (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Subprogram_Parameter = Null_Subprogram_Parameter_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Subprogram_Parameter.Self;
      end if;
   end Get_Subprogram_Parameter_Symbol;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Parameter_Name
     (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref)
     return                     LexTokenManager.Lex_String
   is
   begin
      if The_Subprogram_Parameter = Null_Subprogram_Parameter_Info_Ref then
         return LexTokenManager.Null_String;
      else
         return The_Subprogram_Parameter.Name;
      end if;
   end Get_Subprogram_Parameter_Name;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Parameter_Subprogram
     (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref)
     return                     Subprogram_Info_Ref
   is
   begin
      if The_Subprogram_Parameter = Null_Subprogram_Parameter_Info_Ref then
         return Null_Subprogram_Info_Ref;
      else
         return The_Subprogram_Parameter.Subprogram;
      end if;
   end Get_Subprogram_Parameter_Subprogram;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Parameter_Type (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref) return Type_Info_Ref is
   begin
      if The_Subprogram_Parameter = Null_Subprogram_Parameter_Info_Ref then
         return Null_Type_Info_Ref;
      else
         return The_Subprogram_Parameter.Type_Mark;
      end if;
   end Get_Subprogram_Parameter_Type;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Parameter_Mode (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref) return Modes is
   begin
      if The_Subprogram_Parameter = Null_Subprogram_Parameter_Info_Ref then
         return Dictionary.InvalidMode;
      else
         return The_Subprogram_Parameter.Mode;
      end if;
   end Get_Subprogram_Parameter_Mode;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Parameter_Exported
     (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref;
      Abstraction              : Dictionary.Abstractions)
     return                     Boolean
   is
   begin
      if The_Subprogram_Parameter = Null_Subprogram_Parameter_Info_Ref then
         return False;
      else
         return The_Subprogram_Parameter.Exported (Abstraction);
      end if;
   end Get_Subprogram_Parameter_Exported;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Parameter_Imported
     (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref;
      Abstraction              : Dictionary.Abstractions)
     return                     Boolean
   is
   begin
      if The_Subprogram_Parameter = Null_Subprogram_Parameter_Info_Ref then
         return False;
      else
         return The_Subprogram_Parameter.Imported (Abstraction);
      end if;
   end Get_Subprogram_Parameter_Imported;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Parameter_Dependencies
     (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref;
      Abstraction              : Dictionary.Abstractions)
     return                     Dependency_Info_Ref
   is
   begin
      if The_Subprogram_Parameter = Null_Subprogram_Parameter_Info_Ref then
         return Null_Dependency_Info_Ref;
      else
         return The_Subprogram_Parameter.Dependencies (Abstraction);
      end if;
   end Get_Subprogram_Parameter_Dependencies;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Parameter_Global_References
     (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref;
      Abstraction              : Dictionary.Abstractions)
     return                     Global_Variable_Info_Ref
   is
   begin
      if The_Subprogram_Parameter = Null_Subprogram_Parameter_Info_Ref then
         return Null_Global_Variable_Info_Ref;
      else
         return The_Subprogram_Parameter.Global_References (Abstraction);
      end if;
   end Get_Subprogram_Parameter_Global_References;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Parameter_Subcomponents
     (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref)
     return                     Subcomponent_Info_Ref
   is
   begin
      if The_Subprogram_Parameter = Null_Subprogram_Parameter_Info_Ref then
         return Null_Subcomponent_Info_Ref;
      else
         return The_Subprogram_Parameter.Subcomponents;
      end if;
   end Get_Subprogram_Parameter_Subcomponents;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Parameter_Index_Constraints
     (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref)
     return                     Parameter_Constraint_Info_Ref
   is
   begin
      if The_Subprogram_Parameter = Null_Subprogram_Parameter_Info_Ref then
         return Null_Parameter_Constraint_Info_Ref;
      else
         return The_Subprogram_Parameter.Index_Constraints;
      end if;
   end Get_Subprogram_Parameter_Index_Constraints;

   --------------------------------------------------------------------------------

   function Get_Next_Subprogram_Parameter
     (The_Subprogram_Parameter : Subprogram_Parameter_Info_Ref)
     return                     Subprogram_Parameter_Info_Ref
   is
   begin
      if The_Subprogram_Parameter = Null_Subprogram_Parameter_Info_Ref then
         return Null_Subprogram_Parameter_Info_Ref;
      else
         return The_Subprogram_Parameter.Next;
      end if;
   end Get_Next_Subprogram_Parameter;

   --------------------------------------------------------------------------------
   --  Subprogram_Info
   --------------------------------------------------------------------------------

   function Get_Subprogram_Info_Ref (Item : Dictionary.Symbol) return Subprogram_Info_Ref is
      function RefType_To_Subprogram_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type, Subprogram_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Subprogram_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Subprogram_Symbol then
            DiscriminantDebug ("Get_Subprogram_info_Ref", GetSymbolDiscriminant (Item), Dictionary.Subprogram_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Subprogram_Info_Ref");
         end if;
         return RefType_To_Subprogram_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Subprogram_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Subprogram
     (Name            : in     LexTokenManager.Lex_String;
      The_Declaration : in     Declaration_Info_Ref;
      Comp_Unit       : in     ContextManager.UnitDescriptors;
      Loc             : in     LexTokenManager.Token_Position;
      The_Subprogram  :    out Subprogram_Info_Ref) is
      The_Subprogram_Symbol : Dictionary.Symbol;

      function Subprogram_Info_Ref_To_RefType is new Unchecked_Conversion (Subprogram_Info_Ref, Dictionary.Ref_Type);
   begin
      The_Subprogram :=
        new Subprogram_Info'
        (Self                            => Dictionary.NullSymbol,
         Name                            => Name,
         Implicit_Proof_Functions        => Null_Symbols,
         Implicit_Return_Variables       => Implicit_Return_Variables_T'
           (Dictionary.Abstractions => Null_Implicit_Return_Variable_Info_Ref),
         Signature_Is_Wellformed         => Booleans'(Dictionary.Abstractions => True),
         Precondition                    => Constraints'(Dictionary.Abstractions => 0),
         Postcondition                   => Constraints'(Dictionary.Abstractions => 0),
         Subprogram_Body                 => Null_Declaration_Info_Ref,
         Has_Proper_Body                 => False,
         Body_Is_Hidden                  => False,
         Has_Second_Annotation           => False,
         Has_Second_Constraint           => False,
         Has_Derives_Annotation          => False,
         Has_Delay_Property              => False,
         Delay_Property_Is_Accounted_For => False,
         Generic_Unit                    => Null_Generic_Unit_Info_Ref,
         Instantiation_Of                => Null_Subprogram_Info_Ref,
         First_Generic_Association       => Null_Generic_Association_Info_Ref,
         Last_Generic_Association        => Null_Generic_Association_Info_Ref,
         Suspends_List                   => Dictionary.NullSymbol,
         Is_Entry                        => False,
         Entry_Barrier                   => Dictionary.NullSymbol,
         Is_Interrupt_Handler            => False,
         Uses_Unprotected_Variables      => False,
         Uses_Unchecked_Conversion       => False,
         Assigns_From_External           => False,
         Type_Mark                       => Null_Type_Info_Ref,
         With_Clauses                    => Null_Context_Clause_Info_Ref,
         Use_Type_Clauses                => Null_Use_Type_Clause_Info_Ref,
         Inherit_Clauses                 => Null_Context_Clause_Info_Ref,
         First_Subprogram_Parameter      => Null_Subprogram_Parameter_Info_Ref,
         Last_Subprogram_Parameter       => Null_Subprogram_Parameter_Info_Ref,
         First_Global_Variable           => Global_Variables_T'(Dictionary.Abstractions => Null_Global_Variable_Info_Ref),
         Last_Global_Variable            => Global_Variables_T'(Dictionary.Abstractions => Null_Global_Variable_Info_Ref),
         Renaming_Declarations           => Null_Declaration_Info_Ref,
         First_Declaration               => Null_Declaration_Info_Ref,
         Last_Declaration                => Null_Declaration_Info_Ref,
         First_Loop                      => Dictionary.NullSymbol,
         Last_Loop                       => Dictionary.NullSymbol,
         Specification                   => The_Declaration);
      AddSymbol
        (Discriminant => Dictionary.Subprogram_Symbol,
         Ref          => Subprogram_Info_Ref_To_RefType (The_Subprogram),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Subprogram_Symbol);
      The_Subprogram.Self  := The_Subprogram_Symbol;
      The_Declaration.Item := The_Subprogram_Symbol;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Subprogram");
   end Create_Subprogram;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Implicit_Proof_Function
     (The_Subprogram     : in Subprogram_Info_Ref;
      Abstraction        : in Dictionary.Abstractions;
      The_Proof_Function : in Dictionary.Symbol) is
   begin
      The_Subprogram.Implicit_Proof_Functions (Abstraction) := The_Proof_Function;
   end Set_Subprogram_Implicit_Proof_Function;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Implicit_Return_Variable
     (The_Subprogram               : in Subprogram_Info_Ref;
      Abstraction                  : in Dictionary.Abstractions;
      The_Implicit_Return_Variable : in Implicit_Return_Variable_Info_Ref) is
   begin
      The_Subprogram.Implicit_Return_Variables (Abstraction) := The_Implicit_Return_Variable;
   end Set_Subprogram_Implicit_Return_Variable;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Signature_Not_Wellformed
     (The_Subprogram : in Subprogram_Info_Ref;
      Abstraction    : in Dictionary.Abstractions) is
   begin
      The_Subprogram.Signature_Is_Wellformed (Abstraction) := False;
      --  we may be marking a missing second anno as malformed so we must also mark
      --  it as present, thus
      if Abstraction = Dictionary.IsRefined then
         The_Subprogram.Has_Second_Annotation := True;
      end if;
   end Set_Subprogram_Signature_Not_Wellformed;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Precondition
     (The_Subprogram : in Subprogram_Info_Ref;
      Abstraction    : in Dictionary.Abstractions;
      Precondition   : in ExaminerConstants.RefType) is
   begin
      The_Subprogram.Precondition (Abstraction) := Precondition;
   end Set_Subprogram_Precondition;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Postcondition
     (The_Subprogram : in Subprogram_Info_Ref;
      Abstraction    : in Dictionary.Abstractions;
      Postcondition  : in ExaminerConstants.RefType) is
   begin
      The_Subprogram.Postcondition (Abstraction) := Postcondition;
   end Set_Subprogram_Postcondition;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Body (The_Subprogram : in Subprogram_Info_Ref;
                                  The_Body       : in Declaration_Info_Ref) is
   begin
      The_Subprogram.Subprogram_Body := The_Body;
   end Set_Subprogram_Body;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Has_Proper_Body (The_Subprogram : in Subprogram_Info_Ref;
                                             Is_Hidden      : in Boolean) is
   begin
      The_Subprogram.Has_Proper_Body := True;
      The_Subprogram.Body_Is_Hidden  := Is_Hidden;
   end Set_Subprogram_Has_Proper_Body;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Has_Second_Annotation (The_Subprogram : in Subprogram_Info_Ref) is
   begin
      The_Subprogram.Has_Second_Annotation := True;
   end Set_Subprogram_Has_Second_Annotation;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Has_Second_Constraint (The_Subprogram : in Subprogram_Info_Ref) is
   begin
      The_Subprogram.Has_Second_Constraint := True;
   end Set_Subprogram_Has_Second_Constraint;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Has_Derives_Annotation (The_Subprogram : in Subprogram_Info_Ref) is
   begin
      The_Subprogram.Has_Derives_Annotation := True;
   end Set_Subprogram_Has_Derives_Annotation;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Has_Delay_Property (The_Subprogram : in Subprogram_Info_Ref) is
   begin
      The_Subprogram.Has_Delay_Property := True;
   end Set_Subprogram_Has_Delay_Property;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Mark_Accounts_For_Delay (The_Subprogram : in Subprogram_Info_Ref) is
   begin
      The_Subprogram.Delay_Property_Is_Accounted_For := True;
   end Set_Subprogram_Mark_Accounts_For_Delay;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Generic_Unit (The_Subprogram   : in Subprogram_Info_Ref;
                                          The_Generic_Unit : in Generic_Unit_Info_Ref) is
   begin
      The_Subprogram.Generic_Unit := The_Generic_Unit;
   end Set_Subprogram_Generic_Unit;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Instantiation_Of (The_Subprogram : in Subprogram_Info_Ref;
                                              The_Generic    : in Subprogram_Info_Ref) is
   begin
      The_Subprogram.Instantiation_Of := The_Generic;
   end Set_Subprogram_Instantiation_Of;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_First_Generic_Association
     (The_Subprogram          : in Subprogram_Info_Ref;
      The_Generic_Association : in Generic_Association_Info_Ref) is
   begin
      The_Subprogram.First_Generic_Association := The_Generic_Association;
   end Set_Subprogram_First_Generic_Association;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Last_Generic_Association
     (The_Subprogram          : in Subprogram_Info_Ref;
      The_Generic_Association : in Generic_Association_Info_Ref) is
   begin
      The_Subprogram.Last_Generic_Association := The_Generic_Association;
   end Set_Subprogram_Last_Generic_Association;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Suspends_List (The_Subprogram    : in Subprogram_Info_Ref;
                                           The_Suspends_List : in Dictionary.Symbol) is
   begin
      The_Subprogram.Suspends_List := The_Suspends_List;
   end Set_Subprogram_Suspends_List;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Is_Entry (The_Subprogram : in Subprogram_Info_Ref) is
   begin
      The_Subprogram.Is_Entry := True;
   end Set_Subprogram_Is_Entry;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Entry_Barrier (The_Subprogram : in Subprogram_Info_Ref;
                                           The_Barrier    : in Dictionary.Symbol) is
   begin
      The_Subprogram.Entry_Barrier := The_Barrier;
   end Set_Subprogram_Entry_Barrier;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Is_Interrupt_Handler (The_Subprogram : in Subprogram_Info_Ref) is
   begin
      The_Subprogram.Is_Interrupt_Handler := True;
   end Set_Subprogram_Is_Interrupt_Handler;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Uses_Unprotected_Variables (The_Subprogram : in Subprogram_Info_Ref) is
   begin
      The_Subprogram.Uses_Unprotected_Variables := True;
   end Set_Subprogram_Uses_Unprotected_Variables;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Uses_Unchecked_Conversion (The_Subprogram : in Subprogram_Info_Ref) is
   begin
      The_Subprogram.Uses_Unchecked_Conversion := True;
   end Set_Subprogram_Uses_Unchecked_Conversion;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Assigns_From_External (The_Subprogram : in Subprogram_Info_Ref) is
   begin
      The_Subprogram.Assigns_From_External := True;
   end Set_Subprogram_Assigns_From_External;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Return_Type (The_Subprogram : in Subprogram_Info_Ref;
                                         Type_Mark      : in Type_Info_Ref) is
   begin
      The_Subprogram.Type_Mark := Type_Mark;
   end Set_Subprogram_Return_Type;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_With_Clauses
     (The_Subprogram     : in Subprogram_Info_Ref;
      The_Context_Clause : in Context_Clause_Info_Ref) is
   begin
      The_Subprogram.With_Clauses := The_Context_Clause;
   end Set_Subprogram_With_Clauses;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Use_Type_Clauses
     (The_Subprogram      : in Subprogram_Info_Ref;
      The_Use_Type_Clause : in Use_Type_Clause_Info_Ref) is
   begin
      The_Subprogram.Use_Type_Clauses := The_Use_Type_Clause;
   end Set_Subprogram_Use_Type_Clauses;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Inherit_Clauses
     (The_Subprogram     : in Subprogram_Info_Ref;
      The_Inherit_Clause : in Context_Clause_Info_Ref) is
   begin
      The_Subprogram.Inherit_Clauses := The_Inherit_Clause;
   end Set_Subprogram_Inherit_Clauses;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_First_Parameter
     (The_Subprogram           : in Subprogram_Info_Ref;
      The_Subprogram_Parameter : in Subprogram_Parameter_Info_Ref) is
   begin
      The_Subprogram.First_Subprogram_Parameter := The_Subprogram_Parameter;
   end Set_Subprogram_First_Parameter;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Last_Parameter
     (The_Subprogram           : in Subprogram_Info_Ref;
      The_Subprogram_Parameter : in Subprogram_Parameter_Info_Ref) is
   begin
      The_Subprogram.Last_Subprogram_Parameter := The_Subprogram_Parameter;
   end Set_Subprogram_Last_Parameter;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_First_Global_Variable
     (The_Subprogram      : in Subprogram_Info_Ref;
      Abstraction         : in Dictionary.Abstractions;
      The_Global_Variable : in Global_Variable_Info_Ref) is
   begin
      The_Subprogram.First_Global_Variable (Abstraction) := The_Global_Variable;
   end Set_Subprogram_First_Global_Variable;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Last_Global_Variable
     (The_Subprogram      : in Subprogram_Info_Ref;
      Abstraction         : in Dictionary.Abstractions;
      The_Global_Variable : in Global_Variable_Info_Ref) is
   begin
      The_Subprogram.Last_Global_Variable (Abstraction) := The_Global_Variable;
   end Set_Subprogram_Last_Global_Variable;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Renaming_Declarations
     (The_Subprogram  : in Subprogram_Info_Ref;
      The_Declaration : in Declaration_Info_Ref) is
   begin
      The_Subprogram.Renaming_Declarations := The_Declaration;
   end Set_Subprogram_Renaming_Declarations;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_First_Declaration
     (The_Subprogram  : in Subprogram_Info_Ref;
      The_Declaration : in Declaration_Info_Ref) is
   begin
      The_Subprogram.First_Declaration := The_Declaration;
   end Set_Subprogram_First_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Last_Declaration
     (The_Subprogram  : in Subprogram_Info_Ref;
      The_Declaration : in Declaration_Info_Ref) is
   begin
      The_Subprogram.Last_Declaration := The_Declaration;
   end Set_Subprogram_Last_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_First_Loop (The_Subprogram : in Subprogram_Info_Ref;
                                        The_Loop       : in Dictionary.Symbol) is
   begin
      The_Subprogram.First_Loop := The_Loop;
   end Set_Subprogram_First_Loop;

   --------------------------------------------------------------------------------

   procedure Set_Subprogram_Last_Loop (The_Subprogram : in Subprogram_Info_Ref;
                                       The_Loop       : in Dictionary.Symbol) is
   begin
      The_Subprogram.Last_Loop := The_Loop;
   end Set_Subprogram_Last_Loop;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Symbol (The_Subprogram : Subprogram_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Subprogram.Self;
      end if;
   end Get_Subprogram_Symbol;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Name (The_Subprogram : Subprogram_Info_Ref) return LexTokenManager.Lex_String is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return LexTokenManager.Null_String;
      else
         return The_Subprogram.Name;
      end if;
   end Get_Subprogram_Name;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Implicit_Proof_Function
     (The_Subprogram : Subprogram_Info_Ref;
      Abstraction    : Dictionary.Abstractions)
     return           Dictionary.Symbol
   is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Subprogram.Implicit_Proof_Functions (Abstraction);
      end if;
   end Get_Subprogram_Implicit_Proof_Function;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Implicit_Return_Variable
     (The_Subprogram : Subprogram_Info_Ref;
      Abstraction    : Dictionary.Abstractions)
     return           Implicit_Return_Variable_Info_Ref
   is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Null_Implicit_Return_Variable_Info_Ref;
      else
         return The_Subprogram.Implicit_Return_Variables (Abstraction);
      end if;
   end Get_Subprogram_Implicit_Return_Variable;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Signature_Is_Wellformed
     (The_Subprogram : Subprogram_Info_Ref;
      Abstraction    : Dictionary.Abstractions)
     return           Boolean
   is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return False;
      else
         return The_Subprogram.Signature_Is_Wellformed (Abstraction);
      end if;
   end Get_Subprogram_Signature_Is_Wellformed;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Precondition
     (The_Subprogram : Subprogram_Info_Ref;
      Abstraction    : Dictionary.Abstractions)
     return           ExaminerConstants.RefType
   is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return 0;
      else
         return The_Subprogram.Precondition (Abstraction);
      end if;
   end Get_Subprogram_Precondition;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Postcondition
     (The_Subprogram : Subprogram_Info_Ref;
      Abstraction    : Dictionary.Abstractions)
     return           ExaminerConstants.RefType
   is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return 0;
      else
         return The_Subprogram.Postcondition (Abstraction);
      end if;
   end Get_Subprogram_Postcondition;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Body (The_Subprogram : Subprogram_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return The_Subprogram.Subprogram_Body;
      end if;
   end Get_Subprogram_Body;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Has_Proper_Body (The_Subprogram : Subprogram_Info_Ref) return Boolean is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return False;
      else
         return The_Subprogram.Has_Proper_Body;
      end if;
   end Get_Subprogram_Has_Proper_Body;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Body_Is_Hidden (The_Subprogram : Subprogram_Info_Ref) return Boolean is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return False;
      else
         return The_Subprogram.Body_Is_Hidden;
      end if;
   end Get_Subprogram_Body_Is_Hidden;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Has_Second_Annotation (The_Subprogram : Subprogram_Info_Ref) return Boolean is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return False;
      else
         return The_Subprogram.Has_Second_Annotation;
      end if;
   end Get_Subprogram_Has_Second_Annotation;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Has_Second_Constraint (The_Subprogram : Subprogram_Info_Ref) return Boolean is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return False;
      else
         return The_Subprogram.Has_Second_Constraint;
      end if;
   end Get_Subprogram_Has_Second_Constraint;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Has_Derives_Annotation (The_Subprogram : Subprogram_Info_Ref) return Boolean is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return False;
      else
         return The_Subprogram.Has_Derives_Annotation;
      end if;
   end Get_Subprogram_Has_Derives_Annotation;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Has_Delay_Property (The_Subprogram : Subprogram_Info_Ref) return Boolean is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return False;
      else
         return The_Subprogram.Has_Delay_Property;
      end if;
   end Get_Subprogram_Has_Delay_Property;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Delay_Property_Is_Accounted_For (The_Subprogram : Subprogram_Info_Ref) return Boolean is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return False;
      else
         return The_Subprogram.Delay_Property_Is_Accounted_For;
      end if;
   end Get_Subprogram_Delay_Property_Is_Accounted_For;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Generic_Unit (The_Subprogram : Subprogram_Info_Ref) return Generic_Unit_Info_Ref is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Null_Generic_Unit_Info_Ref;
      else
         return The_Subprogram.Generic_Unit;
      end if;
   end Get_Subprogram_Generic_Unit;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Instantiation_Of (The_Subprogram : Subprogram_Info_Ref) return Subprogram_Info_Ref is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Null_Subprogram_Info_Ref;
      else
         return The_Subprogram.Instantiation_Of;
      end if;
   end Get_Subprogram_Instantiation_Of;

   --------------------------------------------------------------------------------

   function Get_Subprogram_First_Generic_Association (The_Subprogram : Subprogram_Info_Ref) return Generic_Association_Info_Ref is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Null_Generic_Association_Info_Ref;
      else
         return The_Subprogram.First_Generic_Association;
      end if;
   end Get_Subprogram_First_Generic_Association;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Last_Generic_Association (The_Subprogram : Subprogram_Info_Ref) return Generic_Association_Info_Ref is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Null_Generic_Association_Info_Ref;
      else
         return The_Subprogram.Last_Generic_Association;
      end if;
   end Get_Subprogram_Last_Generic_Association;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Suspends_List (The_Subprogram : Subprogram_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Subprogram.Suspends_List;
      end if;
   end Get_Subprogram_Suspends_List;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Is_Entry (The_Subprogram : Subprogram_Info_Ref) return Boolean is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return False;
      else
         return The_Subprogram.Is_Entry;
      end if;
   end Get_Subprogram_Is_Entry;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Entry_Barrier (The_Subprogram : Subprogram_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Subprogram.Entry_Barrier;
      end if;
   end Get_Subprogram_Entry_Barrier;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Is_Interrupt_Handler (The_Subprogram : Subprogram_Info_Ref) return Boolean is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return False;
      else
         return The_Subprogram.Is_Interrupt_Handler;
      end if;
   end Get_Subprogram_Is_Interrupt_Handler;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Uses_Unprotected_Variables (The_Subprogram : Subprogram_Info_Ref) return Boolean is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return False;
      else
         return The_Subprogram.Uses_Unprotected_Variables;
      end if;
   end Get_Subprogram_Uses_Unprotected_Variables;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Uses_Unchecked_Conversion (The_Subprogram : Subprogram_Info_Ref) return Boolean is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return False;
      else
         return The_Subprogram.Uses_Unchecked_Conversion;
      end if;
   end Get_Subprogram_Uses_Unchecked_Conversion;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Assigns_From_External (The_Subprogram : Subprogram_Info_Ref) return Boolean is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return False;
      else
         return The_Subprogram.Assigns_From_External;
      end if;
   end Get_Subprogram_Assigns_From_External;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Return_Type (The_Subprogram : Subprogram_Info_Ref) return Type_Info_Ref is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Null_Type_Info_Ref;
      else
         return The_Subprogram.Type_Mark;
      end if;
   end Get_Subprogram_Return_Type;

   --------------------------------------------------------------------------------

   function Get_Subprogram_With_Clauses (The_Subprogram : Subprogram_Info_Ref) return Context_Clause_Info_Ref is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Null_Context_Clause_Info_Ref;
      else
         return The_Subprogram.With_Clauses;
      end if;
   end Get_Subprogram_With_Clauses;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Use_Type_Clauses (The_Subprogram : Subprogram_Info_Ref) return Use_Type_Clause_Info_Ref is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Null_Use_Type_Clause_Info_Ref;
      else
         return The_Subprogram.Use_Type_Clauses;
      end if;
   end Get_Subprogram_Use_Type_Clauses;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Inherit_Clauses (The_Subprogram : Subprogram_Info_Ref) return Context_Clause_Info_Ref is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Null_Context_Clause_Info_Ref;
      else
         return The_Subprogram.Inherit_Clauses;
      end if;
   end Get_Subprogram_Inherit_Clauses;

   --------------------------------------------------------------------------------

   function Get_Subprogram_First_Parameter (The_Subprogram : Subprogram_Info_Ref) return Subprogram_Parameter_Info_Ref is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Null_Subprogram_Parameter_Info_Ref;
      else
         return The_Subprogram.First_Subprogram_Parameter;
      end if;
   end Get_Subprogram_First_Parameter;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Last_Parameter (The_Subprogram : Subprogram_Info_Ref) return Subprogram_Parameter_Info_Ref is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Null_Subprogram_Parameter_Info_Ref;
      else
         return The_Subprogram.Last_Subprogram_Parameter;
      end if;
   end Get_Subprogram_Last_Parameter;

   --------------------------------------------------------------------------------

   function Get_Subprogram_First_Global_Variable
     (The_Subprogram : Subprogram_Info_Ref;
      Abstraction    : Dictionary.Abstractions)
     return           Global_Variable_Info_Ref
   is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Null_Global_Variable_Info_Ref;
      else
         return The_Subprogram.First_Global_Variable (Abstraction);
      end if;
   end Get_Subprogram_First_Global_Variable;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Last_Global_Variable
     (The_Subprogram : Subprogram_Info_Ref;
      Abstraction    : Dictionary.Abstractions)
     return           Global_Variable_Info_Ref
   is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Null_Global_Variable_Info_Ref;
      else
         return The_Subprogram.Last_Global_Variable (Abstraction);
      end if;
   end Get_Subprogram_Last_Global_Variable;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Renaming_Declarations (The_Subprogram : Subprogram_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return The_Subprogram.Renaming_Declarations;
      end if;
   end Get_Subprogram_Renaming_Declarations;

   --------------------------------------------------------------------------------

   function Get_Subprogram_First_Declaration (The_Subprogram : Subprogram_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return The_Subprogram.First_Declaration;
      end if;
   end Get_Subprogram_First_Declaration;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Last_Declaration (The_Subprogram : Subprogram_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return The_Subprogram.Last_Declaration;
      end if;
   end Get_Subprogram_Last_Declaration;

   --------------------------------------------------------------------------------

   function Get_Subprogram_First_Loop (The_Subprogram : Subprogram_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Subprogram.First_Loop;
      end if;
   end Get_Subprogram_First_Loop;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Last_Loop (The_Subprogram : Subprogram_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Subprogram.Last_Loop;
      end if;
   end Get_Subprogram_Last_Loop;

   --------------------------------------------------------------------------------

   function Get_Subprogram_Specification (The_Subprogram : Subprogram_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Subprogram = Null_Subprogram_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return The_Subprogram.Specification;
      end if;
   end Get_Subprogram_Specification;

   --------------------------------------------------------------------------------
   --  Operator_Info
   --------------------------------------------------------------------------------

   function Get_Operator_Info_Ref (Item : Dictionary.Symbol) return Operator_Info_Ref is
      function RefType_To_Operator_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type, Operator_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Operator_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Operator_Symbol then
            DiscriminantDebug ("Get_Operator_info_Ref", GetSymbolDiscriminant (Item), Dictionary.Operator_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Operator_info_Ref");
         end if;
         return RefType_To_Operator_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Operator_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Unary_Operator
     (Name         : in     SP_Symbols.SP_Symbol;
      Operand      : in     Type_Info_Ref;
      Comp_Unit    : in     ContextManager.UnitDescriptors;
      Loc          : in     LexTokenManager.Token_Position;
      The_Operator :    out Operator_Info_Ref) is
      The_Operator_Symbol : Dictionary.Symbol;

      function Operator_Info_Ref_To_RefType is new Unchecked_Conversion (Operator_Info_Ref, Dictionary.Ref_Type);
   begin
      The_Operator := new Operator_Info'(Self      => Dictionary.NullSymbol,
                                         Name      => Name,
                                         Is_Binary => False,
                                         Operand   => Operand);
      AddSymbol
        (Discriminant => Dictionary.Operator_Symbol,
         Ref          => Operator_Info_Ref_To_RefType (The_Operator),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Operator_Symbol);
      The_Operator.Self := The_Operator_Symbol;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Unary_Operator");
   end Create_Unary_Operator;

   --------------------------------------------------------------------------------

   procedure Create_Binary_Operator
     (Name         : in     SP_Symbols.SP_Symbol;
      Left, Right  : in     Type_Info_Ref;
      Comp_Unit    : in     ContextManager.UnitDescriptors;
      Loc          : in     LexTokenManager.Token_Position;
      The_Operator :    out Operator_Info_Ref) is
      The_Operator_Symbol : Dictionary.Symbol;

      function Operator_Info_Ref_To_RefType is new Unchecked_Conversion (Operator_Info_Ref, Dictionary.Ref_Type);
   begin
      The_Operator :=
        new Operator_Info'
        (Self          => Dictionary.NullSymbol,
         Name          => Name,
         Is_Binary     => True,
         Left_Operand  => Left,
         Right_Operand => Right);
      AddSymbol
        (Discriminant => Dictionary.Operator_Symbol,
         Ref          => Operator_Info_Ref_To_RefType (The_Operator),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Operator_Symbol);
      The_Operator.Self := The_Operator_Symbol;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Binary_Operator");
   end Create_Binary_Operator;

   --------------------------------------------------------------------------------

   function Get_Operator_Symbol (The_Operator : Operator_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Operator = Null_Operator_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Operator.Self;
      end if;
   end Get_Operator_Symbol;

   --------------------------------------------------------------------------------

   function Get_Operator_Name (The_Operator : Operator_Info_Ref) return SP_Symbols.SP_Symbol is
   begin
      if The_Operator = Null_Operator_Info_Ref then
         return SP_Symbols.SPEND;
      else
         return The_Operator.Name;
      end if;
   end Get_Operator_Name;

   --------------------------------------------------------------------------------

   function Get_Operator_Is_Binary (The_Operator : Operator_Info_Ref) return Boolean is
   begin
      if The_Operator = Null_Operator_Info_Ref then
         return False;
      else
         return The_Operator.Is_Binary;
      end if;
   end Get_Operator_Is_Binary;

   --------------------------------------------------------------------------------

   function Get_Operator_Operand (The_Operator : Operator_Info_Ref) return Type_Info_Ref is
   begin
      if The_Operator = Null_Operator_Info_Ref then
         return Null_Type_Info_Ref;
      else
         return The_Operator.Operand;
      end if;
   end Get_Operator_Operand;

   --------------------------------------------------------------------------------

   function Get_Operator_Left_Operand (The_Operator : Operator_Info_Ref) return Type_Info_Ref is
   begin
      if The_Operator = Null_Operator_Info_Ref then
         return Null_Type_Info_Ref;
      else
         return The_Operator.Left_Operand;
      end if;
   end Get_Operator_Left_Operand;

   --------------------------------------------------------------------------------

   function Get_Operator_Right_Operand (The_Operator : Operator_Info_Ref) return Type_Info_Ref is
   begin
      if The_Operator = Null_Operator_Info_Ref then
         return Null_Type_Info_Ref;
      else
         return The_Operator.Right_Operand;
      end if;
   end Get_Operator_Right_Operand;

   --------------------------------------------------------------------------------
   --  Dependency_Info
   --------------------------------------------------------------------------------

   function Get_Dependency_Info_Ref (Item : Dictionary.Symbol) return Dependency_Info_Ref is
      function RefType_To_Dependency_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type,
                                                                           Dependency_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Dependency_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Dependency_Symbol then
            DiscriminantDebug
              ("Get_Dependency_Info_Ref",
               GetSymbolDiscriminant (Item),
               Dictionary.Dependency_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Dependency_Info_Ref");
         end if;
         return RefType_To_Dependency_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Dependency_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Dependency
     (The_Import_Parameter : in     Subprogram_Parameter_Info_Ref;
      The_Import_Variable  : in     Variable_Info_Ref;
      Comp_Unit            : in     ContextManager.UnitDescriptors;
      Loc                  : in     LexTokenManager.Token_Position;
      The_Dependency       :    out Dependency_Info_Ref) is
      The_Dependency_Symbol : Dictionary.Symbol;

      function Dependency_Info_Ref_To_RefType is new Unchecked_Conversion (Dependency_Info_Ref,
                                                                           Dictionary.Ref_Type);
   begin
      if The_Import_Parameter /= Null_Subprogram_Parameter_Info_Ref then
         The_Dependency :=
           new Dependency_Info'
           (Self               => Dictionary.NullSymbol,
            Next               => Null_Dependency_Info_Ref,
            Kind_Of_Dependency => Dependency_Parameter_Item,
            Import_Parameter   => The_Import_Parameter);
      elsif The_Import_Variable /= Null_Variable_Info_Ref then
         The_Dependency :=
           new Dependency_Info'
           (Self               => Dictionary.NullSymbol,
            Next               => Null_Dependency_Info_Ref,
            Kind_Of_Dependency => Dependency_Variable_Item,
            Import_Variable    => The_Import_Variable);
      else
         The_Dependency := Null_Dependency_Info_Ref;
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Symbol_Table,
            Msg     => "Dictionary.RawDict.Create_Dependency");
      end if;
      AddSymbol
        (Discriminant => Dictionary.Dependency_Symbol,
         Ref          => Dependency_Info_Ref_To_RefType (The_Dependency),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Dependency_Symbol);
      The_Dependency.Self := The_Dependency_Symbol;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Dependency");
   end Create_Dependency;

   --------------------------------------------------------------------------------

   procedure Set_Next_Dependency (The_Dependency, Next : in Dependency_Info_Ref) is
   begin
      The_Dependency.Next := Next;
   end Set_Next_Dependency;

   --------------------------------------------------------------------------------

   function Get_Dependency_Symbol (The_Dependency : Dependency_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Dependency = Null_Dependency_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Dependency.Self;
      end if;
   end Get_Dependency_Symbol;

   --------------------------------------------------------------------------------

   function Get_Kind_Of_Dependency (The_Dependency : Dependency_Info_Ref) return Kind_Of_Dependency_T is
   begin
      return The_Dependency.Kind_Of_Dependency;
   end Get_Kind_Of_Dependency;

   --------------------------------------------------------------------------------

   function Get_Dependency_Import_Parameter (The_Dependency : Dependency_Info_Ref) return Subprogram_Parameter_Info_Ref is
   begin
      if The_Dependency = Null_Dependency_Info_Ref then
         return Null_Subprogram_Parameter_Info_Ref;
      else
         return The_Dependency.Import_Parameter;
      end if;
   end Get_Dependency_Import_Parameter;

   --------------------------------------------------------------------------------

   function Get_Dependency_Import_Variable (The_Dependency : Dependency_Info_Ref) return Variable_Info_Ref is
   begin
      if The_Dependency = Null_Dependency_Info_Ref then
         return Null_Variable_Info_Ref;
      else
         return The_Dependency.Import_Variable;
      end if;
   end Get_Dependency_Import_Variable;

   --------------------------------------------------------------------------------

   function Get_Next_Dependency (The_Dependency : Dependency_Info_Ref) return Dependency_Info_Ref is
   begin
      if The_Dependency = Null_Dependency_Info_Ref then
         return Null_Dependency_Info_Ref;
      else
         return The_Dependency.Next;
      end if;
   end Get_Next_Dependency;

   --------------------------------------------------------------------------------
   --  Package_Info
   --------------------------------------------------------------------------------

   function Get_Package_Info_Ref (Item : Dictionary.Symbol) return Package_Info_Ref is
      function RefType_To_Package_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type,
                                                                        Package_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Package_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Package_Symbol then
            DiscriminantDebug
              ("Get_Package_Info_Ref",
               GetSymbolDiscriminant (Item),
               Dictionary.Package_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Package_Info_Ref");
         end if;
         return RefType_To_Package_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Package_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Package
     (Name            : in     LexTokenManager.Lex_String;
      The_Declaration : in     Declaration_Info_Ref;
      Comp_Unit       : in     ContextManager.UnitDescriptors;
      Loc             : in     LexTokenManager.Token_Position;
      The_Package     :    out Package_Info_Ref) is
      The_Package_Symbol : Dictionary.Symbol;

      function Package_Info_Ref_To_RefType is new Unchecked_Conversion (Package_Info_Ref, Dictionary.Ref_Type);
   begin
      The_Package :=
        new Package_Info'
        (Self                    => Dictionary.NullSymbol,
         Name                    => Name,
         Package_Body            => Null_Declaration_Info_Ref,
         Package_Has_Proper_Body => False,
         Inherit_Clauses         => Null_Context_Clause_Info_Ref,
         Own_Variables           => Null_Own_Variable_Info_Ref,
         Task_List               => Dictionary.NullSymbol,
         First_Loop              => Dictionary.NullSymbol,
         Last_Loop               => Dictionary.NullSymbol,
         Specification           => The_Declaration,
         Visible_Part            => Part_Info'(With_Clauses          => Null_Context_Clause_Info_Ref,
                                               Use_Type_Clauses      => Null_Use_Type_Clause_Info_Ref,
                                               Renaming_Declarations => Null_Declaration_Info_Ref,
                                               First_Declaration     => Null_Declaration_Info_Ref,
                                               Last_Declaration      => Null_Declaration_Info_Ref),
         Local_Part              => Part_Info'(With_Clauses          => Null_Context_Clause_Info_Ref,
                                               Use_Type_Clauses      => Null_Use_Type_Clause_Info_Ref,
                                               Renaming_Declarations => Null_Declaration_Info_Ref,
                                               First_Declaration     => Null_Declaration_Info_Ref,
                                               Last_Declaration      => Null_Declaration_Info_Ref),
         Private_Part            => Part_Info'(With_Clauses          => Null_Context_Clause_Info_Ref,
                                               Use_Type_Clauses      => Null_Use_Type_Clause_Info_Ref,
                                               Renaming_Declarations => Null_Declaration_Info_Ref,
                                               First_Declaration     => Null_Declaration_Info_Ref,
                                               Last_Declaration      => Null_Declaration_Info_Ref),
         Elaborate_Body_Found    => False,
         Parent                  => Null_Package_Info_Ref,
         Next_Sibling            => Null_Package_Info_Ref,
         Extends                 => Null_Package_Info_Ref,
         Declares_Tagged_Type    => False,
         Private_Children        => Child_Info'(First_Child => Null_Package_Info_Ref,
                                                Last_Child  => Null_Package_Info_Ref),
         Public_Children         => Child_Info'(First_Child => Null_Package_Info_Ref,
                                                Last_Child  => Null_Package_Info_Ref),
         Is_Private              => False,
         Generic_Unit            => Null_Generic_Unit_Info_Ref);
      AddSymbol
        (Discriminant => Dictionary.Package_Symbol,
         Ref          => Package_Info_Ref_To_RefType (The_Package),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Package_Symbol);
      The_Package.Self     := The_Package_Symbol;
      The_Declaration.Item := The_Package_Symbol;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Package");
   end Create_Package;

   --------------------------------------------------------------------------------

   procedure Set_Package_Body (The_Package : in Package_Info_Ref;
                               The_Body    : in Declaration_Info_Ref) is
   begin
      The_Package.Package_Body := The_Body;
   end Set_Package_Body;

   --------------------------------------------------------------------------------

   procedure Set_Package_Has_Proper_Body (The_Package : in Package_Info_Ref) is
   begin
      The_Package.Package_Has_Proper_Body := True;
   end Set_Package_Has_Proper_Body;

   --------------------------------------------------------------------------------

   procedure Set_Package_Inherit_Clauses (The_Package        : in Package_Info_Ref;
                                          The_Context_Clause : in Context_Clause_Info_Ref) is
   begin
      The_Package.Inherit_Clauses := The_Context_Clause;
   end Set_Package_Inherit_Clauses;

   --------------------------------------------------------------------------------

   procedure Set_Package_Own_Variables (The_Package   : in Package_Info_Ref;
                                        Own_Variables : in Own_Variable_Info_Ref) is
   begin
      The_Package.Own_Variables := Own_Variables;
   end Set_Package_Own_Variables;

   --------------------------------------------------------------------------------

   procedure Set_Package_Task_List (The_Package : in Package_Info_Ref;
                                    Task_List   : in Dictionary.Symbol) is
   begin
      The_Package.Task_List := Task_List;
   end Set_Package_Task_List;

   --------------------------------------------------------------------------------

   procedure Set_Package_First_Loop (The_Package : in Package_Info_Ref;
                                     The_Loop    : in Dictionary.Symbol) is
   begin
      The_Package.First_Loop := The_Loop;
   end Set_Package_First_Loop;

   --------------------------------------------------------------------------------

   procedure Set_Package_Last_Loop (The_Package : in Package_Info_Ref;
                                    The_Loop    : in Dictionary.Symbol) is
   begin
      The_Package.Last_Loop := The_Loop;
   end Set_Package_Last_Loop;

   --------------------------------------------------------------------------------

   procedure Set_Package_Visible_With_Clauses
     (The_Package        : in Package_Info_Ref;
      The_Context_Clause : in Context_Clause_Info_Ref) is
   begin
      The_Package.Visible_Part.With_Clauses := The_Context_Clause;
   end Set_Package_Visible_With_Clauses;

   --------------------------------------------------------------------------------

   procedure Set_Package_Visible_Use_Type_Clauses
     (The_Package         : in Package_Info_Ref;
      The_Use_Type_Clause : in Use_Type_Clause_Info_Ref) is
   begin
      The_Package.Visible_Part.Use_Type_Clauses := The_Use_Type_Clause;
   end Set_Package_Visible_Use_Type_Clauses;

   --------------------------------------------------------------------------------

   procedure Set_Package_Visible_Renaming_Declarations
     (The_Package     : in Package_Info_Ref;
      The_Declaration : in Declaration_Info_Ref) is
   begin
      The_Package.Visible_Part.Renaming_Declarations := The_Declaration;
   end Set_Package_Visible_Renaming_Declarations;

   --------------------------------------------------------------------------------

   procedure Set_Package_First_Visible_Declaration
     (The_Package     : in Package_Info_Ref;
      The_Declaration : in Declaration_Info_Ref) is
   begin
      The_Package.Visible_Part.First_Declaration := The_Declaration;
   end Set_Package_First_Visible_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Package_Last_Visible_Declaration
     (The_Package     : in Package_Info_Ref;
      The_Declaration : in Declaration_Info_Ref) is
   begin
      The_Package.Visible_Part.Last_Declaration := The_Declaration;
   end Set_Package_Last_Visible_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Package_Local_With_Clauses
     (The_Package        : in Package_Info_Ref;
      The_Context_Clause : in Context_Clause_Info_Ref) is
   begin
      The_Package.Local_Part.With_Clauses := The_Context_Clause;
   end Set_Package_Local_With_Clauses;

   --------------------------------------------------------------------------------

   procedure Set_Package_Local_Use_Type_Clauses
     (The_Package         : in Package_Info_Ref;
      The_Use_Type_Clause : in Use_Type_Clause_Info_Ref) is
   begin
      The_Package.Local_Part.Use_Type_Clauses := The_Use_Type_Clause;
   end Set_Package_Local_Use_Type_Clauses;

   --------------------------------------------------------------------------------

   procedure Set_Package_Local_Renaming_Declarations
     (The_Package     : in Package_Info_Ref;
      The_Declaration : in Declaration_Info_Ref) is
   begin
      The_Package.Local_Part.Renaming_Declarations := The_Declaration;
   end Set_Package_Local_Renaming_Declarations;

   --------------------------------------------------------------------------------

   procedure Set_Package_First_Local_Declaration
     (The_Package     : in Package_Info_Ref;
      The_Declaration : in Declaration_Info_Ref) is
   begin
      The_Package.Local_Part.First_Declaration := The_Declaration;
   end Set_Package_First_Local_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Package_Last_Local_Declaration
     (The_Package     : in Package_Info_Ref;
      The_Declaration : in Declaration_Info_Ref) is
   begin
      The_Package.Local_Part.Last_Declaration := The_Declaration;
   end Set_Package_Last_Local_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Package_First_Private_Declaration
     (The_Package     : in Package_Info_Ref;
      The_Declaration : in Declaration_Info_Ref) is
   begin
      The_Package.Private_Part.First_Declaration := The_Declaration;
   end Set_Package_First_Private_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Package_Last_Private_Declaration
     (The_Package     : in Package_Info_Ref;
      The_Declaration : in Declaration_Info_Ref) is
   begin
      The_Package.Private_Part.Last_Declaration := The_Declaration;
   end Set_Package_Last_Private_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Package_Elaborate_Body_Found (The_Package : in Package_Info_Ref) is
   begin
      The_Package.Elaborate_Body_Found := True;
   end Set_Package_Elaborate_Body_Found;

   --------------------------------------------------------------------------------

   procedure Set_Package_Parent (The_Package : in Package_Info_Ref;
                                 The_Parent  : in Package_Info_Ref) is
   begin
      The_Package.Parent := The_Parent;
   end Set_Package_Parent;

   --------------------------------------------------------------------------------

   procedure Set_Package_Next_Sibling (The_Package : in Package_Info_Ref;
                                       The_Sibling : in Package_Info_Ref) is
   begin
      The_Package.Next_Sibling := The_Sibling;
   end Set_Package_Next_Sibling;

   --------------------------------------------------------------------------------

   procedure Set_Package_Extends (The_Package : in Package_Info_Ref;
                                  The_Extends : in Package_Info_Ref) is
   begin
      The_Package.Extends := The_Extends;
   end Set_Package_Extends;

   --------------------------------------------------------------------------------

   procedure Set_Package_Declares_Tagged_Type (The_Package : in Package_Info_Ref) is
   begin
      The_Package.Declares_Tagged_Type := True;
   end Set_Package_Declares_Tagged_Type;

   --------------------------------------------------------------------------------

   procedure Set_Package_First_Private_Child (The_Package : in Package_Info_Ref;
                                              The_Child   : in Package_Info_Ref) is
   begin
      The_Package.Private_Children.First_Child := The_Child;
   end Set_Package_First_Private_Child;

   --------------------------------------------------------------------------------

   procedure Set_Package_Last_Private_Child (The_Package : in Package_Info_Ref;
                                             The_Child   : in Package_Info_Ref) is
   begin
      The_Package.Private_Children.Last_Child := The_Child;
   end Set_Package_Last_Private_Child;

   --------------------------------------------------------------------------------

   procedure Set_Package_First_Public_Child (The_Package : in Package_Info_Ref;
                                             The_Child   : in Package_Info_Ref) is
   begin
      The_Package.Public_Children.First_Child := The_Child;
   end Set_Package_First_Public_Child;

   --------------------------------------------------------------------------------

   procedure Set_Package_Last_Public_Child (The_Package : in Package_Info_Ref;
                                            The_Child   : in Package_Info_Ref) is
   begin
      The_Package.Public_Children.Last_Child := The_Child;
   end Set_Package_Last_Public_Child;

   --------------------------------------------------------------------------------

   procedure Set_Package_Is_Private (The_Package : in Package_Info_Ref) is
   begin
      The_Package.Is_Private := True;
   end Set_Package_Is_Private;

   --------------------------------------------------------------------------------

   procedure Set_Package_Generic_Unit (The_Package      : in Package_Info_Ref;
                                       The_Generic_Unit : in Generic_Unit_Info_Ref) is
   begin
      The_Package.Generic_Unit := The_Generic_Unit;
   end Set_Package_Generic_Unit;

   --------------------------------------------------------------------------------

   function Get_Package_Symbol (The_Package : Package_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Package.Self;
      end if;
   end Get_Package_Symbol;

   --------------------------------------------------------------------------------

   function Get_Package_Name (The_Package : Package_Info_Ref) return LexTokenManager.Lex_String is
   begin
      if The_Package = Null_Package_Info_Ref then
         return LexTokenManager.Null_String;
      else
         return The_Package.Name;
      end if;
   end Get_Package_Name;

   --------------------------------------------------------------------------------

   function Get_Package_Body (The_Package : Package_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return The_Package.Package_Body;
      end if;
   end Get_Package_Body;

   --------------------------------------------------------------------------------

   function Get_Package_Has_Proper_Body (The_Package : Package_Info_Ref) return Boolean is
   begin
      if The_Package = Null_Package_Info_Ref then
         return False;
      else
         return The_Package.Package_Has_Proper_Body;
      end if;
   end Get_Package_Has_Proper_Body;

   --------------------------------------------------------------------------------

   function Get_Package_Inherit_Clauses (The_Package : Package_Info_Ref) return Context_Clause_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Context_Clause_Info_Ref;
      else
         return The_Package.Inherit_Clauses;
      end if;
   end Get_Package_Inherit_Clauses;

   --------------------------------------------------------------------------------

   function Get_Package_Own_Variables (The_Package : Package_Info_Ref) return Own_Variable_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Own_Variable_Info_Ref;
      else
         return The_Package.Own_Variables;
      end if;
   end Get_Package_Own_Variables;

   --------------------------------------------------------------------------------

   function Get_Package_Task_List (The_Package : Package_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Package.Task_List;
      end if;
   end Get_Package_Task_List;

   --------------------------------------------------------------------------------

   function Get_Package_First_Loop (The_Package : Package_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Package.First_Loop;
      end if;
   end Get_Package_First_Loop;

   --------------------------------------------------------------------------------

   function Get_Package_Last_Loop (The_Package : Package_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Package.Last_Loop;
      end if;
   end Get_Package_Last_Loop;

   --------------------------------------------------------------------------------

   function Get_Package_Specification (The_Package : Package_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return The_Package.Specification;
      end if;
   end Get_Package_Specification;

   --------------------------------------------------------------------------------

   function Get_Package_Visible_With_Clauses (The_Package : Package_Info_Ref) return Context_Clause_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Context_Clause_Info_Ref;
      else
         return The_Package.Visible_Part.With_Clauses;
      end if;
   end Get_Package_Visible_With_Clauses;

   --------------------------------------------------------------------------------

   function Get_Package_Visible_Use_Type_Clauses (The_Package : Package_Info_Ref) return Use_Type_Clause_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Use_Type_Clause_Info_Ref;
      else
         return The_Package.Visible_Part.Use_Type_Clauses;
      end if;
   end Get_Package_Visible_Use_Type_Clauses;

   --------------------------------------------------------------------------------

   function Get_Package_Visible_Renaming_Declarations (The_Package : Package_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return The_Package.Visible_Part.Renaming_Declarations;
      end if;
   end Get_Package_Visible_Renaming_Declarations;

   --------------------------------------------------------------------------------

   function Get_Package_First_Visible_Declaration (The_Package : Package_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return The_Package.Visible_Part.First_Declaration;
      end if;
   end Get_Package_First_Visible_Declaration;

   --------------------------------------------------------------------------------

   function Get_Package_Last_Visible_Declaration (The_Package : Package_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return The_Package.Visible_Part.Last_Declaration;
      end if;
   end Get_Package_Last_Visible_Declaration;

   --------------------------------------------------------------------------------

   function Get_Package_Local_With_Clauses (The_Package : Package_Info_Ref) return Context_Clause_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Context_Clause_Info_Ref;
      else
         return The_Package.Local_Part.With_Clauses;
      end if;
   end Get_Package_Local_With_Clauses;

   --------------------------------------------------------------------------------

   function Get_Package_Local_Use_Type_Clauses (The_Package : Package_Info_Ref) return Use_Type_Clause_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Use_Type_Clause_Info_Ref;
      else
         return The_Package.Local_Part.Use_Type_Clauses;
      end if;
   end Get_Package_Local_Use_Type_Clauses;

   --------------------------------------------------------------------------------

   function Get_Package_Local_Renaming_Declarations (The_Package : Package_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return The_Package.Local_Part.Renaming_Declarations;
      end if;
   end Get_Package_Local_Renaming_Declarations;

   --------------------------------------------------------------------------------

   function Get_Package_First_Local_Declaration (The_Package : Package_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return The_Package.Local_Part.First_Declaration;
      end if;
   end Get_Package_First_Local_Declaration;

   --------------------------------------------------------------------------------

   function Get_Package_Last_Local_Declaration (The_Package : Package_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return The_Package.Local_Part.Last_Declaration;
      end if;
   end Get_Package_Last_Local_Declaration;

   --------------------------------------------------------------------------------

   function Get_Package_First_Private_Declaration (The_Package : Package_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return The_Package.Private_Part.First_Declaration;
      end if;
   end Get_Package_First_Private_Declaration;

   --------------------------------------------------------------------------------

   function Get_Package_Last_Private_Declaration (The_Package : Package_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return The_Package.Private_Part.Last_Declaration;
      end if;
   end Get_Package_Last_Private_Declaration;

   --------------------------------------------------------------------------------

   function Get_Package_Elaborate_Body_Found (The_Package : Package_Info_Ref) return Boolean is
   begin
      if The_Package = Null_Package_Info_Ref then
         return False;
      else
         return The_Package.Elaborate_Body_Found;
      end if;
   end Get_Package_Elaborate_Body_Found;

   --------------------------------------------------------------------------------

   function Get_Package_Parent (The_Package : Package_Info_Ref) return Package_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Package_Info_Ref;
      else
         return The_Package.Parent;
      end if;
   end Get_Package_Parent;

   --------------------------------------------------------------------------------

   function Get_Package_Next_Sibling (The_Package : Package_Info_Ref) return Package_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Package_Info_Ref;
      else
         return The_Package.Next_Sibling;
      end if;
   end Get_Package_Next_Sibling;

   --------------------------------------------------------------------------------

   function Get_Package_Extends (The_Package : Package_Info_Ref) return Package_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Package_Info_Ref;
      else
         return The_Package.Extends;
      end if;
   end Get_Package_Extends;

   --------------------------------------------------------------------------------

   function Get_Package_Declares_Tagged_Type (The_Package : Package_Info_Ref) return Boolean is
   begin
      if The_Package = Null_Package_Info_Ref then
         return False;
      else
         return The_Package.Declares_Tagged_Type;
      end if;
   end Get_Package_Declares_Tagged_Type;

   --------------------------------------------------------------------------------

   function Get_Package_First_Private_Child (The_Package : Package_Info_Ref) return Package_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Package_Info_Ref;
      else
         return The_Package.Private_Children.First_Child;
      end if;
   end Get_Package_First_Private_Child;

   --------------------------------------------------------------------------------

   function Get_Package_Last_Private_Child (The_Package : Package_Info_Ref) return Package_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Package_Info_Ref;
      else
         return The_Package.Private_Children.Last_Child;
      end if;
   end Get_Package_Last_Private_Child;

   --------------------------------------------------------------------------------

   function Get_Package_First_Public_Child (The_Package : Package_Info_Ref) return Package_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Package_Info_Ref;
      else
         return The_Package.Public_Children.First_Child;
      end if;
   end Get_Package_First_Public_Child;

   --------------------------------------------------------------------------------

   function Get_Package_Last_Public_Child (The_Package : Package_Info_Ref) return Package_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Package_Info_Ref;
      else
         return The_Package.Public_Children.Last_Child;
      end if;
   end Get_Package_Last_Public_Child;

   --------------------------------------------------------------------------------

   function Get_Package_Is_Private (The_Package : Package_Info_Ref) return Boolean is
   begin
      if The_Package = Null_Package_Info_Ref then
         return False;
      else
         return The_Package.Is_Private;
      end if;
   end Get_Package_Is_Private;

   --------------------------------------------------------------------------------

   function Get_Package_Generic_Unit (The_Package : Package_Info_Ref) return Generic_Unit_Info_Ref is
   begin
      if The_Package = Null_Package_Info_Ref then
         return Null_Generic_Unit_Info_Ref;
      else
         return The_Package.Generic_Unit;
      end if;
   end Get_Package_Generic_Unit;

   --------------------------------------------------------------------------------
   --  Generic_Parameter_Info
   --------------------------------------------------------------------------------

   function Get_Generic_Parameter_Info_Ref (Item : Dictionary.Symbol) return Generic_Parameter_Info_Ref is
      function RefType_To_Generic_Parameter_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type,
                                                                                  Generic_Parameter_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Generic_Parameter_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Generic_Parameter_Symbol then
            DiscriminantDebug
              ("Get_Generic_Parameter_Info_Ref",
               GetSymbolDiscriminant (Item),
               Dictionary.Generic_Parameter_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Generic_Parameter_Info_Ref");
         end if;
         return RefType_To_Generic_Parameter_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Generic_Parameter_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Generic_Parameter
     (Owning_Generic        : in     Generic_Unit_Info_Ref;
      Type_Mark             : in     Type_Info_Ref;
      Object                : in     Constant_Info_Ref;
      Comp_Unit             : in     ContextManager.UnitDescriptors;
      Loc                   : in     LexTokenManager.Token_Position;
      The_Generic_Parameter :    out Generic_Parameter_Info_Ref) is
      The_Generic_Parameter_Symbol : Dictionary.Symbol;

      function Generic_Parameter_Info_Ref_To_RefType is new Unchecked_Conversion (Generic_Parameter_Info_Ref,
                                                                                  Dictionary.Ref_Type);
   begin
      if Type_Mark /= Null_Type_Info_Ref then
         The_Generic_Parameter :=
           new Generic_Parameter_Info'
           (Self           => Dictionary.NullSymbol,
            Owning_Generic => Owning_Generic,
            Kind           => Dictionary.Generic_Type_Parameter,
            Type_Mark      => Type_Mark,
            Next           => Null_Generic_Parameter_Info_Ref);
      elsif Object /= Null_Constant_Info_Ref then
         The_Generic_Parameter :=
           new Generic_Parameter_Info'
           (Self           => Dictionary.NullSymbol,
            Owning_Generic => Owning_Generic,
            Kind           => Dictionary.Generic_Object_Parameter,
            Object         => Object,
            Next           => Null_Generic_Parameter_Info_Ref);
      else
         The_Generic_Parameter := Null_Generic_Parameter_Info_Ref;
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Symbol_Table,
            Msg     => "Dictionary.RawDict.Create_Generic_Parameter");
      end if;
      AddSymbol
        (Discriminant => Dictionary.Generic_Parameter_Symbol,
         Ref          => Generic_Parameter_Info_Ref_To_RefType (The_Generic_Parameter),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Generic_Parameter_Symbol);
      The_Generic_Parameter.Self := The_Generic_Parameter_Symbol;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Generic_Parameter");
   end Create_Generic_Parameter;

   --------------------------------------------------------------------------------

   procedure Set_Next_Generic_Parameter (The_Generic_Parameter, Next : in Generic_Parameter_Info_Ref) is
   begin
      The_Generic_Parameter.Next := Next;
   end Set_Next_Generic_Parameter;

   --------------------------------------------------------------------------------

   function Get_Generic_Parameter_Symbol (The_Generic_Parameter : Generic_Parameter_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Generic_Parameter = Null_Generic_Parameter_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Generic_Parameter.Self;
      end if;
   end Get_Generic_Parameter_Symbol;

   --------------------------------------------------------------------------------

   function Get_Generic_Parameter_Owning_Generic
     (The_Generic_Parameter : Generic_Parameter_Info_Ref)
     return                  Generic_Unit_Info_Ref
   is
   begin
      if The_Generic_Parameter = Null_Generic_Parameter_Info_Ref then
         return Null_Generic_Unit_Info_Ref;
      else
         return The_Generic_Parameter.Owning_Generic;
      end if;
   end Get_Generic_Parameter_Owning_Generic;

   --------------------------------------------------------------------------------

   function Get_Generic_Parameter_Kind
     (The_Generic_Parameter : Generic_Parameter_Info_Ref)
     return                  Dictionary.Generic_Parameter_Kind
   is
   begin
      return The_Generic_Parameter.Kind;
   end Get_Generic_Parameter_Kind;

   --------------------------------------------------------------------------------

   function Get_Generic_Parameter_Type (The_Generic_Parameter : Generic_Parameter_Info_Ref) return Type_Info_Ref is
   begin
      if The_Generic_Parameter = Null_Generic_Parameter_Info_Ref then
         return Null_Type_Info_Ref;
      else
         return The_Generic_Parameter.Type_Mark;
      end if;
   end Get_Generic_Parameter_Type;

   --------------------------------------------------------------------------------

   function Get_Generic_Parameter_Object (The_Generic_Parameter : Generic_Parameter_Info_Ref) return Constant_Info_Ref is
   begin
      if The_Generic_Parameter = Null_Generic_Parameter_Info_Ref then
         return Null_Constant_Info_Ref;
      else
         return The_Generic_Parameter.Object;
      end if;
   end Get_Generic_Parameter_Object;

   --------------------------------------------------------------------------------

   function Get_Next_Generic_Parameter (The_Generic_Parameter : Generic_Parameter_Info_Ref) return Generic_Parameter_Info_Ref is
   begin
      if The_Generic_Parameter = Null_Generic_Parameter_Info_Ref then
         return Null_Generic_Parameter_Info_Ref;
      else
         return The_Generic_Parameter.Next;
      end if;
   end Get_Next_Generic_Parameter;

   --------------------------------------------------------------------------------
   --  Generic_Unit_Info
   --------------------------------------------------------------------------------

   function Get_Generic_Unit_Info_Ref (Item : Dictionary.Symbol) return Generic_Unit_Info_Ref is
      function RefType_To_Generic_Unit_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type, Generic_Unit_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Generic_Unit_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.Generic_Unit_Symbol then
            DiscriminantDebug ("Get_Generic_Unit_Info_Ref", GetSymbolDiscriminant (Item), Dictionary.Generic_Unit_Symbol);
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "Dictionary.RawDict.Get_Generic_Unit_Info_Ref");
         end if;
         return RefType_To_Generic_Unit_Info_Ref (GetSymbolRef (Item));
      end if;
   end Get_Generic_Unit_Info_Ref;

   --------------------------------------------------------------------------------

   procedure Create_Generic_Unit
     (Kind             : in     Dictionary.Generic_Kind;
      Scope            : in     Dictionary.Scopes;
      Comp_Unit        : in     ContextManager.UnitDescriptors;
      Loc              : in     LexTokenManager.Token_Position;
      The_Generic_Unit :    out Generic_Unit_Info_Ref) is
      The_Generic_Unit_Symbol : Dictionary.Symbol;

      function Generic_Unit_Info_Ref_To_RefType is new Unchecked_Conversion (Generic_Unit_Info_Ref, Dictionary.Ref_Type);
   begin
      case Kind is
         when Dictionary.Generic_Of_Subprogram =>
            The_Generic_Unit :=
              new Generic_Unit_Info'
              (Self                    => Dictionary.NullSymbol,
               Scope                   => Scope,
               First_Declaration       => Null_Declaration_Info_Ref,
               Last_Declaration        => Null_Declaration_Info_Ref,
               First_Generic_Parameter => Null_Generic_Parameter_Info_Ref,
               Last_Generic_Parameter  => Null_Generic_Parameter_Info_Ref,
               Kind                    => Dictionary.Generic_Of_Subprogram,
               Owning_Subprogram       => Null_Subprogram_Info_Ref);
         when Dictionary.Generic_Of_Package =>
            The_Generic_Unit :=
              new Generic_Unit_Info'
              (Self                    => Dictionary.NullSymbol,
               Scope                   => Scope,
               First_Declaration       => Null_Declaration_Info_Ref,
               Last_Declaration        => Null_Declaration_Info_Ref,
               First_Generic_Parameter => Null_Generic_Parameter_Info_Ref,
               Last_Generic_Parameter  => Null_Generic_Parameter_Info_Ref,
               Kind                    => Dictionary.Generic_Of_Package,
               Owning_Package          => Null_Package_Info_Ref);
      end case;
      AddSymbol
        (Discriminant => Dictionary.Generic_Unit_Symbol,
         Ref          => Generic_Unit_Info_Ref_To_RefType (The_Generic_Unit),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Generic_Unit_Symbol);
      The_Generic_Unit.Self := The_Generic_Unit_Symbol;
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Generic_Unit");
   end Create_Generic_Unit;

   --------------------------------------------------------------------------------

   procedure Set_Generic_Unit_First_Declaration
     (The_Generic_Unit : in Generic_Unit_Info_Ref;
      The_Declaration  : in Declaration_Info_Ref) is
   begin
      The_Generic_Unit.First_Declaration := The_Declaration;
   end Set_Generic_Unit_First_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Generic_Unit_Last_Declaration
     (The_Generic_Unit : in Generic_Unit_Info_Ref;
      The_Declaration  : in Declaration_Info_Ref) is
   begin
      The_Generic_Unit.Last_Declaration := The_Declaration;
   end Set_Generic_Unit_Last_Declaration;

   --------------------------------------------------------------------------------

   procedure Set_Generic_Unit_First_Generic_Parameter
     (The_Generic_Unit      : in Generic_Unit_Info_Ref;
      The_Generic_Parameter : in Generic_Parameter_Info_Ref) is
   begin
      The_Generic_Unit.First_Generic_Parameter := The_Generic_Parameter;
   end Set_Generic_Unit_First_Generic_Parameter;

   --------------------------------------------------------------------------------

   procedure Set_Generic_Unit_Last_Generic_Parameter
     (The_Generic_Unit      : in Generic_Unit_Info_Ref;
      The_Generic_Parameter : in Generic_Parameter_Info_Ref) is
   begin
      The_Generic_Unit.Last_Generic_Parameter := The_Generic_Parameter;
   end Set_Generic_Unit_Last_Generic_Parameter;

   --------------------------------------------------------------------------------

   procedure Set_Generic_Unit_Owning_Subprogram
     (The_Generic_Unit : in Generic_Unit_Info_Ref;
      The_Subprogram   : in Subprogram_Info_Ref) is
   begin
      The_Generic_Unit.Owning_Subprogram := The_Subprogram;
   end Set_Generic_Unit_Owning_Subprogram;

   --------------------------------------------------------------------------------

   procedure Set_Generic_Unit_Owning_Package (The_Generic_Unit : in Generic_Unit_Info_Ref;
                                              The_Package      : in Package_Info_Ref) is
   begin
      The_Generic_Unit.Owning_Package := The_Package;
   end Set_Generic_Unit_Owning_Package;

   --------------------------------------------------------------------------------

   function Get_Generic_Unit_Symbol (The_Generic_Unit : Generic_Unit_Info_Ref) return Dictionary.Symbol is
   begin
      if The_Generic_Unit = Null_Generic_Unit_Info_Ref then
         return Dictionary.NullSymbol;
      else
         return The_Generic_Unit.Self;
      end if;
   end Get_Generic_Unit_Symbol;

   --------------------------------------------------------------------------------

   function Get_Generic_Unit_Scope (The_Generic_Unit : Generic_Unit_Info_Ref) return Dictionary.Scopes is
   begin
      if The_Generic_Unit = Null_Generic_Unit_Info_Ref then
         return Dictionary.NullScope;
      else
         return The_Generic_Unit.Scope;
      end if;
   end Get_Generic_Unit_Scope;

   --------------------------------------------------------------------------------

   function Get_Generic_Unit_First_Declaration (The_Generic_Unit : Generic_Unit_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Generic_Unit = Null_Generic_Unit_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return The_Generic_Unit.First_Declaration;
      end if;
   end Get_Generic_Unit_First_Declaration;

   --------------------------------------------------------------------------------

   function Get_Generic_Unit_Last_Declaration (The_Generic_Unit : Generic_Unit_Info_Ref) return Declaration_Info_Ref is
   begin
      if The_Generic_Unit = Null_Generic_Unit_Info_Ref then
         return Null_Declaration_Info_Ref;
      else
         return The_Generic_Unit.Last_Declaration;
      end if;
   end Get_Generic_Unit_Last_Declaration;

   --------------------------------------------------------------------------------

   function Get_Generic_Unit_First_Generic_Parameter
     (The_Generic_Unit : Generic_Unit_Info_Ref)
     return             Generic_Parameter_Info_Ref
   is
   begin
      if The_Generic_Unit = Null_Generic_Unit_Info_Ref then
         return Null_Generic_Parameter_Info_Ref;
      else
         return The_Generic_Unit.First_Generic_Parameter;
      end if;
   end Get_Generic_Unit_First_Generic_Parameter;

   --------------------------------------------------------------------------------

   function Get_Generic_Unit_Last_Generic_Parameter (The_Generic_Unit : Generic_Unit_Info_Ref) return Generic_Parameter_Info_Ref is
   begin
      if The_Generic_Unit = Null_Generic_Unit_Info_Ref then
         return Null_Generic_Parameter_Info_Ref;
      else
         return The_Generic_Unit.Last_Generic_Parameter;
      end if;
   end Get_Generic_Unit_Last_Generic_Parameter;

   --------------------------------------------------------------------------------

   function Get_Generic_Unit_Kind (The_Generic_Unit : Generic_Unit_Info_Ref) return Dictionary.Generic_Kind is
   begin
      return The_Generic_Unit.Kind;
   end Get_Generic_Unit_Kind;

   --------------------------------------------------------------------------------

   function Get_Generic_Unit_Owning_Subprogram (The_Generic_Unit : Generic_Unit_Info_Ref) return Subprogram_Info_Ref is
   begin
      if The_Generic_Unit = Null_Generic_Unit_Info_Ref then
         return Null_Subprogram_Info_Ref;
      else
         return The_Generic_Unit.Owning_Subprogram;
      end if;
   end Get_Generic_Unit_Owning_Subprogram;

   --------------------------------------------------------------------------------

   function Get_Generic_Unit_Owning_Package (The_Generic_Unit : Generic_Unit_Info_Ref) return Package_Info_Ref is
   begin
      if The_Generic_Unit = Null_Generic_Unit_Info_Ref then
         return Null_Package_Info_Ref;
      else
         return The_Generic_Unit.Owning_Package;
      end if;
   end Get_Generic_Unit_Owning_Package;

   --------------------------------------------------------------------------------
   --  Generic_Association_Info
   --------------------------------------------------------------------------------

   procedure Create_Generic_Type_Association
     (Formal_Type             : in     Type_Info_Ref;
      Actual_Type             : in     Type_Info_Ref;
      Comp_Unit               : in     ContextManager.UnitDescriptors;
      Loc                     : in     LexTokenManager.Token_Position;
      The_Generic_Association :    out Generic_Association_Info_Ref) is
      The_Generic_Association_Symbol : Dictionary.Symbol;
      pragma Unreferenced (The_Generic_Association_Symbol);

      function Generic_Association_Info_Ref_To_RefType is new Unchecked_Conversion (Generic_Association_Info_Ref, Dictionary.Ref_Type);
   begin
      The_Generic_Association :=
        new Generic_Association_Info'(Is_Object   => False,
                                      Formal_Type => Formal_Type,
                                      Actual_Type => Actual_Type,
                                      Next        => Null_Generic_Association_Info_Ref);
      AddSymbol
        (Discriminant => Dictionary.Generic_Association_Symbol,
         Ref          => Generic_Association_Info_Ref_To_RefType (The_Generic_Association),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Generic_Association_Symbol);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Generic_Type_Association");
   end Create_Generic_Type_Association;

   --------------------------------------------------------------------------------

   procedure Create_Generic_Object_Association
     (Formal_Object           : in     Constant_Info_Ref;
      Actual_Object           : in     Constant_Info_Ref;
      Comp_Unit               : in     ContextManager.UnitDescriptors;
      Loc                     : in     LexTokenManager.Token_Position;
      The_Generic_Association :    out Generic_Association_Info_Ref) is
      The_Generic_Association_Symbol : Dictionary.Symbol;
      pragma Unreferenced (The_Generic_Association_Symbol);

      function Generic_Association_Info_Ref_To_RefType is new Unchecked_Conversion (Generic_Association_Info_Ref, Dictionary.Ref_Type);
   begin
      The_Generic_Association :=
        new Generic_Association_Info'(Is_Object     => True,
                                      Formal_Object => Formal_Object,
                                      Actual_Object => Actual_Object,
                                      Next          => Null_Generic_Association_Info_Ref);
      AddSymbol
        (Discriminant => Dictionary.Generic_Association_Symbol,
         Ref          => Generic_Association_Info_Ref_To_RefType (The_Generic_Association),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => The_Generic_Association_Symbol);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
            Msg     => "Dictionary.RawDict.Create_Generic_Object_Association");
   end Create_Generic_Object_Association;

   --------------------------------------------------------------------------------

   procedure Set_Next_Generic_Association (The_Generic_Association, Next : in Generic_Association_Info_Ref) is
   begin
      The_Generic_Association.Next := Next;
   end Set_Next_Generic_Association;

   --------------------------------------------------------------------------------

   function Get_Generic_Association_Is_Object (The_Generic_Association : Generic_Association_Info_Ref) return Boolean is
   begin
      return The_Generic_Association.Is_Object;
   end Get_Generic_Association_Is_Object;

   --------------------------------------------------------------------------------

   function Get_Generic_Association_Formal_Type (The_Generic_Association : Generic_Association_Info_Ref) return Type_Info_Ref is
   begin
      if The_Generic_Association = Null_Generic_Association_Info_Ref then
         return Null_Type_Info_Ref;
      else
         return The_Generic_Association.Formal_Type;
      end if;
   end Get_Generic_Association_Formal_Type;

   --------------------------------------------------------------------------------

   function Get_Generic_Association_Formal_Object (The_Generic_Association : Generic_Association_Info_Ref) return Constant_Info_Ref is
   begin
      if The_Generic_Association = Null_Generic_Association_Info_Ref then
         return Null_Constant_Info_Ref;
      else
         return The_Generic_Association.Formal_Object;
      end if;
   end Get_Generic_Association_Formal_Object;

   --------------------------------------------------------------------------------

   function Get_Generic_Association_Actual_Type (The_Generic_Association : Generic_Association_Info_Ref) return Type_Info_Ref is
   begin
      if The_Generic_Association = Null_Generic_Association_Info_Ref then
         return Null_Type_Info_Ref;
      else
         return The_Generic_Association.Actual_Type;
      end if;
   end Get_Generic_Association_Actual_Type;

   --------------------------------------------------------------------------------

   function Get_Generic_Association_Actual_Object (The_Generic_Association : Generic_Association_Info_Ref) return Constant_Info_Ref is
   begin
      if The_Generic_Association = Null_Generic_Association_Info_Ref then
         return Null_Constant_Info_Ref;
      else
         return The_Generic_Association.Actual_Object;
      end if;
   end Get_Generic_Association_Actual_Object;

   --------------------------------------------------------------------------------

   function Get_Next_Generic_Association (The_Generic_Association : Generic_Association_Info_Ref) return Generic_Association_Info_Ref is
   begin
      if The_Generic_Association = Null_Generic_Association_Info_Ref then
         return Null_Generic_Association_Info_Ref;
      else
         return The_Generic_Association.Next;
      end if;
   end Get_Next_Generic_Association;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   function GetKnownDiscriminantRef (Item : Dictionary.Symbol) return KnownDiscriminantRef is
      function RefTypeToKnownDiscriminantRef is new Unchecked_Conversion (Dictionary.Ref_Type, KnownDiscriminantRef);
   begin
      if GetSymbolDiscriminant (Item) /= Dictionary.KnownDiscriminantSymbol then
         DiscriminantDebug ("GetKnownDiscriminantRef", GetSymbolDiscriminant (Item), Dictionary.KnownDiscriminantSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToKnownDiscriminantRef (GetSymbolRef (Item));
   end GetKnownDiscriminantRef;

   --------------------------------------------------------------------------------

   function GetSubtypeRef (Item : Dictionary.Symbol) return SubtypeRef is
      function RefTypeToSubtypeRef is new Unchecked_Conversion (Dictionary.Ref_Type, SubtypeRef);
   begin
      if GetSymbolDiscriminant (Item) /= Dictionary.SubtypeSymbol then
         DiscriminantDebug ("GetSubtypeRef", GetSymbolDiscriminant (Item), Dictionary.SubtypeSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToSubtypeRef (GetSymbolRef (Item));
   end GetSubtypeRef;

   --------------------------------------------------------------------------------

   function GetDiscriminantConstraintRef (Item : Dictionary.Symbol) return DiscriminantConstraintRef is
      function RefTypeToDiscriminantConstraintRef is new Unchecked_Conversion (Dictionary.Ref_Type, DiscriminantConstraintRef);
   begin
      if GetSymbolDiscriminant (Item) /= Dictionary.DiscriminantConstraintSymbol then
         DiscriminantDebug ("GetDiscriminantConstraintRef", GetSymbolDiscriminant (Item), Dictionary.DiscriminantConstraintSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToDiscriminantConstraintRef (GetSymbolRef (Item));
   end GetDiscriminantConstraintRef;

   --------------------------------------------------------------------------------

   function GetOwnTaskRef (Item : Dictionary.Symbol) return OwnTaskRef is
      function RefTypeToOwnTaskRef is new Unchecked_Conversion (Dictionary.Ref_Type, OwnTaskRef);
   begin
      if GetSymbolDiscriminant (Item) /= Dictionary.OwnTaskSymbol then
         DiscriminantDebug ("GetOwnTaskRef", GetSymbolDiscriminant (Item), Dictionary.OwnTaskSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToOwnTaskRef (GetSymbolRef (Item));
   end GetOwnTaskRef;

   --------------------------------------------------------------------------------

   function GetImplicitProofFunctionRef (Item : Dictionary.Symbol) return Subprogram_Info_Ref is
      function Reftype_To_Subprogram_Info_Ref is new Unchecked_Conversion (Dictionary.Ref_Type, Subprogram_Info_Ref);
   begin
      if Item = Dictionary.NullSymbol then
         return Null_Subprogram_Info_Ref;
      else
         if GetSymbolDiscriminant (Item) /= Dictionary.ImplicitProofFunctionSymbol then
            DiscriminantDebug
              ("GetImplicitProofFunctionRef",
               GetSymbolDiscriminant (Item),
               Dictionary.ImplicitProofFunctionSymbol);
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                      Msg     => "");
         end if;
         return Reftype_To_Subprogram_Info_Ref (GetSymbolRef (Item));
      end if;
   end GetImplicitProofFunctionRef;

   --------------------------------------------------------------------------------

   function GetVirtualElementRef (Item : Dictionary.Symbol) return VirtualElementInfoRef is
      function RefTypeToVirtualElementRef is new Unchecked_Conversion (Dictionary.Ref_Type, VirtualElementInfoRef);
   begin
      if GetSymbolDiscriminant (Item) /= Dictionary.VirtualElementSymbol then
         DiscriminantDebug ("GetVirtualElementRef", GetSymbolDiscriminant (Item), Dictionary.VirtualElementSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToVirtualElementRef (GetSymbolRef (Item));
   end GetVirtualElementRef;

   --------------------------------------------------------------------------------

   function GetSuspendsListItemRef (Item : Dictionary.Symbol) return SuspendsListItemInfoRef is
      function RefTypeToSuspendsListItemRef is new Unchecked_Conversion (Dictionary.Ref_Type, SuspendsListItemInfoRef);
   begin
      if GetSymbolDiscriminant (Item) /= Dictionary.SuspendsListItemSymbol then
         DiscriminantDebug ("GetSuspendsListItemRef", GetSymbolDiscriminant (Item), Dictionary.SuspendsListItemSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToSuspendsListItemRef (GetSymbolRef (Item));
   end GetSuspendsListItemRef;

   --------------------------------------------------------------------------------

   function GetInterruptStreamMappingRef (Item : Dictionary.Symbol) return InterruptStreamMappingRef is
      function RefTypeToInterruptStreamMappingRef is new Unchecked_Conversion (Dictionary.Ref_Type, InterruptStreamMappingRef);
   begin
      if GetSymbolDiscriminant (Item) /= Dictionary.InterruptStreamMappingSymbol then
         DiscriminantDebug ("GetInterruptStreamMappingRef", GetSymbolDiscriminant (Item), Dictionary.InterruptStreamMappingSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToInterruptStreamMappingRef (GetSymbolRef (Item));
   end GetInterruptStreamMappingRef;

   --------------------------------------------------------------------------------

   function GetLoopRef (Item : Dictionary.Symbol) return LoopRef is
      function RefTypeToLoopRef is new Unchecked_Conversion (Dictionary.Ref_Type, LoopRef);
   begin
      if GetSymbolDiscriminant (Item) /= Dictionary.LoopSymbol then
         DiscriminantDebug ("GetLoopRef", GetSymbolDiscriminant (Item), Dictionary.LoopSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToLoopRef (GetSymbolRef (Item));
   end GetLoopRef;

   --------------------------------------------------------------------------------

   function GetLoopParameterRef (Item : Dictionary.Symbol) return LoopParameterRef is
      function RefTypeToLoopParameterRef is new Unchecked_Conversion (Dictionary.Ref_Type, LoopParameterRef);
   begin
      if GetSymbolDiscriminant (Item) /= Dictionary.LoopParameterSymbol then
         DiscriminantDebug ("GetLoopParameterRef", GetSymbolDiscriminant (Item), Dictionary.LoopParameterSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToLoopParameterRef (GetSymbolRef (Item));
   end GetLoopParameterRef;

   --------------------------------------------------------------------------------

   function GetLoopEntryVariableRef (Item : Dictionary.Symbol) return LoopEntryVariableRef is
      function RefTypeToLoopEntryVariableRef is new Unchecked_Conversion (Dictionary.Ref_Type, LoopEntryVariableRef);
   begin
      if GetSymbolDiscriminant (Item) /= Dictionary.LoopEntryVariableSymbol then
         DiscriminantDebug ("GetLoopEntryVariableRef", GetSymbolDiscriminant (Item), Dictionary.LoopEntryVariableSymbol);
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Symbol_Table,
                                   Msg     => "");
      end if;
      return RefTypeToLoopEntryVariableRef (GetSymbolRef (Item));
   end GetLoopEntryVariableRef;

   --------------------------------------------------------------------------------

   procedure CreateProtectedInfo
     (Comp_Unit : in     ContextManager.UnitDescriptors;
      Loc       : in     LexTokenManager.Token_Position;
      InfoSym   :    out Dictionary.Symbol) is
      NewProtected : ProtectedRef;
      function ProtectedRefToRefType is new Unchecked_Conversion (ProtectedRef, Dictionary.Ref_Type);
   begin
      NewProtected :=
        new ProtectedInfo'
        (Own_Variable      => Null_Own_Variable_Info_Ref,
         ElementsHidden    => False,
         Visible_Part      => Part_Info'(With_Clauses          => Null_Context_Clause_Info_Ref,
                                         Use_Type_Clauses      => Null_Use_Type_Clause_Info_Ref,
                                         Renaming_Declarations => Null_Declaration_Info_Ref,
                                         First_Declaration     => Null_Declaration_Info_Ref,
                                         Last_Declaration      => Null_Declaration_Info_Ref),
         Local_Part        => Part_Info'(With_Clauses          => Null_Context_Clause_Info_Ref,
                                         Use_Type_Clauses      => Null_Use_Type_Clause_Info_Ref,
                                         Renaming_Declarations => Null_Declaration_Info_Ref,
                                         First_Declaration     => Null_Declaration_Info_Ref,
                                         Last_Declaration      => Null_Declaration_Info_Ref),
         Private_Part      => Part_Info'(With_Clauses          => Null_Context_Clause_Info_Ref,
                                         Use_Type_Clauses      => Null_Use_Type_Clause_Info_Ref,
                                         Renaming_Declarations => Null_Declaration_Info_Ref,
                                         First_Declaration     => Null_Declaration_Info_Ref,
                                         Last_Declaration      => Null_Declaration_Info_Ref),
         TheEntry          => Dictionary.NullSymbol,
         FirstDiscriminant => Dictionary.NullSymbol,
         LastDiscriminant  => Dictionary.NullSymbol,
         Pragmas           => PragmaLists'(Dictionary.Priority          => Pragmas'(Given => False,
                                                                                    Value => LexTokenManager.Null_String),
                                           Dictionary.InterruptPriority => Pragmas'(Given => False,
                                                                                    Value => LexTokenManager.Null_String),
                                           Dictionary.AttachHandler     => Pragmas'(Given => False,
                                                                                    Value => LexTokenManager.Null_String)),
         Protected_Body    => Null_Declaration_Info_Ref,
         Has_Proper_Body   => False);
      AddSymbol
        (Discriminant => Dictionary.ProtectedInfoSymbol,
         Ref          => ProtectedRefToRefType (NewProtected),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => InfoSym);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateProtectedInfo;

   --------------------------------------------------------------------------------

   procedure CreateTaskInfo
     (Comp_Unit : in     ContextManager.UnitDescriptors;
      Loc       : in     LexTokenManager.Token_Position;
      InfoSym   :    out Dictionary.Symbol) is
      NewTask : TaskRef;

      function TaskRefToRefType is new Unchecked_Conversion (TaskRef, Dictionary.Ref_Type);
   begin
      NewTask :=
        new TaskInfo'
        (Signature_Is_Wellformed    => Booleans'(Dictionary.Abstractions => True),
         Has_Second_Annotation      => False,
         Has_Derives_Annotation     => False,
         Local_Part                 => Part_Info'(With_Clauses          => Null_Context_Clause_Info_Ref,
                                                  Use_Type_Clauses      => Null_Use_Type_Clause_Info_Ref,
                                                  Renaming_Declarations => Null_Declaration_Info_Ref,
                                                  First_Declaration     => Null_Declaration_Info_Ref,
                                                  Last_Declaration      => Null_Declaration_Info_Ref),
         FirstDiscriminant          => Dictionary.NullSymbol,
         LastDiscriminant           => Dictionary.NullSymbol,
         Pragmas                    => PragmaLists'(Dictionary.Priority          => Pragmas'(Given => False,
                                                                                             Value => LexTokenManager.Null_String),
                                                    Dictionary.InterruptPriority => Pragmas'(Given => False,
                                                                                             Value => LexTokenManager.Null_String),
                                                    Dictionary.AttachHandler     => Pragmas'(Given => False,
                                                                                             Value => LexTokenManager.Null_String)),
         First_Global_Variable      => Global_Variables_T'(Dictionary.Abstractions => Null_Global_Variable_Info_Ref),
         Last_Global_Variable       => Global_Variables_T'(Dictionary.Abstractions => Null_Global_Variable_Info_Ref),
         Task_Body                  => Null_Declaration_Info_Ref,
         Has_Proper_Body            => False,
         Body_Is_Hidden             => False,
         Suspends_List              => Dictionary.NullSymbol,
         First_Loop                 => Dictionary.NullSymbol,
         Last_Loop                  => Dictionary.NullSymbol,
         Uses_Unprotected_Variables => False,
         Uses_Unchecked_Conversion  => False,
         Assigns_From_External      => False);
      AddSymbol
        (Discriminant => Dictionary.TaskInfoSymbol,
         Ref          => TaskRefToRefType (NewTask),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => InfoSym);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateTaskInfo;

   procedure CreateOwnTask
     (Variable  : in     Variable_Info_Ref;
      Owner     : in     Package_Info_Ref;
      Comp_Unit : in     ContextManager.UnitDescriptors;
      Loc       : in     LexTokenManager.Token_Position;
      OwnTask   :    out Dictionary.Symbol) is
      function OwnTaskRefToRefType is new Unchecked_Conversion (OwnTaskRef, Dictionary.Ref_Type);
      NewOwnTask : OwnTaskRef;
   begin
      NewOwnTask := new OwnTaskInfo'(Variable => Variable,
                                     Owner    => Owner,
                                     Next     => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => Dictionary.OwnTaskSymbol,
         Ref          => OwnTaskRefToRefType (NewOwnTask),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => OwnTask);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateOwnTask;

   procedure SetNextOwnTask (Current, Next : in Dictionary.Symbol) is
   begin
      GetOwnTaskRef (Current).Next := Next;
   end SetNextOwnTask;

   function GetNextOwnTask (Current : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetOwnTaskRef (Current).Next;
   end GetNextOwnTask;

   function GetOwnTaskVariable (OwnTask : Dictionary.Symbol) return Variable_Info_Ref is
   begin
      return GetOwnTaskRef (OwnTask).Variable;
   end GetOwnTaskVariable;

   function GetOwnTaskOwner (OwnTask : Dictionary.Symbol) return Package_Info_Ref is
   begin
      return GetOwnTaskRef (OwnTask).Owner;
   end GetOwnTaskOwner;

   -- subprograms --------------------------------------------------------------

   procedure CreateImplicitProofFunction
     (Ada_Function  : in     Subprogram_Info_Ref;
      Comp_Unit     : in     ContextManager.UnitDescriptors;
      Loc           : in     LexTokenManager.Token_Position;
      ProofFunction :    out Dictionary.Symbol) is

      function Subprogram_Info_Ref_To_RefType is new Unchecked_Conversion (Subprogram_Info_Ref, Dictionary.Ref_Type);
   begin
      AddSymbol
        (Discriminant => Dictionary.ImplicitProofFunctionSymbol,
         Ref          => Subprogram_Info_Ref_To_RefType (Ada_Function),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => ProofFunction);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateImplicitProofFunction;

   function GetImplicitProofFunctionAdaFunction (ProofFunction : Dictionary.Symbol) return Subprogram_Info_Ref is
   begin
      return GetImplicitProofFunctionRef (ProofFunction);
   end GetImplicitProofFunctionAdaFunction;

   -- discriminants -----------------------------------------------------------------------

   procedure CreateKnownDiscriminant
     (Name              : in     LexTokenManager.Lex_String;
      Protected_Type    : in     Type_Info_Ref;
      Type_Mark         : in     Type_Info_Ref;
      Comp_Unit         : in     ContextManager.UnitDescriptors;
      Loc               : in     LexTokenManager.Token_Position;
      KnownDiscriminant :    out Dictionary.Symbol) is
      NewKnownDiscriminant : KnownDiscriminantRef;
      function KnownDiscriminantRefToRefType is new Unchecked_Conversion (KnownDiscriminantRef, Dictionary.Ref_Type);
   begin
      NewKnownDiscriminant :=
        new KnownDiscriminantInfo'
        (Name           => Name,
         Protected_Type => Protected_Type,
         Type_Mark      => Type_Mark,
         SetsPriority   => False,
         Next           => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => Dictionary.KnownDiscriminantSymbol,
         Ref          => KnownDiscriminantRefToRefType (NewKnownDiscriminant),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => KnownDiscriminant);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateKnownDiscriminant;

   procedure SetNextDiscriminant (Current, Next : in Dictionary.Symbol) is
   begin
      GetKnownDiscriminantRef (Current).Next := Next;
   end SetNextDiscriminant;

   procedure SetDiscriminantSetsPriority (Discriminant : in Dictionary.Symbol) is
   begin
      GetKnownDiscriminantRef (Discriminant).SetsPriority := True;
   end SetDiscriminantSetsPriority;

   function GetNextDiscriminant (Discriminant : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetKnownDiscriminantRef (Discriminant).Next;
   end GetNextDiscriminant;

   function GetDiscriminantName (Discriminant : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetKnownDiscriminantRef (Discriminant).Name;
   end GetDiscriminantName;

   function GetDiscriminantTypeMark (Discriminant : Dictionary.Symbol) return Type_Info_Ref is
   begin
      return GetKnownDiscriminantRef (Discriminant).Type_Mark;
   end GetDiscriminantTypeMark;

   function GetDiscriminantProtectedType (Discriminant : Dictionary.Symbol) return Type_Info_Ref is
   begin
      return GetKnownDiscriminantRef (Discriminant).Protected_Type;
   end GetDiscriminantProtectedType;

   function GetDiscriminantSetsPriority (Discriminant : Dictionary.Symbol) return Boolean is
   begin
      return GetKnownDiscriminantRef (Discriminant).SetsPriority;
   end GetDiscriminantSetsPriority;

   -- note the following creates a record for extra info about task and protected subtypes, not subtypes in general
   procedure CreateSubtype
     (Comp_Unit  : in     ContextManager.UnitDescriptors;
      Loc        : in     LexTokenManager.Token_Position;
      TheSubtype :    out Dictionary.Symbol) is
      NewSubtype : SubtypeRef;
      function SubtypeRefToRefType is new Unchecked_Conversion (SubtypeRef, Dictionary.Ref_Type);
   begin
      NewSubtype :=
        new SubtypeInfo'
        (Priority        => LexTokenManager.Null_String,
         FirstConstraint => Dictionary.NullSymbol,
         LastConstraint  => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => Dictionary.SubtypeSymbol,
         Ref          => SubtypeRefToRefType (NewSubtype),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => TheSubtype);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateSubtype;

   procedure SetSubtypeInfoPriority (TheSubtype : in Dictionary.Symbol;
                                     Priority   : in LexTokenManager.Lex_String) is
   begin
      GetSubtypeRef (TheSubtype).Priority := Priority;
   end SetSubtypeInfoPriority;

   procedure SetSubtypeInfoFirstConstraint (TheSubtype    : in Dictionary.Symbol;
                                            TheConstraint : in Dictionary.Symbol) is
   begin
      GetSubtypeRef (TheSubtype).FirstConstraint := TheConstraint;
   end SetSubtypeInfoFirstConstraint;

   procedure SetSubtypeInfoLastConstraint (TheSubtype    : in Dictionary.Symbol;
                                           TheConstraint : in Dictionary.Symbol) is
   begin
      GetSubtypeRef (TheSubtype).LastConstraint := TheConstraint;
   end SetSubtypeInfoLastConstraint;

   function GetSubtypeInfoPriority (TheSubtype : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetSubtypeRef (TheSubtype).Priority;
   end GetSubtypeInfoPriority;

   function GetSubtypeInfoFirstConstraint (TheSubtype : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubtypeRef (TheSubtype).FirstConstraint;
   end GetSubtypeInfoFirstConstraint;

   function GetSubtypeInfoLastConstraint (TheSubtype : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSubtypeRef (TheSubtype).LastConstraint;
   end GetSubtypeInfoLastConstraint;

   procedure CreateDiscriminantConstraint
     (Comp_Unit                 : in     ContextManager.UnitDescriptors;
      Loc                       : in     LexTokenManager.Token_Position;
      TheDiscriminantConstraint :    out Dictionary.Symbol) is
      NewDiscriminantConstraint : DiscriminantConstraintRef;
      function DiscriminantConstraintRefToRefType is new Unchecked_Conversion (DiscriminantConstraintRef, Dictionary.Ref_Type);
   begin
      NewDiscriminantConstraint :=
        new DiscriminantConstraintInfo'
        (StaticValue    => LexTokenManager.Null_String,
         AccessedObject => Dictionary.NullSymbol,
         Next           => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => Dictionary.DiscriminantConstraintSymbol,
         Ref          => DiscriminantConstraintRefToRefType (NewDiscriminantConstraint),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => TheDiscriminantConstraint);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateDiscriminantConstraint;

   procedure SetDiscriminantConstraintStaticValue
     (TheConstraint : in Dictionary.Symbol;
      TheValue      : in LexTokenManager.Lex_String) is
   begin
      GetDiscriminantConstraintRef (TheConstraint).StaticValue := TheValue;
   end SetDiscriminantConstraintStaticValue;

   procedure SetDiscriminantConstraintAccessedObject (TheConstraint : in Dictionary.Symbol;
                                                      TheObject     : in Dictionary.Symbol) is
   begin
      GetDiscriminantConstraintRef (TheConstraint).AccessedObject := TheObject;
   end SetDiscriminantConstraintAccessedObject;

   procedure SetNextDiscriminantConstraint (TheConstraint : in Dictionary.Symbol;
                                            Next          : in Dictionary.Symbol) is
   begin
      GetDiscriminantConstraintRef (TheConstraint).Next := Next;
   end SetNextDiscriminantConstraint;

   function GetDiscriminantConstraintStaticValue (TheConstraint : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetDiscriminantConstraintRef (TheConstraint).StaticValue;
   end GetDiscriminantConstraintStaticValue;

   function GetDiscriminantConstraintAccessedObject (TheConstraint : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetDiscriminantConstraintRef (TheConstraint).AccessedObject;
   end GetDiscriminantConstraintAccessedObject;

   function GetNextDiscriminantConstraint (TheConstraint : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetDiscriminantConstraintRef (TheConstraint).Next;
   end GetNextDiscriminantConstraint;

   procedure CreateInterruptStreamMapping
     (TheHandler         : in     LexTokenManager.Lex_String;
      TheInterruptStream : in     LexTokenManager.Lex_String;
      Comp_Unit          : in     ContextManager.UnitDescriptors;
      Loc                : in     LexTokenManager.Token_Position;
      TheMapping         :    out Dictionary.Symbol) is
      NewMapping : InterruptStreamMappingRef;
      function InterruptStreamMappingRefToRefType is new Unchecked_Conversion (InterruptStreamMappingRef, Dictionary.Ref_Type);
   begin
      NewMapping :=
        new InterruptStreamMapping'
        (TheHandler         => TheHandler,
         TheInterruptStream => TheInterruptStream,
         Next               => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => Dictionary.InterruptStreamMappingSymbol,
         Ref          => InterruptStreamMappingRefToRefType (NewMapping),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => TheMapping);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateInterruptStreamMapping;

   procedure SetNextInterruptStreamMapping (Current, Next : in Dictionary.Symbol) is
   begin
      GetInterruptStreamMappingRef (Current).Next := Next;
   end SetNextInterruptStreamMapping;

   function GetNextInterruptStreamMapping (TheMapping : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetInterruptStreamMappingRef (TheMapping).Next;
   end GetNextInterruptStreamMapping;

   function GetInterruptStreamMappingHandler (TheMapping : in Symbol) return LexTokenManager.Lex_String is
   begin
      return GetInterruptStreamMappingRef (TheMapping).TheHandler;
   end GetInterruptStreamMappingHandler;

   function GetInterruptStreamMappingStream (TheMapping : in Symbol) return LexTokenManager.Lex_String is
   begin
      return GetInterruptStreamMappingRef (TheMapping).TheInterruptStream;
   end GetInterruptStreamMappingStream;

   procedure CreateSuspendsListItem
     (ThePOorSO        : in     Dictionary.Symbol;
      Comp_Unit        : in     ContextManager.UnitDescriptors;
      Loc              : in     LexTokenManager.Token_Position;
      SuspendsListItem :    out Dictionary.Symbol) is
      NewSuspendsListItem : SuspendsListItemInfoRef;
      function SuspendsListItemRefToRefType is new Unchecked_Conversion (SuspendsListItemInfoRef, Dictionary.Ref_Type);
   begin
      NewSuspendsListItem :=
        new SuspendsListItemInfo'(ThePOorSO      => ThePOorSO,
                                  IsAccountedFor => False,
                                  Next           => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => Dictionary.SuspendsListItemSymbol,
         Ref          => SuspendsListItemRefToRefType (NewSuspendsListItem),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => SuspendsListItem);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateSuspendsListItem;

   procedure SetNextSuspendsListItem (Current, Next : in Dictionary.Symbol) is
   begin
      GetSuspendsListItemRef (Current).Next := Next;
   end SetNextSuspendsListItem;

   function GetSuspendsListItem (SuspendsListItem : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSuspendsListItemRef (SuspendsListItem).ThePOorSO;
   end GetSuspendsListItem;

   procedure SetSuspendsListItemIsAccountedFor (SuspendsListItem : in Dictionary.Symbol) is
   begin
      GetSuspendsListItemRef (SuspendsListItem).IsAccountedFor := True;
   end SetSuspendsListItemIsAccountedFor;

   function GetSuspendsListItemIsAccountedFor (SuspendsListItem : Dictionary.Symbol) return Boolean is
   begin
      return GetSuspendsListItemRef (SuspendsListItem).IsAccountedFor;
   end GetSuspendsListItemIsAccountedFor;

   function GetNextSuspendsListItem (SuspendsListItem : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetSuspendsListItemRef (SuspendsListItem).Next;
   end GetNextSuspendsListItem;

   procedure CreateVirtualElement
     (The_Variable      : in     RawDict.Variable_Info_Ref;
      TheOwner          : in     Dictionary.Symbol;
      Comp_Unit         : in     ContextManager.UnitDescriptors;
      Loc               : in     LexTokenManager.Token_Position;
      TheVirtualElement :    out Dictionary.Symbol) is
      NewVirtualElement : VirtualElementInfoRef;
      function VirtualElementRefToRefType is new Unchecked_Conversion (VirtualElementInfoRef, Dictionary.Ref_Type);
   begin
      NewVirtualElement :=
        new VirtualElementInfo'
        (The_Variable => The_Variable,
         TheOwner     => TheOwner,
         SeenByOwner  => False,
         Next         => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => Dictionary.VirtualElementSymbol,
         Ref          => VirtualElementRefToRefType (NewVirtualElement),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => TheVirtualElement);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateVirtualElement;

   function GetVirtualElementVariable (VirtualElement : Dictionary.Symbol) return RawDict.Variable_Info_Ref is
   begin
      return GetVirtualElementRef (VirtualElement).The_Variable;
   end GetVirtualElementVariable;

   function GetVirtualElementOwner (VirtualElement : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetVirtualElementRef (VirtualElement).TheOwner;
   end GetVirtualElementOwner;

   procedure SetVirtualElementSeenByOwner (VirtualElement : in Dictionary.Symbol) is
   begin
      GetVirtualElementRef (VirtualElement).SeenByOwner := True;
   end SetVirtualElementSeenByOwner;

   function GetVirtualElementSeenByOwner (VirtualElement : Dictionary.Symbol) return Boolean is
   begin
      return GetVirtualElementRef (VirtualElement).SeenByOwner;
   end GetVirtualElementSeenByOwner;

   procedure SetNextVirtualElement (Current, Next : in Dictionary.Symbol) is
   begin
      GetVirtualElementRef (Current).Next := Next;
   end SetNextVirtualElement;

   function GetNextVirtualElement (VirtualElement : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetVirtualElementRef (VirtualElement).Next;
   end GetNextVirtualElement;

   procedure CreateLoop
     (Region    : in     Dictionary.Symbol;
      Comp_Unit : in     ContextManager.UnitDescriptors;
      Loc       : in     LexTokenManager.Token_Position;
      TheLoop   :    out Dictionary.Symbol) is
      NewLoop : LoopRef;
      function LoopRefToRefType is new Unchecked_Conversion (LoopRef, Dictionary.Ref_Type);
   begin
      NewLoop :=
        new LoopInfo'
        (Name          => LexTokenManager.Null_String,
         Region        => Region,
         LoopParameter => Dictionary.NullSymbol,
         OnEntryVars   => Dictionary.NullSymbol,
         ExitExpn      => 0,
         EntryExpn     => 0,
         HasExits      => False,
         Next          => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => Dictionary.LoopSymbol,
         Ref          => LoopRefToRefType (NewLoop),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => TheLoop);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateLoop;

   procedure SetLoopName (Name    : in LexTokenManager.Lex_String;
                          TheLoop : in Dictionary.Symbol) is
   begin
      GetLoopRef (TheLoop).Name := Name;
   end SetLoopName;

   procedure SetLoopParameter (ForLoop, LoopParameter : in Dictionary.Symbol) is
   begin
      GetLoopRef (ForLoop).LoopParameter := LoopParameter;
   end SetLoopParameter;

   procedure SetLoopOnEntryVars (ForLoop, OnEntryVars : in Dictionary.Symbol) is
   begin
      GetLoopRef (ForLoop).OnEntryVars := OnEntryVars;
   end SetLoopOnEntryVars;

   procedure SetLoopExitExpn (ForLoop : in Dictionary.Symbol;
                              Expn    : in Natural) is
   begin
      GetLoopRef (ForLoop).ExitExpn := Expn;
   end SetLoopExitExpn;

   procedure SetLoopEntryExpn (ForLoop : in Dictionary.Symbol;
                               Expn    : in Natural) is
   begin
      GetLoopRef (ForLoop).EntryExpn := Expn;
   end SetLoopEntryExpn;

   procedure SetLoopHasExits (TheLoop : in Dictionary.Symbol) is
   begin
      GetLoopRef (TheLoop).HasExits := True;
   end SetLoopHasExits;

   procedure SetNextLoop (Current, Next : in Dictionary.Symbol) is
   begin
      GetLoopRef (Current).Next := Next;
   end SetNextLoop;

   function GetLoopName (TheLoop : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetLoopRef (TheLoop).Name;
   end GetLoopName;

   function GetLoopRegion (TheLoop : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetLoopRef (TheLoop).Region;
   end GetLoopRegion;

   function GetLoopParameter (TheLoop : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetLoopRef (TheLoop).LoopParameter;
   end GetLoopParameter;

   function GetLoopOnEntryVars (TheLoop : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetLoopRef (TheLoop).OnEntryVars;
   end GetLoopOnEntryVars;

   function GetLoopExitExpn (TheLoop : Dictionary.Symbol) return Natural is
   begin
      return GetLoopRef (TheLoop).ExitExpn;
   end GetLoopExitExpn;

   function GetLoopEntryExpn (TheLoop : Dictionary.Symbol) return Natural is
   begin
      return GetLoopRef (TheLoop).EntryExpn;
   end GetLoopEntryExpn;

   function GetLoopHasExits (TheLoop : Dictionary.Symbol) return Boolean is
   begin
      return GetLoopRef (TheLoop).HasExits;
   end GetLoopHasExits;

   function GetNextLoop (TheLoop : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetLoopRef (TheLoop).Next;
   end GetNextLoop;

   procedure CreateLoopParameter
     (Name           : in     LexTokenManager.Lex_String;
      Type_Mark      : in     Type_Info_Ref;
      TheLoop        : in     Dictionary.Symbol;
      HasStaticRange : in     Boolean;
      IsReverse      : in     Boolean;
      Comp_Unit      : in     ContextManager.UnitDescriptors;
      Loc            : in     LexTokenManager.Token_Position;
      LoopParameter  :    out Dictionary.Symbol) is
      NewParameter : LoopParameterRef;
      function LoopParameterRefToRefType is new Unchecked_Conversion (LoopParameterRef, Dictionary.Ref_Type);
   begin
      NewParameter :=
        new LoopParameterInfo'
        (Name           => Name,
         Type_Mark      => Type_Mark,
         TheLoop        => TheLoop,
         HasStaticRange => HasStaticRange,
         IsReverse      => IsReverse);
      AddSymbol
        (Discriminant => Dictionary.LoopParameterSymbol,
         Ref          => LoopParameterRefToRefType (NewParameter),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => LoopParameter);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateLoopParameter;

   procedure CreateLoopEntryVariable
     (OriginalVar       : in     Dictionary.Symbol;
      TheLoop           : in     Dictionary.Symbol;
      Comp_Unit         : in     ContextManager.UnitDescriptors;
      Loc               : in     LexTokenManager.Token_Position;
      LoopEntryVariable :    out Dictionary.Symbol) is
      NewBound : LoopEntryVariableRef;
      function LoopEntryVariableRefToRefType is new Unchecked_Conversion (LoopEntryVariableRef, Dictionary.Ref_Type);
   begin
      NewBound := new LoopEntryVariableInfo'(OriginalVar => OriginalVar,
                                             TheLoop     => TheLoop,
                                             Next        => Dictionary.NullSymbol);
      AddSymbol
        (Discriminant => Dictionary.LoopEntryVariableSymbol,
         Ref          => LoopEntryVariableRefToRefType (NewBound),
         Comp_Unit    => Comp_Unit,
         Loc          => Loc,
         Item         => LoopEntryVariable);
   exception
      when Storage_Error =>
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Symbol_Table_Overflow_Dynamic,
                                   Msg     => "");
   end CreateLoopEntryVariable;

   function GetLoopParameterName (LoopParameter : Dictionary.Symbol) return LexTokenManager.Lex_String is
   begin
      return GetLoopParameterRef (LoopParameter).Name;
   end GetLoopParameterName;

   function GetLoopParameterType (LoopParameter : Dictionary.Symbol) return Type_Info_Ref is
   begin
      return GetLoopParameterRef (LoopParameter).Type_Mark;
   end GetLoopParameterType;

   function GetLoopParameterLoop (LoopParameter : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetLoopParameterRef (LoopParameter).TheLoop;
   end GetLoopParameterLoop;

   function GetLoopParameterHasStaticRange (LoopParameter : Dictionary.Symbol) return Boolean is
   begin
      return GetLoopParameterRef (LoopParameter).HasStaticRange;
   end GetLoopParameterHasStaticRange;

   function GetLoopParameterIsReverse (LoopParameter : Dictionary.Symbol) return Boolean is
   begin
      return GetLoopParameterRef (LoopParameter).IsReverse;
   end GetLoopParameterIsReverse;

   procedure SetLoopEntryVariableNext (LoopEntryVariable : in Dictionary.Symbol;
                                       Next              : in Dictionary.Symbol) is
   begin
      GetLoopEntryVariableRef (LoopEntryVariable).Next := Next;
   end SetLoopEntryVariableNext;

   function GetLoopEntryVariableOriginalVar (LoopEntryVariable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetLoopEntryVariableRef (LoopEntryVariable).OriginalVar;
   end GetLoopEntryVariableOriginalVar;

   function GetLoopEntryVariableTheLoop (LoopEntryVariable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetLoopEntryVariableRef (LoopEntryVariable).TheLoop;
   end GetLoopEntryVariableTheLoop;

   function GetLoopEntryVariableNext (LoopEntryVariable : Dictionary.Symbol) return Dictionary.Symbol is
   begin
      return GetLoopEntryVariableRef (LoopEntryVariable).Next;
   end GetLoopEntryVariableNext;

end RawDict; -- hidden because of non-SPARK code
