/*
 * Copyright 2023 KylinSoft Co., Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <QDebug>

#include "powersupplydev.h"

PowerSupplyDev::PowerSupplyDev(QObject *parent) : QObject(parent)
{
    m_upowerDBus = new UPowerDBus(this);
    connect(m_upowerDBus, &UPowerDBus::lidStateChanged, this, &PowerSupplyDev::dealLidStateChanged);
    connect(m_upowerDBus, &UPowerDBus::batteryOnlineStateChanged, this, &PowerSupplyDev::dealBatteryOnlineStateChanged);

    int index = 0;
    QStringList batteryPathList = m_upowerDBus->getBatteryPathList();
    for (index = 0; index < batteryPathList.size(); ++index) {
        UpowerBatteryDBus *upowerBatteryDBus = new UpowerBatteryDBus(index, batteryPathList.at(index), this);
        connect(upowerBatteryDBus, &UpowerBatteryDBus::batteryInfoChanged, this, &PowerSupplyDev::dealBatteryInfoChanged);
        m_batteryDBusVector << upowerBatteryDBus;
    }

    QStringList linePowerPathList = m_upowerDBus->getLinePowerPathList();
    for (index = 0; index < linePowerPathList.size(); ++index) {
        UpowerLinePowerDBus *upowerLinePowerDBus = new UpowerLinePowerDBus(index, linePowerPathList.at(index), this);
        connect(upowerLinePowerDBus, &UpowerLinePowerDBus::acOnlineStateChanged, this, &PowerSupplyDev::dealAcOnlineStateChanged);
        m_linePowerDBusVector << upowerLinePowerDBus;
    }

    qDebug() << "Battery Num:" << m_batteryDBusVector.size();
    qDebug() << "line power Num:" << m_linePowerDBusVector.size();
}

PowerSupplyDev::~PowerSupplyDev() {}

void PowerSupplyDev::dealLidStateChanged(bool value)
{
    qDebug() << "dealLidStateChanged";
    Q_EMIT lidStateChanged(value);
}

void PowerSupplyDev::dealBatteryOnlineStateChanged(bool value)
{
    Q_EMIT batteryOnlineStateChanged(value);
}

void PowerSupplyDev::dealBatteryInfoChanged(int index, QStringList batteryInfoChangedList)
{
    qDebug() << "dealBatteryInfoChanged";
    Q_EMIT batteryInfoChanged(index, batteryInfoChangedList);
}

void PowerSupplyDev::dealAcOnlineStateChanged(int index, bool value)
{
    qDebug() << "dealAcOnlineStateChanged";
    Q_EMIT acOnlineStateChanged(index, value);
}

bool PowerSupplyDev::getLidClosedState()
{
    return m_upowerDBus->getLidClosedState();
}

bool PowerSupplyDev::getBatteryOnlineState()
{
    return m_upowerDBus->getBatteryOnlineState();
}

QString PowerSupplyDev::getUpowerDaemonVersion()
{
    return m_upowerDBus->getUpowerDaemonVersion();
}

int PowerSupplyDev::getBatteryNum()
{
    return m_batteryDBusVector.size();
}

int PowerSupplyDev::getLinePowerNum()
{
    return m_linePowerDBusVector.size();
}

int PowerSupplyDev::getBatteryState(int index)
{
    if (0 == m_batteryDBusVector.size()) {
        return 0;
    }
    return m_batteryDBusVector.at(index)->getBatteryState();
}

double PowerSupplyDev::getOneBatteryPercentage(int index)
{
    if (0 == m_batteryDBusVector.size()) {
        return 0;
    }
    return m_batteryDBusVector.at(index)->getBatteryPercentage();
}

double PowerSupplyDev::getBatteryPercentage()
{
    if (0 == m_batteryDBusVector.size()) {
        return 0;
    }

    double percentage = 0.0;
    for (int index = 0; index < m_batteryDBusVector.size(); ++index) {
        percentage += m_batteryDBusVector.at(index)->getBatteryPercentage();
    }

    return percentage / m_batteryDBusVector.size();
}

double PowerSupplyDev::calcBatteryPercentage()
{
    if (0 == m_batteryDBusVector.size()) {
        return 0;
    }

    double energyNow = 0.0;
    double energyFull = 0.0;
    for (int index = 0; index < m_batteryDBusVector.size(); ++index) {
        energyNow += m_batteryDBusVector.at(index)->getEnergyNow();
        energyFull += m_batteryDBusVector.at(index)->getEnergyFull();
    }

    return energyNow / energyFull * 100;
}

double PowerSupplyDev::getEnergyNow(int index)
{
    if (0 == m_batteryDBusVector.size()) {
        return 0;
    }
    return m_batteryDBusVector.at(index)->getEnergyNow();
}

double PowerSupplyDev::getEnergyFull(int index)
{
    if (0 == m_batteryDBusVector.size()) {
        return 0;
    }
    return m_batteryDBusVector.at(index)->getEnergyFull();
}

double PowerSupplyDev::getEnergyFullDesign(int index)
{
    if (0 == m_batteryDBusVector.size()) {
        return 0;
    }
    return m_batteryDBusVector.at(index)->getEnergyFullDesign();
}

double PowerSupplyDev::getEnergyRate(int index)
{
    if (0 == m_batteryDBusVector.size()) {
        return 0;
    }
    return m_batteryDBusVector.at(index)->getEnergyRate();
}

qlonglong PowerSupplyDev::getTimeToFull(int index)
{
    if (0 == m_batteryDBusVector.size()) {
        return 0;
    }
    return m_batteryDBusVector.at(index)->getTimeToFull();
}

qlonglong PowerSupplyDev::getTimeToEmpty(int index)
{
    if (0 == m_batteryDBusVector.size()) {
        return 0;
    }
    return m_batteryDBusVector.at(index)->getTimeToEmpty();
}

double PowerSupplyDev::getBatteryTemperature(int index)
{
    if (0 == m_batteryDBusVector.size()) {
        return 0;
    }
    return m_batteryDBusVector.at(index)->getBatteryTemperature();
}

int PowerSupplyDev::getWarninglevel(int index)
{
    if (0 == m_batteryDBusVector.size()) {
        return 0;
    }
    return m_batteryDBusVector.at(index)->getWarninglevel();
}

QString PowerSupplyDev::getBatteryVendor(int index)
{
    if (0 == m_batteryDBusVector.size()) {
        return "unknown";
    }
    return m_batteryDBusVector.at(index)->getBatteryVendor();
}

QString PowerSupplyDev::getBatteryModel(int index)
{
    if (0 == m_batteryDBusVector.size()) {
        return "unknown";
    }
    return m_batteryDBusVector.at(index)->getBatteryModel();
}

QString PowerSupplyDev::getOneBatteryIconName(int index)
{
    if (0 == m_batteryDBusVector.size() || 0 == m_linePowerDBusVector.size()) {
        return "battery-level-0-charging-symbolic";
    }
    double percentage = m_batteryDBusVector.at(index)->getBatteryPercentage();

    if (true == m_linePowerDBusVector.at(index)->getAcOnlineState()) {
        m_batteryIconName = QString("battery-level-%1-charging-symbolic").arg((int)percentage / 10 * 10);
    } else {
        m_batteryIconName = QString("battery-level-%1-symbolic").arg((int)percentage / 10 * 10);
    }

    return m_batteryIconName;
}

QString PowerSupplyDev::getBatteryIconName()
{
    if (0 == m_batteryDBusVector.size() || 0 == m_linePowerDBusVector.size()) {
        return "battery-level-0-charging-symbolic";
    }

    double percentage = getBatteryPercentage();
    if (true == getAcOnlineState()) {
        m_batteryIconName = QString("battery-level-%1-charging-symbolic").arg((int)percentage / 10 * 10);
    } else {
        m_batteryIconName = QString("battery-level-%1-symbolic").arg((int)percentage / 10 * 10);
    }

    return m_batteryIconName;
}

bool PowerSupplyDev::getAcOnlineState()
{
    if (0 == m_batteryDBusVector.size() || 0 == m_linePowerDBusVector.size()) {
        return true;
    }
    for (int index = 0; index < m_linePowerDBusVector.size(); ++index) {
        if (true == m_linePowerDBusVector.at(index)->getAcOnlineState()) {
            return true;
        }
    }
    return false;
}
